// =========================================================
// Payment Manager - Unified Numbering System v5.0.0
// =========================================================

class PaymentManager {
    constructor() {
        this.version = '5.0.0';
        this.processors = new Map();
        this.activeProcessor = null;
        this.currentTransaction = null;
        this.transactionHistory = [];
        this.config = this.loadConfig();
        this.events = this.createEventSystem();
        this.ui = null;
        this.initialized = false;
        this.isProcessing = false;
        
        // Setup immediate methods
        this.setupImmediateMethods();
    }

    // ========== INITIALIZATION ==========
    async init() {
        console.log(`💰 Payment Manager v${this.version} initializing...`);
        
        try {
            this.setupEventListeners();
            this.registerDefaultProcessors();
            this.setupUI();
            this.loadTransactionHistory();
            this.setupAutoSave();
            
            // Initialize with a unified number
            await this.initializeUnifiedNumber();
            
            this.initialized = true;
            console.log(`✅ Payment Manager v${this.version} ready`);
            this.trigger('initialized', { version: this.version });
            
        } catch (error) {
            console.error('❌ Payment Manager initialization failed:', error);
            this.handleFatalError(error);
        }
        
        return this;
    }

    // ========== UNIFIED NUMBERING SYSTEM ==========
    
    async initializeUnifiedNumber() {
        console.log('🔄 Initializing unified numbering system...');
        
        try {
            // Try to get initial unified number from server
            const numbers = await this.generateUnifiedNumber();
            
            console.log('✅ Unified numbering initialized:', {
                unified: numbers.unified,
                source: numbers.fromServer ? 'server' : 'fallback'
            });
            
            // Update UI immediately
            this.updateUnifiedNumberUI(numbers.unified);
            
        } catch (error) {
            console.error('❌ Failed to initialize unified numbering:', error);
            // Continue with fallback
        }
    }

    async generateUnifiedNumber() {
        try {
            console.log('🔄 Requesting unified number from server...');
            
            const response = await fetch('/api/unified-number', {
                method: 'GET',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                    'Content-Type': 'application/json'
                },
                cache: 'no-cache',
                timeout: 5000 // 5 second timeout
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    console.log('✅ Got unified number from server:', data.unified_number);
                    
                    // Store in localStorage as backup
                    localStorage.setItem('pos_last_unified_number', data.unified_number);
                    localStorage.setItem('pos_last_invoice_no', data.unified_number);
                    localStorage.setItem('pos_last_receipt_no', data.unified_number);
                    
                    // Update UI if elements exist
                    this.updateUnifiedNumberUI(data.unified_number);
                    
                    return {
                        unified: data.unified_number,
                        invoice: data.unified_number,
                        receipt: data.unified_number,
                        fromServer: true,
                        date: data.date,
                        timestamp: data.timestamp
                    };
                }
            }
            
            throw new Error('Server response not successful');
            
        } catch (error) {
            console.warn('⚠️ Failed to get unified number from server:', error);
            console.log('🔄 Using fallback unified number generation...');
            
            return this.generateUnifiedNumberFallback();
        }
    }

    generateUnifiedNumberFallback() {
        // Get last number from localStorage
        let lastNumber = localStorage.getItem('pos_last_unified_number');
        let counter = 1;
        
        if (lastNumber) {
            // Extract counter from format: INVRCP-YYYYMMDD-0001
            const match = lastNumber.match(/INVRCP-\d{8}-(\d{4})/);
            if (match) {
                counter = parseInt(match[1]) + 1;
            }
        }
        
        // Format: INVRCP-YYYYMMDD-0001
        const today = new Date();
        const year = today.getFullYear();
        const month = String(today.getMonth() + 1).padStart(2, '0');
        const day = String(today.getDate()).padStart(2, '0');
        const dateStr = `${year}${month}${day}`;
        
        const unifiedNumber = `INVRCP-${dateStr}-${String(counter).padStart(4, '0')}`;
        
        // Save for next use
        localStorage.setItem('pos_last_unified_number', unifiedNumber);
        localStorage.setItem('pos_last_invoice_no', unifiedNumber);
        localStorage.setItem('pos_last_receipt_no', unifiedNumber);
        
        console.log('📊 Generated fallback unified number:', unifiedNumber);
        
        return {
            unified: unifiedNumber,
            invoice: unifiedNumber,
            receipt: unifiedNumber,
            fromServer: false,
            isFallback: true,
            date: dateStr,
            timestamp: new Date().toISOString()
        };
    }

    updateUnifiedNumberUI(unifiedNumber) {
        try {
            // Update all elements that display invoice/receipt numbers
            const selectors = [
                '[data-invoice-number]',
                '[data-receipt-number]',
                '[data-unified-number]',
                '.invoice-number',
                '.receipt-number',
                '.unified-number',
                '#invoiceNumberDisplay',
                '#receiptNumberDisplay',
                '#unifiedNumberDisplay'
            ];
            
            selectors.forEach(selector => {
                const elements = document.querySelectorAll(selector);
                elements.forEach(element => {
                    if (element.matches('[data-invoice-number], .invoice-number, #invoiceNumberDisplay')) {
                        element.textContent = unifiedNumber;
                        if (element.dataset) element.dataset.invoiceNumber = unifiedNumber;
                    }
                    if (element.matches('[data-receipt-number], .receipt-number, #receiptNumberDisplay')) {
                        element.textContent = unifiedNumber;
                        if (element.dataset) element.dataset.receiptNumber = unifiedNumber;
                    }
                    if (element.matches('[data-unified-number], .unified-number, #unifiedNumberDisplay')) {
                        element.textContent = unifiedNumber;
                        if (element.dataset) element.dataset.unifiedNumber = unifiedNumber;
                    }
                });
            });
            
            // Update payment modal if open
            const invoiceElement = document.getElementById('paymentInvoiceNo');
            const receiptElement = document.getElementById('paymentReceiptNo');
            const unifiedElement = document.getElementById('paymentUnifiedNo');
            
            if (invoiceElement) {
                invoiceElement.textContent = unifiedNumber;
                if (invoiceElement.dataset) invoiceElement.dataset.invoiceNumber = unifiedNumber;
            }
            
            if (receiptElement) {
                receiptElement.textContent = unifiedNumber;
                if (receiptElement.dataset) receiptElement.dataset.receiptNumber = unifiedNumber;
            }
            
            if (unifiedElement) {
                unifiedElement.textContent = unifiedNumber;
                if (unifiedElement.dataset) unifiedElement.dataset.unifiedNumber = unifiedNumber;
            }
            
            console.log('📊 Updated UI with unified number:', unifiedNumber);
            
        } catch (error) {
            console.error('Error updating unified number UI:', error);
        }
    }

    // ========== EVENT LISTENERS SETUP ==========
    setupEventListeners() {
        console.log('🎮 Setting up payment event listeners...');
        
        // Method selection - use arrow function or bind properly
        document.addEventListener('click', (e) => {
            const card = e.target.closest('.payment-method-card');
            if (card && card.dataset.method) {
                e.preventDefault();
                e.stopPropagation();
                this.selectPaymentMethod(card.dataset.method, card);
            }
        });
        
        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') return;
            
            const modal = document.getElementById('paymentModal');
            const isModalOpen = modal && modal.classList.contains('show');
            
            if (!isModalOpen) return;
            
            switch(e.key) {
                case 'Escape':
                    e.preventDefault();
                    this.hideModal();
                    break;
                case 'Enter':
                    e.preventDefault();
                    const nextBtn = document.getElementById('nextStepBtn');
                    if (nextBtn && nextBtn.style.display !== 'none') {
                        nextBtn.click();
                    }
                    break;
                case 'F2':
                    e.preventDefault();
                    this.showModal();
                    break;
            }
        });
        
        console.log('✅ Event listeners setup complete');
    }

    setupImmediateMethods() {
        // Make methods available immediately
        this.showModal = this.showModal.bind(this);
        this.selectPaymentMethod = this.selectPaymentMethod.bind(this);
        this.hideModal = this.hideModal.bind(this);
        this.generateUnifiedNumber = this.generateUnifiedNumber.bind(this);
        
        // Set global functions
        window.PaymentManager = this;
        window.showPaymentModal = () => this.showModal();
        window.selectPaymentMethod = (method) => this.selectPaymentMethod(method);
        window.generateUnifiedNumber = () => this.generateUnifiedNumber();
    }

    // ========== TRANSACTION NUMBERING SYSTEM ==========
    generateTransactionNumber() {
        try {
            // Daily transaction counter
            const today = new Date().toISOString().split('T')[0].replace(/-/g, '');
            const storageKey = `pos_transaction_counter_${today}`;
            
            let transactionCounter = parseInt(localStorage.getItem(storageKey)) || 1;
            
            const transactionNumber = `TXN-${today}-${String(transactionCounter).padStart(4, '0')}`;
            
            // Increment for next transaction
            localStorage.setItem(storageKey, transactionCounter + 1);
            
            return transactionNumber;
            
        } catch (error) {
            console.error('Error generating transaction number:', error);
            return `TXN-${Date.now()}`;
        }
    }

    setupDailyCounterReset() {
        const now = new Date();
        const tomorrow = new Date(now);
        tomorrow.setDate(tomorrow.getDate() + 1);
        tomorrow.setHours(0, 0, 1, 0);
        
        const timeUntilMidnight = tomorrow.getTime() - now.getTime();
        
        setTimeout(() => {
            // Clear only transaction counters (unified numbers persist)
            const keysToRemove = [];
            for (let i = 0; i < localStorage.length; i++) {
                const key = localStorage.key(i);
                if (key.includes('pos_transaction_counter_')) {
                    keysToRemove.push(key);
                }
            }
            
            keysToRemove.forEach(key => {
                localStorage.removeItem(key);
            });
            
            console.log('🔁 Daily transaction counters reset at midnight');
            
            // Schedule next reset
            this.setupDailyCounterReset();
        }, timeUntilMidnight);
    }

    // ========== PROCESSOR MANAGEMENT ==========
    registerDefaultProcessors() {
        console.log('📋 Registering default processors...');
        
        this.registerProcessor('cash', this.createCashProcessor());
        this.registerProcessor('mpesa', this.createMpesaProcessor());
        this.registerProcessor('card', this.createCardProcessor());
        this.registerProcessor('credit', this.createCreditProcessor());
        this.registerProcessor('multiple', this.createSplitProcessor());
        this.registerProcessor('bank', this.createBankProcessor());
        
        console.log(`✅ Registered ${this.processors.size} payment processors`);
    }

    registerProcessor(code, processor) {
        if (!processor || typeof processor.process !== 'function') {
            console.error(`Invalid processor for code: ${code}`, processor);
            return;
        }
        
        this.processors.set(code, processor);
        console.log(`✅ Registered processor: ${code}`);
    }

    getProcessor(code) {
        const processor = this.processors.get(code);
        if (!processor) {
            throw new Error(`Processor not found: ${code}`);
        }
        return processor;
    }

    // ========== SIMPLE PROCESSOR CREATORS ==========
    createCashProcessor() {
        return {
            methodCode: 'cash',
            methodName: 'Cash',
            async validate(data) {
                const amount = parseFloat(data.amount || 0);
                const received = parseFloat(data.cash_received || 0);
                
                if (received < amount) {
                    throw new Error(`Insufficient cash. Need KES ${(amount - received).toFixed(2)} more`);
                }
                
                return { success: true, data: { amount, received } };
            },
            async process(data) {
                const amount = parseFloat(data.amount || 0);
                const received = parseFloat(data.cash_received || amount);
                const change = Math.max(received - amount, 0);
                
                return {
                    success: true,
                    data: {
                        transactionId: `CASH_${Date.now()}`,
                        amount: amount,
                        cash_received: received,
                        change: change,
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    createMpesaProcessor() {
        return {
            methodCode: 'mpesa',
            methodName: 'M-Pesa',
            async validate(data) {
                const phone = data.mpesa_phone || '';
                if (!phone || phone.length !== 9 || !/^[17]/.test(phone)) {
                    throw new Error('Please enter valid 9-digit Safaricom number');
                }
                return { success: true, data };
            },
            async process(data) {
                return {
                    success: true,
                    data: {
                        transactionId: `MPESA_${Date.now()}`,
                        amount: parseFloat(data.amount || 0),
                        mpesa_phone: data.mpesa_phone,
                        mpesa_type: data.mpesa_type || 'stk',
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    createCardProcessor() {
        return {
            methodCode: 'card',
            methodName: 'Card',
            async validate(data) {
                const cardNumber = data.card_number || '';
                if (!cardNumber || cardNumber.replace(/\s/g, '').length !== 16) {
                    throw new Error('Please enter valid 16-digit card number');
                }
                return { success: true, data };
            },
            async process(data) {
                return {
                    success: true,
                    data: {
                        transactionId: `CARD_${Date.now()}`,
                        amount: parseFloat(data.amount || 0),
                        card_number: data.card_number,
                        card_last4: (data.card_number || '').slice(-4),
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    createCreditProcessor() {
        return {
            methodCode: 'credit',
            methodName: 'Credit',
            async validate(data) {
                if (!data.customer_id) {
                    throw new Error('Please select a customer for credit sale');
                }
                return { success: true, data };
            },
            async process(data) {
                return {
                    success: true,
                    data: {
                        transactionId: `CREDIT_${Date.now()}`,
                        amount: parseFloat(data.amount || 0),
                        customer_id: data.customer_id,
                        terms: data.terms || '30',
                        due_date: data.due_date || new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    createSplitProcessor() {
        return {
            methodCode: 'multiple',
            methodName: 'Split Payment',
            async validate(data) {
                const total = parseFloat(data.amount || 0);
                const splits = data.splits || {};
                const splitTotal = Object.values(splits).reduce((sum, val) => sum + parseFloat(val || 0), 0);
                
                if (Math.abs(splitTotal - total) > 0.01) {
                    throw new Error(`Split total (KES ${splitTotal.toFixed(2)}) doesn't match sale total (KES ${total.toFixed(2)})`);
                }
                
                return { success: true, data };
            },
            async process(data) {
                return {
                    success: true,
                    data: {
                        transactionId: `SPLIT_${Date.now()}`,
                        amount: parseFloat(data.amount || 0),
                        splits: data.splits || {},
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    createBankProcessor() {
        return {
            methodCode: 'bank',
            methodName: 'Bank Transfer',
            async validate(data) {
                if (!data.bank_reference) {
                    throw new Error('Bank reference number is required');
                }
                return { success: true, data };
            },
            async process(data) {
                return {
                    success: true,
                    data: {
                        transactionId: `BANK_${Date.now()}`,
                        amount: parseFloat(data.amount || 0),
                        bank_reference: data.bank_reference,
                        bank_name: data.bank_name || '',
                        timestamp: new Date().toISOString()
                    }
                };
            }
        };
    }

    // ========== VAT CALCULATION METHODS ==========
    
    /**
     * Calculate VAT with customer status consideration
     */
    calculateVATWithCustomerStatus(cart, customer) {
        try {
            // Use shared VAT service if available
            if (window.VATService && window.VATService.calculateCartVAT) {
                return window.VATService.calculateCartVAT(
                    cart.items || cart || [],
                    customer,
                    cart.discount || 0
                );
            }
            
            // Fallback calculation
            return this.calculateVATWithCustomerStatusFallback(cart, customer);
            
        } catch (error) {
            console.error('Error calculating VAT with customer status:', error);
            return this.getEmptyVATData(customer);
        }
    }
    
    calculateVATWithCustomerStatusFallback(cart, customer) {
        // Initialize all values with defaults
        let subtotal = 0;
        let totalVAT = 0;
        let taxableAmount = 0;
        let nonTaxableAmount = 0;
        let exemptedAmount = 0;
        let zeroRatedAmount = 0;
        
        const vatByRate = {};
        
        // Check if cart exists and has items
        const cartItems = cart.items || cart || [];
        if (!Array.isArray(cartItems) || cartItems.length === 0) {
            console.warn('Invalid cart data in calculateVATWithCustomerStatus');
            return this.getEmptyVATData(customer);
        }
        
        // Get customer VAT status with safe defaults
        const customerVatStatus = customer?.vat_status || 'vatable';
        const isExempted = customerVatStatus === 'exempted';
        const isZeroRated = customerVatStatus === 'zero_rated';
        const isVatable = customerVatStatus === 'vatable' || !customer;
        
        // VAT disclaimer/notes
        let vatNotes = [];
        
        if (isExempted) {
            vatNotes.push('VAT Exempted Customer - No VAT Applied');
        } else if (isZeroRated) {
            vatNotes.push('Zero-Rated Customer - 0% VAT Rate');
        } else if (isVatable) {
            vatNotes.push('VATable Customer - Standard VAT Rates Apply');
        }
        
        // Process each cart item safely
        cartItems.forEach((item) => {
            try {
                // Safely extract values with defaults
                const quantity = parseInt(item.qty || item.quantity || 1);
                const price = parseFloat(item.price || item.unit_price || item.selling_price || 0);
                const priceExTax = parseFloat(item.price_ex_tax || price);
                const itemTotal = priceExTax * quantity;
                
                // Get product-specific tax rate (default 16% if not specified)
                const productTaxRate = parseFloat(item.tax_rate) || 16;
                
                // Check if product has VAT (with multiple fallbacks)
                const isProductVatable = item.has_vat || item.is_vatable || (productTaxRate > 0);
                
                // Skip invalid items
                if (isNaN(quantity) || isNaN(priceExTax) || quantity <= 0) {
                    console.warn('Invalid cart item:', item);
                    return;
                }
                
                subtotal += itemTotal;
                
                if (isExempted) {
                    // VAT exempted customer - no VAT on any items
                    exemptedAmount += itemTotal;
                    nonTaxableAmount += itemTotal;
                    
                } else if (isZeroRated) {
                    // Zero-rated customer - items are taxable but at 0% rate
                    zeroRatedAmount += itemTotal;
                    taxableAmount += itemTotal;
                    
                    // Record as zero-rated
                    if (!vatByRate[0]) {
                        vatByRate[0] = { amount: 0, taxableAmount: 0, type: 'zero_rated' };
                    }
                    vatByRate[0].taxableAmount += itemTotal;
                    
                } else if (isVatable && isProductVatable) {
                    // VATable customer with VATable product
                    const itemVAT = (itemTotal * productTaxRate) / 100;
                    totalVAT += itemVAT;
                    taxableAmount += itemTotal;
                    
                    if (!vatByRate[productTaxRate]) {
                        vatByRate[productTaxRate] = { 
                            amount: 0, 
                            taxableAmount: 0,
                            type: 'standard',
                            rate: productTaxRate 
                        };
                    }
                    vatByRate[productTaxRate].amount += itemVAT;
                    vatByRate[productTaxRate].taxableAmount += itemTotal;
                    
                } else {
                    // Non-VATable product or non-VATable customer
                    nonTaxableAmount += itemTotal;
                }
            } catch (itemError) {
                console.error('Error processing cart item:', itemError, item);
            }
        });
        
        // Apply discount safely
        const discount = parseFloat(cart.discount) || 0;
        
        // Apply discount proportionally
        let discountedVAT = totalVAT;
        let discountedTaxableAmount = taxableAmount;
        
        if (discount > 0 && subtotal > 0) {
            const discountRatio = discount / subtotal;
            discountedVAT = totalVAT * (1 - discountRatio);
            discountedTaxableAmount = taxableAmount * (1 - discountRatio);
        }
        
        const totalWithVAT = Math.max(0, (subtotal - discount) + discountedVAT);
        
        return {
            subtotal: subtotal,
            discount: discount,
            taxableAmount: discountedTaxableAmount,
            nonTaxableAmount: nonTaxableAmount,
            exemptedAmount: exemptedAmount,
            zeroRatedAmount: zeroRatedAmount,
            totalVAT: discountedVAT,
            totalWithVAT: totalWithVAT,
            vatByRate: vatByRate,
            customerVatStatus: customerVatStatus,
            isVatApplicable: !isExempted, // VAT not applicable for exempted
            isZeroRated: isZeroRated,
            isVatable: isVatable,
            isExempted: isExempted,
            vatNotes: vatNotes
        };
    }
    
    getEmptyVATData(customer) {
        return {
            subtotal: 0,
            discount: 0,
            taxableAmount: 0,
            nonTaxableAmount: 0,
            exemptedAmount: 0,
            zeroRatedAmount: 0,
            totalVAT: 0,
            totalWithVAT: 0,
            vatByRate: {},
            customerVatStatus: customer?.vat_status || 'vatable',
            isVatApplicable: true,
            isZeroRated: false,
            isVatable: true,
            isExempted: false,
            vatNotes: ['No items in cart']
        };
    }

    // ========== TRANSACTION MANAGEMENT ==========
    async startTransaction(cartData) {
        if (!cartData || !cartData.items || cartData.items.length === 0) {
            throw new Error('Cart is empty');
        }
        
        // Generate unified numbers for this transaction
        const numbers = await this.generateUnifiedNumber();
        const transactionNumber = this.generateTransactionNumber();
        
        this.currentTransaction = {
            id: transactionNumber,
            cart: cartData,
            status: 'pending',
            steps: { current: 1, total: 3 },
            startedAt: new Date().toISOString(),
            selectedMethod: null,
            paymentData: {},
            metadata: {
                unifiedNumber: numbers.unified,
                invoiceNumber: numbers.invoice, // Same as unified
                receiptNumber: numbers.receipt, // Same as unified
                transactionNumber: transactionNumber,
                numbersSource: numbers.fromServer ? 'server' : 'fallback'
            }
        };
        
        console.log('💰 Transaction started with unified numbers:', {
            unified: numbers.unified,
            invoice: numbers.invoice,
            receipt: numbers.receipt,
            source: numbers.fromServer ? 'server' : 'fallback',
            transaction: transactionNumber
        });
        
        return this.currentTransaction;
    }

    async selectPaymentMethod(methodCode) {
        try {
            if (!this.currentTransaction) {
                const cartData = this.prepareCartData();
                await this.startTransaction(cartData);
            }
            
            const processor = this.getProcessor(methodCode);
            
            this.currentTransaction.selectedMethod = methodCode;
            this.currentTransaction.processor = processor;
            this.activeProcessor = processor;
            
            // Update UI
            this.updateUIForMethod(methodCode);
            
            console.log(`💰 Selected payment method: ${methodCode}`);
            
            return processor;
            
        } catch (error) {
            console.error('Failed to select payment method:', error);
            this.showToast(error.message, 'error');
            throw error;
        }
    }

    // ========== UI MANAGEMENT ==========
    setupUI() {
        this.ui = {
            updateMethodSelection: (methodCode) => {
                document.querySelectorAll('.payment-method-card').forEach(card => {
                    card.classList.remove('selected');
                });
                
                const selectedCard = document.querySelector(`[data-method="${methodCode}"]`);
                if (selectedCard) {
                    selectedCard.classList.add('selected');
                }
            },
            
            updateStep: (step) => {
                document.querySelectorAll('.payment-step').forEach(el => {
                    el.style.display = 'none';
                });
                
                const currentStepElement = document.getElementById(`step${step}`);
                if (currentStepElement) {
                    currentStepElement.style.display = 'block';
                }
                
                document.querySelectorAll('.step-circle').forEach((circle, index) => {
                    circle.classList.remove('active', 'completed');
                    if (index + 1 === step) {
                        circle.classList.add('active');
                    } else if (index + 1 < step) {
                        circle.classList.add('completed');
                    }
                });
                
                const prevBtn = document.getElementById('prevStepBtn');
                const nextBtn = document.getElementById('nextStepBtn');
                const completeBtn = document.getElementById('completePaymentBtn');
                
                if (prevBtn) prevBtn.style.display = step > 1 ? 'inline-block' : 'none';
                if (nextBtn) nextBtn.style.display = step < 3 ? 'inline-block' : 'none';
                if (completeBtn) completeBtn.style.display = step === 3 ? 'inline-block' : 'none';
                
                const nextStepText = document.getElementById('nextStepText');
                if (nextStepText) {
                    nextStepText.textContent = step === 2 ? 'Review' : 'Next';
                }
            },
            
            loadForm: (methodCode) => {
                const container = document.getElementById('paymentFormContainer');
                if (!container) return;
                
                const template = document.getElementById(`${methodCode}FormTemplate`);
                if (!template) {
                    container.innerHTML = `
                        <div class="alert alert-warning">
                            <i class="bx bx-error-circle me-2"></i>
                            Form not available for ${methodCode}
                        </div>
                    `;
                    return;
                }
                
                container.innerHTML = template.innerHTML;
                this.initializeForm(methodCode);
            },
            
            updateSummary: (cart) => {
                if (!cart) return;
                
                const totalElement = document.getElementById('paymentTotalAmount');
                const countElement = document.getElementById('paymentItemCount');
                const customerElement = document.getElementById('paymentCustomer');
                const invoiceElement = document.getElementById('paymentInvoiceNo');
                const receiptElement = document.getElementById('paymentReceiptNo');
                const unifiedElement = document.getElementById('paymentUnifiedNo');
                
                if (totalElement) totalElement.textContent = this.formatCurrency(cart.total);
                if (countElement) countElement.textContent = cart.items?.length || 0;
                if (customerElement) customerElement.textContent = cart.customer?.name || 'Walk-in Customer';
                
                // Update unified number display
                const unifiedNumber = this.currentTransaction?.metadata?.unifiedNumber;
                if (unifiedNumber) {
                    if (invoiceElement) {
                        invoiceElement.textContent = unifiedNumber;
                        invoiceElement.dataset.invoiceNumber = unifiedNumber;
                    }
                    if (receiptElement) {
                        receiptElement.textContent = unifiedNumber;
                        receiptElement.dataset.receiptNumber = unifiedNumber;
                    }
                    if (unifiedElement) {
                        unifiedElement.textContent = unifiedNumber;
                        unifiedElement.dataset.unifiedNumber = unifiedNumber;
                    }
                }
            },
            
            updateUnifiedDisplay: (unifiedNumber) => {
                const invoiceElement = document.getElementById('paymentInvoiceNo');
                const receiptElement = document.getElementById('paymentReceiptNo');
                const unifiedElement = document.getElementById('paymentUnifiedNo');
                
                if (invoiceElement) {
                    invoiceElement.textContent = unifiedNumber;
                    invoiceElement.dataset.invoiceNumber = unifiedNumber;
                }
                
                if (receiptElement) {
                    receiptElement.textContent = unifiedNumber;
                    receiptElement.dataset.receiptNumber = unifiedNumber;
                }
                
                if (unifiedElement) {
                    unifiedElement.textContent = unifiedNumber;
                    unifiedElement.dataset.unifiedNumber = unifiedNumber;
                }
            }
        };
        
        this.setupModalEvents();
    }

    setupModalEvents() {
        const modal = document.getElementById('paymentModal');
        if (!modal) return;
        
        modal.addEventListener('shown.bs.modal', async () => {
            const cart = this.getCurrentCart();
            this.ui.updateSummary(cart);
            this.ui.updateStep(1);
            
            if (!this.currentTransaction) {
                await this.startTransaction(cart);
            }
            
            // Update unified number display
            if (this.currentTransaction?.metadata) {
                const unifiedNumber = this.currentTransaction.metadata.unifiedNumber;
                this.ui.updateUnifiedDisplay(unifiedNumber);
                
                // Also update the main unified number
                this.updateUnifiedNumberUI(unifiedNumber);
            }
        });
        
        modal.addEventListener('hidden.bs.modal', () => {
            this.reset();
        });
        
        const nextBtn = document.getElementById('nextStepBtn');
        const prevBtn = document.getElementById('prevStepBtn');
        const completeBtn = document.getElementById('completePaymentBtn');
        
        if (nextBtn) {
            nextBtn.addEventListener('click', () => this.nextStep());
        }
        
        if (prevBtn) {
            prevBtn.addEventListener('click', () => this.prevStep());
        }
        
        if (completeBtn) {
            completeBtn.addEventListener('click', () => this.completePayment());
        }
    }

    updateUIForMethod(methodCode) {
        if (this.ui) {
            this.ui.updateMethodSelection(methodCode);
            this.ui.loadForm(methodCode);
            
            const nextBtn = document.getElementById('nextStepBtn');
            if (nextBtn) nextBtn.disabled = false;
        }
    }

    initializeForm(methodCode) {
        const cart = this.currentTransaction?.cart;
        if (!cart) return;
        
        const total = cart.total || 0;
        
        switch(methodCode) {
            case 'cash':
                this.initializeCashForm(total);
                break;
            case 'mpesa':
                this.initializeMpesaForm(total);
                break;
            case 'card':
                this.initializeCardForm(total);
                break;
        }
    }

    initializeCashForm(total) {
        const cashTotal = document.getElementById('cashTotalAmount');
        const cashReceived = document.getElementById('cashReceived');
        
        if (cashTotal) cashTotal.value = total.toFixed(2);
        if (cashReceived) {
            cashReceived.value = total.toFixed(2);
            cashReceived.addEventListener('input', () => this.calculateCashChange());
            this.calculateCashChange();
        }
        
        document.querySelectorAll('.quick-amount-btn').forEach(btn => {
            btn.addEventListener('click', () => {
                const amount = parseFloat(btn.dataset.amount);
                this.setQuickCashAmount(amount);
            });
        });
    }

    calculateCashChange() {
        const total = parseFloat(document.getElementById('cashTotalAmount')?.value || 0);
        const received = parseFloat(document.getElementById('cashReceived')?.value || 0);
        
        if (isNaN(received)) return;
        
        const change = received - total;
        const changeDisplay = document.getElementById('cashChangeDisplay');
        const warningDisplay = document.getElementById('cashWarningDisplay');
        
        if (!changeDisplay || !warningDisplay) return;
        
        if (change >= 0) {
            changeDisplay.style.display = 'block';
            warningDisplay.style.display = 'none';
            document.getElementById('changeAmount').textContent = `KES ${change.toFixed(2)}`;
        } else {
            changeDisplay.style.display = 'none';
            warningDisplay.style.display = 'block';
            document.getElementById('insufficientAmount').textContent = `KES ${Math.abs(change).toFixed(2)}`;
        }
    }

    setQuickCashAmount(amount) {
        const total = this.currentTransaction?.cart?.total || 0;
        const finalAmount = Math.max(amount, total);
        const cashReceived = document.getElementById('cashReceived');
        if (cashReceived) {
            cashReceived.value = finalAmount.toFixed(2);
            this.calculateCashChange();
            cashReceived.focus();
        }
    }

    // ========== STEP MANAGEMENT ==========
    nextStep() {
        if (!this.currentTransaction) return;
        
        const currentStep = this.currentTransaction.steps.current;
        
        switch(currentStep) {
            case 1:
                if (!this.currentTransaction.selectedMethod) {
                    this.showToast('Please select a payment method', 'error');
                    return;
                }
                this.updateStep(2);
                break;
                
            case 2:
                this.validateAndProceed().catch(error => {
                    this.showToast(error.message, 'error');
                });
                break;
        }
    }

    prevStep() {
        if (!this.currentTransaction || this.currentTransaction.steps.current <= 1) return;
        this.updateStep(this.currentTransaction.steps.current - 1);
    }

    async validateAndProceed() {
        const formData = this.collectFormData();
        
        if (this.activeProcessor) {
            await this.activeProcessor.validate({
                ...this.currentTransaction.cart,
                ...formData
            });
        }
        
        this.currentTransaction.paymentData = {
            ...this.currentTransaction.paymentData,
            ...formData
        };
        
        this.updateStep(3);
    }

    updateStep(step) {
        if (!this.currentTransaction) return;
        
        this.currentTransaction.steps.current = step;
        
        if (this.ui) {
            this.ui.updateStep(step);
        }
    }

    // ========== PAYMENT PROCESSING ==========
    async completePayment() {
        console.log('💰 Starting completePayment process with unified numbering...');
        
        // Prevent multiple simultaneous payments
        if (this.isProcessing) {
            this.showToast('Payment already in progress', 'warning');
            return;
        }
        
        const spinner = document.getElementById('paymentSpinner');
        const completeBtn = document.getElementById('completePaymentBtn');
        
        try {
            this.isProcessing = true;
            
            // Show loading state
            if (spinner) spinner.classList.remove('d-none');
            if (completeBtn) completeBtn.disabled = true;
            
            // 1. Validate current transaction exists
            if (!this.currentTransaction) {
                throw new Error('No active transaction found. Please start a new sale.');
            }
            
            // 2. Get current cart data with customer
            const cart = this.getCurrentCart();
            if (!cart || !cart.items || cart.items.length === 0) {
                throw new Error('Cart is empty. Add items before completing payment.');
            }
            
            const customer = cart.customer || null;
            
            console.log('Cart and customer data:', { 
                itemCount: cart.items.length,
                customer: customer?.name || 'Walk-in',
                customerVatStatus: customer?.vat_status || 'vatable'
            });
            
            // 3. Get unified number from current transaction
            const unifiedNumber = this.currentTransaction.metadata.unifiedNumber;
            const invoiceNumber = unifiedNumber; // Same as unified
            const receiptNumber = unifiedNumber; // Same as unified
            const transactionNumber = this.currentTransaction.id;
            
            console.log('Using unified numbers:', { 
                unified: unifiedNumber, 
                invoice: invoiceNumber, 
                receipt: receiptNumber,
                transaction: transactionNumber 
            });
            
            // 4. Get form data
            const formData = this.collectFormData();
            
            // 5. Calculate VAT with customer status
            const vatData = this.calculateVATWithCustomerStatus(cart, customer);
            
            console.log('VAT calculation result with customer status:', {
                subtotal: vatData.subtotal,
                vatAmount: vatData.totalVAT,
                total: vatData.totalWithVAT,
                customerStatus: vatData.customerVatStatus,
                isExempted: vatData.isExempted
            });
            
            // 6. Prepare cart items for backend with VAT details
            const cartItemsForBackend = cart.items.map(item => {
                return {
                    product_id: item.id || item.product_id,
                    qty: item.quantity || item.qty || 1,
                    price: item.price,
                    price_ex_tax: item.price_ex_tax || item.price,
                    tax_rate: item.tax_rate || (item.has_vat ? 16 : 0),
                    is_vatable: item.has_vat || item.is_vatable || false,
                    tax_amount: item.vat_amount || 0,
                    total: (item.price * (item.quantity || item.qty || 1))
                };
            });
            
            // 7. Prepare payment data for backend with unified number
            const paymentData = {
                // Unified numbering
                unified_number: unifiedNumber,
                invoice_no: invoiceNumber,
                receipt_no: receiptNumber,
                
                // Customer info
                customer_id: customer?.id || null,
                customer_vat_status: customer?.vat_status || 'vatable',
                user_id: window.authUser?.id || 1,
                
                // Totals
                amount: vatData.totalWithVAT,
                subtotal: vatData.subtotal,
                tax_total: vatData.totalVAT,
                discount: vatData.discount,
                grand_total: vatData.totalWithVAT,
                
                // Payment info
                payment_method: this.currentTransaction.selectedMethod,
                payment_status: 'paid',
                status: 'completed',
                transaction_id: transactionNumber,
                cart: cartItemsForBackend,
                
                // VAT breakdown fields
                taxable_amount: vatData.taxableAmount,
                non_taxable_amount: vatData.nonTaxableAmount,
                exempted_amount: vatData.exemptedAmount,
                zero_rated_amount: vatData.zeroRatedAmount,
                vat_amount: vatData.totalVAT,
                is_vat_applied: vatData.isVatApplicable,
                zero_rated: vatData.isZeroRated,
                exempted: vatData.isExempted,
                vat_breakdown: vatData.vatByRate
            };
            
            // Add payment method specific data
            const method = this.currentTransaction.selectedMethod;
            if (method === 'cash') {
                paymentData.cash_received = formData.cash_received || vatData.totalWithVAT;
                paymentData.customer_change = Math.max(0, paymentData.cash_received - vatData.totalWithVAT);
            } else if (method === 'mpesa') {
                paymentData.mpesa_received = vatData.totalWithVAT;
                paymentData.mpesa_phone = formData.mpesa_phone || customer?.phone;
                paymentData.mpesa_transaction_id = formData.mpesa_transaction_id || `MPESA_${Date.now()}`;
            }
            
            console.log('Sending payment to backend with unified number:', paymentData);
            
            // 8. Send to backend
            const response = await fetch('/api/payments/process', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(paymentData)
            });
            
            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.message || `Payment failed with status ${response.status}`);
            }
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'Payment processing failed');
            }
            
            console.log('✅ Payment successful with unified number:', result);
            
            // 9. Store receipt data for reprinting
            const receiptData = {
                unified_number: unifiedNumber,
                invoice_no: invoiceNumber,
                receipt_no: receiptNumber,
                transaction_id: transactionNumber,
                date: new Date().toLocaleDateString('en-KE'),
                time: new Date().toLocaleTimeString('en-KE'),
                customer: customer ? {
                    name: customer.name,
                    phone: customer.phone,
                    vat_status: customer.vat_status
                } : null,
                items: cart.items.map(item => ({
                    name: item.name,
                    quantity: item.quantity || item.qty || 1,
                    unit_price: item.price,
                    total: item.price * (item.quantity || item.qty || 1)
                })),
                totals: {
                    subtotal: vatData.subtotal,
                    discount: vatData.discount,
                    vat_amount: vatData.totalVAT,
                    grand_total: vatData.totalWithVAT
                },
                payment: {
                    method: this.currentTransaction.selectedMethod,
                    amount_paid: vatData.totalWithVAT,
                    ...(method === 'cash' ? {
                        cash_received: paymentData.cash_received,
                        change: paymentData.customer_change
                    } : {})
                },
                store: {
                    name: window.appConfig?.name || 'Kenyan Supermarket',
                    address: window.appConfig?.address || 'Nairobi CBD',
                    phone: window.appConfig?.phone || '0700 000 000',
                    vat_pin: window.appConfig?.vat_pin || ''
                },
                cashier: window.authUser?.name || 'Cashier',
                notes: [
                    '🎯 Unified Invoice & Receipt Number',
                    '✓ Invoice and Receipt numbers match',
                    'Thank you for your business!'
                ]
            };
            
            localStorage.setItem('pos_last_receipt_data', JSON.stringify(receiptData));
            localStorage.setItem('pos_last_completed_sale', Date.now().toString());
            localStorage.setItem('pos_last_unified_number', unifiedNumber);
            
            // 10. Show success message
            this.showToast(`Payment completed! Unified Number: ${unifiedNumber}`, 'success');
            
            // 11. Clear cart SAFELY
            await this.clearCartSafely();
            
            // 12. Open receipt for printing
            setTimeout(() => {
                const receiptUrl = `/pos/receipt/print/${unifiedNumber}?autoprint=true`;
                const receiptWindow = window.open(receiptUrl, '_blank', 'width=800,height=600');
                
                // Close modal
                this.hideModal();
                this.reset();
                
                // Ask if they want to print again
                setTimeout(() => {
                    if (receiptWindow && !receiptWindow.closed && confirm('Print another copy?')) {
                        receiptWindow.print();
                    }
                }, 2000);
            }, 1500);
            
            // 13. Generate next unified number for next transaction
            setTimeout(async () => {
                try {
                    const nextNumbers = await this.generateUnifiedNumber();
                    console.log('🔄 Prepared next unified number:', nextNumbers.unified);
                } catch (error) {
                    console.warn('Failed to prepare next unified number:', error);
                }
            }, 100);
            
        } catch (error) {
            console.error('❌ Payment error:', error);
            
            // Show error message
            let errorMessage = error.message || 'Payment failed';
            
            // Make backend errors more user-friendly
            if (errorMessage.includes('Validation failed')) {
                errorMessage = 'Please check your payment information and try again.';
            } else if (errorMessage.includes('SQLSTATE') || errorMessage.includes('database')) {
                errorMessage = 'Database error. Please contact support.';
            } else if (errorMessage.includes('change')) {
                errorMessage = 'Please check the cash received amount.';
            } else if (errorMessage.includes('unified_number')) {
                errorMessage = 'Number generation error. Please try again.';
            }
            
            this.showToast(errorMessage, 'error');
            
        } finally {
            // Reset processing state
            this.isProcessing = false;
            
            // Reset UI state
            if (spinner) spinner.classList.add('d-none');
            if (completeBtn) completeBtn.disabled = false;
        }
    }
    
    /**
     * Clear cart safely after payment
     */
    async clearCartSafely() {
        console.log('🧹 Clearing cart after payment...');
        
        try {
            // Method 1: Use posState if available
            if (window.posState && typeof window.posState.clearCart === 'function') {
                window.posState.clearCart();
            } 
            // Method 2: Use POSCart if available
            else if (window.POSCart && typeof window.POSCart.clear === 'function') {
                window.POSCart.clear();
            }
            // Method 3: Manually clear cart
            else if (window.posState && Array.isArray(window.posState.cart)) {
                window.posState.cart = [];
                window.posState.currentDiscount = 0;
                window.posState.currentCustomer = null;
                window.posState.cartVATData = null;
                
                // Update display if possible
                if (window.POSCart && typeof window.POSCart.updateDisplay === 'function') {
                    window.POSCart.updateDisplay();
                }
            }
            
            // Clear localStorage cart data but keep unified number
            localStorage.removeItem('pos_cart');
            localStorage.removeItem('pos_current_customer');
            localStorage.removeItem('pos_current_discount');
            localStorage.setItem('pos_last_completed_sale', Date.now().toString());
            
            console.log('✅ Cart cleared successfully');
            
        } catch (error) {
            console.error('❌ Error clearing cart:', error);
            // Don't throw, just log
        }
    }

    // ========== FORM DATA COLLECTION ==========
    collectFormData() {
        if (!this.currentTransaction?.selectedMethod) {
            throw new Error('No payment method selected');
        }
        
        const method = this.currentTransaction.selectedMethod;
        const data = {};
        
        switch(method) {
            case 'cash':
                data.cash_received = parseFloat(document.getElementById('cashReceived')?.value || 0);
                data.notes = document.getElementById('cashNotes')?.value || '';
                break;
                
            case 'mpesa':
                data.mpesa_phone = document.getElementById('mpesaPhone')?.value || '';
                data.mpesa_type = document.querySelector('input[name="mpesaType"]:checked')?.value || 'stk';
                data.mpesa_transaction_id = document.getElementById('mpesaTransactionId')?.value || '';
                data.mpesa_receipt_no = document.getElementById('mpesaReceiptNo')?.value || '';
                break;
                
            case 'card':
                data.card_number = document.getElementById('cardNumber')?.value || '';
                data.card_expiry = document.getElementById('cardExpiry')?.value || '';
                data.card_cvv = document.getElementById('cardCvv')?.value || '';
                data.card_holder = document.getElementById('cardHolder')?.value || '';
                break;
                
            case 'credit':
                data.customer_id = document.getElementById('creditCustomerSelect')?.value || '';
                data.terms = document.getElementById('creditTerms')?.value || '30';
                data.due_date = document.getElementById('creditDueDate')?.value || '';
                break;
                
            case 'multiple':
                data.splits = {
                    cash: parseFloat(document.getElementById('splitCash')?.value || 0),
                    mpesa: parseFloat(document.getElementById('splitMpesa')?.value || 0),
                    card: parseFloat(document.getElementById('splitCard')?.value || 0)
                };
                break;
                
            case 'bank':
                data.bank_reference = document.getElementById('bankReference')?.value || '';
                data.bank_name = document.getElementById('bankName')?.value || '';
                break;
        }
        
        return data;
    }

    // ========== PUBLIC API ==========
    showModal() {
        if (!this.initialized) {
            this.showToast('Payment system initializing...', 'warning');
            return false;
        }
        
        const cart = this.getCurrentCart();
        if (!cart.items || cart.items.length === 0) {
            this.showToast('Add items to cart first', 'warning');
            return false;
        }
        
        try {
            const modal = document.getElementById('paymentModal');
            if (modal) {
                const bsModal = new bootstrap.Modal(modal);
                bsModal.show();
                return true;
            }
        } catch (error) {
            console.error('Error showing payment modal:', error);
            this.showToast('Cannot open payment modal', 'error');
        }
        
        return false;
    }

    hideModal() {
        const modal = document.getElementById('paymentModal');
        if (modal) {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
        }
    }

    // ========== HELPER METHODS ==========
    getCurrentCart() {
        if (window.posState) {
            return {
                items: window.posState.cart || [],
                subtotal: window.posState.getCartSubtotal ? window.posState.getCartSubtotal() : 0,
                discount: window.posState.currentDiscount || 0,
                total: window.posState.getCartTotal ? window.posState.getCartTotal() : 0,
                customer: window.posState.currentCustomer || null
            };
        }
        
        return {
            items: [],
            subtotal: 0,
            discount: 0,
            total: 0,
            customer: null
        };
    }

    prepareCartData() {
        return this.getCurrentCart();
    }

    formatCurrency(amount) {
        return `KES ${parseFloat(amount).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
    }

    showToast(message, type = 'info') {
        if (window.POSUtils?.showToast) {
            window.POSUtils.showToast(message, type);
        } else {
            console.log(`[${type}] ${message}`);
            const alertDiv = document.createElement('div');
            alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed top-0 end-0 m-3`;
            alertDiv.style.zIndex = '9999';
            alertDiv.innerHTML = `
                <strong>${type.toUpperCase()}:</strong> ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(alertDiv);
            setTimeout(() => alertDiv.remove(), 5000);
        }
    }

    // ========== CONFIG & STATE ==========
    loadConfig() {
        return {
            autoSaveInterval: 30000,
            maxHistoryItems: 100,
            autoPrintReceipt: true,
            autoSubmitETIMS: false,
            useUnifiedNumbering: true,
            unifiedNumberPrefix: 'INVRCP'
        };
    }

    saveTransactionToHistory() {
        if (!this.currentTransaction) return;
        
        this.transactionHistory.unshift(this.currentTransaction);
        
        if (this.transactionHistory.length > this.config.maxHistoryItems) {
            this.transactionHistory = this.transactionHistory.slice(0, this.config.maxHistoryItems);
        }
        
        localStorage.setItem('pos_payment_history', JSON.stringify(this.transactionHistory));
    }

    loadTransactionHistory() {
        try {
            const history = JSON.parse(localStorage.getItem('pos_payment_history') || '[]');
            this.transactionHistory = history;
        } catch (error) {
            console.warn('Failed to load transaction history:', error);
            this.transactionHistory = [];
        }
    }

    setupAutoSave() {
        setInterval(() => {
            if (this.currentTransaction) {
                localStorage.setItem('pos_current_payment_transaction', JSON.stringify(this.currentTransaction));
            }
        }, this.config.autoSaveInterval);
    }

    // ========== EVENT SYSTEM ==========
    createEventSystem() {
        const events = new Map();
        
        return {
            on: (event, callback) => {
                if (!events.has(event)) {
                    events.set(event, []);
                }
                events.get(event).push(callback);
            },
            off: (event, callback) => {
                if (events.has(event)) {
                    const callbacks = events.get(event);
                    const index = callbacks.indexOf(callback);
                    if (index > -1) {
                        callbacks.splice(index, 1);
                    }
                }
            },
            trigger: (event, data) => {
                if (events.has(event)) {
                    events.get(event).forEach(callback => {
                        try {
                            callback(data);
                        } catch (error) {
                            console.error(`Error in ${event} handler:`, error);
                        }
                    });
                }
            }
        };
    }

    on(event, callback) {
        this.events.on(event, callback);
    }

    off(event, callback) {
        this.events.off(event, callback);
    }

    trigger(event, data) {
        this.events.trigger(event, data);
    }

    // ========== ERROR HANDLING ==========
    handleFatalError(error) {
        console.error('[PaymentManager] Fatal error:', error);
        
        this.processors.clear();
        this.currentTransaction = null;
        this.initialized = false;
        
        // Setup minimal functional methods
        this.showModal = function() {
            const modal = document.getElementById('paymentModal');
            if (modal) {
                try {
                    new bootstrap.Modal(modal).show();
                } catch (e) {
                    console.error('Cannot show payment modal:', e);
                }
            }
        };
        
        this.showToast('Payment system loaded with limited functionality', 'warning');
    }

    reset() {
        this.currentTransaction = null;
        this.activeProcessor = null;
        this.isProcessing = false;
        
        if (this.ui) {
            this.ui.updateStep(1);
            
            const container = document.getElementById('paymentFormContainer');
            if (container) {
                container.innerHTML = `
                    <div class="alert alert-info text-center py-5">
                        <i class="bx bx-credit-card fs-1 mb-3"></i>
                        <p>Please select a payment method to continue</p>
                    </div>
                `;
            }
            
            document.querySelectorAll('.payment-method-card').forEach(card => {
                card.classList.remove('selected');
            });
        }
        
        localStorage.removeItem('pos_current_payment_transaction');
    }
}

// ========== GLOBAL INITIALIZATION ==========

let paymentManagerInstance = null;

function initializePaymentManager() {
    if (!paymentManagerInstance) {
        paymentManagerInstance = new PaymentManager();
        
        const initTimeout = setTimeout(() => {
            console.warn('Payment Manager initialization timeout');
            paymentManagerInstance.initialized = true;
        }, 10000); // 10 second timeout
        
        paymentManagerInstance.init()
            .then(() => {
                clearTimeout(initTimeout);
                console.log('💰 Payment Manager with Unified Numbering ready v5.0.0');
            })
            .catch(error => {
                clearTimeout(initTimeout);
                console.error('Payment Manager init error:', error);
                paymentManagerInstance.handleFatalError(error);
            });
    }
    
    return paymentManagerInstance;
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        setTimeout(() => {
            window.PaymentManager = initializePaymentManager();
        }, 1000);
    });
} else {
    setTimeout(() => {
        window.PaymentManager = initializePaymentManager();
    }, 1000);
}

// Global helper functions (backup)
if (!window.showPaymentModal) {
    window.showPaymentModal = function() {
        if (window.PaymentManager && window.PaymentManager.showModal) {
            return window.PaymentManager.showModal();
        } else {
            const modal = document.getElementById('paymentModal');
            if (modal) {
                try {
                    new bootstrap.Modal(modal).show();
                    return true;
                } catch (e) {
                    console.error('Cannot show payment modal:', e);
                    return false;
                }
            }
        }
        return false;
    };
}

if (!window.selectPaymentMethod) {
    window.selectPaymentMethod = function(method) {
        if (window.PaymentManager && window.PaymentManager.selectPaymentMethod) {
            return window.PaymentManager.selectPaymentMethod(method).catch(() => {
                document.querySelectorAll('.payment-method-card').forEach(card => {
                    card.classList.remove('selected');
                });
                
                const selectedCard = document.querySelector(`[data-method="${method}"]`);
                if (selectedCard) {
                    selectedCard.classList.add('selected');
                }
            });
        } else {
            document.querySelectorAll('.payment-method-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            const selectedCard = document.querySelector(`[data-method="${method}"]`);
            if (selectedCard) {
                selectedCard.classList.add('selected');
            }
        }
    };
}

// Add unified number generator to window
if (!window.generateUnifiedNumber) {
    window.generateUnifiedNumber = async function() {
        if (window.PaymentManager && window.PaymentManager.generateUnifiedNumber) {
            return window.PaymentManager.generateUnifiedNumber();
        } else {
            // Simple fallback
            const today = new Date();
            const year = today.getFullYear();
            const month = String(today.getMonth() + 1).padStart(2, '0');
            const day = String(today.getDate()).padStart(2, '0');
            const dateStr = `${year}${month}${day}`;
            
            let counter = parseInt(localStorage.getItem('pos_unified_counter') || '0') + 1;
            localStorage.setItem('pos_unified_counter', counter);
            
            return `INVRCP-${dateStr}-${String(counter).padStart(4, '0')}`;
        }
    };
}

console.log('✅ Payment Manager module loaded v5.0.0 with Unified Numbering');