// POS Held Sales Management Module
// =========================================================

class POSHeldSales {
    static init() {
        // Setup event listeners
        const heldSalesModal = document.getElementById('heldSalesModal');
        if (heldSalesModal) {
            heldSalesModal.addEventListener('shown.bs.modal', () => {
                this.load();
                this.loadStats();
            });
        }
    }
    
    static async holdCurrent() {
        console.log('⏸️ Hold current sale called');
        
        if (window.posState.cart.length === 0) {
            POSUtils.showToast('Cart is empty', 'error');
            return false;
        }
        
        try {
            const holdName = prompt('Enter a name for this hold (optional):') || 
                            `Hold ${new Date().toLocaleTimeString()}`;
            
            if (holdName === null) return false;
            
            const holdData = {
                name: holdName,
                items: window.posState.cart.map(item => ({
                    product_id: item.id,
                    name: item.name,
                    sku: item.sku,
                    price: item.price,
                    quantity: item.quantity,
                    tax_rate: item.tax_rate,
                    is_vatable: item.is_vatable
                })),
                customer_id: window.posState.currentCustomer ? window.posState.currentCustomer.id : null,
                subtotal: window.posState.getCartSubtotal(),
                vat: window.posState.getCartVAT(),
                discount: window.posState.currentDiscount,
                total: window.posState.getCartTotal(),
                notes: document.getElementById('saleNotes')?.value || ''
            };
            
            POSUtils.showLoading(true);
            
            const response = await fetch('/api/hold-sales', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content
                },
                body: JSON.stringify(holdData)
            });
            
            const data = await response.json();
            POSUtils.showLoading(false);
            
            if (data.success) {
                // Clear current cart
                window.posState.clearCart();
                POSCart.updateDisplay();
                
                POSUtils.showToast(`✅ Sale held successfully: ${holdName}`, 'success');
                
                // Refresh if modal is open
                if (document.getElementById('heldSalesModal')?.classList.contains('show')) {
                    await this.load();
                }
                
                return true;
            } else {
                POSUtils.showToast(data.error || 'Failed to hold sale', 'error');
                return false;
            }
        } catch (error) {
            POSUtils.showLoading(false);
            console.error('Hold sale error:', error);
            POSUtils.showToast('Failed to hold sale', 'error');
            return false;
        }
    }
    
    static async load() {
        try {
            POSUtils.showLoading(true, 'heldSalesModal');
            
            const response = await fetch('/api/hold-sales');
            const data = await response.json();
            
            POSUtils.showLoading(false, 'heldSalesModal');
            
            if (data.success) {
                this.renderTable(data.holds || data.data || []);
            } else {
                POSUtils.showToast(data.error || 'Failed to load held sales', 'error');
                this.renderTable([]);
            }
        } catch (error) {
            POSUtils.showLoading(false, 'heldSalesModal');
            console.error('Load held sales error:', error);
            POSUtils.showToast('Failed to load held sales', 'error');
            this.renderTable([]);
        }
    }
    
    static renderTable(holds) {
        const tbody = document.getElementById('heldTableBody');
        if (!tbody) return;
        
        if (!holds || holds.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-5">
                        <i class="bx bx-package fs-1 text-muted mb-3"></i>
                        <p class="text-muted">No held sales found</p>
                        <p class="small text-muted">Hold sales will appear here</p>
                    </td>
                </tr>
            `;
            return;
        }
        
        tbody.innerHTML = '';
        
        holds.forEach(hold => {
            try {
                const isExpired = hold.status === 'expired' || 
                                (hold.expires_at && new Date(hold.expires_at) < new Date());
                const isActive = hold.status === 'active' && !isExpired;
                
                let statusClass = 'bg-secondary';
                let statusText = hold.status || 'unknown';
                
                if (isExpired) {
                    statusClass = 'bg-danger';
                    statusText = 'Expired';
                } else if (isActive) {
                    statusClass = 'bg-success';
                    statusText = 'Active';
                } else if (hold.status === 'converted') {
                    statusClass = 'bg-info';
                    statusText = 'Converted';
                }
                
                const createdDate = hold.created_at ? new Date(hold.created_at) : new Date();
                const expiresDate = hold.expires_at ? new Date(hold.expires_at) : null;
                
                let timeRemaining = 'N/A';
                if (expiresDate && isActive) {
                    const diffMs = expiresDate - new Date();
                    if (diffMs > 0) {
                        const diffHours = Math.floor(diffMs / (1000 * 60 * 60));
                        const diffMinutes = Math.floor((diffMs % (1000 * 60 * 60)) / (1000 * 60));
                        timeRemaining = diffHours > 0 ? `${diffHours}h ${diffMinutes}m` : `${diffMinutes}m`;
                    } else {
                        timeRemaining = 'Expired';
                    }
                }
                
                const row = document.createElement('tr');
                row.dataset.holdId = hold.id;
                
                row.innerHTML = `
                    <td>
                        <div class="fw-medium">${hold.name || 'Unnamed Hold'}</div>
                        <div class="small text-muted">${hold.reference || 'REF-' + hold.id}</div>
                    </td>
                    <td>
                        ${hold.customer_name || 'Walk-in'}
                    </td>
                    <td class="text-center">
                        <span class="badge bg-secondary">${hold.item_count || 0} items</span>
                    </td>
                    <td class="text-end fw-bold">
                        KES ${parseFloat(hold.total || 0).toFixed(2)}
                    </td>
                    <td>
                        <span class="badge ${statusClass}">${statusText}</span>
                    </td>
                    <td>
                        <div class="small">${createdDate.toLocaleDateString()}</div>
                        <div class="small text-muted">${createdDate.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}</div>
                    </td>
                    <td>
                        ${hold.held_by_name || 'System'}
                    </td>
                    <td class="${isExpired ? 'text-danger' : 'text-success'}">
                        ${timeRemaining}
                    </td>
                    <td>
                        <div class="btn-group btn-group-sm">
                            <button class="btn btn-outline-primary" 
                                    onclick="POSHeldSales.loadHold('${hold.id}')"
                                    ${!isActive ? 'disabled' : ''}
                                    title="${isActive ? 'Load into cart' : 'Cannot load expired/converted hold'}">
                                <i class="bx bx-upload"></i>
                            </button>
                            <button class="btn btn-outline-secondary" 
                                    onclick="POSHeldSales.viewDetails('${hold.id}')"
                                    title="View details">
                                <i class="bx bx-show"></i>
                            </button>
                            <button class="btn btn-outline-danger" 
                                    onclick="POSHeldSales.delete('${hold.id}')"
                                    title="Delete">
                                <i class="bx bx-trash"></i>
                            </button>
                        </div>
                    </td>
                `;
                
                tbody.appendChild(row);
            } catch (error) {
                console.error('Error rendering hold row:', error, hold);
            }
        });
    }
    
    static async loadHold(holdId) {
        if (!confirm('Load this hold sale into cart? Current cart will be replaced.')) {
            return;
        }
        
        try {
            POSUtils.showLoading(true);
            
            const response = await fetch(`/api/hold-sales/${holdId}/load`);
            const data = await response.json();
            
            POSUtils.showLoading(false);
            
            if (data.success) {
                // Clear current cart and load hold items
                window.posState.cart = [];
                const holdItems = data.items || data.data || [];
                holdItems.forEach(item => {
                    window.posState.cart.push({
                        id: item.product_id,
                        name: item.name,
                        sku: item.sku,
                        barcode: item.barcode,
                        price: item.price,
                        cost_price: item.cost_price,
                        quantity: item.quantity,
                        tax_rate: item.tax_rate,
                        is_vatable: item.is_vatable,
                        stock: item.stock
                    });
                });
                
                window.posState.currentDiscount = data.discount || 0;
                
                if (data.customer) {
                    POSCustomers.setCurrent(data.customer);
                }
                
                POSCart.updateDisplay();
                
                // Close modal
                const modalElement = document.getElementById('heldSalesModal');
                if (modalElement) {
                    const modal = bootstrap.Modal.getInstance(modalElement);
                    if (modal) modal.hide();
                }
                
                POSUtils.showToast(`✅ Hold sale loaded: ${data.name || data.reference}`, 'success');
            } else {
                POSUtils.showToast(data.error || 'Failed to load hold sale', 'error');
            }
        } catch (error) {
            POSUtils.showLoading(false);
            console.error('Load hold sale error:', error);
            POSUtils.showToast('Failed to load hold sale', 'error');
        }
    }
    
    static async delete(holdId) {
        if (!confirm('Are you sure you want to delete this hold sale? This action cannot be undone.')) {
            return;
        }
        
        try {
            POSUtils.showLoading(true, 'heldSalesModal');
            
            const response = await fetch(`/api/hold-sales/${holdId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content
                }
            });
            
            const data = await response.json();
            POSUtils.showLoading(false, 'heldSalesModal');
            
            if (data.success) {
                POSUtils.showToast('✅ Hold sale deleted', 'success');
                await this.load();
            } else {
                POSUtils.showToast(data.error || 'Failed to delete hold sale', 'error');
            }
        } catch (error) {
            POSUtils.showLoading(false, 'heldSalesModal');
            console.error('Delete hold sale error:', error);
            POSUtils.showToast('Failed to delete hold sale', 'error');
        }
    }
    
    static async loadStats() {
        try {
            const response = await fetch('/api/hold-sales/stats');
            const data = await response.json();
            
            if (data.success) {
                const stats = data.stats || data.data || {};
                
                const elements = {
                    'activeHolds': stats.active_holds || 0,
                    'totalHeldValue': `KES ${(stats.total_value || 0).toFixed(2)}`,
                    'expiringSoon': stats.expiring_soon || 0,
                    'todaysHolds': stats.todays_holds || 0
                };
                
                for (const [id, value] of Object.entries(elements)) {
                    const element = document.getElementById(id);
                    if (element) element.textContent = value;
                }
            }
        } catch (error) {
            console.error('Load held stats error:', error);
        }
    }
    
    static async viewDetails(holdId) {
        try {
            const response = await fetch(`/api/hold-sales/${holdId}`);
            const data = await response.json();
            
            if (data.success) {
                this.showDetailsModal(data.hold || data.data);
            } else {
                POSUtils.showToast(data.error || 'Failed to load details', 'error');
            }
        } catch (error) {
            console.error('View details error:', error);
            POSUtils.showToast('Failed to load details', 'error');
        }
    }
    
    static showDetailsModal(hold) {
        // Implementation for showing modal with hold details
        // Similar to original viewHoldDetails function
    }
    
    static showModal() {
        try {
            const modal = new bootstrap.Modal(document.getElementById('heldSalesModal'));
            modal.show();
        } catch (error) {
            console.error('Error showing held sales modal:', error);
            POSUtils.showToast('Failed to open held sales', 'error');
        }
    }
}

// Export for global access
window.POSHeldSales = POSHeldSales;
window.showHeldSalesModal = () => POSHeldSales.showModal();
window.holdCurrentSale = () => POSHeldSales.holdCurrent();
window.loadHeldSales = () => POSHeldSales.load();
window.loadHoldSale = (holdId) => POSHeldSales.loadHold(holdId);
window.deleteHoldSale = (holdId) => POSHeldSales.delete(holdId);
window.viewHoldDetails = (holdId) => POSHeldSales.viewDetails(holdId);