// public/js/pos/modules/customers.js
// Customer Management Module - Enhanced for Laravel Backend with VAT Integration
// =========================================================

class POSCustomers {
    constructor() {
        this.customers = [];
        this.selectedCustomer = null;
        this.isLoading = false;
        this.currentPage = 1;
        this.totalPages = 1;
        this.searchQuery = '';
        this.customerFilter = 'all';
        this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
        this.baseUrl = window.location.origin;
    }

    init() {
        this.loadCustomers();
        this.setupEventListeners();
        
        // Check for stored customer
        if (window.posState && window.posState.currentCustomer) {
            this.setCurrent(window.posState.currentCustomer);
        }
    }

    // Setup event listeners
    setupEventListeners() {
        // Customer search
        const customerSearch = document.getElementById('customerSearch');
        if (customerSearch) {
            customerSearch.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    this.searchCustomers();
                }
            });
            
            // Debounce search
            let timeout;
            customerSearch.addEventListener('input', (e) => {
                clearTimeout(timeout);
                timeout = setTimeout(() => {
                    this.searchCustomers();
                }, 500);
            });
        }

        // Customer filter
        const typeFilter = document.getElementById('customerTypeFilter');
        if (typeFilter) {
            typeFilter.addEventListener('change', () => {
                this.customerFilter = typeFilter.value;
                this.currentPage = 1;
                this.loadCustomers();
            });
        }

        // Customer modal events
        const customerModal = document.getElementById('customerModal');
        if (customerModal) {
            customerModal.addEventListener('shown.bs.modal', () => {
                customerSearch?.focus();
                this.loadCustomers();
                this.loadStats();
            });

            customerModal.addEventListener('hidden.bs.modal', () => {
                this.searchQuery = '';
                if (customerSearch) customerSearch.value = '';
                this.customerFilter = 'all';
                if (typeFilter) typeFilter.value = 'all';
            });
        }
        
        // Keyboard shortcuts for customer selection
        document.addEventListener('keydown', (e) => {
            if (e.key === 'F2' && !e.target.matches('input, textarea, select')) {
                e.preventDefault();
                this.showModal();
            }
        });
    }

    // Load customers from Laravel API
    async loadCustomers() {
        if (this.isLoading) return;
        
        this.isLoading = true;
        this.showLoading(true);
        
        try {
            const params = new URLSearchParams({
                search: this.searchQuery,
                type: this.customerFilter,
                page: this.currentPage,
                with_vat_status: true
            });
            
            const response = await fetch(`${this.baseUrl}/api/customers?${params}`, {
                headers: {
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                this.customers = data.customers || data.data || [];
                this.totalPages = data.last_page || data.total_pages || 1;
                this.renderCustomerTable();
            } else {
                console.error('Failed to load customers:', response.status);
                // Show fallback customers
                this.showFallbackCustomers();
            }
        } catch (error) {
            console.error('Error loading customers:', error);
            this.showFallbackCustomers();
        } finally {
            this.isLoading = false;
            this.showLoading(false);
        }
    }

    // Load customer statistics
    async loadStats() {
        try {
            // Change this from /api/customers/stats to /api/customers?stats=true
            const response = await fetch(`${this.baseUrl}/api/customers?stats=true`, {
                headers: {
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.updateStats(data);
                }
            } else {
                // Try the stats endpoint directly
                const altResponse = await fetch(`${this.baseUrl}/api/customers/stats`);
                if (altResponse.ok) {
                    const altData = await altResponse.json();
                    this.updateStats(altData);
                }
            }
        } catch (error) {
            console.error('Error loading stats:', error);
            this.updateStats({
                total_customers: 0,
                active_today: 0,
                total_credit: 0,
                avg_sale: 0
            });
        }
    }

    // Search customers
    searchCustomers() {
        const searchInput = document.getElementById('customerSearch');
        if (searchInput) {
            this.searchQuery = searchInput.value.trim();
            this.currentPage = 1;
            this.loadCustomers();
        }
    }

    // Show/hide loading state
    showLoading(show) {
        const tableBody = document.getElementById('customerTableBody');
        if (!tableBody) return;

        if (show) {
            tableBody.innerHTML = `
                <tr>
                    <td colspan="6" class="text-center py-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading customers...</span>
                        </div>
                        <p class="mt-2 text-muted">Loading customers...</p>
                    </td>
                </tr>
            `;
        }
    }

    // Render customer table
    renderCustomerTable() {
        const tableBody = document.getElementById('customerTableBody');
        if (!tableBody) return;

        if (this.customers.length === 0) {
            tableBody.innerHTML = `
                <tr>
                    <td colspan="6" class="text-center py-5">
                        <div class="text-muted">
                            <i class="bx bx-user-x fs-1 mb-2"></i>
                            <p>No customers found</p>
                            ${this.searchQuery ? '<small>Try a different search term</small>' : ''}
                        </div>
                    </td>
                </tr>
            `;
            return;
        }

        tableBody.innerHTML = this.customers.map(customer => this.getCustomerRowHtml(customer)).join('');
        
        // Add pagination if needed
        if (this.totalPages > 1) {
            this.renderPagination();
        }
    }

    // Render pagination controls
    renderPagination() {
        const tableBody = document.getElementById('customerTableBody');
        if (!tableBody || this.totalPages <= 1) return;

        let paginationHtml = `
            <tr>
                <td colspan="6">
                    <nav aria-label="Customer pagination">
                        <ul class="pagination justify-content-center pagination-sm mb-0">
        `;

        // Previous button
        paginationHtml += `
            <li class="page-item ${this.currentPage === 1 ? 'disabled' : ''}">
                <a class="page-link" href="#" onclick="window.POSCustomers.goToPage(${this.currentPage - 1})">
                    &laquo;
                </a>
            </li>
        `;

        // Page numbers
        const maxVisiblePages = 5;
        let startPage = Math.max(1, this.currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(this.totalPages, startPage + maxVisiblePages - 1);

        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        for (let i = startPage; i <= endPage; i++) {
            paginationHtml += `
                <li class="page-item ${i === this.currentPage ? 'active' : ''}">
                    <a class="page-link" href="#" onclick="window.POSCustomers.goToPage(${i})">
                        ${i}
                    </a>
                </li>
            `;
        }

        // Next button
        paginationHtml += `
            <li class="page-item ${this.currentPage === this.totalPages ? 'disabled' : ''}">
                <a class="page-link" href="#" onclick="window.POSCustomers.goToPage(${this.currentPage + 1})">
                    &raquo;
                </a>
            </li>
        `;

        paginationHtml += `
                        </ul>
                        <div class="text-center small text-muted mt-1">
                            Page ${this.currentPage} of ${this.totalPages}
                        </div>
                    </nav>
                </td>
            </tr>
        `;

        tableBody.insertAdjacentHTML('beforeend', paginationHtml);
    }

    // Go to specific page
    goToPage(page) {
        if (page >= 1 && page <= this.totalPages && page !== this.currentPage) {
            this.currentPage = page;
            this.loadCustomers();
        }
    }

    // Get customer row HTML
    getCustomerRowHtml(customer) {
        const isSelected = this.selectedCustomer?.id === customer.id;
        const initials = this.getInitials(customer.name || 'Walk-in Customer');
        const typeBadge = this.getTypeBadge(customer.customer_type || customer.type || 'walk-in');
        const creditInfo = this.getCreditInfo(customer);
        const lastPurchase = customer.last_purchase_date 
            ? new Date(customer.last_purchase_date).toLocaleDateString()
            : customer.last_purchase 
                ? new Date(customer.last_purchase).toLocaleDateString()
                : 'Never';
        
        // Add VAT status indicator
        let vatBadge = '';
        if (customer.vat_status && customer.vat_status !== 'vatable') {
            let vatBadgeClass = '';
            let vatBadgeText = '';
            
            switch(customer.vat_status) {
                case 'exempted':
                    vatBadgeClass = 'badge bg-danger';
                    vatBadgeText = 'VAT Exempted';
                    break;
                case 'zero_rated':
                    vatBadgeClass = 'badge bg-warning text-dark';
                    vatBadgeText = 'Zero-Rated';
                    break;
            }
            
            if (vatBadgeClass) {
                vatBadge = `<span class="${vatBadgeClass} ms-1" style="font-size: 0.7em;">${vatBadgeText}</span>`;
            }
        }

        return `
            <tr class="${isSelected ? 'selected' : ''}" 
                onclick="window.POSCustomers.selectCustomer(${customer.id})"
                style="cursor: pointer;">
                <td data-label="Customer">
                    <div class="customer-info">
                        <div class="customer-avatar">${initials}</div>
                        <div>
                            <div class="customer-name">
                                ${this.escapeHtml(customer.name)}
                                ${vatBadge}
                            </div>
                            <div class="customer-phone">${customer.customer_code || ''} ${customer.customer_code ? '•' : ''} ${customer.phone || 'N/A'}</div>
                        </div>
                    </div>
                </td>
                <td data-label="Contact">
                    <div>${customer.phone || 'N/A'}</div>
                    <small class="text-muted">${customer.email || ''}</small>
                </td>
                <td data-label="Type">
                    <span class="badge ${typeBadge.class} customer-type-badge">
                        ${typeBadge.text}
                    </span>
                </td>
                <td data-label="Credit" class="credit-info">
                    ${creditInfo}
                </td>
                <td data-label="Last Purchase">
                    <div class="last-purchase">${lastPurchase}</div>
                    <small class="text-muted">Total: KES ${parseFloat(customer.total_spent || 0).toFixed(2)}</small>
                </td>
                <td data-label="Actions" class="customer-actions">
                    <button class="btn btn-sm btn-outline-primary" 
                            onclick="event.stopPropagation(); window.POSCustomers.viewDetails(${customer.id})"
                            title="View Details">
                        <i class="bx bx-show"></i>
                    </button>
                    ${customer.customer_type !== 'walk-in' ? `
                    <button class="btn btn-sm btn-outline-warning" 
                            onclick="event.stopPropagation(); window.POSCustomers.editCustomer(${customer.id})"
                            title="Edit">
                        <i class="bx bx-edit"></i>
                    </button>
                    ` : ''}
                </td>
            </tr>
        `;
    }

    // Escape HTML to prevent XSS
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Get customer initials
    getInitials(name) {
        if (!name || name === 'Walk-in Customer') return 'WI';
        return name.split(' ')
            .map(part => part.charAt(0))
            .join('')
            .toUpperCase()
            .substring(0, 2);
    }

    // Get type badge
    getTypeBadge(type) {
        const badges = {
            'walk-in': { class: 'badge-walk-in', text: 'Walk-in' },
            'regular': { class: 'badge-regular', text: 'Regular' },
            'business': { class: 'badge-business', text: 'Business' },
            'credit': { class: 'badge-credit', text: 'Credit' }
        };
        return badges[type] || badges['walk-in'];
    }

    // Get credit info
    getCreditInfo(customer) {
        const creditLimit = parseFloat(customer.credit_limit || 0);
        const creditBalance = parseFloat(customer.credit_balance || 0);
        const availableCredit = creditLimit - creditBalance;
        
        if (creditLimit <= 0) {
            return '<span class="text-muted">No credit</span>';
        }
        
        const percentage = creditBalance > 0 ? (creditBalance / creditLimit) * 100 : 0;
        const progressBarClass = percentage > 80 ? 'bg-danger' : percentage > 50 ? 'bg-warning' : 'bg-success';
        
        return `
            <div>
                <div class="d-flex justify-content-between small">
                    <span>Limit: KES ${creditLimit.toFixed(2)}</span>
                    <span>Used: ${percentage.toFixed(0)}%</span>
                </div>
                <div class="progress" style="height: 4px;">
                    <div class="progress-bar ${progressBarClass}" 
                         style="width: ${percentage}%"></div>
                </div>
                <div class="credit-available mt-1">
                    Available: KES ${availableCredit.toFixed(2)}
                </div>
            </div>
        `;
    }

    // Update stats display
    updateStats(stats) {
        const elements = {
            totalCustomers: document.getElementById('totalCustomers'),
            activeToday: document.getElementById('activeToday'),
            totalCredit: document.getElementById('totalCredit'),
            avgCustomerSale: document.getElementById('avgCustomerSale')
        };

        if (elements.totalCustomers) {
            elements.totalCustomers.textContent = stats.total_customers || stats.total || 0;
        }
        if (elements.activeToday) {
            elements.activeToday.textContent = stats.active_today || stats.active || 0;
        }
        if (elements.totalCredit) {
            elements.totalCredit.textContent = `KES ${parseFloat(stats.total_credit || 0).toFixed(2)}`;
        }
        if (elements.avgCustomerSale) {
            elements.avgCustomerSale.textContent = `KES ${parseFloat(stats.avg_sale || stats.avg_order_value || 0).toFixed(2)}`;
        }
    }

    // Show fallback customers (for demo/offline)
    showFallbackCustomers() {
        this.customers = [
            {
                id: 1,
                name: 'Walk-in Customer',
                customer_code: 'WALK-IN',
                phone: '',
                email: '',
                customer_type: 'walk-in',
                credit_limit: 0,
                credit_balance: 0,
                total_spent: 0,
                total_orders: 0,
                last_purchase_date: null,
                vat_status: 'vatable',
                vat_number: null
            },
            {
                id: 2,
                name: 'John Doe (VAT Exempted)',
                customer_code: 'CUST-001',
                phone: '0712345678',
                email: 'john@example.com',
                customer_type: 'regular',
                credit_limit: 50000,
                credit_balance: 12500,
                total_spent: 45000,
                total_orders: 12,
                last_purchase_date: new Date().toISOString(),
                vat_status: 'exempted',
                vat_number: 'EX12345678',
                company_name: 'John Doe Enterprises'
            },
            {
                id: 3,
                name: 'Jane Smith (Zero-Rated)',
                customer_code: 'CUST-002',
                phone: '0723456789',
                email: 'jane@example.com',
                customer_type: 'business',
                credit_limit: 100000,
                credit_balance: 45000,
                total_spent: 120000,
                total_orders: 25,
                last_purchase_date: new Date(Date.now() - 86400000).toISOString(),
                vat_status: 'zero_rated',
                vat_number: 'ZR12345678',
                company_name: 'Smith Ltd'
            },
            {
                id: 4,
                name: 'ABC Enterprises (VATable)',
                customer_code: 'CUST-003',
                phone: '0734567890',
                email: 'accounts@abc.com',
                customer_type: 'credit',
                credit_limit: 200000,
                credit_balance: 175000,
                total_spent: 350000,
                total_orders: 45,
                last_purchase_date: new Date(Date.now() - 172800000).toISOString(),
                vat_status: 'vatable',
                vat_number: 'VAT12345678',
                company_name: 'ABC Enterprises Ltd'
            }
        ];
        
        this.renderCustomerTable();
        this.updateStats({
            total_customers: this.customers.length,
            active_today: 2,
            total_credit: 350000,
            avg_sale: 4500
        });
    }

    // Show customer modal
    showModal() {
        const modalElement = document.getElementById('customerModal');
        if (modalElement) {
            const modal = new bootstrap.Modal(modalElement);
            modal.show();
        }
    }

    /**
     * Enhanced selectCustomer method with VAT integration
     */
    async selectCustomer(customerId) {
        try {
            // Try the main customers endpoint first to get all customers
            const response = await fetch(`${this.baseUrl}/api/customers`, {
                headers: {
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.customers) {
                    // Find the specific customer from the list
                    const customer = data.customers.find(c => c.id == customerId);
                    
                    if (customer) {
                        this.selectedCustomer = customer;
                        
                        // Update state with customer VAT status
                        if (window.posState) {
                            window.posState.currentCustomer = {
                                id: customer.id,
                                name: customer.name,
                                phone: customer.phone,
                                email: customer.email,
                                vat_number: customer.vat_number,
                                vat_status: customer.vat_status || 'vatable',
                                pin: customer.pin,
                                company_name: customer.company_name,
                                address: customer.address,
                                customer_type: customer.customer_type || customer.type,
                                discount_percent: customer.discount_percent || 0,
                                credit_limit: customer.credit_limit || 0,
                                available_credit: customer.available_credit || 0,
                                loyalty_points: customer.loyalty_points || 0
                            };
                            
                            window.posState.saveCart();
                            
                            // Trigger VAT recalculation with new customer status
                            this.triggerCustomerVATUpdate();
                            
                            // Apply customer discount if any
                            if (customer.discount_percent > 0 && window.POSCart) {
                                window.POSCart.applyCustomerDiscount(customer.discount_percent);
                            }
                        }
                        
                        // Update UI
                        this.updateCurrentCustomerDisplay();
                        this.renderCustomerTable();
                        
                        // Show success message with VAT status info
                        const vatStatus = customer.vat_status || 'vatable';
                        let vatMessage = '';
                        
                        switch(vatStatus) {
                            case 'exempted':
                                vatMessage = ' (VAT Exempted)';
                                break;
                            case 'zero_rated':
                                vatMessage = ' (Zero-Rated VAT)';
                                break;
                            case 'vatable':
                                vatMessage = ' (VATable Customer)';
                                break;
                        }
                        
                        if (window.POSUtils) {
                            window.POSUtils.showToast(`Customer selected: ${customer.name}${vatMessage}`, 'success');
                        }
                        
                        // Update VAT display in payment modal if open
                        this.updatePaymentModalVAT();
                        
                        // Close modal after short delay
                        setTimeout(() => {
                            const modal = bootstrap.Modal.getInstance(document.getElementById('customerModal'));
                            if (modal) modal.hide();
                        }, 500);
                    } else {
                        throw new Error('Customer not found in list');
                    }
                } else {
                    throw new Error('Failed to load customers list');
                }
            } else {
                throw new Error('Failed to load customer details');
            }
        } catch (error) {
            console.error('Error selecting customer:', error);
            if (window.POSUtils) {
                window.POSUtils.showToast('Failed to load customer details', 'error');
            }
        }
    }

    /**
     * Trigger VAT recalculation when customer changes
     */
    triggerCustomerVATUpdate() {
        if (!window.posState || !window.posState.currentCustomer) return;
        
        // Recalculate VAT with new customer status
        if (window.POSCart && typeof window.POSCart.calculateVATTotals === 'function') {
            const vatData = window.POSCart.calculateVATTotals(window.posState.currentCustomer);
            
            // Update cart display
            if (window.POSCart.updateDisplay) {
                window.POSCart.updateDisplay();
            }
            
            // Update any open payment modal
            this.updatePaymentModalVAT();
        }
    }

    /**
     * Update VAT display in payment modal
     */
    updatePaymentModalVAT() {
        if (!window.posState || !window.posState.currentCustomer) return;
        
        const paymentModal = document.getElementById('paymentModal');
        if (!paymentModal || !paymentModal.classList.contains('show')) return;
        
        // Update payment modal VAT display
        if (window.updatePaymentModalVAT) {
            window.updatePaymentModalVAT();
        }
        
        // Update customer VAT status indicator in modal
        this.updateCustomerVATIndicator();
    }

    /**
     * Update customer VAT status indicator in payment modal
     */
    updateCustomerVATIndicator() {
        if (!window.posState || !window.posState.currentCustomer) return;
        
        const customer = window.posState.currentCustomer;
        const vatStatus = customer.vat_status || 'vatable';
        
        // Find or create VAT status indicator in payment modal
        let vatIndicator = document.getElementById('customerVATIndicator');
        if (!vatIndicator) {
            // Create indicator if it doesn't exist
            const paymentCustomerElement = document.getElementById('paymentCustomer');
            if (paymentCustomerElement) {
                vatIndicator = document.createElement('span');
                vatIndicator.id = 'customerVATIndicator';
                vatIndicator.className = 'badge ms-2';
                paymentCustomerElement.parentNode.appendChild(vatIndicator);
            }
        }
        
        if (vatIndicator) {
            // Update indicator based on VAT status
            switch(vatStatus) {
                case 'exempted':
                    vatIndicator.className = 'badge bg-danger ms-2';
                    vatIndicator.textContent = 'VAT Exempted';
                    vatIndicator.title = 'Customer is VAT exempted - no VAT applied';
                    break;
                case 'zero_rated':
                    vatIndicator.className = 'badge bg-warning text-dark ms-2';
                    vatIndicator.textContent = 'Zero-Rated';
                    vatIndicator.title = 'Customer is zero-rated - 0% VAT rate';
                    break;
                case 'vatable':
                    vatIndicator.className = 'badge bg-success ms-2';
                    vatIndicator.textContent = 'VATable';
                    vatIndicator.title = 'Customer is VATable - standard VAT rates apply';
                    break;
                default:
                    vatIndicator.className = 'badge bg-secondary ms-2';
                    vatIndicator.textContent = 'Standard';
                    vatIndicator.title = 'Standard VAT treatment';
            }
        }
    }

    /**
     * Enhanced customer display with VAT info
     */
    updateCurrentCustomerDisplay() {
        const customer = this.selectedCustomer || window.posState?.currentCustomer;
        
        const nameElement = document.getElementById('currentCustomerName');
        const metaElement = document.getElementById('currentCustomerMeta');
        const statsElement = document.getElementById('customerStats');
        const customerIdElement = document.getElementById('customerId');
        
        if (customer && customer.id) {
            // Customer is selected
            if (nameElement) {
                nameElement.textContent = customer.name || 'Unnamed Customer';
            }
            if (metaElement) {
                const parts = [];
                if (customer.customer_code) parts.push(customer.customer_code);
                if (customer.phone) parts.push(customer.phone);
                if (customer.customer_type) parts.push(customer.customer_type.charAt(0).toUpperCase() + customer.customer_type.slice(1));
                
                // Add VAT status if available
                if (customer.vat_status) {
                    let vatStatusText = '';
                    switch(customer.vat_status) {
                        case 'exempted':
                            vatStatusText = 'VAT Exempted';
                            break;
                        case 'zero_rated':
                            vatStatusText = 'Zero-Rated';
                            break;
                        case 'vatable':
                            vatStatusText = 'VATable';
                            break;
                    }
                    if (vatStatusText) parts.push(vatStatusText);
                }
                
                metaElement.textContent = parts.join(' • ');
            }
            if (customerIdElement) {
                customerIdElement.value = customer.id;
            }
            if (statsElement) {
                statsElement.style.display = 'block';
                this.updateCustomerStats();
            }
            
            // Show tax info if available
            if (customer.vat_number || customer.vat_status) {
                this.showTaxInfo(customer.vat_number, customer.vat_status);
            }
            
            // Update VAT status badge in cart section
            this.updateCartVATStatusBadge(customer);
        } else {
            // Walk-in customer (default)
            if (nameElement) nameElement.textContent = 'Walk-in Customer';
            if (metaElement) metaElement.textContent = 'Tap to select customer';
            if (customerIdElement) customerIdElement.value = '';
            if (statsElement) statsElement.style.display = 'none';
            
            // Remove VAT status badge
            this.removeCartVATStatusBadge();
        }
    }

    /**
     * Update VAT status badge in cart section
     */
    updateCartVATStatusBadge(customer) {
        const cartSection = document.querySelector('.card.cart-wrapper') || 
                           document.querySelector('#cartItemsList')?.closest('.card');
        
        if (!cartSection) return;
        
        // Remove existing badge
        this.removeCartVATStatusBadge();
        
        const vatStatus = customer.vat_status || 'vatable';
        if (vatStatus !== 'vatable') {
            let badgeHtml = '';
            let badgeClass = '';
            
            switch(vatStatus) {
                case 'exempted':
                    badgeClass = 'bg-danger';
                    badgeHtml = '<i class="bx bx-shield-x me-1"></i>VAT Exempted';
                    break;
                case 'zero_rated':
                    badgeClass = 'bg-warning text-dark';
                    badgeHtml = '<i class="bx bx-shield-alt-2 me-1"></i>Zero-Rated VAT';
                    break;
            }
            
            if (badgeHtml) {
                const badge = document.createElement('div');
                badge.className = `vat-customer-badge badge ${badgeClass} mb-3 p-2`;
                badge.innerHTML = `
                    <div class="d-flex align-items-center">
                        ${badgeHtml}
                        <small class="ms-2">Customer: ${customer.name}</small>
                    </div>
                `;
                
                const cardBody = cartSection.querySelector('.card-body');
                if (cardBody) {
                    cardBody.insertBefore(badge, cardBody.firstChild);
                }
            }
        }
    }

    /**
     * Remove VAT status badge from cart
     */
    removeCartVATStatusBadge() {
        const existingBadge = document.querySelector('.vat-customer-badge');
        if (existingBadge) {
            existingBadge.remove();
        }
    }

    /**
     * Enhanced tax info display
     */
    showTaxInfo(vatNumber, vatStatus = 'vatable') {
        if (!window.POSUtils) return;
        
        let message = '';
        let type = 'info';
        
        if (vatNumber) {
            message = `VAT/Tax PIN: ${vatNumber}`;
        }
        
        switch(vatStatus) {
            case 'exempted':
                message += ' (VAT Exempted Customer - No VAT Applied)';
                type = 'warning';
                break;
            case 'zero_rated':
                message += ' (Zero-Rated Customer - 0% VAT Rate)';
                type = 'info';
                break;
            case 'vatable':
                message += message ? ' (VATable Customer)' : 'VATable Customer';
                type = 'success';
                break;
        }
        
        window.POSUtils.showToast(message, type, 4000);
    }

    /**
     * Update customer stats
     */
    async updateCustomerStats() {
        const customer = this.selectedCustomer || window.posState?.currentCustomer;
        
        if (!customer || !customer.id) return;
        
        const elements = {
            customerTotalSpent: document.getElementById('customerTotalSpent'),
            customerVisits: document.getElementById('customerVisits'),
            customerCreditLimit: document.getElementById('customerCreditLimit'),
            customerLastVisit: document.getElementById('customerLastVisit')
        };
        
        // Update from customer object
        if (elements.customerTotalSpent) {
            elements.customerTotalSpent.textContent = `KES ${parseFloat(customer.total_spent || 0).toFixed(2)}`;
        }
        if (elements.customerVisits) {
            elements.customerVisits.textContent = customer.total_orders || 0;
        }
        if (elements.customerCreditLimit) {
            elements.customerCreditLimit.textContent = `KES ${parseFloat(customer.credit_limit || 0).toFixed(2)}`;
        }
        if (elements.customerLastVisit) {
            elements.customerLastVisit.textContent = customer.last_purchase_date 
                ? new Date(customer.last_purchase_date).toLocaleDateString()
                : customer.last_purchase 
                    ? new Date(customer.last_purchase).toLocaleDateString()
                    : '-';
        }
        
        // Load detailed stats from API if needed
        if (!customer.total_spent || !customer.total_orders) {
            try {
                const response = await fetch(`${this.baseUrl}/api/customers/${customer.id}/stats`);
                if (response.ok) {
                    const data = await response.json();
                    if (data.success && data.stats) {
                        if (elements.customerTotalSpent && data.stats.total_spent) {
                            elements.customerTotalSpent.textContent = `KES ${parseFloat(data.stats.total_spent).toFixed(2)}`;
                        }
                        if (elements.customerVisits && data.stats.total_orders) {
                            elements.customerVisits.textContent = data.stats.total_orders;
                        }
                    }
                }
            } catch (error) {
                console.error('Error loading customer stats:', error);
            }
        }
    }

    /**
     * Get current customer with VAT status
     */
    getCurrentWithVAT() {
        const customer = this.selectedCustomer || window.posState?.currentCustomer;
        
        if (customer) {
            // Ensure customer has VAT status property
            return {
                ...customer,
                vat_status: customer.vat_status || 'vatable',
                is_vat_exempted: customer.vat_status === 'exempted',
                is_zero_rated: customer.vat_status === 'zero_rated',
                is_vatable: customer.vat_status === 'vatable' || !customer.vat_status
            };
        }
        
        return null;
    }

    /**
     * Set current customer with VAT status handling
     */
    setCurrent(customer) {
        if (customer === null || customer === undefined) {
            this.selectedCustomer = null;
            if (window.posState) {
                window.posState.currentCustomer = null;
                window.posState.saveCart();
            }
        } else {
            // Ensure customer has VAT status
            const customerWithVAT = {
                ...customer,
                vat_status: customer.vat_status || 'vatable'
            };
            
            this.selectedCustomer = customerWithVAT;
            if (window.posState) {
                window.posState.currentCustomer = customerWithVAT;
                window.posState.saveCart();
            }
            
            // Trigger VAT recalculation
            this.triggerCustomerVATUpdate();
        }
        
        this.updateCurrentCustomerDisplay();
    }

    /**
     * Select walk-in customer with VAT status
     */
    selectWalkInCustomer() {
        this.setCurrent({
            id: null,
            name: 'Walk-in Customer',
            customer_type: 'walk-in',
            vat_status: 'vatable' // Walk-in customers are vatable by default
        });
        
        const modal = bootstrap.Modal.getInstance(document.getElementById('customerModal'));
        if (modal) modal.hide();
        
        if (window.POSUtils) {
            window.POSUtils.showToast('Walk-in customer selected (VATable)', 'info');
        }
    }

    /**
     * Quick add customer using Laravel API
     */
    async addQuickCustomer() {
        const name = document.getElementById('quickName')?.value.trim();
        const phone = document.getElementById('quickPhone')?.value.trim();
        const email = document.getElementById('quickEmail')?.value.trim();
        const type = document.getElementById('quickType')?.value;
        const credit = parseFloat(document.getElementById('quickCredit')?.value || 0);
        const vatStatus = document.getElementById('quickVatStatus')?.value || 'vatable';
        const vatNumber = document.getElementById('quickVatNumber')?.value.trim();
        
        if (!name) {
            if (window.POSUtils) {
                window.POSUtils.showToast('Customer name is required', 'error');
            }
            return;
        }
        
        try {
            const response = await fetch(`${this.baseUrl}/api/customers/quick-add`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                },
                body: JSON.stringify({
                    name: name,
                    phone: phone || null,
                    email: email || null,
                    customer_type: type || 'regular',
                    credit_limit: credit,
                    vat_status: vatStatus,
                    vat_number: vatNumber || null,
                    company_name: document.getElementById('quickCompany')?.value.trim() || null
                })
            });
            
            const data = await response.json();
            
            if (response.ok && data.success) {
                // Select the new customer
                this.selectCustomer(data.customer.id);
                
                // Clear form
                ['quickName', 'quickPhone', 'quickEmail', 'quickCredit', 'quickVatNumber', 'quickCompany'].forEach(id => {
                    const element = document.getElementById(id);
                    if (element) element.value = '';
                });
                
                if (window.POSUtils) {
                    window.POSUtils.showToast('Customer added successfully', 'success');
                }
                
                // Refresh customer list
                this.loadCustomers();
            } else {
                throw new Error(data.message || data.error || 'Failed to add customer');
            }
        } catch (error) {
            console.error('Error adding customer:', error);
            if (window.POSUtils) {
                window.POSUtils.showToast(error.message || 'Failed to add customer', 'error');
            }
        }
    }

    /**
     * Enhanced customer details modal with VAT info
     */
    async viewDetails(customerId) {
        try {
            const response = await fetch(`${this.baseUrl}/api/customers/${customerId}/details`);
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.customer) {
                    this.showCustomerDetailsModal(data.customer);
                }
            } else {
                // Fallback to simple alert
                this.selectCustomer(customerId);
            }
        } catch (error) {
            console.error('Error viewing customer details:', error);
            this.selectCustomer(customerId);
        }
    }

    /**
     * Enhanced customer details modal with VAT info
     */
    showCustomerDetailsModal(customer) {
        // Create VAT status display
        let vatStatusDisplay = '';
        const vatStatus = customer.vat_status || 'vatable';
        
        let vatStatusClass = '';
        let vatStatusText = '';
        
        switch(vatStatus) {
            case 'exempted':
                vatStatusClass = 'vat-status-exempted';
                vatStatusText = 'VAT Exempted';
                break;
            case 'zero_rated':
                vatStatusClass = 'vat-status-zero-rated';
                vatStatusText = 'Zero-Rated';
                break;
            case 'vatable':
                vatStatusClass = 'vat-status-vatable';
                vatStatusText = 'VATable';
                break;
        }
        
        if (vatStatusClass) {
            vatStatusDisplay = `
                <div class="customer-vat-info ${vatStatusClass} mb-3 p-2 rounded">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <strong>VAT Status:</strong> ${vatStatusText}
                            ${customer.vat_number ? `<div><strong>VAT Number:</strong> ${customer.vat_number}</div>` : ''}
                            ${customer.company_name ? `<div><strong>Company:</strong> ${customer.company_name}</div>` : ''}
                        </div>
                        <span class="badge ${vatStatus === 'exempted' ? 'bg-danger' : vatStatus === 'zero_rated' ? 'bg-warning text-dark' : 'bg-success'}">
                            ${vatStatusText}
                        </span>
                    </div>
                </div>
            `;
        }
        
        // Create and show a details modal
        const modalHtml = `
            <div class="modal fade" id="customerDetailsModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header bg-primary text-white">
                            <h5 class="modal-title">Customer Details</h5>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            ${vatStatusDisplay}
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="text-center mb-3">
                                        <div class="customer-avatar-large mb-2">
                                            ${this.getInitials(customer.name)}
                                        </div>
                                        <h5>${this.escapeHtml(customer.name)}</h5>
                                        <span class="badge ${this.getTypeBadge(customer.customer_type).class}">
                                            ${this.getTypeBadge(customer.customer_type).text}
                                        </span>
                                    </div>
                                </div>
                                <div class="col-md-8">
                                    <dl class="row">
                                        <dt class="col-sm-4">Customer Code:</dt>
                                        <dd class="col-sm-8">${customer.customer_code || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">Phone:</dt>
                                        <dd class="col-sm-8">${customer.phone || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">Email:</dt>
                                        <dd class="col-sm-8">${customer.email || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">Company:</dt>
                                        <dd class="col-sm-8">${customer.company_name || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">VAT Number:</dt>
                                        <dd class="col-sm-8">${customer.vat_number || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">Address:</dt>
                                        <dd class="col-sm-8">${customer.address || 'N/A'}</dd>
                                        
                                        <dt class="col-sm-4">Total Spent:</dt>
                                        <dd class="col-sm-8">KES ${parseFloat(customer.total_spent || 0).toFixed(2)}</dd>
                                        
                                        <dt class="col-sm-4">Total Orders:</dt>
                                        <dd class="col-sm-8">${customer.total_orders || 0}</dd>
                                        
                                        <dt class="col-sm-4">Credit Limit:</dt>
                                        <dd class="col-sm-8">KES ${parseFloat(customer.credit_limit || 0).toFixed(2)}</dd>
                                        
                                        <dt class="col-sm-4">Available Credit:</dt>
                                        <dd class="col-sm-8">KES ${parseFloat(customer.available_credit || 0).toFixed(2)}</dd>
                                        
                                        <dt class="col-sm-4">Last Purchase:</dt>
                                        <dd class="col-sm-8">${customer.last_purchase_date ? new Date(customer.last_purchase_date).toLocaleDateString() : 'Never'}</dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="button" class="btn btn-primary" onclick="window.POSCustomers.selectCustomer(${customer.id});">
                                Select This Customer
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        const existingModal = document.getElementById('customerDetailsModal');
        if (existingModal) existingModal.remove();
        
        // Add new modal to DOM
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modalElement = document.getElementById('customerDetailsModal');
        const modal = new bootstrap.Modal(modalElement);
        modal.show();
        
        // Clean up on hide
        modalElement.addEventListener('hidden.bs.modal', () => {
            modalElement.remove();
        });
    }

    /**
     * Edit customer
     */
    async editCustomer(customerId) {
        const customer = this.customers.find(c => c.id === customerId);
        if (!customer || customer.customer_type === 'walk-in') return;
        
        // Simple edit dialog
        const newName = prompt('Edit customer name:', customer.name);
        if (newName && newName !== customer.name) {
            try {
                const response = await fetch(`${this.baseUrl}/api/customers/${customerId}`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': this.csrfToken
                    },
                    body: JSON.stringify({ name: newName })
                });
                
                if (response.ok) {
                    customer.name = newName;
                    this.renderCustomerTable();
                    
                    // Update selected customer if it's the same
                    if (this.selectedCustomer?.id === customerId) {
                        this.selectedCustomer.name = newName;
                        this.updateCurrentCustomerDisplay();
                    }
                    
                    if (window.POSUtils) {
                        window.POSUtils.showToast('Customer updated', 'success');
                    }
                }
            } catch (error) {
                console.error('Error updating customer:', error);
                if (window.POSUtils) {
                    window.POSUtils.showToast('Failed to update customer', 'error');
                }
            }
        }
    }

    /**
     * Get current customer
     */
    getCurrent() {
        return this.selectedCustomer || window.posState?.currentCustomer;
    }

    /**
     * Quick add customer (from main page)
     */
    quickAddCustomer() {
        this.showModal();
        
        setTimeout(() => {
            const quickAddForm = document.getElementById('quickAddForm');
            const nameInput = document.getElementById('quickName');
            
            if (quickAddForm) {
                quickAddForm.scrollIntoView({ behavior: 'smooth' });
            }
            if (nameInput) {
                nameInput.focus();
            }
        }, 500);
    }

    /**
     * View customer history
     */
    async viewCustomerHistory() {
        const customer = this.getCurrent();
        if (!customer || !customer.id) {
            if (window.POSUtils) {
                window.POSUtils.showToast('Select a customer first', 'warning');
            }
            return;
        }
        
        try {
            const response = await fetch(`${this.baseUrl}/api/customers/${customer.id}/history`);
            if (response.ok) {
                const data = await response.json();
                this.showPurchaseHistoryModal(data);
            } else {
                if (window.POSUtils) {
                    window.POSUtils.showToast('Purchase history not available', 'info');
                }
            }
        } catch (error) {
            console.error('Error loading purchase history:', error);
            if (window.POSUtils) {
                window.POSUtils.showToast('Failed to load purchase history', 'error');
            }
        }
    }

    /**
     * Check credit balance
     */
    checkCreditBalance() {
        const customer = this.getCurrent();
        if (!customer || !customer.id) {
            if (window.POSUtils) {
                window.POSUtils.showToast('Select a customer first', 'warning');
            }
            return;
        }
        
        const availableCredit = parseFloat(customer.available_credit || 0);
        const creditLimit = parseFloat(customer.credit_limit || 0);
        const creditBalance = parseFloat(customer.credit_balance || 0);
        
        const message = `
            Credit Information for ${customer.name}:
            ------------------------------------------
            Credit Limit: KES ${creditLimit.toFixed(2)}
            Credit Used: KES ${creditBalance.toFixed(2)}
            Available Credit: KES ${availableCredit.toFixed(2)}
            ------------------------------------------
            ${availableCredit > 0 ? '✅ Credit available for purchase' : '❌ No credit available'}
        `;
        
        alert(message);
    }

    /**
     * Apply loyalty points
     */
    applyLoyaltyPoints() {
        const customer = this.getCurrent();
        if (!customer || !customer.id) {
            if (window.POSUtils) {
                window.POSUtils.showToast('Select a customer first', 'warning');
            }
            return;
        }
        
        const points = customer.loyalty_points || 0;
        if (points > 0) {
            const discountAmount = Math.min(points * 0.1, 1000); // 0.1 KES per point, max 1000 KES
            if (window.POSCart) {
                window.POSCart.applyLoyaltyDiscount(discountAmount);
            }
        } else {
            if (window.POSUtils) {
                window.POSUtils.showToast('No loyalty points available', 'info');
            }
        }
    }

    /**
     * Apply customer discount to cart
     */
    applyCustomerDiscountToCart() {
        const customer = this.getCurrent();
        if (!customer || !customer.id || !customer.discount_percent) return;
        
        if (window.POSCart) {
            window.POSCart.applyCustomerDiscount(customer.discount_percent);
        }
    }

    /**
     * Validate credit purchase
     */
    validateCreditPurchase(amount) {
        const customer = this.getCurrent();
        if (!customer || !customer.id) {
            return { allowed: false, reason: 'No customer selected' };
        }
        
        const availableCredit = parseFloat(customer.available_credit || 0);
        if (availableCredit <= 0) {
            return { allowed: false, reason: 'No credit limit available' };
        }
        
        if (amount > availableCredit) {
            return { 
                allowed: false, 
                reason: `Credit limit exceeded. Available: KES ${availableCredit.toFixed(2)}` 
            };
        }
        
        return { allowed: true, available: availableCredit };
    }

    /**
     * Include tax/VAT information in sale
     */
    includeTaxInSale() {
        const customer = this.getCurrent();
        if (!customer || !customer.vat_number) return null;
        
        return {
            vat_number: customer.vat_number,
            vat_status: customer.vat_status || 'vatable',
            company_name: customer.company_name,
            tax_id: customer.tax_id
        };
    }

    /**
     * UPDATE VAT DISPLAY FOR CUSTOMER - FIXED METHOD
     */
    updateVATDisplayForCustomer() {
        const customer = this.selectedCustomer || window.posState?.currentCustomer;
        
        if (!customer) return;
        
        // Update VAT status badge
        this.updateCartVATStatusBadge(customer);
        
        // Update payment modal if open
        this.updatePaymentModalVAT();
        
        // Update current customer display
        this.updateCurrentCustomerDisplay();
    }
}

// Initialize and export
window.POSCustomers = new POSCustomers();

// Global functions for inline onclick handlers
window.showCustomerModal = function() {
    window.POSCustomers.showModal();
};

window.selectWalkInCustomer = function() {
    window.POSCustomers.selectWalkInCustomer();
};

window.addQuickCustomer = function() {
    window.POSCustomers.addQuickCustomer();
};

window.quickAddCustomer = function() {
    window.POSCustomers.quickAddCustomer();
};

window.viewCustomerHistory = function() {
    window.POSCustomers.viewCustomerHistory();
};

window.checkCreditBalance = function() {
    window.POSCustomers.checkCreditBalance();
};

window.applyLoyaltyPoints = function() {
    window.POSCustomers.applyLoyaltyPoints();
};

// Global helper function for VAT status integration
window.updateCartWithCustomerVAT = function(customer) {
    if (window.POSCustomers) {
        window.POSCustomers.setCurrent(customer);
    }
};

// Enhanced customer selection that includes VAT recalculation
window.selectCustomerWithVAT = function(customerId) {
    if (window.POSCustomers && window.POSCustomers.selectCustomer) {
        window.POSCustomers.selectCustomer(customerId);
    }
};

// Auto-initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    if (window.POSCustomers && window.POSCustomers.init) {
        window.POSCustomers.init();
    }
});