// public/js/pos/modules/cart.js
// POS Cart Management Module - Production Grade with VAT Support
// =========================================================

class POSCart {
    static init() {
        console.log('🛒 Initializing POS Cart with VAT Support');
        this.loadSavedState();
        this.updateDisplay();
        return this;
    }
    
    static loadSavedState() {
        try {
            if (window.posState && typeof window.posState.loadSavedState === 'function') {
                window.posState.loadSavedState();
            } else if (window.posState) {
                // Fallback loading
                const savedCart = localStorage.getItem('pos_cart');
                if (savedCart) {
                    window.posState.cart = JSON.parse(savedCart);
                }
                
                const savedCustomer = localStorage.getItem('pos_current_customer');
                if (savedCustomer) {
                    window.posState.currentCustomer = JSON.parse(savedCustomer);
                }
                
                const savedDiscount = localStorage.getItem('pos_current_discount');
                if (savedDiscount) {
                    window.posState.currentDiscount = parseFloat(savedDiscount);
                }
            }
        } catch (error) {
            console.error('Error loading saved state:', error);
        }
    }
    
    static add(product, quantity = 1) {
        console.log('➕ Adding to cart:', product?.name || 'Unknown product');
        
        try {
            // Validate product
            if (!product || typeof product !== 'object') {
                console.error('Invalid product data:', product);
                this.safeToast('Invalid product data', 'error');
                return false;
            }
            
            if (!product.id || !product.name) {
                console.error('Product missing required properties:', product);
                this.safeToast('Product data is incomplete', 'error');
                return false;
            }
            
            // Convert quantity
            quantity = parseInt(quantity);
            if (isNaN(quantity) || quantity < 1) quantity = 1;
            
            // Check stock
            if (product.stock && quantity > product.stock) {
                this.safeToast(`Only ${product.stock} units available`, 'error');
                return false;
            }
            
            // Check if posState exists
            if (!window.posState || !window.posState.cart) {
                console.error('posState not initialized');
                this.safeToast('System error: Cart not available', 'error');
                return false;
            }
            
            // Find existing item
            const existingIndex = window.posState.cart.findIndex(item => item.id === product.id);
            
            if (existingIndex !== -1) {
                // Update existing
                const existingItem = window.posState.cart[existingIndex];
                const newQuantity = existingItem.quantity + quantity;
                
                if (product.stock && newQuantity > product.stock) {
                    this.safeToast(`Only ${product.stock} units available`, 'error');
                    return false;
                }
                
                existingItem.quantity = newQuantity;
            } else {
                // Add new item
                window.posState.cart.push({
                    id: product.id,
                    sku: product.sku || '',
                    barcode: product.barcode || '',
                    name: product.name,
                    price: product.price || product.final_price || product.sale_price || 0,
                    cost_price: product.cost_price || 0,
                    quantity: quantity,
                    tax_rate: product.tax_rate || 16,
                    is_vatable: product.has_vat || product.is_vatable || false,
                    has_vat: product.has_vat || product.is_vatable || (product.tax_rate > 0),
                    stock: product.stock || 0,
                    category_id: product.category_id || null,
                    vat_amount: 0 // Will be calculated in updateDisplay
                });
            }
            
            // Update UI and save
            this.updateDisplay();
            
            // Save to storage
            this.saveCart();
            
            this.safeToast(`${product.name} added to cart`, 'success');
            
            // Play sound if enabled
            if (window.posState?.storeSettings?.sound_effects) {
                this.safePlaySound('add_to_cart');
            }
            
            return true;
            
        } catch (error) {
            console.error('Error adding to cart:', error);
            this.safeToast('Failed to add product to cart', 'error');
            return false;
        }
    }
    
    static updateDisplay() {
        try {
            if (!window.posState || !Array.isArray(window.posState.cart)) {
                console.warn('Cart state not available');
                return;
            }
            
            console.log('🔄 Updating cart display with', window.posState.cart.length, 'items');
            
            const cartItemsContainer = document.getElementById('cartItemsList');
            const emptyCartState = document.getElementById('emptyCartState');
            const cartCount = document.getElementById('cartCount');
            const checkoutBtn = document.getElementById('checkoutBtn');
            
            if (window.posState.cart.length === 0) {
                // Show empty state
                if (cartItemsContainer) cartItemsContainer.classList.add('d-none');
                if (emptyCartState) emptyCartState.classList.remove('d-none');
                if (cartCount) cartCount.textContent = '0';
                if (checkoutBtn) checkoutBtn.disabled = true;
                this.updateSummary();
                return;
            }
            
            // Show cart items
            if (emptyCartState) emptyCartState.classList.add('d-none');
            if (cartItemsContainer) cartItemsContainer.classList.remove('d-none');
            
            // Clear existing items
            if (cartItemsContainer) cartItemsContainer.innerHTML = '';
            
            // Calculate totals using VAT service if available
            const vatData = this.calculateVATTotals();
            
            // Store calculated VAT data
            window.posState.cartVATData = vatData;
            
            // Add cart items
            window.posState.cart.forEach((item, index) => {
                try {
                    const itemTotal = item.price * item.quantity;
                    
                    const itemTemplate = document.getElementById('cartItemTemplate');
                    if (!itemTemplate) {
                        console.warn('Cart item template not found');
                        return;
                    }
                    
                    const templateContent = itemTemplate.content.cloneNode(true);
                    
                    // Fill template
                    const cartItemElement = templateContent.querySelector('.cart-item');
                    if (cartItemElement) {
                        cartItemElement.dataset.itemId = item.id;
                        cartItemElement.dataset.vatRate = item.tax_rate || 0;
                        cartItemElement.dataset.isVatable = item.is_vatable || false;
                    }
                    
                    const nameElement = templateContent.querySelector('.cart-item-name');
                    if (nameElement) nameElement.textContent = item.name;
                    
                    const priceElement = templateContent.querySelector('.cart-item-price');
                    if (priceElement) priceElement.textContent = `KES ${(item.price || 0).toFixed(2)} each`;
                    
                    const skuElement = templateContent.querySelector('.cart-item-sku');
                    if (skuElement) skuElement.textContent = item.sku || 'N/A';
                    
                    const totalElement = templateContent.querySelector('.cart-item-total');
                    if (totalElement) totalElement.textContent = `KES ${itemTotal.toFixed(2)}`;
                    
                    // Add VAT information to item display
                    const vatInfoElement = templateContent.querySelector('.cart-item-vat-info');
                    if (vatInfoElement) {
                        if (item.is_vatable || item.has_vat || item.tax_rate > 0) {
                            const vatRate = item.tax_rate || 16;
                            const priceExcludingVAT = item.price_ex_tax || item.price || 0;
                            vatInfoElement.innerHTML = `
                                <small class="text-success">
                                    Price: KES ${priceExcludingVAT.toFixed(2)} + VAT ${vatRate}%
                                </small>
                            `;
                            vatInfoElement.classList.remove('d-none');
                        } else {
                            vatInfoElement.innerHTML = `
                                <small class="text-muted">No VAT applicable</small>
                            `;
                            vatInfoElement.classList.remove('d-none');
                        }
                    }
                    
                    const quantityInput = templateContent.querySelector('.quantity-input');
                    if (quantityInput) {
                        quantityInput.value = item.quantity;
                        quantityInput.dataset.itemIndex = index;
                    }
                    
                    // Event listeners
                    const decreaseBtn = templateContent.querySelector('[onclick*="decreaseQuantity"]');
                    if (decreaseBtn) {
                        decreaseBtn.addEventListener('click', (e) => {
                            e.stopPropagation();
                            e.preventDefault();
                            this.updateQuantity(index, -1);
                        });
                    }
                    
                    const increaseBtn = templateContent.querySelector('[onclick*="increaseQuantity"]');
                    if (increaseBtn) {
                        increaseBtn.addEventListener('click', (e) => {
                            e.stopPropagation();
                            e.preventDefault();
                            this.updateQuantity(index, 1);
                        });
                    }
                    
                    if (quantityInput) {
                        quantityInput.addEventListener('change', (e) => {
                            const newValue = parseInt(e.target.value);
                            const itemIndex = parseInt(e.target.dataset.itemIndex);
                            this.updateQuantity(itemIndex, 0, newValue);
                        });
                    }
                    
                    const removeBtn = templateContent.querySelector('[onclick*="removeFromCart"]');
                    if (removeBtn) {
                        removeBtn.addEventListener('click', (e) => {
                            e.stopPropagation();
                            e.preventDefault();
                            this.remove(index);
                        });
                    }
                    
                    if (cartItemsContainer) cartItemsContainer.appendChild(templateContent);
                } catch (itemError) {
                    console.error('Error rendering cart item:', itemError, item);
                }
            });
            
            // Update cart count
            if (cartCount) cartCount.textContent = window.posState.cart.length;
            
            // Update checkout button
            if (checkoutBtn) {
                const total = vatData.totalWithVAT || 0;
                checkoutBtn.innerHTML = `<i class="bx bx-credit-card me-2"></i>CHECKOUT KES ${total.toFixed(2)}`;
                checkoutBtn.disabled = total <= 0;
            }
            
            // Update summary with VAT calculation
            this.updateSummary();
            
        } catch (error) {
            console.error('Error updating cart display:', error);
        }
    }
    
    static updateQuantity(index, change, newValue = null) {
        try {
            if (!window.posState || !Array.isArray(window.posState.cart)) {
                console.error('Cart state not available');
                return;
            }
            
            if (index < 0 || index >= window.posState.cart.length) {
                console.error('Invalid cart index:', index);
                return;
            }
            
            const item = window.posState.cart[index];
            
            if (newValue !== null) {
                newValue = parseInt(newValue);
                if (isNaN(newValue) || newValue < 1) newValue = 1;
                if (item.stock && newValue > item.stock) {
                    this.safeToast(`Only ${item.stock} units available`, 'error');
                    newValue = item.stock;
                }
                item.quantity = newValue;
            } else {
                const newQuantity = item.quantity + change;
                if (newQuantity < 1) {
                    this.remove(index);
                    return;
                }
                if (item.stock && newQuantity > item.stock) {
                    this.safeToast(`Only ${item.stock} units available`, 'error');
                    return;
                }
                item.quantity = newQuantity;
            }
            
            this.updateDisplay();
            
            // Save to storage
            this.saveCart();
            
        } catch (error) {
            console.error('Error updating quantity:', error);
            this.safeToast('Failed to update quantity', 'error');
        }
    }
    
    static remove(index) {
        try {
            if (!window.posState || !Array.isArray(window.posState.cart)) {
                console.error('Cart state not available');
                return;
            }
            
            if (index < 0 || index >= window.posState.cart.length) {
                console.error('Invalid cart index:', index);
                return;
            }
            
            const item = window.posState.cart[index];
            window.posState.cart.splice(index, 1);
            this.updateDisplay();
            
            // Save to storage
            this.saveCart();
            
            this.safeToast(`${item.name} removed from cart`, 'info');
        } catch (error) {
            console.error('Error removing item from cart:', error);
            this.safeToast('Failed to remove item', 'error');
        }
    }
    
    static clear() {
        try {
            if (!window.posState || !Array.isArray(window.posState.cart)) {
                console.error('Cart state not available');
                return;
            }
            
            if (window.posState.cart.length === 0) return;
            
            if (confirm('Clear all items from cart?')) {
                window.posState.cart = [];
                window.posState.currentDiscount = 0;
                window.posState.cartVATData = null;
                
                this.updateDisplay();
                this.saveCart();
                
                this.safeToast('Cart cleared', 'info');
            }
        } catch (error) {
            console.error('Error clearing cart:', error);
            this.safeToast('Failed to clear cart', 'error');
        }
    }
    
    static updateSummary() {
        try {
            if (!window.posState) {
                console.warn('posState not available for summary update');
                return;
            }
            
            // Get current customer
            const customer = window.posState.currentCustomer || null;
            
            // Calculate VAT with customer status
            const vatData = this.calculateVATTotals(customer);
            
            // Store VAT data for later use
            window.posState.cartVATData = vatData;
            
            const subtotal = vatData.subtotal || 0;
            const totalVAT = vatData.totalVAT || 0;
            const discountAmount = vatData.discount || 0;
            const total = vatData.totalWithVAT || 0;
            
            // Update main summary elements
            const subtotalElement = document.getElementById('cartSubtotal');
            const vatElement = document.getElementById('cartVat');
            const discountElement = document.getElementById('cartDiscount');
            const totalElement = document.getElementById('cartTotal');
            
            if (subtotalElement) subtotalElement.textContent = `KES ${subtotal.toFixed(2)}`;
            if (vatElement) vatElement.textContent = `KES ${totalVAT.toFixed(2)}`;
            if (discountElement) discountElement.textContent = `-KES ${discountAmount.toFixed(2)}`;
            if (totalElement) totalElement.textContent = `KES ${total.toFixed(2)}`;
            
            // Update customer VAT status display
            this.updateCustomerVATStatusDisplay(customer);
            
            // Update checkout button
            const checkoutBtn = document.getElementById('checkoutBtn');
            if (checkoutBtn) {
                checkoutBtn.innerHTML = `<i class="bx bx-credit-card me-2"></i>CHECKOUT KES ${total.toFixed(2)}`;
                checkoutBtn.disabled = total <= 0;
            }
            
        } catch (error) {
            console.error('Error updating cart summary:', error);
        }
    }
    
    static updateCustomerVATStatusDisplay(customer) {
        const customerVatStatusElement = document.getElementById('customerVatStatus');
        if (!customerVatStatusElement) return;
        
        if (customer) {
            const vatStatus = customer.vat_status || 'vatable';
            if (vatStatus === 'exempted') {
                customerVatStatusElement.innerHTML = `
                    <span class="badge bg-danger">
                        <i class="bx bx-shield-x me-1"></i>VAT Exempted
                    </span>
                `;
                customerVatStatusElement.classList.remove('d-none');
            } else if (vatStatus === 'zero_rated') {
                customerVatStatusElement.innerHTML = `
                    <span class="badge bg-warning text-dark">
                        <i class="bx bx-shield-alt-2 me-1"></i>Zero-Rated
                    </span>
                `;
                customerVatStatusElement.classList.remove('d-none');
            } else {
                customerVatStatusElement.classList.add('d-none');
            }
        } else {
            customerVatStatusElement.classList.add('d-none');
        }
    }
    
    static calculateVATTotals(customer = null) {
        try {
            if (!window.posState || !Array.isArray(window.posState.cart)) {
                return {
                    totalVAT: 0,
                    vatByRate: {},
                    taxableAmount: 0,
                    nonTaxableAmount: 0,
                    exemptedAmount: 0,
                    zeroRatedAmount: 0,
                    subtotal: 0,
                    customerVatStatus: customer?.vat_status || 'vatable',
                    isVatable: true,
                    isZeroRated: false,
                    isExempted: false
                };
            }
            
            // Use VAT service if available
            if (window.VATService && window.VATService.calculateCartVAT) {
                return window.VATService.calculateCartVAT(
                    window.posState.cart,
                    customer,
                    window.posState.currentDiscount || 0
                );
            }
            
            // Fallback calculation
            return this.calculateVATTotalsFallback(customer);
            
        } catch (error) {
            console.error('Error calculating VAT totals:', error);
            return {
                totalVAT: 0,
                vatByRate: {},
                taxableAmount: 0,
                nonTaxableAmount: 0,
                exemptedAmount: 0,
                zeroRatedAmount: 0,
                subtotal: 0,
                discount: 0,
                totalWithVAT: 0,
                customerVatStatus: 'vatable',
                isVatApplicable: true,
                isZeroRated: false,
                isVatable: true,
                isExempted: false,
                vatNotes: ['Error calculating VAT']
            };
        }
    }
    
//////

// In public/js/pos/modules/cart.js
static calculateVATTotalsFallback(customer = null) {
    try {
        // Use centralized VAT service if available
        if (window.VATService && window.VATService.calculateCartVAT) {
            const vatData = window.VATService.calculateCartVAT(
                window.posState.cart,
                customer,
                window.posState.currentDiscount || 0
            );
            
            console.log('VAT calculated by VATService:', vatData);
            return vatData;
        }
        
        // Fallback calculation (simplified)
        return this.simpleVATCalculationFallback(customer);
        
    } catch (error) {
        console.error('Error calculating VAT totals:', error);
        return this.getEmptyVATData(customer);
    }
}

static simpleVATCalculationFallback(customer = null) {
    const customerStatus = customer?.vat_status || 'vatable';
    const isExempted = customerStatus === 'exempted';
    const isZeroRated = customerStatus === 'zero_rated';
    
    let subtotal = 0;
    let totalVAT = 0;
    
    // Simple calculation: if exempted, no VAT at all
    if (isExempted) {
        window.posState.cart.forEach(item => {
            const price = item.price || 0;
            const quantity = item.quantity || 1;
            subtotal += price * quantity;
        });
        
        return {
            subtotal: subtotal,
            totalVAT: 0,
            totalWithVAT: subtotal,
            customerVatStatus: 'exempted',
            isExempted: true,
            vatNotes: ['VAT Exempted Customer - No VAT Applied']
        };
    }
    
    // For zero-rated or vatable, calculate normally
    window.posState.cart.forEach(item => {
        const price = item.price || 0;
        const quantity = item.quantity || 1;
        const itemTotal = price * quantity;
        subtotal += itemTotal;
        
        // Apply VAT if product has VAT
        if (item.has_vat || item.is_vatable) {
            const taxRate = item.tax_rate || 16;
            const itemVAT = (itemTotal * taxRate) / 100;
            totalVAT += itemVAT;
        }
    });
    
    // For zero-rated customers, VAT is 0
    if (isZeroRated) {
        totalVAT = 0;
    }
    
    const discount = window.posState.currentDiscount || 0;
    const totalWithVAT = Math.max(0, (subtotal - discount) + totalVAT);
    
    return {
        subtotal: subtotal,
        totalVAT: totalVAT,
        totalWithVAT: totalWithVAT,
        customerVatStatus: customerStatus,
        isZeroRated: isZeroRated,
        isExempted: false,
        vatNotes: [isZeroRated ? 'Zero-Rated Customer' : 'VATable Customer']
    };
}
/////
    
    static saveCart() {
        try {
            if (!window.posState) return;
            
            // Save to localStorage
            localStorage.setItem('pos_cart', JSON.stringify(window.posState.cart));
            
            if (window.posState.currentCustomer) {
                localStorage.setItem('pos_current_customer', JSON.stringify(window.posState.currentCustomer));
            }
            
            localStorage.setItem('pos_current_discount', (window.posState.currentDiscount || 0).toString());
            
        } catch (error) {
            console.error('Error saving cart:', error);
        }
    }
    
    // Safe utility methods
    static safeToast(message, type = 'info') {
        try {
            if (window.POSUtils && typeof window.POSUtils.showToast === 'function') {
                window.POSUtils.showToast(message, type);
            } else if (window.safeToast) {
                window.safeToast(message, type);
            } else {
                console.log(`💬 ${type.toUpperCase()}: ${message}`);
                
                // Fallback toast
                const toastContainer = document.getElementById('toastContainer');
                if (!toastContainer) return;
                
                const toastId = 'toast-' + Date.now();
                const toastHtml = `
                    <div id="${toastId}" class="toast align-items-center border-0 bg-${type === 'error' ? 'danger' : type === 'success' ? 'success' : type === 'warning' ? 'warning' : 'info'} text-white" role="alert">
                        <div class="d-flex">
                            <div class="toast-body">${message}</div>
                            <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="document.getElementById('${toastId}').remove()"></button>
                        </div>
                    </div>
                `;
                
                toastContainer.insertAdjacentHTML('beforeend', toastHtml);
                
                // Auto-remove after 3 seconds
                setTimeout(() => {
                    const toast = document.getElementById(toastId);
                    if (toast) toast.remove();
                }, 3000);
            }
        } catch (error) {
            console.error('Error showing toast:', error);
        }
    }
    
    static safePlaySound(soundName) {
        try {
            if (window.POSUtils && typeof window.POSUtils.playSound === 'function') {
                window.POSUtils.playSound(soundName);
            }
        } catch (error) {
            // Silently fail for sound playback
        }
    }
    
    static getCartTotal() {
        try {
            const vatData = this.calculateVATTotals();
            return vatData.totalWithVAT || 0;
        } catch (error) {
            console.error('Error getting cart total:', error);
            return 0;
        }
    }
    
    static getItemCount() {
        try {
            if (window.posState && Array.isArray(window.posState.cart)) {
                return window.posState.cart.length;
            }
            return 0;
        } catch (error) {
            console.error('Error getting item count:', error);
            return 0;
        }
    }
    
    static getVATBreakdown() {
        try {
            const vatData = this.calculateVATTotals();
            return {
                summary: {
                    subtotal: vatData.subtotal,
                    taxableAmount: vatData.taxableAmount,
                    nonTaxableAmount: vatData.nonTaxableAmount,
                    totalVAT: vatData.totalVAT,
                    discount: window.posState?.currentDiscount || 0
                },
                byRate: vatData.vatByRate
            };
        } catch (error) {
            console.error('Error getting VAT breakdown:', error);
            return {
                summary: {
                    subtotal: 0,
                    taxableAmount: 0,
                    nonTaxableAmount: 0,
                    totalVAT: 0,
                    discount: 0
                },
                byRate: {}
            };
        }
    }
    
    static applyCustomerDiscount(discountPercent) {
        try {
            if (!window.posState) return false;
            
            const vatData = this.calculateVATTotals();
            const subtotal = vatData.subtotal || 0;
            const discountAmount = (subtotal * discountPercent) / 100;
            window.posState.currentDiscount = discountAmount;
            
            this.updateDisplay();
            this.saveCart();
            
            this.safeToast(`Customer discount applied: ${discountPercent}%`, 'success');
            return true;
        } catch (error) {
            console.error('Error applying customer discount:', error);
            return false;
        }
    }
    
    static applyLoyaltyDiscount(discountAmount) {
        try {
            if (!window.posState) return false;
            
            const currentTotal = this.getCartTotal();
            const maxDiscount = Math.min(discountAmount, currentTotal * 0.5); // Max 50% of cart total
            
            window.posState.currentDiscount = (window.posState.currentDiscount || 0) + maxDiscount;
            
            this.updateDisplay();
            this.saveCart();
            
            this.safeToast(`Loyalty discount applied: KES ${maxDiscount.toFixed(2)}`, 'success');
            return true;
        } catch (error) {
            console.error('Error applying loyalty discount:', error);
            return false;
        }
    }
    
    static removeDiscount() {
        try {
            if (!window.posState) return false;
            
            window.posState.currentDiscount = 0;
            this.updateDisplay();
            this.saveCart();
            
            this.safeToast('Discount removed', 'info');
            return true;
        } catch (error) {
            console.error('Error removing discount:', error);
            return false;
        }
    }
    
    static setCustomer(customer) {
        try {
            if (!window.posState) return false;
            
            window.posState.currentCustomer = customer;
            this.updateDisplay();
            this.saveCart();
            
            return true;
        } catch (error) {
            console.error('Error setting customer:', error);
            return false;
        }
    }
}

// Safe global exports with fallbacks
window.POSCart = POSCart;

// Safe wrapper functions
window.addToCart = function(product, quantity) {
    try {
        if (window.POSCart && typeof window.POSCart.add === 'function') {
            return window.POSCart.add(product, quantity);
        }
        console.error('POSCart.add not available');
        return false;
    } catch (error) {
        console.error('Error in addToCart:', error);
        return false;
    }
};

window.updateQuantity = function(index, change, newValue) {
    try {
        if (window.POSCart && typeof window.POSCart.updateQuantity === 'function') {
            return window.POSCart.updateQuantity(index, change, newValue);
        }
        console.error('POSCart.updateQuantity not available');
        return false;
    } catch (error) {
        console.error('Error in updateQuantity:', error);
        return false;
    }
};

window.removeFromCart = function(index) {
    try {
        if (window.POSCart && typeof window.POSCart.remove === 'function') {
            return window.POSCart.remove(index);
        }
        console.error('POSCart.remove not available');
        return false;
    } catch (error) {
        console.error('Error in removeFromCart:', error);
        return false;
    }
};

window.clearCart = function() {
    try {
        if (window.POSCart && typeof window.POSCart.clear === 'function') {
            return window.POSCart.clear();
        }
        console.error('POSCart.clear not available');
        return false;
    } catch (error) {
        console.error('Error in clearCart:', error);
        return false;
    }
};

window.getCartTotal = function() {
    try {
        if (window.POSCart && typeof window.POSCart.getCartTotal === 'function') {
            return window.POSCart.getCartTotal();
        }
        console.error('POSCart.getCartTotal not available');
        return 0;
    } catch (error) {
        console.error('Error in getCartTotal:', error);
        return 0;
    }
};

window.getVATBreakdown = function() {
    try {
        if (window.POSCart && typeof window.POSCart.getVATBreakdown === 'function') {
            return window.POSCart.getVATBreakdown();
        }
        console.error('POSCart.getVATBreakdown not available');
        return { summary: {}, byRate: {} };
    } catch (error) {
        console.error('Error in getVATBreakdown:', error);
        return { summary: {}, byRate: {} };
    }
};

window.applyCustomerDiscount = function(discountPercent) {
    try {
        if (window.POSCart && typeof window.POSCart.applyCustomerDiscount === 'function') {
            return window.POSCart.applyCustomerDiscount(discountPercent);
        }
        console.error('POSCart.applyCustomerDiscount not available');
        return false;
    } catch (error) {
        console.error('Error in applyCustomerDiscount:', error);
        return false;
    }
};

window.applyLoyaltyDiscount = function(discountAmount) {
    try {
        if (window.POSCart && typeof window.POSCart.applyLoyaltyDiscount === 'function') {
            return window.POSCart.applyLoyaltyDiscount(discountAmount);
        }
        console.error('POSCart.applyLoyaltyDiscount not available');
        return false;
    } catch (error) {
        console.error('Error in applyLoyaltyDiscount:', error);
        return false;
    }
};

window.removeDiscount = function() {
    try {
        if (window.POSCart && typeof window.POSCart.removeDiscount === 'function') {
            return window.POSCart.removeDiscount();
        }
        console.error('POSCart.removeDiscount not available');
        return false;
    } catch (error) {
        console.error('Error in removeDiscount:', error);
        return false;
    }
};

// Initialize when safe
setTimeout(() => {
    if (window.POSCart && typeof window.POSCart.init === 'function') {
        try {
            window.POSCart.init();
            console.log('✅ POS Cart initialized with VAT Support');
        } catch (error) {
            console.error('Failed to initialize POS Cart:', error);
        }
    }
}, 100);

console.log('🛒 POS Cart module loaded with VAT Support');