// public/js/pos/module-manager.js
class POSModuleManager {
    constructor() {
        this.modules = new Map();
        this.loadedModules = new Set();
        this.dependencies = {
            'POSUtils': [],
            'posState': ['POSUtils'],
            'POSCart': ['POSUtils', 'posState'],
            'POSProducts': ['POSUtils', 'posState'],
            'POSCustomers': ['POSUtils', 'posState'],
            'POSScanner': ['POSUtils'],
            'POSDiscounts': ['POSUtils', 'posState', 'POSCart'],
            'POSHeldSales': ['POSUtils', 'posState'],
            'POSStats': ['POSUtils'],
            'POSReceipts': ['POSUtils', 'posState'],
            'PaymentManager': ['POSUtils', 'posState'],
            'POSPayments': ['POSUtils', 'posState', 'POSCart', 'PaymentManager']
        };
    }
    
    async loadModule(name) {
        // If already loaded
        if (this.loadedModules.has(name)) {
            return window[name];
        }
        
        // If already loading
        if (this.loadingPromises && this.loadingPromises.has(name)) {
            return this.loadingPromises.get(name);
        }
        
        // Create loading promises map if it doesn't exist
        if (!this.loadingPromises) {
            this.loadingPromises = new Map();
        }
        
        // Create loading promise
        const loadPromise = (async () => {
            try {
                console.log(`📦 Loading module: ${name}`);
                
                // Load dependencies first
                const deps = this.dependencies[name] || [];
                for (const dep of deps) {
                    await this.loadModule(dep);
                }
                
                // Get module from window (should be loaded by script tag)
                const module = window[name];
                
                if (!module) {
                    // Try to create default instance for missing modules
                    if (name === 'PaymentManager') {
                        console.warn('Creating default PaymentManager');
                        window.PaymentManager = {
                            init: () => Promise.resolve(),
                            processPayment: () => Promise.resolve({ success: true }),
                            loadPaymentMethods: () => Promise.resolve([])
                        };
                        return window.PaymentManager;
                    }
                    throw new Error(`Module ${name} not found`);
                }
                
                // Initialize module if it has init method
                if (typeof module.init === 'function') {
                    await Promise.resolve(module.init());
                } else if (typeof module === 'object' && module.init) {
                    await Promise.resolve(module.init());
                } else if (typeof module === 'function' && module.prototype && module.prototype.init) {
                    const instance = new module();
                    await Promise.resolve(instance.init());
                    window[name] = instance;
                } else {
                    // Just return the module if it doesn't need initialization
                    console.log(`✅ Module ${name} loaded (no init required)`);
                }
                
                this.loadedModules.add(name);
                this.loadingPromises.delete(name);
                console.log(`✅ Module loaded: ${name}`);
                
                return window[name];
                
            } catch (error) {
                console.error(`❌ Failed to load module ${name}:`, error);
                this.loadedModules.delete(name);
                this.loadingPromises.delete(name);
                throw error;
            }
        })();
        
        this.loadingPromises.set(name, loadPromise);
        return loadPromise;
    }
    
    async loadAll() {
        const modulesToLoad = [
            'POSUtils',
            'posState',
            'POSCart',
            'POSProducts',
            'POSCustomers',
            'POSScanner',
            'POSDiscounts',
            'POSHeldSales',
            'POSStats',
            'POSReceipts',
            'PaymentManager',
            'POSPayments'
        ];
        
        const results = [];
        for (const moduleName of modulesToLoad) {
            try {
                const module = await this.loadModule(moduleName);
                results.push({ name: moduleName, success: true, module });
            } catch (error) {
                console.warn(`⚠️ Module ${moduleName} failed:`, error.message);
                results.push({ name: moduleName, success: false, error: error.message });
                
                // Create fallback for critical modules
                if (moduleName === 'PaymentManager') {
                    window.PaymentManager = {
                        init: () => Promise.resolve(),
                        processPayment: async (paymentData) => {
                            console.log('Fallback payment processing:', paymentData);
                            return { success: true, transaction_id: 'FALLBACK-' + Date.now() };
                        },
                        loadPaymentMethods: () => Promise.resolve([
                            { code: 'cash', name: 'Cash', type: 'cash' },
                            { code: 'mpesa', name: 'M-Pesa', type: 'mobile' }
                        ])
                    };
                }
            }
        }
        
        return results;
    }
    
    isModuleLoaded(moduleName) {
        return this.loadedModules.has(moduleName);
    }
    
    getModuleStatus(moduleName) {
        return this.loadedModules.has(moduleName) ? 'loaded' : 
               (this.loadingPromises?.has(moduleName) ? 'loading' : 'not-loaded');
    }
}

// Create global module manager
window.POSModuleManager = new POSModuleManager();
console.log('✅ POSModuleManager initialized');