// =========================================================
// POS State Management - Fixed with proper cart initialization
// =========================================================

class POSState {
    constructor() {
        console.log('🔄 Initializing POS State');
        
        // Initialize with proper defaults
        this.cart = []; // Ensure cart is always an array
        this.currentCustomer = null;
        this.currentDiscount = 0;
        this.cartVATData = null;
        this.customerVatStatus = 'vatable'; // Default VAT status
        
        // Load saved state
        this.loadSavedState();
        
        // Bind methods
        this.getCartSubtotal = this.getCartSubtotal.bind(this);
        this.getCartTotal = this.getCartTotal.bind(this);
        this.clearCart = this.clearCart.bind(this);
        this.saveCart = this.saveCart.bind(this);
        this.loadSavedState = this.loadSavedState.bind(this);
        this.addItem = this.addItem.bind(this);
        this.removeItem = this.removeItem.bind(this);
        this.updateQuantity = this.updateQuantity.bind(this);
        
        // Export to window
        window.posState = this;
        
        console.log('✅ POS State initialized');
    }

    // Load saved state from localStorage
    loadSavedState() {
        try {
            // Load cart - ensure it's always an array
            const savedCart = localStorage.getItem('pos_cart');
            if (savedCart) {
                const parsedCart = JSON.parse(savedCart);
                this.cart = Array.isArray(parsedCart) ? parsedCart : [];
            } else {
                this.cart = []; // Default to empty array
            }
            
            // Load customer
            const savedCustomer = localStorage.getItem('pos_current_customer');
            if (savedCustomer) {
                this.currentCustomer = JSON.parse(savedCustomer);
            }
            
            // Load discount
            const savedDiscount = localStorage.getItem('pos_current_discount');
            if (savedDiscount) {
                this.currentDiscount = parseFloat(savedDiscount) || 0;
            }
            
            // Load VAT status
            const savedVatStatus = localStorage.getItem('pos_customer_vat_status');
            if (savedVatStatus) {
                this.customerVatStatus = savedVatStatus;
            }
            
            console.log('📥 Loaded saved state:', {
                cartItems: this.cart.length,
                customer: this.currentCustomer?.name || 'None',
                discount: this.currentDiscount
            });
            
        } catch (error) {
            console.error('❌ Error loading saved state:', error);
            // Reset to defaults
            this.cart = [];
            this.currentCustomer = null;
            this.currentDiscount = 0;
            this.customerVatStatus = 'vatable';
        }
    }

    // Get cart subtotal
    getCartSubtotal() {
        try {
            if (!Array.isArray(this.cart) || this.cart.length === 0) {
                return 0;
            }
            
            return this.cart.reduce((total, item) => {
                const price = parseFloat(item.price) || 0;
                const quantity = parseInt(item.quantity) || parseInt(item.qty) || 1;
                return total + (price * quantity);
            }, 0);
            
        } catch (error) {
            console.error('❌ Error calculating cart subtotal:', error);
            return 0;
        }
    }

    // Get cart total with VAT and discount
    getCartTotal() {
        try {
            const subtotal = this.getCartSubtotal();
            const discount = this.currentDiscount || 0;
            
            // Calculate VAT based on customer status
            let vatAmount = 0;
            
            if (this.cartVATData && this.cartVATData.totalVAT) {
                vatAmount = this.cartVATData.totalVAT;
            } else if (window.VATService) {
                // Calculate VAT using VAT service
                const vatData = window.VATService.calculateCartVAT(this.cart, this.currentCustomer, discount);
                vatAmount = vatData.totalVAT || 0;
                this.cartVATData = vatData;
            } else {
                // Simple VAT calculation (16% default)
                vatAmount = (subtotal - discount) * 0.16;
            }
            
            const total = Math.max(0, (subtotal - discount) + vatAmount);
            
            console.log('💰 Cart totals:', {
                subtotal: subtotal,
                discount: discount,
                vatAmount: vatAmount,
                total: total
            });
            
            return total;
            
        } catch (error) {
            console.error('❌ Error calculating cart total:', error);
            return 0;
        }
    }

    // Clear cart
    clearCart() {
        console.log('🧹 Clearing cart...');
        
        this.cart = [];
        this.currentDiscount = 0;
        this.cartVATData = null;
        
        // Clear localStorage
        localStorage.removeItem('pos_cart');
        localStorage.removeItem('pos_current_discount');
        localStorage.removeItem('pos_cart_vat_data');
        
        // Keep customer (optional)
        // localStorage.removeItem('pos_current_customer');
        // this.currentCustomer = null;
        
        console.log('✅ Cart cleared');
        
        // Notify other modules
        if (window.POSCart && window.POSCart.updateDisplay) {
            window.POSCart.updateDisplay();
        }
    }

    // Save cart state
    saveCart() {
        try {
            localStorage.setItem('pos_cart', JSON.stringify(this.cart));
            
            if (this.currentCustomer) {
                localStorage.setItem('pos_current_customer', JSON.stringify(this.currentCustomer));
            }
            
            localStorage.setItem('pos_current_discount', this.currentDiscount.toString());
            
            if (this.cartVATData) {
                localStorage.setItem('pos_cart_vat_data', JSON.stringify(this.cartVATData));
            }
            
            console.log('💾 Cart saved:', {
                items: this.cart.length,
                discount: this.currentDiscount
            });
            
        } catch (error) {
            console.error('❌ Error saving cart:', error);
        }
    }

    // Add item to cart
    addItem(product, quantity = 1) {
        try {
            if (!product || !product.id) {
                console.error('❌ Invalid product:', product);
                return false;
            }
            
            if (!Array.isArray(this.cart)) {
                this.cart = [];
            }
            
            // Find existing item
            const existingIndex = this.cart.findIndex(item => 
                item.id === product.id || 
                item.product_id === product.id ||
                (item.barcode && product.barcode && item.barcode === product.barcode)
            );
            
            if (existingIndex > -1) {
                // Update existing item
                const currentQty = parseInt(this.cart[existingIndex].quantity) || parseInt(this.cart[existingIndex].qty) || 0;
                this.cart[existingIndex].quantity = currentQty + quantity;
                this.cart[existingIndex].qty = currentQty + quantity;
            } else {
                // Add new item
                const cartItem = {
                    id: product.id,
                    product_id: product.id,
                    name: product.name || 'Unknown Product',
                    sku: product.sku || '',
                    barcode: product.barcode || '',
                    price: parseFloat(product.price) || 0,
                    price_ex_tax: parseFloat(product.price_ex_tax) || parseFloat(product.price) || 0,
                    tax_rate: product.tax_rate || (product.has_vat ? 16 : 0),
                    has_vat: product.has_vat || product.is_vatable || false,
                    is_vatable: product.has_vat || product.is_vatable || false,
                    quantity: quantity,
                    qty: quantity,
                    stock: product.stock || 0,
                    category_id: product.category_id,
                    category_name: product.category_name,
                    image: product.image || product.image_url
                };
                
                this.cart.push(cartItem);
            }
            
            // Save to localStorage
            this.saveCart();
            
            console.log('➕ Item added:', {
                product: product.name,
                quantity: quantity,
                cartSize: this.cart.length
            });
            
            return true;
            
        } catch (error) {
            console.error('❌ Error adding item to cart:', error);
            return false;
        }
    }

    // Remove item from cart
    removeItem(productId) {
        try {
            if (!Array.isArray(this.cart)) {
                this.cart = [];
                return false;
            }
            
            const initialLength = this.cart.length;
            this.cart = this.cart.filter(item => 
                item.id !== productId && 
                item.product_id !== productId
            );
            
            const removed = initialLength !== this.cart.length;
            
            if (removed) {
                this.saveCart();
                console.log('➖ Item removed:', productId);
            }
            
            return removed;
            
        } catch (error) {
            console.error('❌ Error removing item from cart:', error);
            return false;
        }
    }

    // Update item quantity
    updateQuantity(productId, quantity) {
        try {
            if (!Array.isArray(this.cart)) {
                this.cart = [];
                return false;
            }
            
            if (quantity <= 0) {
                return this.removeItem(productId);
            }
            
            const itemIndex = this.cart.findIndex(item => 
                item.id === productId || 
                item.product_id === productId
            );
            
            if (itemIndex > -1) {
                this.cart[itemIndex].quantity = quantity;
                this.cart[itemIndex].qty = quantity;
                this.saveCart();
                
                console.log('✏️ Quantity updated:', {
                    productId: productId,
                    quantity: quantity
                });
                
                return true;
            }
            
            return false;
            
        } catch (error) {
            console.error('❌ Error updating quantity:', error);
            return false;
        }
    }

    // Get item count
    getItemCount() {
        try {
            if (!Array.isArray(this.cart)) {
                return 0;
            }
            
            return this.cart.reduce((total, item) => {
                const qty = parseInt(item.quantity) || parseInt(item.qty) || 1;
                return total + qty;
            }, 0);
            
        } catch (error) {
            console.error('❌ Error counting items:', error);
            return 0;
        }
    }

    // Set customer with VAT status
    setCustomer(customer) {
        this.currentCustomer = customer;
        
        // Update VAT status
        this.customerVatStatus = customer?.vat_status || 'vatable';
        
        // Save to localStorage
        localStorage.setItem('pos_current_customer', JSON.stringify(customer));
        localStorage.setItem('pos_customer_vat_status', this.customerVatStatus);
        
        console.log('👤 Customer set:', {
            name: customer?.name,
            vatStatus: this.customerVatStatus
        });
        
        // Recalculate VAT
        this.recalculateVAT();
    }

    // Recalculate VAT for entire cart
    recalculateVAT() {
        try {
            if (!Array.isArray(this.cart) || this.cart.length === 0) {
                this.cartVATData = null;
                return;
            }
            
            if (window.VATService && window.VATService.calculateCartVAT) {
                this.cartVATData = window.VATService.calculateCartVAT(
                    this.cart,
                    this.currentCustomer,
                    this.currentDiscount
                );
                
                console.log('🧾 VAT recalculated:', this.cartVATData);
            }
            
        } catch (error) {
            console.error('❌ Error recalculating VAT:', error);
            this.cartVATData = null;
        }
    }

    // Apply discount
    applyDiscount(amount, type = 'amount') {
        try {
            if (type === 'percentage') {
                const subtotal = this.getCartSubtotal();
                this.currentDiscount = subtotal * (Math.min(amount, 100) / 100);
            } else {
                const subtotal = this.getCartSubtotal();
                this.currentDiscount = Math.min(amount, subtotal);
            }
            
            localStorage.setItem('pos_current_discount', this.currentDiscount.toString());
            
            // Recalculate VAT with discount
            this.recalculateVAT();
            
            console.log('🎫 Discount applied:', {
                amount: this.currentDiscount,
                type: type
            });
            
            return true;
            
        } catch (error) {
            console.error('❌ Error applying discount:', error);
            return false;
        }
    }

    // Remove discount
    removeDiscount() {
        this.currentDiscount = 0;
        localStorage.removeItem('pos_current_discount');
        
        // Recalculate VAT without discount
        this.recalculateVAT();
        
        console.log('🎫 Discount removed');
    }
}

// Initialize immediately
if (typeof window !== 'undefined') {
    // Check if posState already exists
    if (!window.posState) {
        window.posState = new POSState();
    } else {
        // Ensure existing posState has cart as array
        if (!Array.isArray(window.posState.cart)) {
            window.posState.cart = [];
        }
        
        // Ensure methods exist
        const proto = POSState.prototype;
        Object.getOwnPropertyNames(proto).forEach(method => {
            if (method !== 'constructor' && typeof proto[method] === 'function') {
                if (!window.posState[method]) {
                    window.posState[method] = proto[method].bind(window.posState);
                }
            }
        });
    }
}

console.log('✅ POS State module loaded');