// public/js/pos/core/monitor.js - FIXED
class POSMonitor {
    constructor() {
        this.metrics = {
            startupTime: null,
            moduleLoadTimes: {},
            transactionCount: 0,
            errorCount: 0,
            lastTransactionTime: null
        };
        
        this.startTime = performance.now();
        this.setupMonitoring();
    }
    
    setupMonitoring() {
        // Track module load times
        const originalSetTimeout = window.setTimeout;
        window.setTimeout = (callback, delay, ...args) => {
            const stack = new Error().stack;
            return originalSetTimeout(() => {
                try {
                    callback(...args);
                } catch (error) {
                    this.recordError(error, { type: 'setTimeout', stack });
                    throw error;
                }
            }, delay);
        };
        
        // Track fetch requests
        const originalFetch = window.fetch;
        window.fetch = (...args) => {
            const start = performance.now();
            return originalFetch(...args).then(response => {
                const duration = performance.now() - start;
                this.recordMetric('fetch_duration', duration, { url: args[0] });
                return response;
            }).catch(error => {
                this.recordError(error, { type: 'fetch', args });
                throw error;
            });
        };
        
        // Periodic health check
        setInterval(() => this.healthCheck(), 60000);
    }
    
    recordMetric(name, value, tags = {}) {
        this.metrics[name] = this.metrics[name] || [];
        this.metrics[name].push({
            value,
            tags,
            timestamp: Date.now()
        });
        
        // Keep only last 1000 metrics
        if (this.metrics[name].length > 1000) {
            this.metrics[name].shift();
        }
    }
    
    recordError(error, context = {}) {
        this.metrics.errorCount++;
        window.POSErrorHandler?.capture(error, context);
    }
    
    /**
     * Health check with proper CSRF token handling
     */
    healthCheck() {
        const health = {
            timestamp: Date.now(),
            modulesLoaded: Object.keys(window).filter(key => 
                key.startsWith('POS') || key === 'posState'
            ).length,
            cartItems: window.posState?.cart?.length || 0,
            memory: performance.memory ? {
                usedMB: Math.round(performance.memory.usedJSHeapSize / 1048576),
                totalMB: Math.round(performance.memory.totalJSHeapSize / 1048576)
            } : null,
            online: navigator.onLine,
            userAgent: navigator.userAgent,
            screenResolution: `${window.screen.width}x${window.screen.height}`,
            connectionType: navigator.connection?.effectiveType || 'unknown'
        };
        
        // Log to console
        console.log('🏥 POS Health:', health);
        
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
        
        // Prepare headers with CSRF token
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        };
        
        // Add CSRF token if available (for POST requests)
        if (csrfToken) {
            headers['X-CSRF-TOKEN'] = csrfToken;
        }
        
        // Send to server using GET first (no CSRF needed)
        fetch('/api/pos/health-check', {
            method: 'GET',
            headers: headers,
            credentials: 'same-origin' // Important for sending cookies/session
        })
        .then(response => {
            if (!response.ok) {
                console.warn('Health check GET failed:', response.status);
                // Fallback to POST with token if GET fails
                return this.sendHealthCheckPOST(health, csrfToken);
            }
            return response.json();
        })
        .then(data => {
            if (data && data.status === 'healthy') {
                console.log('✅ Health check successful');
            }
        })
        .catch(error => {
            console.warn('Health check failed:', error.message);
            // Last resort: try POST without CSRF (might work if route is exempt)
            this.sendHealthCheckPOST(health, csrfToken).catch(() => {
                // Silently fail - health check is not critical
            });
        });
        
        return health;
    }
    
    /**
     * Send health check via POST with CSRF token
     */
    async sendHealthCheckPOST(health, csrfToken) {
        if (!csrfToken) {
            console.warn('No CSRF token available for POST health check');
            return Promise.reject('No CSRF token');
        }
        
        const headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        };
        
        return fetch('/api/pos/health', {
            method: 'POST',
            headers: headers,
            credentials: 'same-origin',
            body: JSON.stringify(health)
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`POST health check failed: ${response.status}`);
            }
            return response.json();
        });
    }
    
    /**
     * Get system health status
     */
    getSystemHealth() {
        return {
            metrics: this.metrics,
            uptime: performance.now() - this.startTime,
            errors: this.metrics.errorCount,
            transactions: this.metrics.transactionCount,
            memory: performance.memory ? {
                usedMB: Math.round(performance.memory.usedJSHeapSize / 1048576),
                totalMB: Math.round(performance.memory.totalJSHeapSize / 1048576),
                limitMB: Math.round(performance.memory.jsHeapSizeLimit / 1048576)
            } : null
        };
    }
}

// Initialize
window.POSMonitor = new POSMonitor();