// public/js/pos/core/error-handler.js
class POSErrorHandler {
    constructor() {
        this.errors = [];
        this.maxErrors = 100;
        this.reportEndpoint = '/api/log-error';
    }

    capture(error, context = {}) {
        const errorData = {
            id: Date.now(),
            message: error.message || String(error),
            stack: error.stack,
            timestamp: new Date().toISOString(),
            url: window.location.href,
            userAgent: navigator.userAgent,
            context: context
        };

        // Store locally
        this.errors.unshift(errorData);
        if (this.errors.length > this.maxErrors) {
            this.errors.pop();
        }

        // Log to console
        console.error('🚨 POS Error:', errorData);

        // Send to server if not in development
            if (!window.location.hostname.includes('localhost') && 
                !window.location.hostname.includes('127.0.0.1')) {
                this.reportToServer(errorData);
            }

        // Show user-friendly message
        this.showUserMessage(error);

        return errorData;
    }

    showUserMessage(error) {
        const message = this.getFriendlyMessage(error);
        
        // Use toast if available
        if (window.safeToast) {
            window.safeToast(message, 'error');
        } else if (window.POSUtils?.showToast) {
            window.POSUtils.showToast(message, 'error');
        } else {
            alert(`Error: ${message}`);
        }
    }

    getFriendlyMessage(error) {
        const msg = error.message || String(error);
        
        if (msg.includes('clearCart is not a function')) {
            return 'Cart system error. Please refresh the page.';
        } else if (msg.includes('printLastReceipt')) {
            return 'Receipt printing unavailable. Try again.';
        } else if (msg.includes('SQL') || msg.includes('database')) {
            return 'Database error. Please contact support.';
        } else if (msg.includes('fetch') || msg.includes('network')) {
            return 'Network error. Check your connection.';
        } else if (msg.includes('VAT') || msg.includes('tax')) {
            return 'Tax calculation error. Please verify prices.';
        } else {
            return 'System error occurred. Please try again.';
        }
    }

    reportToServer(errorData) {
        if (!navigator.onLine) return;

        fetch(this.reportEndpoint, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify(errorData)
        }).catch(() => {
            // Silently fail
        });
    }

    clearOldErrors() {
        const oneHourAgo = Date.now() - (60 * 60 * 1000);
        this.errors = this.errors.filter(err => err.timestamp > oneHourAgo);
    }
}

// Initialize globally
window.POSErrorHandler = new POSErrorHandler();

// Global error handler
window.onerror = function(msg, url, lineNo, columnNo, error) {
    window.POSErrorHandler.capture(error || { message: msg }, {
        url, lineNo, columnNo
    });
    return false;
};

// Unhandled promise rejections
window.addEventListener('unhandledrejection', function(event) {
    window.POSErrorHandler.capture(event.reason, {
        type: 'unhandled_rejection'
    });
});