<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up()
    {
        // Update mpesa_credentials table
        Schema::table('mpesa_credentials', function (Blueprint $table) {
            // Add new columns
            $table->string('validation_url')->nullable()->after('callback_url');
            $table->string('confirmation_url')->nullable()->after('validation_url');
            $table->timestamp('urls_registered_at')->nullable()->after('last_failure');
        });
        
        // Change is_active default to false using raw SQL (for MariaDB/MySQL)
        DB::statement("ALTER TABLE mpesa_credentials ALTER is_active SET DEFAULT 0");

        // Update mpesa_transactions table - FIRST drop indexes that might interfere
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            // Drop indexes that might cause issues
            $table->dropIndex('mpesa_transactions_phone_status_index');
        });
        
        // Now modify the columns
        DB::statement("ALTER TABLE mpesa_transactions MODIFY reference VARCHAR(255) NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY phone VARCHAR(20) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY amount DECIMAL(12,2) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY expected_amount DECIMAL(12,2) NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY transaction_date DATETIME NOT NULL");
        
        // Add all new columns
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            // Add new columns
            $table->foreignId('credential_id')->nullable()->after('customer_id')
                  ->constrained('mpesa_credentials')->nullOnDelete();
            $table->timestamp('matched_at')->nullable()->after('credential_id');
            $table->foreignId('matched_by')->nullable()->after('matched_at')
                  ->constrained('users')->nullOnDelete();
            $table->enum('match_type', ['exact', 'partial', 'overpayment', 'manual', 'auto', 'auto_reference'])
                  ->nullable()->after('matched_by');
            $table->text('match_notes')->nullable()->after('match_type');
            $table->timestamp('reversed_at')->nullable()->after('match_notes');
            $table->foreignId('reversed_by')->nullable()->after('reversed_at')
                  ->constrained('users')->nullOnDelete();
            $table->text('reversal_reason')->nullable()->after('reversed_by');
            $table->decimal('refund_amount', 12, 2)->nullable()->after('reversal_reason');
            $table->decimal('amount_difference', 12, 2)->nullable()->after('refund_amount');
            $table->string('transaction_type')->nullable()->after('amount_difference');
            $table->string('mpesa_receipt_number')->nullable()->after('transaction_type');
            $table->integer('result_code')->nullable()->after('mpesa_receipt_number');
            $table->text('result_desc')->nullable()->after('result_code');
            $table->string('account_reference')->nullable()->after('result_desc');
            $table->string('third_party_trans_id')->nullable()->after('account_reference');
            $table->string('transaction_status')->nullable()->after('third_party_trans_id');
            $table->softDeletes();
        });
        
        // Change status to ENUM - need to handle this carefully
        // First, update any invalid status values
        DB::update("UPDATE mpesa_transactions SET status = 'pending' WHERE status NOT IN ('pending', 'completed', 'failed', 'reversed', 'matched')");
        
        // Then change the column type
        DB::statement("ALTER TABLE mpesa_transactions 
            CHANGE status status ENUM('pending', 'completed', 'failed', 'reversed', 'matched') 
            NOT NULL DEFAULT 'pending'");
            
        // Add new indexes
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            $table->index(['status', 'sale_id']);
            $table->index(['phone', 'transaction_date']);
            $table->index(['transaction_date', 'status']);
            $table->index(['credential_id']);
        });
        
        // Add indexes to mpesa_credentials
        Schema::table('mpesa_credentials', function (Blueprint $table) {
            $table->index(['is_active', 'environment']);
            $table->unique(['business_shortcode', 'environment']);
        });
    }

    public function down()
    {
        // Remove indexes from mpesa_credentials
        Schema::table('mpesa_credentials', function (Blueprint $table) {
            $table->dropUnique(['business_shortcode', 'environment']);
            $table->dropIndex(['is_active', 'environment']);
        });
        
        // Remove new columns from mpesa_credentials
        Schema::table('mpesa_credentials', function (Blueprint $table) {
            $table->dropColumn(['validation_url', 'confirmation_url', 'urls_registered_at']);
        });
        
        // Revert is_active default
        DB::statement("ALTER TABLE mpesa_credentials ALTER is_active SET DEFAULT 1");
        
        // Remove new indexes from mpesa_transactions
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            $table->dropIndex(['status', 'sale_id']);
            $table->dropIndex(['phone', 'transaction_date']);
            $table->dropIndex(['transaction_date', 'status']);
            $table->dropIndex(['credential_id']);
        });
        
        // Change status back to VARCHAR
        DB::statement("ALTER TABLE mpesa_transactions 
            CHANGE status status VARCHAR(255) NOT NULL DEFAULT 'pending'");
        
        // Remove new columns from mpesa_transactions
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            $table->dropSoftDeletes();
            $table->dropColumn([
                'credential_id',
                'matched_at',
                'matched_by',
                'match_type',
                'match_notes',
                'reversed_at',
                'reversed_by',
                'reversal_reason',
                'refund_amount',
                'amount_difference',
                'transaction_type',
                'mpesa_receipt_number',
                'result_code',
                'result_desc',
                'account_reference',
                'third_party_trans_id',
                'transaction_status',
            ]);
        });
        
        // Revert column changes
        DB::statement("ALTER TABLE mpesa_transactions MODIFY reference VARCHAR(255) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY phone VARCHAR(255) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY amount DECIMAL(10,2) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY expected_amount DECIMAL(10,2) NOT NULL");
        DB::statement("ALTER TABLE mpesa_transactions MODIFY transaction_date TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP");
        
        // Restore old index
        Schema::table('mpesa_transactions', function (Blueprint $table) {
            $table->index(['phone', 'status']);
        });
    }
};