<?php

namespace App\Traits;

use Carbon\Carbon;

trait InvoiceReceiptGenerator
{
    /**
     * Generate invoice number based on type and date
     */
    public function generateInvoiceNumber(string $type = 'INV'): string
    {
        $date = Carbon::now();
        $year = $date->format('Y');
        $month = $date->format('m');
        $day = $date->format('d');
        
        // Get sequence for today
        $sequence = $this->getTodaySequence($type);
        
        // Format: INV-YYYYMMDD-001
        return sprintf('%s-%s%s%s-%03d', 
            $type, 
            $year, 
            $month, 
            $day, 
            $sequence
        );
    }
    
    /**
     * Generate receipt number
     */
    public function generateReceiptNumber(): string
    {
        $date = Carbon::now();
        $year = $date->format('Y');
        $month = $date->format('m');
        $day = $date->format('d');
        
        // Get sequence for today
        $sequence = $this->getTodaySequence('RCPT');
        
        // Format: RCPT-YYYYMMDD-001
        return sprintf('RCPT-%s%s%s-%03d', 
            $year, 
            $month, 
            $day, 
            $sequence
        );
    }
    
    /**
     * Get today's sequence number for a given type
     */
    private function getTodaySequence(string $type): int
    {
        $today = Carbon::today();
        $tomorrow = Carbon::tomorrow();
        
        // For invoices
        if (in_array($type, ['INV', 'INVOICE'])) {
            $count = \App\Models\Sale::whereBetween('created_at', [$today, $tomorrow])
                ->whereNotNull('invoice_no')
                ->count();
        } 
        // For receipts
        else if ($type === 'RCPT') {
            $count = \App\Models\Sale::whereBetween('created_at', [$today, $tomorrow])
                ->whereNotNull('receipt_no')
                ->count();
        }
        // For credit notes
        else if ($type === 'CN') {
            $count = \App\Models\Sale::whereBetween('created_at', [$today, $tomorrow])
                ->where('invoice_no', 'LIKE', 'CN-%')
                ->count();
        }
        // For refunds
        else if ($type === 'REF') {
            $count = \App\Models\Sale::whereBetween('created_at', [$today, $tomorrow])
                ->where('invoice_no', 'LIKE', 'REF-%')
                ->count();
        }
        // Default
        else {
            $count = \App\Models\Sale::whereBetween('created_at', [$today, $tomorrow])->count();
        }
        
        return $count + 1;
    }
    
    /**
     * Generate credit note number
     */
    public function generateCreditNoteNumber(): string
    {
        return $this->generateInvoiceNumber('CN');
    }
    
    /**
     * Generate refund number
     */
    public function generateRefundNumber(): string
    {
        return $this->generateInvoiceNumber('REF');
    }
    
    /**
     * Generate proforma invoice number
     */
    public function generateProformaNumber(): string
    {
        return $this->generateInvoiceNumber('PRO');
    }
    
    /**
     * Get the display number (invoice if exists, otherwise receipt)
     */
    public function getDisplayNumber(\App\Models\Sale $sale): string
    {
        return $sale->invoice_no ?? $sale->receipt_no ?? 'N/A';
    }
}