<?php
namespace App\Traits;

trait ActivityLogger
{
    /**
     * Log an activity safely
     */
    protected function logActivity(string $description, array $properties = [], $subject = null, $causer = null): void
    {
        // Disable during high-load operations
        if ($this->shouldSkipLogging()) {
            return;
        }
        
        $activity = activity();
        
        if ($subject) {
            $activity->performedOn($subject);
        }
        
        if ($causer) {
            $activity->causedBy($causer);
        } elseif (auth()->check()) {
            $activity->causedBy(auth()->user());
        }
        
        // Add IP and user agent
        $properties = array_merge($properties, [
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
        
        // Limit property size
        $properties = $this->limitPropertySize($properties);
        
        $activity->withProperties($properties)
                ->log($description);
    }
    
    /**
     * Log a model creation
     */
    protected function logModelCreated($model, array $extraProperties = []): void
    {
        if (!$this->shouldLogModel($model)) {
            return;
        }
        
        $modelName = class_basename($model);
        $properties = array_merge([
            'attributes' => $this->getSafeModelAttributes($model),
        ], $extraProperties);
        
        $this->logActivity("created {$modelName}", $properties, $model);
    }
    
    /**
     * Log a model update
     */
    protected function logModelUpdated($model, array $extraProperties = []): void
    {
        if (!$this->shouldLogModel($model)) {
            return;
        }
        
        $modelName = class_basename($model);
        $properties = array_merge([
            'old' => $model->getOriginal(),
            'new' => $this->getSafeModelAttributes($model),
        ], $extraProperties);
        
        $this->logActivity("updated {$modelName}", $properties, $model);
    }
    
    /**
     * Log a model deletion
     */
    protected function logModelDeleted($model, array $extraProperties = []): void
    {
        if (!$this->shouldLogModel($model)) {
            return;
        }
        
        $modelName = class_basename($model);
        $properties = array_merge([
            'attributes' => $this->getSafeModelAttributes($model),
        ], $extraProperties);
        
        $this->logActivity("deleted {$modelName}", $properties, $model);
    }
    
    /**
     * Check if we should skip logging
     */
    protected function shouldSkipLogging(): bool
    {
        // Skip during payment processing
        if ($this->isPaymentRoute()) {
            return true;
        }
        
        // Skip during high-load operations
        if (app()->runningInConsole()) {
            return true;
        }
        
        // Skip if memory usage is high
        if (memory_get_usage() > 100 * 1024 * 1024) { // 100MB
            return true;
        }
        
        return false;
    }
    
    /**
     * Check if we should log this model
     */
    protected function shouldLogModel($model): bool
    {
        $modelName = class_basename($model);
        
        // Never log ActivityLog itself (to avoid recursion)
        if ($modelName === 'ActivityLog') {
            return false;
        }
        
        // Skip high-frequency models during payment processing
        $skipDuringPayment = [
            'PaymentItem', 'InventoryHistory', 'StockMovement', 
            'Payment', 'SaleItem', 'Transaction'
        ];
        
        if ($this->isPaymentRoute() && in_array($modelName, $skipDuringPayment)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Get safe model attributes (remove large fields)
     */
    protected function getSafeModelAttributes($model): array
    {
        $attributes = $model->getAttributes();
        
        // Remove potentially large fields
        $unsafeFields = [
            'cart', 'data', 'meta', 'content', 'description', 
            'notes', 'details', 'properties', 'config', 'settings',
            'image', 'photo', 'avatar', 'file', 'document'
        ];
        
        foreach ($unsafeFields as $field) {
            if (array_key_exists($field, $attributes)) {
                if (is_array($attributes[$field]) || is_object($attributes[$field])) {
                    $attributes[$field] = '[DATA]';
                } elseif (strlen($attributes[$field]) > 500) {
                    $attributes[$field] = substr($attributes[$field], 0, 500) . '...';
                }
            }
        }
        
        return $attributes;
    }
    
    /**
     * Limit property size to prevent memory issues
     */
    protected function limitPropertySize(array $properties, int $maxSize = 10000): array
    {
        $jsonSize = strlen(json_encode($properties));
        
        if ($jsonSize > $maxSize) {
            // Remove largest properties
            foreach ($properties as $key => $value) {
                if (strlen(json_encode($value)) > 1000) {
                    $properties[$key] = '[TRUNCATED]';
                }
            }
        }
        
        return $properties;
    }
    
    /**
     * Check if current route is payment-related
     */
    protected function isPaymentRoute(): bool
    {
        $route = request()->route();
        if (!$route) {
            return false;
        }
        
        $paymentRoutes = [
            'api.v1.payments.process',
            'pos.payments.process',
            'api.payment.process',
            'sales.process',
            'pos.process-sale',
        ];
        
        $routeName = $route->getName();
        
        return in_array($routeName, $paymentRoutes) || 
               str_contains($routeName ?? '', 'payment');
    }
}