<?php

namespace App\Services;

use App\Models\PaymentMethod;
use App\Models\Payment;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Product;
use App\Models\Customer;
use App\Models\NumberSequence;
use App\Interfaces\PaymentProviderInterface;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class PaymentService
{
    protected array $providers = [];
    protected NumberGeneratorService $numberGenerator;
    
    public function __construct()
    {
        $this->numberGenerator = new NumberGeneratorService();
        $this->registerProviders();
    }

    // Register payment providers dynamically
    protected function registerProviders(): void
    {
        $providers = config('payment.providers', []);
        
        foreach ($providers as $providerClass) {
            if (class_exists($providerClass)) {
                $provider = app($providerClass);
                if ($provider instanceof PaymentProviderInterface) {
                    $this->providers[$provider->getCode()] = $provider;
                }
            }
        }
    }

    /**
     * Create a sale with invoice and receipt numbers
     */
    public function createSaleWithNumbers(array $saleData, array $cartData): Sale
    {
        DB::beginTransaction();
        
        try {
            // Calculate totals with VAT
            $calculation = $this->calculateSaleTotals($cartData);
            
            // Generate invoice and receipt numbers
            $numbers = $this->numberGenerator->generateForSale();
            
            // Get current user
            $user = auth()->user();
            
            // Create sale record
            $sale = Sale::create([
                'invoice_no' => $numbers['invoice'],
                'receipt_no' => $numbers['receipt'],
                'customer_id' => $saleData['customer_id'] ?? null,
                'user_id' => $user ? $user->id : 1,
                'subtotal' => $calculation['subtotal'],
                'tax_total' => $calculation['tax_total'],
                'discount' => $saleData['discount'] ?? 0,
                'grand_total' => $calculation['grand_total'],
                'status' => 'pending',
                'payment_status' => 'pending',
                'is_vat_applied' => $calculation['tax_total'] > 0,
                'payment_method' => $saleData['payment_method'] ?? 'cash',
                'transaction_id' => $this->generateTransactionId($saleData['payment_method'] ?? 'cash'),
                'cash_received' => $calculation['grand_total'],
                'customer_change' => 0,
                'notes' => $saleData['notes'] ?? null,
                'sale_date' => Carbon::now(),
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
            
            // Create sale items
            $this->createSaleItems($sale->id, $cartData, $calculation['items']);
            
            // Update stock
            $this->updateStock($cartData);
            
            DB::commit();
            
            Log::info('Sale created with numbers', [
                'sale_id' => $sale->id,
                'invoice' => $sale->invoice_no,
                'receipt' => $sale->receipt_no,
                'total' => $sale->grand_total
            ]);
            
            return $sale;
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create sale with numbers: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Process payment for an existing sale
     */
    public function processPaymentForSale(array $paymentData): array
    {
        DB::beginTransaction();
        
        try {
            // Find payment method
            $paymentMethod = PaymentMethod::findOrFail($paymentData['payment_method_id']);
            
            // Validate sale
            $sale = Sale::findOrFail($paymentData['sale_id']);
            
            // Calculate amount
            $amount = $paymentData['amount'] ?? $sale->grand_total;
            
            // Different logic based on payment type
            switch ($paymentMethod->type) {
                case 'cash':
                    return $this->processCashPayment($sale, $paymentMethod, $amount, $paymentData);
                
                case 'mobile':
                    return $this->processMobilePayment($sale, $paymentMethod, $amount, $paymentData);
                
                case 'card':
                    return $this->processCardPayment($sale, $paymentMethod, $amount, $paymentData);
                
                case 'bank':
                    return $this->processBankPayment($sale, $paymentMethod, $amount, $paymentData);
                
                case 'credit':
                    return $this->processCreditPayment($sale, $paymentMethod, $amount, $paymentData);
                
                case 'digital':
                    return $this->processDigitalPayment($sale, $paymentMethod, $amount, $paymentData);
                
                default:
                    throw new \Exception("Unsupported payment type: {$paymentMethod->type}");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Payment processing error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Process cash payment
     */
    protected function processCashPayment(Sale $sale, PaymentMethod $method, float $amount, array $data): array
    {
        // Calculate change if cash received is provided
        $change = 0;
        if (isset($data['cash_received']) && $data['cash_received'] > $amount) {
            $change = $data['cash_received'] - $amount;
        }
        
        $payment = $this->createPaymentRecord([
            'sale_id' => $sale->id,
            'payment_method_id' => $method->id,
            'amount' => $amount,
            'reference' => $sale->receipt_no,
            'status' => 'completed',
            'metadata' => [
                'invoice_no' => $sale->invoice_no,
                'receipt_no' => $sale->receipt_no,
                'cash_received' => $data['cash_received'] ?? $amount,
                'change' => $change,
                'notes' => $data['notes'] ?? null,
            ]
        ]);
        
        // Update sale payment status
        $this->updateSalePayment($sale, $amount);
        
        // Update sale with cash details
        $sale->update([
            'cash_received' => $data['cash_received'] ?? $amount,
            'customer_change' => $change,
            'payment_status' => 'paid'
        ]);
        
        return [
            'success' => true,
            'payment' => $payment,
            'sale' => $sale->fresh(),
            'numbers' => $this->getSaleNumbers($sale),
            'change' => $change,
            'message' => 'Cash payment recorded successfully'
        ];
    }

    /**
     * Process mobile payment (M-Pesa, etc.)
     */
    protected function processMobilePayment(Sale $sale, PaymentMethod $method, float $amount, array $data): array
    {
        $providerCode = $method->provider ?? 'mpesa';
        
        if (!isset($this->providers[$providerCode])) {
            throw new \Exception("Payment provider {$providerCode} not found");
        }
        
        $provider = $this->providers[$providerCode];
        
        // Prepare provider data
        $providerData = [
            'amount' => $amount,
            'phone' => $data['phone'],
            'reference' => $sale->receipt_no,
            'description' => "Payment for Invoice #{$sale->invoice_no}",
            'callback_url' => route('api.payment.callback', ['provider' => $providerCode]),
        ];
        
        // Initiate payment with provider
        $result = $provider->initiatePayment($providerData);
        
        // Create pending payment record
        $payment = $this->createPaymentRecord([
            'sale_id' => $sale->id,
            'payment_method_id' => $method->id,
            'amount' => $amount,
            'reference' => $sale->receipt_no,
            'status' => 'pending',
            'metadata' => array_merge($providerData, [
                'invoice_no' => $sale->invoice_no,
                'receipt_no' => $sale->receipt_no,
                'provider_reference' => $result['provider_reference'] ?? null,
                'provider_response' => $result,
                'checkout_request_id' => $result['checkout_request_id'] ?? null,
            ])
        ]);
        
        return [
            'success' => true,
            'payment' => $payment,
            'sale' => $sale,
            'numbers' => $this->getSaleNumbers($sale),
            'provider_response' => $result,
            'requires_verification' => $method->requires_confirmation,
            'message' => 'Payment initiated successfully'
        ];
    }

    /**
     * Get all active payment methods from database
     */
    public function getActivePaymentMethods(): array
    {
        return PaymentMethod::active()
            ->orderBy('sort_order')
            ->get()
            ->map(function ($method) {
                return [
                    'id' => $method->id,
                    'name' => $method->name,
                    'code' => $method->code,
                    'type' => $method->type,
                    'provider' => $method->provider,
                    'icon' => $method->icon,
                    'description' => $method->description,
                    'config' => $method->config,
                    'requires_confirmation' => $method->requires_confirmation,
                ];
            })->toArray();
    }

    /**
     * Calculate sale totals with VAT
     */
    private function calculateSaleTotals(array $cartItems): array
    {
        $subtotal = 0;
        $taxTotal = 0;
        $taxableAmount = 0;
        $nonTaxableAmount = 0;
        $items = [];
        
        foreach ($cartItems as $cartItem) {
            $product = Product::with('tax')->find($cartItem['product_id']);
            
            if (!$product) {
                throw new \Exception("Product not found: {$cartItem['product_id']}");
            }
            
            $quantity = $cartItem['quantity'];
            $price = $cartItem['price'];
            $itemTotal = $price * $quantity;
            
            // Calculate VAT
            $itemTaxAmount = 0;
            $itemTaxRate = 0;
            $isVatable = $product->has_vat || $product->is_vatable || false;
            
            if ($isVatable && $product->tax) {
                $itemTaxRate = $product->tax->rate;
                $itemTaxAmount = ($itemTotal * $itemTaxRate) / 100;
                $taxableAmount += $itemTotal;
            } else {
                $nonTaxableAmount += $itemTotal;
            }
            
            $items[] = [
                'product' => $product,
                'quantity' => $quantity,
                'price' => $price,
                'total' => $itemTotal,
                'tax_rate' => $itemTaxRate,
                'tax_amount' => $itemTaxAmount,
                'is_vatable' => $isVatable,
            ];
            
            $subtotal += $itemTotal;
            $taxTotal += $itemTaxAmount;
        }
        
        $grandTotal = $subtotal + $taxTotal;
        
        return [
            'subtotal' => $subtotal,
            'tax_total' => $taxTotal,
            'taxable_amount' => $taxableAmount,
            'non_taxable_amount' => $nonTaxableAmount,
            'grand_total' => $grandTotal,
            'items' => $items
        ];
    }

    /**
     * Create sale items
     */
    private function createSaleItems(int $saleId, array $cartItems, array $calculatedItems): void
    {
        foreach ($calculatedItems as $index => $itemData) {
            $cartItem = $cartItems[$index];
            $product = $itemData['product'];
            
            SaleItem::create([
                'sale_id' => $saleId,
                'product_id' => $product->id,
                'quantity' => $itemData['quantity'],
                'unit_price' => $itemData['price'],
                'tax_percent' => $itemData['tax_rate'],
                'tax_amount' => $itemData['tax_amount'],
                'total' => $itemData['total'],
                'batch_id' => $cartItem['batch_id'] ?? null,
                'batch_number' => $cartItem['batch_number'] ?? null,
                'expiry_date' => $cartItem['expiry_date'] ?? null,
            ]);
        }
    }

    /**
     * Update stock
     */
    private function updateStock(array $cartItems): void
    {
        foreach ($cartItems as $item) {
            $product = Product::find($item['product_id']);
            if ($product && $product->track_inventory) {
                $product->decrement('stock', $item['quantity']);
                
                // Update stock status
                if ($product->stock <= 0) {
                    $product->stock_status = 'out_of_stock';
                } elseif ($product->stock <= $product->minimum_stock) {
                    $product->stock_status = 'low_stock';
                } else {
                    $product->stock_status = 'in_stock';
                }
                
                $product->save();
            }
        }
    }

    /**
     * Create payment record
     */
    protected function createPaymentRecord(array $data): Payment
    {
        return Payment::create([
            'sale_id' => $data['sale_id'],
            'payment_method_id' => $data['payment_method_id'],
            'amount' => $data['amount'],
            'reference' => $data['reference'],
            'status' => $data['status'],
            'metadata' => $data['metadata'] ?? [],
            'paid_at' => $data['status'] === 'completed' ? now() : null,
        ]);
    }

    /**
     * Update sale payment status
     */
    protected function updateSalePayment(Sale $sale, float $amount): void
    {
        $sale->update([
            'payment_status' => 'paid',
            'amount_paid' => $amount,
            'paid_at' => now()
        ]);
    }

    /**
     * Generate transaction ID
     */
    private function generateTransactionId(string $method): string
    {
        $timestamp = date('YmdHis');
        $random = rand(1000, 9999);
        return strtoupper($method) . '_' . $timestamp . '_' . $random;
    }

    /**
     * Get sale numbers in array format
     */
    private function getSaleNumbers(Sale $sale): array
    {
        return [
            'invoice' => $sale->invoice_no,
            'receipt' => $sale->receipt_no,
            'display' => $sale->invoice_no ?? $sale->receipt_no ?? 'N/A',
            'has_invoice' => !empty($sale->invoice_no),
            'has_receipt' => !empty($sale->receipt_no)
        ];
    }

    /**
     * Generate receipt data for printing
     */
    public function generateReceiptData(Sale $sale): array
    {
        $sale->load(['items.product', 'customer', 'user']);
        
        $items = [];
        foreach ($sale->items as $item) {
            $items[] = [
                'name' => $item->product->name ?? 'Product',
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'total' => $item->total,
                'tax_rate' => $item->tax_percent,
                'tax_amount' => $item->tax_amount,
                'is_vatable' => $item->tax_percent > 0,
            ];
        }
        
        return [
            'invoice_no' => $sale->invoice_no,
            'receipt_no' => $sale->receipt_no,
            'display_number' => $sale->invoice_no ?? $sale->receipt_no,
            'date' => $sale->created_at->format('Y-m-d'),
            'time' => $sale->created_at->format('H:i:s'),
            'customer' => $sale->customer ? [
                'name' => $sale->customer->name,
                'phone' => $sale->customer->phone,
                'email' => $sale->customer->email,
                'vat_number' => $sale->customer->vat_number,
                'pin' => $sale->customer->pin,
                'company_name' => $sale->customer->company_name,
                'address' => $sale->customer->address,
            ] : null,
            'items' => $items,
            'totals' => [
                'subtotal' => $sale->subtotal,
                'discount' => $sale->discount,
                'vat_amount' => $sale->tax_total,
                'grand_total' => $sale->grand_total,
            ],
            'payment' => [
                'method' => $sale->payment_method,
                'amount_paid' => $sale->grand_total,
                'transaction_id' => $sale->transaction_id,
                'cash_received' => $sale->cash_received,
                'change' => $sale->customer_change,
            ],
            'store' => [
                'name' => config('app.name', 'Kenyan Supermarket'),
                'address' => config('app.address', 'Nairobi CBD'),
                'phone' => config('app.phone', '0700 000 000'),
                'email' => config('app.email', 'info@supermarket.co.ke'),
                'pin' => config('app.pin', 'P051234567N'),
                'vat_number' => config('app.vat_number', 'VAT001234567'),
            ],
            'cashier' => $sale->user->name ?? 'Cashier',
            'autoprint' => true,
        ];
    }

    /**
     * Get next available numbers
     */
    public function getNextNumbers(): array
    {
        return $this->numberGenerator->generateForSale();
    }

    /**
     * Validate a number
     */
    public function validateNumber(string $number, string $type = 'invoice'): array
    {
        $valid = $this->numberGenerator->validateNumber($number, $type);
        
        return [
            'valid' => $valid,
            'parsed' => $valid ? $this->numberGenerator->parseNumber($number) : null
        ];
    }

    // The other payment processing methods (card, bank, credit, digital) would be updated similarly
    // to include number generation and handling...
}