<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use App\Models\MpesaCredential;
use App\Models\MpesaTransaction;
use Carbon\Carbon;

class MpesaService
{
    protected $credential;
    protected $accessToken;
    protected $environment;

    public function __construct(MpesaCredential $credential = null)
    {
        if ($credential) {
            $this->setCredential($credential);
        }
    }

    /**
     * Set active credential
     */
    public function setCredential(MpesaCredential $credential)
    {
        $this->credential = $credential;
        $this->environment = $credential->environment;
        return $this;
    }

    /**
     * Get access token
     */
    public function getAccessToken()
    {
        if ($this->accessToken && Cache::has('mpesa_access_token')) {
            return Cache::get('mpesa_access_token');
        }

        try {
            $url = $this->environment == 'production' 
                ? 'https://api.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials'
                : 'https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials';

            $response = Http::withBasicAuth(
                $this->credential->getDecryptedConsumerKey(),
                $this->credential->getDecryptedConsumerSecret()
            )->get($url);

            if ($response->successful()) {
                $data = $response->json();
                $this->accessToken = $data['access_token'];
                
                // Cache token for 55 minutes (tokens expire in 1 hour)
                Cache::put('mpesa_access_token', $this->accessToken, 55 * 60);
                
                return $this->accessToken;
            }

            Log::error('MPESA Access Token Error: ' . $response->body());
            return null;

        } catch (\Exception $e) {
            Log::error('MPESA Get Access Token Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * STK Push - Initiate payment request
     */
    public function stkPush($phone, $amount, $accountReference, $transactionDesc)
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $timestamp = date('YmdHis');
            $password = base64_encode(
                $this->credential->business_shortcode . 
                $this->credential->getDecryptedPasskey() . 
                $timestamp
            );

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/stkpush/v1/processrequest'
                : 'https://sandbox.safaricom.co.ke/mpesa/stkpush/v1/processrequest';

            $payload = [
                'BusinessShortCode' => $this->credential->business_shortcode,
                'Password' => $password,
                'Timestamp' => $timestamp,
                'TransactionType' => 'CustomerPayBillOnline',
                'Amount' => $amount,
                'PartyA' => $phone,
                'PartyB' => $this->credential->business_shortcode,
                'PhoneNumber' => $phone,
                'CallBackURL' => $this->credential->callback_url . '/api/mpesa/callback',
                'AccountReference' => $accountReference,
                'TransactionDesc' => $transactionDesc
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful() && $responseData['ResponseCode'] == '0') {
                return [
                    'success' => true,
                    'checkout_request_id' => $responseData['CheckoutRequestID'],
                    'merchant_request_id' => $responseData['MerchantRequestID'],
                    'customer_message' => $responseData['CustomerMessage'],
                    'response_description' => $responseData['ResponseDescription']
                ];
            }

            return [
                'success' => false,
                'message' => $responseData['errorMessage'] ?? 'STK Push failed',
                'response_code' => $responseData['ResponseCode'] ?? 'ERROR'
            ];

        } catch (\Exception $e) {
            Log::error('MPESA STK Push Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Check STK Push status
     */
    public function checkStkPushStatus($checkoutRequestId)
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/stkpushquery/v1/query'
                : 'https://sandbox.safaricom.co.ke/mpesa/stkpushquery/v1/query';

            $timestamp = date('YmdHis');
            $password = base64_encode(
                $this->credential->business_shortcode . 
                $this->credential->getDecryptedPasskey() . 
                $timestamp
            );

            $payload = [
                'BusinessShortCode' => $this->credential->business_shortcode,
                'Password' => $password,
                'Timestamp' => $timestamp,
                'CheckoutRequestID' => $checkoutRequestId
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful()) {
                return [
                    'success' => true,
                    'result_code' => $responseData['ResultCode'],
                    'result_desc' => $responseData['ResultDesc'],
                    'checkout_request_id' => $responseData['CheckoutRequestID'],
                    'merchant_request_id' => $responseData['MerchantRequestID'],
                    'response_code' => $responseData['ResponseCode'] ?? null,
                    'response_description' => $responseData['ResponseDescription'] ?? null,
                    'mpesa_receipt_number' => $responseData['MpesaReceiptNumber'] ?? null,
                    'phone_number' => $responseData['PhoneNumber'] ?? null,
                    'amount' => $responseData['Amount'] ?? null,
                    'transaction_date' => $responseData['TransactionDate'] ?? null
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to check status',
                'response' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('MPESA STK Push Query Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Register C2B URLs
     */
    public function registerUrls()
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/c2b/v1/registerurl'
                : 'https://sandbox.safaricom.co.ke/mpesa/c2b/v1/registerurl';

            $payload = [
                'ShortCode' => $this->credential->business_shortcode,
                'ResponseType' => 'Completed',
                'ConfirmationURL' => $this->credential->confirmation_url,
                'ValidationURL' => $this->credential->validation_url
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful() && $responseData['ResponseCode'] == '0') {
                return [
                    'success' => true,
                    'message' => $responseData['ResponseDescription'],
                    'response_code' => $responseData['ResponseCode']
                ];
            }

            return [
                'success' => false,
                'message' => $responseData['errorMessage'] ?? 'URL registration failed',
                'response_code' => $responseData['ResponseCode'] ?? 'ERROR'
            ];

        } catch (\Exception $e) {
            Log::error('MPESA URL Registration Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Check transaction status
     */
    public function checkTransactionStatus($transactionId)
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/transactionstatus/v1/query'
                : 'https://sandbox.safaricom.co.ke/mpesa/transactionstatus/v1/query';

            $timestamp = date('YmdHis');
            $password = base64_encode(
                $this->credential->business_shortcode . 
                $this->credential->getDecryptedPasskey() . 
                $timestamp
            );

            $payload = [
                'Initiator' => $this->credential->initiator_name,
                'SecurityCredential' => $this->credential->getDecryptedSecurityCredential(),
                'CommandID' => 'TransactionStatusQuery',
                'TransactionID' => $transactionId,
                'PartyA' => $this->credential->business_shortcode,
                'IdentifierType' => '4',
                'ResultURL' => $this->credential->callback_url . '/api/mpesa/transaction-status',
                'QueueTimeOutURL' => $this->credential->callback_url . '/api/mpesa/timeout',
                'Remarks' => 'Transaction status query',
                'Occasion' => 'Check status'
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful()) {
                return [
                    'success' => true,
                    'conversation_id' => $responseData['ConversationID'],
                    'originator_conversation_id' => $responseData['OriginatorConversationID'],
                    'response_code' => $responseData['ResponseCode'],
                    'response_description' => $responseData['ResponseDescription']
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to check transaction status',
                'response' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('MPESA Transaction Status Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Get account balance
     */
    public function getAccountBalance()
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/accountbalance/v1/query'
                : 'https://sandbox.safaricom.co.ke/mpesa/accountbalance/v1/query';

            $timestamp = date('YmdHis');
            $password = base64_encode(
                $this->credential->business_shortcode . 
                $this->credential->getDecryptedPasskey() . 
                $timestamp
            );

            $payload = [
                'Initiator' => $this->credential->initiator_name,
                'SecurityCredential' => $this->credential->getDecryptedSecurityCredential(),
                'CommandID' => 'AccountBalance',
                'PartyA' => $this->credential->business_shortcode,
                'IdentifierType' => '4',
                'Remarks' => 'Account balance check',
                'QueueTimeOutURL' => $this->credential->callback_url . '/api/mpesa/timeout',
                'ResultURL' => $this->credential->callback_url . '/api/mpesa/account-balance'
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful()) {
                return [
                    'success' => true,
                    'conversation_id' => $responseData['ConversationID'],
                    'originator_conversation_id' => $responseData['OriginatorConversationID'],
                    'response_code' => $responseData['ResponseCode'],
                    'response_description' => $responseData['ResponseDescription']
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to get account balance',
                'response' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('MPESA Account Balance Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Reverse transaction
     */
    public function reverseTransaction($transactionId, $amount, $receiverPhone = null)
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = $this->environment == 'production'
                ? 'https://api.safaricom.co.ke/mpesa/reversal/v1/request'
                : 'https://sandbox.safaricom.co.ke/mpesa/reversal/v1/request';

            $timestamp = date('YmdHis');
            $password = base64_encode(
                $this->credential->business_shortcode . 
                $this->credential->getDecryptedPasskey() . 
                $timestamp
            );

            $payload = [
                'Initiator' => $this->credential->initiator_name,
                'SecurityCredential' => $this->credential->getDecryptedSecurityCredential(),
                'CommandID' => 'TransactionReversal',
                'TransactionID' => $transactionId,
                'Amount' => $amount,
                'ReceiverParty' => $receiverPhone ?: $this->credential->business_shortcode,
                'RecieverIdentifierType' => '11', // MSISDN
                'ResultURL' => $this->credential->callback_url . '/api/mpesa/reversal-result',
                'QueueTimeOutURL' => $this->credential->callback_url . '/api/mpesa/timeout',
                'Remarks' => 'Transaction reversal',
                'Occasion' => 'Refund'
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful()) {
                return [
                    'success' => true,
                    'conversation_id' => $responseData['ConversationID'],
                    'originator_conversation_id' => $responseData['OriginatorConversationID'],
                    'response_code' => $responseData['ResponseCode'],
                    'response_description' => $responseData['ResponseDescription']
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to reverse transaction',
                'response' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('MPESA Reversal Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Simulate C2B payment (for sandbox testing)
     */
    public function simulateC2BPayment($phone, $amount, $reference)
    {
        try {
            $accessToken = $this->getAccessToken();
            if (!$accessToken) {
                throw new \Exception('Failed to get access token');
            }

            $url = 'https://sandbox.safaricom.co.ke/mpesa/c2b/v1/simulate';

            $payload = [
                'ShortCode' => $this->credential->business_shortcode,
                'CommandID' => 'CustomerPayBillOnline',
                'Amount' => $amount,
                'Msisdn' => $phone,
                'BillRefNumber' => $reference
            ];

            $response = Http::withToken($accessToken)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $responseData = $response->json();

            if ($response->successful()) {
                return [
                    'success' => true,
                    'conversation_id' => $responseData['ConversationID'],
                    'originator_conversation_id' => $responseData['OriginatorConversationID'],
                    'response_code' => $responseData['ResponseCode'],
                    'response_description' => $responseData['ResponseDescription']
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to simulate payment',
                'response' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('MPESA C2B Simulation Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Format phone number to MPESA format (2547XXXXXXXX)
     */
    public static function formatPhoneNumber($phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($phone) === 9 && !str_starts_with($phone, '0')) {
            return '254' . $phone;
        }
        
        if (strlen($phone) === 10 && str_starts_with($phone, '0')) {
            return '254' . substr($phone, 1);
        }
        
        if (strlen($phone) === 12 && str_starts_with($phone, '254')) {
            return $phone;
        }
        
        return $phone;
    }

    /**
     * Parse callback data
     */
    public static function parseCallbackData($callbackData)
    {
        try {
            $result = [];
            
            if (isset($callbackData['Body']['stkCallback'])) {
                // STK Push callback
                $stkCallback = $callbackData['Body']['stkCallback'];
                $result['type'] = 'stk_push';
                $result['checkout_request_id'] = $stkCallback['CheckoutRequestID'];
                $result['merchant_request_id'] = $stkCallback['MerchantRequestID'];
                $result['result_code'] = $stkCallback['ResultCode'];
                $result['result_desc'] = $stkCallback['ResultDesc'];
                
                if (isset($stkCallback['CallbackMetadata']['Item'])) {
                    foreach ($stkCallback['CallbackMetadata']['Item'] as $item) {
                        $result[$item['Name']] = $item['Value'] ?? null;
                    }
                }
                
            } elseif (isset($callbackData['TransactionType'])) {
                // C2B callback
                $result['type'] = 'c2b';
                $result['transaction_type'] = $callbackData['TransactionType'];
                $result['transaction_id'] = $callbackData['TransID'];
                $result['transaction_time'] = $callbackData['TransTime'];
                $result['amount'] = $callbackData['TransAmount'];
                $result['business_shortcode'] = $callbackData['BusinessShortCode'];
                $result['bill_ref_number'] = $callbackData['BillRefNumber'];
                $result['invoice_number'] = $callbackData['InvoiceNumber'];
                $result['org_account_balance'] = $callbackData['OrgAccountBalance'];
                $result['third_party_trans_id'] = $callbackData['ThirdPartyTransID'];
                $result['msisdn'] = $callbackData['MSISDN'];
                $result['first_name'] = $callbackData['FirstName'];
                $result['middle_name'] = $callbackData['MiddleName'];
                $result['last_name'] = $callbackData['LastName'];
                
            } else {
                // Generic callback
                $result['type'] = 'generic';
                $result['data'] = $callbackData;
            }
            
            return $result;
            
        } catch (\Exception $e) {
            Log::error('MPESA Parse Callback Error: ' . $e->getMessage());
            return ['type' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Create transaction from callback
     */
    public static function createTransactionFromCallback($callbackData, $credentialId = null)
    {
        try {
            $parsedData = self::parseCallbackData($callbackData);
            
            if ($parsedData['type'] === 'stk_push') {
                // STK Push callback
                $status = $parsedData['result_code'] == 0 ? 'completed' : 'failed';
                
                $transaction = MpesaTransaction::create([
                    'transaction_id' => $parsedData['MpesaReceiptNumber'] ?? null,
                    'reference' => $parsedData['AccountReference'] ?? null,
                    'phone' => self::formatPhoneNumber($parsedData['PhoneNumber'] ?? ''),
                    'amount' => $parsedData['Amount'] ?? 0,
                    'transaction_date' => $parsedData['TransactionDate'] ? 
                        Carbon::parse($parsedData['TransactionDate']) : now(),
                    'status' => $status,
                    'description' => $parsedData['result_desc'] ?? 'STK Push Payment',
                    'first_name' => null,
                    'middle_name' => null,
                    'last_name' => null,
                    'credential_id' => $credentialId,
                    'raw_response' => $callbackData,
                    'mpesa_receipt_number' => $parsedData['MpesaReceiptNumber'] ?? null,
                    'result_code' => $parsedData['result_code'],
                    'result_desc' => $parsedData['result_desc'],
                    'account_reference' => $parsedData['AccountReference'] ?? null
                ]);
                
            } elseif ($parsedData['type'] === 'c2b') {
                // C2B callback
                $transaction = MpesaTransaction::create([
                    'transaction_id' => $parsedData['transaction_id'],
                    'reference' => $parsedData['bill_ref_number'],
                    'phone' => self::formatPhoneNumber($parsedData['msisdn']),
                    'amount' => $parsedData['amount'],
                    'transaction_date' => Carbon::parse($parsedData['transaction_time']),
                    'status' => 'completed',
                    'description' => $parsedData['transaction_type'] . ' Payment',
                    'first_name' => $parsedData['first_name'],
                    'middle_name' => $parsedData['middle_name'],
                    'last_name' => $parsedData['last_name'],
                    'credential_id' => $credentialId,
                    'raw_response' => $callbackData,
                    'third_party_trans_id' => $parsedData['third_party_trans_id'],
                    'transaction_type' => $parsedData['transaction_type']
                ]);
                
            } else {
                throw new \Exception('Unsupported callback type');
            }
            
            // Try to auto-match with pending sales
            if ($transaction->status === 'completed') {
                self::autoMatchTransaction($transaction);
            }
            
            return $transaction;
            
        } catch (\Exception $e) {
            Log::error('MPESA Create Transaction Error: ' . $e->getMessage(), [
                'callback_data' => $callbackData
            ]);
            return null;
        }
    }

    /**
     * Auto-match transaction with pending sales
     */
    private static function autoMatchTransaction(MpesaTransaction $transaction)
    {
        try {
            // Find pending sales for this phone
            $customer = \App\Models\Customer::where('phone', $transaction->phone)->first();
            
            if ($customer) {
                // Find unpaid sales for this customer
                $pendingSales = \App\Models\Sale::where('customer_id', $customer->id)
                    ->where('payment_status', '!=', 'paid')
                    ->where('grand_total', '>=', $transaction->amount * 0.95) // Within 5%
                    ->where('grand_total', '<=', $transaction->amount * 1.05) // Within 5%
                    ->orderBy('created_at', 'desc')
                    ->get();
                
                foreach ($pendingSales as $sale) {
                    // Check for exact match first
                    if (abs($sale->grand_total - $transaction->amount) < 0.01) {
                        self::matchTransactionToSale($transaction, $sale, 'auto');
                        return true;
                    }
                }
            }
            
            // Try to match by reference (invoice number)
            if ($transaction->reference) {
                $sale = \App\Models\Sale::where('invoice_no', $transaction->reference)
                    ->where('payment_status', '!=', 'paid')
                    ->first();
                
                if ($sale) {
                    self::matchTransactionToSale($transaction, $sale, 'auto_reference');
                    return true;
                }
            }
            
            return false;
            
        } catch (\Exception $e) {
            Log::error('MPESA Auto-match Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Match transaction to sale
     */
    private static function matchTransactionToSale(MpesaTransaction $transaction, \App\Models\Sale $sale, $matchType)
    {
        try {
            $transaction->update([
                'sale_id' => $sale->id,
                'customer_id' => $sale->customer_id,
                'status' => 'matched',
                'matched_at' => now(),
                'match_type' => $matchType,
                'amount_difference' => $transaction->amount - $sale->grand_total
            ]);
            
            $sale->update([
                'payment_status' => 'paid',
                'payment_method' => 'mpesa',
                'payment_reference' => $transaction->transaction_id,
                'paid_at' => now()
            ]);
            
            // Create payment record
            \App\Models\Payment::create([
                'sale_id' => $sale->id,
                'amount' => $transaction->amount,
                'payment_method' => 'mpesa',
                'reference' => $transaction->transaction_id,
                'transaction_id' => $transaction->transaction_id,
                'phone' => $transaction->phone,
                'status' => 'completed',
                'payment_date' => $transaction->transaction_date,
                'notes' => 'Auto-matched with MPESA transaction'
            ]);
            
            Log::info('MPESA Transaction auto-matched', [
                'transaction_id' => $transaction->id,
                'sale_id' => $sale->id,
                'match_type' => $matchType
            ]);
            
        } catch (\Exception $e) {
            Log::error('MPESA Match Transaction Error: ' . $e->getMessage());
        }
    }
}