<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;

class ActivityLogServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // IMPORTANT: Only log SAFE events (no Eloquent model events)
        $this->registerSafeEventListeners();
    }

    /**
     * Register only SAFE event listeners that won't cause recursion
     */
    protected function registerSafeEventListeners(): void
    {
        // 1. Authentication events (safe, no recursion)
        $this->registerAuthEvents();
        
        // 2. Custom events (safe, triggered manually)
        $this->registerCustomEvents();
        
        // 3. Critical system events only (no payment-related)
        $this->registerCriticalEvents();
    }

    /**
     * Authentication events - SAFE
     */
    protected function registerAuthEvents(): void
    {
        Event::listen('Illuminate\Auth\Events\Login', function ($event) {
            activity()
                ->causedBy($event->user)
                ->withProperties([
                    'ip_address' => Request::ip(),
                    'user_agent' => Request::userAgent(),
                ])
                ->log('login');
        });

        Event::listen('Illuminate\Auth\Events\Logout', function ($event) {
            activity()
                ->causedBy($event->user)
                ->withProperties([
                    'ip_address' => Request::ip(),
                    'user_agent' => Request::userAgent(),
                ])
                ->log('logout');
        });

        Event::listen('Illuminate\Auth\Events\PasswordReset', function ($event) {
            activity()
                ->causedBy($event->user)
                ->withProperties([
                    'ip_address' => Request::ip(),
                ])
                ->log('password_reset');
        });
    }

    /**
     * Custom events triggered manually - SAFE
     */
    protected function registerCustomEvents(): void
    {
        // These events are triggered manually from controllers where we know it's safe
        Event::listen('activity.system', function ($eventName, $data) {
            if (isset($data['user']) && isset($data['properties'])) {
                activity()
                    ->causedBy($data['user'])
                    ->withProperties(array_merge($data['properties'], [
                        'ip_address' => Request::ip(),
                        'user_agent' => Request::userAgent(),
                    ]))
                    ->log($eventName);
            }
        });

        Event::listen('activity.user', function ($eventName, $data) {
            if (isset($data['user']) && isset($data['properties'])) {
                activity()
                    ->causedBy($data['user'])
                    ->withProperties(array_merge($data['properties'], [
                        'ip_address' => Request::ip(),
                    ]))
                    ->log($eventName);
            }
        });
    }

    /**
     * Critical system events only - SAFE
     */
    protected function registerCriticalEvents(): void
    {
        // Only listen to CRITICAL events, not all Eloquent events
        $criticalModels = ['User', 'Role', 'Permission', 'Settings'];
        
        foreach ($criticalModels as $model) {
            $modelClass = "App\\Models\\{$model}";
            
            // Created
            Event::listen("eloquent.created: {$modelClass}", function ($eventModel) {
                if (Auth::check() && !$this->isPaymentProcessing()) {
                    $this->logModelEvent('created', $eventModel, [
                        'attributes' => $this->getSafeAttributes($eventModel),
                    ]);
                }
            });
            
            // Updated
            Event::listen("eloquent.updated: {$modelClass}", function ($eventModel) {
                if (Auth::check() && !$this->isPaymentProcessing()) {
                    $this->logModelEvent('updated', $eventModel, [
                        'old' => $eventModel->getOriginal(),
                        'new' => $this->getSafeAttributes($eventModel),
                    ]);
                }
            });
            
            // Deleted
            Event::listen("eloquent.deleted: {$modelClass}", function ($eventModel) {
                if (Auth::check() && !$this->isPaymentProcessing()) {
                    $this->logModelEvent('deleted', $eventModel, [
                        'attributes' => $this->getSafeAttributes($eventModel),
                    ]);
                }
            });
        }
    }

    /**
     * Log model event safely
     */
    protected function logModelEvent(string $event, $model, array $properties = []): void
    {
        $modelName = class_basename($model);
        
        // Double-check we're not in ActivityLog to avoid recursion
        if ($modelName === 'ActivityLog') {
            return;
        }
        
        activity()
            ->causedBy(Auth::user())
            ->performedOn($model)
            ->withProperties(array_merge($properties, [
                'ip_address' => Request::ip(),
                'user_agent' => Request::userAgent(),
            ]))
            ->log("{$event} {$modelName}");
    }

    /**
     * Get safe attributes (remove large fields)
     */
    protected function getSafeAttributes($model): array
    {
        $attributes = $model->getAttributes();
        
        // Remove potentially large fields
        $unsafeFields = [
            'cart', 'data', 'meta', 'content', 'description', 
            'notes', 'details', 'properties', 'config', 'settings',
            'image', 'photo', 'avatar', 'file', 'document'
        ];
        
        foreach ($unsafeFields as $field) {
            if (array_key_exists($field, $attributes)) {
                if (is_array($attributes[$field]) || is_object($attributes[$field])) {
                    $attributes[$field] = '[DATA]';
                } elseif (strlen($attributes[$field]) > 500) {
                    $attributes[$field] = substr($attributes[$field], 0, 500) . '...';
                }
            }
        }
        
        return $attributes;
    }

    /**
     * Check if we're processing a payment
     */
    protected function isPaymentProcessing(): bool
    {
        // Temporarily disable during all requests until we fix it properly
        return true; // Set to true to DISABLE logging during payment processing
        
        // For production, use this:
        /*
        $currentRoute = request()->route();
        if (!$currentRoute) {
            return false;
        }
        
        $paymentRoutes = [
            'api.v1.payments.process',
            'pos.payments.process',
            'api.payment.process',
        ];
        
        $routeName = $currentRoute->getName();
        
        return in_array($routeName, $paymentRoutes) || 
               str_contains($routeName ?? '', 'payment') ||
               str_contains($currentRoute->uri ?? '', 'process-payment') ||
               str_contains($currentRoute->uri ?? '', 'checkout');
        */
    }
}