<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

class Purchase extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'purchases';

    protected $fillable = [
        'supplier_id',
        'invoice_no',
        'purchase_date',
        'reference_number',
        'expected_delivery_date',
        'subtotal',
        'discount_amount',
        'tax_amount',
        'shipping_charges',
        'grand_total',
        'paid_amount',
        'payment_status',
        'payment_method',
        'payment_notes',
        'status',
        'notes',
        'terms_conditions',
        'created_by',
        'updated_by',
        'due_date',
        'is_credit_purchase',
        'approved_at',
        'approved_by',
        'received_at',
        'received_by',
        'delivered_at',
    ];

    protected $casts = [
        'purchase_date' => 'datetime',
        'expected_delivery_date' => 'date',
        'due_date' => 'date',
        'approved_at' => 'datetime',
        'received_at' => 'datetime',
        'delivered_at' => 'datetime',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_charges' => 'decimal:2',
        'grand_total' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'is_credit_purchase' => 'boolean',
    ];

    protected $appends = [
        'balance_due',
        'is_fully_paid',
        'can_edit',
        'payment_summary',
        'credit_info',
        'next_step',
        'previous_step',
    ];

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($purchase) {
            if (empty($purchase->invoice_no)) {
                $purchase->invoice_no = $purchase->generateUniqueInvoiceNumber();
            }
            
            // Set due date based on supplier terms
            if ($purchase->supplier && $purchase->supplier->credit_terms) {
                $days = $purchase->parseCreditTerms($purchase->supplier->credit_terms);
                $purchase->due_date = now()->addDays($days);
            }
        });

        static::created(function ($purchase) {
            // Update supplier balance if it's a credit purchase
            if ($purchase->is_credit_purchase && $purchase->grand_total > 0) {
                $purchase->supplier->updateBalance($purchase->grand_total);
            }
        });

        static::updated(function ($purchase) {
            // Update payment status automatically
            $purchase->updatePaymentStatus();
            
            // Update supplier balance if payment is made
            if ($purchase->wasChanged('paid_amount')) {
                $difference = $purchase->paid_amount - $purchase->getOriginal('paid_amount');
                if ($difference > 0) {
                    $purchase->supplier->reduceBalance($difference);
                }
            }
        });
    }

    /**
     * Generate unique invoice number
     */
    public function generateUniqueInvoiceNumber(): string
    {
        $prefix = 'PO-';
        $date = date('Ymd');
        
        // Try to find existing invoice with same pattern today
        $lastInvoice = Purchase::where('invoice_no', 'like', "{$prefix}{$date}-%")
            ->orderBy('invoice_no', 'desc')
            ->first();
        
        if ($lastInvoice) {
            // Extract the sequence number
            $lastNumber = (int) substr($lastInvoice->invoice_no, -4);
            $sequence = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $sequence = '0001';
        }
        
        // If this sequence exists, find next available
        $newNumber = "{$prefix}{$date}-{$sequence}";
        while (Purchase::where('invoice_no', $newNumber)->exists()) {
            $sequence = str_pad((int)$sequence + 1, 4, '0', STR_PAD_LEFT);
            $newNumber = "{$prefix}{$date}-{$sequence}";
        }
        
        return $newNumber;
    }

    /**
     * Relationships
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function items()
    {
        return $this->hasMany(PurchaseItem::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function receiver()
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Accessors
     */
    public function getBalanceDueAttribute()
    {
        return $this->grand_total - $this->paid_amount;
    }

    public function getIsFullyPaidAttribute()
    {
        return $this->paid_amount >= $this->grand_total;
    }

    public function getCanEditAttribute()
    {
        return $this->status === 'pending';
    }

    public function getPaymentSummaryAttribute()
    {
        return [
            'total_amount' => $this->grand_total,
            'paid_amount' => $this->paid_amount,
            'balance_due' => $this->balance_due,
            'payment_status' => $this->payment_status,
            'payment_method' => $this->payment_method,
            'payment_percentage' => $this->grand_total > 0 ? 
                round(($this->paid_amount / $this->grand_total) * 100, 2) : 0,
            'is_credit' => $this->is_credit_purchase,
        ];
    }

    public function getCreditInfoAttribute()
    {
        if (!$this->is_credit_purchase) {
            return null;
        }

        return [
            'credit_limit' => $this->supplier->credit_limit ?? 0,
            'current_balance' => $this->supplier->current_balance ?? 0,
            'credit_available' => $this->supplier->credit_available ?? 0,
            'credit_utilization' => $this->supplier->credit_utilization ?? 0,
            'is_over_limit' => $this->supplier->is_over_limit ?? false,
            'credit_terms' => $this->supplier->credit_terms,
            'due_date' => $this->due_date,
        ];
    }

    /**
     * Get next available step in workflow
     */
    public function getNextStepAttribute()
    {
        $steps = [
            'pending' => 'approved',
            'approved' => 'received',
            'received' => 'completed',
            'completed' => 'delivered',
            'delivered' => null,
            'cancelled' => null,
        ];
        
        return $steps[$this->status] ?? null;
    }

    /**
     * Get previous step in workflow
     */
    public function getPreviousStepAttribute()
    {
        $steps = [
            'approved' => 'pending',
            'received' => 'approved',
            'completed' => 'received',
            'delivered' => 'completed',
            'pending' => null,
            'cancelled' => null,
        ];
        
        return $steps[$this->status] ?? null;
    }

    /**
     * Check if purchase can proceed to next step
     */
    public function canProceedToNextStep(): bool
    {
        if (!$this->next_step) {
            return false;
        }

        if ($this->next_step === 'approved' && $this->is_credit_purchase) {
            return $this->supplier->canMakePurchase($this->grand_total);
        }

        return true;
    }

    /**
     * Update payment status based on paid amount
     */
    public function updatePaymentStatus()
    {
        if ($this->grand_total <= 0) {
            $status = 'paid';
        } elseif ($this->paid_amount <= 0) {
            $status = 'pending';
        } elseif ($this->paid_amount >= $this->grand_total) {
            $status = 'paid';
        } else {
            $status = 'partial';
        }

        if ($this->payment_status !== $status) {
            $this->payment_status = $status;
            $this->saveQuietly();
        }
    }

    /**
     * Parse credit terms string to days
     */
    private function parseCreditTerms(string $terms): int
    {
        preg_match('/(\d+)/', $terms, $matches);
        return $matches[1] ?? 30; // Default 30 days
    }

    /**
     * Check if purchase can be approved (within credit limit)
     */
    public function canApprove(): bool
    {
        if (!$this->is_credit_purchase) {
            return true;
        }

        return $this->supplier->canMakePurchase($this->grand_total);
    }

    /**
     * Make a payment towards this purchase
     */
    public function makePayment(float $amount, string $method, ?string $notes = null): bool
    {
        if ($amount <= 0 || $amount > $this->balance_due) {
            return false;
        }

        DB::beginTransaction();
        try {
            $oldPaidAmount = $this->paid_amount;
            $this->paid_amount += $amount;
            $this->payment_method = $method;
            $this->payment_notes = $notes;
            $this->save();

            // Update supplier balance
            $this->supplier->reduceBalance($amount);

            // Record payment activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($this)
                ->withProperties([
                    'amount' => $amount,
                    'method' => $method,
                    'notes' => $notes,
                ])
                ->log('made payment');

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            return false;
        }
    }

    /**
     * Get purchase breakdown
     */
    public function getBreakdownAttribute()
    {
        return [
            'subtotal' => $this->subtotal,
            'discount' => $this->discount_amount,
            'tax' => $this->tax_amount,
            'shipping' => $this->shipping_charges,
            'grand_total' => $this->grand_total,
        ];
    }

    /**
     * Get workflow timeline
     */
    public function getTimelineAttribute()
    {
        $timeline = [];
        
        if ($this->created_at) {
            $timeline[] = [
                'event' => 'created',
                'date' => $this->created_at,
                'user' => $this->creator,
                'description' => 'Purchase order created'
            ];
        }
        
        if ($this->approved_at) {
            $timeline[] = [
                'event' => 'approved',
                'date' => $this->approved_at,
                'user' => $this->approver,
                'description' => 'Purchase order approved'
            ];
        }
        
        if ($this->received_at) {
            $timeline[] = [
                'event' => 'received',
                'date' => $this->received_at,
                'user' => $this->receiver,
                'description' => 'Items received'
            ];
        }
        
        if ($this->status === 'completed') {
            $timeline[] = [
                'event' => 'completed',
                'date' => $this->updated_at,
                'user' => $this->updater,
                'description' => 'Items added to inventory'
            ];
        }
        
        if ($this->delivered_at) {
            $timeline[] = [
                'event' => 'delivered',
                'date' => $this->delivered_at,
                'user' => $this->updater,
                'description' => 'Services delivered'
            ];
        }
        
        return $timeline;
    }
}