<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class PaymentMethod extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'code',
        'type', // 'cash', 'mobile', 'card', 'bank', 'credit', 'digital'
        'provider', // 'mpesa', 'pesapal', 'njenga', 'equity', 'kcb', 'etc'
        'is_active',
        'requires_confirmation',
        'config', // JSON config for provider-specific settings
        'sort_order',
        'icon',
        'description'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'requires_confirmation' => 'boolean',
        'config' => 'array',
        'sort_order' => 'integer'
    ];

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    // Scope for active methods
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope by type
    public function scopeType($query, $type)
    {
        return $query->where('type', $type);
    }

    // Get payment method by code
    public static function findByCode($code)
    {
        return static::where('code', $code)->first();
    }

    // Check if this method requires manual verification
    public function requiresVerification(): bool
    {
        return $this->requires_confirmation || in_array($this->type, ['bank', 'cheque']);
    }

    // Get display name with icon
    public function getDisplayNameAttribute(): string
    {
        $icon = $this->icon ? "<i class='bx bx-{$this->icon}'></i> " : '';
        return $icon . $this->name;
    }
}