<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class MpesaTransaction extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'transaction_id',
        'reference',
        'phone',
        'amount',
        'expected_amount',
        'transaction_date',
        'status',
        'description',
        'first_name',
        'middle_name',
        'last_name',
        'sale_id',
        'customer_id',
        'credential_id',
        'matched_at',
        'matched_by',
        'match_type',
        'match_notes',
        'reversed_at',
        'reversed_by',
        'reversal_reason',
        'refund_amount',
        'amount_difference',
        'raw_response',
        'transaction_type',
        'mpesa_receipt_number',
        'result_code',
        'result_desc',
        'account_reference',
        'third_party_trans_id',
        'transaction_status'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'expected_amount' => 'decimal:2',
        'refund_amount' => 'decimal:2',
        'amount_difference' => 'decimal:2',
        'transaction_date' => 'datetime',
        'matched_at' => 'datetime',
        'reversed_at' => 'datetime',
        'raw_response' => 'array'
    ];

    public function sale()
    {
        return $this->belongsTo(Sale::class);
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function credential()
    {
        return $this->belongsTo(MpesaCredential::class);
    }

    public function matchedBy()
    {
        return $this->belongsTo(User::class, 'matched_by');
    }

    public function reversedBy()
    {
        return $this->belongsTo(User::class, 'reversed_by');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeReversed($query)
    {
        return $query->where('status', 'reversed');
    }

    public function scopeUnmatched($query)
    {
        return $query->whereNull('sale_id')->where('status', 'completed');
    }

    public function scopeMatched($query)
    {
        return $query->whereNotNull('sale_id');
    }

    public function scopeByPhone($query, $phone)
    {
        return $query->where('phone', $phone);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    public function scopeByAmountRange($query, $minAmount, $maxAmount)
    {
        return $query->whereBetween('amount', [$minAmount, $maxAmount]);
    }

    // Methods
    public function isMatched()
    {
        return !empty($this->sale_id);
    }

    public function isReversed()
    {
        return $this->status === 'reversed';
    }

    public function canBeReversed()
    {
        return $this->status === 'completed' && !$this->isReversed();
    }

    public function matchesExpectedAmount(): bool
    {
        return abs($this->amount - $this->expected_amount) < 0.01;
    }

    public function getFullNameAttribute(): string
    {
        $names = array_filter([$this->first_name, $this->middle_name, $this->last_name]);
        return implode(' ', $names);
    }

    public function getFormattedPhoneAttribute(): string
    {
        $phone = $this->phone;
        if (strlen($phone) === 12 && str_starts_with($phone, '254')) {
            return '0' . substr($phone, 3);
        }
        return $phone;
    }

    public function getAmountDifferenceTextAttribute(): string
    {
        if ($this->amount_difference === null) {
            return 'Exact';
        }
        
        if ($this->amount_difference > 0) {
            return 'Overpayment: +' . number_format($this->amount_difference, 2);
        }
        
        return 'Underpayment: ' . number_format($this->amount_difference, 2);
    }
}