<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Customer extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'customer_code',
        'name',
        'email',
        'phone',
        'address',
        'contact_person',
        'customer_type',
        'company_name',
        'website',
        'tax_id',
        'status',
        'credit_limit',
        'credit_balance',
        'balance',
        'total_spent',
        'taxable_sales',
        'vat_collected',
        'total_vat_paid',
        'total_orders',
        'last_purchase_date',
        'notes',
        'loyalty_points',
        'discount_percent',
        'is_vat_registered',
        'vat_status',
        'vat_exemption_number',
        'vat_exemption_certificate_no',
        'vat_exemption_reason',
        'vat_exemption_start_date',
        'vat_exemption_end_date',
        'vat_exemption_approved_by',
        'is_zero_rated',
        'zero_rated_start_date',
        'zero_rated_end_date',
        'zero_rated_reason',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'credit_limit' => 'decimal:2',
        'credit_balance' => 'decimal:2',
        'balance' => 'decimal:2',
        'total_spent' => 'decimal:2',
        'taxable_sales' => 'decimal:2',
        'vat_collected' => 'decimal:2',
        'total_vat_paid' => 'decimal:2',
        'total_orders' => 'integer',
        'loyalty_points' => 'integer',
        'discount_percent' => 'decimal:2',
        'is_vat_registered' => 'boolean',
        'is_zero_rated' => 'boolean',
        'vat_exemption_start_date' => 'date',
        'vat_exemption_end_date' => 'date',
        'zero_rated_start_date' => 'date',
        'zero_rated_end_date' => 'date',
        'last_purchase_date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    protected $appends = [
        'is_walk_in',
        'is_vat_registered',
        'has_credit_available',
        'available_credit',
        'average_order_value',
        'is_active_customer',
        'days_since_last_purchase',
        'customer_level',
        'total_vat_paid',
        'vat_rate_effective',
        'non_taxable_sales',
        'current_vat_status',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($customer) {
            // Generate customer code if not provided
            if (empty($customer->customer_code)) {
                $customer->customer_code = 'CUST-' . strtoupper(uniqid());
            }
            
            // Set default VAT status if not provided
            if (empty($customer->vat_status)) {
                $customer->vat_status = 'vatable';
            }
            
            // Auto-set is_vat_registered based on VAT number
            if (!empty($customer->vat_number) && !isset($customer->is_vat_registered)) {
                $customer->is_vat_registered = true;
            }
        });
        
        static::updating(function ($customer) {
            // Auto-update is_vat_registered based on VAT number
            if (!empty($customer->vat_number) && !$customer->is_vat_registered) {
                $customer->is_vat_registered = true;
            }
        });
    }

    /**
     * Sales relationship
     */
    public function sales(): HasMany
    {
        return $this->hasMany(Sale::class);
    }

    /**
     * Credit notes relationship
     */
    public function creditNotes(): HasMany
    {
        return $this->hasMany(CreditNote::class);
    }

    /**
     * Payments relationship (through sales)
     */
    public function payments(): HasManyThrough
    {
        return $this->hasManyThrough(
            Payment::class,
            Sale::class,
            'customer_id', // Foreign key on sales table
            'sale_id',     // Foreign key on payments table
            'id',          // Local key on customers table
            'id'           // Local key on sales table
        );
    }

    /**
     * Scope a query to only include active customers.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for walk-in customers.
     */
    public function scopeWalkIn($query)
    {
        return $query->where('customer_type', 'walk-in');
    }

    /**
     * Scope for regular customers.
     */
    public function scopeRegular($query)
    {
        return $query->where('customer_type', 'regular');
    }

    /**
     * Scope for business customers.
     */
    public function scopeBusiness($query)
    {
        return $query->where('customer_type', 'business');
    }

    /**
     * Scope for customers with credit.
     */
    public function scopeHasCredit($query)
    {
        return $query->where('credit_limit', '>', 0);
    }

    /**
     * Scope for VAT registered customers.
     */
    public function scopeVatRegistered($query)
    {
        return $query->where('is_vat_registered', true);
    }

    /**
     * Scope for non-VAT registered customers.
     */
    public function scopeNonVat($query)
    {
        return $query->where('is_vat_registered', false);
    }

    /**
     * Scope for VAT exempted customers.
     */
    public function scopeVatExempted($query)
    {
        return $query->where('vat_status', 'exempted');
    }

    /**
     * Scope for zero-rated customers.
     */
    public function scopeZeroRated($query)
    {
        return $query->where('vat_status', 'zero_rated');
    }

    /**
     * Search customers by name, phone, email, or code.
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('customer_code', 'like', "%{$search}%")
              ->orWhere('vat_number', 'like', "%{$search}%")
              ->orWhere('company_name', 'like', "%{$search}%")
              ->orWhere('contact_person', 'like', "%{$search}%");
        });
    }

    /**
     * Filter customers by VAT registration status.
     */
    public function scopeByVatStatus($query, $status)
    {
        if ($status === 'registered') {
            return $query->where('is_vat_registered', true);
        } elseif ($status === 'non-registered') {
            return $query->where('is_vat_registered', false);
        } elseif ($status === 'exempted') {
            return $query->where('vat_status', 'exempted');
        } elseif ($status === 'zero_rated') {
            return $query->where('vat_status', 'zero_rated');
        }
        return $query;
    }

    /**
     * Check if customer is walk-in.
     */
    public function getIsWalkInAttribute(): bool
    {
        return ($this->attributes['customer_type'] ?? '') === 'walk-in';
    }

    /**
     * Check if customer is VAT registered.
     */
    public function getIsVatRegisteredAttribute(): bool
    {
        return (bool) ($this->attributes['is_vat_registered'] ?? false);
    }

    /**
     * Get current VAT status for sales calculation
     */
    public function getCurrentVatStatusAttribute(): string
    {
        $status = $this->attributes['vat_status'] ?? 'vatable';
        
        // Check if exemption or zero-rated status is still valid
        if ($status === 'exempted') {
            $startDate = $this->attributes['vat_exemption_start_date'] ?? null;
            $endDate = $this->attributes['vat_exemption_end_date'] ?? null;
            
            if ($startDate && $endDate) {
                $now = now();
                $start = \Carbon\Carbon::parse($startDate);
                $end = \Carbon\Carbon::parse($endDate);
                
                if ($now->isBetween($start, $end)) {
                    return 'exempted';
                }
            }
        } elseif ($status === 'zero_rated') {
            $startDate = $this->attributes['zero_rated_start_date'] ?? null;
            $endDate = $this->attributes['zero_rated_end_date'] ?? null;
            
            if ($startDate && $endDate) {
                $now = now();
                $start = \Carbon\Carbon::parse($startDate);
                $end = \Carbon\Carbon::parse($endDate);
                
                if ($now->isBetween($start, $end)) {
                    return 'zero_rated';
                }
            }
        }
        
        return $status;
    }

    /**
     * Check if customer has credit available.
     */
    public function getHasCreditAvailableAttribute(): bool
    {
        $creditLimit = $this->attributes['credit_limit'] ?? 0;
        $creditBalance = $this->attributes['credit_balance'] ?? 0;
        
        if (!$creditLimit || $creditLimit <= 0) {
            return false;
        }
        
        return ($creditLimit - $creditBalance) > 0;
    }

    /**
     * Get available credit amount.
     */
    public function getAvailableCreditAttribute(): float
    {
        $creditLimit = $this->attributes['credit_limit'] ?? 0;
        $creditBalance = $this->attributes['credit_balance'] ?? 0;
        
        if (!$creditLimit || $creditLimit <= 0) {
            return 0;
        }
        
        return max(0, $creditLimit - $creditBalance);
    }

    /**
     * Get average order value.
     */
    public function getAverageOrderValueAttribute(): float
    {
        $totalOrders = $this->attributes['total_orders'] ?? 0;
        $totalSpent = $this->attributes['total_spent'] ?? 0;
        
        if ($totalOrders === 0) {
            return 0;
        }
        
        return round($totalSpent / $totalOrders, 2);
    }

    /**
     * Check if customer is active (made purchase in last 90 days).
     */
    public function getIsActiveCustomerAttribute(): bool
    {
        $lastPurchaseDate = $this->attributes['last_purchase_date'] ?? null;
        
        if (!$lastPurchaseDate) {
            return false;
        }
        
        try {
            $lastPurchase = \Carbon\Carbon::parse($lastPurchaseDate);
            return $lastPurchase->diffInDays(now()) <= 90;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get days since last purchase.
     */
    public function getDaysSinceLastPurchaseAttribute(): ?int
    {
        $lastPurchaseDate = $this->attributes['last_purchase_date'] ?? null;
        
        if (!$lastPurchaseDate) {
            return null;
        }
        
        try {
            $lastPurchase = \Carbon\Carbon::parse($lastPurchaseDate);
            return $lastPurchase->diffInDays(now());
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get customer level based on spending.
     */
    public function getCustomerLevelAttribute(): string
    {
        $totalSpent = $this->attributes['total_spent'] ?? 0;
        
        if ($totalSpent >= 100000) {
            return 'platinum';
        } elseif ($totalSpent >= 50000) {
            return 'gold';
        } elseif ($totalSpent >= 10000) {
            return 'silver';
        } else {
            return 'bronze';
        }
    }

    /**
     * Get total VAT paid by customer.
     */
    public function getTotalVatPaidAttribute(): float
    {
        return (float) ($this->attributes['vat_collected'] ?? 0);
    }

    /**
     * Get effective VAT rate for the customer.
     */
    public function getVatRateEffectiveAttribute(): float
    {
        $taxableSales = $this->attributes['taxable_sales'] ?? 0;
        $vatCollected = $this->attributes['vat_collected'] ?? 0;
        
        if ($taxableSales == 0) {
            return 0;
        }
        
        return round(($vatCollected / $taxableSales) * 100, 2);
    }

    /**
     * Get non-taxable sales amount.
     */
    public function getNonTaxableSalesAttribute(): float
    {
        $totalSpent = $this->attributes['total_spent'] ?? 0;
        $taxableSales = $this->attributes['taxable_sales'] ?? 0;
        
        return (float) ($totalSpent - $taxableSales);
    }

    /**
     * Update customer statistics after a sale.
     */
    public function updateAfterSale(float $amount, float $taxableAmount = 0, float $vatAmount = 0): void
    {
        $this->increment('total_spent', $amount);
        $this->increment('taxable_sales', $taxableAmount);
        $this->increment('vat_collected', $vatAmount);
        $this->increment('total_orders', 1);
        $this->last_purchase_date = now();
        
        // Add loyalty points (1 point per 100 spent)
        $points = floor($amount / 100);
        if ($points > 0) {
            $this->increment('loyalty_points', $points);
        }
        
        $this->save();
    }

    /**
     * Update VAT statistics for customer.
     */
    public function updateVatStats(float $taxableAmount, float $vatAmount): void
    {
        $this->increment('taxable_sales', $taxableAmount);
        $this->increment('vat_collected', $vatAmount);
        $this->save();
    }

    /**
     * Add credit balance for credit sales.
     */
    public function addCredit(float $amount): bool
    {
        $creditLimit = $this->attributes['credit_limit'] ?? 0;
        $creditBalance = $this->attributes['credit_balance'] ?? 0;
        
        if ($creditLimit > 0 && ($creditBalance + $amount) > $creditLimit) {
            return false;
        }
        
        $this->increment('credit_balance', $amount);
        $this->increment('balance', $amount);
        return true;
    }

    /**
     * Reduce credit balance for payments.
     */
    public function reduceCredit(float $amount): bool
    {
        $creditBalance = $this->attributes['credit_balance'] ?? 0;
        
        if ($creditBalance < $amount) {
            return false;
        }
        
        $this->decrement('credit_balance', $amount);
        $this->decrement('balance', $amount);
        
        // Ensure balance doesn't go negative
        if ($this->balance < 0) {
            $this->balance = 0;
            $this->save();
        }
        
        return true;
    }

    /**
     * Check if customer can make credit purchase.
     */
    public function canPurchaseOnCredit(float $amount): array
    {
        $creditLimit = $this->attributes['credit_limit'] ?? 0;
        $creditBalance = $this->attributes['credit_balance'] ?? 0;
        
        if ($creditLimit <= 0) {
            return ['allowed' => false, 'reason' => 'No credit limit'];
        }
        
        $available = $creditLimit - $creditBalance;
        if ($amount > $available) {
            return [
                'allowed' => false, 
                'reason' => "Credit limit exceeded. Available: KES" . number_format($available, 2) . ", Requested: KES" . number_format($amount, 2)
            ];
        }
        
        return ['allowed' => true, 'available' => $available];
    }

    /**
     * Get customer's purchase history.
     */
    public function getPurchaseHistory(int $limit = 10)
    {
        return $this->sales()
            ->with(['items.product'])
            ->orderBy('sale_date', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Get total outstanding amount.
     */
    public function getOutstandingAmount(): float
    {
        return $this->sales()
            ->where('payment_status', 'pending')
            ->sum('grand_total');
    }

    /**
     * Get payment history.
     */
    public function getPaymentHistory(int $limit = 10)
    {
        return $this->payments()
            ->with('sale')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Generate customer statement with VAT details.
     */
    public function generateStatement($startDate = null, $endDate = null): array
    {
        $startDate = $startDate ?: now()->subMonth();
        $endDate = $endDate ?: now();
        
        $sales = $this->sales()
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->with(['items.product', 'payments'])
            ->orderBy('sale_date', 'desc')
            ->get();
        
        $payments = $this->payments()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();
        
        $openingBalance = $this->sales()
            ->where('sale_date', '<', $startDate)
            ->where('payment_status', 'pending')
            ->sum('grand_total');
        
        $totalSales = $sales->sum('grand_total');
        $totalPayments = $payments->sum('amount');
        $closingBalance = $openingBalance + $totalSales - $totalPayments;
        
        // Calculate VAT summary
        $taxableAmount = $sales->where('is_taxable', true)->sum('taxable_amount');
        $vatAmount = $sales->where('is_taxable', true)->sum('tax_amount');
        
        return [
            'customer' => $this->only(['name', 'customer_code', 'phone', 'email', 'vat_number', 'is_vat_registered', 'vat_status']),
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d'),
            ],
            'opening_balance' => $openingBalance,
            'sales' => $sales->map(function($sale) {
                return [
                    'date' => $sale->sale_date->format('Y-m-d H:i'),
                    'invoice_no' => $sale->invoice_no,
                    'amount' => $sale->grand_total,
                    'taxable_amount' => $sale->taxable_amount ?? 0,
                    'tax_amount' => $sale->tax_amount ?? 0,
                    'is_taxable' => $sale->is_taxable ?? false,
                    'status' => $sale->payment_status,
                ];
            }),
            'payments' => $payments->map(function($payment) {
                return [
                    'date' => $payment->created_at->format('Y-m-d H:i'),
                    'reference' => $payment->reference,
                    'method' => $payment->payment_method,
                    'amount' => $payment->amount,
                ];
            }),
            'vat_summary' => [
                'taxable_sales' => $taxableAmount,
                'vat_collected' => $vatAmount,
                'non_taxable_sales' => $totalSales - $taxableAmount,
                'vat_rate_effective' => $taxableAmount > 0 ? round(($vatAmount / $taxableAmount) * 100, 2) : 0,
            ],
            'summary' => [
                'total_sales' => $totalSales,
                'total_payments' => $totalPayments,
                'closing_balance' => $closingBalance,
                'credit_limit' => $this->credit_limit,
                'credit_available' => $this->available_credit,
            ],
        ];
    }

    /**
     * Apply customer discount if applicable.
     */
    public function applyDiscount(float $amount): float
    {
        $discountPercent = $this->attributes['discount_percent'] ?? 0;
        
        if ($discountPercent > 0) {
            return $amount * (1 - ($discountPercent / 100));
        }
        
        return $amount;
    }

    /**
     * Reset customer statistics.
     */
    public function resetStatistics(): void
    {
        $this->update([
            'total_spent' => 0,
            'taxable_sales' => 0,
            'vat_collected' => 0,
            'total_vat_paid' => 0,
            'total_orders' => 0,
            'loyalty_points' => 0,
            'credit_balance' => 0,
            'balance' => 0,
            'last_purchase_date' => null,
        ]);
    }

    /**
     * Validate customer data.
     */
    public static function validateData(array $data, $customerId = null): array
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:customers,email,' . ($customerId ?: 'NULL'),
            'phone' => 'required|string|max:20|unique:customers,phone,' . ($customerId ?: 'NULL'),
            'customer_type' => 'required|in:walk-in,regular,business',
            'credit_limit' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive',
            'vat_number' => 'nullable|string|max:50|unique:customers,vat_number,' . ($customerId ?: 'NULL'),
            'is_vat_registered' => 'boolean',
            'vat_status' => 'nullable|in:vatable,exempted,zero_rated',
            'vat_exemption_number' => 'nullable|string|max:100',
            'vat_exemption_certificate_no' => 'nullable|string|max:100',
            'vat_exemption_reason' => 'nullable|string',
            'vat_exemption_start_date' => 'nullable|date',
            'vat_exemption_end_date' => 'nullable|date|after_or_equal:vat_exemption_start_date',
            'vat_exemption_approved_by' => 'nullable|string|max:255',
            'is_zero_rated' => 'boolean',
            'zero_rated_start_date' => 'nullable|date',
            'zero_rated_end_date' => 'nullable|date|after_or_equal:zero_rated_start_date',
            'zero_rated_reason' => 'nullable|string',
            'discount_percent' => 'nullable|numeric|min:0|max:100',
            'loyalty_points' => 'nullable|integer|min:0',
            'taxable_sales' => 'nullable|numeric|min:0',
            'vat_collected' => 'nullable|numeric|min:0',
            'total_vat_paid' => 'nullable|numeric|min:0',
            'balance' => 'nullable|numeric',
        ];
        
        $messages = [
            'name.required' => 'Customer name is required',
            'phone.required' => 'Phone number is required',
            'phone.unique' => 'Phone number already exists',
            'email.unique' => 'Email already exists',
            'vat_number.unique' => 'VAT number already exists',
            'vat_exemption_end_date.after_or_equal' => 'Exemption end date must be after or equal to start date',
            'zero_rated_end_date.after_or_equal' => 'Zero rated end date must be after or equal to start date',
        ];
        
        return [$rules, $messages];
    }

    /**
     * Check if VAT number is valid format.
     */
    public function isValidVatNumber(): bool
    {
        if (empty($this->vat_number)) {
            return false;
        }
        
        // Basic VAT number validation (country code + numbers)
        // You can customize this based on your country's VAT format
        return preg_match('/^[A-Z]{2}[0-9A-Z]{8,12}$/', $this->vat_number) === 1;
    }

    /**
     * Get VAT summary for reporting.
     */
    public function getVatSummary(): array
    {
        return [
            'customer_name' => $this->name,
            'vat_number' => $this->vat_number,
            'is_vat_registered' => $this->is_vat_registered,
            'vat_status' => $this->vat_status,
            'current_vat_status' => $this->current_vat_status,
            'taxable_sales' => (float) $this->taxable_sales,
            'vat_collected' => (float) $this->vat_collected,
            'total_sales' => (float) $this->total_spent,
            'non_taxable_sales' => $this->non_taxable_sales,
            'vat_rate_effective' => $this->vat_rate_effective,
        ];
    }

    /**
     * Check if customer has active VAT exemption
     */
    public function hasActiveVatExemption(): bool
    {
        if ($this->vat_status !== 'exempted') {
            return false;
        }
        
        $startDate = $this->vat_exemption_start_date;
        $endDate = $this->vat_exemption_end_date;
        
        if (!$startDate || !$endDate) {
            return false;
        }
        
        $now = now();
        return $now->between($startDate, $endDate);
    }

    /**
     * Check if customer has active zero-rated status
     */
    public function hasActiveZeroRated(): bool
    {
        if ($this->vat_status !== 'zero_rated') {
            return false;
        }
        
        $startDate = $this->zero_rated_start_date;
        $endDate = $this->zero_rated_end_date;
        
        if (!$startDate || !$endDate) {
            return false;
        }
        
        $now = now();
        return $now->between($startDate, $endDate);
    }

    /**
     * Get VAT exemption details
     */
    public function getVatExemptionDetails(): array
    {
        if ($this->vat_status !== 'exempted') {
            return [];
        }
        
        return [
            'exemption_number' => $this->vat_exemption_number,
            'certificate_no' => $this->vat_exemption_certificate_no,
            'reason' => $this->vat_exemption_reason,
            'start_date' => $this->vat_exemption_start_date,
            'end_date' => $this->vat_exemption_end_date,
            'approved_by' => $this->vat_exemption_approved_by,
            'is_active' => $this->hasActiveVatExemption(),
        ];
    }

    /**
     * Get zero-rated details
     */
    public function getZeroRatedDetails(): array
    {
        if ($this->vat_status !== 'zero_rated') {
            return [];
        }
        
        return [
            'start_date' => $this->zero_rated_start_date,
            'end_date' => $this->zero_rated_end_date,
            'reason' => $this->zero_rated_reason,
            'is_active' => $this->hasActiveZeroRated(),
        ];
    }

    /**
     * Calculate VAT for a sale amount based on customer's VAT status
     */
    public function calculateVatForAmount(float $amount): array
    {
        $vatStatus = $this->current_vat_status;
        
        switch ($vatStatus) {
            case 'exempted':
                return [
                    'taxable_amount' => 0,
                    'vat_amount' => 0,
                    'total_with_vat' => $amount,
                    'vat_rate' => 0,
                    'vat_status' => 'exempted',
                ];
                
            case 'zero_rated':
                return [
                    'taxable_amount' => $amount,
                    'vat_amount' => 0,
                    'total_with_vat' => $amount,
                    'vat_rate' => 0,
                    'vat_status' => 'zero_rated',
                ];
                
            case 'vatable':
            default:
                // Standard 16% VAT (Kenya)
                $vatRate = 16;
                $vatAmount = ($amount * $vatRate) / 100;
                $totalWithVat = $amount + $vatAmount;
                
                return [
                    'taxable_amount' => $amount,
                    'vat_amount' => $vatAmount,
                    'total_with_vat' => $totalWithVat,
                    'vat_rate' => $vatRate,
                    'vat_status' => 'vatable',
                ];
        }
    }
}