<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Builder;

class ActivityLog extends Model
{
    use HasFactory;

    protected $table = 'activity_log';

    protected $fillable = [
        'log_name',
        'description',
        'subject_id',
        'subject_type',
        'causer_id',
        'causer_type',
        'properties',
        'event',
        'batch_uuid',
        'ip_address',
        'user_agent',
    ];

    protected $casts = [
        'properties' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the subject of the activity.
     */
    public function subject(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Get the causer of the activity.
     */
    public function causer(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Scope a query to filter by log name.
     */
    public function scopeByLogName(Builder $query, $logName): Builder
    {
        return $query->where('log_name', $logName);
    }

    /**
     * Scope a query to filter by causer.
     */
    public function scopeByCauser(Builder $query, $userId): Builder
    {
        return $query->where('causer_id', $userId);
    }

    /**
     * Scope a query to filter by date range.
     */
    public function scopeDateRange(Builder $query, $startDate, $endDate): Builder
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Scope a query to filter by event type.
     */
    public function scopeByEvent(Builder $query, $event): Builder
    {
        return $query->where('event', $event);
    }

    /**
     * Scope a query to search in description.
     */
    public function scopeSearch(Builder $query, $searchTerm): Builder
    {
        return $query->where('description', 'like', "%{$searchTerm}%")
                    ->orWhere('properties', 'like', "%{$searchTerm}%");
    }

    /**
     * Get formatted properties for display.
     */
    public function getFormattedPropertiesAttribute(): string
    {
        if (empty($this->properties)) {
            return 'No details';
        }

        return json_encode($this->properties, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }

    /**
     * Get user-friendly event name.
     */
    public function getEventNameAttribute(): string
    {
        $eventNames = [
            'created' => 'Created',
            'updated' => 'Updated',
            'deleted' => 'Deleted',
            'restored' => 'Restored',
            'login' => 'Login',
            'logout' => 'Logout',
            'password_reset' => 'Password Reset',
            'exported' => 'Exported',
            'imported' => 'Imported',
            'backup' => 'Backup',
            'restore' => 'Restore',
            'payment' => 'Payment',
            'refund' => 'Refund',
            'stock_adjustment' => 'Stock Adjustment',
        ];

        return $eventNames[$this->event] ?? ucfirst(str_replace('_', ' ', $this->event));
    }

    /**
     * Get activity icon based on log name.
     */
    public function getIconAttribute(): string
    {
        $icons = [
            'default' => 'bx bx-history',
            'user' => 'bx bx-user',
            'product' => 'bx bx-package',
            'category' => 'bx bx-category',
            'customer' => 'bx bx-user-circle',
            'supplier' => 'bx bx-truck',
            'sale' => 'bx bx-cart',
            'purchase' => 'bx bx-shopping-bag',
            'stock' => 'bx bx-box',
            'payment' => 'bx bx-credit-card',
            'mpesa' => 'bx bx-mobile',
            'system' => 'bx bx-cog',
            'auth' => 'bx bx-shield',
            'error' => 'bx bx-error',
            'audit' => 'bx bx-clipboard',
        ];

        return $icons[$this->log_name] ?? $icons['default'];
    }

    /**
     * Get badge color based on event.
     */
    public function getBadgeColorAttribute(): string
    {
        $colors = [
            'created' => 'success',
            'updated' => 'warning',
            'deleted' => 'danger',
            'login' => 'info',
            'logout' => 'secondary',
            'error' => 'danger',
            'warning' => 'warning',
            'info' => 'info',
            'success' => 'success',
        ];

        return $colors[$this->event] ?? 'primary';
    }

    /**
     * Clean old logs automatically.
     */
    public static function cleanOldLogs($days = 90): int
    {
        return self::where('created_at', '<', now()->subDays($days))
                  ->delete();
    }
}