<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class CheckRole
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, ...$roles): Response
    {
        // Debug logging
        Log::info('CheckRole middleware triggered', [
            'user_id' => Auth::id(),
            'required_roles' => $roles,
            'path' => $request->path(),
        ]);

        // Check if user is authenticated
        if (!Auth::check()) {
            Log::warning('User not authenticated, redirecting to login');
            return redirect()->route('login');
        }

        $user = Auth::user();
        
        // Debug user roles
        Log::info('User roles check', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_roles' => $user->roles->pluck('name')->toArray(),
        ]);
        
        // Check if user has any of the required roles
        foreach ($roles as $role) {
            if ($user->hasRole($role)) {
                Log::info('Access granted', [
                    'user_id' => $user->id,
                    'granted_role' => $role,
                ]);
                return $next($request);
            }
        }

        // Log unauthorized access attempt
        Log::warning('Unauthorized access attempt', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'required_roles' => $roles,
            'user_roles' => $user->roles->pluck('name')->toArray(),
        ]);

        // If user doesn't have required role
        return redirect()->route('dashboard')
            ->with('error', 'You do not have permission to access this page.');
    }
}