<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
// Note: Removed Intervention Image import
use App\Exports\UsersExport;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view users', ['only' => ['index', 'show', 'export', 'statistics']]);
        $this->middleware('permission:create users', ['only' => ['create', 'store']]);
        $this->middleware('permission:edit users', ['only' => ['edit', 'update', 'updateProfileImage', 'updateStatus']]);
        $this->middleware('permission:delete users', ['only' => ['destroy', 'bulkDelete', 'forceDelete']]);
        $this->middleware('permission:export users', ['only' => ['export', 'exportPdf']]);
        $this->middleware('permission:import users', ['only' => ['import', 'processImport']]);
    }

    /**
     * Display a listing of users.
     */
    public function index(Request $request)
    {
        $query = User::with('roles')->latest();

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('employee_id', 'like', "%{$search}%");
            });
        }

        // Filter by role
        if ($request->has('role') && $request->role) {
            $query->whereHas('roles', function($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        // Filter by status
        if ($request->has('status') && in_array($request->status, ['active', 'inactive'])) {
            $query->where('status', $request->status === 'active');
        }

        // Filter by department
        if ($request->has('department') && $request->department) {
            $query->where('department', $request->department);
        }

        // Filter by date range
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $users = $query->paginate(20);
        $roles = Role::all();
        $departments = User::select('department')->distinct()->whereNotNull('department')->pluck('department');
        
        $totalUsers = User::count();
        $activeUsers = User::where('status', true)->count();
        $inactiveUsers = User::where('status', false)->count();

        return view('users.index', compact(
            'users', 'roles', 'departments', 'totalUsers', 
            'activeUsers', 'inactiveUsers'
        ));
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        $roles = Role::all();
        return view('users.create', compact('roles'));
    }

    /**
     * Store a newly created user in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'phone' => ['nullable', 'string', 'max:20'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'profile_image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'roles' => ['required', 'array'],
            'roles.*' => ['exists:roles,id'],
            'department' => ['nullable', 'string', 'max:100'],
            'position' => ['nullable', 'string', 'max:100'],
            'employee_id' => ['nullable', 'string', 'max:50', 'unique:users'],
            'address' => ['nullable', 'string', 'max:255'],
            'date_of_birth' => ['nullable', 'date'],
            'gender' => ['nullable', 'in:male,female,other'],
            'hire_date' => ['nullable', 'date'],
            'notes' => ['nullable', 'string'],
            'status' => ['required', 'boolean'],
        ]);

        try {
            DB::beginTransaction();

            $userData = [
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'password' => Hash::make($request->password),
                'department' => $request->department,
                'position' => $request->position,
                'employee_id' => $request->employee_id,
                'address' => $request->address,
                'date_of_birth' => $request->date_of_birth,
                'gender' => $request->gender,
                'hire_date' => $request->hire_date,
                'notes' => $request->notes,
                'status' => $request->status,
            ];

            // Handle profile image upload - SIMPLE VERSION without Intervention Image
            if ($request->hasFile('profile_image')) {
                $image = $request->file('profile_image');
                $filename = 'user_' . time() . '.' . $image->getClientOriginalExtension();
                
                // Store the image
                $path = $image->storeAs('profile-images', $filename, 'public');
                $userData['profile_image'] = $filename;
            }

            $user = User::create($userData);

            // Assign roles
            $roles = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
            $user->assignRole($roles);

            // Log activity
            activity()
                ->causedBy(Auth::user())
                ->performedOn($user)
                ->withProperties(['roles' => $roles])
                ->log('created user');

            DB::commit();

            return redirect()->route('users.index')
                ->with('success', 'User created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating user: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified user.
     */
    public function show(User $user)
    {
        $user->load('roles', 'activities', 'sales', 'purchases', 'loginHistory');
        $permissions = $user->getAllPermissions();
        $statistics = $user->getStatistics();
        
        // Get recent activities
        $recentActivities = $user->activities()
            ->latest()
            ->take(10)
            ->get();
            
        // Get recent sales
        $recentSales = $user->sales()
            ->with(['customer', 'items'])
            ->latest()
            ->take(5)
            ->get();
            
        // Get login history
        $loginHistory = $user->loginHistory()
            ->latest()
            ->take(10)
            ->get();

        return view('users.show', compact(
            'user', 'permissions', 'statistics', 
            'recentActivities', 'recentSales', 'loginHistory'
        ));
    }

    /**
     * Show the form for editing the specified user.
     */
    public function edit(User $user)
    {
        // Prevent editing super admin unless you're super admin
        if ($user->hasRole('super-admin') && !auth()->user()->hasRole('super-admin')) {
            abort(403, 'Unauthorized action.');
        }

        $roles = Role::all();
        $userRoles = $user->roles->pluck('id')->toArray();

        return view('users.edit', compact('user', 'roles', 'userRoles'));
    }

    /**
     * Update the specified user in storage.
     */
    public function update(Request $request, User $user)
    {
        // Prevent editing super admin unless you're super admin
        if ($user->hasRole('super-admin') && !auth()->user()->hasRole('super-admin')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
            'profile_image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'roles' => ['required', 'array'],
            'roles.*' => ['exists:roles,id'],
            'department' => ['nullable', 'string', 'max:100'],
            'position' => ['nullable', 'string', 'max:100'],
            'employee_id' => ['nullable', 'string', 'max:50', 'unique:users,employee_id,' . $user->id],
            'address' => ['nullable', 'string', 'max:255'],
            'date_of_birth' => ['nullable', 'date'],
            'gender' => ['nullable', 'in:male,female,other'],
            'hire_date' => ['nullable', 'date'],
            'notes' => ['nullable', 'string'],
            'status' => ['required', 'boolean'],
        ]);

        try {
            DB::beginTransaction();

            $userData = [
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'department' => $request->department,
                'position' => $request->position,
                'employee_id' => $request->employee_id,
                'address' => $request->address,
                'date_of_birth' => $request->date_of_birth,
                'gender' => $request->gender,
                'hire_date' => $request->hire_date,
                'notes' => $request->notes,
                'status' => $request->status,
            ];

            // Update password if provided
            if ($request->filled('password')) {
                $userData['password'] = Hash::make($request->password);
            }

            // Handle profile image upload - SIMPLE VERSION without Intervention Image
            if ($request->hasFile('profile_image')) {
                // Delete old image if exists
                if ($user->profile_image) {
                    Storage::disk('public')->delete('profile-images/' . $user->profile_image);
                }
                
                $image = $request->file('profile_image');
                $filename = 'user_' . $user->id . '_' . time() . '.' . $image->getClientOriginalExtension();
                
                // Store the image
                $path = $image->storeAs('profile-images', $filename, 'public');
                $userData['profile_image'] = $filename;
            }

            $user->update($userData);

            // Sync roles
            $roles = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
            $user->syncRoles($roles);

            // Log activity
            activity()
                ->causedBy(Auth::user())
                ->performedOn($user)
                ->withProperties(['roles' => $roles])
                ->log('updated user');

            DB::commit();

            return redirect()->route('users.show', $user)
                ->with('success', 'User updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating user: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified user from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deleting yourself
        if ($user->id === Auth::id()) {
            return redirect()->back()->with('error', 'You cannot delete your own account.');
        }

        // Prevent deleting super admin unless you're super admin
        if ($user->hasRole('super-admin') && !auth()->user()->hasRole('super-admin')) {
            abort(403, 'Unauthorized action.');
        }

        // Check if user has related records
        if ($user->hasSales() || $user->hasPurchases()) {
            return redirect()->back()->with('warning', 
                'Cannot delete user with related records. Consider deactivating instead.');
        }

        try {
            // Delete profile image if exists
            if ($user->profile_image) {
                Storage::disk('public')->delete('profile-images/' . $user->profile_image);
            }

            // Log activity before deletion
            activity()
                ->causedBy(Auth::user())
                ->performedOn($user)
                ->log('deleted user');

            $user->delete();

            return redirect()->route('users.index')
                ->with('success', 'User deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting user: ' . $e->getMessage());
        }
    }

    /**
     * Update user status (activate/deactivate)
     */
    public function updateStatus(Request $request, User $user)
    {
        $request->validate([
            'status' => ['required', 'boolean']
        ]);

        $user->update(['status' => $request->status]);

        $statusText = $request->status ? 'activated' : 'deactivated';
        
        activity()
            ->causedBy(Auth::user())
            ->performedOn($user)
            ->withProperties(['status' => $request->status])
            ->log("{$statusText} user");

        return response()->json([
            'success' => true,
            'message' => "User {$statusText} successfully.",
            'status' => $user->status
        ]);
    }

    /**
     * Update profile image only
     */
    public function updateProfileImage(Request $request, User $user)
    {
        $request->validate([
            'profile_image' => ['required', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048']
        ]);

        try {
            // Delete old image if exists
            if ($user->profile_image) {
                Storage::disk('public')->delete('profile-images/' . $user->profile_image);
            }
            
            $image = $request->file('profile_image');
            $filename = 'user_' . $user->id . '_' . time() . '.' . $image->getClientOriginalExtension();
            
            // Store the image
            $path = $image->storeAs('profile-images', $filename, 'public');
            
            $user->update(['profile_image' => $filename]);

            activity()
                ->causedBy(Auth::user())
                ->performedOn($user)
                ->log('updated profile image');

            return response()->json([
                'success' => true,
                'message' => 'Profile image updated successfully.',
                'image_url' => $user->profile_image_url
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating profile image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete users
     */
    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['exists:users,id']
        ]);

        $users = User::whereIn('id', $request->ids)
            ->where('id', '!=', Auth::id()) // Prevent deleting yourself
            ->get();

        $deletedCount = 0;
        $errors = [];

        foreach ($users as $user) {
            try {
                // Check if user has related records
                if ($user->hasSales() || $user->hasPurchases()) {
                    $errors[] = "User {$user->name} has related records and cannot be deleted.";
                    continue;
                }

                // Delete profile image if exists
                if ($user->profile_image) {
                    Storage::disk('public')->delete('profile-images/' . $user->profile_image);
                }

                $user->delete();
                $deletedCount++;

                activity()
                    ->causedBy(Auth::user())
                    ->performedOn($user)
                    ->log('deleted user via bulk action');

            } catch (\Exception $e) {
                $errors[] = "Error deleting user {$user->name}: " . $e->getMessage();
            }
        }

        $message = "Successfully deleted {$deletedCount} users.";
        if (!empty($errors)) {
            $message .= ' Errors: ' . implode(', ', $errors);
        }

        return response()->json([
            'success' => $deletedCount > 0,
            'message' => $message,
            'deleted_count' => $deletedCount,
            'errors' => $errors
        ]);
    }

    /**
     * Export users to Excel
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'excel');
        
        if ($type === 'pdf') {
            return $this->exportPdf($request);
        }

        $filename = 'users_' . date('Y-m-d_H-i-s') . '.xlsx';
        return Excel::download(new UsersExport($request->all()), $filename);
    }

    /**
     * Export users to PDF
     */
    public function exportPdf(Request $request)
    {
        $query = User::with('roles');

        // Apply filters
        if ($request->has('role') && $request->role) {
            $query->whereHas('roles', function($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        if ($request->has('status') && in_array($request->status, ['active', 'inactive'])) {
            $query->where('status', $request->status === 'active');
        }

        $users = $query->get();

        $pdf = Pdf::loadView('users.exports.pdf', compact('users'))
            ->setPaper('a4', 'landscape');

        return $pdf->download('users_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Show import form
     */
    public function import()
    {
        return view('users.import');
    }

    /**
     * Process user import
     */
    public function processImport(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:xlsx,xls,csv'],
            'import_type' => ['required', 'in:create,update']
        ]);

        try {
            $import = new UsersImport($request->import_type);
            Excel::import($import, $request->file('file'));

            $imported = $import->getImportedCount();
            $updated = $import->getUpdatedCount();
            $failed = $import->getFailedCount();
            $errors = $import->getErrors();

            activity()
                ->causedBy(Auth::user())
                ->withProperties([
                    'imported' => $imported,
                    'updated' => $updated,
                    'failed' => $failed
                ])
                ->log('imported users');

            return redirect()->route('users.index')
                ->with('success', "Import completed: {$imported} created, {$updated} updated, {$failed} failed.")
                ->with('import_errors', $errors);

        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error importing users: ' . $e->getMessage());
        }
    }

    /**
     * Get user statistics
     */
    public function statistics()
    {
        $totalUsers = User::count();
        $activeUsers = User::where('status', true)->count();
        $inactiveUsers = User::where('status', false)->count();
        
        // Users by role
        $roles = Role::withCount('users')->get();
        
        // Users by department
        $usersByDepartment = User::select('department', DB::raw('count(*) as count'))
            ->whereNotNull('department')
            ->groupBy('department')
            ->get();
            
        // Monthly user growth
        $monthlyGrowth = User::select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
                DB::raw('count(*) as count')
            )
            ->where('created_at', '>=', now()->subYear())
            ->groupBy('month')
            ->orderBy('month')
            ->get();
            
        // Recent logins
        $recentLogins = User::whereNotNull('last_login_at')
            ->orderBy('last_login_at', 'desc')
            ->take(10)
            ->get();

        return view('users.statistics', compact(
            'totalUsers', 'activeUsers', 'inactiveUsers',
            'roles', 'usersByDepartment', 'monthlyGrowth', 'recentLogins'
        ));
    }

    /**
     * Force delete user (for admins only)
     */
    public function forceDelete(User $user)
    {
        if (!auth()->user()->hasRole('super-admin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            // Get user data for logging
            $userData = $user->toArray();
            
            // Delete all related data
            $user->sales()->delete();
            $user->purchases()->delete();
            $user->activities()->delete();
            $user->loginHistory()->delete();
            $user->notifications()->delete();
            $user->auditLogs()->delete();
            
            // Delete profile image
            if ($user->profile_image) {
                Storage::disk('public')->delete('profile-images/' . $user->profile_image);
            }
            
            // Force delete user
            $user->forceDelete();

            activity()
                ->causedBy(Auth::user())
                ->withProperties(['user_data' => $userData])
                ->log('force deleted user');

            return redirect()->route('users.index')
                ->with('success', 'User permanently deleted.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error permanently deleting user: ' . $e->getMessage());
        }
    }

    /**
     * Reset user password (admin function)
     */
    public function resetPassword(Request $request, User $user)
    {
        $request->validate([
            'password' => ['required', 'confirmed', Rules\Password::defaults()]
        ]);

        $user->update([
            'password' => Hash::make($request->password)
        ]);

        activity()
            ->causedBy(Auth::user())
            ->performedOn($user)
            ->log('reset user password');

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully.'
        ]);
    }
}