<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use App\Models\StoreInformation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class UnifiedReceiptController extends Controller
{
    /**
     * Cache TTL for store information (1 hour)
     */
    const STORE_CACHE_TTL = 3600;
    
    /**
     * Display unified receipt
     * 
     * @param string $identifier Unified number, invoice, or receipt number
     * @return \Illuminate\View\View
     */
    public function show($identifier, Request $request)
    {
        // Log receipt request for audit trail
        Log::info('Unified receipt request', [
            'identifier' => $identifier,
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'autoprint' => $request->boolean('autoprint'),
            'referrer' => $request->header('referer'),
        ]);
        
        try {
            // 1. Find the sale with unified number priority
            $sale = $this->findSaleByIdentifier($identifier);
            
            if (!$sale) {
                Log::warning('Sale not found for receipt', ['identifier' => $identifier]);
                return $this->renderNotFound($identifier, $request);
            }
            
            // 2. Verify sale belongs to current user's store (if needed)
            if (!$this->authorizeSale($sale)) {
                Log::warning('Unauthorized receipt access', [
                    'identifier' => $identifier,
                    'user_id' => auth()->id(),
                    'sale_user_id' => $sale->user_id
                ]);
                return $this->renderUnauthorized($identifier);
            }
            
            // 3. Prepare receipt data
            $receiptData = $this->prepareReceiptData($sale);
            
            // 4. Get store information (cached)
            $storeData = $this->getStoreInformation();
            $receiptData['store'] = $storeData;
            
            // 5. Add request parameters
            $receiptData['autoprint'] = $request->boolean('autoprint');
            $receiptData['download'] = $request->boolean('download');
            $receiptData['timestamp'] = now()->toISOString();
            
            // 6. Log successful receipt generation
            Log::info('Receipt generated successfully', [
                'sale_id' => $sale->id,
                'unified_number' => $sale->unified_number,
                'total' => $sale->grand_total,
                'customer_id' => $sale->customer_id,
            ]);
            
            // 7. Determine view template based on request
            $template = $this->getTemplate($request);
            
            return view($template, [
                'data' => $receiptData,
                'sale' => $sale,
                'store' => $storeData,
                'cache_info' => [
                    'store_cached' => Cache::has('store_info_receipt'),
                    'timestamp' => now()->toISOString(),
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Unified receipt generation failed', [
                'identifier' => $identifier,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            return $this->renderError($identifier, $e->getMessage(), $request);
        }
    }
    
    /**
     * Find sale by any identifier
     */
    private function findSaleByIdentifier(string $identifier): ?Sale
    {
        // Priority order: unified_number -> invoice_no -> receipt_no -> id
        return Sale::with([
            'items.product', 
            'customer', 
            'user', 
            'payments.paymentMethod',
            'mpesaTransactions'
        ])
        ->where('unified_number', $identifier)
        ->orWhere('invoice_no', $identifier)
        ->orWhere('receipt_no', $identifier)
        ->orWhere('id', $identifier)
        ->first();
    }
    
    /**
     * Authorize the sale belongs to current user's store
     */
    private function authorizeSale(Sale $sale): bool
    {
        // If no auth required, return true
        if (!auth()->check()) {
            return true;
        }
        
        $user = auth()->user();
        
        // Admin can view all receipts
        if ($user->hasRole('admin')) {
            return true;
        }
        
        // Cashier can view their own receipts
        if ($user->hasRole('cashier') && $sale->user_id === $user->id) {
            return true;
        }
        
        // Manager can view all receipts from their branch
        if ($user->hasRole('manager')) {
            // Implement branch logic if you have multi-branch
            return true;
        }
        
        return false;
    }
    
    /**
     * Prepare comprehensive receipt data
     */
    private function prepareReceiptData(Sale $sale): array
    {
        // Calculate VAT breakdown
        $vatBreakdown = $this->calculateVatBreakdown($sale);
        
        // Prepare items
        $items = [];
        foreach ($sale->items as $item) {
            $items[] = [
                'id' => $item->product_id,
                'name' => $item->product->name ?? 'Product',
                'sku' => $item->product->sku ?? '',
                'quantity' => $item->qty,
                'unit_price' => $item->price,
                'price_ex_tax' => $item->price_ex_tax ?? $item->price,
                'tax_rate' => $item->tax_rate,
                'tax_amount' => $item->tax_amount,
                'total' => $item->total,
                'is_vatable' => $item->tax_rate > 0,
                'vat_type' => $this->getVatType($item->tax_rate, $sale->customer->vat_status ?? 'vatable'),
            ];
        }
        
        // Customer data
        $customer = $sale->customer ? [
            'id' => $sale->customer_id,
            'name' => $sale->customer->name,
            'phone' => $sale->customer->phone,
            'email' => $sale->customer->email,
            'company_name' => $sale->customer->company_name,
            'address' => $sale->customer->address,
            'pin' => $sale->customer->pin,
            'vat_number' => $sale->customer->vat_number,
            'vat_status' => $sale->customer->vat_status ?? 'vatable',
        ] : [
            'name' => 'Walk-in Customer',
            'vat_status' => 'vatable',
        ];
        
        // Payment data
        $paymentData = $this->preparePaymentData($sale);
        
        // Totals
        $totals = [
            'subtotal' => $sale->subtotal,
            'discount' => $sale->discount,
            'vat_amount' => $sale->tax_total,
            'taxable_amount' => $sale->taxable_amount ?? 0,
            'non_taxable_amount' => $sale->non_taxable_amount ?? 0,
            'zero_rated_amount' => $sale->zero_rated_amount ?? 0,
            'exempted_amount' => $sale->exempted_amount ?? 0,
            'grand_total' => $sale->grand_total,
            'total_items' => $sale->items->sum('qty'),
        ];
        
        // VAT info
        $vatInfo = [
            'customer_status' => $customer['vat_status'],
            'is_vat_applicable' => $sale->is_vat_applied ?? true,
            'is_zero_rated' => $sale->zero_rated ?? false,
            'is_exempted' => $sale->exempted ?? false,
            'breakdown' => $vatBreakdown,
            'total_vat' => $sale->tax_total,
        ];
        
        return [
            // Basic Info
            'id' => $sale->id,
            'unified_number' => $sale->unified_number,
            'invoice_no' => $sale->invoice_no,
            'receipt_no' => $sale->receipt_no,
            'transaction_id' => $sale->transaction_id,
            
            // Dates
            'date' => $sale->created_at->format('Y-m-d'),
            'time' => $sale->created_at->format('H:i:s'),
            'timestamp' => $sale->created_at->toISOString(),
            
            // Entities
            'customer' => $customer,
            
            // Items
            'items' => $items,
            
            // Totals
            'totals' => $totals,
            
            // VAT Info
            'vat_info' => $vatInfo,
            
            // Payment
            'payment' => $paymentData,
            
            // Personnel
            'cashier' => $sale->user->name ?? 'Cashier',
            'user_id' => $sale->user_id,
            
            // Notes (configurable)
            'notes' => $this->getReceiptNotes($sale),
            
            // Metadata
            'metadata' => [
                'sale_status' => $sale->status,
                'payment_status' => $sale->payment_status,
                'is_vat_applied' => $sale->is_vat_applied,
                'created_at' => $sale->created_at->toIso8601String(),
            ]
        ];
    }
    
    /**
     * Get store information with caching
     */
    private function getStoreInformation(): array
    {
        return Cache::remember('store_info_receipt', self::STORE_CACHE_TTL, function () {
            $store = StoreInformation::first();
            
            if ($store) {
                return [
                    'name' => $store->store_name,
                    'address' => $store->address,
                    'phone' => $store->telephone,
                    'whatsapp' => $store->whatsapp_number,
                    'email' => $store->email,
                    'pin' => $store->company_pin,
                    'vat_number' => $store->vat_number,
                    'branch_code' => $store->branch_code,
                    'city' => $store->city,
                    'country' => $store->country,
                    'currency' => $store->currency,
                    'timezone' => $store->timezone,
                    'logo' => $this->getLogoUrl($store->logo),
                    'business_registration' => $store->business_registration,
                    'kra_pin' => $store->kra_pin,
                    'permit_number' => $store->permit_number,
                    'source' => 'database',
                ];
            }
            
            // Fallback to config
            return [
                'name' => config('pos.store.name', 'Kenyan Supermarket'),
                'address' => config('pos.store.address', 'Nairobi CBD'),
                'phone' => config('pos.store.phone', '0700 000 000'),
                'email' => config('pos.store.email', ''),
                'pin' => config('pos.store.company_pin', 'P051234567N'),
                'vat_number' => config('pos.store.vat_number', 'VAT001234567'),
                'branch_code' => config('pos.store.branch_code', '001'),
                'city' => config('pos.store.city', 'Nairobi'),
                'country' => config('pos.store.country', 'Kenya'),
                'currency' => config('pos.store.currency', 'KES'),
                'timezone' => config('pos.store.timezone', 'Africa/Nairobi'),
                'logo' => config('pos.store.logo', ''),
                'source' => 'config',
            ];
        });
    }
    
    /**
     * Get logo URL
     */
    private function getLogoUrl($logo): string
    {
        if (!$logo) {
            return '';
        }
        
        if (filter_var($logo, FILTER_VALIDATE_URL)) {
            return $logo;
        }
        
        // Local file
        return asset('storage/' . $logo);
    }
    
    /**
     * Calculate VAT breakdown
     */
    private function calculateVatBreakdown(Sale $sale): array
    {
        $breakdown = [];
        
        foreach ($sale->items as $item) {
            if ($item->tax_rate > 0) {
                $rateKey = (string)$item->tax_rate;
                if (!isset($breakdown[$rateKey])) {
                    $breakdown[$rateKey] = [
                        'rate' => $item->tax_rate,
                        'taxable_amount' => 0,
                        'vat_amount' => 0,
                        'type' => 'standard',
                    ];
                }
                
                $itemTotal = $item->price_ex_tax * $item->qty;
                $breakdown[$rateKey]['taxable_amount'] += $itemTotal;
                $breakdown[$rateKey]['vat_amount'] += $item->tax_amount;
            }
        }
        
        return array_values($breakdown);
    }
    
    /**
     * Prepare payment data
     */
    private function preparePaymentData(Sale $sale): array
    {
        $data = [
            'method' => $sale->payment_method,
            'amount_paid' => $sale->grand_total,
            'status' => $sale->payment_status,
            'transaction_id' => $sale->transaction_id,
        ];
        
        // Method-specific details
        switch ($sale->payment_method) {
            case 'cash':
                $data['cash_received'] = $sale->cash_received;
                $data['change'] = $sale->customer_change;
                break;
            case 'mpesa':
                $mpesa = $sale->mpesaTransactions->first();
                if ($mpesa) {
                    $data['mpesa_details'] = [
                        'phone' => $mpesa->phone,
                        'transaction_id' => $mpesa->transaction_id,
                        'receipt_no' => $mpesa->receipt_no,
                    ];
                }
                break;
            case 'card':
                $data['card_last4'] = substr($sale->transaction_id ?? '', -4) ?: '****';
                break;
        }
        
        return $data;
    }
    
    /**
     * Get receipt notes
     */
    private function getReceiptNotes(Sale $sale): array
    {
        $notes = config('pos.receipts.notes', [
            'Thank you for your business!',
            'Items sold are not returnable unless defective',
            'Please check goods before leaving the counter',
            'Valid for 7 days from date of purchase',
            'Keep this receipt for warranty claims',
            'VAT Invoice - Valid for Tax Purposes',
        ]);
        
        // Add VAT-specific notes
        if ($sale->exempted) {
            $notes[] = 'VAT Exempted Customer - No VAT Applied';
        } elseif ($sale->zero_rated) {
            $notes[] = 'Zero-Rated Customer - 0% VAT Rate';
        }
        
        return $notes;
    }
    
    /**
     * Determine template based on request
     */
    private function getTemplate(Request $request): string
    {
        // Check for specific template parameter
        if ($request->has('template')) {
            $template = $request->get('template');
            if (view()->exists("receipts.templates.{$template}")) {
                return "receipts.templates.{$template}";
            }
        }
        
        // Default to your test.blade.php
        return 'receipt.test';
    }
    
    /**
     * Get VAT type
     */
    private function getVatType($taxRate, $customerStatus): string
    {
        if ($customerStatus === 'exempted') {
            return 'exempted';
        }
        
        if ($customerStatus === 'zero_rated') {
            return 'zero_rated';
        }
        
        return $taxRate > 0 ? 'standard' : 'none';
    }
    
    /**
     * Render not found page
     */
    private function renderNotFound($identifier, Request $request)
    {
        if ($request->expectsJson() || $request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Receipt not found',
                'identifier' => $identifier,
                'suggestions' => [
                    'Check the unified number format',
                    'Verify the sale exists in database',
                    'Contact support if issue persists',
                ]
            ], 404);
        }
        
        return view('receipt.error', [
            'identifier' => $identifier,
            'error' => 'Receipt not found',
            'title' => 'Receipt Not Found',
            'suggestions' => [
                'Please check the receipt number',
                'The sale may not be in our system',
                'Contact customer support for assistance',
            ]
        ]);
    }
    
    /**
     * Render unauthorized page
     */
    private function renderUnauthorized($identifier)
    {
        return view('receipt.error', [
            'identifier' => $identifier,
            'error' => 'Unauthorized access',
            'title' => 'Access Denied',
            'suggestions' => [
                'You are not authorized to view this receipt',
                'Please contact your supervisor',
                'Log in with appropriate credentials',
            ]
        ]);
    }
    
    /**
     * Render error page
     */
    private function renderError($identifier, $error, Request $request)
    {
        if ($request->expectsJson() || $request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Receipt generation failed',
                'error' => $error,
                'identifier' => $identifier,
            ], 500);
        }
        
        return view('receipt.error', [
            'identifier' => $identifier,
            'error' => 'System error: ' . $error,
            'title' => 'System Error',
            'suggestions' => [
                'Please try again later',
                'Contact technical support',
                'Reference this error ID: ' . uniqid(),
            ]
        ]);
    }
    
    /**
     * Download receipt as PDF
     */
    public function download($identifier, Request $request)
    {
        // Similar to show() but return PDF
        // You'll need to implement PDF generation
        // For now, redirect to show with download param
        return redirect()->route('receipt.show', [
            'identifier' => $identifier,
            'download' => true
        ]);
    }
    
    /**
     * Verify receipt exists (for API)
     */
    public function verify($identifier, Request $request)
    {
        $sale = $this->findSaleByIdentifier($identifier);
        
        if (!$sale) {
            return response()->json([
                'exists' => false,
                'identifier' => $identifier,
                'message' => 'Receipt not found',
            ], 404);
        }
        
        return response()->json([
            'exists' => true,
            'sale' => [
                'id' => $sale->id,
                'unified_number' => $sale->unified_number,
                'invoice_no' => $sale->invoice_no,
                'receipt_no' => $sale->receipt_no,
                'grand_total' => $sale->grand_total,
                'date' => $sale->created_at->format('Y-m-d H:i:s'),
                'status' => $sale->status,
            ],
            'verification' => [
                'timestamp' => now()->toISOString(),
                'verified_by' => auth()->check() ? auth()->user()->name : 'system',
            ]
        ]);
    }
}