<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\Purchase;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;

class SupplierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Supplier::query();
            
            // Search functionality
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->input('search');
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('address', 'like', "%{$search}%")
                      ->orWhere('contact_person', 'like', "%{$search}%");
                });
            }
            
            // Filter by purchase count
            if ($request->has('filter') && !empty($request->filter)) {
                switch ($request->filter) {
                    case 'active':
                        $query->has('purchases');
                        break;
                    case 'inactive':
                        $query->doesntHave('purchases');
                        break;
                    case 'status_active':
                        $query->where('status', 'active');
                        break;
                    case 'status_inactive':
                        $query->where('status', 'inactive');
                        break;
                }
            }
            
            // Sorting
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            
            if (in_array($sortBy, ['name', 'email', 'created_at', 'updated_at'])) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Load purchases count
            $query->withCount('purchases');
            
            $suppliers = $query->paginate(20)->withQueryString();
            
            // Statistics for dashboard
            $stats = [
                'total' => Supplier::count(),
                'with_purchases' => Supplier::has('purchases')->count(),
                'active_this_month' => Supplier::whereHas('purchases', function($q) {
                    $q->whereMonth('created_at', now()->month);
                })->count(),
            ];
            
            return view('suppliers.index', compact('suppliers', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('Supplier index error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('dashboard')
                ->with('error', 'Failed to load suppliers. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('suppliers.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $validated = $request->validate([
                'name' => 'required|string|max:255|unique:suppliers,name',
                'email' => 'nullable|email|unique:suppliers,email',
                'phone' => 'nullable|string|max:20',
                'address' => 'nullable|string|max:500',
                'contact_person' => 'nullable|string|max:255',
                'website' => 'nullable|url|max:255',
                'tax_id' => 'nullable|string|max:50',
                'payment_terms' => 'nullable|string|max:100',
                'notes' => 'nullable|string',
                'status' => 'nullable|in:active,inactive',
            ]);
            
            $supplier = Supplier::create($validated);
            
            DB::commit();
            
            return redirect()->route('suppliers.show', $supplier)
                ->with('success', 'Supplier created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Supplier store error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->back()
                ->with('error', 'Failed to create supplier. Please try again.')
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Supplier $supplier)
    {
        try {
            // Debug logging
            Log::info('Showing supplier: ' . $supplier->id);
            
            // Load relationships efficiently
            $supplier->loadCount('purchases');
            
            // Get recent purchases with pagination
            $purchases = $supplier->purchases()
                ->with(['items.product'])
                ->orderBy('created_at', 'desc')
                ->paginate(10);
            
            // Get purchase statistics
            $purchaseStats = [
                'total_purchases' => $supplier->purchases_count,
                'total_spent' => $supplier->purchases()->sum('grand_total') ?? 0,
                'avg_order_value' => $supplier->purchases()->avg('grand_total') ?? 0,
                'last_purchase' => $supplier->purchases()->latest()->first()?->created_at?->format('M d, Y'),
            ];
            
            return view('suppliers.show', compact('supplier', 'purchases', 'purchaseStats'));
            
        } catch (\Exception $e) {
            Log::error('Supplier show error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('suppliers.index')
                ->with('error', 'Failed to load supplier details: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Supplier $supplier)
    {
        return view('suppliers.edit', compact('supplier'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Supplier $supplier)
    {
        try {
            DB::beginTransaction();
            
            $validated = $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('suppliers')->ignore($supplier->id),
                ],
                'email' => [
                    'nullable',
                    'email',
                    Rule::unique('suppliers')->ignore($supplier->id),
                ],
                'phone' => 'nullable|string|max:20',
                'address' => 'nullable|string|max:500',
                'contact_person' => 'nullable|string|max:255',
                'website' => 'nullable|url|max:255',
                'tax_id' => 'nullable|string|max:50',
                'payment_terms' => 'nullable|string|max:100',
                'notes' => 'nullable|string',
                'status' => 'nullable|in:active,inactive',
            ]);
            
            $supplier->update($validated);
            
            DB::commit();
            
            return redirect()->route('suppliers.show', $supplier)
                ->with('success', 'Supplier updated successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Supplier update error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->back()
                ->with('error', 'Failed to update supplier. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     * FIXED: Properly checks for purchases before deletion
     */
    public function destroy(Supplier $supplier)
    {
        try {
            DB::beginTransaction();
            
            // Check if supplier has any purchases (including soft deleted purchases if you have that)
            if ($supplier->purchases()->count() > 0) {
                return redirect()->route('suppliers.index')
                    ->with('error', 'Cannot delete supplier with existing purchase records. Please delete purchases first or deactivate the supplier.');
            }
            
            $supplier->delete();
            
            DB::commit();
            
            return redirect()->route('suppliers.index')
                ->with('success', 'Supplier deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Supplier destroy error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('suppliers.index')
                ->with('error', 'Failed to delete supplier. Please try again.');
        }
    }

    /**
     * Bulk delete suppliers.
     * FIXED: Proper validation and error handling
     */
    public function bulkDelete(Request $request)
    {
        try {
            $request->validate([
                'ids' => 'required|array',
                'ids.*' => 'exists:suppliers,id'
            ]);

            $suppliers = Supplier::withCount('purchases')->whereIn('id', $request->ids)->get();
            
            // Separate suppliers with and without purchases
            $suppliersWithPurchases = $suppliers->filter(function($supplier) {
                return $supplier->purchases_count > 0;
            });

            $suppliersWithoutPurchases = $suppliers->filter(function($supplier) {
                return $supplier->purchases_count === 0;
            });

            if ($suppliersWithPurchases->count() > 0) {
                $names = $suppliersWithPurchases->pluck('name')->implode(', ');
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete suppliers with purchase records: ' . $names
                ], 400);
            }

            DB::beginTransaction();
            
            foreach ($suppliersWithoutPurchases as $supplier) {
                $supplier->delete();
            }
            
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $suppliersWithoutPurchases->count() . ' supplier(s) deleted successfully.'
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error: ' . implode(', ', $e->errors())
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Supplier bulk delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete suppliers. Please try again.'
            ], 500);
        }
    }

    /**
     * Search suppliers for AJAX requests.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            if (empty($search)) {
                return response()->json([]);
            }
            
            $suppliers = Supplier::where('name', 'like', "%{$search}%")
                ->orWhere('email', 'like', "%{$search}%")
                ->orWhere('phone', 'like', "%{$search}%")
                ->limit(10)
                ->get(['id', 'name', 'email', 'phone']);
            
            return response()->json($suppliers);
            
        } catch (\Exception $e) {
            Log::error('Supplier search error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([], 500);
        }
    }

    /**
     * Export suppliers to CSV.
     */
    public function export(Request $request)
    {
        try {
            $suppliers = Supplier::withCount('purchases')->get();
            
            $fileName = 'suppliers-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($suppliers) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'ID', 'Name', 'Email', 'Phone', 'Address', 
                    'Contact Person', 'Website', 'Tax ID', 'Payment Terms',
                    'Status', 'Total Purchases', 'Created At'
                ]);
                
                // Data
                foreach ($suppliers as $supplier) {
                    fputcsv($file, [
                        $supplier->id,
                        $supplier->name,
                        $supplier->email ?? '',
                        $supplier->phone ?? '',
                        $supplier->address ?? '',
                        $supplier->contact_person ?? '',
                        $supplier->website ?? '',
                        $supplier->tax_id ?? '',
                        $supplier->payment_terms ?? '',
                        $supplier->status ?? 'active',
                        $supplier->purchases_count,
                        $supplier->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Supplier export error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('suppliers.index')
                ->with('error', 'Failed to export suppliers.');
        }
    }

    /**
     * Get supplier statistics for dashboard.
     */
    public function statistics()
    {
        try {
            $stats = [
                'total' => Supplier::count(),
                'active' => Supplier::has('purchases')->count(),
                'new_this_month' => Supplier::whereMonth('created_at', now()->month)->count(),
                'top_suppliers' => Supplier::withCount('purchases')
                    ->orderBy('purchases_count', 'desc')
                    ->limit(5)
                    ->get(['id', 'name', 'email', 'purchases_count']),
            ];
            
            return response()->json($stats);
            
        } catch (\Exception $e) {
            Log::error('Supplier statistics error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json(['error' => 'Failed to load statistics'], 500);
        }
    }

    /**
     * Show import form.
     */
    public function import()
    {
        return view('suppliers.import');
    }

    /**
     * Process imported suppliers.
     */
    public function processImport(Request $request)
    {
        try {
            $request->validate([
                'file' => 'required|mimes:csv,txt|max:2048'
            ]);

            $file = $request->file('file');
            $imported = 0;
            $errors = [];

            if (($handle = fopen($file->path(), 'r')) !== FALSE) {
                // Skip header
                fgetcsv($handle);
                
                while (($data = fgetcsv($handle)) !== FALSE) {
                    try {
                        Supplier::create([
                            'name' => $data[0] ?? null,
                            'email' => $data[1] ?? null,
                            'phone' => $data[2] ?? null,
                            'address' => $data[3] ?? null,
                            'contact_person' => $data[4] ?? null,
                            'website' => $data[5] ?? null,
                            'tax_id' => $data[6] ?? null,
                            'payment_terms' => $data[7] ?? null,
                            'notes' => $data[8] ?? null,
                            'status' => $data[9] ?? 'active',
                        ]);
                        $imported++;
                    } catch (\Exception $e) {
                        $errors[] = "Row " . ($imported + 1) . ": " . $e->getMessage();
                    }
                }
                fclose($handle);
            }

            $message = "Successfully imported {$imported} suppliers.";
            if (!empty($errors)) {
                $message .= " " . count($errors) . " errors occurred.";
            }

            return redirect()->route('suppliers.index')
                ->with('success', $message)
                ->with('import_errors', $errors);

        } catch (\Exception $e) {
            Log::error('Supplier import error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to import suppliers. Please check the file format.')
                ->withInput();
        }
    }

    /**
     * Get suppliers for dropdown/select2.
     */
    public function getSuppliers(Request $request)
    {
        try {
            $search = $request->input('search', '');
            
            $suppliers = Supplier::when($search, function($query) use ($search) {
                return $query->where('name', 'like', "%{$search}%");
            })
            ->orderBy('name')
            ->limit(50)
            ->get(['id', 'name', 'email', 'phone']);
            
            return response()->json($suppliers);
            
        } catch (\Exception $e) {
            Log::error('Get suppliers error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }
}