<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use Illuminate\Http\Request;
use PDF;

class ReceiptController extends Controller
{
    /**
     * Show receipt for a sale
     */
    public function show($invoice_no)
    {
        try {
            // Find the sale by invoice or receipt number
            $sale = Sale::where('invoice_no', $invoice_no)
                        ->orWhere('receipt_no', $invoice_no)
                        ->first();
            
            if (!$sale) {
                // For testing, create a mock sale
                return $this->showMockReceipt($invoice_no);
            }
            
            // Load relationships
            $sale->load(['items.product', 'customer', 'user', 'payments']);
            
            // Prepare receipt data
            $data = $this->prepareReceiptData($sale);
            
            // Default settings
            $setting = (object)[
                'print_store_name' => true,
                'print_store_address' => true,
                'print_store_phone' => true,
                'print_store_logo' => false,
                'print_date_time' => true,
                'print_transaction_number' => true,
                'print_cashier_name' => true,
                'print_customer_info' => true,
                'print_item_barcode' => false,
                'print_tax_breakdown' => true,
                'print_discounts' => true,
                'print_payment_method' => true,
                'print_change_amount' => true,
                'print_qr_code' => true,
                'print_footer' => true,
                'print_vat_number' => true,
                'print_branch_code' => true,
            ];
            
            $autoprint = request()->has('autoprint') || request()->has('print');
            
            return view('pos.receipts.receipt', [
                'data' => $data,
                'setting' => $setting,
                'autoprint' => $autoprint
            ]);
            
        } catch (\Exception $e) {
            // Fallback to mock receipt
            return $this->showMockReceipt($invoice_no);
        }
    }
    
    /**
     * Show mock receipt for testing
     */
    private function showMockReceipt($invoice_no)
    {
        $data = [
            'store' => [
                'name' => config('app.name', 'Kenyan Supermarket'),
                'address' => config('app.address', 'Nairobi CBD'),
                'phone' => config('app.phone', '0700 000 000'),
                'email' => config('app.email', 'info@supermarket.co.ke'),
                'company_pin' => 'P051234567N',
                'vat_number' => 'VAT001234567',
                'branch_code' => '001',
            ],
            'transaction' => [
                'id' => 'TXN-' . time(),
                'invoice_no' => $invoice_no,
                'receipt_no' => 'RCP-' . date('Ymd') . '-001',
                'date' => date('Y-m-d'),
                'time' => date('H:i:s'),
                'cashier' => auth()->user()->name ?? 'Cashier',
            ],
            'customer' => [
                'name' => 'Walk-in Customer',
                'phone' => '0700 000 000',
                'email' => '',
                'vat_number' => '',
            ],
            'items' => [
                [
                    'name' => 'Test Product',
                    'quantity' => 1,
                    'unit_price' => 100.00,
                    'total' => 100.00,
                    'tax_rate' => 16,
                ]
            ],
            'totals' => [
                'subtotal' => 100.00,
                'discount' => 0,
                'vat_amount' => 16.00,
                'grand_total' => 116.00,
                'total_items' => 1,
            ],
            'payment' => [
                'method' => 'cash',
                'amount_paid' => 116.00,
                'change' => 0,
                'transaction_id' => 'TXN-' . time(),
            ],
            'invoice_no' => $invoice_no,
            'receipt_no' => 'RCP-' . date('Ymd') . '-001',
            'cashier' => auth()->user()->name ?? 'Cashier',
            'date' => date('Y-m-d'),
            'time' => date('H:i:s'),
        ];
        
        $setting = (object)[
            'print_store_name' => true,
            'print_store_address' => true,
            'print_store_phone' => true,
            'print_store_logo' => false,
            'print_date_time' => true,
            'print_transaction_number' => true,
            'print_cashier_name' => true,
            'print_customer_info' => true,
            'print_item_barcode' => false,
            'print_tax_breakdown' => true,
            'print_discounts' => true,
            'print_payment_method' => true,
            'print_change_amount' => true,
            'print_qr_code' => true,
            'print_footer' => true,
            'print_vat_number' => true,
            'print_branch_code' => true,
        ];
        
        $autoprint = request()->has('autoprint') || request()->has('print');
        
        return view('pos.receipts.receipt', [
            'data' => $data,
            'setting' => $setting,
            'autoprint' => $autoprint
        ]);
    }
    
    /**
     * Prepare receipt data from sale
     */
    private function prepareReceiptData(Sale $sale): array
    {
        // Store info
        $storeInfo = [
            'name' => config('app.name', 'POS System'),
            'address' => config('app.address', 'Nairobi, Kenya'),
            'phone' => config('app.phone', '0700 000 000'),
            'email' => config('app.email', 'info@example.com'),
            'company_pin' => config('app.company_pin', 'P000000000A'),
            'vat_number' => config('app.vat_number', 'VAT000000000'),
            'branch_code' => config('app.branch_code', '001'),
        ];
        
        // Prepare items
        $items = [];
        $totalItems = 0;
        $subtotal = 0;
        $vatAmount = 0;
        
        foreach ($sale->items as $item) {
            $itemTotal = $item->qty * $item->price;
            $itemVat = $item->tax_amount ?? 0;
            
            $items[] = [
                'name' => $item->product->name ?? 'Product',
                'barcode' => $item->product->barcode ?? null,
                'quantity' => $item->qty,
                'qty' => $item->qty,
                'price' => $item->price,
                'unit_price' => $item->price,
                'total' => $itemTotal,
                'tax_rate' => $item->tax_rate ?? 16,
                'vat_amount' => $itemVat,
            ];
            
            $totalItems += $item->qty;
            $subtotal += $itemTotal;
            $vatAmount += $itemVat;
        }
        
        // Payment info
        $payment = $sale->payments->first();
        $paymentMethod = $sale->payment_method ?? 'cash';
        
        $paymentData = [
            'method' => $paymentMethod,
            'amount_paid' => $sale->grand_total,
            'transaction_id' => $sale->transaction_id,
            'change' => $sale->customer_change ?? 0,
        ];
        
        return [
            'store' => $storeInfo,
            'transaction' => [
                'id' => $sale->id,
                'invoice_no' => $sale->invoice_no,
                'receipt_no' => $sale->receipt_no,
                'date' => $sale->created_at->format('Y-m-d'),
                'time' => $sale->created_at->format('H:i:s'),
                'cashier' => $sale->user->name ?? 'System',
            ],
            'customer' => $sale->customer ? [
                'name' => $sale->customer->name,
                'phone' => $sale->customer->phone,
                'email' => $sale->customer->email,
                'vat_number' => $sale->customer->vat_number,
                'pin' => $sale->customer->pin,
            ] : null,
            'items' => $items,
            'totals' => [
                'total_items' => $totalItems,
                'subtotal' => $subtotal,
                'discount' => $sale->discount,
                'vat_amount' => $vatAmount,
                'grand_total' => $sale->grand_total,
            ],
            'payment' => $paymentData,
            'invoice_no' => $sale->invoice_no,
            'receipt_no' => $sale->receipt_no,
            'cashier' => $sale->user->name ?? 'Cashier',
            'date' => $sale->created_at->format('Y-m-d'),
            'time' => $sale->created_at->format('H:i:s'),
        ];
    }
    
    /**
     * Download receipt as PDF
     */
    public function pdf($invoice_no)
    {
        $data = $this->showMockReceipt($invoice_no)->getData();
        
        $pdf = PDF::loadView('pos.receipts.receipt', $data);
        return $pdf->download('receipt-' . $invoice_no . '.pdf');
    }
}