<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\Supplier;
use App\Models\Product;
use App\Models\PurchaseItem;
use App\Models\ProductBatch;
use App\Models\BatchStockMovement;
use App\Models\StockMovement;
use App\Models\Tax;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Purchase::with(['supplier', 'items'])
                ->withCount('items');
            
            // Search functionality
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->input('search');
                $query->where(function($q) use ($search) {
                    $q->where('invoice_no', 'like', "%{$search}%")
                      ->orWhere('reference_number', 'like', "%{$search}%")
                      ->orWhereHas('supplier', function($q) use ($search) {
                          $q->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                      });
                });
            }
            
            // Filter by status
            if ($request->has('status') && !empty($request->status)) {
                $query->where('status', $request->status);
            }
            
            // Filter by supplier
            if ($request->has('supplier_id') && !empty($request->supplier_id)) {
                $query->where('supplier_id', $request->supplier_id);
            }
            
            // Filter by date range
            if ($request->has('start_date') && !empty($request->start_date)) {
                $query->whereDate('purchase_date', '>=', $request->start_date);
            }
            
            if ($request->has('end_date') && !empty($request->end_date)) {
                $query->whereDate('purchase_date', '<=', $request->end_date);
            }
            
            // Sorting
            $sortBy = $request->get('sort_by', 'purchase_date');
            $sortOrder = $request->get('sort_order', 'desc');
            
            if (in_array($sortBy, ['invoice_no', 'purchase_date', 'grand_total', 'created_at'])) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->orderBy('purchase_date', 'desc');
            }
            
            $purchases = $query->paginate(20)->withQueryString();
            
            // Get statistics
            $stats = [
                'total_purchases' => Purchase::count(),
                'completed_purchases' => Purchase::where('status', 'completed')->count(),
                'pending_purchases' => Purchase::where('status', 'pending')->count(),
                'cancelled_purchases' => Purchase::where('status', 'cancelled')->count(),
                'total_amount' => Purchase::sum('grand_total'),
                'total_paid' => Purchase::sum('paid_amount'),
                'total_outstanding' => Purchase::sum(DB::raw('grand_total - paid_amount')),
            ];
            
            // Get suppliers for filter
            $suppliers = Supplier::where('status', 'active')->get();
            
            // Get recent suppliers
            $recentSuppliers = Supplier::has('purchases')
                ->withCount('purchases')
                ->orderBy('created_at', 'desc')
                ->limit(8)
                ->get();
            
            return view('purchases.index', compact('purchases', 'stats', 'suppliers', 'recentSuppliers'));
            
        } catch (\Exception $e) {
            Log::error('Purchase index error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('dashboard')
                ->with('error', 'Failed to load purchases. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        try {
            // Get inventory products only (not services or digital)
            $products = Product::where('status', 'active')
                ->where('is_service', false)
                ->where('is_digital', false)
                ->where('track_inventory', true)
                ->orderBy('name')
                ->get(['id', 'sku', 'name', 'cost_price', 'sale_price', 'stock', 
                       'has_expiry', 'track_batches', 'unit_id', 'category_id']);
            
            // Get services for service purchases
            $services = Product::where('status', 'active')
                ->where('is_service', true)
                ->orderBy('name')
                ->get(['id', 'name', 'sale_price', 'sku']);
            
            $suppliers = Supplier::where('status', 'active')->get();
            
            // Generate initial invoice number
            $invoiceNumber = (new Purchase())->generateUniqueInvoiceNumber();
            
            return view('purchases.create', compact('products', 'services', 'suppliers', 'invoiceNumber'));
            
        } catch (\Exception $e) {
            Log::error('Purchase create error: ' . $e->getMessage());
            return redirect()->route('purchases.index')
                ->with('error', 'Failed to load purchase form.');
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            // Convert checkbox values to boolean
            $addToStock = $request->has('add_to_stock') && $request->input('add_to_stock') !== 'false';
            $isCreditPurchase = $request->has('is_credit_purchase') && $request->input('is_credit_purchase') !== 'false';

            $validated = $request->validate([
                'supplier_id' => 'required|exists:suppliers,id',
                'invoice_no' => 'nullable|string|max:100',
                'purchase_date' => 'required|date',
                'reference_number' => 'nullable|string|max:100',
                'expected_delivery_date' => 'nullable|date|after_or_equal:purchase_date',
                'payment_status' => 'required|in:pending,partial,paid',
                'payment_method' => 'nullable|string|max:50',
                'paid_amount' => 'required|numeric|min:0',
                'status' => 'required|in:draft,pending,completed,cancelled',
                'notes' => 'nullable|string',
                'terms_conditions' => 'nullable|string',
                'due_date' => 'nullable|date|after_or_equal:purchase_date',
                
                // Purchase items
                'items' => 'required|array|min:1',
                'items.*.type' => 'required|in:product,service',
                'items.*.product_id' => 'required_if:items.*.type,product|exists:products,id',
                'items.*.description' => 'nullable|string|max:255',
                'items.*.quantity' => 'required|numeric|min:0.001',
                'items.*.unit_price' => 'required|numeric|min:0',
                'items.*.batch_number' => 'nullable|string|max:100',
                'items.*.expiry_date' => 'nullable|date|after:today',
                'items.*.notes' => 'nullable|string',
            ]);

            // Get supplier
            $supplier = Supplier::find($validated['supplier_id']);
            
            // Check credit limit if this is a credit purchase
            if ($isCreditPurchase) {
                // Calculate total amount
                $totalAmount = 0;
                foreach ($validated['items'] as $item) {
                    $totalAmount += $item['quantity'] * $item['unit_price'];
                }
                
                // Check if supplier can make this purchase within credit limit
                if (!$supplier->canMakePurchase($totalAmount)) {
                    return redirect()->back()
                        ->with('error', 'Purchase amount exceeds available credit limit. Available: KES ' . 
                               number_format($supplier->credit_available, 2))
                        ->withInput();
                }
            }

            // Generate invoice number if not provided or if it exists
            if (empty($validated['invoice_no'])) {
                $validated['invoice_no'] = (new Purchase())->generateUniqueInvoiceNumber();
            } else {
                // Check if invoice number already exists
                $existing = Purchase::where('invoice_no', $validated['invoice_no'])->first();
                if ($existing) {
                    // Generate a new unique number
                    $validated['invoice_no'] = (new Purchase())->generateUniqueInvoiceNumber();
                }
            }

            // Calculate totals with tax per item
            $subtotal = 0;
            $totalTax = 0;
            $itemsWithTax = [];

            foreach ($validated['items'] as $itemData) {
                if ($itemData['type'] === 'product') {
                    $product = Product::find($itemData['product_id']);
                    $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                    $subtotal += $itemTotal;
                    
                    // Calculate tax for this item
                    $taxRate = $product->tax_rate ?? 16.00;
                    $itemTax = $this->calculateTaxAmount($itemTotal, $taxRate);
                    $totalTax += $itemTax;
                    
                    $itemsWithTax[] = [
                        'product' => $product,
                        'tax_rate' => $taxRate,
                        'item_tax' => $itemTax,
                        'item_total' => $itemTotal,
                    ];
                } else {
                    // For services, use default tax rate
                    $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                    $subtotal += $itemTotal;
                    
                    $defaultTax = Tax::where('is_default', true)->first();
                    $taxRate = $defaultTax ? $defaultTax->rate : 16.00;
                    $itemTax = $itemTotal * ($taxRate / 100);
                    $totalTax += $itemTax;
                    
                    $itemsWithTax[] = [
                        'product' => null,
                        'tax_rate' => $taxRate,
                        'item_tax' => $itemTax,
                        'item_total' => $itemTotal,
                    ];
                }
            }

            $grandTotal = $subtotal + $totalTax;

            // Create purchase with tax total
            $purchase = Purchase::create([
                'supplier_id' => $validated['supplier_id'],
                'invoice_no' => $validated['invoice_no'],
                'purchase_date' => $validated['purchase_date'],
                'reference_number' => $validated['reference_number'] ?? null,
                'expected_delivery_date' => $validated['expected_delivery_date'] ?? null,
                'subtotal' => $subtotal,
                'tax_amount' => $totalTax,
                'grand_total' => $grandTotal,
                'paid_amount' => $validated['paid_amount'],
                'payment_status' => $validated['payment_status'],
                'payment_method' => $validated['payment_method'] ?? null,
                'status' => $validated['status'],
                'notes' => $validated['notes'] ?? null,
                'terms_conditions' => $validated['terms_conditions'] ?? null,
                'created_by' => auth()->id(),
                'is_credit_purchase' => $isCreditPurchase,
                'due_date' => $validated['due_date'] ?? null,
            ]);

            // Process items with tax
            foreach ($validated['items'] as $index => $itemData) {
                $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                $itemDataWithTax = $itemsWithTax[$index] ?? null;
                
                if ($itemData['type'] === 'product') {
                    $product = Product::find($itemData['product_id']);
                    $description = $product ? $product->name : 'Product #' . $itemData['product_id'];
                } else {
                    $description = $itemData['description'] ?? 'Service';
                }

                $purchaseItem = PurchaseItem::create([
                    'purchase_id' => $purchase->id,
                    'product_id' => $itemData['type'] === 'product' ? $itemData['product_id'] : null,
                    'item_type' => $itemData['type'],
                    'description' => $description,
                    'quantity' => $itemData['quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'tax_percent' => $itemDataWithTax ? $itemDataWithTax['tax_rate'] : 16.00,
                    'tax_amount' => $itemDataWithTax ? $itemDataWithTax['item_tax'] : 0,
                    'total' => $itemTotal + ($itemDataWithTax ? $itemDataWithTax['item_tax'] : 0),
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                    'added_to_stock' => false,
                    'notes' => $itemData['notes'] ?? null,
                ]);

                // Add to stock if it's a product and purchase is completed
                if ($itemData['type'] === 'product' && $validated['status'] === 'completed' && $addToStock) {
                    $product = Product::find($itemData['product_id']);
                    if ($product && $product->track_inventory) {
                        $this->addProductToStock($purchaseItem, $product);
                    }
                }
            }

            // Update supplier balance if credit purchase
            if ($isCreditPurchase) {
                $supplier->updateBalance($grandTotal);
            }

            DB::commit();

            return redirect()->route('purchases.show', $purchase)
                ->with('success', 'Purchase created successfully!')
                ->with('purchase_id', $purchase->id);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase store error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->back()
                ->with('error', 'Failed to create purchase: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Purchase $purchase)
    {
        try {
            $purchase->load([
                'supplier',
                'items.product',
                'items.product.unit',
                'creator',
                'updater',
                'approver',
                'receiver'
            ]);

            // Get payment summary
            $paymentSummary = [
                'total_amount' => $purchase->grand_total,
                'paid_amount' => $purchase->paid_amount,
                'balance_due' => $purchase->grand_total - $purchase->paid_amount,
                'payment_status' => $purchase->payment_status,
                'payment_method' => $purchase->payment_method,
                'payment_percentage' => $purchase->grand_total > 0 ? 
                    round(($purchase->paid_amount / $purchase->grand_total) * 100, 2) : 0,
            ];
            
            // Get credit information if applicable
            $creditInfo = null;
            if ($purchase->is_credit_purchase) {
                $creditInfo = [
                    'credit_limit' => $purchase->supplier->credit_limit ?? 0,
                    'current_balance' => $purchase->supplier->current_balance ?? 0,
                    'credit_available' => $purchase->supplier->credit_available ?? 0,
                    'credit_utilization' => $purchase->supplier->credit_utilization ?? 0,
                    'is_over_limit' => $purchase->supplier->is_over_limit ?? false,
                    'credit_terms' => $purchase->supplier->credit_terms,
                    'due_date' => $purchase->due_date,
                ];
            }
            
            // Get purchase breakdown
            $breakdown = [
                'subtotal' => $purchase->subtotal,
                'discount' => $purchase->discount_amount ?? 0,
                'tax' => $purchase->tax_amount,
                'grand_total' => $purchase->grand_total,
            ];
            
            // Get supplier statistics
            $supplierStats = [
                'total_purchases' => $purchase->supplier->purchases()->count(),
                'total_spent' => $purchase->supplier->purchases()->sum('grand_total'),
                'total_paid' => $purchase->supplier->purchases()->sum('paid_amount'),
                'outstanding' => $purchase->supplier->purchases()->sum(DB::raw('grand_total - paid_amount')),
                'credit_limit' => $purchase->supplier->credit_limit ?? 0,
                'current_balance' => $purchase->supplier->current_balance ?? 0,
                'credit_available' => $purchase->supplier->credit_available ?? 0,
                'credit_utilization' => $purchase->supplier->credit_utilization ?? 0,
            ];

            // Get timeline
            $timeline = $purchase->timeline;

            // Check inventory status
            $inventoryItems = $purchase->items()
                ->where('item_type', 'product')
                ->where('added_to_stock', false)
                ->count();

            return view('purchases.show', compact(
                'purchase',
                'paymentSummary',
                'creditInfo',
                'breakdown',
                'supplierStats',
                'timeline',
                'inventoryItems'
            ));
            
        } catch (\Exception $e) {
            Log::error('Purchase show error: ' . $e->getMessage());
            return redirect()->route('purchases.index')
                ->with('error', 'Failed to load purchase details.');
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Purchase $purchase)
    {
        try {
            // Check if purchase can be edited
            if ($purchase->status !== 'pending') {
                return redirect()->route('purchases.show', $purchase)
                    ->with('error', 'Only pending purchases can be edited.');
            }

            $purchase->load(['items.product']);
            
            $products = Product::where('status', 'active')
                ->where('is_service', false)
                ->where('track_inventory', true)
                ->orderBy('name')
                ->get(['id', 'sku', 'name', 'cost_price', 'sale_price', 'stock', 
                       'has_expiry', 'track_batches', 'unit_id']);
            
            $services = Product::where('status', 'active')
                ->where('is_service', true)
                ->orderBy('name')
                ->get(['id', 'name', 'sale_price', 'sku']);
            
            $suppliers = Supplier::where('status', 'active')->get();
            
            return view('purchases.edit', compact('purchase', 'products', 'services', 'suppliers'));
            
        } catch (\Exception $e) {
            Log::error('Purchase edit error: ' . $e->getMessage());
            return redirect()->route('purchases.index')
                ->with('error', 'Failed to load edit form.');
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Purchase $purchase)
    {
        try {
            // Check if purchase can be edited
            if ($purchase->status !== 'pending') {
                return redirect()->route('purchases.show', $purchase)
                    ->with('error', 'Only pending purchases can be edited.');
            }

            DB::beginTransaction();

            // Convert checkbox values to boolean
            $isCreditPurchase = $request->boolean('is_credit_purchase');

            $validated = $request->validate([
                'supplier_id' => 'required|exists:suppliers,id',
                'purchase_date' => 'required|date',
                'reference_number' => 'nullable|string|max:100',
                'expected_delivery_date' => 'nullable|date|after_or_equal:purchase_date',
                'payment_status' => 'required|in:pending,partial,paid',
                'payment_method' => 'nullable|string|max:50',
                'paid_amount' => 'required|numeric|min:0',
                'status' => 'required|in:draft,pending,completed,cancelled',
                'notes' => 'nullable|string',
                'terms_conditions' => 'nullable|string',
                'due_date' => 'nullable|date|after_or_equal:purchase_date',
                
                // Purchase items
                'items' => 'required|array|min:1',
                'items.*.id' => 'nullable|exists:purchase_items,id',
                'items.*.type' => 'required|in:product,service',
                'items.*.product_id' => 'required_if:items.*.type,product|exists:products,id',
                'items.*.description' => 'nullable|string|max:255',
                'items.*.quantity' => 'required|numeric|min:0.001',
                'items.*.unit_price' => 'required|numeric|min:0',
                'items.*.batch_number' => 'nullable|string|max:100',
                'items.*.expiry_date' => 'nullable|date|after:today',
                'items.*.notes' => 'nullable|string',
            ]);

            // Get supplier
            $supplier = Supplier::find($validated['supplier_id']);
            
            // Check credit limit if this is a credit purchase
            if ($isCreditPurchase && $supplier) {
                // Calculate total amount
                $totalAmount = 0;
                foreach ($validated['items'] as $item) {
                    $totalAmount += $item['quantity'] * $item['unit_price'];
                }
                
                // Check if supplier can make this purchase within credit limit
                if (!$supplier->canMakePurchase($totalAmount)) {
                    return redirect()->back()
                        ->with('error', 'Purchase amount exceeds available credit limit. Available: KES ' . 
                               number_format($supplier->credit_available, 2))
                        ->withInput();
                }
            }

            // Calculate totals with tax per item
            $subtotal = 0;
            $totalTax = 0;
            $itemsWithTax = [];

            foreach ($validated['items'] as $itemData) {
                if ($itemData['type'] === 'product') {
                    $product = Product::find($itemData['product_id']);
                    $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                    $subtotal += $itemTotal;
                    
                    // Calculate tax for this item
                    $taxRate = $product->tax_rate ?? 16.00;
                    $itemTax = $this->calculateTaxAmount($itemTotal, $taxRate);
                    $totalTax += $itemTax;
                    
                    $itemsWithTax[] = [
                        'product' => $product,
                        'tax_rate' => $taxRate,
                        'item_tax' => $itemTax,
                        'item_total' => $itemTotal,
                    ];
                } else {
                    // For services, use default tax rate
                    $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                    $subtotal += $itemTotal;
                    
                    $defaultTax = Tax::where('is_default', true)->first();
                    $taxRate = $defaultTax ? $defaultTax->rate : 16.00;
                    $itemTax = $itemTotal * ($taxRate / 100);
                    $totalTax += $itemTax;
                    
                    $itemsWithTax[] = [
                        'product' => null,
                        'tax_rate' => $taxRate,
                        'item_tax' => $itemTax,
                        'item_total' => $itemTotal,
                    ];
                }
            }

            $grandTotal = $subtotal + $totalTax;

            // Update purchase
            $purchase->update([
                'supplier_id' => $validated['supplier_id'],
                'purchase_date' => $validated['purchase_date'],
                'reference_number' => $validated['reference_number'] ?? null,
                'expected_delivery_date' => $validated['expected_delivery_date'] ?? null,
                'subtotal' => $subtotal,
                'tax_amount' => $totalTax,
                'grand_total' => $grandTotal,
                'paid_amount' => $validated['paid_amount'],
                'payment_status' => $validated['payment_status'],
                'payment_method' => $validated['payment_method'] ?? null,
                'status' => $validated['status'],
                'notes' => $validated['notes'] ?? null,
                'terms_conditions' => $validated['terms_conditions'] ?? null,
                'updated_by' => auth()->id(),
                'is_credit_purchase' => $isCreditPurchase,
                'due_date' => $validated['due_date'] ?? null,
            ]);

            // Update or create items
            $existingItemIds = [];
            foreach ($validated['items'] as $index => $itemData) {
                $itemTotal = $itemData['quantity'] * $itemData['unit_price'];
                $itemDataWithTax = $itemsWithTax[$index] ?? null;
                
                // Get description
                if ($itemData['type'] === 'product') {
                    $product = Product::find($itemData['product_id']);
                    $description = $product ? $product->name : 'Product #' . $itemData['product_id'];
                } else {
                    $description = $itemData['description'] ?? 'Service';
                }
                
                if (!empty($itemData['id'])) {
                    // Update existing item
                    $item = PurchaseItem::find($itemData['id']);
                    if ($item) {
                        $item->update([
                            'product_id' => $itemData['type'] === 'product' ? $itemData['product_id'] : null,
                            'item_type' => $itemData['type'],
                            'description' => $description,
                            'quantity' => $itemData['quantity'],
                            'unit_price' => $itemData['unit_price'],
                            'tax_percent' => $itemDataWithTax ? $itemDataWithTax['tax_rate'] : 16.00,
                            'tax_amount' => $itemDataWithTax ? $itemDataWithTax['item_tax'] : 0,
                            'total' => $itemTotal + ($itemDataWithTax ? $itemDataWithTax['item_tax'] : 0),
                            'batch_number' => $itemData['batch_number'] ?? null,
                            'expiry_date' => $itemData['expiry_date'] ?? null,
                            'notes' => $itemData['notes'] ?? null,
                        ]);
                        $existingItemIds[] = $itemData['id'];
                    }
                } else {
                    // Create new item
                    $item = PurchaseItem::create([
                        'purchase_id' => $purchase->id,
                        'product_id' => $itemData['type'] === 'product' ? $itemData['product_id'] : null,
                        'item_type' => $itemData['type'],
                        'description' => $description,
                        'quantity' => $itemData['quantity'],
                        'unit_price' => $itemData['unit_price'],
                        'tax_percent' => $itemDataWithTax ? $itemDataWithTax['tax_rate'] : 16.00,
                        'tax_amount' => $itemDataWithTax ? $itemDataWithTax['item_tax'] : 0,
                        'total' => $itemTotal + ($itemDataWithTax ? $itemDataWithTax['item_tax'] : 0),
                        'batch_number' => $itemData['batch_number'] ?? null,
                        'expiry_date' => $itemData['expiry_date'] ?? null,
                        'added_to_stock' => false,
                        'notes' => $itemData['notes'] ?? null,
                    ]);
                    $existingItemIds[] = $item->id;
                }
            }

            // Delete removed items
            PurchaseItem::where('purchase_id', $purchase->id)
                ->whereNotIn('id', $existingItemIds)
                ->delete();

            // Update supplier balance if credit purchase
            if ($isCreditPurchase && $supplier) {
                // Remove old balance and add new
                $oldBalance = $purchase->getOriginal('grand_total') ?? 0;
                $supplier->reduceBalance($oldBalance);
                $supplier->updateBalance($grandTotal);
            }

            DB::commit();

            return redirect()->route('purchases.show', $purchase)
                ->with('success', 'Purchase updated successfully!');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase update error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->back()
                ->with('error', 'Failed to update purchase: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            // Check if purchase can be deleted
            if ($purchase->status === 'completed' && $purchase->items()->where('added_to_stock', true)->exists()) {
                return redirect()->route('purchases.show', $purchase)
                    ->with('error', 'Cannot delete completed purchase with items added to stock.');
            }

            // Remove from supplier balance if credit purchase
            if ($purchase->is_credit_purchase && $purchase->supplier) {
                $purchase->supplier->reduceBalance($purchase->grand_total);
            }

            // Delete purchase items
            $purchase->items()->delete();
            
            // Delete the purchase
            $purchase->delete();

            DB::commit();

            return redirect()->route('purchases.index')
                ->with('success', 'Purchase deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase destroy error: ' . $e->getMessage());
            return redirect()->route('purchases.index')
                ->with('error', 'Failed to delete purchase. Please try again.');
        }
    }

    /**
     * Approve purchase.
     */
    public function approve(Request $request, Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            if ($purchase->status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only pending purchases can be approved.'
                ], 400);
            }

            // Check credit limit for credit purchases
            if ($purchase->is_credit_purchase && !$purchase->canApprove()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Purchase amount exceeds available credit limit.'
                ], 400);
            }

            // Update purchase status to approved
            $purchase->update([
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => auth()->id(),
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Purchase approved successfully.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase approve error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve purchase: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark purchase as received.
     */
    public function receive(Request $request, Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            if ($purchase->status !== 'approved') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only approved purchases can be marked as received.'
                ], 400);
            }

            // Update purchase status to received
            $purchase->update([
                'status' => 'received',
                'received_at' => now(),
                'received_by' => auth()->id(),
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Purchase marked as received.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase receive error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark purchase as received: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark purchase as completed and add items to stock.
     */
    public function complete(Request $request, Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            if ($purchase->status !== 'received' && $purchase->status !== 'approved') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only approved or received purchases can be completed.'
                ], 400);
            }

            // Add all inventory items to stock
            $addedCount = 0;
            $failedCount = 0;
            
            foreach ($purchase->items as $item) {
                if ($item->item_type === 'product' && $item->product && $item->product->track_inventory) {
                    if (!$item->added_to_stock) {
                        if ($this->addProductToStock($item, $item->product)) {
                            $addedCount++;
                        } else {
                            $failedCount++;
                        }
                    }
                }
            }

            // Update purchase status
            $purchase->update([
                'status' => 'completed',
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            $message = "Purchase marked as completed.";
            if ($addedCount > 0) {
                $message .= " {$addedCount} item(s) added to stock.";
            }
            if ($failedCount > 0) {
                $message .= " {$failedCount} item(s) failed to add to stock.";
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'added_count' => $addedCount,
                'failed_count' => $failedCount,
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase complete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to complete purchase: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Deliver services.
     */
    public function deliver(Request $request, Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            if ($purchase->status !== 'completed') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only completed purchases can have services delivered.'
                ], 400);
            }

            // Update purchase status to delivered
            $purchase->update([
                'status' => 'delivered',
                'delivered_at' => now(),
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Services marked as delivered.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase deliver error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark services as delivered: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check if purchase can be added to inventory.
     */
    public function checkInventoryAdd(Purchase $purchase)
    {
        try {
            $inventoryItems = $purchase->items()
                ->where('item_type', 'product')
                ->where('added_to_stock', false)
                ->with('product')
                ->get();
            
            $serviceItems = $purchase->items()
                ->where('item_type', 'service')
                ->count();

            $canAddToInventory = $purchase->status === 'received' || $purchase->status === 'approved';

            return response()->json([
                'success' => true,
                'can_add_to_inventory' => $canAddToInventory,
                'inventory_items_count' => $inventoryItems->count(),
                'service_items_count' => $serviceItems,
                'items' => $inventoryItems->map(function($item) {
                    return [
                        'id' => $item->id,
                        'product_name' => $item->product ? $item->product->name : 'N/A',
                        'quantity' => $item->quantity,
                        'unit_price' => $item->unit_price,
                        'batch_number' => $item->batch_number,
                        'expiry_date' => $item->expiry_date,
                        'current_stock' => $item->product ? $item->product->stock : 0,
                    ];
                })
            ]);
            
        } catch (\Exception $e) {
            Log::error('Check inventory add error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to check inventory: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add product to stock.
     */
    private function addProductToStock(PurchaseItem $purchaseItem, Product $product)
    {
        if ($purchaseItem->added_to_stock) {
            return true; // Already added to stock
        }

        if (!$product->track_inventory) {
            return false; // Don't add to stock if inventory tracking is disabled
        }

        try {
            // Calculate new stock
            $newQuantity = $product->stock + $purchaseItem->quantity;
            
            if ($product->track_batches || $purchaseItem->batch_number || $purchaseItem->expiry_date) {
                // Check if batch already exists
                $batch = ProductBatch::where('product_id', $product->id)
                    ->where('batch_number', $purchaseItem->batch_number)
                    ->where('expiry_date', $purchaseItem->expiry_date)
                    ->first();
                
                if ($batch) {
                    // Update existing batch
                    $oldQuantity = $batch->current_quantity;
                    $batch->update([
                        'current_quantity' => $batch->current_quantity + $purchaseItem->quantity,
                        'updated_at' => now(),
                    ]);
                    
                    // Log batch movement
                    BatchStockMovement::create([
                        'batch_id' => $batch->id,
                        'product_id' => $product->id,
                        'movement_type' => 'in',
                        'quantity' => $purchaseItem->quantity,
                        'quantity_before' => $oldQuantity,
                        'reason' => 'purchase',
                        'reference_number' => $purchaseItem->purchase->invoice_no,
                        'notes' => 'Added from purchase',
                        'user_id' => auth()->id(),
                    ]);
                } else {
                    // Create new batch
                    $batch = ProductBatch::create([
                        'product_id' => $product->id,
                        'supplier_id' => $purchaseItem->purchase->supplier_id,
                        'purchase_id' => $purchaseItem->purchase_id,
                        'purchase_item_id' => $purchaseItem->id,
                        'batch_number' => $purchaseItem->batch_number ?? $this->generateBatchNumber(),
                        'expiry_date' => $purchaseItem->expiry_date,
                        'initial_quantity' => $purchaseItem->quantity,
                        'current_quantity' => $purchaseItem->quantity,
                        'batch_cost_price' => $purchaseItem->unit_price,
                        'batch_sale_price' => $product->sale_price,
                        'created_by' => auth()->id(),
                    ]);

                    // Log batch movement
                    BatchStockMovement::create([
                        'batch_id' => $batch->id,
                        'product_id' => $product->id,
                        'movement_type' => 'in',
                        'quantity' => $purchaseItem->quantity,
                        'quantity_before' => 0,
                        'reason' => 'purchase',
                        'reference_number' => $purchaseItem->purchase->invoice_no,
                        'notes' => 'Added from purchase',
                        'user_id' => auth()->id(),
                    ]);
                }
            } else {
                // Update general stock
                $oldStock = $product->stock;
                $product->increment('stock', $purchaseItem->quantity);

                // Log stock movement
                StockMovement::create([
                    'product_id' => $product->id,
                    'type' => 'in',
                    'qty_change' => $purchaseItem->quantity,
                    'qty_before' => $oldStock,
                    'reason' => 'purchase',
                    'notes' => 'Added from purchase ' . $purchaseItem->purchase->invoice_no,
                    'user_id' => auth()->id(),
                ]);
            }

            // Update product stock directly
            $product->update([
                'stock' => $newQuantity,
                'stock_status' => $newQuantity > 0 ? 'in_stock' : 'out_of_stock',
            ]);

            // Mark purchase item as added to stock
            $purchaseItem->update(['added_to_stock' => true]);

            return true;

        } catch (\Exception $e) {
            Log::error('Add product to stock error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Record payment for purchase.
     */
    public function recordPayment(Request $request, Purchase $purchase)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0.01|max:' . ($purchase->grand_total - $purchase->paid_amount),
                'method' => 'required|string|max:50',
                'notes' => 'nullable|string',
            ]);

            DB::beginTransaction();

            // Update paid amount
            $oldPaidAmount = $purchase->paid_amount;
            $newPaidAmount = $oldPaidAmount + $validated['amount'];
            
            $purchase->update([
                'paid_amount' => $newPaidAmount,
                'payment_method' => $validated['method'],
                'payment_notes' => $validated['notes'] ?? null,
                'updated_by' => auth()->id(),
            ]);

            // Update supplier balance if credit purchase
            if ($purchase->is_credit_purchase && $purchase->supplier) {
                $purchase->supplier->reduceBalance($validated['amount']);
            }

            // Update payment status
            $purchase->updatePaymentStatus();

            DB::commit();

            return redirect()->route('purchases.show', $purchase)
                ->with('success', 'Payment of KES ' . number_format($validated['amount'], 2) . ' recorded successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Record payment error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to record payment: ' . $e->getMessage());
        }
    }

    /**
     * Cancel purchase.
     */
    public function cancel(Request $request, Purchase $purchase)
    {
        try {
            DB::beginTransaction();

            if ($purchase->status === 'cancelled') {
                return redirect()->route('purchases.show', $purchase)
                    ->with('error', 'Purchase is already cancelled.');
            }

            // Remove from supplier balance if credit purchase
            if ($purchase->is_credit_purchase && $purchase->supplier) {
                $purchase->supplier->reduceBalance($purchase->grand_total);
            }

            // Update purchase status
            $purchase->update([
                'status' => 'cancelled',
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('purchases.show', $purchase)
                ->with('success', 'Purchase cancelled successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase cancel error: ' . $e->getMessage());
            return redirect()->route('purchases.show', $purchase)
                ->with('error', 'Failed to cancel purchase. Please try again.');
        }
    }

    /**
     * Show payment form for purchase.
     */
    public function payment(Purchase $purchase)
    {
        try {
            // Check if purchase can accept payments
            if ($purchase->status !== 'pending' && $purchase->status !== 'completed') {
                return redirect()->route('purchases.show', $purchase)
                    ->with('error', 'Cannot record payment for this purchase status.');
            }

            if ($purchase->paid_amount >= $purchase->grand_total) {
                return redirect()->route('purchases.show', $purchase)
                    ->with('info', 'Purchase is already fully paid.');
            }

            $purchase->load(['supplier']);
            
            return view('purchases.payment', compact('purchase'));
            
        } catch (\Exception $e) {
            Log::error('Purchase payment error: ' . $e->getMessage());
            return redirect()->route('purchases.show', $purchase)
                ->with('error', 'Failed to load payment form.');
        }
    }

    /**
     * Print purchase order.
     */
    public function printInvoice(Purchase $purchase)
    {
        try {
            $purchase->load([
                'supplier',
                'items.product',
                'items.product.unit',
                'creator',
                'approver',
                'receiver'
            ]);

            // Get company information
            $company = [
                'name' => config('app.name', 'Your Company'),
                'address' => '123 Business Street, City, Country',
                'phone' => '+123 456 7890',
                'email' => 'info@company.com',
                'website' => 'www.company.com',
                'tax_id' => 'TAX-123456',
            ];

            return view('purchases.print.purchase-order', compact('purchase', 'company'));
            
        } catch (\Exception $e) {
            Log::error('Print purchase order error: ' . $e->getMessage());
            return redirect()->route('purchases.show', $purchase)
                ->with('error', 'Failed to generate print view.');
        }
    }

    /**
     * Quick view for purchase details modal.
     */
    public function quickView(Purchase $purchase)
    {
        $purchase->load([
            'supplier',
            'items.product',
            'items.product.unit',
            'creator'
        ]);

        // Get payment summary
        $paymentSummary = [
            'total_amount' => $purchase->grand_total,
            'paid_amount' => $purchase->paid_amount,
            'balance_due' => $purchase->grand_total - $purchase->paid_amount,
            'payment_status' => $purchase->payment_status,
            'payment_method' => $purchase->payment_method,
        ];

        $html = view('purchases.partials.quick-view-content', compact('purchase', 'paymentSummary'))->render();
        
        return response()->json(['html' => $html]);
    }

    /**
     * Export purchases.
     */
    public function export(Request $request)
    {
        try {
            $query = Purchase::with(['supplier', 'items']);
            
            // Apply filters
            if ($request->has('status') && !empty($request->status)) {
                $query->where('status', $request->status);
            }
            
            if ($request->has('supplier_id') && !empty($request->supplier_id)) {
                $query->where('supplier_id', $request->supplier_id);
            }
            
            if ($request->has('start_date') && !empty($request->start_date)) {
                $query->whereDate('purchase_date', '>=', $request->start_date);
            }
            
            if ($request->has('end_date') && !empty($request->end_date)) {
                $query->whereDate('purchase_date', '<=', $request->end_date);
            }
            
            $purchases = $query->orderBy('purchase_date', 'desc')->get();
            
            $fileName = 'purchases-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($purchases) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Invoice No', 'Date', 'Supplier', 'Reference', 'Status',
                    'Subtotal', 'Tax', 'Grand Total', 'Paid Amount', 'Balance Due',
                    'Payment Status', 'Payment Method', 'Items Count', 'Created At'
                ]);
                
                // Data
                foreach ($purchases as $purchase) {
                    fputcsv($file, [
                        $purchase->invoice_no,
                        $purchase->purchase_date->format('Y-m-d'),
                        $purchase->supplier->name ?? 'N/A',
                        $purchase->reference_number ?? 'N/A',
                        ucfirst($purchase->status),
                        $purchase->subtotal,
                        $purchase->tax_amount,
                        $purchase->grand_total,
                        $purchase->paid_amount,
                        $purchase->grand_total - $purchase->paid_amount,
                        ucfirst($purchase->payment_status),
                        $purchase->payment_method ?? 'N/A',
                        $purchase->items->count(),
                        $purchase->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Purchase export error: ' . $e->getMessage());
            return redirect()->route('purchases.index')
                ->with('error', 'Failed to export purchases.');
        }
    }

    /**
     * Search purchases.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            if (empty($search)) {
                return response()->json([]);
            }
            
            $purchases = Purchase::where('invoice_no', 'like', "%{$search}%")
                ->orWhere('reference_number', 'like', "%{$search}%")
                ->orWhereHas('supplier', function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                })
                ->with(['supplier'])
                ->limit(10)
                ->get(['id', 'invoice_no', 'purchase_date', 'grand_total', 'supplier_id'])
                ->map(function($purchase) {
                    return [
                        'id' => $purchase->id,
                        'text' => "{$purchase->invoice_no} - {$purchase->supplier->name} - KES " . number_format($purchase->grand_total, 2),
                    ];
                });
            
            return response()->json(['results' => $purchases]);
            
        } catch (\Exception $e) {
            Log::error('Purchase search error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /**
     * Update purchase status.
     */
    public function updateStatus(Request $request, Purchase $purchase)
    {
        try {
            $validated = $request->validate([
                'status' => 'required|in:completed,pending,cancelled'
            ]);

            DB::beginTransaction();

            if ($validated['status'] === 'completed') {
                // Add all inventory items to stock
                foreach ($purchase->items as $item) {
                    if ($item->item_type === 'product' && $item->product && $item->product->track_inventory) {
                        if (!$item->added_to_stock) {
                            $this->addProductToStock($item, $item->product);
                        }
                    }
                }
            }

            // Update purchase status
            $purchase->update([
                'status' => $validated['status'],
                'updated_by' => auth()->id(),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Purchase status updated successfully.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase status update error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update purchase status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add items to stock without completing purchase.
     */
    public function addToStock(Request $request, Purchase $purchase)
    {
        try {
            if ($purchase->status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only pending purchases can add items to stock.'
                ], 400);
            }

            DB::beginTransaction();

            // Add all inventory items to stock
            foreach ($purchase->items as $item) {
                if ($item->item_type === 'product' && $item->product && $item->product->track_inventory) {
                    if (!$item->added_to_stock) {
                        $this->addProductToStock($item, $item->product);
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Items added to stock successfully.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Add to stock error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to add items to stock: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate batch number.
     */
    private function generateBatchNumber(): string
    {
        return 'BATCH-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 6));
    }

    /**
     * Calculate tax amount.
     */
    private function calculateTaxAmount($amount, $taxRate)
    {
        return $amount * ($taxRate / 100);
    }
}