<?php

namespace App\Http\Controllers;

use App\Models\ProductBatch;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\BatchStockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProductBatchController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = ProductBatch::with(['product', 'supplier', 'creator'])
                ->withCount('stockMovements');

            // Search functionality
            if ($request->has('search')) {
                $search = $request->input('search');
                $query->where(function($q) use ($search) {
                    $q->where('batch_number', 'like', "%{$search}%")
                      ->orWhereHas('product', function($q) use ($search) {
                          $q->where('name', 'like', "%{$search}%")
                            ->orWhere('sku', 'like', "%{$search}%");
                      });
                });
            }

            // Filter by status
            if ($request->has('status') && $request->status != '') {
                switch ($request->status) {
                    case 'active':
                        $query->where('current_quantity', '>', 0)
                              ->where(function($q) {
                                  $q->whereNull('expiry_date')
                                    ->orWhere('expiry_date', '>', now()->addDays(30));
                              });
                        break;
                    case 'expiring':
                        $query->where('current_quantity', '>', 0)
                              ->whereNotNull('expiry_date')
                              ->where('expiry_date', '>', now())
                              ->where('expiry_date', '<=', now()->addDays(30));
                        break;
                    case 'expired':
                        $query->where('current_quantity', '>', 0)
                              ->whereNotNull('expiry_date')
                              ->where('expiry_date', '<', now());
                        break;
                    case 'empty':
                        $query->where('current_quantity', '<=', 0);
                        break;
                }
            }

            // Filter by supplier
            if ($request->has('supplier_id')) {
                $query->where('supplier_id', $request->supplier_id);
            }

            // Filter by expiry date range
            if ($request->has('expiry_from')) {
                $query->whereDate('expiry_date', '>=', $request->expiry_from);
            }

            if ($request->has('expiry_to')) {
                $query->whereDate('expiry_date', '<=', $request->expiry_to);
            }

            // Filter by product
            if ($request->has('product_id')) {
                $query->where('product_id', $request->product_id);
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'expiry_date');
            $sortOrder = $request->get('sort_order', 'asc');
            
            $allowedSort = ['batch_number', 'expiry_date', 'manufacture_date', 'current_quantity', 'batch_cost_price', 'created_at'];
            if (in_array($sortBy, $allowedSort)) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->orderBy('expiry_date')->orderBy('created_at');
            }

            $batches = $query->paginate(20)->withQueryString();

            // Batch statistics
            $batchStats = [
                'total_batches' => ProductBatch::count(),
                'active_batches' => ProductBatch::where('current_quantity', '>', 0)
                    ->where(function($q) {
                        $q->whereNull('expiry_date')
                          ->orWhere('expiry_date', '>', now()->addDays(30));
                    })->count(),
                'expiring_soon' => ProductBatch::where('current_quantity', '>', 0)
                    ->whereNotNull('expiry_date')
                    ->where('expiry_date', '>', now())
                    ->where('expiry_date', '<=', now()->addDays(30))->count(),
                'expired_batches' => ProductBatch::where('current_quantity', '>', 0)
                    ->whereNotNull('expiry_date')
                    ->where('expiry_date', '<', now())->count(),
                'empty_batches' => ProductBatch::where('current_quantity', '<=', 0)->count(),
                'total_quantity' => ProductBatch::sum('current_quantity'),
                'total_value' => ProductBatch::sum(DB::raw('batch_cost_price * current_quantity')),
            ];

            $suppliers = Supplier::where('status', 'active')->get();

return view('products.product-batches.index', compact('batches', 'batchStats', 'suppliers'));
            
        } catch (\Exception $e) {
            Log::error('ProductBatch index error: ' . $e->getMessage());
            return redirect()->route('products.index')
                ->with('error', 'Failed to load batches. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        try {
            $products = Product::where('track_batches', true)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'sku', 'cost_price', 'sale_price']);
            
            $suppliers = Supplier::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name']);
            
            return view('products.product-batches.create', compact('products', 'suppliers'));
            
        } catch (\Exception $e) {
            Log::error('Batch create form error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to load batch creation form.');
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_id' => 'required|exists:products,id',
                'supplier_id' => 'nullable|exists:suppliers,id',
                'batch_number' => 'required|string|max:100|unique:product_batches,batch_number',
                'expiry_date' => 'nullable|date|after_or_equal:today',
                'manufacture_date' => 'nullable|date|before_or_equal:today',
                'initial_quantity' => 'required|integer|min:1',
                'current_quantity' => 'required|integer|min:0',
                'batch_cost_price' => 'required|numeric|min:0',
                'batch_sale_price' => 'nullable|numeric|min:0',
                'warehouse_location' => 'nullable|string|max:100',
                'notes' => 'nullable|string|max:500',
            ]);

            // Get product details
            $product = Product::findOrFail($validated['product_id']);
            
            // Set current quantity if not provided
            if (!isset($validated['current_quantity'])) {
                $validated['current_quantity'] = $validated['initial_quantity'];
            }
            
            // Set batch sale price if not provided
            if (!isset($validated['batch_sale_price'])) {
                $validated['batch_sale_price'] = $product->sale_price;
            }
            
            // Add created_by
            $validated['created_by'] = auth()->id();
            
            // Create the batch
            $batch = ProductBatch::create($validated);
            
            // Update product stock
            $product->increment('stock', $validated['initial_quantity']);
            
            // Record initial stock movement
            BatchStockMovement::create([
                'batch_id' => $batch->id,
                'product_id' => $batch->product_id,
                'movement_type' => 'in',
                'quantity' => $validated['initial_quantity'],
                'quantity_before' => 0,
                'reason' => 'initial_batch_creation',
                'reference_number' => 'BATCH-CREATE-' . now()->format('Ymd'),
                'notes' => $validated['notes'] ?? 'Initial batch creation',
                'user_id' => auth()->id(),
            ]);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($batch)
                ->log('created product batch');

            DB::commit();

            return redirect()->route('batches.show', $batch)
                ->with('success', 'Batch created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Batch store error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to create batch: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(ProductBatch $batch)
    {
        try {
            // Load relationships
            $batch->load(['product', 'supplier', 'creator', 'stockMovements.user']);
            
            // Get batch statistics
            $batchStats = [
                'total_movements' => $batch->stockMovements()->count(),
                'total_in' => $batch->stockMovements()->where('movement_type', 'in')->sum('quantity'),
                'total_out' => abs($batch->stockMovements()->where('movement_type', 'out')->sum('quantity')),
                'remaining_quantity' => $batch->current_quantity,
                'remaining_percentage' => $batch->initial_quantity > 0 
                    ? ($batch->current_quantity / $batch->initial_quantity) * 100 
                    : 0,
            ];
            
            // Get recent stock movements
            $stockMovements = $batch->stockMovements()
                ->with('user')
                ->latest()
                ->limit(20)
                ->get();
            
            return view('products.product-batches.show', compact('batch', 'batchStats', 'stockMovements'));
            
        } catch (\Exception $e) {
            Log::error('Batch show error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to load batch details.');
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ProductBatch $batch)
    {
        try {
            $products = Product::where('track_batches', true)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'sku']);
            
            $suppliers = Supplier::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name']);
            
            return view('products.product-batches.edit', compact('batch', 'products', 'suppliers'));
            
        } catch (\Exception $e) {
            Log::error('Batch edit form error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to load batch edit form.');
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ProductBatch $batch)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'product_id' => 'required|exists:products,id',
                'supplier_id' => 'nullable|exists:suppliers,id',
                'batch_number' => [
                    'required',
                    'string',
                    'max:100',
                    'unique:product_batches,batch_number,' . $batch->id,
                ],
                'expiry_date' => 'nullable|date|after_or_equal:today',
                'manufacture_date' => 'nullable|date|before_or_equal:today',
                'batch_cost_price' => 'required|numeric|min:0',
                'batch_sale_price' => 'nullable|numeric|min:0',
                'warehouse_location' => 'nullable|string|max:100',
                'notes' => 'nullable|string|max:500',
            ]);

            // Store old data for logging
            $oldData = $batch->toArray();
            
            // Update the batch
            $batch->update($validated);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($batch)
                ->withProperties([
                    'old' => $oldData,
                    'new' => $validated
                ])
                ->log('updated product batch');

            DB::commit();

            return redirect()->route('batches.show', $batch)
                ->with('success', 'Batch updated successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Batch update error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to update batch: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProductBatch $batch)
    {
        try {
            DB::beginTransaction();

            // Check if batch has stock
            if ($batch->current_quantity > 0) {
                // Update product stock
                $batch->product->decrement('stock', $batch->current_quantity);
            }

            // Log activity before deletion
            activity()
                ->causedBy(auth()->user())
                ->performedOn($batch)
                ->log('deleted product batch');

            $batch->delete();

            DB::commit();

            return redirect()->route('batches.index')
                ->with('success', 'Batch deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Batch destroy error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to delete batch. Please try again.');
        }
    }

    /**
     * Adjust stock for a batch.
     */
    public function adjustStock(Request $request, ProductBatch $batch)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'adjustment_type' => 'required|in:add,remove,set',
                'quantity' => 'required|integer|min:0',
                'reason' => 'required|string|max:255',
                'notes' => 'nullable|string|max:500',
            ]);

            $oldQuantity = $batch->current_quantity;
            $newQuantity = $oldQuantity;
            
            // Calculate new quantity based on adjustment type
            switch ($validated['adjustment_type']) {
                case 'add':
                    $newQuantity = $oldQuantity + $validated['quantity'];
                    $movementType = 'in';
                    $quantityChange = $validated['quantity'];
                    break;
                case 'remove':
                    if ($validated['quantity'] > $oldQuantity) {
                        throw new \Exception("Cannot remove more than available stock. Available: {$oldQuantity}");
                    }
                    $newQuantity = $oldQuantity - $validated['quantity'];
                    $movementType = 'out';
                    $quantityChange = -$validated['quantity'];
                    break;
                case 'set':
                    $newQuantity = $validated['quantity'];
                    $movementType = $validated['quantity'] > $oldQuantity ? 'in' : 'out';
                    $quantityChange = $validated['quantity'] - $oldQuantity;
                    break;
            }

            // Update batch quantity
            $batch->update(['current_quantity' => $newQuantity]);
            
            // Update product stock
            if ($quantityChange != 0) {
                if ($quantityChange > 0) {
                    $batch->product->increment('stock', $quantityChange);
                } else {
                    $batch->product->decrement('stock', abs($quantityChange));
                }
            }
            
            // Record stock movement
            BatchStockMovement::create([
                'batch_id' => $batch->id,
                'product_id' => $batch->product_id,
                'movement_type' => $movementType,
                'quantity' => $quantityChange,
                'quantity_before' => $oldQuantity,
                'reason' => $validated['reason'],
                'reference_number' => 'ADJUST-' . now()->format('YmdHis'),
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($batch)
                ->withProperties([
                    'old_quantity' => $oldQuantity,
                    'new_quantity' => $newQuantity,
                    'adjustment_type' => $validated['adjustment_type'],
                    'reason' => $validated['reason'],
                ])
                ->log('adjusted batch stock');

            DB::commit();

            return redirect()->route('batches.show', $batch)
                ->with('success', "Stock adjusted successfully. New quantity: {$newQuantity}");
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Adjust stock error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to adjust stock: ' . $e->getMessage());
        }
    }

    /**
     * Show expiry alerts.
     */
    public function expiryAlerts(Request $request)
    {
        try {
            $days = $request->input('days', 30);
            
            $query = ProductBatch::with(['product', 'supplier'])
                ->whereNotNull('expiry_date')
                ->where('current_quantity', '>', 0)
                ->where('expiry_date', '<=', now()->addDays($days))
                ->orderBy('expiry_date');
            
            // Filter by severity
            if ($request->has('severity')) {
                switch ($request->severity) {
                    case 'expired':
                        $query->where('expiry_date', '<', now());
                        break;
                    case 'critical':
                        $query->where('expiry_date', '>=', now())
                              ->where('expiry_date', '<=', now()->addDays(7));
                        break;
                    case 'warning':
                        $query->where('expiry_date', '>', now()->addDays(7))
                              ->where('expiry_date', '<=', now()->addDays(30));
                        break;
                }
            }
            
            $batches = $query->paginate(20)->withQueryString();
            
            $stats = [
                'expired' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '<', now())
                    ->where('current_quantity', '>', 0)
                    ->count(),
                'critical' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '>=', now())
                    ->where('expiry_date', '<=', now()->addDays(7))
                    ->where('current_quantity', '>', 0)
                    ->count(),
                'warning' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '>', now()->addDays(7))
                    ->where('expiry_date', '<=', now()->addDays(30))
                    ->where('current_quantity', '>', 0)
                    ->count(),
                'total_value' => ProductBatch::whereNotNull('expiry_date')
                    ->where('expiry_date', '<=', now()->addDays(30))
                    ->where('current_quantity', '>', 0)
                    ->sum(DB::raw('batch_cost_price * current_quantity')),
            ];
            
            return view('products.product-batches.expiry-alerts', compact('batches', 'stats', 'days'));
            
        } catch (\Exception $e) {
            Log::error('Expiry alerts error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to load expiry alerts.');
        }
    }

    /**
     * Export batches.
     */
    public function export(Request $request)
    {
        try {
            $query = ProductBatch::with(['product', 'supplier']);
            
            // Apply filters
            if ($request->has('status')) {
                switch ($request->status) {
                    case 'active':
                        $query->where('current_quantity', '>', 0)
                              ->where(function($q) {
                                  $q->whereNull('expiry_date')
                                    ->orWhere('expiry_date', '>', now()->addDays(30));
                              });
                        break;
                    case 'expiring':
                        $query->where('current_quantity', '>', 0)
                              ->whereNotNull('expiry_date')
                              ->where('expiry_date', '>', now())
                              ->where('expiry_date', '<=', now()->addDays(30));
                        break;
                    case 'expired':
                        $query->where('current_quantity', '>', 0)
                              ->whereNotNull('expiry_date')
                              ->where('expiry_date', '<', now());
                        break;
                    case 'empty':
                        $query->where('current_quantity', '<=', 0);
                        break;
                }
            }
            
            if ($request->has('product_id')) {
                $query->where('product_id', $request->product_id);
            }
            
            if ($request->has('supplier_id')) {
                $query->where('supplier_id', $request->supplier_id);
            }
            
            $batches = $query->get();
            
            $fileName = 'batches-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($batches) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Batch Number', 'Product', 'SKU', 'Supplier', 'Manufacture Date',
                    'Expiry Date', 'Initial Quantity', 'Current Quantity', 'Cost Price',
                    'Sale Price', 'Warehouse Location', 'Created By', 'Created At'
                ]);
                
                // Data
                foreach ($batches as $batch) {
                    fputcsv($file, [
                        $batch->batch_number,
                        $batch->product->name ?? '',
                        $batch->product->sku ?? '',
                        $batch->supplier->name ?? '',
                        $batch->manufacture_date?->format('Y-m-d') ?? '',
                        $batch->expiry_date?->format('Y-m-d') ?? '',
                        $batch->initial_quantity,
                        $batch->current_quantity,
                        $batch->batch_cost_price,
                        $batch->batch_sale_price,
                        $batch->warehouse_location ?? '',
                        $batch->creator->name ?? '',
                        $batch->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Batches export error: ' . $e->getMessage());
            return redirect()->route('batches.index')
                ->with('error', 'Failed to export batches.');
        }
    }
}