<?php

namespace App\Http\Controllers;

use App\Models\Printer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PrinterController extends Controller
{
    /**
     * Display a listing of printers.
     */
    public function index(Request $request)
    {
        $printers = Printer::with(['creator', 'updater', 'assignedUser'])
            ->orderBy('is_default', 'desc')
            ->orderBy('is_active', 'desc')
            ->orderBy('name')
            ->paginate(20);
            
        return view('printers.index', compact('printers'));
    }
    
    /**
     * Show the form for creating a new printer.
     */
    public function create()
    {
        return view('printers.create');
    }
    
    /**
     * Store a newly created printer.
     */
    public function store(Request $request)
    {
        $validated = $request->validate(Printer::getValidationRules());
        
        $printer = Printer::create($validated);
        
        // Test connection
        $testResult = $printer->testConnection();
        
        if ($testResult['success']) {
            return redirect()->route('printers.show', $printer)
                ->with('success', 'Printer created and tested successfully');
        } else {
            return redirect()->route('printers.show', $printer)
                ->with('warning', 'Printer created but connection test failed: ' . $testResult['error']);
        }
    }
    
    /**
     * Display the specified printer.
     */
    public function show(Printer $printer)
    {
        $printer->load(['creator', 'updater', 'assignedUser', 'jobs' => function($query) {
            $query->orderBy('created_at', 'desc')->limit(10);
        }, 'logs' => function($query) {
            $query->orderBy('created_at', 'desc')->limit(10);
        }]);
        
        $health = $printer->checkHealth();
        $queue = $printer->getQueue();
        $stats = $printer->getStatistics(now()->subDays(7), now());
        
        return view('printers.show', compact('printer', 'health', 'queue', 'stats'));
    }
    
    /**
     * Show the form for editing the specified printer.
     */
    public function edit(Printer $printer)
    {
        return view('printers.edit', compact('printer'));
    }
    
    /**
     * Update the specified printer.
     */
    public function update(Request $request, Printer $printer)
    {
        $validated = $request->validate(Printer::getValidationRules($printer->id));
        
        $printer->update($validated);
        
        // Test connection if connection details changed
        if ($request->hasAny(['ip_address', 'port', 'device_path', 'connection_type'])) {
            $testResult = $printer->testConnection();
            
            if ($testResult['success']) {
                return redirect()->route('printers.show', $printer)
                    ->with('success', 'Printer updated and tested successfully');
            } else {
                return redirect()->route('printers.show', $printer)
                    ->with('warning', 'Printer updated but connection test failed: ' . $testResult['error']);
            }
        }
        
        return redirect()->route('printers.show', $printer)
            ->with('success', 'Printer updated successfully');
    }
    
    /**
     * Remove the specified printer.
     */
    public function destroy(Printer $printer)
    {
        if ($printer->is_default) {
            return redirect()->route('printers.index')
                ->with('error', 'Cannot delete default printer');
        }
        
        $printer->delete();
        
        return redirect()->route('printers.index')
            ->with('success', 'Printer deleted successfully');
    }
    
    /**
     * Test printer connection.
     */
    public function testConnection(Printer $printer)
    {
        $result = $printer->testConnection();
        
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Printer connection test successful',
                'printer' => $printer->name,
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => $result['error'],
                'printer' => $printer->name,
            ], 500);
        }
    }
    
    /**
     * Print test page.
     */
    public function printTestPage(Printer $printer)
    {
        $result = $printer->printReceipt([
            'store' => [
                'name' => 'Test Store',
                'address' => 'Test Address',
                'phone' => '0712 345 678',
                'vat_number' => 'TEST123456',
            ],
            'sale' => [
                'invoice_no' => 'TEST-' . date('YmdHis'),
                'date' => now()->format('Y-m-d H:i:s'),
                'cashier' => 'System',
            ],
            'customer' => null,
            'items' => [
                ['name' => 'Test Item 1', 'quantity' => 1, 'unit_price' => 100, 'total' => 100],
                ['name' => 'Test Item 2', 'quantity' => 2, 'unit_price' => 50, 'total' => 100],
            ],
            'totals' => [
                'subtotal' => 200,
                'vat' => 32,
                'discount' => 0,
                'total' => 232,
                'paid' => 250,
                'change' => 18,
            ],
            'payment' => [
                'method' => 'cash',
                'status' => 'paid',
            ],
            'footer' => 'This is a test receipt',
        ]);
        
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Test page printed successfully',
                'job_id' => $result['job_id'],
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => $result['error'],
            ], 500);
        }
    }
    
    /**
     * Check printer health.
     */
    public function checkHealth(Printer $printer)
    {
        $health = $printer->checkHealth();
        
        return response()->json($health);
    }
    
    /**
     * Perform maintenance.
     */
    public function performMaintenance(Printer $printer, Request $request)
    {
        $notes = $request->input('notes');
        
        $result = $printer->performMaintenance($notes);
        
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => $result['message'],
                'next_maintenance' => $result['next_maintenance'],
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => $result['error'],
            ], 500);
        }
    }
    
    /**
     * Change paper roll.
     */
    public function changePaper(Printer $printer)
    {
        $result = $printer->changePaper();
        
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => $result['message'],
                'paper_last_changed' => $result['paper_last_changed'],
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => $result['error'],
            ], 500);
        }
    }
    
    /**
     * Clear printer queue.
     */
    public function clearQueue(Printer $printer)
    {
        $result = $printer->clearQueue();
        
        return response()->json($result);
    }
    
    /**
     * Get printer statistics.
     */
    public function getStatistics(Printer $printer, Request $request)
    {
        $startDate = $request->input('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        $stats = $printer->getStatistics($startDate, $endDate);
        
        return response()->json($stats);
    }
    
    /**
     * Get printer queue.
     */
    public function getQueue(Printer $printer)
    {
        $queue = $printer->getQueue();
        
        return response()->json($queue);
    }
    
    /**
     * Reset printer statistics.
     */
    public function resetStatistics(Printer $printer)
    {
        $printer->resetStatistics();
        
        return response()->json([
            'success' => true,
            'message' => 'Printer statistics reset successfully',
        ]);
    }
    
    /**
     * Export printer configuration.
     */
    public function exportConfiguration(Printer $printer)
    {
        $config = $printer->exportConfiguration();
        
        return response()->json($config);
    }
    
    /**
     * Import printer configuration.
     */
    public function importConfiguration(Request $request)
    {
        $config = $request->validate([
            'config' => 'required|array',
        ])['config'];
        
        $printer = Printer::importConfiguration($config);
        
        // Validate configuration
        $validation = $printer->validateConfiguration();
        
        if (!$validation['valid']) {
            return response()->json([
                'success' => false,
                'errors' => $validation['errors'],
            ], 422);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Configuration imported successfully',
            'printer' => $printer->toArray(),
        ]);
    }
    
    /**
     * Get available printers for POS.
     */
    public function getAvailablePrinters()
    {
        $printers = Printer::active()
            ->online()
            ->orderBy('is_default', 'desc')
            ->orderBy('name')
            ->get(['id', 'name', 'alias', 'location', 'is_default', 'paper_size']);
        
        return response()->json($printers);
    }
    
    /**
     * Set default printer.
     */
    public function setDefault(Printer $printer)
    {
        Printer::where('is_default', true)->update(['is_default' => false]);
        
        $printer->update(['is_default' => true]);
        
        return response()->json([
            'success' => true,
            'message' => 'Default printer set successfully',
            'printer' => $printer->name,
        ]);
    }
}