<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Customer;
use App\Models\Category;
use App\Models\Tax;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\ProductBatch;
use App\Models\Payment;
use App\Models\HoldSale;
use App\Models\Printer;
use App\Models\Report;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Mike42\Escpos\PrintConnectors\NetworkPrintConnector;
use Mike42\Escpos\PrintConnectors\FilePrintConnector;
use Mike42\Escpos\Printer as ThermalPrinter;
use Mike42\Escpos\EscposImage;

class PosController extends Controller
{
    private $storeSettings;
    private $currentPrinter;
    
    public function __construct()
    {
        $this->storeSettings = $this->getStoreSettings();
        $this->currentPrinter = Printer::where('is_default', true)->first();
    }
    
    /**
     * Display POS interface
     */
    public function index()
    {
        try {
            // Get active products with relationships
            $products = Product::with(['category', 'tax'])
                ->select([
                    'id', 'sku', 'barcode', 'name', 'sale_price', 'cost_price',
                    'discount_price', 'discount_percent', 'stock', 'has_vat',
                    'has_expiry', 'track_batches', 'category_id', 'tax_id'
                ])
                ->where('status', 'active')
                ->where('track_inventory', true)
                ->orderBy('name')
                ->limit(100)
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'sku' => $product->sku,
                        'barcode' => $product->barcode,
                        'name' => $product->name,
                        'sale_price' => (float) $product->sale_price,
                        'cost_price' => (float) $product->cost_price,
                        'final_price' => (float) $product->final_price,
                        'stock' => $product->stock,
                        'available_stock' => $product->available_stock,
                        'has_vat' => (bool) $product->has_vat,
                        'tax_rate' => (float) $product->tax_rate,
                        'category_id' => $product->category_id,
                        'is_low_stock' => $product->is_low_stock,
                    ];
                });
            
            // Get active customers
            $customers = Customer::active()
                ->orderBy('name')
                ->limit(50)
                ->get(['id', 'name', 'phone', 'email', 'credit_limit', 'credit_balance', 'customer_code']);
            
            // Get categories for filtering
            $categories = Category::active()
                ->orderBy('name')
                ->get(['id', 'name']);
            
            // Get default tax
            $defaultTax = Tax::default()->first() ?? Tax::first();
            $defaultTaxRate = $defaultTax ? $defaultTax->rate : 16.00;
            
            // Get today's stats
            $todayStats = $this->getTodayStats();
            
            // Get printers
            $printers = Printer::active()->get();
            
            return view('pos.index', array_merge(compact(
                'products', 
                'customers', 
                'categories', 
                'defaultTaxRate',
                'todayStats',
                'printers'
            ), [
                'storeSettings' => $this->storeSettings
            ]));
            
        } catch (\Exception $e) {
            Log::error('POS Index Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load POS: ' . $e->getMessage());
        }
    }
    
    /**
     * Search products by barcode, SKU, or name
     */
    public function searchProducts(Request $request)
    {
        try {
            $search = $request->input('q', '');
            $categoryId = $request->input('category_id');
            $withStock = $request->input('with_stock', true);
            
            $query = Product::with(['tax', 'category', 'activeBatches'])
                ->where('status', 'active');
            
            if ($withStock) {
                $query->where('track_inventory', true);
            }
            
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%")
                      ->orWhere('barcode', 'like', "%{$search}%");
                });
            }
            
            if ($categoryId) {
                $query->where('category_id', $categoryId);
            }
            
            $products = $query->orderBy('name')
                ->limit(30)
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'sku' => $product->sku,
                        'barcode' => $product->barcode,
                        'name' => $product->name,
                        'sale_price' => (float) $product->sale_price,
                        'cost_price' => (float) $product->cost_price,
                        'final_price' => (float) $product->final_price,
                        'stock' => $product->stock,
                        'available_stock' => $product->available_stock,
                        'has_vat' => (bool) $product->has_vat,
                        'tax_rate' => (float) $product->tax_rate,
                        'tax_id' => $product->tax_id,
                        'has_expiry' => (bool) $product->has_expiry,
                        'track_batches' => (bool) $product->track_batches,
                        'batches' => $product->track_batches ? $product->activeBatches->map(function($batch) {
                            return [
                                'id' => $batch->id,
                                'batch_number' => $batch->batch_number,
                                'expiry_date' => $batch->expiry_date ? $batch->expiry_date->format('Y-m-d') : null,
                                'current_quantity' => $batch->current_quantity,
                                'batch_sale_price' => (float) $batch->batch_sale_price,
                            ];
                        }) : [],
                        'is_low_stock' => $product->is_low_stock,
                        'category' => $product->category ? $product->category->name : null,
                    ];
                });
            
            return response()->json([
                'success' => true,
                'products' => $products
            ]);
            
        } catch (\Exception $e) {
            Log::error('Product Search Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Search failed'], 500);
        }
    }
    
    /**
     * Process barcode scan
     */
    public function scanBarcode(Request $request)
    {
        try {
            $barcode = trim($request->input('barcode'));
            
            if (empty($barcode)) {
                return response()->json(['success' => false, 'error' => 'No barcode provided']);
            }
            
            // Search by barcode, SKU, or product code
            $product = Product::with(['tax', 'activeBatches'])
                ->where(function($query) use ($barcode) {
                    $query->where('barcode', $barcode)
                          ->orWhere('sku', $barcode)
                          ->orWhere('ref_number', $barcode);
                })
                ->where('status', 'active')
                ->first();
            
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'error' => 'Product not found',
                    'barcode' => $barcode
                ]);
            }
            
            // Check stock
            if ($product->track_inventory && $product->available_stock <= 0) {
                return response()->json([
                    'success' => false,
                    'error' => 'Out of stock',
                    'product' => [
                        'id' => $product->id,
                        'name' => $product->name,
                        'stock' => $product->stock
                    ]
                ]);
            }
            
            return response()->json([
                'success' => true,
                'product' => [
                    'id' => $product->id,
                    'sku' => $product->sku,
                    'barcode' => $product->barcode,
                    'name' => $product->name,
                    'sale_price' => (float) $product->sale_price,
                    'final_price' => (float) $product->final_price,
                    'stock' => $product->stock,
                    'available_stock' => $product->available_stock,
                    'has_vat' => (bool) $product->has_vat,
                    'tax_rate' => (float) $product->tax_rate,
                    'tax_id' => $product->tax_id,
                    'has_expiry' => (bool) $product->has_expiry,
                    'track_batches' => (bool) $product->track_batches,
                    'batches' => $product->track_batches ? $product->activeBatches->map(function($batch) {
                        return [
                            'id' => $batch->id,
                            'batch_number' => $batch->batch_number,
                            'expiry_date' => $batch->expiry_date ? $batch->expiry_date->format('Y-m-d') : null,
                            'current_quantity' => $batch->current_quantity,
                            'batch_sale_price' => (float) $batch->batch_sale_price,
                        ];
                    }) : [],
                    'is_low_stock' => $product->is_low_stock,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Barcode Scan Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Scan failed'], 500);
        }
    }
    
    /**
     * Process sale with comprehensive VAT handling and cash management
     */
    public function processSale(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validated = $this->validateSaleRequest($request);
            
            // Generate invoice number
            $invoiceNo = $this->generateInvoiceNumber();
            
            // Calculate totals with VAT options
            $totals = $this->calculateSaleTotals($validated);
            
            // For cash payments, validate cash received
            if ($validated['payment_method'] === 'cash') {
                $cashReceived = $validated['cash_received'] ?? $totals['grand_total'];
                $change = $cashReceived - $totals['grand_total'];
                
                if ($change < 0) {
                    throw new \Exception("Insufficient cash. Received: KES {$cashReceived}, Required: KES {$totals['grand_total']}");
                }
                
                // Add cash details to validated data
                $validated['cash_received'] = $cashReceived;
                $validated['change'] = $change;
            }
            
            // Check customer credit if applicable
            if ($validated['payment_method'] === 'credit') {
                $creditCheck = $this->checkCustomerCredit($validated['customer_id'] ?? null, $totals['grand_total']);
                if (!$creditCheck['allowed']) {
                    throw new \Exception($creditCheck['reason']);
                }
            }
            
            // Check payment amount for non-credit methods
            if ($validated['payment_method'] !== 'credit') {
                $amountPaid = $validated['amount_paid'] ?? $totals['grand_total'];
                if ($amountPaid < $totals['grand_total']) {
                    throw new \Exception("Insufficient payment. Required: {$totals['grand_total']}, Paid: {$amountPaid}");
                }
            }
            
            // Create sale record
            $sale = $this->createSaleRecord($validated, $invoiceNo, $totals);
            
            // Create sale items and update stock
            $this->processSaleItems($validated['items'], $sale, $invoiceNo);
            
            // Handle payment
            $paymentResult = $this->processPayment($validated, $sale, $totals['grand_total']);
            
            // Update customer statistics
            if ($validated['customer_id']) {
                $this->updateCustomerStats($validated['customer_id'], $totals['grand_total']);
            }
            
            // Log sale activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($sale)
                ->withProperties([
                    'invoice_no' => $invoiceNo,
                    'total' => $totals['grand_total'],
                    'payment_method' => $validated['payment_method'],
                    'cash_received' => $validated['cash_received'] ?? null,
                    'change' => $validated['change'] ?? null,
                ])
                ->log('completed sale');
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'sale' => [
                    'id' => $sale->id,
                    'invoice_no' => $sale->invoice_no,
                    'grand_total' => $sale->grand_total,
                    'cash_received' => $validated['cash_received'] ?? null,
                    'change' => $validated['change'] ?? 0,
                    'customer_name' => $sale->customer ? $sale->customer->name : 'Walk-in',
                    'payment_method' => $validated['payment_method'],
                    'vat_included' => $validated['include_vat'] ?? true,
                    'vat_amount' => $totals['vat_amount'],
                    'receipt_data' => $this->prepareReceiptData($sale, $validated, $paymentResult),
                ],
                'message' => 'Sale processed successfully'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Sale Processing Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Validate sale request with cash validation
     */
    private function validateSaleRequest(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.001',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.batch_id' => 'nullable|exists:product_batches,id',
            'items.*.vat_included' => 'nullable|boolean',
            'discount_amount' => 'nullable|numeric|min:0',
            'discount_percent' => 'nullable|numeric|min:0|max:100',
            'include_vat' => 'required|boolean',
            'payment_method' => 'required|in:cash,card,mpesa,bank_transfer,credit,multiple',
            'amount_paid' => 'required|numeric|min:0',
            'cash_received' => 'nullable|required_if:payment_method,cash|numeric|min:0',
            'change' => 'nullable|numeric|min:0',
            'cash_amount' => 'nullable|numeric|min:0',
            'mpesa_amount' => 'nullable|numeric|min:0',
            'card_amount' => 'nullable|numeric|min:0',
            'mpesa_phone' => 'nullable|string|max:20',
            'mpesa_transaction_id' => 'nullable|string|max:100',
            'card_transaction_id' => 'nullable|string|max:100',
            'bank_reference' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:1000',
            'hold_sale_id' => 'nullable|exists:hold_sales,id',
            'print_receipt' => 'nullable|boolean',
            'receipt_type' => 'nullable|in:thermal,normal,both',
        ]);
        
        // Additional validation for cash payments
        if ($validated['payment_method'] === 'cash') {
            if (empty($validated['cash_received'])) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'cash_received' => ['Cash received amount is required for cash payments']
                ]);
            }
            
            // Calculate total to validate cash received
            $subtotal = 0;
            foreach ($validated['items'] as $item) {
                $itemTotal = $item['quantity'] * $item['unit_price'];
                $subtotal += $itemTotal;
            }
            
            $vatAmount = $subtotal * 0.16; // 16% VAT
            $total = $subtotal + $vatAmount;
            
            if ($validated['cash_received'] < $total) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'cash_received' => ["Insufficient cash. Required: KES {$total}, Received: KES {$validated['cash_received']}"]
                ]);
            }
            
            // Calculate change
            $validated['change'] = $validated['cash_received'] - $total;
        }
        
        return $validated;
    }
    
    /**
     * Calculate sale totals with VAT handling
     */
    private function calculateSaleTotals(array $data): array
    {
        $subtotal = 0;
        $vatableAmount = 0;
        $vatAmount = 0;
        
        foreach ($data['items'] as $item) {
            $itemTotal = $item['quantity'] * $item['unit_price'];
            $subtotal += $itemTotal;
            
            // Check if item has VAT
            $product = Product::find($item['product_id']);
            if ($product && $product->has_vat && ($data['include_vat'] || $item['vat_included'])) {
                $vatableAmount += $itemTotal;
            }
        }
        
        // Calculate VAT
        $defaultTax = Tax::default()->first();
        $vatRate = $defaultTax ? $defaultTax->rate : 16.00;
        
        if ($data['include_vat']) {
            // VAT included in prices
            $vatAmount = $vatableAmount * ($vatRate / (100 + $vatRate));
        } else {
            // VAT excluded, add to total
            $vatAmount = $vatableAmount * ($vatRate / 100);
        }
        
        // Apply discount
        $discountAmount = 0;
        if (!empty($data['discount_amount'])) {
            $discountAmount = min($data['discount_amount'], $subtotal);
        } elseif (!empty($data['discount_percent'])) {
            $discountAmount = $subtotal * ($data['discount_percent'] / 100);
        }
        
        $grandTotal = $subtotal + $vatAmount - $discountAmount;
        
        return [
            'subtotal' => $subtotal,
            'vatable_amount' => $vatableAmount,
            'vat_amount' => $vatAmount,
            'discount_amount' => $discountAmount,
            'grand_total' => $grandTotal,
            'vat_rate' => $vatRate,
        ];
    }
    
    /**
     * Check customer credit
     */
    private function checkCustomerCredit(?int $customerId, float $amount): array
    {
        if (!$customerId) {
            return ['allowed' => false, 'reason' => 'Customer required for credit sales'];
        }
        
        $customer = Customer::find($customerId);
        if (!$customer) {
            return ['allowed' => false, 'reason' => 'Customer not found'];
        }
        
        if ($customer->credit_limit <= 0) {
            return ['allowed' => false, 'reason' => 'Customer has no credit limit'];
        }
        
        $availableCredit = $customer->available_credit;
        if ($amount > $availableCredit) {
            return [
                'allowed' => false,
                'reason' => "Credit limit exceeded. Available: {$availableCredit}, Required: {$amount}"
            ];
        }
        
        return ['allowed' => true, 'available' => $availableCredit];
    }
    
    /**
     * Generate invoice number
     */
    private function generateInvoiceNumber(): string
    {
        $prefix = $this->storeSettings['invoice_prefix'] ?? 'INV';
        $year = date('Y');
        $month = date('m');
        
        $lastInvoice = Sale::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->orderBy('id', 'desc')
            ->first();
        
        $sequence = $lastInvoice ? (int) substr($lastInvoice->invoice_no, -5) + 1 : 1;
        
        return sprintf('%s-%s%s-%05d', $prefix, $year, $month, $sequence);
    }
    
    /**
     * Create sale record
     */
        private function createSaleRecord(array $data, string $invoiceNo, array $totals): Sale
        {
            $saleData = [
                'customer_id' => $data['customer_id'] ?? null,
                'invoice_no' => $invoiceNo,
                'sale_date' => now(),
                'subtotal' => $totals['subtotal'],
                'vat_amount' => $totals['vat_amount'],
                'discount_amount' => $totals['discount_amount'],
                'grand_total' => $totals['grand_total'],
                'vat_included' => $data['include_vat'],
                'vat_rate' => $totals['vat_rate'],
                'status' => 'completed',
                'payment_status' => $data['payment_method'] === 'credit' ? 'pending' : 'paid',
                'payment_method' => $data['payment_method'],
                'notes' => $data['notes'] ?? null,
                'created_by' => auth()->id(),
                'is_etims_invoice' => config('services.etims.enabled', false),
                'etims_sent_at' => config('services.etims.enabled', false) ? now() : null,
                'cash_received' => $data['cash_received'] ?? null,
                'change' => $data['change'] ?? null,
            ];
            
            // Delete hold sale if exists
            if (!empty($data['hold_sale_id'])) {
                HoldSale::where('id', $data['hold_sale_id'])->delete();
            }
            
            return Sale::create($saleData);
        }
        
        /**
         * Process sale items
         */
    private function processSaleItems(array $items, Sale $sale, string $invoiceNo): void
    {
        foreach ($items as $item) {
            $product = Product::findOrFail($item['product_id']);
            $batchId = $item['batch_id'] ?? null;
            
            // Check stock availability
            if ($product->track_inventory && $product->available_stock < $item['quantity']) {
                throw new \Exception("Insufficient stock for {$product->name}. Available: {$product->available_stock}");
            }
            
            // Calculate item tax
            $itemTotal = $item['quantity'] * $item['unit_price'];
            $taxAmount = 0;
            
            if ($product->has_vat) {
                $taxRate = $product->tax_rate;
                if ($sale->vat_included) {
                    $taxAmount = $itemTotal * ($taxRate / (100 + $taxRate));
                } else {
                    $taxAmount = $itemTotal * ($taxRate / 100);
                }
            }
            
            // Create sale item - USE 'qty' NOT 'quantity'
            $saleItem = SaleItem::create([
                'sale_id' => $sale->id,
                'product_id' => $product->id,
                'qty' => $item['quantity'], // CHANGED HERE
                'unit_price' => $item['unit_price'],
                'tax_percent' => $product->tax_rate,
                'tax_amount' => $taxAmount,
                'total' => $itemTotal,
                'batch_id' => $batchId,
                'batch_number' => $batchId ? ProductBatch::find($batchId)->batch_number : null,
            ]);
            
            // Update product stock
            $product->removeStock($item['quantity'], [
                'reason' => 'sale',
                'reference' => $invoiceNo,
                'batch_id' => $batchId,
            ]);
            
            // Update batch if applicable
            if ($batchId) {
                $batch = ProductBatch::find($batchId);
                if ($batch) {
                    $batch->decrement('current_quantity', $item['quantity']);
                }
            }
        }
    }
    
    /**
     * Process payment with cash handling
     */
    private function processPayment(array $data, Sale $sale, float $grandTotal): array
    {
        $result = [
            'total_paid' => $data['amount_paid'],
            'change' => 0,
        ];
        
        if ($data['payment_method'] === 'multiple') {
            $totalPaid = ($data['cash_amount'] ?? 0) + 
                        ($data['mpesa_amount'] ?? 0) + 
                        ($data['card_amount'] ?? 0);
            
            if ($totalPaid < $grandTotal) {
                throw new \Exception("Insufficient payment in multi-method");
            }
            
            $result['change'] = $totalPaid - $grandTotal;
            $result['total_paid'] = $totalPaid;
            
            // Record multiple payments
            if ($data['cash_amount'] > 0) {
                Payment::create([
                    'sale_id' => $sale->id,
                    'customer_id' => $data['customer_id'] ?? null,
                    'amount' => $data['cash_amount'],
                    'payment_method' => 'cash',
                    'reference' => $sale->invoice_no . '-CASH',
                    'status' => 'completed',
                    'payment_date' => now(),
                    'received_by' => auth()->id(),
                    'notes' => 'Part of split payment',
                ]);
            }
            
            if ($data['mpesa_amount'] > 0) {
                Payment::create([
                    'sale_id' => $sale->id,
                    'customer_id' => $data['customer_id'] ?? null,
                    'amount' => $data['mpesa_amount'],
                    'payment_method' => 'mpesa',
                    'reference' => $data['mpesa_transaction_id'] ?? $sale->invoice_no . '-MPESA',
                    'transaction_id' => $data['mpesa_transaction_id'] ?? null,
                    'phone' => $data['mpesa_phone'] ?? null,
                    'status' => 'completed',
                    'payment_date' => now(),
                    'received_by' => auth()->id(),
                    'notes' => 'Part of split payment',
                ]);
            }
            
            if ($data['card_amount'] > 0) {
                Payment::create([
                    'sale_id' => $sale->id,
                    'customer_id' => $data['customer_id'] ?? null,
                    'amount' => $data['card_amount'],
                    'payment_method' => 'card',
                    'reference' => $data['card_transaction_id'] ?? $sale->invoice_no . '-CARD',
                    'transaction_id' => $data['card_transaction_id'] ?? null,
                    'status' => 'completed',
                    'payment_date' => now(),
                    'received_by' => auth()->id(),
                    'notes' => 'Part of split payment',
                ]);
            }
            
        } elseif ($data['payment_method'] === 'cash') {
            // Single cash payment
            $cashReceived = $data['cash_received'] ?? $grandTotal;
            $result['change'] = $cashReceived - $grandTotal;
            $result['total_paid'] = $cashReceived;
            
            Payment::create([
                'sale_id' => $sale->id,
                'customer_id' => $data['customer_id'] ?? null,
                'amount' => $grandTotal,
                'payment_method' => 'cash',
                'reference' => $sale->invoice_no,
                'cash_received' => $cashReceived,
                'change' => $result['change'],
                'status' => 'completed',
                'payment_date' => now(),
                'received_by' => auth()->id(),
                'notes' => $data['notes'] ?? null,
            ]);
        } elseif ($data['payment_method'] === 'credit') {
            // Update customer credit balance
            if ($data['customer_id']) {
                $customer = Customer::find($data['customer_id']);
                if ($customer) {
                    $customer->addCredit($grandTotal);
                }
            }
            
        } else {
            // Single payment method (non-cash)
            if ($data['amount_paid'] > $grandTotal) {
                $result['change'] = $data['amount_paid'] - $grandTotal;
            }
            
            Payment::create([
                'sale_id' => $sale->id,
                'customer_id' => $data['customer_id'] ?? null,
                'amount' => $data['amount_paid'],
                'payment_method' => $data['payment_method'],
                'reference' => $this->getPaymentReference($data, $sale),
                'transaction_id' => $data['mpesa_transaction_id'] ?? $data['card_transaction_id'] ?? null,
                'phone' => $data['mpesa_phone'] ?? null,
                'bank_reference' => $data['bank_reference'] ?? null,
                'status' => 'completed',
                'payment_date' => now(),
                'received_by' => auth()->id(),
            ]);
        }
        
        return $result;
    }
    
    /**
     * Get payment reference
     */
    private function getPaymentReference(array $data, Sale $sale): string
    {
        switch ($data['payment_method']) {
            case 'mpesa':
                return $data['mpesa_transaction_id'] ?? $sale->invoice_no . '-MPESA';
            case 'card':
                return $data['card_transaction_id'] ?? $sale->invoice_no . '-CARD';
            case 'bank_transfer':
                return $data['bank_reference'] ?? $sale->invoice_no . '-BANK';
            default:
                return $sale->invoice_no;
        }
    }
    
    /**
     * Update customer statistics
     */
    private function updateCustomerStats(?int $customerId, float $amount): void
    {
        if ($customerId) {
            $customer = Customer::find($customerId);
            if ($customer) {
                $customer->updateAfterSale($amount);
            }
        }
    }
    
    /**
     * Prepare receipt data
     */
    private function prepareReceiptData(Sale $sale, array $data, array $payment): array
    {
        $receipt = [
            'store' => [
                'name' => $this->storeSettings['store_name'] ?? 'POS Store',
                'address' => $this->storeSettings['store_address'] ?? '',
                'phone' => $this->storeSettings['store_phone'] ?? '',
                'vat_number' => $this->storeSettings['vat_number'] ?? '',
            ],
            'sale' => [
                'invoice_no' => $sale->invoice_no,
                'date' => $sale->sale_date->format('Y-m-d H:i:s'),
                'cashier' => auth()->user()->name ?? 'System',
            ],
            'customer' => $sale->customer ? [
                'name' => $sale->customer->name,
                'phone' => $sale->customer->phone,
                'vat_number' => $sale->customer->vat_number,
            ] : null,
            'items' => $sale->items->map(function($item) {
                return [
                    'name' => $item->product->name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'total' => $item->total,
                    'vat' => $item->tax_amount > 0 ? 'Yes' : 'No',
                ];
            }),
            'totals' => [
                'subtotal' => $sale->subtotal,
                'vat' => $sale->vat_amount,
                'discount' => $sale->discount_amount,
                'total' => $sale->grand_total,
                'paid' => $payment['total_paid'] ?? $data['amount_paid'],
                'change' => $payment['change'] ?? $data['change'] ?? 0,
                'cash_received' => $data['cash_received'] ?? null,
            ],
            'payment' => [
                'method' => $sale->payment_method,
                'status' => $sale->payment_status,
            ],
            'footer' => $this->storeSettings['receipt_footer'] ?? 'Thank you for your business!',
        ];
        
        return $receipt;
    }
    
    /**
     * Print receipt
     */
    public function printReceipt(Request $request, $saleId = null)
    {
        try {
            if ($saleId) {
                $sale = Sale::with(['customer', 'items.product'])->findOrFail($saleId);
                $receiptData = $this->prepareReceiptData($sale, [], ['total_paid' => $sale->grand_total, 'change' => 0]);
            } else {
                $receiptData = $request->validate([
                    'store' => 'required|array',
                    'sale' => 'required|array',
                    'items' => 'required|array',
                    'totals' => 'required|array',
                    'payment' => 'required|array',
                ]);
            }
            
            $printType = $request->input('type', 'thermal');
            
            if ($printType === 'thermal' && $this->currentPrinter) {
                return $this->printThermalReceipt($receiptData);
            } else {
                return $this->printNormalReceipt($receiptData);
            }
            
        } catch (\Exception $e) {
            Log::error('Print Receipt Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Print failed'], 500);
        }
    }
    
    /**
     * Print thermal receipt
     */
    private function printThermalReceipt(array $receiptData)
    {
        try {
            $printer = $this->currentPrinter;
            
            if ($printer->connection_type === 'network') {
                $connector = new NetworkPrintConnector($printer->ip_address, $printer->port);
            } else {
                // For USB/Serial printers
                $connector = new FilePrintConnector($printer->device_path);
            }
            
            $thermalPrinter = new ThermalPrinter($connector);
            
            // Store information
            $thermalPrinter->setJustification(ThermalPrinter::JUSTIFY_CENTER);
            $thermalPrinter->selectPrintMode(ThermalPrinter::MODE_DOUBLE_HEIGHT | ThermalPrinter::MODE_DOUBLE_WIDTH);
            $thermalPrinter->text($receiptData['store']['name'] . "\n");
            $thermalPrinter->selectPrintMode();
            
            if (!empty($receiptData['store']['address'])) {
                $thermalPrinter->text($receiptData['store']['address'] . "\n");
            }
            if (!empty($receiptData['store']['phone'])) {
                $thermalPrinter->text("Tel: " . $receiptData['store']['phone'] . "\n");
            }
            if (!empty($receiptData['store']['vat_number'])) {
                $thermalPrinter->text("VAT: " . $receiptData['store']['vat_number'] . "\n");
            }
            
            $thermalPrinter->text("--------------------------------\n");
            
            // Sale information
            $thermalPrinter->setJustification(ThermalPrinter::JUSTIFY_LEFT);
            $thermalPrinter->text("Invoice: " . $receiptData['sale']['invoice_no'] . "\n");
            $thermalPrinter->text("Date: " . $receiptData['sale']['date'] . "\n");
            $thermalPrinter->text("Cashier: " . $receiptData['sale']['cashier'] . "\n");
            
            if ($receiptData['customer']) {
                $thermalPrinter->text("Customer: " . $receiptData['customer']['name'] . "\n");
            }
            
            $thermalPrinter->text("--------------------------------\n");
            
            // Items
            $thermalPrinter->text(str_pad("Item", 20) . str_pad("Qty", 6) . str_pad("Price", 8) . str_pad("Total", 10) . "\n");
            $thermalPrinter->text("--------------------------------\n");
            
            foreach ($receiptData['items'] as $item) {
                $name = substr($item['name'], 0, 20);
                $qty = str_pad($item['quantity'], 5);
                $price = str_pad(number_format($item['unit_price'], 2), 7);
                $total = str_pad(number_format($item['total'], 2), 9);
                
                $thermalPrinter->text(str_pad($name, 20) . $qty . $price . $total . "\n");
            }
            
            $thermalPrinter->text("--------------------------------\n");
            
            // Totals
            $thermalPrinter->setEmphasis(true);
            $thermalPrinter->text(str_pad("Subtotal:", 30) . str_pad(number_format($receiptData['totals']['subtotal'], 2), 10, ' ', STR_PAD_LEFT) . "\n");
            
            if ($receiptData['totals']['vat'] > 0) {
                $thermalPrinter->text(str_pad("VAT:", 30) . str_pad(number_format($receiptData['totals']['vat'], 2), 10, ' ', STR_PAD_LEFT) . "\n");
            }
            
            if ($receiptData['totals']['discount'] > 0) {
                $thermalPrinter->text(str_pad("Discount:", 30) . str_pad(number_format($receiptData['totals']['discount'], 2), 10, ' ', STR_PAD_LEFT) . "\n");
            }
            
            $thermalPrinter->text(str_pad("TOTAL:", 30) . str_pad(number_format($receiptData['totals']['total'], 2), 10, ' ', STR_PAD_LEFT) . "\n");
            $thermalPrinter->setEmphasis(false);
            
            $thermalPrinter->text("--------------------------------\n");
            
            // Payment
            $thermalPrinter->text("Payment: " . strtoupper($receiptData['payment']['method']) . "\n");
            
            if ($receiptData['totals']['cash_received'] > 0) {
                $thermalPrinter->text("Cash Received: " . number_format($receiptData['totals']['cash_received'], 2) . "\n");
            }
            
            $thermalPrinter->text("Paid: " . number_format($receiptData['totals']['paid'], 2) . "\n");
            
            if ($receiptData['totals']['change'] > 0) {
                $thermalPrinter->text("Change: " . number_format($receiptData['totals']['change'], 2) . "\n");
            }
            
            $thermalPrinter->text("Status: " . strtoupper($receiptData['payment']['status']) . "\n");
            
            $thermalPrinter->text("--------------------------------\n");
            $thermalPrinter->setJustification(ThermalPrinter::JUSTIFY_CENTER);
            $thermalPrinter->text($receiptData['footer'] . "\n");
            $thermalPrinter->text("\n\n");
            
            // Cut paper
            $thermalPrinter->cut();
            
            $thermalPrinter->close();
            
            return response()->json(['success' => true, 'message' => 'Receipt printed']);
            
        } catch (\Exception $e) {
            Log::error('Thermal Print Error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Print normal receipt (PDF/HTML)
     */
    private function printNormalReceipt(array $receiptData)
    {
        $pdf = Pdf::loadView('pos.receipt-pdf', $receiptData);
        
        return $pdf->download('receipt-' . $receiptData['sale']['invoice_no'] . '.pdf');
    }
    
/**
 * Get today's statistics
 */
public function getTodayStats()
{
    try {
        $today = now()->format('Y-m-d');
        
        // FIXED: Use 'qty' instead of 'quantity'
        $stats = Sale::whereDate('sale_date', $today)
            ->where('status', 'completed')
            ->select([
                DB::raw('COUNT(*) as total_sales'),
                DB::raw('SUM(grand_total) as total_revenue'),
                DB::raw('SUM(vat_amount) as total_vat'),
                DB::raw('SUM(discount_amount) as total_discount'),
                DB::raw('SUM(cash_received) as total_cash_received'),
                DB::raw('SUM(customer_change) as total_change'),
                DB::raw('AVG(grand_total) as average_sale'),
            ])->first();
        
        // FIXED: Get payment method breakdown
        $paymentBreakdown = Payment::whereDate('paid_at', $today)
            ->join('payment_methods', 'payments.payment_method_id', '=', 'payment_methods.id')
            ->select('payment_methods.name as payment_method', DB::raw('SUM(payments.amount) as total'))
            ->groupBy('payment_methods.name')
            ->get();
        
        // FIXED: Use 'qty' column
        $topProducts = SaleItem::whereHas('sale', function($query) use ($today) {
                $query->whereDate('sale_date', $today)
                      ->where('status', 'completed');
            })
            ->with('product')
            ->select('product_id', DB::raw('SUM(qty) as total_quantity')) // CHANGED HERE
            ->groupBy('product_id')
            ->orderByDesc('total_quantity')
            ->limit(5)
            ->get();
        
        return [
            'sales' => [
                'count' => $stats->total_sales ?? 0,
                'revenue' => $stats->total_revenue ?? 0,
                'vat' => $stats->total_vat ?? 0,
                'discount' => $stats->total_discount ?? 0,
                'cash_received' => $stats->total_cash_received ?? 0,
                'change' => $stats->total_change ?? 0,
                'average' => $stats->average_sale ?? 0,
            ],
            'payments' => $paymentBreakdown,
            'top_products' => $topProducts,
        ];
        
    } catch (\Exception $e) {
        Log::error('Today Stats Error: ' . $e->getMessage());
        return [];
    }
}
    
    /**
     * Generate reports
     */
    public function generateReport(Request $request)
    {
        try {
            $type = $request->input('type', 'daily');
            $startDate = $request->input('start_date', now()->format('Y-m-d'));
            $endDate = $request->input('end_date', now()->format('Y-m-d'));
            
            switch ($type) {
                case 'daily':
                    return $this->generateDailyReport($startDate);
                case 'weekly':
                    return $this->generateWeeklyReport($startDate);
                case 'monthly':
                    return $this->generateMonthlyReport($startDate);
                case 'yearly':
                    return $this->generateYearlyReport($startDate);
                case 'custom':
                    return $this->generateCustomReport($startDate, $endDate);
                default:
                    throw new \Exception('Invalid report type');
            }
            
        } catch (\Exception $e) {
            Log::error('Report Generation Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => $e->getMessage()], 500);
        }
    }
    
    /**
     * Generate daily report
     */
    private function generateDailyReport(string $date)
    {
        $startDate = Carbon::parse($date)->startOfDay();
        $endDate = Carbon::parse($date)->endOfDay();
        
        $sales = Sale::with(['customer', 'items.product', 'payments'])
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->orderBy('sale_date')
            ->get();
        
        $summary = $this->calculateReportSummary($sales);
        
        // Hourly breakdown
        $hourlyData = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $hourStart = $startDate->copy()->addHours($hour);
            $hourEnd = $hourStart->copy()->addHour();
            
            $hourSales = $sales->filter(function($sale) use ($hourStart, $hourEnd) {
                return $sale->sale_date >= $hourStart && $sale->sale_date < $hourEnd;
            });
            
            $hourlyData[] = [
                'hour' => sprintf('%02d:00', $hour),
                'sales_count' => $hourSales->count(),
                'revenue' => $hourSales->sum('grand_total'),
                'cash_received' => $hourSales->sum('cash_received'),
                'change' => $hourSales->sum('change'),
                'average' => $hourSales->count() > 0 ? $hourSales->sum('grand_total') / $hourSales->count() : 0,
            ];
        }
        
        return [
            'type' => 'daily',
            'date' => $date,
            'summary' => $summary,
            'hourly_data' => $hourlyData,
            'sales' => $sales->take(100)->map(function($sale) {
                return [
                    'invoice_no' => $sale->invoice_no,
                    'time' => $sale->sale_date->format('H:i:s'),
                    'customer' => $sale->customer ? $sale->customer->name : 'Walk-in',
                    'total' => $sale->grand_total,
                    'cash_received' => $sale->cash_received,
                    'change' => $sale->change,
                    'payment_method' => $sale->payment_method,
                    'items_count' => $sale->items->count(),
                ];
            }),
        ];
    }
    
    /**
     * Generate weekly report
     */
    private function generateWeeklyReport(string $date)
    {
        $startDate = Carbon::parse($date)->startOfWeek();
        $endDate = Carbon::parse($date)->endOfWeek();
        
        $sales = Sale::with(['customer'])
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->get();
        
        $summary = $this->calculateReportSummary($sales);
        
        // Daily breakdown
        $dailyData = [];
        $currentDate = $startDate->copy();
        
        while ($currentDate <= $endDate) {
            $daySales = $sales->filter(function($sale) use ($currentDate) {
                return $sale->sale_date->format('Y-m-d') === $currentDate->format('Y-m-d');
            });
            
            $dailyData[] = [
                'date' => $currentDate->format('Y-m-d'),
                'day' => $currentDate->format('D'),
                'sales_count' => $daySales->count(),
                'revenue' => $daySales->sum('grand_total'),
                'vat' => $daySales->sum('vat_amount'),
                'cash_received' => $daySales->sum('cash_received'),
                'average' => $daySales->count() > 0 ? $daySales->sum('grand_total') / $daySales->count() : 0,
            ];
            
            $currentDate->addDay();
        }
        
        // Top products for the week
        $topProducts = SaleItem::whereHas('sale', function($query) use ($startDate, $endDate) {
                $query->whereBetween('sale_date', [$startDate, $endDate])
                      ->where('status', 'completed');
            })
            ->with('product')
            ->select('product_id', DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total) as total_revenue'))
            ->groupBy('product_id')
            ->orderByDesc('total_revenue')
            ->limit(10)
            ->get()
            ->map(function($item) {
                return [
                    'product' => $item->product->name ?? 'Unknown',
                    'quantity' => $item->total_quantity,
                    'revenue' => $item->total_revenue,
                ];
            });
        
        return [
            'type' => 'weekly',
            'week' => $startDate->format('M d') . ' - ' . $endDate->format('M d'),
            'summary' => $summary,
            'daily_data' => $dailyData,
            'top_products' => $topProducts,
        ];
    }
    
    /**
     * Generate monthly report
     */
    private function generateMonthlyReport(string $date)
    {
        $startDate = Carbon::parse($date)->startOfMonth();
        $endDate = Carbon::parse($date)->endOfMonth();
        
        $sales = Sale::with(['customer'])
            ->whereBetween('sale_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->get();
        
        $summary = $this->calculateReportSummary($sales);
        
        // Weekly breakdown
        $weeklyData = [];
        $currentWeek = $startDate->copy();
        $weekNumber = 1;
        
        while ($currentWeek <= $endDate) {
            $weekEnd = $currentWeek->copy()->endOfWeek();
            if ($weekEnd > $endDate) {
                $weekEnd = $endDate;
            }
            
            $weekSales = $sales->filter(function($sale) use ($currentWeek, $weekEnd) {
                return $sale->sale_date >= $currentWeek && $sale->sale_date <= $weekEnd;
            });
            
            $weeklyData[] = [
                'week' => 'Week ' . $weekNumber,
                'period' => $currentWeek->format('M d') . ' - ' . $weekEnd->format('M d'),
                'sales_count' => $weekSales->count(),
                'revenue' => $weekSales->sum('grand_total'),
                'vat' => $weekSales->sum('vat_amount'),
                'cash_received' => $weekSales->sum('cash_received'),
                'average' => $weekSales->count() > 0 ? $weekSales->sum('grand_total') / $weekSales->count() : 0,
            ];
            
            $currentWeek = $weekEnd->copy()->addDay();
            $weekNumber++;
        }
        
        // Category breakdown
        $categoryData = SaleItem::whereHas('sale', function($query) use ($startDate, $endDate) {
                $query->whereBetween('sale_date', [$startDate, $endDate])
                      ->where('status', 'completed');
            })
            ->with(['product.category'])
            ->select(DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total) as total_revenue'))
            ->groupBy('product_id')
            ->get()
            ->groupBy('product.category.name')
            ->map(function($items, $category) {
                return [
                    'category' => $category ?: 'Uncategorized',
                    'quantity' => $items->sum('total_quantity'),
                    'revenue' => $items->sum('total_revenue'),
                ];
            })
            ->sortByDesc('revenue')
            ->values();
        
        // Payment method analysis
        $paymentData = Payment::whereBetween('payment_date', [$startDate, $endDate])
            ->select('payment_method', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total'))
            ->groupBy('payment_method')
            ->get();
        
        return [
            'type' => 'monthly',
            'month' => $startDate->format('F Y'),
            'summary' => $summary,
            'weekly_data' => $weeklyData,
            'category_data' => $categoryData,
            'payment_data' => $paymentData,
        ];
    }
    
    /**
     * Generate yearly report
     */
    private function generateYearlyReport(string $date)
    {
        $year = Carbon::parse($date)->year;
        $startDate = Carbon::create($year, 1, 1)->startOfDay();
        $endDate = Carbon::create($year, 12, 31)->endOfDay();
        
        $sales = Sale::whereBetween('sale_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->get();
        
        $summary = $this->calculateReportSummary($sales);
        
        // Monthly breakdown
        $monthlyData = [];
        for ($month = 1; $month <= 12; $month++) {
            $monthStart = Carbon::create($year, $month, 1)->startOfMonth();
            $monthEnd = $monthStart->copy()->endOfMonth();
            
            $monthSales = $sales->filter(function($sale) use ($monthStart, $monthEnd) {
                return $sale->sale_date >= $monthStart && $sale->sale_date <= $monthEnd;
            });
            
            $monthlyData[] = [
                'month' => $monthStart->format('M'),
                'sales_count' => $monthSales->count(),
                'revenue' => $monthSales->sum('grand_total'),
                'vat' => $monthSales->sum('vat_amount'),
                'discount' => $monthSales->sum('discount_amount'),
                'cash_received' => $monthSales->sum('cash_received'),
                'average' => $monthSales->count() > 0 ? $monthSales->sum('grand_total') / $monthSales->count() : 0,
            ];
        }
        
        // Year-over-year comparison
        $previousYear = $year - 1;
        $prevYearSales = Sale::whereYear('sale_date', $previousYear)
            ->where('status', 'completed')
            ->get();
        
        $prevYearSummary = $this->calculateReportSummary($prevYearSales);
        
        // Growth calculation
        $growth = [
            'sales_count' => $summary['sales_count'] > 0 ? 
                (($summary['sales_count'] - $prevYearSummary['sales_count']) / $prevYearSummary['sales_count']) * 100 : 0,
            'revenue' => $summary['revenue'] > 0 ? 
                (($summary['revenue'] - $prevYearSummary['revenue']) / $prevYearSummary['revenue']) * 100 : 0,
            'average_sale' => $summary['average_sale'] > 0 ? 
                (($summary['average_sale'] - $prevYearSummary['average_sale']) / $prevYearSummary['average_sale']) * 100 : 0,
        ];
        
        // Customer analysis
        $customerStats = Customer::whereYear('created_at', '<=', $year)
            ->select([
                DB::raw('COUNT(*) as total_customers'),
                DB::raw('SUM(total_orders) as total_orders'),
                DB::raw('SUM(total_spent) as total_spent'),
            ])
            ->first();
        
        $newCustomers = Customer::whereYear('created_at', $year)->count();
        $activeCustomers = Customer::where('last_purchase_date', '>=', $startDate)->count();
        
        return [
            'type' => 'yearly',
            'year' => $year,
            'summary' => $summary,
            'monthly_data' => $monthlyData,
            'comparison' => [
                'current_year' => $summary,
                'previous_year' => $prevYearSummary,
                'growth' => $growth,
            ],
            'customer_analysis' => [
                'total_customers' => $customerStats->total_customers ?? 0,
                'new_customers' => $newCustomers,
                'active_customers' => $activeCustomers,
                'average_spend' => $customerStats->total_customers > 0 ? 
                    ($customerStats->total_spent / $customerStats->total_customers) : 0,
                'repeat_rate' => $customerStats->total_customers > 0 ? 
                    ($activeCustomers / $customerStats->total_customers) * 100 : 0,
            ],
        ];
    }
    
    /**
     * Generate custom report
     */
    private function generateCustomReport(string $startDate, string $endDate)
    {
        $start = Carbon::parse($startDate)->startOfDay();
        $end = Carbon::parse($endDate)->endOfDay();
        
        $sales = Sale::with(['customer', 'items.product.category'])
            ->whereBetween('sale_date', [$start, $end])
            ->where('status', 'completed')
            ->get();
        
        $summary = $this->calculateReportSummary($sales);
        
        // Daily breakdown
        $dailyData = [];
        $currentDate = $start->copy();
        
        while ($currentDate <= $end) {
            $daySales = $sales->filter(function($sale) use ($currentDate) {
                return $sale->sale_date->format('Y-m-d') === $currentDate->format('Y-m-d');
            });
            
            $dailyData[] = [
                'date' => $currentDate->format('Y-m-d'),
                'sales_count' => $daySales->count(),
                'revenue' => $daySales->sum('grand_total'),
                'vat' => $daySales->sum('vat_amount'),
                'discount' => $daySales->sum('discount_amount'),
                'cash_received' => $daySales->sum('cash_received'),
                'change' => $daySales->sum('change'),
                'average' => $daySales->count() > 0 ? $daySales->sum('grand_total') / $daySales->count() : 0,
            ];
            
            $currentDate->addDay();
        }
        
        // Top products
        $topProducts = SaleItem::whereHas('sale', function($query) use ($start, $end) {
                $query->whereBetween('sale_date', [$start, $end])
                      ->where('status', 'completed');
            })
            ->with('product')
            ->select('product_id', DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total) as total_revenue'))
            ->groupBy('product_id')
            ->orderByDesc('total_revenue')
            ->limit(10)
            ->get()
            ->map(function($item) {
                return [
                    'product' => $item->product->name ?? 'Unknown',
                    'quantity' => $item->total_quantity,
                    'revenue' => $item->total_revenue,
                    'profit_margin' => $item->product ? 
                        (($item->total_revenue - ($item->product->cost_price * $item->total_quantity)) / $item->total_revenue) * 100 : 0,
                ];
            });
        
        // Customer analysis
        $customerData = Sale::whereBetween('sale_date', [$start, $end])
            ->where('status', 'completed')
            ->whereNotNull('customer_id')
            ->with('customer')
            ->select('customer_id', DB::raw('COUNT(*) as visit_count'), DB::raw('SUM(grand_total) as total_spent'))
            ->groupBy('customer_id')
            ->orderByDesc('total_spent')
            ->limit(10)
            ->get()
            ->map(function($sale) {
                return [
                    'customer' => $sale->customer->name ?? 'Unknown',
                    'visits' => $sale->visit_count,
                    'total_spent' => $sale->total_spent,
                    'average_spend' => $sale->visit_count > 0 ? $sale->total_spent / $sale->visit_count : 0,
                ];
            });
        
        // Hourly trends
        $hourlyTrends = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $hourSales = $sales->filter(function($sale) use ($hour) {
                return $sale->sale_date->hour === $hour;
            });
            
            $hourlyTrends[] = [
                'hour' => sprintf('%02d:00', $hour),
                'sales_count' => $hourSales->count(),
                'revenue' => $hourSales->sum('grand_total'),
                'cash_received' => $hourSales->sum('cash_received'),
                'percentage' => $summary['revenue'] > 0 ? 
                    ($hourSales->sum('grand_total') / $summary['revenue']) * 100 : 0,
            ];
        }
        
        return [
            'type' => 'custom',
            'period' => $start->format('Y-m-d') . ' to ' . $end->format('Y-m-d'),
            'summary' => $summary,
            'daily_data' => $dailyData,
            'top_products' => $topProducts,
            'customer_data' => $customerData,
            'hourly_trends' => $hourlyTrends,
            'payment_analysis' => $this->analyzePayments($start, $end),
            'category_analysis' => $this->analyzeCategories($start, $end),
        ];
    }
    
    /**
     * Calculate report summary
     */
    private function calculateReportSummary($sales): array
    {
        return [
            'sales_count' => $sales->count(),
            'revenue' => $sales->sum('grand_total'),
            'vat' => $sales->sum('vat_amount'),
            'discount' => $sales->sum('discount_amount'),
            'cash_received' => $sales->sum('cash_received'),
            'change' => $sales->sum('change'),
            'average_sale' => $sales->count() > 0 ? $sales->sum('grand_total') / $sales->count() : 0,
            'max_sale' => $sales->max('grand_total') ?? 0,
            'min_sale' => $sales->min('grand_total') ?? 0,
            'refunds' => $sales->where('status', 'refunded')->sum('grand_total'),
            'net_revenue' => $sales->sum('grand_total') - $sales->where('status', 'refunded')->sum('grand_total'),
        ];
    }
    
    /**
     * Analyze payments
     */
    private function analyzePayments(Carbon $start, Carbon $end): array
    {
        $payments = Payment::whereBetween('payment_date', [$start, $end])
            ->select([
                'payment_method',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(amount) as total'),
                DB::raw('AVG(amount) as average'),
            ])
            ->groupBy('payment_method')
            ->get();
        
        $totalAmount = $payments->sum('total');
        
        return $payments->map(function($payment) use ($totalAmount) {
            return [
                'method' => $payment->payment_method,
                'count' => $payment->count,
                'total' => $payment->total,
                'average' => $payment->average,
                'percentage' => $totalAmount > 0 ? ($payment->total / $totalAmount) * 100 : 0,
            ];
        })->toArray();
    }
    
    /**
     * Analyze categories
     */
    private function analyzeCategories(Carbon $start, Carbon $end): array
    {
        $categorySales = SaleItem::whereHas('sale', function($query) use ($start, $end) {
                $query->whereBetween('sale_date', [$start, $end])
                      ->where('status', 'completed');
            })
            ->join('products', 'sale_items.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->select([
                'categories.name as category',
                DB::raw('SUM(sale_items.quantity) as quantity'),
                DB::raw('SUM(sale_items.total) as revenue'),
                DB::raw('AVG(products.cost_price) as avg_cost'),
            ])
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('revenue')
            ->get();
        
        $totalRevenue = $categorySales->sum('revenue');
        
        return $categorySales->map(function($sale) use ($totalRevenue) {
            $cost = $sale->avg_cost * $sale->quantity;
            $profit = $sale->revenue - $cost;
            
            return [
                'category' => $sale->category,
                'quantity' => $sale->quantity,
                'revenue' => $sale->revenue,
                'cost' => $cost,
                'profit' => $profit,
                'margin' => $sale->revenue > 0 ? ($profit / $sale->revenue) * 100 : 0,
                'percentage' => $totalRevenue > 0 ? ($sale->revenue / $totalRevenue) * 100 : 0,
            ];
        })->toArray();
    }
    
    /**
     * Get store settings
     */
    private function getStoreSettings(): array
    {
        return [
            'store_name' => config('pos.store_name', 'My POS Store'),
            'store_address' => config('pos.store_address', ''),
            'store_phone' => config('pos.store_phone', ''),
            'vat_number' => config('pos.vat_number', ''),
            'invoice_prefix' => config('pos.invoice_prefix', 'INV'),
            'receipt_footer' => config('pos.receipt_footer', 'Thank you for your business!'),
            'default_vat_rate' => config('pos.default_vat_rate', 16.00),
            'currency' => config('pos.currency', 'KES'),
            'currency_symbol' => config('pos.currency_symbol', 'KSh'),
            'print_automatic' => config('pos.print_automatic', true),
            'require_customer' => config('pos.require_customer', false),
            'low_stock_threshold' => config('pos.low_stock_threshold', 5),
        ];
    }
    
    /**
     * Hold sale for later
     */
    public function holdSale(Request $request)
    {
        try {
            $validated = $request->validate([
                'items' => 'required|array|min:1',
                'customer_id' => 'nullable|exists:customers,id',
                'notes' => 'nullable|string|max:500',
            ]);
            
            // Calculate totals
            $subtotal = 0;
            $vatAmount = 0;
            
            foreach ($validated['items'] as $item) {
                $product = Product::find($item['product_id']);
                $itemTotal = $item['quantity'] * $item['unit_price'];
                $subtotal += $itemTotal;
                
                if ($product && $product->has_vat) {
                    $vatAmount += $itemTotal * ($product->tax_rate / 100);
                }
            }
            
            $grandTotal = $subtotal + $vatAmount;
            
            $holdSale = HoldSale::create([
                'customer_id' => $validated['customer_id'] ?? null,
                'items' => json_encode($validated['items']),
                'subtotal' => $subtotal,
                'vat_amount' => $vatAmount,
                'grand_total' => $grandTotal,
                'notes' => $validated['notes'] ?? null,
                'held_by' => auth()->id(),
                'expires_at' => now()->addHours(24),
            ]);
            
            return response()->json([
                'success' => true,
                'hold_sale_id' => $holdSale->id,
                'message' => 'Sale held successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Hold Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to hold sale'], 500);
        }
    }
    
    /**
     * Get held sales
     */
    public function getHeldSales()
    {
        try {
            $heldSales = HoldSale::with('customer')
                ->where('held_by', auth()->id())
                ->where('expires_at', '>', now())
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function($sale) {
                    $items = json_decode($sale->items, true);
                    return [
                        'id' => $sale->id,
                        'customer' => $sale->customer ? $sale->customer->name : 'Walk-in',
                        'item_count' => count($items),
                        'total' => $sale->grand_total,
                        'created_at' => $sale->created_at->format('H:i:s'),
                        'notes' => $sale->notes,
                    ];
                });
            
            return response()->json(['success' => true, 'sales' => $heldSales]);
            
        } catch (\Exception $e) {
            Log::error('Get Held Sales Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get held sales'], 500);
        }
    }
    
    /**
     * Load held sale
     */
    public function loadHeldSale($id)
    {
        try {
            $holdSale = HoldSale::with('customer')->findOrFail($id);
            
            $items = json_decode($holdSale->items, true);
            $productIds = array_column($items, 'product_id');
            
            $products = Product::with('tax')
                ->whereIn('id', $productIds)
                ->get()
                ->keyBy('id');
            
            $detailedItems = [];
            foreach ($items as $item) {
                $product = $products[$item['product_id']] ?? null;
                if ($product) {
                    $detailedItems[] = [
                        'product_id' => $product->id,
                        'sku' => $product->sku,
                        'name' => $product->name,
                        'quantity' => $item['quantity'],
                        'unit_price' => $item['unit_price'],
                        'has_vat' => $product->has_vat,
                        'tax_rate' => $product->tax_rate,
                        'stock' => $product->stock,
                    ];
                }
            }
            
            return response()->json([
                'success' => true,
                'sale' => [
                    'customer_id' => $holdSale->customer_id,
                    'customer' => $holdSale->customer ? [
                        'id' => $holdSale->customer->id,
                        'name' => $holdSale->customer->name,
                        'phone' => $holdSale->customer->phone,
                    ] : null,
                    'items' => $detailedItems,
                    'subtotal' => $holdSale->subtotal,
                    'vat_amount' => $holdSale->vat_amount,
                    'grand_total' => $holdSale->grand_total,
                    'notes' => $holdSale->notes,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Load Held Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to load held sale'], 500);
        }
    }
    
    /**
     * Delete held sale
     */
    public function deleteHeldSale($id)
    {
        try {
            $holdSale = HoldSale::findOrFail($id);
            
            if ($holdSale->held_by !== auth()->id()) {
                return response()->json(['success' => false, 'error' => 'Unauthorized'], 403);
            }
            
            $holdSale->delete();
            
            return response()->json(['success' => true, 'message' => 'Held sale deleted']);
            
        } catch (\Exception $e) {
            Log::error('Delete Held Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to delete held sale'], 500);
        }
    }
    
    /**
     * Get dashboard statistics
     */
/**
 * Get dashboard statistics
 */
        public function getDashboardStats()
        {
            try {
                $today = now()->format('Y-m-d');
                $weekStart = now()->startOfWeek()->format('Y-m-d');
                $monthStart = now()->startOfMonth()->format('Y-m-d');
                
                // Today's stats - FIXED COLUMNS
                $todayStats = Sale::whereDate('sale_date', $today)
                    ->where('status', 'completed')
                    ->select([
                        DB::raw('COUNT(*) as sales_count'),
                        DB::raw('SUM(grand_total) as revenue'),
                        DB::raw('SUM(vat_amount) as vat'),
                        DB::raw('SUM(cash_received) as cash_received'),
                        DB::raw('SUM(customer_change) as change'),
                    ])->first();
                
                // Week's stats
                $weekStats = Sale::whereDate('sale_date', '>=', $weekStart)
                    ->where('status', 'completed')
                    ->select([
                        DB::raw('COUNT(*) as sales_count'),
                        DB::raw('SUM(grand_total) as revenue'),
                        DB::raw('SUM(cash_received) as cash_received'),
                    ])->first();
                
                // Month's stats
                $monthStats = Sale::whereDate('sale_date', '>=', $monthStart)
                    ->where('status', 'completed')
                    ->select([
                        DB::raw('COUNT(*) as sales_count'),
                        DB::raw('SUM(grand_total) as revenue'),
                        DB::raw('SUM(cash_received) as cash_received'),
                    ])->first();
                
                // Low stock products
                $lowStockProducts = Product::where('stock', '<=', DB::raw('minimum_stock'))
                    ->where('stock', '>', 0)
                    ->count();
                
                // Out of stock products
                $outOfStockProducts = Product::where('stock', '<=', 0)
                    ->where('track_inventory', true)
                    ->count();
                
                // Recent sales
                $recentSales = Sale::with('customer')
                    ->where('status', 'completed')
                    ->orderBy('sale_date', 'desc')
                    ->limit(5)
                    ->get()
                    ->map(function($sale) {
                        return [
                            'invoice_no' => $sale->invoice_no,
                            'customer' => $sale->customer ? $sale->customer->name : 'Walk-in',
                            'total' => $sale->grand_total,
                            'cash_received' => $sale->cash_received,
                            'change' => $sale->customer_change,
                            'time' => $sale->sale_date->format('H:i'),
                        ];
                    });
                
                // Top products today
                $topProducts = SaleItem::whereHas('sale', function($query) use ($today) {
                        $query->whereDate('sale_date', $today)
                            ->where('status', 'completed');
                    })
                    ->with('product')
                    ->select('product_id', DB::raw('SUM(quantity) as total_quantity'))
                    ->groupBy('product_id')
                    ->orderByDesc('total_quantity')
                    ->limit(5)
                    ->get()
                    ->map(function($item) {
                        return [
                            'product' => $item->product->name ?? 'Unknown',
                            'quantity' => $item->total_quantity,
                        ];
                    });
                
                return response()->json([
                    'success' => true,
                    'stats' => [
                        'today' => [
                            'sales' => $todayStats->sales_count ?? 0,
                            'revenue' => $todayStats->revenue ?? 0,
                            'vat' => $todayStats->vat ?? 0,
                            'cash_received' => $todayStats->cash_received ?? 0,
                            'change' => $todayStats->change ?? 0,
                        ],
                        'week' => [
                            'sales' => $weekStats->sales_count ?? 0,
                            'revenue' => $weekStats->revenue ?? 0,
                            'cash_received' => $weekStats->cash_received ?? 0,
                        ],
                        'month' => [
                            'sales' => $monthStats->sales_count ?? 0,
                            'revenue' => $monthStats->revenue ?? 0,
                            'cash_received' => $monthStats->cash_received ?? 0,
                        ],
                        'inventory' => [
                            'low_stock' => $lowStockProducts,
                            'out_of_stock' => $outOfStockProducts,
                        ],
                    ],
                    'recent_sales' => $recentSales,
                    'top_products' => $topProducts,
                ]);
                
            } catch (\Exception $e) {
                Log::error('Dashboard Stats Error: ' . $e->getMessage());
                return response()->json(['success' => false, 'error' => 'Failed to get stats'], 500);
            }
        }
    
    /**
     * Quick customer add from POS
     */
    public function quickAddCustomer(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:20',
                'email' => 'nullable|email|max:255',
                'customer_type' => 'nullable|in:walk-in,regular,business',
            ]);
            
            // Check if phone already exists
            $existingCustomer = Customer::where('phone', $validated['phone'])->first();
            if ($existingCustomer) {
                return response()->json([
                    'success' => true,
                    'customer' => [
                        'id' => $existingCustomer->id,
                        'name' => $existingCustomer->name,
                        'phone' => $existingCustomer->phone,
                        'email' => $existingCustomer->email,
                        'customer_code' => $existingCustomer->customer_code,
                    ]
                ]);
            }
            
            $customer = Customer::create([
                'name' => $validated['name'],
                'phone' => $validated['phone'],
                'email' => $validated['email'] ?? null,
                'customer_type' => $validated['customer_type'] ?? 'walk-in',
                'status' => 'active',
                'created_by' => auth()->id(),
            ]);
            
            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'email' => $customer->email,
                    'customer_code' => $customer->customer_code,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Quick Add Customer Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to add customer'], 500);
        }
    }
    
    /**
     * Search customers for POS
     */
    public function searchCustomers(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            $customers = Customer::where('status', 'active')
                ->where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('phone', 'like', "%{$search}%")
                          ->orWhere('email', 'like', "%{$search}%")
                          ->orWhere('customer_code', 'like', "%{$search}%");
                })
                ->orderBy('name')
                ->limit(20)
                ->get(['id', 'name', 'phone', 'email', 'customer_code', 'credit_limit', 'credit_balance']);
            
            return response()->json($customers);
            
        } catch (\Exception $e) {
            Log::error('Search Customers Error: ' . $e->getMessage());
            return response()->json(['error' => 'Search failed'], 500);
        }
    }
    
    /**
     * Get customer credit info
     */
    public function getCustomerCreditInfo($customerId)
    {
        try {
            $customer = Customer::findOrFail($customerId);
            
            return response()->json([
                'success' => true,
                'credit_limit' => $customer->credit_limit,
                'credit_balance' => $customer->credit_balance,
                'available_credit' => $customer->available_credit,
                'total_spent' => $customer->total_spent,
                'total_orders' => $customer->total_orders,
            ]);
            
        } catch (\Exception $e) {
            Log::error('Customer Credit Info Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Customer not found'], 404);
        }
    }

    /**
     * API: Get held sales for modern POS
     */
    public function apiGetHeldSales(Request $request)
    {
        try {
            $query = HoldSale::with(['customer', 'user'])
                ->where('user_id', auth()->id()) // Only show current user's holds
                ->orderBy('created_at', 'desc');
            
            // Check if expired
            $query->where(function($q) {
                $q->where('expires_at', '>', now())
                  ->orWhere('status', '!=', 'expired');
            });
            
            $holds = $query->get()->map(function($hold) {
                $items = json_decode($hold->items, true);
                
                return [
                    'id' => $hold->id,
                    'reference' => $hold->reference ?? 'HOLD-' . $hold->id,
                    'customer_name' => $hold->customer ? $hold->customer->name : 'Walk-in',
                    'customer_id' => $hold->customer_id,
                    'item_count' => count($items),
                    'total' => (float) $hold->total,
                    'subtotal' => (float) $hold->subtotal,
                    'vat' => (float) $hold->vat,
                    'discount' => (float) $hold->discount,
                    'status' => $hold->expires_at < now() ? 'expired' : ($hold->status ?? 'active'),
                    'held_by_name' => $hold->user ? $hold->user->name : auth()->user()->name,
                    'created_at' => $hold->created_at->toISOString(),
                    'expires_at' => $hold->expires_at->toISOString(),
                    'notes' => $hold->notes,
                ];
            });
            
            return response()->json([
                'success' => true,
                'holds' => $holds,
                'count' => $holds->count()
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Get Held Sales Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get held sales'], 500);
        }
    }

    /**
     * API: Get held sale details
     */
    public function apiGetHoldSale($id)
    {
        try {
            $hold = HoldSale::with(['customer', 'user'])
                ->where('id', $id)
                ->where('user_id', auth()->id())
                ->firstOrFail();
            
            $items = json_decode($hold->items, true);
            
            // Get product details for items
            $detailedItems = [];
            foreach ($items as $item) {
                $product = Product::find($item['product_id'] ?? null);
                $detailedItems[] = [
                    'product_id' => $item['product_id'] ?? null,
                    'name' => $product->name ?? 'Unknown Product',
                    'quantity' => $item['quantity'] ?? 1,
                    'price' => $item['unit_price'] ?? $item['price'] ?? 0,
                    'total' => ($item['quantity'] ?? 1) * ($item['unit_price'] ?? $item['price'] ?? 0),
                ];
            }
            
            return response()->json([
                'success' => true,
                'hold' => [
                    'id' => $hold->id,
                    'reference' => $hold->reference ?? 'HOLD-' . $hold->id,
                    'customer_name' => $hold->customer ? $hold->customer->name : 'Walk-in',
                    'customer' => $hold->customer ? [
                        'id' => $hold->customer->id,
                        'name' => $hold->customer->name,
                        'phone' => $hold->customer->phone,
                        'email' => $hold->customer->email,
                    ] : null,
                    'items' => $detailedItems,
                    'total' => (float) $hold->total,
                    'subtotal' => (float) $hold->subtotal,
                    'vat' => (float) $hold->vat,
                    'discount' => (float) $hold->discount,
                    'status' => $hold->expires_at < now() ? 'expired' : ($hold->status ?? 'active'),
                    'held_by_name' => $hold->user ? $hold->user->name : auth()->user()->name,
                    'created_at' => $hold->created_at->toISOString(),
                    'expires_at' => $hold->expires_at->toISOString(),
                    'notes' => $hold->notes,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Get Hold Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Held sale not found'], 404);
        }
    }

    /**
     * API: Load held sale into POS
     */
    public function apiLoadHoldSale($id)
    {
        try {
            $hold = HoldSale::with(['customer'])
                ->where('id', $id)
                ->where('user_id', auth()->id())
                ->firstOrFail();
            
            // Check if expired
            if ($hold->expires_at < now()) {
                return response()->json([
                    'success' => false,
                    'error' => 'This hold sale has expired'
                ]);
            }
            
            $items = json_decode($hold->items, true);
            
            // Prepare items with current product data
            $loadedItems = [];
            foreach ($items as $item) {
                $product = Product::with('tax')->find($item['product_id'] ?? null);
                if ($product) {
                    $loadedItems[] = [
                        'id' => $product->id,
                        'sku' => $product->sku,
                        'barcode' => $product->barcode,
                        'name' => $product->name,
                        'price' => $item['unit_price'] ?? $item['price'] ?? $product->sale_price,
                        'quantity' => $item['quantity'] ?? 1,
                        'has_vat' => $product->has_vat,
                        'tax_rate' => $product->tax_rate,
                        'stock' => $product->stock,
                    ];
                }
            }
            
            return response()->json([
                'success' => true,
                'items' => $loadedItems,
                'customer' => $hold->customer ? [
                    'id' => $hold->customer->id,
                    'name' => $hold->customer->name,
                    'phone' => $hold->customer->phone,
                    'email' => $hold->customer->email,
                ] : null,
                'discount' => (float) $hold->discount,
                'total' => (float) $hold->total,
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Load Hold Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to load hold sale'], 500);
        }
    }

    /**
     * API: Delete held sale
     */
    public function apiDeleteHoldSale($id)
    {
        try {
            $hold = HoldSale::where('id', $id)
                ->where('user_id', auth()->id())
                ->firstOrFail();
            
            $hold->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Hold sale deleted successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Delete Hold Sale Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to delete hold sale'], 500);
        }
    }

    /**
     * API: Clear expired holds
     */
    public function apiClearExpiredHolds()
    {
        try {
            $expiredCount = HoldSale::where('user_id', auth()->id())
                ->where('expires_at', '<', now())
                ->delete();
            
            return response()->json([
                'success' => true,
                'cleared_count' => $expiredCount,
                'message' => "Cleared {$expiredCount} expired hold sales"
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Clear Expired Holds Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to clear expired holds'], 500);
        }
    }

    /**
     * API: Get held sales statistics
     */
    public function apiGetHeldStats()
    {
        try {
            $userId = auth()->id();
            $now = now();
            
            $stats = [
                'active_holds' => HoldSale::where('user_id', $userId)
                    ->where('expires_at', '>', $now)
                    ->count(),
                'total_value' => HoldSale::where('user_id', $userId)
                    ->where('expires_at', '>', $now)
                    ->sum('total'),
                'expiring_soon' => HoldSale::where('user_id', $userId)
                    ->where('expires_at', '>', $now)
                    ->where('expires_at', '<=', $now->addHour())
                    ->count(),
                'todays_holds' => HoldSale::where('user_id', $userId)
                    ->whereDate('created_at', today())
                    ->count(),
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Held Stats Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get stats'], 500);
        }
    }

    /**
     * API: Search held sales
     */
    public function apiSearchHeldSales(Request $request)
    {
        try {
            $search = $request->input('q', '');
            $status = $request->input('status', 'all');
            $date = $request->input('date', 'all');
            
            $query = HoldSale::with(['customer', 'user'])
                ->where('user_id', auth()->id())
                ->orderBy('created_at', 'desc');
            
            // Apply search filter
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->whereHas('customer', function($q2) use ($search) {
                        $q2->where('name', 'like', "%{$search}%")
                           ->orWhere('phone', 'like', "%{$search}%");
                    })
                    ->orWhere('reference', 'like', "%{$search}%")
                    ->orWhere('notes', 'like', "%{$search}%");
                });
            }
            
            // Apply status filter
            if ($status !== 'all') {
                if ($status === 'active') {
                    $query->where('expires_at', '>', now());
                } elseif ($status === 'expired') {
                    $query->where('expires_at', '<=', now());
                } elseif ($status === 'converted') {
                    $query->where('status', 'converted');
                }
            }
            
            // Apply date filter
            if ($date !== 'all') {
                if ($date === 'today') {
                    $query->whereDate('created_at', today());
                } elseif ($date === 'yesterday') {
                    $query->whereDate('created_at', today()->subDay());
                } elseif ($date === 'week') {
                    $query->whereDate('created_at', '>=', today()->subDays(7));
                }
            }
            
            $holds = $query->get()->map(function($hold) {
                $items = json_decode($hold->items, true);
                
                return [
                    'id' => $hold->id,
                    'reference' => $hold->reference ?? 'HOLD-' . $hold->id,
                    'customer_name' => $hold->customer ? $hold->customer->name : 'Walk-in',
                    'item_count' => count($items),
                    'total' => (float) $hold->total,
                    'status' => $hold->expires_at < now() ? 'expired' : ($hold->status ?? 'active'),
                    'held_by_name' => $hold->user ? $hold->user->name : auth()->user()->name,
                    'created_at' => $hold->created_at->toISOString(),
                    'expires_at' => $hold->expires_at->toISOString(),
                ];
            });
            
            return response()->json([
                'success' => true,
                'holds' => $holds,
                'count' => $holds->count()
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Search Held Sales Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Search failed'], 500);
        }
    }   
    
    /**
     * Process sale from modern POS frontend
     */
    public function processSaleFromPOS(Request $request)
    {
        DB::beginTransaction();
        
        try {
            // Validate the request
            $validated = $request->validate([
                'customer_id' => 'nullable|exists:customers,id',
                'items' => 'required|array|min:1',
                'items.*.product_id' => 'required|exists:products,id',
                'items.*.quantity' => 'required|numeric|min:0.001',
                'items.*.price' => 'required|numeric|min:0',
                'payment_method' => 'required|in:cash,mpesa,card,credit,multiple',
                'amount_paid' => 'required|numeric|min:0',
                'cash_amount' => 'nullable|numeric|min:0',
                'mpesa_amount' => 'nullable|numeric|min:0',
                'card_amount' => 'nullable|numeric|min:0',
                'mpesa_phone' => 'nullable|string|max:20',
                'mpesa_transaction_id' => 'nullable|string|max:100',
                'card_transaction_id' => 'nullable|string|max:100',
                'include_vat' => 'boolean',
                'discount_amount' => 'nullable|numeric|min:0',
                'cash_received' => 'nullable|numeric|min:0',
                'change' => 'nullable|numeric|min:0',
                'print_receipt' => 'boolean',
                'receipt_type' => 'nullable|in:thermal,normal,both',
            ]);
            
            // Calculate total from items
            $subtotal = 0;
            foreach ($validated['items'] as $item) {
                $subtotal += $item['quantity'] * $item['price'];
            }
            
            // Calculate VAT (16%)
            $vatRate = 16.00;
            $vatAmount = $validated['include_vat'] ?? true ? 
                $subtotal * ($vatRate / (100 + $vatRate)) : 
                $subtotal * ($vatRate / 100);
            
            // Apply discount
            $discount = $validated['discount_amount'] ?? 0;
            
            // Grand total
            $grandTotal = $subtotal + $vatAmount - $discount;
            
            // For cash payments, validate cash received
            if ($validated['payment_method'] === 'cash') {
                $cashReceived = $validated['cash_received'] ?? $validated['amount_paid'];
                $change = $cashReceived - $grandTotal;
                
                if ($change < 0) {
                    throw new \Exception("Insufficient cash. Received: KES {$cashReceived}, Required: KES {$grandTotal}");
                }
                
                $validated['cash_received'] = $cashReceived;
                $validated['change'] = $change;
            }
            
            // Generate invoice number
            $invoiceNo = $this->generateInvoiceNumber();
            
            // Create sale
            $sale = Sale::create([
                'customer_id' => $validated['customer_id'] ?? null,
                'invoice_no' => $invoiceNo,
                'sale_date' => now(),
                'subtotal' => $subtotal,
                'vat_amount' => $vatAmount,
                'discount_amount' => $discount,
                'grand_total' => $grandTotal,
                'vat_included' => $validated['include_vat'] ?? true,
                'vat_rate' => $vatRate,
                'status' => 'completed',
                'payment_status' => 'paid',
                'payment_method' => $validated['payment_method'],
                'cash_received' => $validated['cash_received'] ?? null,
                'change' => $validated['change'] ?? 0,
                'created_by' => auth()->id(),
            ]);
            
            // Create sale items
            foreach ($validated['items'] as $item) {
                $product = Product::find($item['product_id']);
                
                SaleItem::create([
                    'sale_id' => $sale->id,
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'sku' => $product->sku,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['price'],
                    'tax_percent' => $product->has_vat ? $vatRate : 0,
                    'tax_amount' => $product->has_vat ? $item['quantity'] * $item['price'] * ($vatRate / 100) : 0,
                    'total' => $item['quantity'] * $item['price'],
                ]);
                
                // Update stock
                if ($product->track_inventory) {
                    $product->decrement('stock', $item['quantity']);
                }
            }
            
            // Create payment record
            $payment = Payment::create([
                'sale_id' => $sale->id,
                'customer_id' => $validated['customer_id'] ?? null,
                'amount' => $validated['amount_paid'],
                'payment_method' => $validated['payment_method'],
                'reference' => $invoiceNo,
                'transaction_id' => $validated['mpesa_transaction_id'] ?? $validated['card_transaction_id'] ?? null,
                'phone' => $validated['mpesa_phone'] ?? null,
                'cash_received' => $validated['cash_received'] ?? null,
                'change' => $validated['change'] ?? 0,
                'status' => 'completed',
                'payment_date' => now(),
                'received_by' => auth()->id(),
            ]);
            
            // Update customer stats
            if ($validated['customer_id']) {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    $customer->increment('total_orders');
                    $customer->increment('total_spent', $grandTotal);
                    $customer->last_purchase_date = now();
                    $customer->save();
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Sale processed successfully',
                'sale' => [
                    'id' => $sale->id,
                    'invoice_no' => $sale->invoice_no,
                    'grand_total' => $sale->grand_total,
                    'cash_received' => $sale->cash_received,
                    'change' => $sale->change,
                ],
                'receipt_data' => $this->prepareSimpleReceipt($sale, $validated)
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('POS Sale Processing Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Prepare simple receipt for modern POS
     */
    private function prepareSimpleReceipt(Sale $sale, array $data): array
    {
        return [
            'store' => [
                'name' => config('app.name', 'POS Store'),
                'address' => config('pos.store_address', ''),
                'phone' => config('pos.store_phone', ''),
            ],
            'sale' => [
                'invoice_no' => $sale->invoice_no,
                'date' => $sale->sale_date->format('Y-m-d H:i:s'),
                'cashier' => auth()->user()->name ?? 'System',
            ],
            'items' => $sale->items->map(function($item) {
                return [
                    'name' => $item->product_name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'total' => $item->total,
                ];
            }),
            'totals' => [
                'subtotal' => $sale->subtotal,
                'vat' => $sale->vat_amount,
                'discount' => $sale->discount_amount,
                'total' => $sale->grand_total,
                'paid' => $sale->cash_received ?? $sale->grand_total,
                'change' => $sale->change ?? 0,
            ],
            'payment' => [
                'method' => $sale->payment_method,
            ],
        ];
    }
}