<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use App\Models\MpesaTransaction;
use App\Models\Sale;
use App\Models\Customer;
use App\Models\Payment;
use Carbon\Carbon;
use App\Exports\MpesaTransactionsExport;
use Maatwebsite\Excel\Facades\Excel;
use PDF;

class MpesaTransactionController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:mpesa.view', ['only' => ['transactions', 'showTransaction', 'transactionsByPhone', 'dailyReport', 'unmatchedReport', 'reconciliationReport']]);
        $this->middleware('permission:mpesa.manage', ['only' => ['matchTransaction', 'reverseTransaction', 'bulkMatchTransactions', 'exportTransactions']]);
    }
    
    /**
     * Display MPESA Transactions with Filters
     */
    public function transactions(Request $request)
    {
        try {
            $query = MpesaTransaction::with(['sale', 'customer'])
                ->orderBy('transaction_date', 'desc');
            
            // Search
            if ($search = $request->input('search')) {
                $query->where(function($q) use ($search) {
                    $q->where('transaction_id', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('reference', 'like', "%{$search}%")
                      ->orWhere('first_name', 'like', "%{$search}%")
                      ->orWhere('last_name', 'like', "%{$search}%")
                      ->orWhereHas('sale', function($q2) use ($search) {
                          $q2->where('invoice_no', 'like', "%{$search}%");
                      })
                      ->orWhereHas('customer', function($q3) use ($search) {
                          $q3->where('name', 'like', "%{$search}%")
                             ->orWhere('phone', 'like', "%{$search}%");
                      });
                });
            }
            
            // Status filter
            if ($status = $request->input('status')) {
                $query->where('status', $status);
            }
            
            // Date filter
            if ($date = $request->input('date')) {
                $query->whereDate('transaction_date', $date);
            }
            
            // Date range filter
            if ($startDate = $request->input('start_date')) {
                $query->whereDate('transaction_date', '>=', $startDate);
            }
            
            if ($endDate = $request->input('end_date')) {
                $query->whereDate('transaction_date', '<=', $endDate);
            }
            
            // Phone filter
            if ($phone = $request->input('phone')) {
                $normalizedPhone = $this->formatPhoneNumber($phone);
                $query->where('phone', $normalizedPhone);
            }
            
            // Amount range filter
            if ($minAmount = $request->input('min_amount')) {
                $query->where('amount', '>=', $minAmount);
            }
            
            if ($maxAmount = $request->input('max_amount')) {
                $query->where('amount', '<=', $maxAmount);
            }
            
            // Unmatched filter
            if ($request->boolean('unmatched')) {
                $query->whereNull('sale_id')->where('status', 'completed');
            }
            
            // Matched filter
            if ($request->boolean('matched')) {
                $query->whereNotNull('sale_id');
            }
            
            // Failed filter
            if ($request->boolean('failed')) {
                $query->where('status', 'failed');
            }
            
            $perPage = $request->input('per_page', 25);
            $transactions = $query->paginate($perPage)
                ->withQueryString();
            
            // Stats for display
            $stats = Cache::remember('mpesa_transactions_stats', 300, function () {
                return [
                    'total' => MpesaTransaction::count(),
                    'completed' => MpesaTransaction::where('status', 'completed')->count(),
                    'unmatched' => MpesaTransaction::whereNull('sale_id')->where('status', 'completed')->count(),
                    'matched' => MpesaTransaction::whereNotNull('sale_id')->count(),
                    'failed' => MpesaTransaction::where('status', 'failed')->count(),
                    'pending' => MpesaTransaction::where('status', 'pending')->count(),
                    'reversed' => MpesaTransaction::where('status', 'reversed')->count(),
                    'total_amount' => MpesaTransaction::where('status', 'completed')->sum('amount'),
                    'matched_amount' => MpesaTransaction::whereNotNull('sale_id')->sum('amount'),
                    'unmatched_amount' => MpesaTransaction::whereNull('sale_id')->where('status', 'completed')->sum('amount'),
                ];
            });
            
            // Status options for filter
            $statusOptions = [
                'completed' => 'Completed',
                'pending' => 'Pending',
                'failed' => 'Failed',
                'reversed' => 'Reversed',
                'matched' => 'Matched'
            ];
            
            return view('payments.mpesa.transactions', compact('transactions', 'stats', 'statusOptions'));
            
        } catch (\Exception $e) {
            Log::error('MPESA Transactions View Error: ' . $e->getMessage(), [
                'filters' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return view('payments.mpesa.transactions')->withErrors([
                'error' => 'Failed to load transactions. Please try again.'
            ]);
        }
    }
    
    /**
     * Show Single Transaction Details
     */
    public function showTransaction($id)
    {
        try {
            $transaction = MpesaTransaction::with([
                'sale' => function($query) {
                    $query->with(['customer', 'items.product', 'user']);
                },
                'customer',
                'reversedBy'
            ])->findOrFail($id);
            
            // Get similar transactions (same phone)
            $similarTransactions = MpesaTransaction::where('phone', $transaction->phone)
                ->where('id', '!=', $transaction->id)
                ->where('status', 'completed')
                ->orderBy('transaction_date', 'desc')
                ->limit(5)
                ->get();
            
            // Get potential matching sales (unpaid sales for this customer or similar amount)
            $potentialSales = [];
            if ($transaction->customer_id) {
                $potentialSales = Sale::where('customer_id', $transaction->customer_id)
                    ->where('payment_status', '!=', 'paid')
                    ->where('grand_total', '>=', $transaction->amount * 0.9) // Within 10%
                    ->where('grand_total', '<=', $transaction->amount * 1.1) // Within 10%
                    ->orderBy('created_at', 'desc')
                    ->limit(5)
                    ->get();
            }
            
            // Transaction timeline
            $timeline = $this->getTransactionTimeline($transaction);
            
            return view('payments.mpesa.transaction-details', compact(
                'transaction',
                'similarTransactions',
                'potentialSales',
                'timeline'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Transaction Details Error: ' . $e->getMessage(), [
                'transaction_id' => $id,
                'user_id' => auth()->id()
            ]);
            
            return redirect()->route('payments.mpesa.transactions')
                ->with('error', 'Transaction not found or could not be loaded.');
        }
    }
    
    /**
     * Get Transactions by Phone Number
     */
    public function transactionsByPhone($phone)
    {
        try {
            $normalizedPhone = $this->formatPhoneNumber($phone);
            
            $transactions = MpesaTransaction::with(['sale', 'customer'])
                ->where('phone', $normalizedPhone)
                ->where('status', 'completed')
                ->orderBy('transaction_date', 'desc')
                ->paginate(20);
            
            // Customer info (if exists)
            $customer = Customer::where('phone', $normalizedPhone)->first();
            
            // Stats
            $totalTransactions = $transactions->total();
            $totalAmount = $transactions->sum('amount');
            $matchedCount = $transactions->whereNotNull('sale_id')->count();
            $unmatchedCount = $transactions->whereNull('sale_id')->count();
            
            $stats = [
                'total_transactions' => $totalTransactions,
                'total_amount' => $totalAmount,
                'matched_count' => $matchedCount,
                'unmatched_count' => $unmatchedCount,
                'matched_amount' => $transactions->whereNotNull('sale_id')->sum('amount'),
                'unmatched_amount' => $transactions->whereNull('sale_id')->sum('amount'),
                'avg_amount' => $totalTransactions > 0 ? $totalAmount / $totalTransactions : 0,
                'first_transaction' => $transactions->last()->transaction_date ?? null,
                'last_transaction' => $transactions->first()->transaction_date ?? null,
            ];
            
            return view('payments.mpesa.phone-transactions', compact(
                'transactions',
                'customer',
                'stats',
                'phone'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Phone Transactions Error: ' . $e->getMessage(), [
                'phone' => $phone,
                'user_id' => auth()->id()
            ]);
            
            return redirect()->route('payments.mpesa.transactions')
                ->with('error', 'Failed to load transactions for phone: ' . $phone);
        }
    }
    
    /**
     * Match Transaction with Sale
     */
    public function matchTransaction(Request $request, $id)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'sale_id' => 'required|exists:sales,id',
                'match_type' => ['nullable', Rule::in(['exact', 'partial', 'overpayment'])],
                'notes' => 'nullable|string|max:500'
            ]);
            
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            
            $transaction = MpesaTransaction::findOrFail($id);
            $sale = Sale::with('customer')->findOrFail($request->sale_id);
            
            // Validation checks
            if ($transaction->status !== 'completed') {
                return back()->with('error', 'Only completed transactions can be matched.');
            }
            
            if ($transaction->sale_id) {
                return back()->with('error', 'Transaction already matched with sale #' . $transaction->sale_id);
            }
            
            if ($sale->payment_status === 'paid') {
                return back()->with('error', 'Sale #' . $sale->id . ' is already paid.');
            }
            
            // Check amount match
            $amountDiff = abs($transaction->amount - $sale->grand_total);
            $isExactMatch = $amountDiff < 0.01;
            
            if (!$isExactMatch && !$request->match_type) {
                return back()->with('warning', 'Amount mismatch. Please confirm match type.')
                    ->with('amount_diff', $amountDiff)
                    ->with('sale_amount', $sale->grand_total)
                    ->with('transaction_amount', $transaction->amount);
            }
            
            // Update transaction
            $transaction->update([
                'sale_id' => $sale->id,
                'customer_id' => $sale->customer_id,
                'status' => 'matched',
                'matched_at' => now(),
                'matched_by' => auth()->id(),
                'match_type' => $request->match_type ?? ($isExactMatch ? 'exact' : 'manual'),
                'match_notes' => $request->notes,
                'amount_difference' => $isExactMatch ? null : ($transaction->amount - $sale->grand_total)
            ]);
            
            // Update sale
            $sale->update([
                'payment_status' => 'paid',
                'payment_method' => 'mpesa',
                'payment_reference' => $transaction->transaction_id,
                'paid_at' => now(),
                'paid_by' => auth()->id()
            ]);
            
            // Create payment record
            $payment = Payment::create([
                'sale_id' => $sale->id,
                'amount' => $transaction->amount,
                'payment_method' => 'mpesa',
                'reference' => $transaction->transaction_id,
                'transaction_id' => $transaction->transaction_id,
                'phone' => $transaction->phone,
                'status' => 'completed',
                'payment_date' => $transaction->transaction_date,
                'received_by' => auth()->id(),
                'notes' => 'Matched with MPESA transaction ' . $transaction->transaction_id . 
                         ($request->notes ? ' - ' . $request->notes : ''),
                'metadata' => [
                    'mpesa_transaction_id' => $transaction->id,
                    'match_type' => $request->match_type ?? ($isExactMatch ? 'exact' : 'manual'),
                    'original_amount' => $transaction->amount,
                    'sale_amount' => $sale->grand_total,
                    'difference' => $isExactMatch ? 0 : ($transaction->amount - $sale->grand_total)
                ]
            ]);
            
            // Handle overpayment (create credit note)
            if ($transaction->amount > $sale->grand_total && $request->match_type === 'overpayment') {
                $overpayment = $transaction->amount - $sale->grand_total;
                
                // Create customer credit
                if ($sale->customer_id) {
                    $customer = Customer::find($sale->customer_id);
                    if ($customer) {
                        $customer->increment('credit_balance', $overpayment);
                        
                        // Create credit transaction
                        \App\Models\CreditTransaction::create([
                            'customer_id' => $customer->id,
                            'type' => 'credit',
                            'amount' => $overpayment,
                            'balance_after' => $customer->credit_balance,
                            'reference' => 'Overpayment from MPESA ' . $transaction->transaction_id,
                            'notes' => 'Overpayment from sale #' . $sale->id . ' matching',
                            'created_by' => auth()->id()
                        ]);
                    }
                }
            }
            
            DB::commit();
            
            // Clear cache
            Cache::forget('mpesa_dashboard_stats');
            Cache::forget('mpesa_transactions_stats');
            
            // Log activity
            activity('mpesa-match')
                ->causedBy(auth()->user())
                ->performedOn($transaction)
                ->withProperties([
                    'sale_id' => $sale->id,
                    'amount' => $transaction->amount,
                    'sale_amount' => $sale->grand_total,
                    'match_type' => $request->match_type ?? ($isExactMatch ? 'exact' : 'manual'),
                    'difference' => $isExactMatch ? 0 : ($transaction->amount - $sale->grand_total)
                ])
                ->log('MPESA transaction matched with sale');
            
            return redirect()->route('payments.mpesa.transactions.show', $id)
                ->with('success', 'Transaction successfully matched with Sale #' . $sale->id)
                ->with('payment_id', $payment->id);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('MPESA Match Transaction Error: ' . $e->getMessage(), [
                'transaction_id' => $id,
                'sale_id' => $request->sale_id,
                'user_id' => auth()->id()
            ]);
            
            return back()->with('error', 'Failed to match transaction: ' . $e->getMessage())->withInput();
        }
    }
    
    /**
     * Reverse Transaction
     */
    public function reverseTransaction(Request $request, $id)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'reason' => 'required|string|max:500',
                'notify_customer' => 'boolean',
                'refund_amount' => 'nullable|numeric|min:0'
            ]);
            
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            
            $transaction = MpesaTransaction::findOrFail($id);
            
            // Validation checks
            if ($transaction->status !== 'completed') {
                return back()->with('error', 'Only completed transactions can be reversed.');
            }
            
            if ($transaction->status === 'reversed') {
                return back()->with('error', 'Transaction is already reversed.');
            }
            
            // Mark as reversed
            $transaction->update([
                'status' => 'reversed',
                'reversed_at' => now(),
                'reversed_by' => auth()->id(),
                'reversal_reason' => $request->reason,
                'refund_amount' => $request->refund_amount ?? $transaction->amount
            ]);
            
            // If linked to sale, update sale payment status
            if ($transaction->sale_id) {
                $sale = Sale::find($transaction->sale_id);
                if ($sale) {
                    $sale->update([
                        'payment_status' => 'pending',
                        'payment_reference' => null,
                        'paid_at' => null,
                        'payment_notes' => 'Payment reversed: ' . $request->reason
                    ]);
                    
                    // Update payment record
                    Payment::where('sale_id', $sale->id)
                        ->where('transaction_id', $transaction->transaction_id)
                        ->update([
                            'status' => 'reversed',
                            'reversed_at' => now(),
                            'reversal_reason' => $request->reason
                        ]);
                    
                    // Create reversal transaction for customer credit
                    if ($sale->customer_id && $request->refund_amount) {
                        $customer = Customer::find($sale->customer_id);
                        if ($customer) {
                            $customer->increment('credit_balance', $request->refund_amount);
                            
                            \App\Models\CreditTransaction::create([
                                'customer_id' => $customer->id,
                                'type' => 'refund',
                                'amount' => $request->refund_amount,
                                'balance_after' => $customer->credit_balance,
                                'reference' => 'Reversal of MPESA ' . $transaction->transaction_id,
                                'notes' => $request->reason,
                                'created_by' => auth()->id()
                            ]);
                        }
                    }
                }
            }
            
            DB::commit();
            
            // Clear cache
            Cache::forget('mpesa_dashboard_stats');
            Cache::forget('mpesa_transactions_stats');
            
            // Log activity
            activity('mpesa-reversal')
                ->causedBy(auth()->user())
                ->performedOn($transaction)
                ->withProperties([
                    'reason' => $request->reason,
                    'refund_amount' => $request->refund_amount ?? $transaction->amount,
                    'original_amount' => $transaction->amount
                ])
                ->log('MPESA transaction reversed');
            
            // Send notification if requested
            if ($request->boolean('notify_customer') && $transaction->phone) {
                $this->sendReversalNotification($transaction, $request->reason);
            }
            
            return redirect()->route('payments.mpesa.transactions.show', $id)
                ->with('success', 'Transaction reversed successfully.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('MPESA Reverse Transaction Error: ' . $e->getMessage(), [
                'transaction_id' => $id,
                'user_id' => auth()->id()
            ]);
            
            return back()->with('error', 'Failed to reverse transaction: ' . $e->getMessage())->withInput();
        }
    }
    
    /**
     * Bulk Match Transactions
     */
    public function bulkMatchTransactions(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'matches' => 'required|array|min:1',
                'matches.*.transaction_id' => 'required|exists:mpesa_transactions,id',
                'matches.*.sale_id' => 'required|exists:sales,id',
                'matches.*.match_type' => ['nullable', Rule::in(['exact', 'partial', 'overpayment'])],
                'matches.*.notes' => 'nullable|string|max:500'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            $results = [];
            $successCount = 0;
            $failedCount = 0;
            
            foreach ($request->matches as $match) {
                try {
                    $transaction = MpesaTransaction::find($match['transaction_id']);
                    $sale = Sale::find($match['sale_id']);
                    
                    // Validate
                    if (!$transaction || !$sale) {
                        $results[] = [
                            'transaction_id' => $match['transaction_id'],
                            'sale_id' => $match['sale_id'],
                            'success' => false,
                            'message' => 'Transaction or sale not found'
                        ];
                        $failedCount++;
                        continue;
                    }
                    
                    if ($transaction->status !== 'completed') {
                        $results[] = [
                            'transaction_id' => $match['transaction_id'],
                            'sale_id' => $match['sale_id'],
                            'success' => false,
                            'message' => 'Transaction not completed'
                        ];
                        $failedCount++;
                        continue;
                    }
                    
                    if ($transaction->sale_id) {
                        $results[] = [
                            'transaction_id' => $match['transaction_id'],
                            'sale_id' => $match['sale_id'],
                            'success' => false,
                            'message' => 'Transaction already matched'
                        ];
                        $failedCount++;
                        continue;
                    }
                    
                    if ($sale->payment_status === 'paid') {
                        $results[] = [
                            'transaction_id' => $match['transaction_id'],
                            'sale_id' => $match['sale_id'],
                            'success' => false,
                            'message' => 'Sale already paid'
                        ];
                        $failedCount++;
                        continue;
                    }
                    
                    // Match transaction
                    $transaction->update([
                        'sale_id' => $sale->id,
                        'customer_id' => $sale->customer_id,
                        'status' => 'matched',
                        'matched_at' => now(),
                        'matched_by' => auth()->id(),
                        'match_type' => $match['match_type'] ?? 'manual',
                        'match_notes' => $match['notes'] ?? null
                    ]);
                    
                    // Update sale
                    $sale->update([
                        'payment_status' => 'paid',
                        'payment_method' => 'mpesa',
                        'payment_reference' => $transaction->transaction_id,
                        'paid_at' => now(),
                        'paid_by' => auth()->id()
                    ]);
                    
                    // Create payment record
                    Payment::create([
                        'sale_id' => $sale->id,
                        'amount' => $transaction->amount,
                        'payment_method' => 'mpesa',
                        'reference' => $transaction->transaction_id,
                        'transaction_id' => $transaction->transaction_id,
                        'phone' => $transaction->phone,
                        'status' => 'completed',
                        'payment_date' => $transaction->transaction_date,
                        'received_by' => auth()->id(),
                        'notes' => 'Bulk matched with MPESA transaction'
                    ]);
                    
                    $results[] = [
                        'transaction_id' => $transaction->id,
                        'sale_id' => $sale->id,
                        'success' => true,
                        'message' => 'Matched successfully',
                        'transaction_amount' => $transaction->amount,
                        'sale_amount' => $sale->grand_total
                    ];
                    $successCount++;
                    
                } catch (\Exception $e) {
                    Log::error('Bulk match item error: ' . $e->getMessage(), $match);
                    
                    $results[] = [
                        'transaction_id' => $match['transaction_id'],
                        'sale_id' => $match['sale_id'],
                        'success' => false,
                        'message' => 'Error: ' . $e->getMessage()
                    ];
                    $failedCount++;
                }
            }
            
            DB::commit();
            
            // Clear cache
            Cache::forget('mpesa_dashboard_stats');
            Cache::forget('mpesa_transactions_stats');
            
            // Log activity
            activity('mpesa-bulk-match')
                ->causedBy(auth()->user())
                ->withProperties([
                    'total' => count($request->matches),
                    'successful' => $successCount,
                    'failed' => $failedCount
                ])
                ->log('Bulk MPESA transaction matching');
            
            return response()->json([
                'success' => true,
                'message' => "Bulk match completed. Successful: {$successCount}, Failed: {$failedCount}",
                'results' => $results,
                'summary' => [
                    'total' => count($request->matches),
                    'successful' => $successCount,
                    'failed' => $failedCount
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('MPESA Bulk Match Error: ' . $e->getMessage(), [
                'matches_count' => count($request->matches ?? []),
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Bulk match failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Daily Report
     */
    public function dailyReport(Request $request)
    {
        try {
            $date = $request->input('date', now()->format('Y-m-d'));
            
            $transactions = MpesaTransaction::with(['sale', 'customer'])
                ->whereDate('transaction_date', $date)
                ->where('status', 'completed')
                ->orderBy('transaction_date', 'desc')
                ->get();
            
            $summary = [
                'total_amount' => $transactions->sum('amount'),
                'total_count' => $transactions->count(),
                'matched_count' => $transactions->whereNotNull('sale_id')->count(),
                'unmatched_count' => $transactions->whereNull('sale_id')->count(),
                'matched_amount' => $transactions->whereNotNull('sale_id')->sum('amount'),
                'unmatched_amount' => $transactions->whereNull('sale_id')->sum('amount'),
                'avg_amount' => $transactions->count() > 0 ? $transactions->sum('amount') / $transactions->count() : 0,
                'max_amount' => $transactions->max('amount'),
                'min_amount' => $transactions->min('amount')
            ];
            
            // Top phones
            $topPhones = $transactions->groupBy('phone')
                ->map(function($group) {
                    return [
                        'count' => $group->count(),
                        'amount' => $group->sum('amount'),
                        'matched' => $group->whereNotNull('sale_id')->count(),
                        'unmatched' => $group->whereNull('sale_id')->count()
                    ];
                })
                ->sortByDesc('amount')
                ->take(10);
            
            // Hourly distribution
            $hourlyData = [];
            for ($hour = 0; $hour < 24; $hour++) {
                $hourTransactions = $transactions->filter(function($transaction) use ($hour) {
                    return $transaction->transaction_date->hour == $hour;
                });
                
                $hourlyData[] = [
                    'hour' => sprintf('%02d:00', $hour),
                    'count' => $hourTransactions->count(),
                    'amount' => $hourTransactions->sum('amount')
                ];
            }
            
            return view('payments.mpesa.reports.daily', compact(
                'transactions',
                'summary',
                'date',
                'topPhones',
                'hourlyData'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Daily Report Error: ' . $e->getMessage());
            
            return redirect()->route('payments.mpesa.transactions')
                ->with('error', 'Failed to generate daily report: ' . $e->getMessage());
        }
    }
    
    /**
     * Unmatched Transactions Report
     */
    public function unmatchedReport(Request $request)
    {
        try {
            $days = $request->input('days', 7);
            $since = now()->subDays($days);
            
            $transactions = MpesaTransaction::with(['customer'])
                ->whereNull('sale_id')
                ->where('status', 'completed')
                ->where('transaction_date', '>=', $since)
                ->orderBy('transaction_date', 'desc')
                ->paginate(50);
            
            $summary = [
                'total_count' => $transactions->total(),
                'total_amount' => $transactions->sum('amount'),
                'avg_amount' => $transactions->count() > 0 ? $transactions->sum('amount') / $transactions->count() : 0,
                'oldest' => $transactions->last()->transaction_date ?? null,
                'newest' => $transactions->first()->transaction_date ?? null
            ];
            
            // Group by phone for potential customer matching
            $phoneGroups = collect();
            if ($transactions->count() > 0) {
                $phoneGroups = $transactions->groupBy('phone')
                    ->map(function($group) {
                        return [
                            'count' => $group->count(),
                            'amount' => $group->sum('amount'),
                            'transactions' => $group,
                            'customer' => Customer::where('phone', $group->first()->phone)->first()
                        ];
                    })
                    ->sortByDesc('amount');
            }
            
            // Potential sales for matching
            $potentialSales = [];
            if ($phoneGroups->isNotEmpty()) {
                foreach ($phoneGroups->take(5) as $phone => $group) {
                    if ($group['customer']) {
                        $potentialSales[$phone] = Sale::where('customer_id', $group['customer']->id)
                            ->where('payment_status', '!=', 'paid')
                            ->orderBy('created_at', 'desc')
                            ->limit(3)
                            ->get();
                    }
                }
            }
            
            return view('payments.mpesa.reports.unmatched', compact(
                'transactions',
                'summary',
                'days',
                'phoneGroups',
                'potentialSales'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Unmatched Report Error: ' . $e->getMessage());
            
            return redirect()->route('payments.mpesa.transactions')
                ->with('error', 'Failed to generate unmatched report: ' . $e->getMessage());
        }
    }
    
    /**
     * Reconciliation Report
     */
    public function reconciliationReport(Request $request)
    {
        try {
            $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
            $endDate = $request->input('end_date', now()->format('Y-m-d'));
            
            $transactions = MpesaTransaction::whereBetween('transaction_date', [$startDate, $endDate])
                ->where('status', 'completed')
                ->get();
            
            // Daily breakdown
            $dailyBreakdown = $transactions->groupBy(function($item) {
                return $item->transaction_date->format('Y-m-d');
            })->map(function($dayTransactions) {
                return [
                    'count' => $dayTransactions->count(),
                    'amount' => $dayTransactions->sum('amount'),
                    'matched' => $dayTransactions->whereNotNull('sale_id')->count(),
                    'unmatched' => $dayTransactions->whereNull('sale_id')->count(),
                    'matched_amount' => $dayTransactions->whereNotNull('sale_id')->sum('amount'),
                    'unmatched_amount' => $dayTransactions->whereNull('sale_id')->sum('amount')
                ];
            })->sortKeys();
            
            // Summary
            $summary = [
                'total_transactions' => $transactions->count(),
                'total_amount' => $transactions->sum('amount'),
                'matched_count' => $transactions->whereNotNull('sale_id')->count(),
                'unmatched_count' => $transactions->whereNull('sale_id')->count(),
                'matched_amount' => $transactions->whereNotNull('sale_id')->sum('amount'),
                'unmatched_amount' => $transactions->whereNull('sale_id')->sum('amount'),
                'match_rate' => $transactions->count() > 0 ? 
                    ($transactions->whereNotNull('sale_id')->count() / $transactions->count()) * 100 : 0,
                'days_covered' => count($dailyBreakdown),
                'avg_daily_amount' => count($dailyBreakdown) > 0 ? 
                    $transactions->sum('amount') / count($dailyBreakdown) : 0
            ];
            
            // Top performing days
            $topDays = $dailyBreakdown->sortByDesc('amount')->take(5);
            
            // Discrepancies (if any)
            $discrepancies = [];
            
            // Check for sales without matching MPESA transactions
            $salesWithoutMpesa = Sale::whereBetween('created_at', [$startDate, $endDate])
                ->where('payment_method', 'mpesa')
                ->where('payment_status', 'paid')
                ->whereDoesntHave('payments', function($query) {
                    $query->where('payment_method', 'mpesa');
                })
                ->count();
            
            if ($salesWithoutMpesa > 0) {
                $discrepancies[] = [
                    'type' => 'sales_without_mpesa',
                    'count' => $salesWithoutMpesa,
                    'message' => "{$salesWithoutMpesa} sales marked as MPESA paid but no MPESA payment record found"
                ];
            }
            
            // Check for MPESA transactions without matching sales
            $mpesaWithoutSales = $transactions->whereNull('sale_id')->count();
            if ($mpesaWithoutSales > 0) {
                $discrepancies[] = [
                    'type' => 'mpesa_without_sales',
                    'count' => $mpesaWithoutSales,
                    'amount' => $transactions->whereNull('sale_id')->sum('amount'),
                    'message' => "{$mpesaWithoutSales} MPESA transactions ({$transactions->whereNull('sale_id')->sum('amount')}) not matched to sales"
                ];
            }
            
            return view('payments.mpesa.reports.reconciliation', compact(
                'summary',
                'dailyBreakdown',
                'topDays',
                'discrepancies',
                'startDate',
                'endDate'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Reconciliation Report Error: ' . $e->getMessage());
            
            return redirect()->route('payments.mpesa.transactions')
                ->with('error', 'Failed to generate reconciliation report: ' . $e->getMessage());
        }
    }
    
    /**
     * Export Transactions
     */
    public function exportTransactions(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'format' => ['required', Rule::in(['csv', 'excel', 'pdf'])],
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'status' => 'nullable|string',
                'include_details' => 'boolean'
            ]);
            
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            
            $query = MpesaTransaction::query();
            
            if ($request->start_date) {
                $query->whereDate('transaction_date', '>=', $request->start_date);
            }
            
            if ($request->end_date) {
                $query->whereDate('transaction_date', '<=', $request->end_date);
            }
            
            if ($request->status) {
                $query->where('status', $request->status);
            }
            
            $transactions = $query->orderBy('transaction_date', 'desc')->get();
            
            $filename = 'mpesa-transactions-' . now()->format('Ymd-His') . '.' . $request->format;
            
            if ($request->format === 'csv') {
                return Excel::download(new MpesaTransactionsExport($transactions), $filename);
            } elseif ($request->format === 'excel') {
                return Excel::download(new MpesaTransactionsExport($transactions, true), $filename);
            } else {
                $pdf = PDF::loadView('payments.mpesa.exports.pdf', compact('transactions'));
                return $pdf->download($filename);
            }
            
        } catch (\Exception $e) {
            Log::error('MPESA Export Error: ' . $e->getMessage());
            
            return back()->with('error', 'Failed to export transactions: ' . $e->getMessage());
        }
    }
    
    /**
     * Get Transaction Timeline
     */
    private function getTransactionTimeline($transaction)
    {
        $timeline = [];
        
        // Transaction received
        $timeline[] = [
            'event' => 'Transaction Received',
            'timestamp' => $transaction->transaction_date,
            'description' => 'MPESA transaction received from ' . $transaction->phone,
            'icon' => 'bx bx-receipt',
            'color' => 'primary'
        ];
        
        // Stored in system
        $timeline[] = [
            'event' => 'Stored in System',
            'timestamp' => $transaction->created_at,
            'description' => 'Transaction stored in database',
            'icon' => 'bx bx-save',
            'color' => 'info'
        ];
        
        // Matched (if applicable)
        if ($transaction->matched_at) {
            $timeline[] = [
                'event' => 'Matched with Sale',
                'timestamp' => $transaction->matched_at,
                'description' => 'Matched with Sale #' . $transaction->sale_id,
                'icon' => 'bx bx-link',
                'color' => 'success'
            ];
        }
        
        // Reversed (if applicable)
        if ($transaction->reversed_at) {
            $timeline[] = [
                'event' => 'Transaction Reversed',
                'timestamp' => $transaction->reversed_at,
                'description' => 'Transaction reversed: ' . ($transaction->reversal_reason ?? 'No reason provided'),
                'icon' => 'bx bx-undo',
                'color' => 'danger'
            ];
        }
        
        // Sort by timestamp
        usort($timeline, function($a, $b) {
            return $a['timestamp'] <=> $b['timestamp'];
        });
        
        return $timeline;
    }
    
    /**
     * Send Reversal Notification
     */
    private function sendReversalNotification($transaction, $reason)
    {
        try {
            // This would integrate with your notification system
            // For now, just log it
            
            Log::info('Reversal notification sent', [
                'transaction_id' => $transaction->id,
                'phone' => $transaction->phone,
                'amount' => $transaction->amount,
                'reason' => $reason
            ]);
            
        } catch (\Exception $e) {
            Log::error('Reversal notification failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Format Phone Number
     */
    private function formatPhoneNumber($phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($phone) === 9 && !str_starts_with($phone, '0')) {
            return '254' . $phone;
        }
        
        if (strlen($phone) === 10 && str_starts_with($phone, '0')) {
            return '254' . substr($phone, 1);
        }
        
        if (strlen($phone) === 12 && str_starts_with($phone, '254')) {
            return $phone;
        }
        
        return $phone;
    }
    
    /**
     * API: Get Transactions (for backward compatibility)
     */
    public function index(Request $request)
    {
        try {
            $query = MpesaTransaction::with(['sale', 'customer'])
                ->orderBy('transaction_date', 'desc');
            
            // Apply filters
            if ($search = $request->input('search')) {
                $query->where(function($q) use ($search) {
                    $q->where('transaction_id', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('reference', 'like', "%{$search}%");
                });
            }
            
            if ($status = $request->input('status')) {
                $query->where('status', $status);
            }
            
            if ($date = $request->input('date')) {
                $query->whereDate('transaction_date', $date);
            }
            
            if ($request->boolean('unmatched')) {
                $query->whereNull('sale_id')->where('status', 'completed');
            }
            
            if ($phone = $request->input('phone')) {
                $normalizedPhone = $this->formatPhoneNumber($phone);
                $query->where('phone', $normalizedPhone);
            }
            
            $perPage = $request->input('per_page', 20);
            $transactions = $query->paginate($perPage);
            
            return response()->json([
                'success' => true,
                'transactions' => $transactions->items(),
                'pagination' => [
                    'current_page' => $transactions->currentPage(),
                    'last_page' => $transactions->lastPage(),
                    'per_page' => $transactions->perPage(),
                    'total' => $transactions->total(),
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get MPESA transactions API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get transactions'], 500);
        }
    }
    
    /**
     * API: Show Single Transaction
     */
    public function show($id)
    {
        try {
            $transaction = MpesaTransaction::with(['sale', 'customer'])->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'transaction' => $transaction
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get MPESA transaction API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Transaction not found'], 404);
        }
    }
    
    /**
     * API: Check Pending Payment
     */
    public function checkPendingPayment(Request $request)
    {
        try {
            $validated = $request->validate([
                'phone' => 'required|string',
                'amount' => 'required|numeric|min:0',
                'reference' => 'nullable|string',
                'within_minutes' => 'nullable|integer|min:1|max:60'
            ]);
            
            $normalizedPhone = $this->formatPhoneNumber($validated['phone']);
            $withinMinutes = $validated['within_minutes'] ?? 10;
            
            $since = now()->subMinutes($withinMinutes);
            
            $transactions = MpesaTransaction::where('phone', $normalizedPhone)
                ->where('status', 'completed')
                ->whereNull('sale_id')
                ->where('transaction_date', '>=', $since)
                ->orderBy('transaction_date', 'desc')
                ->get();
            
            $exactMatch = $transactions->first(function($transaction) use ($validated) {
                return abs($transaction->amount - $validated['amount']) < 0.01;
            });
            
            if ($exactMatch) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'exact',
                    'transaction' => $exactMatch,
                    'message' => 'Exact MPESA payment found!'
                ]);
            }
            
            if ($transactions->count() > 0) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'multiple',
                    'transactions' => $transactions->values(),
                    'message' => 'Found ' . $transactions->count() . ' MPESA transactions for this phone'
                ]);
            }
            
            return response()->json([
                'success' => true,
                'found' => false,
                'message' => 'No matching MPESA payment found'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Check pending payment API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to check payment'], 500);
        }
    }
    
    /**
     * API: Match Transaction
     */
    public function match(Request $request, $id)
    {
        DB::beginTransaction();
        
        try {
            $validated = $request->validate([
                'sale_id' => 'required|exists:sales,id'
            ]);
            
            $transaction = MpesaTransaction::findOrFail($id);
            $sale = Sale::with('customer')->findOrFail($validated['sale_id']);
            
            if ($transaction->sale_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction already matched with sale #' . $transaction->sale_id
                ], 400);
            }
            
            $transaction->update([
                'sale_id' => $sale->id,
                'customer_id' => $sale->customer_id,
                'status' => 'matched',
                'matched_at' => now()
            ]);
            
            if ($sale->payment_status !== 'paid') {
                $sale->update([
                    'payment_status' => 'paid',
                    'payment_reference' => $transaction->transaction_id,
                    'paid_at' => now()
                ]);
                
                $sale->payments()->create([
                    'amount' => $transaction->amount,
                    'payment_method' => 'mpesa',
                    'reference' => $transaction->transaction_id,
                    'transaction_id' => $transaction->transaction_id,
                    'phone' => $transaction->phone,
                    'status' => 'completed',
                    'payment_date' => $transaction->transaction_date,
                    'received_by' => auth()->id(),
                    'notes' => 'Matched with MPESA transaction ' . $transaction->transaction_id
                ]);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Transaction matched successfully',
                'transaction' => $transaction,
                'sale' => $sale
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Match transaction API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to match transaction'], 500);
        }
    }
    
    /**
     * API: Reverse Transaction
     */
    public function reverse($id)
    {
        DB::beginTransaction();
        
        try {
            $transaction = MpesaTransaction::findOrFail($id);
            
            if ($transaction->status !== 'completed') {
                return response()->json([
                    'success' => false,
                    'error' => 'Only completed transactions can be reversed'
                ], 400);
            }
            
            $transaction->update([
                'status' => 'reversed',
                'reversed_at' => now(),
                'reversed_by' => auth()->id()
            ]);
            
            if ($transaction->sale_id) {
                $sale = Sale::find($transaction->sale_id);
                if ($sale) {
                    $sale->update([
                        'payment_status' => 'pending',
                        'payment_reference' => null,
                        'paid_at' => null
                    ]);
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Transaction reversed successfully',
                'transaction' => $transaction
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Reverse transaction API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to reverse transaction'], 500);
        }
    }
    
    /**
     * API: Get Stats
     */
    public function stats()
    {
        try {
            $today = now()->format('Y-m-d');
            $weekStart = now()->startOfWeek()->format('Y-m-d');
            $monthStart = now()->startOfMonth()->format('Y-m-d');
            
            $stats = [
                'today' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', $today)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', $today)->sum('amount'),
                    'matched' => MpesaTransaction::whereDate('transaction_date', $today)->whereNotNull('sale_id')->count(),
                    'unmatched' => MpesaTransaction::whereDate('transaction_date', $today)->whereNull('sale_id')->count(),
                ],
                'week' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', '>=', $weekStart)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', '>=', $weekStart)->sum('amount'),
                ],
                'month' => [
                    'count' => MpesaTransaction::whereDate('transaction_date', '>=', $monthStart)->count(),
                    'amount' => MpesaTransaction::whereDate('transaction_date', '>=', $monthStart)->sum('amount'),
                ],
                'total' => [
                    'count' => MpesaTransaction::count(),
                    'amount' => MpesaTransaction::sum('amount'),
                    'matched_count' => MpesaTransaction::whereNotNull('sale_id')->count(),
                    'unmatched_count' => MpesaTransaction::whereNull('sale_id')->where('status', 'completed')->count(),
                ]
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get MPESA stats API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get stats'], 500);
        }
    }
    
    /**
     * API: Dashboard Stats
     */
    public function dashboardStats()
    {
        try {
            $today = now()->format('Y-m-d');
            
            $stats = [
                'today_total' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->sum('amount'),
                'today_count' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->count(),
                'unmatched_today' => MpesaTransaction::whereDate('transaction_date', $today)
                    ->where('status', 'completed')
                    ->whereNull('sale_id')
                    ->count(),
                'recent_unmatched' => MpesaTransaction::whereNull('sale_id')
                    ->where('status', 'completed')
                    ->where('transaction_date', '>=', now()->subHours(6))
                    ->orderBy('transaction_date', 'desc')
                    ->limit(5)
                    ->get()
                    ->map(function($transaction) {
                        return [
                            'id' => $transaction->id,
                            'transaction_id' => $transaction->transaction_id,
                            'phone' => $transaction->phone,
                            'amount' => $transaction->amount,
                            'transaction_date' => $transaction->transaction_date->format('H:i:s'),
                            'full_name' => $transaction->full_name,
                        ];
                    }),
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get MPESA dashboard stats API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to get dashboard stats'], 500);
        }
    }
}