<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use App\Models\MpesaCredential;
use App\Models\MpesaTransaction;
use App\Models\Sale;
use App\Models\Customer;
use App\Models\Setting;
use Carbon\Carbon;
use App\Services\MpesaService;

class MpesaController extends Controller
{
    protected $mpesaService;
    
    public function __construct(MpesaService $mpesaService)
    {
        $this->mpesaService = $mpesaService;
        
        $this->middleware('permission:mpesa.view', ['only' => ['index', 'dashboard', 'configuration', 'settings', 'test']]);
        $this->middleware('permission:mpesa.manage', ['only' => ['updateConfiguration', 'updateSettings', 'testConnection']]);
    }
    
    /**
     * Main MPESA Dashboard
     */
    public function index()
    {
        return redirect()->route('payments.mpesa.dashboard');
    }
    
    /**
     * Display MPESA Dashboard with Stats
     */
    public function dashboard(Request $request)
    {
        try {
            $today = Carbon::today();
            $yesterday = Carbon::yesterday();
            $weekStart = $today->copy()->startOfWeek();
            $monthStart = $today->copy()->startOfMonth();
            
            // Cache stats for performance
            $stats = Cache::remember('mpesa_dashboard_stats', 300, function () use ($today, $yesterday, $weekStart, $monthStart) {
                return [
                    'total' => [
                        'transactions' => MpesaTransaction::count(),
                        'amount' => MpesaTransaction::sum('amount'),
                        'matched' => MpesaTransaction::whereNotNull('sale_id')->count(),
                        'unmatched' => MpesaTransaction::whereNull('sale_id')->where('status', 'completed')->count(),
                        'failed' => MpesaTransaction::where('status', 'failed')->count(),
                    ],
                    'today' => [
                        'transactions' => MpesaTransaction::whereDate('created_at', $today)->count(),
                        'amount' => MpesaTransaction::whereDate('created_at', $today)->sum('amount'),
                        'matched' => MpesaTransaction::whereDate('created_at', $today)->whereNotNull('sale_id')->count(),
                        'unmatched' => MpesaTransaction::whereDate('created_at', $today)->whereNull('sale_id')->where('status', 'completed')->count(),
                    ],
                    'yesterday' => [
                        'transactions' => MpesaTransaction::whereDate('created_at', $yesterday)->count(),
                        'amount' => MpesaTransaction::whereDate('created_at', $yesterday)->sum('amount'),
                    ],
                    'week' => [
                        'transactions' => MpesaTransaction::whereDate('created_at', '>=', $weekStart)->count(),
                        'amount' => MpesaTransaction::whereDate('created_at', '>=', $weekStart)->sum('amount'),
                    ],
                    'month' => [
                        'transactions' => MpesaTransaction::whereDate('created_at', '>=', $monthStart)->count(),
                        'amount' => MpesaTransaction::whereDate('created_at', '>=', $monthStart)->sum('amount'),
                    ],
                ];
            });
            
            // Recent transactions
            $recentTransactions = MpesaTransaction::with(['sale', 'customer'])
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();
            
            // Active credentials
            $credentials = MpesaCredential::active()->get();
            
            // Top phones by transaction volume
            $topPhones = MpesaTransaction::select('phone', DB::raw('COUNT(*) as transaction_count'), DB::raw('SUM(amount) as total_amount'))
                ->where('status', 'completed')
                ->groupBy('phone')
                ->orderBy('total_amount', 'desc')
                ->limit(5)
                ->get();
            
            // Daily chart data (last 7 days)
            $chartData = $this->getChartData();
            
            return view('payments.mpesa.dashboard', compact('stats', 'recentTransactions', 'credentials', 'topPhones', 'chartData'));
            
        } catch (\Exception $e) {
            Log::error('MPESA Dashboard Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);
            
            return view('payments.mpesa.dashboard')->withErrors([
                'error' => 'Failed to load dashboard data. Please try again.'
            ]);
        }
    }
    
    /**
     * Display Configuration Page
     */
    public function configuration()
    {
        try {
            $credentials = MpesaCredential::orderBy('is_active', 'desc')
                ->orderBy('created_at', 'desc')
                ->get();
            
            $activeCredential = MpesaCredential::active()->first();
            
            // Default configuration for new credentials
            $defaults = [
                'name' => 'Production Credentials',
                'business_shortcode' => '174379', // Sandbox default
                'environment' => 'sandbox',
                'callback_url' => url('/api/mpesa/callback'),
                'is_active' => true
            ];
            
            return view('payments.mpesa.configuration', compact('credentials', 'activeCredential', 'defaults'));
            
        } catch (\Exception $e) {
            Log::error('MPESA Configuration View Error: ' . $e->getMessage());
            
            return redirect()->route('payments.mpesa.dashboard')
                ->with('error', 'Failed to load configuration page.');
        }
    }
    
    /**
     * Get Credential Data for Editing
     */
    public function getCredential(Request $request)
    {
        try {
            $request->validate(['id' => 'required|exists:mpesa_credentials,id']);
            
            $credential = MpesaCredential::findOrFail($request->id);
            
            // Return credential data (without sensitive fields)
            return response()->json([
                'success' => true,
                'credential' => [
                    'id' => $credential->id,
                    'name' => $credential->name,
                    'business_shortcode' => $credential->business_shortcode,
                    'initiator_name' => $credential->initiator_name,
                    'security_credential' => $credential->security_credential,
                    'environment' => $credential->environment,
                    'is_active' => $credential->is_active,
                    'callback_url' => $credential->callback_url,
                    'validation_url' => $credential->validation_url,
                    'confirmation_url' => $credential->confirmation_url,
                    'notes' => $credential->notes,
                    'created_at' => $credential->created_at->format('Y-m-d H:i:s'),
                    'updated_at' => $credential->updated_at->format('Y-m-d H:i:s'),
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get MPESA Credential Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to get credential data'
            ], 500);
        }
    }
    
    /**
     * Activate Credential
     */
    public function activateCredential(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $request->validate(['id' => 'required|exists:mpesa_credentials,id']);
            
            // Deactivate all credentials first
            MpesaCredential::where('is_active', true)->update(['is_active' => false]);
            
            // Activate the selected one
            $credential = MpesaCredential::findOrFail($request->id);
            $credential->update(['is_active' => true]);
            
            DB::commit();
            
            Cache::forget('mpesa_active_credential');
            Cache::forget('mpesa_dashboard_stats');
            
            return response()->json([
                'success' => true,
                'message' => 'Credential activated successfully'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Activate MPESA Credential Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate credential'
            ], 500);
        }
    }
    
    /**
     * Delete Credential
     */
    public function deleteCredential(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $request->validate(['id' => 'required|exists:mpesa_credentials,id']);
            
            $credential = MpesaCredential::findOrFail($request->id);
            
            // Check if this is the only credential
            $totalCredentials = MpesaCredential::count();
            
            if ($totalCredentials <= 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete the only credential'
                ], 400);
            }
            
            // Check if it's active
            if ($credential->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete active credential. Please activate another credential first.'
                ], 400);
            }
            
            // Check if there are transactions using this credential
            $transactionCount = MpesaTransaction::where('credential_id', $credential->id)->count();
            
            if ($transactionCount > 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Cannot delete credential with {$transactionCount} associated transactions"
                ], 400);
            }
            
            $credential->delete();
            
            DB::commit();
            
            Cache::forget('mpesa_active_credential');
            Cache::forget('mpesa_dashboard_stats');
            
            return response()->json([
                'success' => true,
                'message' => 'Credential deleted successfully'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Delete MPESA Credential Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete credential'
            ], 500);
        }
    }
    
    /**
     * Save MPESA Configuration
     */
    public function updateConfiguration(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'business_shortcode' => 'required|string|max:20',
                'consumer_key' => 'required|string',
                'consumer_secret' => 'required|string',
                'passkey' => 'required|string',
                'initiator_name' => 'nullable|string|max:100',
                'initiator_password' => 'nullable|string',
                'security_credential' => 'nullable|string',
                'environment' => ['required', Rule::in(['sandbox', 'production'])],
                'callback_url' => 'required|url|max:500',
                'validation_url' => 'nullable|url|max:500',
                'confirmation_url' => 'nullable|url|max:500',
                'is_active' => 'boolean',
                'notes' => 'nullable|string|max:1000',
                'test_connection' => 'boolean',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            // Check for duplicate shortcode in same environment
            $existing = MpesaCredential::where('business_shortcode', $request->business_shortcode)
                ->where('environment', $request->environment)
                ->when($request->has('id') && $request->id, function($query) use ($request) {
                    return $query->where('id', '!=', $request->id);
                })
                ->first();
            
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Credentials already exist for this shortcode in ' . $request->environment . ' environment'
                ], 400);
            }
            
            // Prepare data
            $data = $request->only([
                'name', 'business_shortcode', 'initiator_name', 
                'security_credential', 'environment', 'callback_url',
                'validation_url', 'confirmation_url', 'notes'
            ]);
            
            // Encrypt sensitive data
            $data['consumer_key'] = encrypt($request->consumer_key);
            $data['consumer_secret'] = encrypt($request->consumer_secret);
            $data['passkey'] = encrypt($request->passkey);
            
            if ($request->initiator_password) {
                $data['initiator_password'] = encrypt($request->initiator_password);
            }
            
            $data['is_active'] = $request->boolean('is_active');
            
            // If activating this credential, deactivate others
            if ($data['is_active']) {
                MpesaCredential::where('is_active', true)->update(['is_active' => false]);
            }
            
            // Check if we're updating or creating
            if ($request->has('id') && $request->id) {
                $credential = MpesaCredential::findOrFail($request->id);
                $credential->update($data);
                $message = 'Configuration updated successfully';
                $isNew = false;
            } else {
                $credential = MpesaCredential::create($data);
                $message = 'Configuration saved successfully';
                $isNew = true;
            }
            
            // Test connection if requested
            $testResult = null;
            if ($request->boolean('test_connection')) {
                $testResult = $this->testConnectionInternal($credential);
                
                if ($testResult['success']) {
                    $credential->update(['last_success' => now()]);
                    $message .= ' and connection test passed';
                } else {
                    $credential->update(['last_failure' => now()]);
                    $message .= ' but connection test failed';
                }
            }
            
            DB::commit();
            
            // Clear cache
            Cache::forget('mpesa_dashboard_stats');
            Cache::forget('mpesa_active_credential');
            
            return response()->json([
                'success' => true,
                'message' => $message,
                'credential' => $credential,
                'test_result' => $testResult,
                'is_new' => $isNew
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('MPESA Configuration Save Error: ' . $e->getMessage(), [
                'data' => $request->except(['consumer_key', 'consumer_secret', 'passkey', 'initiator_password', 'security_credential']),
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to save configuration: ' . $e->getMessage()
            ], 500);
        }
    }
    

        /**
         * Display Settings Page
         */
        public function settings()
        {
            try {
                // Default settings
                $defaultSettings = [
                    'auto_check_payments' => '1',
                    'auto_check_minutes' => '5',
                    'auto_match_payments' => '1',
                    'auto_sync_hours' => '24',
                    'notification_email' => '',
                    'notification_channels' => '["email","sms"]',
                    'daily_report_time' => '09:00',
                    'reconciliation_days' => '30',
                    'max_amount' => '150000',
                    'min_amount' => '10',
                    'max_retry_attempts' => '3',
                    'retry_delay_minutes' => '5',
                    'enable_logging' => '1',
                    'log_level' => 'info',
                ];
                
                // Get saved settings from database
                $savedSettings = [];
                $dbSettings = Setting::where('key', 'like', 'mpesa.%')->get();
                
                foreach ($dbSettings as $setting) {
                    $key = str_replace('mpesa.', '', $setting->key);
                    $savedSettings[$key] = $setting->value;
                }
                
                // Merge saved settings with defaults
                $settings = array_merge($defaultSettings, $savedSettings);
                
                // Decode JSON fields
                if (isset($settings['notification_channels'])) {
                    $channels = json_decode($settings['notification_channels'], true);
                    $settings['notification_channels'] = is_array($channels) ? $channels : ['email', 'sms'];
                } else {
                    $settings['notification_channels'] = ['email', 'sms'];
                }
                
                // Convert string booleans to actual booleans for checkboxes
                $booleanFields = ['auto_check_payments', 'auto_match_payments', 'enable_logging'];
                foreach ($booleanFields as $field) {
                    if (isset($settings[$field])) {
                        $settings[$field] = $settings[$field] === '1' || $settings[$field] === 'true' || $settings[$field] === true;
                    }
                }
                
                // Get available notification emails
                $notificationEmails = \App\Models\User::whereNotNull('email')
                    ->where('email', '!=', '')
                    ->pluck('email', 'id')
                    ->toArray();
                
                return view('payments.mpesa.settings', compact('settings', 'notificationEmails'));
                
            } catch (\Exception $e) {
                Log::error('MPESA Settings View Error: ' . $e->getMessage(), [
                    'trace' => $e->getTraceAsString()
                ]);
                
                return redirect()->route('payments.mpesa.dashboard')
                    ->with('error', 'Failed to load settings page: ' . $e->getMessage());
            }
        }
    /**
     * Update MPESA Settings
     */
    public function updateSettings(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'auto_check_payments' => 'boolean',
                'auto_check_minutes' => 'integer|min:1|max:1440',
                'auto_match_payments' => 'boolean',
                'auto_sync_hours' => 'integer|min:1|max:168',
                'notification_email' => 'nullable|email|max:255',
                'notification_channels' => 'nullable|array',
                'notification_channels.*' => 'in:email,sms,dashboard',
                'daily_report_time' => 'nullable|date_format:H:i',
                'reconciliation_days' => 'integer|min:1|max:365',
                'max_amount' => 'numeric|min:1|max:150000',
                'min_amount' => 'numeric|min:10|max:150000',
                'max_retry_attempts' => 'integer|min:0|max:10',
                'retry_delay_minutes' => 'integer|min:1|max:60',
                'enable_logging' => 'boolean',
                'log_level' => ['nullable', Rule::in(['debug', 'info', 'warning', 'error'])],
            ]);
            
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            
            // Save settings to database
            $settingsToSave = $request->only([
                'auto_check_payments',
                'auto_check_minutes',
                'auto_match_payments',
                'auto_sync_hours',
                'notification_email',
                'notification_channels',
                'daily_report_time',
                'reconciliation_days',
                'max_amount',
                'min_amount',
                'max_retry_attempts',
                'retry_delay_minutes',
                'enable_logging',
                'log_level'
            ]);
            
            // Convert arrays to JSON
            foreach ($settingsToSave as $key => $value) {
                if (is_array($value)) {
                    $settingsToSave[$key] = json_encode($value);
                }
            }
            
            // Save to database
            foreach ($settingsToSave as $key => $value) {
                Setting::updateOrCreate(
                    ['key' => 'mpesa.' . $key],
                    ['value' => $value]
                );
            }
            
            // Clear relevant caches
            Cache::forget('mpesa_settings');
            Cache::forget('mpesa_dashboard_stats');
            
            // Log activity
            activity('mpesa-settings')
                ->causedBy(auth()->user())
                ->withProperties($settingsToSave)
                ->log('MPESA settings updated');
            
            return redirect()->route('payments.mpesa.settings')
                ->with('success', 'Settings updated successfully.')
                ->with('settings_updated', true);
            
        } catch (\Exception $e) {
            Log::error('MPESA Settings Update Error: ' . $e->getMessage(), [
                'settings' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return back()->withErrors(['error' => 'Failed to update settings: ' . $e->getMessage()])->withInput();
        }
    }
    
    /**
     * Display Test Page
     */
    public function test()
    {
        try {
            $credentials = MpesaCredential::orderBy('is_active', 'desc')
                ->orderBy('created_at', 'desc')
                ->get();
            
            $activeCredential = MpesaCredential::active()->first();
            
            // Recent test transactions
            $recentTransactions = MpesaTransaction::where('transaction_type', 'like', '%test%')
                ->orWhere('status', 'pending')
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();
            
            // Test customers (for STK Push)
            $testCustomers = Customer::where('phone', 'like', '2547%')
                ->limit(5)
                ->get();
            
            // Test amounts
            $testAmounts = [100, 500, 1000, 5000, 10000];
            
            return view('payments.mpesa.test', compact(
                'credentials', 
                'activeCredential', 
                'recentTransactions',
                'testCustomers',
                'testAmounts'
            ));
            
        } catch (\Exception $e) {
            Log::error('MPESA Test View Error: ' . $e->getMessage());
            
            return redirect()->route('payments.mpesa.dashboard')
                ->with('error', 'Failed to load test page.');
        }
    }
    
    /**
     * Test MPESA Connection
     */
    public function testConnection(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'credential_id' => 'nullable|exists:mpesa_credentials,id',
                'consumer_key' => 'required_without:credential_id|string',
                'consumer_secret' => 'required_without:credential_id|string',
                'environment' => ['required', Rule::in(['sandbox', 'production'])]
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            if ($request->credential_id) {
                $credential = MpesaCredential::findOrFail($request->credential_id);
                $result = $this->testConnectionInternal($credential);
                
                // Update credential status
                if ($result['success']) {
                    $credential->update(['last_success' => now()]);
                } else {
                    $credential->update(['last_failure' => now()]);
                }
            } else {
                $result = $this->testConnectionDirect(
                    $request->consumer_key,
                    $request->consumer_secret,
                    $request->environment
                );
            }
            
            return response()->json($result);
            
        } catch (\Exception $e) {
            Log::error('MPESA Connection Test Error: ' . $e->getMessage(), [
                'data' => $request->except(['consumer_key', 'consumer_secret']),
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
                'error_details' => config('app.debug') ? $e->getTraceAsString() : null
            ], 500);
        }
    }
    
    /**
     * Test STK Push
     */
    public function testStkPush(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'phone' => 'required|string|regex:/^254[17]\d{8}$/',
                'amount' => 'required|numeric|min:10|max:150000',
                'credential_id' => 'required|exists:mpesa_credentials,id',
                'reference' => 'nullable|string|max:255',
                'description' => 'nullable|string|max:255'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            $credential = MpesaCredential::findOrFail($request->credential_id);
            
            // Ensure credential is active
            if (!$credential->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Selected credentials are not active'
                ], 400);
            }
            
            $result = $this->mpesaService->stkPush(
                $credential,
                $request->phone,
                $request->amount,
                $request->reference ?? 'Test Payment ' . now()->format('YmdHis'),
                $request->description ?? 'POS Test Payment'
            );
            
            if ($result['success']) {
                // Log activity
                activity('mpesa-test')
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'phone' => $request->phone,
                        'amount' => $request->amount,
                        'credential' => $credential->name,
                        'reference' => $request->reference
                    ])
                    ->log('STK Push test initiated');
            }
            
            return response()->json($result);
            
        } catch (\Exception $e) {
            Log::error('MPESA STK Push Test Error: ' . $e->getMessage(), [
                'data' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'STK Push failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get Transaction Status
     */
    public function getTransactionStatus(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'transaction_id' => 'required|string',
                'credential_id' => 'required|exists:mpesa_credentials,id'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            $credential = MpesaCredential::findOrFail($request->credential_id);
            $result = $this->mpesaService->checkTransactionStatus($credential, $request->transaction_id);
            
            return response()->json($result);
            
        } catch (\Exception $e) {
            Log::error('MPESA Transaction Status Check Error: ' . $e->getMessage(), [
                'transaction_id' => $request->transaction_id,
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Status check failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Register URLs
     */
    public function registerUrls(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'credential_id' => 'required|exists:mpesa_credentials,id'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            $credential = MpesaCredential::findOrFail($request->credential_id);
            $result = $this->mpesaService->registerUrls($credential);
            
            if ($result['success']) {
                $credential->update(['urls_registered_at' => now()]);
                
                activity('mpesa-urls')
                    ->causedBy(auth()->user())
                    ->withProperties(['credential' => $credential->name])
                    ->log('MPESA URLs registered');
            }
            
            return response()->json($result);
            
        } catch (\Exception $e) {
            Log::error('MPESA Register URLs Error: ' . $e->getMessage(), [
                'credential_id' => $request->credential_id,
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to register URLs: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get Account Balance
     */
    public function getBalance(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'credential_id' => 'required|exists:mpesa_credentials,id'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            $credential = MpesaCredential::findOrFail($request->credential_id);
            $result = $this->mpesaService->getAccountBalance($credential);
            
            return response()->json($result);
            
        } catch (\Exception $e) {
            Log::error('MPESA Balance Check Error: ' . $e->getMessage(), [
                'credential_id' => $request->credential_id,
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to get balance: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Internal Test Connection Method
     */
    private function testConnectionInternal(MpesaCredential $credential)
    {
        try {
            $consumerKey = decrypt($credential->consumer_key);
            $consumerSecret = decrypt($credential->consumer_secret);
            
            return $this->testConnectionDirect($consumerKey, $consumerSecret, $credential->environment);
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Failed to decrypt credentials or connect: ' . $e->getMessage(),
                'credential_name' => $credential->name,
                'environment' => $credential->environment
            ];
        }
    }
    
    /**
     * Direct Test Connection Method
     */
    private function testConnectionDirect($consumerKey, $consumerSecret, $environment)
    {
        try {
            $url = $environment == 'production' 
                ? 'https://api.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials'
                : 'https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials';
            
            $response = Http::withBasicAuth($consumerKey, $consumerSecret)
                ->timeout(30)
                ->retry(3, 1000)
                ->get($url);
            
            if ($response->successful()) {
                $data = $response->json();
                
                return [
                    'success' => true,
                    'message' => 'Connection successful',
                    'data' => [
                        'access_token' => substr($data['access_token'] ?? '', 0, 20) . '...',
                        'expires_in' => $data['expires_in'] ?? 0,
                        'environment' => $environment,
                        'timestamp' => now()->toDateTimeString()
                    ]
                ];
            }
            
            $errorDetails = $response->body();
            $statusCode = $response->status();
            
            return [
                'success' => false,
                'message' => "Connection failed (HTTP {$statusCode})",
                'error_details' => $errorDetails,
                'environment' => $environment
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
                'environment' => $environment
            ];
        }
    }
    
    /**
     * Get Chart Data for Dashboard
     */
    private function getChartData()
    {
        $days = 7;
        $chartData = [];
        
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = Carbon::today()->subDays($i);
            $dateStr = $date->format('Y-m-d');
            
            $transactions = MpesaTransaction::whereDate('created_at', $dateStr)
                ->where('status', 'completed')
                ->get();
            
            $chartData[] = [
                'date' => $date->format('M d'),
                'count' => $transactions->count(),
                'amount' => $transactions->sum('amount'),
                'matched' => $transactions->whereNotNull('sale_id')->count(),
                'unmatched' => $transactions->whereNull('sale_id')->count()
            ];
        }
        
        return $chartData;
    }
    
    /**
     * Clear MPESA Cache
     */
    public function clearCache(Request $request)
    {
        try {
            Cache::forget('mpesa_dashboard_stats');
            Cache::forget('mpesa_active_credential');
            Cache::forget('mpesa_settings');
            Cache::forget('mpesa_transactions_stats');
            
            return response()->json([
                'success' => true,
                'message' => 'MPESA cache cleared successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Clear MPESA Cache Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear cache'
            ], 500);
        }
    }
    
    /**
     * Reset MPESA Settings to Defaults
     */
    public function resetSettings(Request $request)
    {
        DB::beginTransaction();
        
        try {
            // Delete all MPESA settings from database
            Setting::where('key', 'like', 'mpesa.%')->delete();
            
            DB::commit();
            
            // Clear cache
            Cache::forget('mpesa_settings');
            
            return response()->json([
                'success' => true,
                'message' => 'MPESA settings reset to defaults'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Reset MPESA Settings Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to reset settings'
            ], 500);
        }
    }
    
    /**
     * Check Payment (used by POS)
     */
    public function checkPayment(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'phone' => 'required|string',
                'amount' => 'required|numeric|min:1',
                'within_minutes' => 'nullable|integer|min:1|max:60',
                'reference' => 'nullable|string|max:255'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            
            // Normalize phone number
            $phone = $this->normalizePhone($request->phone);
            $withinMinutes = $request->within_minutes ?? 10;
            $since = now()->subMinutes($withinMinutes);
            
            // Check for matching transactions
            $transactions = MpesaTransaction::where('phone', $phone)
                ->where('status', 'completed')
                ->whereNull('sale_id')
                ->where('transaction_date', '>=', $since)
                ->orderBy('transaction_date', 'desc')
                ->get();
            
            // Find exact amount match
            $exactMatch = $transactions->first(function($transaction) use ($request) {
                return abs($transaction->amount - $request->amount) < 0.01;
            });
            
            if ($exactMatch) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'exact',
                    'transaction' => $exactMatch,
                    'message' => 'Exact MPESA payment found!'
                ]);
            }
            
            // Find partial matches
            $partialMatches = $transactions->filter(function($transaction) use ($request) {
                return $transaction->amount >= $request->amount;
            });
            
            if ($partialMatches->count() > 0) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'partial',
                    'transactions' => $partialMatches->values(),
                    'message' => 'Found ' . $partialMatches->count() . ' MPESA transactions with sufficient amount'
                ]);
            }
            
            // Return all transactions for this phone
            if ($transactions->count() > 0) {
                return response()->json([
                    'success' => true,
                    'found' => true,
                    'match_type' => 'multiple',
                    'transactions' => $transactions->values(),
                    'message' => 'Found ' . $transactions->count() . ' MPESA transactions for this phone'
                ]);
            }
            
            return response()->json([
                'success' => true,
                'found' => false,
                'message' => 'No matching MPESA payment found for phone ' . $phone
            ]);
            
        } catch (\Exception $e) {
            Log::error('MPESA Check Payment Error: ' . $e->getMessage(), [
                'data' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to check payment: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Normalize phone number to MPESA format (2547XXXXXXXX)
     */
    private function normalizePhone($phone)
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Convert to 254 format
        if (strlen($phone) === 9 && !str_starts_with($phone, '0')) {
            return '254' . $phone;
        }
        
        if (strlen($phone) === 10 && str_starts_with($phone, '0')) {
            return '254' . substr($phone, 1);
        }
        
        if (strlen($phone) === 12 && str_starts_with($phone, '254')) {
            return $phone;
        }
        
        return $phone;
    }
    
    /**
     * Export MPESA Logs
     */
    public function exportLogs(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'start_date' => 'required|date',
                'end_date' => 'required|date|after_or_equal:start_date',
                'format' => ['required', Rule::in(['csv', 'excel', 'pdf'])]
            ]);
            
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            
            $transactions = MpesaTransaction::with(['sale', 'customer'])
                ->whereBetween('created_at', [$request->start_date, $request->end_date])
                ->orderBy('created_at', 'desc')
                ->get();
            
            $filename = 'mpesa-transactions-' . now()->format('Ymd-His') . '.' . $request->format;
            
            // Generate export based on format
            return $this->generateExport($transactions, $filename, $request->format);
            
        } catch (\Exception $e) {
            Log::error('MPESA Export Error: ' . $e->getMessage());
            
            return back()->with('error', 'Failed to export logs: ' . $e->getMessage());
        }
    }
    
    /**
     * Generate Export File
     */
    private function generateExport($data, $filename, $format)
    {
        // Implementation depends on your export library
        // This is a placeholder for Excel/CSV/PDF generation
        
        if ($format === 'csv') {
            return $this->exportToCsv($data, $filename);
        } elseif ($format === 'excel') {
            return $this->exportToExcel($data, $filename);
        } else {
            return $this->exportToPdf($data, $filename);
        }
    }
    
    /**
     * Export to CSV
     */
    private function exportToCsv($data, $filename)
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($file, [
                'Transaction ID', 'Date', 'Phone', 'Amount', 'Status',
                'Customer Name', 'Reference', 'Matched Sale', 'Created At'
            ]);
            
            // Add data rows
            foreach ($data as $transaction) {
                fputcsv($file, [
                    $transaction->transaction_id,
                    $transaction->transaction_date->format('Y-m-d H:i:s'),
                    $transaction->phone,
                    $transaction->amount,
                    $transaction->status,
                    $transaction->full_name,
                    $transaction->reference,
                    $transaction->sale_id ? 'Sale #' . $transaction->sale_id : 'Not Matched',
                    $transaction->created_at->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    /**
     * Export to Excel
     */
    private function exportToExcel($data, $filename)
    {
        // You can use Laravel Excel package here
        // For now, we'll return CSV as fallback
        return $this->exportToCsv($data, $filename);
    }
    
    /**
     * Export to PDF
     */
    private function exportToPdf($data, $filename)
    {
        // You can use DomPDF or similar package here
        // For now, we'll return CSV as fallback
        return $this->exportToCsv($data, $filename);
    }
}