<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\Sale;
use App\Models\Product;
use App\Models\Customer;
use App\Models\Purchase;
use App\Models\ActivityLog;
use App\Models\User;

class DashboardController extends Controller
{
    /**
     * Display the dashboard.
     */
    public function index()
    {
        return view('dashboard.index');
    }

    /**
     * Get dashboard statistics data.
     */

    public function getDashboardData(Request $request)
{
    try {
        $period = $request->get('period', 30); // Default to 30 days
        
        $startDate = Carbon::now()->subDays($period);
        $previousStartDate = Carbon::now()->subDays($period * 2);
        $previousEndDate = Carbon::now()->subDays($period);
        
        // Total Revenue
        $totalRevenue = Sale::where('created_at', '>=', $startDate)->sum('grand_total');
        $previousRevenue = Sale::whereBetween('created_at', [$previousStartDate, $previousEndDate])->sum('grand_total');
        $revenueChange = $previousRevenue > 0 ? round((($totalRevenue - $previousRevenue) / $previousRevenue) * 100, 1) : 0;
        
        // Total Sales Count
        $totalSales = Sale::where('created_at', '>=', $startDate)->count();
        $previousSales = Sale::whereBetween('created_at', [$previousStartDate, $previousEndDate])->count();
        $salesChange = $previousSales > 0 ? round((($totalSales - $previousSales) / $previousSales) * 100, 1) : 0;
        
        // Total Customers
        $totalCustomers = Customer::count();
        $previousCustomers = Customer::where('created_at', '<', $startDate)->count();
        $customersChange = $previousCustomers > 0 ? round((($totalCustomers - $previousCustomers) / $previousCustomers) * 100, 1) : 0;
        
        // Total Products
        $totalProducts = Product::where('status', 'active')->count();
        $previousProducts = Product::where('status', 'active')->where('created_at', '<', $startDate)->count();
        $productsChange = $previousProducts > 0 ? round((($totalProducts - $previousProducts) / $previousProducts) * 100, 1) : 0;
        
        // Today's Sales
        $todaySales = Sale::whereDate('created_at', Carbon::today())->sum('grand_total');
        $yesterdaySales = Sale::whereDate('created_at', Carbon::yesterday())->sum('grand_total');
        $todaySalesChange = $yesterdaySales > 0 ? round((($todaySales - $yesterdaySales) / $yesterdaySales) * 100, 1) : 0;
        
        // Today's Stats - FIXED COLUMNS
        $todayStats = Sale::whereDate('created_at', Carbon::today())
            ->where('status', 'completed')
            ->selectRaw('COUNT(*) as total_sales')
            ->selectRaw('SUM(grand_total) as total_revenue')
            ->selectRaw('SUM(vat_amount) as total_vat') // Changed to vat_amount (your actual column)
            ->selectRaw('SUM(discount_amount) as total_discount') // Changed to discount_amount
            ->selectRaw('SUM(cash_received) as total_cash_received')
            ->selectRaw('SUM(customer_change) as total_change') // Using customer_change (your actual column)
            ->selectRaw('AVG(grand_total) as average_sale')
            ->first();
        
        // If todayStats is null, create an empty object
        if (!$todayStats) {
            $todayStats = (object) [
                'total_sales' => 0,
                'total_revenue' => 0,
                'total_vat' => 0,
                'total_discount' => 0,
                'total_cash_received' => 0,
                'total_change' => 0,
                'average_sale' => 0
            ];
        }
        
        // Pending Orders
        $pendingOrders = Sale::where('status', 'pending')->count();
        $previousPendingOrders = Sale::where('status', 'pending')->where('created_at', '<', $startDate)->count();
        $pendingOrdersChange = $previousPendingOrders > 0 ? round((($pendingOrders - $previousPendingOrders) / $previousPendingOrders) * 100, 1) : 0;
        
        // Sales overview data for chart - Ensure we have data for all days
        $salesData = [];
        for ($i = $period - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $dateStr = $date->format('M d');
            $salesAmount = Sale::whereDate('created_at', $date)->sum('grand_total');
            
            $salesData[] = [
                'date' => $dateStr,
                'sales' => $salesAmount
            ];
        }
        
        return response()->json([
            'success' => true,
            'total_revenue' => $totalRevenue,
            'revenue_change' => $revenueChange,
            'total_sales' => $totalSales,
            'sales_change' => $salesChange,
            'total_customers' => $totalCustomers,
            'customers_change' => $customersChange,
            'total_products' => $totalProducts,
            'products_change' => $productsChange,
            'today_sales' => $todaySales,
            'today_sales_change' => $todaySalesChange,
            'today_stats' => $todayStats,
            'pending_orders' => $pendingOrders,
            'pending_orders_change' => $pendingOrdersChange,
            'sales_overview' => [
                'labels' => array_column($salesData, 'date'),
                'series' => array_column($salesData, 'sales')
            ]
        ]);
        
    } catch (\Exception $e) {
        \Log::error('Dashboard data error: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);
        return response()->json([
            'success' => false,
            'message' => 'Error loading dashboard data',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Get recent activities.
     */
    public function getRecentActivities()
    {
        try {
            $activities = ActivityLog::with('causer')
                ->latest()
                ->take(10)
                ->get();
                
            if ($activities->isEmpty()) {
                return '<div class="text-center py-4 text-muted">No activities found</div>';
            }
            
            $html = '';
            foreach ($activities as $activity) {
                $html .= '
                <div class="activity-item d-flex align-items-center border-bottom py-2">
                    <div class="activity-icon bg-light-primary text-primary rounded-circle p-2 me-3">
                        <i class="bx bx-' . $this->getActivityIcon($activity->log_name) . '"></i>
                    </div>
                    <div class="flex-grow-1">
                        <h6 class="mb-0">' . $activity->description . '</h6>
                        <small class="text-muted">' . $activity->created_at->diffForHumans() . '</small>
                    </div>
                </div>';
            }
            
            return $html;
            
        } catch (\Exception $e) {
            \Log::error('Recent activities error: ' . $e->getMessage());
            return '<div class="text-center py-4 text-danger">Error loading activities</div>';
        }
    }

    /**
     * Get low stock products.
     */
    public function getLowStockProducts()
    {
        try {
            $lowStockProducts = Product::where('stock', '<=', DB::raw('reorder_point'))
                ->where('status', 'active')
                ->whereNull('deleted_at')
                ->orderBy('stock', 'asc')
                ->limit(5)
                ->get();
            
            if ($lowStockProducts->isEmpty()) {
                return '<div class="text-center py-4 text-muted">All products are well stocked</div>';
            }
            
            $html = '';
            foreach ($lowStockProducts as $product) {
                $stockPercentage = $product->reorder_point > 0 ? ($product->stock / $product->reorder_point) * 100 : 0;
                $progressClass = $stockPercentage <= 25 ? 'bg-danger' : 
                                ($stockPercentage <= 50 ? 'bg-warning' : 'bg-info');
                
                $html .= '
                <div class="d-flex align-items-center mb-3">
                    <div class="flex-grow-1">
                        <h6 class="mb-1">' . $product->name . '</h6>
                        <div class="d-flex align-items-center">
                            <div class="progress flex-grow-1 me-2" style="height: 6px;">
                                <div class="progress-bar ' . $progressClass . '" role="progressbar" 
                                     style="width: ' . min($stockPercentage, 100) . '%"></div>
                            </div>
                            <span class="text-muted">' . $product->stock . ' left</span>
                        </div>
                    </div>
                    <div class="text-end">
                        <small class="text-muted d-block">Min: ' . $product->reorder_point . '</small>
                        <a href="' . route('products.edit', $product->id) . '" class="btn btn-sm btn-outline-primary">Restock</a>
                    </div>
                </div>';
            }
            
            return $html;
            
        } catch (\Exception $e) {
            \Log::error('Low stock products error: ' . $e->getMessage());
            return '<div class="text-center py-4 text-danger">Error loading low stock products</div>';
        }
    }

    /**
     * Get recent sales.
     */
    public function getRecentSales()
    {
        try {
            $recentSales = Sale::with('customer')
                ->latest()
                ->take(8)
                ->get();
                
            if ($recentSales->isEmpty()) {
                return '<tr><td colspan="5" class="text-center py-4 text-muted">No recent sales</td></tr>';
            }
            
            $html = '';
            foreach ($recentSales as $sale) {
                $statusClass = $this->getStatusClass($sale->status);
                
                $html .= '
                <tr>
                    <td>
                        <a href="' . route('sales.show', $sale->id) . '" class="text-primary">
                            ' . ($sale->unified_number ?? $sale->invoice_no ?? '#' . str_pad($sale->id, 6, '0', STR_PAD_LEFT)) . '
                        </a>
                    </td>
                    <td>' . ($sale->customer ? $sale->customer->name : 'Walk-in Customer') . '</td>
                    <td>' . $sale->created_at->format('M d, Y h:i A') . '</td>
                    <td class="fw-bold">KES ' . number_format($sale->grand_total, 2) . '</td>
                    <td>
                        <span class="badge bg-' . $statusClass . '">' . ucfirst($sale->status) . '</span>
                    </td>
                </tr>';
            }
            
            return $html;
            
        } catch (\Exception $e) {
            \Log::error('Recent sales error: ' . $e->getMessage());
            return '<tr><td colspan="5" class="text-center py-4 text-danger">Error loading recent sales</td></tr>';
        }
    }

    /**
     * Get top products.
     */
    public function getTopProducts()
    {
        try {
            // Check if sale_items table exists and has qty column
            $tableExists = DB::select("SHOW TABLES LIKE 'sale_items'");
            
            if (empty($tableExists)) {
                return '<div class="text-center py-4 text-muted">Sales data not available</div>';
            }
            
            // Check if qty column exists
            $columns = DB::select("SHOW COLUMNS FROM sale_items LIKE 'qty'");
            
            if (empty($columns)) {
                // Try with quantity column instead
                $topProducts = DB::table('sale_items')
                    ->select('product_id', DB::raw('SUM(quantity) as total_sold'))
                    ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
                    ->where('sales.status', 'completed')
                    ->groupBy('product_id')
                    ->orderByDesc('total_sold')
                    ->take(5)
                    ->get();
            } else {
                // Use qty column
                $topProducts = DB::table('sale_items')
                    ->select('product_id', DB::raw('SUM(qty) as total_sold'))
                    ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
                    ->where('sales.status', 'completed')
                    ->groupBy('product_id')
                    ->orderByDesc('total_sold')
                    ->take(5)
                    ->get();
            }
                
            if ($topProducts->isEmpty()) {
                return '<div class="text-center py-4 text-muted">No sales data available</div>';
            }
            
            $html = '';
            $counter = 1;
            foreach ($topProducts as $item) {
                $product = Product::find($item->product_id);
                if (!$product) continue;
                
                $imageSrc = $product->primary_image ? 
                    asset('storage/' . $product->primary_image) : 
                    asset('assets/images/no-image.png');
                
                $html .= '
                <div class="row border mx-0 mb-3 py-2 radius-10">
                    <div class="col-sm-6">
                        <div class="d-flex align-items-center">
                            <div class="product-img">
                                <img src="' . $imageSrc . '" alt="' . $product->name . '" 
                                     style="width: 40px; height: 40px; object-fit: cover; border-radius: 4px;">
                            </div>
                            <div class="ms-2">
                                <h6 class="mb-1">' . $product->name . '</h6>
                                <p class="mb-0 text-muted">#' . $counter . ' Best Seller</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-sm">
                        <h6 class="mb-1">' . $item->total_sold . ' Sold</h6>
                        <p class="mb-0">Stock: ' . $product->stock . '</p>
                    </div>
                </div>';
                
                $counter++;
            }
            
            return $html;
            
        } catch (\Exception $e) {
            \Log::error('Top products error: ' . $e->getMessage());
            return '<div class="text-center py-4 text-danger">Error loading top products</div>';
        }
    }

    /**
     * Get recent orders for dashboard.
     */
    public function getRecentOrders()
    {
        try {
            $recentOrders = Sale::with('customer')
                ->latest()
                ->take(10)
                ->get();
                
            return response()->json([
                'success' => true,
                'orders' => $recentOrders->map(function($order) {
                    return [
                        'id' => $order->id,
                        'order_id' => $order->invoice_no ?? $order->unified_number ?? '#' . str_pad($order->id, 6, '0', STR_PAD_LEFT),
                        'product_name' => 'Multiple Products',
                        'customer_name' => $order->customer ? $order->customer->name : 'Walk-in Customer',
                        'order_date' => $order->created_at->format('M d, Y'),
                        'total' => $order->grand_total,
                        'status' => $order->status
                    ];
                })
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Recent orders error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading recent orders',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get monthly metrics for dashboard.
     */
    public function getMonthlyMetrics()
    {
        try {
            $currentMonthStart = Carbon::now()->startOfMonth();
            $lastMonthStart = Carbon::now()->subMonth()->startOfMonth();
            $lastMonthEnd = Carbon::now()->subMonth()->endOfMonth();
            
            // Current month revenue
            $monthRevenue = Sale::where('created_at', '>=', $currentMonthStart)->sum('grand_total');
            $lastMonthRevenue = Sale::whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])->sum('grand_total');
            $monthRevenueChange = $lastMonthRevenue > 0 ? 
                round((($monthRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100, 1) : 0;
            
            // Current month sales count
            $monthSales = Sale::where('created_at', '>=', $currentMonthStart)->count();
            $lastMonthSales = Sale::whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])->count();
            $monthSalesChange = $lastMonthSales > 0 ? 
                round((($monthSales - $lastMonthSales) / $lastMonthSales) * 100, 1) : 0;
            
            // Current month new customers
            $monthCustomers = Customer::where('created_at', '>=', $currentMonthStart)->count();
            $lastMonthCustomers = Customer::whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])->count();
            $monthCustomersChange = $lastMonthCustomers > 0 ? 
                round((($monthCustomers - $lastMonthCustomers) / $lastMonthCustomers) * 100, 1) : 0;
            
            return response()->json([
                'success' => true,
                'month_revenue' => $monthRevenue,
                'month_revenue_change' => $monthRevenueChange,
                'month_sales' => $monthSales,
                'month_sales_change' => $monthSalesChange,
                'month_customers' => $monthCustomers,
                'month_customers_change' => $monthCustomersChange
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Monthly metrics error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading monthly metrics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * API: Get today's stats (for other API calls)
     */
public function apiGetTodayStats()
{
    try {
        // FIXED COLUMNS
        $todayStats = Sale::whereDate('created_at', Carbon::today())
            ->where('status', 'completed')
            ->selectRaw('COUNT(*) as total_sales')
            ->selectRaw('SUM(grand_total) as total_revenue')
            ->selectRaw('SUM(vat_amount) as total_vat') // Changed to vat_amount
            ->selectRaw('SUM(discount_amount) as total_discount') // Changed to discount_amount
            ->selectRaw('SUM(cash_received) as total_cash_received')
            ->selectRaw('SUM(customer_change) as total_change') // Changed to customer_change
            ->selectRaw('AVG(grand_total) as average_sale')
            ->first();
        
        // If no stats, return empty object
        if (!$todayStats) {
            $todayStats = (object) [
                'total_sales' => 0,
                'total_revenue' => 0,
                'total_vat' => 0,
                'total_discount' => 0,
                'total_cash_received' => 0,
                'total_change' => 0,
                'average_sale' => 0
            ];
        }
        
        return response()->json([
            'success' => true,
            'data' => $todayStats
        ]);
        
    } catch (\Exception $e) {
        \Log::error('Today stats error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Helper function to get activity icon.
     */
    private function getActivityIcon($logName)
    {
        $icons = [
            'default' => 'bell',
            'sale' => 'cart',
            'purchase' => 'package',
            'product' => 'box',
            'customer' => 'user',
            'user' => 'user-circle',
            'category' => 'category'
        ];
        
        return $icons[$logName] ?? 'bell';
    }

    /**
     * Helper function to get status class.
     */
    private function getStatusClass($status)
    {
        $classes = [
            'pending' => 'warning',
            'completed' => 'success',
            'cancelled' => 'danger',
            'refunded' => 'info',
            'paid' => 'success',
            'unpaid' => 'warning',
            'processing' => 'primary',
            'delivered' => 'success',
            'shipped' => 'info'
        ];
        
        return $classes[strtolower($status)] ?? 'secondary';
    }
}