<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Customer::query();
            
            // Search functionality
            if ($request->has('search')) {
                $search = $request->input('search');
                $query->search($search);
            }
            
            // Filter by customer type
            if ($request->has('type')) {
                switch ($request->type) {
                    case 'walk-in':
                        $query->walkIn();
                        break;
                    case 'regular':
                        $query->regular();
                        break;
                    case 'business':
                        $query->business();
                        break;
                    case 'active':
                        $query->active();
                        break;
                    case 'with-balance':
                        $query->where('credit_balance', '>', 0);
                        break;
                    case 'vat-registered':
                        $query->vatRegistered();
                        break;
                    case 'non-vat':
                        $query->nonVat();
                        break;
                    case 'has-credit':
                        $query->hasCredit();
                        break;
                }
            }
            
            // Filter by VAT status
            if ($request->has('vat_status')) {
                $query->byVatStatus($request->vat_status);
            }
            
            // Sorting
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            
            $sortableColumns = ['name', 'total_spent', 'taxable_sales', 'vat_collected', 
                               'total_orders', 'created_at', 'last_purchase_date', 
                               'credit_balance', 'loyalty_points'];
            
            if (in_array($sortBy, $sortableColumns)) {
                $query->orderBy($sortBy, $sortOrder);
            } else {
                $query->latest();
            }
            
            $customers = $query->withCount('sales')->paginate(20)->withQueryString();
            
            // Statistics for dashboard
            $stats = [
                'total' => Customer::count(),
                'walk_in' => Customer::walkIn()->count(),
                'regular' => Customer::regular()->count(),
                'business' => Customer::business()->count(),
                'active' => Customer::active()->count(),
                'vat_registered' => Customer::vatRegistered()->count(),
                'total_revenue' => Customer::sum('total_spent'),
                'total_taxable_sales' => Customer::sum('taxable_sales'),
                'total_vat_collected' => Customer::sum('vat_collected'),
                'avg_order_value' => Customer::where('total_orders', '>', 0)->avg('total_spent'),
                'total_credit_balance' => Customer::sum('credit_balance'),
                'total_loyalty_points' => Customer::sum('loyalty_points'),
            ];
            
            return view('customers.index', compact('customers', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('Customer index error: ' . $e->getMessage());
            return redirect()->route('dashboard')
                ->with('error', 'Failed to load customers. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Default values for new customer
        $defaultCustomer = new Customer([
            'customer_type' => 'regular',
            'status' => 'active',
            'is_vat_registered' => false,
            'discount_percent' => 0,
            'credit_limit' => 0,
            'credit_balance' => 0,
            'balance' => 0,
            'total_spent' => 0,
            'taxable_sales' => 0,
            'vat_collected' => 0,
            'total_orders' => 0,
            'loyalty_points' => 0,
        ]);
        
        return view('customers.create', compact('defaultCustomer'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            
            [$rules, $messages] = Customer::validateData($request->all());
            
            $validated = $request->validate($rules, $messages);
            
            // Set defaults for walk-in customers
            if ($validated['customer_type'] === 'walk-in') {
                $validated['name'] = $validated['name'] ?: 'Walk-in Customer';
                $validated['is_vat_registered'] = false;
            }
            
            // Set default VAT registration status based on VAT number
            if (!empty($validated['vat_number'])) {
                $validated['is_vat_registered'] = true;
            } elseif (!isset($validated['is_vat_registered'])) {
                $validated['is_vat_registered'] = false;
            }
            
            // Set default values for statistics
            $validated['total_spent'] = $validated['total_spent'] ?? 0;
            $validated['taxable_sales'] = $validated['taxable_sales'] ?? 0;
            $validated['vat_collected'] = $validated['vat_collected'] ?? 0;
            $validated['total_orders'] = 0;
            $validated['loyalty_points'] = 0;
            $validated['credit_balance'] = $validated['credit_balance'] ?? 0;
            $validated['balance'] = $validated['balance'] ?? 0;
            $validated['discount_percent'] = $validated['discount_percent'] ?? 0;
            
            // Set created_by if user is authenticated
            if (auth()->check()) {
                $validated['created_by'] = auth()->id();
            }
            
            $customer = Customer::create($validated);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('created customer');
            
            DB::commit();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'customer' => $customer,
                    'message' => 'Customer created successfully.'
                ]);
            }
            
            return redirect()->route('customers.show', $customer)
                ->with('success', 'Customer created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $e->errors()
                ], 422);
            }
            
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Customer store error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create customer. Please try again.'
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', 'Failed to create customer. Please try again.')
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
public function show(Customer $customer)
{
    try {
        \Log::info('Showing customer', ['customer_id' => $customer->id, 'customer_name' => $customer->name]);
        
        // Load relationships efficiently
        $customer->loadCount(['sales', 'payments' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        \Log::info('Customer loaded with counts', [
            'sales_count' => $customer->sales_count,
            'payments_count' => $customer->payments_count,
        ]);

        // Get recent sales with pagination
        $sales = $customer->sales()
            ->with(['items.product', 'payments'])
            ->latest()
            ->paginate(10);
            
        \Log::info('Sales loaded', ['sales_count' => $sales->count()]);

        // Get payment history
        $payments = $customer->payments()
            ->latest()
            ->limit(10)
            ->get();
            
        \Log::info('Payments loaded', ['payments_count' => $payments->count()]);

        // Calculate sales statistics safely
        $salesStats = [
            'total_sales' => (int) ($customer->sales_count ?? 0),
            'total_spent' => (float) ($customer->total_spent ?? 0),
            'taxable_sales' => (float) ($customer->taxable_sales ?? 0),
            'vat_collected' => (float) ($customer->vat_collected ?? 0),
            'non_taxable_sales' => ($customer->total_spent ?? 0) - ($customer->taxable_sales ?? 0),
            'vat_rate_effective' => ($customer->taxable_sales ?? 0) > 0 
                ? round((($customer->vat_collected ?? 0) / ($customer->taxable_sales ?? 1)) * 100, 2) 
                : 0,
            'avg_order_value' => ($customer->total_orders ?? 0) > 0 
                ? round(($customer->total_spent ?? 0) / ($customer->total_orders ?? 1), 2) 
                : 0,
            'last_purchase' => $customer->last_purchase_date ? $customer->last_purchase_date->format('M d, Y') : 'Never',
            'credit_balance' => (float) ($customer->credit_balance ?? 0),
            'available_credit' => (float) ($customer->available_credit ?? 0),
            'balance' => (float) ($customer->balance ?? 0),
            'has_credit_available' => (bool) ($customer->has_credit_available ?? false),
            'loyalty_points' => (int) ($customer->loyalty_points ?? 0),
            'discount_percent' => (float) ($customer->discount_percent ?? 0),
            'customer_level' => $customer->customer_level ?? 'bronze',
            'days_since_last_purchase' => $customer->days_since_last_purchase,
            'is_active_customer' => $customer->is_active_customer ?? false,
        ];
        
        \Log::info('Sales stats calculated', $salesStats);

        // Get VAT summary safely
        try {
            $vatSummary = $customer->getVatSummary();
        } catch (\Exception $e) {
            \Log::warning('Failed to get VAT summary', ['error' => $e->getMessage()]);
            $vatSummary = [
                'customer_name' => $customer->name,
                'vat_number' => $customer->vat_number,
                'is_vat_registered' => (bool) ($customer->is_vat_registered ?? false),
                'taxable_sales' => (float) ($customer->taxable_sales ?? 0),
                'vat_collected' => (float) ($customer->vat_collected ?? 0),
                'total_sales' => (float) ($customer->total_spent ?? 0),
                'non_taxable_sales' => ($customer->total_spent ?? 0) - ($customer->taxable_sales ?? 0),
                'vat_rate_effective' => ($customer->taxable_sales ?? 0) > 0 
                    ? round((($customer->vat_collected ?? 0) / ($customer->taxable_sales ?? 1)) * 100, 2) 
                    : 0,
            ];
        }

        // Get purchase history safely
        try {
            $purchaseHistory = $customer->getPurchaseHistory(5);
        } catch (\Exception $e) {
            \Log::warning('Failed to get purchase history', ['error' => $e->getMessage()]);
            $purchaseHistory = collect([]);
        }

        // Get outstanding amount safely
        try {
            $outstandingAmount = $customer->getOutstandingAmount();
        } catch (\Exception $e) {
            \Log::warning('Failed to get outstanding amount', ['error' => $e->getMessage()]);
            $outstandingAmount = 0;
        }

        \Log::info('All data loaded successfully for customer', ['customer_id' => $customer->id]);

        return view('customers.show', compact(
            'customer', 
            'sales', 
            'payments',
            'salesStats', 
            'vatSummary',
            'purchaseHistory',
            'outstandingAmount'
        ));
        
    } catch (\Exception $e) {
        \Log::error('Customer show error: ' . $e->getMessage(), [
            'customer_id' => $customer->id ?? 'unknown',
            'trace' => $e->getTraceAsString(),
            'customer_data' => $customer->toArray() ?? 'no customer data'
        ]);
        
        return redirect()->route('customers.index')
            ->with('error', 'Failed to load customer details: ' . ($e->getMessage() ?? 'Unknown error'));
    }
}
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Customer $customer)
    {
        return view('customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Customer $customer)
    {
        try {
            DB::beginTransaction();
            
            [$rules, $messages] = Customer::validateData($request->all(), $customer->id);
            
            $validated = $request->validate($rules, $messages);
            
            // Set VAT registration status based on VAT number
            if (!empty($validated['vat_number'])) {
                $validated['is_vat_registered'] = true;
            } elseif (!isset($validated['is_vat_registered'])) {
                $validated['is_vat_registered'] = false;
            }
            
            // Preserve statistics fields that shouldn't be changed directly
            $protectedFields = ['total_spent', 'taxable_sales', 'vat_collected', 'total_orders', 
                              'loyalty_points', 'last_purchase_date'];
            foreach ($protectedFields as $field) {
                if (isset($validated[$field])) {
                    unset($validated[$field]);
                }
            }
            
            $oldData = $customer->toArray();
            
            // Set updated_by if user is authenticated
            if (auth()->check()) {
                $validated['updated_by'] = auth()->id();
            }
            
            $customer->update($validated);
            
            // Log activity with changes
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->withProperties([
                    'old' => $oldData,
                    'new' => $validated
                ])
                ->log('updated customer');
            
            DB::commit();
            
            return redirect()->route('customers.show', $customer)
                ->with('success', 'Customer updated successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Customer update error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Failed to update customer. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Customer $customer)
    {
        try {
            DB::beginTransaction();
            
            // Check if customer has any sales
            if ($customer->sales()->exists()) {
                return redirect()->route('customers.index')
                    ->with('error', 'Cannot delete customer with existing sales records.');
            }
            
            // Log activity before deletion
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('deleted customer');
            
            $customer->delete();
            
            DB::commit();
            
            return redirect()->route('customers.index')
                ->with('success', 'Customer deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Customer destroy error: ' . $e->getMessage());
            return redirect()->route('customers.index')
                ->with('error', 'Failed to delete customer. Please try again.');
        }
    }

    /**
     * Search customers for AJAX requests.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            // If search is empty, return walk-in customer option
            if (empty($search)) {
                return response()->json([
                    [
                        'id' => 'walk-in',
                        'name' => 'Walk-in Customer',
                        'email' => null,
                        'phone' => null,
                        'customer_code' => 'WALK-IN',
                        'is_walk_in' => true,
                        'is_vat_registered' => false,
                        'credit_limit' => 0,
                        'credit_balance' => 0,
                        'available_credit' => 0,
                        'vat_number' => null,
                    ]
                ]);
            }
            
            $customers = Customer::where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('email', 'like', "%{$search}%")
                          ->orWhere('phone', 'like', "%{$search}%")
                          ->orWhere('company_name', 'like', "%{$search}%")
                          ->orWhere('vat_number', 'like', "%{$search}%")
                          ->orWhere('customer_code', 'like', "%{$search}%");
                })
                ->where('status', 'active')
                ->limit(10)
                ->get()
                ->map(function($customer) {
                    return [
                        'id' => $customer->id,
                        'name' => $customer->name,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'customer_type' => $customer->customer_type,
                        'company_name' => $customer->company_name,
                        'customer_code' => $customer->customer_code,
                        'credit_balance' => (float) $customer->credit_balance,
                        'credit_limit' => (float) $customer->credit_limit,
                        'available_credit' => $customer->available_credit,
                        'vat_number' => $customer->vat_number,
                        'is_vat_registered' => (bool) $customer->is_vat_registered,
                        'total_spent' => (float) $customer->total_spent,
                        'loyalty_points' => $customer->loyalty_points,
                        'discount_percent' => (float) $customer->discount_percent,
                        'is_walk_in' => $customer->customer_type === 'walk-in'
                    ];
                });
            
            // Add walk-in option at the beginning
            $results = $customers->toArray();
            array_unshift($results, [
                'id' => 'walk-in',
                'name' => 'Walk-in Customer',
                'email' => null,
                'phone' => null,
                'customer_type' => 'walk-in',
                'company_name' => null,
                'customer_code' => 'WALK-IN',
                'credit_balance' => 0,
                'credit_limit' => 0,
                'available_credit' => 0,
                'vat_number' => null,
                'is_vat_registered' => false,
                'total_spent' => 0,
                'loyalty_points' => 0,
                'discount_percent' => 0,
                'is_walk_in' => true
            ]);
            
            return response()->json($results);
            
        } catch (\Exception $e) {
            Log::error('Customer search error: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /**
     * Quick create customer for POS.
     */
    public function quickCreate(Request $request)
    {
        try {
            [$rules, $messages] = Customer::validateData($request->all());
            
            $validated = $request->validate($rules, $messages);
            
            // Set VAT registration based on VAT number
            if (!empty($validated['vat_number'])) {
                $validated['is_vat_registered'] = true;
            } elseif (!isset($validated['is_vat_registered'])) {
                $validated['is_vat_registered'] = false;
            }
            
            // Set default values
            $validated['status'] = 'active';
            $validated['total_spent'] = 0;
            $validated['taxable_sales'] = 0;
            $validated['vat_collected'] = 0;
            $validated['total_orders'] = 0;
            $validated['loyalty_points'] = 0;
            $validated['credit_balance'] = $validated['credit_balance'] ?? 0;
            $validated['balance'] = 0;
            $validated['discount_percent'] = $validated['discount_percent'] ?? 0;
            
            // Set created_by if user is authenticated
            if (auth()->check()) {
                $validated['created_by'] = auth()->id();
            }
            
            $customer = Customer::create($validated);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('quick created customer via POS');
            
            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'customer_type' => $customer->customer_type,
                    'customer_code' => $customer->customer_code,
                    'company_name' => $customer->company_name,
                    'vat_number' => $customer->vat_number,
                    'is_vat_registered' => (bool) $customer->is_vat_registered,
                    'credit_limit' => (float) $customer->credit_limit,
                    'credit_balance' => (float) $customer->credit_balance,
                    'available_credit' => $customer->available_credit,
                    'discount_percent' => (float) $customer->discount_percent,
                    'loyalty_points' => $customer->loyalty_points,
                ],
                'message' => 'Customer created successfully.'
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Customer quick create error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create customer.'
            ], 500);
        }
    }

    /**
     * Get or create walk-in customer.
     */
    public function getWalkInCustomer()
    {
        try {
            // Find or create a generic walk-in customer
            $customer = Customer::firstOrCreate(
                ['customer_code' => 'WALK-IN'],
                [
                    'name' => 'Walk-in Customer',
                    'customer_type' => 'walk-in',
                    'is_vat_registered' => false,
                    'status' => 'active',
                    'credit_limit' => 0,
                    'credit_balance' => 0,
                    'total_spent' => 0,
                    'total_orders' => 0,
                    'loyalty_points' => 0,
                    'discount_percent' => 0,
                ]
            );
            
            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'customer_code' => $customer->customer_code,
                    'is_vat_registered' => (bool) $customer->is_vat_registered,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get walk-in customer error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get walk-in customer.'
            ], 500);
        }
    }

    /**
     * Export customers to CSV with VAT information.
     */
    public function export(Request $request)
    {
        try {
            $query = Customer::query();
            
            if ($request->has('type')) {
                switch ($request->type) {
                    case 'walk-in':
                        $query->walkIn();
                        break;
                    case 'regular':
                        $query->regular();
                        break;
                    case 'business':
                        $query->business();
                        break;
                    case 'vat-registered':
                        $query->vatRegistered();
                        break;
                }
            }
            
            if ($request->has('vat_status')) {
                $query->byVatStatus($request->vat_status);
            }
            
            $customers = $query->get();
            
            $fileName = 'customers-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($customers) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Customer ID', 'Code', 'Name', 'Type', 'Email', 'Phone', 'Company', 
                    'Address', 'VAT Number', 'VAT Registered', 'Contact Person',
                    'Tax ID', 'Website', 'Status', 'Credit Limit', 'Credit Balance',
                    'Available Credit', 'Balance', 'Total Spent', 'Taxable Sales',
                    'VAT Collected', 'Non-Taxable Sales', 'Total Orders', 
                    'Average Order Value', 'Last Purchase', 'Loyalty Points',
                    'Discount %', 'Customer Level', 'Days Since Last Purchase',
                    'Created At', 'Updated At'
                ]);
                
                // Data
                foreach ($customers as $customer) {
                    fputcsv($file, [
                        $customer->id,
                        $customer->customer_code,
                        $customer->name,
                        $customer->customer_type,
                        $customer->email ?? '',
                        $customer->phone ?? '',
                        $customer->company_name ?? '',
                        $customer->address ?? '',
                        $customer->vat_number ?? '',
                        $customer->is_vat_registered ? 'Yes' : 'No',
                        $customer->contact_person ?? '',
                        $customer->tax_id ?? '',
                        $customer->website ?? '',
                        $customer->status,
                        $customer->credit_limit,
                        $customer->credit_balance,
                        $customer->available_credit,
                        $customer->balance,
                        $customer->total_spent,
                        $customer->taxable_sales,
                        $customer->vat_collected,
                        $customer->non_taxable_sales,
                        $customer->total_orders,
                        $customer->average_order_value,
                        $customer->last_purchase_date ? $customer->last_purchase_date->format('Y-m-d H:i:s') : 'Never',
                        $customer->loyalty_points,
                        $customer->discount_percent,
                        $customer->customer_level,
                        $customer->days_since_last_purchase,
                        $customer->created_at->format('Y-m-d H:i:s'),
                        $customer->updated_at->format('Y-m-d H:i:s'),
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Customer export error: ' . $e->getMessage());
            return redirect()->route('customers.index')
                ->with('error', 'Failed to export customers.');
        }
    }

    /**
     * Export VAT report for customers.
     */
    public function exportVatReport(Request $request)
    {
        try {
            $query = Customer::vatRegistered();
            
            if ($request->has('start_date') && $request->has('end_date')) {
                $query->whereBetween('last_purchase_date', [$request->start_date, $request->end_date]);
            }
            
            $customers = $query->get();
            
            $fileName = 'vat-customers-report-' . date('Y-m-d') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=$fileName",
            ];
            
            $callback = function() use ($customers) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'Customer Name', 'VAT Number', 'Company Name', 'Phone', 'Email',
                    'Contact Person', 'Address', 'Tax ID', 'Website',
                    'Total Sales', 'Taxable Sales', 'VAT Collected', 
                    'Non-Taxable Sales', 'Effective VAT Rate', 'Total Orders',
                    'Average Order Value', 'Last Purchase Date', 'Customer Level',
                    'Days Since Last Purchase', 'Status', 'Created At'
                ]);
                
                // Data
                foreach ($customers as $customer) {
                    fputcsv($file, [
                        $customer->name,
                        $customer->vat_number ?? '',
                        $customer->company_name ?? '',
                        $customer->phone ?? '',
                        $customer->email ?? '',
                        $customer->contact_person ?? '',
                        $customer->address ?? '',
                        $customer->tax_id ?? '',
                        $customer->website ?? '',
                        $customer->total_spent,
                        $customer->taxable_sales,
                        $customer->vat_collected,
                        $customer->non_taxable_sales,
                        $customer->vat_rate_effective . '%',
                        $customer->total_orders,
                        $customer->average_order_value,
                        $customer->last_purchase_date ? $customer->last_purchase_date->format('Y-m-d') : 'Never',
                        $customer->customer_level,
                        $customer->days_since_last_purchase,
                        $customer->status,
                        $customer->created_at->format('Y-m-d'),
                    ]);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('VAT report export error: ' . $e->getMessage());
            return redirect()->route('customers.index')
                ->with('error', 'Failed to export VAT report.');
        }
    }

    /**
     * Bulk delete customers.
     */
    public function bulkDelete(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $validated = $request->validate([
                'ids' => 'required|array',
                'ids.*' => 'exists:customers,id',
            ]);
            
            $customers = Customer::whereIn('id', $validated['ids'])->get();
            
            // Check if any customer has sales
            $customersWithSales = $customers->filter(function($customer) {
                return $customer->sales()->exists();
            });
            
            if ($customersWithSales->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete customers with existing sales records.',
                ], 422);
            }
            
            // Log activity for each customer
            foreach ($customers as $customer) {
                activity()
                    ->causedBy(auth()->user())
                    ->performedOn($customer)
                    ->log('bulk deleted customer');
            }
            
            Customer::whereIn('id', $validated['ids'])->delete();
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Selected customers deleted successfully.',
                'count' => count($validated['ids'])
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Customer bulk delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete customers. Please try again.'
            ], 500);
        }
    }

    /**
     * Get customer statistics for dashboard.
     */
    public function statistics()
    {
        try {
            $stats = [
                'total' => Customer::count(),
                'walk_in' => Customer::walkIn()->count(),
                'regular' => Customer::regular()->count(),
                'business' => Customer::business()->count(),
                'active' => Customer::active()->count(),
                'vat_registered' => Customer::vatRegistered()->count(),
                'total_revenue' => Customer::sum('total_spent'),
                'total_taxable_sales' => Customer::sum('taxable_sales'),
                'total_vat_collected' => Customer::sum('vat_collected'),
                'total_credit_balance' => Customer::sum('credit_balance'),
                'total_loyalty_points' => Customer::sum('loyalty_points'),
                'avg_order_value' => Customer::where('total_orders', '>', 0)->avg('total_spent'),
                'top_customers' => Customer::orderBy('total_spent', 'desc')->limit(5)->get(),
                'top_vat_customers' => Customer::vatRegistered()->orderBy('vat_collected', 'desc')->limit(5)->get(),
            ];
            
            return response()->json($stats);
            
        } catch (\Exception $e) {
            Log::error('Customer statistics error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load statistics'], 500);
        }
    }

    /**
     * Update customer balance (for credit sales).
     */
    public function updateBalance(Request $request, Customer $customer)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'type' => 'required|in:payment,sale', // payment reduces balance, sale increases
                'notes' => 'nullable|string',
            ]);
            
            DB::beginTransaction();
            
            $success = false;
            $message = '';
            
            if ($validated['type'] === 'sale') {
                // Check if customer can make credit purchase
                $creditCheck = $customer->canPurchaseOnCredit($validated['amount']);
                if (!$creditCheck['allowed']) {
                    return response()->json([
                        'success' => false,
                        'message' => $creditCheck['reason']
                    ], 422);
                }
                $success = $customer->addCredit($validated['amount']);
                $message = 'Credit sale recorded.';
            } else {
                $success = $customer->reduceCredit($validated['amount']);
                $message = 'Payment recorded.';
            }
            
            if (!$success) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update credit balance. Check credit limit.'
                ], 422);
            }
            
            // Update balance field as well
            if ($validated['type'] === 'sale') {
                $customer->increment('balance', $validated['amount']);
            } else {
                $customer->decrement('balance', $validated['amount']);
                if ($customer->balance < 0) {
                    $customer->balance = 0;
                    $customer->save();
                }
            }
            
            // Log the balance update
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->withProperties([
                    'amount' => $validated['amount'],
                    'type' => $validated['type'],
                    'notes' => $validated['notes'] ?? null,
                    'new_credit_balance' => $customer->credit_balance,
                    'new_balance' => $customer->balance,
                    'available_credit' => $customer->available_credit
                ])
                ->log('updated customer balance');
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'credit_balance' => (float) $customer->credit_balance,
                'balance' => (float) $customer->balance,
                'available_credit' => (float) $customer->available_credit,
                'message' => $message
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update customer balance error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update balance.'
            ], 500);
        }
    }

    /**
     * Update customer VAT statistics.
     */
    public function updateVatStats(Request $request, Customer $customer)
    {
        try {
            $validated = $request->validate([
                'taxable_amount' => 'required|numeric|min:0',
                'vat_amount' => 'required|numeric|min:0',
            ]);
            
            DB::beginTransaction();
            
            $customer->updateVatStats($validated['taxable_amount'], $validated['vat_amount']);
            
            // Log the VAT update
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->withProperties([
                    'taxable_amount' => $validated['taxable_amount'],
                    'vat_amount' => $validated['vat_amount'],
                    'new_taxable_sales' => $customer->taxable_sales,
                    'new_vat_collected' => $customer->vat_collected
                ])
                ->log('updated customer VAT statistics');
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'taxable_sales' => (float) $customer->taxable_sales,
                'vat_collected' => (float) $customer->vat_collected,
                'vat_rate_effective' => $customer->vat_rate_effective,
                'message' => 'VAT statistics updated successfully.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update customer VAT stats error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update VAT statistics.'
            ], 500);
        }
    }

    /**
     * API: Get all customers with pagination
     */
    public function apiGetCustomers(Request $request): JsonResponse
    {
        try {
            \Log::info('========== API GET CUSTOMERS START ==========');
            \Log::info('Request params:', $request->all());
            \Log::info('User authenticated:', ['auth' => auth()->check()]);
            \Log::info('User ID:', ['id' => auth()->id()]);
            
            $search = $request->input('search', '');
            $type = $request->input('type', 'all');
            $page = $request->input('page', 1);
            $perPage = $request->input('per_page', 50);
            
            \Log::info('Query parameters:', [
                'search' => $search,
                'type' => $type,
                'page' => $page,
                'perPage' => $perPage
            ]);
            
            // Test database connection first
            try {
                $testQuery = \DB::table('customers')->count();
                \Log::info('Database connection test:', ['customer_count' => $testQuery]);
            } catch (\Exception $dbError) {
                \Log::error('Database connection failed:', ['error' => $dbError->getMessage()]);
                throw $dbError;
            }
            
            // Use DB query directly for safety
            $query = \DB::table('customers')->where('status', 'active');
            
            if (!empty($search)) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('customer_code', 'like', "%{$search}%")
                      ->orWhere('vat_number', 'like', "%{$search}%");
                });
            }
            
            if ($type !== 'all') {
                $query->where('customer_type', $type);
            }
            
            \Log::info('SQL Query:', ['sql' => $query->toSql(), 'bindings' => $query->getBindings()]);
            
            $total = $query->count();
            $offset = ($page - 1) * $perPage;
            $lastPage = ceil($total / $perPage);
            
            \Log::info('Pagination info:', [
                'total' => $total,
                'offset' => $offset,
                'lastPage' => $lastPage
            ]);
            
            $customers = $query->orderBy('name')
                ->offset($offset)
                ->limit($perPage)
                ->get();
            
            \Log::info('Customers fetched:', ['count' => $customers->count()]);
            
            $customerData = $customers->map(function($customer) {
                // Calculate available credit
                $creditLimit = (float) ($customer->credit_limit ?? 0);
                $creditBalance = (float) ($customer->credit_balance ?? 0);
                $availableCredit = max(0, $creditLimit - $creditBalance);
                
                // Calculate non-taxable sales
                $nonTaxableSales = ($customer->total_spent ?? 0) - ($customer->taxable_sales ?? 0);
                
                // Calculate VAT rate effective
                $vatRateEffective = 0;
                if (($customer->taxable_sales ?? 0) > 0) {
                    $vatRateEffective = (($customer->vat_collected ?? 0) / ($customer->taxable_sales ?? 1)) * 100;
                }
                
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'customer_type' => $customer->customer_type,
                    'customer_code' => $customer->customer_code,
                    'credit_limit' => $creditLimit,
                    'credit_balance' => $creditBalance,
                    'available_credit' => $availableCredit,
                    'balance' => (float) ($customer->balance ?? 0),
                    'total_spent' => (float) ($customer->total_spent ?? 0),
                    'taxable_sales' => (float) ($customer->taxable_sales ?? 0),
                    'vat_collected' => (float) ($customer->vat_collected ?? 0),
                    'non_taxable_sales' => (float) $nonTaxableSales,
                    'vat_rate_effective' => round($vatRateEffective, 2),
                    'total_orders' => (int) ($customer->total_orders ?? 0),
                    'average_order_value' => ($customer->total_orders ?? 0) > 0 
                        ? round(($customer->total_spent ?? 0) / ($customer->total_orders ?? 1), 2) 
                        : 0,
                    'last_purchase_date' => $customer->last_purchase_date,
                    'created_at' => $customer->created_at,
                    'address' => $customer->address,
                    'status' => $customer->status,
                    'company_name' => $customer->company_name,
                    'vat_number' => $customer->vat_number,
                    'is_vat_registered' => (bool) ($customer->is_vat_registered ?? false),
                    'discount_percent' => (float) ($customer->discount_percent ?? 0),
                    'loyalty_points' => (int) ($customer->loyalty_points ?? 0),
                    'contact_person' => $customer->contact_person,
                    'tax_id' => $customer->tax_id,
                    'website' => $customer->website,
                    'notes' => $customer->notes,
                ];
            });
            
            \Log::info('Customers processed successfully');
            
            $response = [
                'success' => true,
                'customers' => $customerData,
                'pagination' => [
                    'current_page' => (int) $page,
                    'last_page' => $lastPage,
                    'per_page' => (int) $perPage,
                    'total' => $total,
                ],
                'count' => $customers->count()
            ];
            
            \Log::info('API Response ready', ['customer_count' => count($customerData)]);
            \Log::info('========== API GET CUSTOMERS END ==========');
            
            return response()->json($response);
            
        } catch (\Exception $e) {
            \Log::error('========== API GET CUSTOMERS ERROR ==========');
            \Log::error('Error message: ' . $e->getMessage());
            \Log::error('Error trace: ' . $e->getTraceAsString());
            \Log::error('Request data:', $request->all());
            \Log::error('========== END ERROR ==========');
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to load customers',
                'message' => $e->getMessage(),
                'debug' => config('app.debug') ? [
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ] : null
            ], 500);
        }
    }

    /**
     * API: Store a new customer
     */
    public function apiStore(Request $request): JsonResponse
    {
        try {
            [$rules, $messages] = Customer::validateData($request->all());
            
            $validated = $request->validate($rules, $messages);
            
            // Set default values
            $validated['status'] = 'active';
            $validated['total_spent'] = 0;
            $validated['taxable_sales'] = 0;
            $validated['vat_collected'] = 0;
            $validated['total_orders'] = 0;
            $validated['loyalty_points'] = 0;
            $validated['credit_balance'] = $validated['credit_balance'] ?? 0;
            $validated['balance'] = 0;
            $validated['discount_percent'] = $validated['discount_percent'] ?? 0;
            
            // Set created_by if user is authenticated
            if (auth()->check()) {
                $validated['created_by'] = auth()->id();
            }
            
            $customer = Customer::create($validated);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('created customer via API');
            
            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'customer_type' => $customer->customer_type,
                    'customer_code' => $customer->customer_code,
                    'company_name' => $customer->company_name,
                    'vat_number' => $customer->vat_number,
                    'is_vat_registered' => (bool) $customer->is_vat_registered,
                    'credit_limit' => (float) $customer->credit_limit,
                    'credit_balance' => (float) $customer->credit_balance,
                    'available_credit' => (float) $customer->available_credit,
                    'discount_percent' => (float) $customer->discount_percent,
                    'loyalty_points' => $customer->loyalty_points,
                ],
                'message' => 'Customer created successfully'
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'error' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            Log::error('API Store Customer Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to create customer'
            ], 500);
        }
    }

    /**
     * API: Get customer credit information
     */
    public function apiGetCreditInfo($id): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($id);
            
            return response()->json([
                'success' => true,
                'credit_info' => [
                    'credit_limit' => (float) $customer->credit_limit,
                    'credit_balance' => (float) $customer->credit_balance,
                    'available_credit' => (float) $customer->available_credit,
                    'balance' => (float) $customer->balance,
                    'credit_utilization' => $customer->credit_limit > 0 
                        ? ($customer->credit_balance / $customer->credit_limit) * 100 
                        : 0,
                    'credit_status' => $customer->credit_balance > $customer->credit_limit 
                        ? 'over_limit' 
                        : ($customer->credit_balance > 0 ? 'active' : 'clear'),
                    'can_purchase_on_credit' => $customer->has_credit_available,
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Get Credit Info Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load credit information'
            ], 500);
        }
    }

    /**
     * Update customer credit balance
     */
    public function updateCreditBalance(Request $request, $id): JsonResponse
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric',
                'type' => 'required|in:payment,sale', // payment reduces balance, sale increases
                'notes' => 'nullable|string',
            ]);
            
            $customer = Customer::findOrFail($id);
            
            DB::beginTransaction();
            
            $success = false;
            $message = '';
            
            if ($validated['type'] === 'sale') {
                // Check if customer can make credit purchase
                $creditCheck = $customer->canPurchaseOnCredit($validated['amount']);
                if (!$creditCheck['allowed']) {
                    return response()->json([
                        'success' => false,
                        'message' => $creditCheck['reason']
                    ], 422);
                }
                $success = $customer->addCredit($validated['amount']);
                $message = 'Credit sale recorded.';
            } else {
                $success = $customer->reduceCredit($validated['amount']);
                $message = 'Payment recorded.';
            }
            
            if (!$success) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update credit balance. Check credit limit.'
                ], 422);
            }
            
            // Update balance field as well
            if ($validated['type'] === 'sale') {
                $customer->increment('balance', $validated['amount']);
            } else {
                $customer->decrement('balance', $validated['amount']);
                if ($customer->balance < 0) {
                    $customer->balance = 0;
                    $customer->save();
                }
            }
            
            // Log the balance update
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->withProperties([
                    'amount' => $validated['amount'],
                    'type' => $validated['type'],
                    'notes' => $validated['notes'] ?? null,
                    'new_credit_balance' => $customer->credit_balance,
                    'new_balance' => $customer->balance,
                    'available_credit' => $customer->available_credit
                ])
                ->log('updated customer credit balance via API');
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'credit_balance' => (float) $customer->credit_balance,
                'balance' => (float) $customer->balance,
                'available_credit' => (float) $customer->available_credit,
                'message' => $message
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update customer credit balance error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update credit balance.'
            ], 500);
        }
    }

    /**
     * API: Quick add customer for POS
     */
    public function apiQuickAdd(Request $request): JsonResponse
    {
        try {
            [$rules, $messages] = Customer::validateData($request->all());
            
            $validated = $request->validate($rules, $messages);
            
            // Set VAT registration based on VAT number
            if (!empty($validated['vat_number'])) {
                $validated['is_vat_registered'] = true;
            } elseif (!isset($validated['is_vat_registered'])) {
                $validated['is_vat_registered'] = false;
            }
            
            // Set default values for POS
            $validated['status'] = 'active';
            $validated['total_spent'] = 0;
            $validated['taxable_sales'] = 0;
            $validated['vat_collected'] = 0;
            $validated['total_orders'] = 0;
            $validated['loyalty_points'] = 0;
            $validated['credit_balance'] = $validated['credit_balance'] ?? 0;
            $validated['balance'] = 0;
            $validated['discount_percent'] = $validated['discount_percent'] ?? 0;
            
            // Set created_by if user is authenticated
            if (auth()->check()) {
                $validated['created_by'] = auth()->id();
            }
            
            $customer = Customer::create($validated);
            
            // Log activity
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('quick added customer via POS');
            
            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'email' => $customer->email,
                    'vat_number' => $customer->vat_number,
                    'is_vat_registered' => (bool) $customer->is_vat_registered,
                    'customer_type' => $customer->customer_type,
                    'customer_code' => $customer->customer_code,
                    'company_name' => $customer->company_name,
                    'credit_limit' => (float) ($customer->credit_limit ?? 0),
                    'credit_balance' => (float) ($customer->credit_balance ?? 0),
                    'available_credit' => (float) $customer->available_credit,
                    'discount_percent' => (float) $customer->discount_percent,
                    'loyalty_points' => $customer->loyalty_points,
                ],
                'message' => 'Customer added successfully'
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'error' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            Log::error('API Quick Add Customer Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to add customer'
            ], 500);
        }
    }

    /**
     * API: Search customers
     */
    public function apiSearchCustomers(Request $request): JsonResponse
    {
        try {
            $search = $request->input('q', '');
            $type = $request->input('type', 'all');
            $limit = $request->input('limit', 20);
            
            $query = Customer::where('status', 'active');
            
            // Apply search filter
            if (!empty($search)) {
                $query->search($search);
            }
            
            // Apply type filter
            if ($type !== 'all') {
                $query->where('customer_type', $type);
            }
            
            $customers = $query->orderBy('name')
                ->limit($limit)
                ->get()
                ->map(function($customer) {
                    return [
                        'id' => $customer->id,
                        'name' => $customer->name,
                        'email' => $customer->email,
                        'phone' => $customer->phone,
                        'customer_type' => $customer->customer_type,
                        'customer_code' => $customer->customer_code,
                        'company_name' => $customer->company_name,
                        'credit_limit' => (float) ($customer->credit_limit ?? 0),
                        'credit_balance' => (float) ($customer->credit_balance ?? 0),
                        'available_credit' => (float) $customer->available_credit,
                        'balance' => (float) $customer->balance,
                        'total_spent' => (float) ($customer->total_spent ?? 0),
                        'taxable_sales' => (float) ($customer->taxable_sales ?? 0),
                        'vat_collected' => (float) ($customer->vat_collected ?? 0),
                        'non_taxable_sales' => $customer->non_taxable_sales,
                        'vat_rate_effective' => $customer->vat_rate_effective,
                        'total_orders' => (int) ($customer->total_orders ?? 0),
                        'average_order_value' => $customer->average_order_value,
                        'last_purchase_date' => $customer->last_purchase_date,
                        'vat_number' => $customer->vat_number,
                        'is_vat_registered' => (bool) $customer->is_vat_registered,
                        'discount_percent' => (float) $customer->discount_percent,
                        'loyalty_points' => $customer->loyalty_points,
                        'customer_level' => $customer->customer_level,
                    ];
                });
            
            return response()->json([
                'success' => true,
                'customers' => $customers
            ]);
            
        } catch (\Exception $e) {
            Log::error('API Search Customers Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to search customers'
            ], 500);
        }
    }

    /**
     * API: Get customer statistics
     */
    public function apiGetStats(): JsonResponse
    {
        try {
            $totalCustomers = Customer::count();
            $activeToday = Customer::whereDate('last_purchase_date', today())->count();
            $totalCreditLimit = Customer::sum('credit_limit');
            $totalCreditBalance = Customer::sum('credit_balance');
            $totalRevenue = Customer::sum('total_spent');
            $totalVatCollected = Customer::sum('vat_collected');
            $avgSale = Customer::where('total_spent', '>', 0)
                ->avg('total_spent');
            
            return response()->json([
                'success' => true,
                'stats' => [
                    'total_customers' => $totalCustomers,
                    'active_today' => $activeToday,
                    'total_credit_limit' => (float) $totalCreditLimit,
                    'total_credit_balance' => (float) $totalCreditBalance,
                    'total_revenue' => (float) $totalRevenue,
                    'total_vat_collected' => (float) $totalVatCollected,
                    'avg_sale' => (float) ($avgSale ?? 0),
                    'vat_registered_count' => Customer::vatRegistered()->count(),
                    'top_customer_type' => Customer::groupBy('customer_type')
                        ->selectRaw('customer_type, count(*) as count')
                        ->orderBy('count', 'desc')
                        ->first()->customer_type ?? 'N/A',
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Customer stats error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load stats',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * API: Get customers for dropdown/AJAX
     */
    public function ajaxSearch(Request $request): JsonResponse
    {
        try {
            $search = $request->input('q', '');
            
            $query = Customer::where('status', 'active');
            
            if (!empty($search)) {
                $query->search($search);
            }
            
            $customers = $query->orderBy('name')
                ->limit(10)
                ->get(['id', 'name', 'email', 'phone', 'customer_code', 
                      'customer_type', 'vat_number', 'is_vat_registered',
                      'credit_limit', 'credit_balance', 'balance',
                      'total_spent', 'loyalty_points', 'discount_percent']);
            
            return response()->json([
                'success' => true,
                'customers' => $customers
            ]);
            
        } catch (\Exception $e) {
            Log::error('Customer AJAX Search Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to search customers'
            ], 500);
        }
    }

    /**
     * API: Get single customer for POS
     */
    public function apiGetCustomer($id): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($id);
            
            $customerData = [
                'id' => $customer->id,
                'name' => $customer->name,
                'email' => $customer->email,
                'phone' => $customer->phone,
                'customer_type' => $customer->customer_type,
                'customer_code' => $customer->customer_code,
                'company_name' => $customer->company_name,
                'credit_limit' => (float) ($customer->credit_limit ?? 0),
                'credit_balance' => (float) ($customer->credit_balance ?? 0),
                'available_credit' => (float) $customer->available_credit,
                'balance' => (float) $customer->balance,
                'total_spent' => (float) ($customer->total_spent ?? 0),
                'taxable_sales' => (float) ($customer->taxable_sales ?? 0),
                'vat_collected' => (float) ($customer->vat_collected ?? 0),
                'non_taxable_sales' => $customer->non_taxable_sales,
                'vat_rate_effective' => $customer->vat_rate_effective,
                'total_orders' => (int) ($customer->total_orders ?? 0),
                'average_order_value' => $customer->average_order_value,
                'last_purchase_date' => $customer->last_purchase_date,
                'vat_number' => $customer->vat_number,
                'is_vat_registered' => (bool) $customer->is_vat_registered,
                'discount_percent' => (float) ($customer->discount_percent ?? 0),
                'loyalty_points' => (int) ($customer->loyalty_points ?? 0),
                'customer_level' => $customer->customer_level,
                'days_since_last_purchase' => $customer->days_since_last_purchase,
                'is_active_customer' => $customer->is_active_customer,
                'has_credit_available' => $customer->has_credit_available,
                'address' => $customer->address,
                'contact_person' => $customer->contact_person,
                'tax_id' => $customer->tax_id,
                'website' => $customer->website,
                'notes' => $customer->notes,
            ];
            
            return response()->json([
                'success' => true,
                'customer' => $customerData
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Get customer error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Customer not found'
            ], 404);
        }
    }

    /**
 * API: Get customer statistics
 */
public function apiGetCustomerStats(): JsonResponse
{
    try {
        // Get basic customer stats
        $totalCustomers = Customer::count();
        $activeToday = Customer::whereDate('last_purchase_date', today())->count();
        $totalRevenue = Customer::sum('total_spent');
        $totalVatCollected = Customer::sum('vat_collected');
        $totalCreditBalance = Customer::sum('credit_balance');
        
        // Customer type breakdown
        $customerTypes = Customer::selectRaw('customer_type, COUNT(*) as count')
            ->groupBy('customer_type')
            ->get()
            ->pluck('count', 'customer_type')
            ->toArray();
        
        // Get top customers by revenue
        $topCustomers = Customer::orderBy('total_spent', 'desc')
            ->limit(5)
            ->get(['id', 'name', 'total_spent', 'total_orders']);
        
        // VAT registered stats
        $vatStats = [
            'vat_registered' => Customer::where('is_vat_registered', true)->count(),
            'non_vat' => Customer::where('is_vat_registered', false)->count(),
            'total_vat_collected' => Customer::sum('vat_collected'),
            'total_taxable_sales' => Customer::sum('taxable_sales'),
        ];
        
        // Monthly growth
        $monthlyGrowth = Customer::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, COUNT(*) as count')
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderBy('month')
            ->get();
        
        return response()->json([
            'success' => true,
            'stats' => [
                'total_customers' => $totalCustomers,
                'active_today' => $activeToday,
                'total_revenue' => (float) $totalRevenue,
                'total_vat_collected' => (float) $totalVatCollected,
                'total_credit_balance' => (float) $totalCreditBalance,
                'avg_revenue_per_customer' => $totalCustomers > 0 ? round($totalRevenue / $totalCustomers, 2) : 0,
                'customer_types' => $customerTypes,
                'vat_stats' => $vatStats,
                'monthly_growth' => $monthlyGrowth,
                'top_customers' => $topCustomers,
            ],
            'timestamp' => now()->toISOString()
        ]);
        
    } catch (\Exception $e) {
        \Log::error('Customer stats API error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'error' => 'Failed to load customer statistics',
            'message' => config('app.debug') ? $e->getMessage() : 'Internal server error'
        ], 500);
    }
}
    /**
     * Get customer purchase history for API
     */
    public function getPurchaseHistory($id)
    {
        try {
            $customer = Customer::findOrFail($id);
            
            $history = $customer->sales()
                ->with(['items.product', 'payments'])
                ->orderBy('sale_date', 'desc')
                ->limit(10)
                ->get()
                ->map(function($sale) {
                    return [
                        'id' => $sale->id,
                        'invoice_no' => $sale->invoice_no,
                        'date' => $sale->sale_date->format('Y-m-d H:i'),
                        'total_amount' => $sale->grand_total,
                        'taxable_amount' => $sale->taxable_amount ?? 0,
                        'tax_amount' => $sale->tax_amount ?? 0,
                        'payment_method' => $sale->payment_method,
                        'payment_status' => $sale->payment_status,
                        'is_taxable' => $sale->is_taxable ?? false,
                        'items' => $sale->items->map(function($item) {
                            return [
                                'product_name' => $item->product->name ?? 'Unknown Product',
                                'quantity' => $item->quantity,
                                'price' => $item->price,
                                'total' => $item->quantity * $item->price
                            ];
                        })
                    ];
                });
            
            return response()->json([
                'success' => true,
                'history' => $history,
                'customer_name' => $customer->name,
                'total_purchases' => $customer->sales()->count(),
                'total_spent' => $customer->total_spent
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Customer history error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load purchase history'
            ], 500);
        }
    }

    /**
     * Generate customer statement
     */
    public function generateStatement(Request $request, $id)
    {
        try {
            $customer = Customer::findOrFail($id);
            
            $startDate = $request->input('start_date') ? \Carbon\Carbon::parse($request->input('start_date')) : now()->subMonth();
            $endDate = $request->input('end_date') ? \Carbon\Carbon::parse($request->input('end_date')) : now();
            
            $statement = $customer->generateStatement($startDate, $endDate);
            
            return response()->json([
                'success' => true,
                'statement' => $statement,
                'download_url' => route('customers.statement.pdf', [
                    'customer' => $id,
                    'start_date' => $startDate->format('Y-m-d'),
                    'end_date' => $endDate->format('Y-m-d')
                ])
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Generate statement error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to generate statement'
            ], 500);
        }
    }

    /**
     * Get VAT registered customers report.
     */
    public function vatReport(Request $request)
    {
        try {
            $query = Customer::vatRegistered();
            
            // Search functionality
            if ($request->has('search')) {
                $search = $request->input('search');
                $query->search($search);
            }
            
            // Filter by date range
            if ($request->has('start_date') && $request->has('end_date')) {
                $query->whereBetween('last_purchase_date', [$request->start_date, $request->end_date]);
            }
            
            $customers = $query->orderBy('vat_collected', 'desc')
                ->paginate(20)
                ->withQueryString();
            
            $stats = [
                'total_vat_customers' => Customer::vatRegistered()->count(),
                'total_vat_collected' => Customer::vatRegistered()->sum('vat_collected'),
                'total_taxable_sales' => Customer::vatRegistered()->sum('taxable_sales'),
                'avg_vat_per_customer' => Customer::vatRegistered()->avg('vat_collected'),
                'total_non_taxable_sales' => Customer::vatRegistered()->sum('total_spent') - Customer::vatRegistered()->sum('taxable_sales'),
            ];
            
            return view('customers.vat-report', compact('customers', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('VAT report error: ' . $e->getMessage());
            return redirect()->route('customers.index')
                ->with('error', 'Failed to load VAT report.');
        }
    }

    /**
     * Check if customer has sales
     */
    public function hasSales($id)
    {
        try {
            $customer = Customer::findOrFail($id);
            $hasSales = $customer->sales()->exists();
            
            return response()->json([
                'success' => true,
                'has_sales' => $hasSales,
                'sales_count' => $customer->sales()->count()
            ]);
            
        } catch (\Exception $e) {
            Log::error('Check customer sales error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to check sales'
            ], 500);
        }
    }

    /**
     * Reset customer statistics
     */
    public function resetStatistics($id)
    {
        try {
            $customer = Customer::findOrFail($id);
            
            DB::beginTransaction();
            
            $customer->resetStatistics();
            
            // Log the reset
            activity()
                ->causedBy(auth()->user())
                ->performedOn($customer)
                ->log('reset customer statistics');
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Customer statistics reset successfully.',
                'customer' => [
                    'total_spent' => $customer->total_spent,
                    'taxable_sales' => $customer->taxable_sales,
                    'vat_collected' => $customer->vat_collected,
                    'total_orders' => $customer->total_orders,
                    'loyalty_points' => $customer->loyalty_points,
                    'credit_balance' => $customer->credit_balance,
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Reset statistics error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to reset statistics'
            ], 500);
        }
    }

    /**
     * Validate VAT number
     */
    public function validateVatNumber(Request $request)
    {
        try {
            $vatNumber = $request->input('vat_number');
            
            if (empty($vatNumber)) {
                return response()->json([
                    'success' => true,
                    'is_valid' => false,
                    'message' => 'VAT number is empty'
                ]);
            }
            
            // Check if VAT number already exists
            $existing = Customer::where('vat_number', $vatNumber)->first();
            if ($existing) {
                return response()->json([
                    'success' => true,
                    'is_valid' => false,
                    'message' => 'VAT number already exists for customer: ' . $existing->name,
                    'customer' => $existing->only(['id', 'name', 'email'])
                ]);
            }
            
            // Basic validation pattern
            $isValid = preg_match('/^[A-Z]{2}[0-9A-Z]{8,12}$/', $vatNumber) === 1;
            
            return response()->json([
                'success' => true,
                'is_valid' => $isValid,
                'message' => $isValid ? 'VAT number format is valid' : 'Invalid VAT number format'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Validate VAT number error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to validate VAT number'
            ], 500);
        }
    }


}