<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class ProductApiController extends Controller
{
    /**
     * Get products for POS
     */
    public function index(Request $request): JsonResponse
    {
        try {
            Log::info('POS Products API called', $request->all());
            
            // Build query
            $query = Product::with(['category', 'tax'])
                ->where('status', 'active')
                ->orderBy('name');
            
            // Apply filters
            $this->applyFilters($query, $request);
            
            // Get paginated results
            $perPage = $request->get('per_page', 100);
            $page = $request->get('page', 1);
            $products = $query->paginate($perPage, ['*'], 'page', $page);
            
            // Transform products for POS
            $transformedProducts = $products->map(function ($product) {
                return $this->transformProduct($product);
            });
            
            // Get categories for filter
            $categories = Category::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'parent_id']);
            
            return response()->json([
                'success' => true,
                'products' => $transformedProducts,
                'categories' => $categories,
                'stats' => [
                    'total_products' => $products->total(),
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'per_page' => $products->perPage(),
                    'has_more_pages' => $products->hasMorePages(),
                ],
                'message' => 'Products loaded successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Products API Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to load products',
                'products' => [],
                'categories' => [],
                'stats' => [],
                'message' => 'Error loading products'
            ], 500);
        }
    }
    
    /**
     * Search products
     */
    public function search(Request $request): JsonResponse
    {
        try {
            $search = $request->get('q', '');
            $limit = $request->get('limit', 50);
            
            $products = Product::with(['category'])
                ->where('status', 'active')
                ->where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('sku', 'like', "%{$search}%")
                          ->orWhere('barcode', 'like', "%{$search}%");
                })
                ->orderBy('name')
                ->limit($limit)
                ->get()
                ->map(function ($product) {
                    return $this->transformProduct($product);
                });
            
            return response()->json([
                'success' => true,
                'products' => $products,
                'count' => $products->count(),
                'search_term' => $search
            ]);
            
        } catch (\Exception $e) {
            Log::error('Product search error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Search failed'], 500);
        }
    }
    
    /**
     * Get product by barcode
     */
    public function getByBarcode(Request $request, $barcode): JsonResponse
    {
        try {
            $product = Product::with(['category', 'tax'])
                ->where(function($query) use ($barcode) {
                    $query->where('barcode', $barcode)
                          ->orWhere('sku', $barcode);
                })
                ->where('status', 'active')
                ->first();
            
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'error' => 'Product not found',
                    'barcode' => $barcode
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'product' => $this->transformProduct($product)
            ]);
            
        } catch (\Exception $e) {
            Log::error('Barcode lookup error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Lookup failed'], 500);
        }
    }
    
    /**
     * Get single product
     */
    public function show($id): JsonResponse
    {
        try {
            $product = Product::with(['category', 'tax'])
                ->where('status', 'active')
                ->find($id);
            
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'error' => 'Product not found'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'product' => $this->transformProduct($product)
            ]);
            
        } catch (\Exception $e) {
            Log::error('Product show error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to load product'], 500);
        }
    }
    
    /**
     * Get categories
     */
    public function categories(): JsonResponse
    {
        try {
            $categories = Category::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'parent_id', 'description']);
            
            return response()->json([
                'success' => true,
                'categories' => $categories
            ]);
            
        } catch (\Exception $e) {
            Log::error('Categories API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to load categories'], 500);
        }
    }
    
    /**
     * Apply filters to query
     */
    private function applyFilters($query, Request $request): void
    {
        // Category filter
        if ($request->has('category_id') && $request->category_id !== 'all') {
            $categoryId = $request->category_id;
            if ($categoryId === 'null') {
                $query->whereNull('category_id');
            } else {
                $query->where('category_id', $categoryId);
            }
        }
        
        // Search filter
        if ($request->has('q') && !empty($request->q)) {
            $search = $request->q;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        // Stock filter
        if ($request->has('stock_status')) {
            switch ($request->stock_status) {
                case 'in_stock':
                    $query->where('stock', '>', 0);
                    break;
                case 'low_stock':
                    $query->where('stock', '>', 0)
                          ->whereColumn('stock', '<=', 'minimum_stock');
                    break;
                case 'out_of_stock':
                    $query->where('stock', '<=', 0);
                    break;
            }
        }
        
        // In stock only filter
        if ($request->has('in_stock_only') && $request->in_stock_only) {
            $query->where('stock', '>', 0);
        }
    }
    
    /**
     * Transform product for API response
     */
    private function transformProduct(Product $product): array
    {
        return [
            'id' => $product->id,
            'sku' => $product->sku,
            'barcode' => $product->barcode,
            'name' => $product->name,
            'description' => $product->description,
            'short_description' => $product->short_description,
            'price' => (float) $product->sale_price,
            'sale_price' => (float) $product->sale_price,
            'cost_price' => (float) $product->cost_price,
            'final_price' => (float) $product->final_price,
            'discount_price' => (float) $product->discount_price,
            'discount_percent' => (float) $product->discount_percent,
            'stock' => $product->stock,
            'available_stock' => $product->available_stock,
            'has_vat' => (bool) $product->has_vat,
            'is_vatable' => $product->is_vatable,
            'tax_rate' => (float) $product->tax_rate,
            'tax_id' => $product->tax_id,
            'category_id' => $product->category_id,
            'category_name' => $product->category ? $product->category->name : null,
            'brand_id' => $product->brand_id,
            'supplier_id' => $product->supplier_id,
            'unit_id' => $product->unit_id,
            'minimum_stock' => $product->minimum_stock,
            'reorder_point' => $product->reorder_point,
            'maximum_stock' => $product->maximum_stock,
            'is_low_stock' => $product->is_low_stock,
            'low_stock_threshold' => $product->low_stock_threshold,
            'track_inventory' => $product->track_inventory,
            'has_expiry' => $product->has_expiry,
            'track_batches' => $product->track_batches,
            'image_url' => $product->image_path ? asset('storage/' . $product->image_path) : null,
            'status' => $product->status,
            'created_at' => $product->created_at,
            'updated_at' => $product->updated_at,
        ];
    }
}