<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class PosProductController extends Controller
{
    /**
     * Get products for POS with proper response format
     */
    public function index(Request $request): JsonResponse
    {
        try {
            Log::info('POS Products API called', [
                'category_id' => $request->input('category_id'),
                'search' => $request->input('q'),
                'page' => $request->input('page', 1)
            ]);
            
            $query = Product::with(['category', 'tax'])
                ->where('status', 'active')
                ->orderBy('name');
            
            // Apply category filter
            if ($request->has('category_id') && $request->category_id !== 'all') {
                $categoryId = $request->category_id;
                
                // Handle "null" string
                if ($categoryId === 'null') {
                    $query->whereNull('category_id');
                } else {
                    $query->where('category_id', $categoryId);
                }
            }
            
            // Apply search filter
            if ($request->has('q') && !empty($request->q)) {
                $search = $request->q;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%")
                      ->orWhere('barcode', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                });
            }
            
            // Filter by stock if needed
            if ($request->has('in_stock_only') && $request->in_stock_only) {
                $query->where('stock', '>', 0);
            }
            
            // Get pagination or all
            $perPage = $request->has('per_page') ? (int)$request->per_page : 100;
            $products = $query->paginate($perPage);
            
            // Transform products for POS
            $transformedProducts = $products->map(function ($product) {
                return [
                    'id' => $product->id,
                    'sku' => $product->sku,
                    'barcode' => $product->barcode,
                    'name' => $product->name,
                    'description' => $product->description,
                    'short_description' => $product->short_description,
                    'price' => (float) $product->sale_price,
                    'sale_price' => (float) $product->sale_price,
                    'cost_price' => (float) $product->cost_price,
                    'final_price' => (float) $product->final_price,
                    'stock' => $product->stock,
                    'available_stock' => $product->available_stock,
                    'has_vat' => (bool) $product->has_vat,
                    'is_vatable' => $product->is_vatable,
                    'tax_rate' => (float) $product->tax_rate,
                    'tax_id' => $product->tax_id,
                    'category_id' => $product->category_id,
                    'category_name' => $product->category ? $product->category->name : null,
                    'brand_id' => $product->brand_id,
                    'supplier_id' => $product->supplier_id,
                    'unit_id' => $product->unit_id,
                    'minimum_stock' => $product->minimum_stock,
                    'reorder_point' => $product->reorder_point,
                    'maximum_stock' => $product->maximum_stock,
                    'is_low_stock' => $product->is_low_stock,
                    'low_stock_threshold' => $product->low_stock_threshold,
                    'track_inventory' => $product->track_inventory,
                    'has_expiry' => $product->has_expiry,
                    'track_batches' => $product->track_batches,
                    'image_url' => $product->image_path ? asset('storage/' . $product->image_path) : null,
                    'status' => $product->status,
                    'created_at' => $product->created_at,
                    'updated_at' => $product->updated_at,
                ];
            });
            
            // Get categories for filter
            $categories = Category::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'parent_id']);
            
            // Stats
            $stats = [
                'total_products' => Product::where('status', 'active')->count(),
                'in_stock_count' => Product::where('status', 'active')->where('stock', '>', 0)->count(),
                'low_stock_count' => Product::where('status', 'active')
                    ->where('stock', '>', 0)
                    ->whereColumn('stock', '<=', 'minimum_stock')
                    ->count(),
                'out_of_stock_count' => Product::where('status', 'active')->where('stock', '<=', 0)->count(),
            ];
            
            return response()->json([
                'success' => true,
                'products' => $transformedProducts,
                'categories' => $categories,
                'stats' => $stats,
                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'per_page' => $products->perPage(),
                    'total' => $products->total(),
                    'has_more_pages' => $products->hasMorePages(),
                ],
                'message' => 'Products loaded successfully'
            ]);
            
        } catch (\Exception $e) {
            Log::error('POS Products API Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'Failed to load products: ' . $e->getMessage(),
                'products' => [],
                'categories' => [],
                'stats' => [],
                'pagination' => [],
                'message' => 'Error loading products'
            ], 500);
        }
    }
    
    /**
     * Search products quickly for POS
     */
    public function search(Request $request): JsonResponse
    {
        try {
            $search = $request->input('q', '');
            $limit = $request->input('limit', 50);
            
            $products = Product::with(['category'])
                ->where('status', 'active')
                ->where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('sku', 'like', "%{$search}%")
                          ->orWhere('barcode', 'like', "%{$search}%");
                })
                ->orderBy('name')
                ->limit($limit)
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'name' => $product->name,
                        'sku' => $product->sku,
                        'barcode' => $product->barcode,
                        'price' => (float) $product->final_price,
                        'stock' => $product->stock,
                        'has_vat' => (bool) $product->has_vat,
                        'category' => $product->category ? $product->category->name : null,
                        'image' => $product->image_path ? asset('storage/' . $product->image_path) : null,
                    ];
                });
            
            return response()->json([
                'success' => true,
                'products' => $products,
                'count' => $products->count(),
                'search_term' => $search
            ]);
            
        } catch (\Exception $e) {
            Log::error('POS Search Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Search failed'], 500);
        }
    }
    
    /**
     * Get product by barcode for POS
     */
    public function barcode(Request $request, $barcode): JsonResponse
    {
        try {
            $product = Product::with(['category', 'tax', 'batches'])
                ->where(function($query) use ($barcode) {
                    $query->where('barcode', $barcode)
                          ->orWhere('sku', $barcode);
                })
                ->where('status', 'active')
                ->first();
            
            if (!$product) {
                return response()->json([
                    'success' => false,
                    'error' => 'Product not found',
                    'barcode' => $barcode
                ], 404);
            }
            
            $productData = [
                'id' => $product->id,
                'sku' => $product->sku,
                'barcode' => $product->barcode,
                'name' => $product->name,
                'price' => (float) $product->final_price,
                'sale_price' => (float) $product->sale_price,
                'stock' => $product->stock,
                'has_vat' => (bool) $product->has_vat,
                'tax_rate' => (float) $product->tax_rate,
                'category_id' => $product->category_id,
                'category_name' => $product->category ? $product->category->name : null,
                'is_low_stock' => $product->is_low_stock,
                'batches' => $product->batches->map(function($batch) {
                    return [
                        'id' => $batch->id,
                        'batch_number' => $batch->batch_number,
                        'expiry_date' => $batch->expiry_date,
                        'current_quantity' => $batch->current_quantity,
                    ];
                })
            ];
            
            return response()->json([
                'success' => true,
                'product' => $productData
            ]);
            
        } catch (\Exception $e) {
            Log::error('Barcode lookup error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Lookup failed'], 500);
        }
    }
    
    /**
     * Get categories for POS filter
     */
    public function categories(): JsonResponse
    {
        try {
            $categories = Category::where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'parent_id', 'description']);
            
            return response()->json([
                'success' => true,
                'categories' => $categories
            ]);
            
        } catch (\Exception $e) {
            Log::error('Categories API error: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => 'Failed to load categories'], 500);
        }
    }
}