<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\SalesService;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PosApiController extends Controller
{
    protected SalesService $sales;

    public function __construct(SalesService $sales) { $this->sales = $sales; }

    public function checkout(Request $req)
    {
        $req->validate([
            'items'=>'required|array|min:1',
            'items.*.product_id'=>'required|exists:products,id',
            'items.*.qty'=>'required|integer|min:1',
            'items.*.unit_price'=>'required|numeric|min:0',
            'payment_method_id'=>'required|exists:payment_methods,id',
            'customer_id'=>'nullable|exists:customers,id',
            'is_vat_applied'=>'nullable|boolean'
        ]);

        try {
            $sale = DB::transaction(function() use ($req) {
                return $this->sales->createSaleFromPayload($req->user(), $req->all());
            }, 5); // retry up to 5 times in case of deadlocks

            return response()->json([
                'sale' => $sale->load('items','payments'),
                'receipt_url' => route('sales.show', $sale->id)
            ]);
        } catch (\Exception $e) {
            \Log::error('POS Checkout failed: '.$e->getMessage(), ['trace'=>$e->getTraceAsString()]);
            throw ValidationException::withMessages(['error'=>[$e->getMessage()]]);
        }
    }
}
