<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use App\Models\Sale;
use App\Models\Payment;
use App\Models\Customer;
use App\Models\MpesaTransaction;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class PaymentProcessingController extends Controller
{
    /**
     * Process a payment
     */
    public function process(Request $request)
    {
        DB::beginTransaction();
        
        try {
            // Validate request
            $validator = Validator::make($request->all(), [
                'payment_method' => 'required|string|exists:payment_methods,code',
                'amount' => 'required|numeric|min:0.01',
                'cart' => 'required|array|min:1',
                'cart.*.product_id' => 'required|exists:products,id',
                'cart.*.quantity' => 'required|integer|min:1',
                'customer_id' => 'nullable|exists:customers,id',
                'discount' => 'nullable|numeric|min:0',
                'vat_rate' => 'nullable|numeric|min:0|max:100',
                'notes' => 'nullable|string|max:500',
                'mpesa_phone' => 'required_if:payment_method,mpesa|string|min:9|max:12',
                'mpesa_transaction_id' => 'nullable|string',
                'cash_received' => 'required_if:payment_method,cash|numeric|min:0',
                'card_number' => 'nullable|string',
                'card_holder' => 'nullable|string',
                'card_expiry' => 'nullable|string',
                'bank_name' => 'nullable|string',
                'bank_reference' => 'nullable|string',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            $data = $validator->validated();
            $paymentMethod = PaymentMethod::where('code', $data['payment_method'])->firstOrFail();
            
            // Check if cart items are available
            $cartItems = $data['cart'];
            $products = [];
            $subtotal = 0;
            
            foreach ($cartItems as $item) {
                $product = Product::find($item['product_id']);
                
                if (!$product) {
                    throw new \Exception("Product ID {$item['product_id']} not found");
                }
                
                if ($product->available_stock < $item['quantity']) {
                    throw new \Exception("Insufficient stock for {$product->name}. Available: {$product->available_stock}");
                }
                
                $itemTotal = $product->sale_price * $item['quantity'];
                $subtotal += $itemTotal;
                
                $products[] = [
                    'product' => $product,
                    'quantity' => $item['quantity'],
                    'price' => $product->sale_price,
                    'total' => $itemTotal
                ];
            }
            
            // Calculate totals
            $discount = $data['discount'] ?? 0;
            $vatRate = $data['vat_rate'] ?? config('pos.vat_rate', 16);
            $vatAmount = ($subtotal - $discount) * ($vatRate / 100);
            $totalAmount = ($subtotal - $discount) + $vatAmount;
            
            // Validate amount matches
            if (abs($totalAmount - $data['amount']) > 0.01) {
                throw new \Exception("Calculated total ({$totalAmount}) doesn't match provided amount ({$data['amount']})");
            }
            
            // Check payment method specific validations
            $this->validatePaymentMethod($paymentMethod, $data);
            
            // Generate invoice number
            $invoiceNo = $this->generateInvoiceNumber();
            
            // Create sale
            $sale = Sale::create([
                'invoice_no' => $invoiceNo,
                'customer_id' => $data['customer_id'] ?? null,
                'subtotal' => $subtotal,
                'discount' => $discount,
                'vat_rate' => $vatRate,
                'vat_amount' => $vatAmount,
                'total' => $totalAmount,
                'status' => 'completed',
                'payment_status' => 'paid',
                'notes' => $data['notes'] ?? null,
                'user_id' => auth()->id(),
                'sale_date' => now(),
            ]);
            
            // Add sale items
            foreach ($products as $item) {
                $sale->items()->create([
                    'product_id' => $item['product']->id,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['price'],
                    'total' => $item['total'],
                    'vat_rate' => $vatRate,
                    'vat_amount' => $item['total'] * ($vatRate / 100),
                ]);
                
                // Update product stock
                $item['product']->decrement('available_stock', $item['quantity']);
                $item['product']->increment('total_sold', $item['quantity']);
            }
            
            // Create payment
            $payment = Payment::create([
                'sale_id' => $sale->id,
                'payment_method_id' => $paymentMethod->id,
                'amount' => $totalAmount,
                'reference' => $this->generatePaymentReference($paymentMethod->code),
                'status' => 'completed',
                'notes' => $data['notes'] ?? null,
                'payment_details' => $this->getPaymentDetails($paymentMethod, $data),
                'user_id' => auth()->id(),
            ]);
            
            // Handle method-specific logic
            $this->handlePaymentMethodLogic($paymentMethod, $payment, $data);
            
            // Update customer if exists
            if ($sale->customer_id) {
                $customer = Customer::find($sale->customer_id);
                $customer->increment('total_purchases', $totalAmount);
                $customer->last_purchase = now();
                $customer->save();
            }
            
            DB::commit();
            
            // Generate receipt data
            $receipt = $this->generateReceiptData($sale, $payment);
            
            return response()->json([
                'success' => true,
                'message' => 'Payment processed successfully',
                'sale' => [
                    'id' => $sale->id,
                    'invoice_no' => $sale->invoice_no,
                    'total' => $sale->total,
                    'date' => $sale->sale_date->format('Y-m-d H:i:s'),
                ],
                'payment' => [
                    'id' => $payment->id,
                    'method' => $paymentMethod->name,
                    'reference' => $payment->reference,
                    'status' => $payment->status,
                ],
                'receipt' => $receipt,
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Payment processing error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request' => $request->all(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Payment failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Process cash payment
     */
    public function cash(Request $request)
    {
        $request->merge(['payment_method' => 'cash']);
        return $this->process($request);
    }
    
    /**
     * Process M-Pesa payment
     */
    public function mpesa(Request $request)
    {
        $request->merge(['payment_method' => 'mpesa']);
        
        // If STK push
        if ($request->has('stk_push') && $request->stk_push) {
            return $this->stkPush($request);
        }
        
        // Manual M-Pesa entry
        return $this->manualMpesa($request);
    }
    
    /**
     * Process M-Pesa STK Push
     */
    public function stkPush(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:1|max:150000',
            'phone' => 'required|string|min:9|max:12',
            'reference' => 'nullable|string',
            'description' => 'nullable|string',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $data = $validator->validated();
        
        try {
            // Format phone
            $phone = $this->formatPhone($data['phone']);
            
            // In production, call Daraja API here
            // For now, simulate STK push
            
            $checkoutRequestId = 'STK_' . Str::random(20);
            
            return response()->json([
                'success' => true,
                'message' => 'STK Push initiated',
                'checkout_request_id' => $checkoutRequestId,
                'phone' => $phone,
                'amount' => $data['amount'],
                'instructions' => 'Customer should check their phone for STK prompt'
            ]);
            
        } catch (\Exception $e) {
            Log::error('STK Push error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'STK Push failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Manual M-Pesa entry
     */
    public function manualMpesa(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'phone' => 'required|string',
            'transaction_id' => 'required|string',
            'receipt_no' => 'nullable|string',
            'transaction_date' => 'nullable|date',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $data = $validator->validated();
        
        try {
            // Check if transaction already exists
            $existing = MpesaTransaction::where('transaction_id', $data['transaction_id'])
                ->where('status', 'completed')
                ->first();
            
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction ID already used'
                ], 400);
            }
            
            // Create M-Pesa transaction record
            $mpesa = MpesaTransaction::create([
                'transaction_id' => $data['transaction_id'],
                'receipt_no' => $data['receipt_no'] ?? null,
                'amount' => $data['amount'],
                'phone' => $this->formatPhone($data['phone']),
                'transaction_date' => $data['transaction_date'] ?? now(),
                'status' => 'completed',
                'type' => 'manual',
                'user_id' => auth()->id(),
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'M-Pesa transaction recorded',
                'transaction' => $mpesa
            ]);
            
        } catch (\Exception $e) {
            Log::error('Manual M-Pesa error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to record M-Pesa transaction'
            ], 500);
        }
    }
    
    /**
     * Check M-Pesa payment status
     */
    public function checkMpesaStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'checkout_request_id' => 'required|string',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        // In production, check with Daraja API
        // For now, simulate status check
        
        $statuses = ['pending', 'completed', 'failed', 'cancelled'];
        $status = $statuses[array_rand($statuses)];
        
        return response()->json([
            'success' => true,
            'status' => $status,
            'checkout_request_id' => $request->checkout_request_id,
            'message' => 'Payment ' . $status,
            'transaction_id' => $status === 'completed' ? 'MPESA_' . Str::random(10) : null,
            'receipt_no' => $status === 'completed' ? 'RCPT' . rand(100000, 999999) : null,
        ]);
    }
    
    /**
     * Process card payment
     */
    public function card(Request $request)
    {
        $request->merge(['payment_method' => 'card']);
        
        $validator = Validator::make($request->all(), [
            'card_number' => 'required|string|min:16|max:19',
            'card_holder' => 'required|string',
            'card_expiry' => 'required|string',
            'card_cvv' => 'required|string|min:3|max:4',
            'card_type' => 'nullable|in:visa,mastercard,amex',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        // In production, integrate with payment gateway
        // For now, simulate card processing
        
        return $this->process($request);
    }
    
    /**
     * Process credit sale
     */
    public function credit(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'terms' => 'required|integer|min:1|max:90',
            'due_date' => 'required|date',
            'credit_limit_check' => 'required|boolean',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $data = $validator->validated();
        
        // Check customer credit limit
        if ($data['credit_limit_check']) {
            $customer = Customer::find($data['customer_id']);
            $creditLimit = $customer->credit_limit ?? 0;
            $creditBalance = $customer->credit_balance ?? 0;
            
            if ($creditLimit > 0 && ($creditBalance + $request->amount) > $creditLimit) {
                return response()->json([
                    'success' => false,
                    'message' => "Credit limit exceeded. Available credit: KES " . ($creditLimit - $creditBalance)
                ], 400);
            }
        }
        
        $request->merge([
            'payment_method' => 'credit',
            'notes' => 'Credit sale - Due: ' . $data['due_date'] . ', Terms: ' . $data['terms'] . ' days'
        ]);
        
        return $this->process($request);
    }
    
    /**
     * Process split payment
     */
    public function split(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'split_payments' => 'required|array|min:2',
            'split_payments.*.method' => 'required|string|exists:payment_methods,code',
            'split_payments.*.amount' => 'required|numeric|min:0.01',
            'split_payments.*.details' => 'nullable|array',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $data = $validator->validated();
        
        // Verify total matches
        $total = array_sum(array_column($data['split_payments'], 'amount'));
        
        if (abs($total - $request->amount) > 0.01) {
            return response()->json([
                'success' => false,
                'message' => 'Split total does not match sale total'
            ], 422);
        }
        
        // Process main payment
        $response = $this->process($request);
        
        if ($response->getData()->success) {
            $saleId = $response->getData()->sale->id;
            
            // Record split payment details
            foreach ($data['split_payments'] as $split) {
                $method = PaymentMethod::where('code', $split['method'])->first();
                
                Payment::create([
                    'sale_id' => $saleId,
                    'payment_method_id' => $method->id,
                    'amount' => $split['amount'],
                    'reference' => 'SPLIT_' . $this->generatePaymentReference($method->code),
                    'status' => 'completed',
                    'payment_details' => $split['details'] ?? [],
                    'user_id' => auth()->id(),
                ]);
            }
        }
        
        return $response;
    }
    
    /**
     * Verify a payment
     */
    public function verify(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'reference' => 'required|string',
            'payment_method' => 'required|string|exists:payment_methods,code',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        try {
            $payment = Payment::where('reference', $request->reference)
                ->whereHas('method', function($q) use ($request) {
                    $q->where('code', $request->payment_method);
                })
                ->first();
            
            if (!$payment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment not found'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'payment' => $payment,
                'sale' => $payment->sale,
                'status' => $payment->status,
                'verified' => $payment->status === 'completed',
            ]);
            
        } catch (\Exception $e) {
            Log::error('Payment verification error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Verification failed'
            ], 500);
        }
    }
    
    /**
     * Refund a payment
     */
    public function refund(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'payment_id' => 'required|exists:payments,id',
            'amount' => 'required|numeric|min:0.01',
            'reason' => 'required|string|max:255',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        DB::beginTransaction();
        
        try {
            $payment = Payment::findOrFail($request->payment_id);
            
            if ($payment->status !== 'completed') {
                throw new \Exception('Only completed payments can be refunded');
            }
            
            if ($request->amount > $payment->amount) {
                throw new \Exception('Refund amount cannot exceed original payment');
            }
            
            // Create refund payment
            $refund = Payment::create([
                'sale_id' => $payment->sale_id,
                'payment_method_id' => $payment->payment_method_id,
                'amount' => -$request->amount, // Negative for refund
                'reference' => 'REFUND_' . $payment->reference,
                'status' => 'completed',
                'notes' => $request->reason,
                'payment_details' => ['original_payment_id' => $payment->id],
                'user_id' => auth()->id(),
            ]);
            
            // Update sale if fully refunded
            if ($request->amount == $payment->amount) {
                $payment->sale->update(['payment_status' => 'refunded']);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Refund processed successfully',
                'refund_id' => $refund->id,
                'refund_reference' => $refund->reference,
                'amount' => $request->amount,
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Refund error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Refund failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get payment status
     */
    public function status($reference)
    {
        try {
            $payment = Payment::where('reference', $reference)->firstOrFail();
            
            return response()->json([
                'success' => true,
                'reference' => $reference,
                'status' => $payment->status,
                'amount' => $payment->amount,
                'method' => $payment->method->name,
                'created_at' => $payment->created_at->format('Y-m-d H:i:s'),
                'sale_invoice' => $payment->sale->invoice_no ?? null,
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Payment not found'
            ], 404);
        }
    }
    
    /**
     * Get payment history
     */
    public function history(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'method' => 'nullable|string|exists:payment_methods,code',
            'status' => 'nullable|string|in:pending,completed,failed,refunded',
            'limit' => 'nullable|integer|min:1|max:100',
        ]);
        
        $query = Payment::with(['method', 'sale.customer', 'user'])
            ->where('amount', '>', 0) // Exclude refunds
            ->orderBy('created_at', 'desc');
        
        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->method) {
            $query->whereHas('method', function($q) use ($request) {
                $q->where('code', $request->method);
            });
        }
        
        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        $payments = $query->paginate($request->limit ?? 20);
        
        return response()->json([
            'success' => true,
            'payments' => $payments,
            'summary' => [
                'total' => $payments->total(),
                'total_amount' => $payments->sum('amount'),
                'today_count' => Payment::whereDate('created_at', today())->count(),
                'today_amount' => Payment::whereDate('created_at', today())->sum('amount'),
            ]
        ]);
    }
    
    /**
     * Get today's payments
     */
    public function today()
    {
        $today = today();
        
        $payments = Payment::with(['method', 'sale'])
            ->whereDate('created_at', $today)
            ->where('amount', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get();
        
        $byMethod = Payment::whereDate('created_at', $today)
            ->where('amount', '>', 0)
            ->select('payment_method_id', DB::raw('count(*) as count'), DB::raw('sum(amount) as total'))
            ->groupBy('payment_method_id')
            ->get();
        
        return response()->json([
            'success' => true,
            'date' => $today->format('Y-m-d'),
            'payments' => $payments,
            'summary' => [
                'count' => $payments->count(),
                'total' => $payments->sum('amount'),
                'average' => $payments->count() > 0 ? $payments->sum('amount') / $payments->count() : 0,
            ],
            'by_method' => $byMethod->map(function($item) {
                $method = PaymentMethod::find($item->payment_method_id);
                return [
                    'method' => $method ? $method->name : 'Unknown',
                    'count' => $item->count,
                    'total' => $item->total,
                ];
            }),
        ]);
    }
    
    /**
     * Get dashboard payment stats
     */
    public function dashboardStats(Request $request)
    {
        $today = today();
        $yesterday = today()->subDay();
        
        // Today's stats
        $todayPayments = Payment::whereDate('created_at', $today)
            ->where('amount', '>', 0)
            ->where('status', 'completed');
        
        $yesterdayPayments = Payment::whereDate('created_at', $yesterday)
            ->where('amount', '>', 0)
            ->where('status', 'completed');
        
        $todayTotal = $todayPayments->sum('amount');
        $yesterdayTotal = $yesterdayPayments->sum('amount');
        
        // Method distribution
        $methodDistribution = Payment::whereDate('created_at', $today)
            ->where('amount', '>', 0)
            ->select('payment_method_id', DB::raw('sum(amount) as total'))
            ->groupBy('payment_method_id')
            ->get()
            ->map(function($item) {
                $method = PaymentMethod::find($item->payment_method_id);
                return [
                    'name' => $method ? $method->name : 'Unknown',
                    'total' => $item->total,
                    'percentage' => 0, // Will calculate below
                ];
            });
        
        // Calculate percentages
        $total = $methodDistribution->sum('total');
        $methodDistribution = $methodDistribution->map(function($item) use ($total) {
            $item['percentage'] = $total > 0 ? round(($item['total'] / $total) * 100, 1) : 0;
            return $item;
        });
        
        // Hourly distribution for today
        $hourlyData = [];
        for ($i = 0; $i < 24; $i++) {
            $hourStart = $today->copy()->addHours($i);
            $hourEnd = $today->copy()->addHours($i + 1);
            
            $hourTotal = Payment::whereBetween('created_at', [$hourStart, $hourEnd])
                ->where('amount', '>', 0)
                ->sum('amount');
            
            $hourlyData[] = [
                'hour' => $hourStart->format('H:00'),
                'total' => $hourTotal,
                'count' => Payment::whereBetween('created_at', [$hourStart, $hourEnd])
                    ->where('amount', '>', 0)
                    ->count(),
            ];
        }
        
        return response()->json([
            'success' => true,
            'stats' => [
                'today_total' => $todayTotal,
                'yesterday_total' => $yesterdayTotal,
                'change_percent' => $yesterdayTotal > 0 
                    ? round((($todayTotal - $yesterdayTotal) / $yesterdayTotal) * 100, 1)
                    : 0,
                'today_count' => $todayPayments->count(),
                'average_ticket' => $todayPayments->count() > 0 
                    ? round($todayTotal / $todayPayments->count(), 2)
                    : 0,
                'method_distribution' => $methodDistribution,
                'hourly_data' => $hourlyData,
                'top_payments' => $todayPayments
                    ->orderBy('amount', 'desc')
                    ->limit(5)
                    ->get()
                    ->map(function($payment) {
                        return [
                            'id' => $payment->id,
                            'amount' => $payment->amount,
                            'method' => $payment->method->name,
                            'reference' => $payment->reference,
                            'time' => $payment->created_at->format('H:i'),
                        ];
                    }),
            ],
        ]);
    }
    
    /**
     * Get payment settings
     */
    public function getSettings()
    {
        $settings = [
            'default_vat_rate' => config('pos.vat_rate', 16),
            'currency' => 'KES',
            'cash_rounding' => config('pos.cash_rounding', true),
            'round_to' => config('pos.round_to', 5),
            'mpesa_enabled' => config('pos.mpesa_enabled', true),
            'mpesa_business_shortcode' => config('pos.mpesa_business_shortcode'),
            'mpesa_passkey' => config('pos.mpesa_passkey'),
            'card_processing_enabled' => config('pos.card_processing_enabled', false),
            'auto_print_receipt' => config('pos.auto_print_receipt', true),
            'require_customer_for_credit' => config('pos.require_customer_for_credit', true),
            'max_split_payments' => config('pos.max_split_payments', 3),
            'credit_terms' => [7, 14, 30, 60, 90],
            'receipt_footer' => config('pos.receipt_footer', 'Thank you for shopping with us!'),
        ];
        
        return response()->json([
            'success' => true,
            'settings' => $settings,
        ]);
    }
    
    // ==================== PRIVATE HELPER METHODS ====================
    
    private function validatePaymentMethod(PaymentMethod $method, array $data)
    {
        switch ($method->code) {
            case 'cash':
                if (!isset($data['cash_received']) || $data['cash_received'] < $data['amount']) {
                    throw new \Exception('Cash received must be greater than or equal to amount');
                }
                break;
                
            case 'mpesa':
                if (!isset($data['mpesa_phone'])) {
                    throw new \Exception('Phone number required for M-Pesa');
                }
                
                $phone = $this->formatPhone($data['mpesa_phone']);
                if (!$phone) {
                    throw new \Exception('Invalid phone number format');
                }
                break;
                
            case 'card':
                if (!isset($data['card_number']) || strlen($data['card_number']) < 16) {
                    throw new \Exception('Valid card number required');
                }
                break;
                
            case 'bank':
                if (!isset($data['bank_reference'])) {
                    throw new \Exception('Bank reference required');
                }
                break;
                
            case 'credit':
                if (!isset($data['customer_id'])) {
                    throw new \Exception('Customer required for credit sale');
                }
                break;
        }
    }
    
    private function handlePaymentMethodLogic(PaymentMethod $method, Payment $payment, array $data)
    {
        switch ($method->code) {
            case 'cash':
                // Record cash received and change
                $change = $data['cash_received'] - $payment->amount;
                $payment->payment_details = array_merge($payment->payment_details, [
                    'cash_received' => $data['cash_received'],
                    'change' => $change,
                ]);
                $payment->save();
                break;
                
            case 'mpesa':
                // Record M-Pesa transaction
                $mpesa = MpesaTransaction::create([
                    'transaction_id' => $data['mpesa_transaction_id'] ?? 'MPESA_' . time(),
                    'receipt_no' => $data['mpesa_receipt_no'] ?? null,
                    'amount' => $payment->amount,
                    'phone' => $this->formatPhone($data['mpesa_phone']),
                    'transaction_date' => now(),
                    'status' => 'completed',
                    'type' => 'stk',
                    'payment_id' => $payment->id,
                    'user_id' => auth()->id(),
                ]);
                break;
                
            case 'card':
                // Record card details (masked)
                $payment->payment_details = array_merge($payment->payment_details, [
                    'card_last4' => substr($data['card_number'], -4),
                    'card_type' => $data['card_type'] ?? 'unknown',
                    'auth_code' => $data['card_auth_code'] ?? null,
                ]);
                $payment->save();
                break;
        }
    }
    
    private function generateInvoiceNumber()
    {
        $date = date('Ymd');
        $lastSale = Sale::whereDate('created_at', today())->orderBy('id', 'desc')->first();
        
        if ($lastSale && str_starts_with($lastSale->invoice_no, 'INV-' . $date)) {
            $lastNum = intval(substr($lastSale->invoice_no, -4));
            $nextNum = str_pad($lastNum + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $nextNum = '0001';
        }
        
        return 'INV-' . $date . '-' . $nextNum;
    }
    
    private function generatePaymentReference($methodCode)
    {
        $prefix = strtoupper($methodCode);
        return $prefix . '-' . date('YmdHis') . '-' . Str::random(6);
    }
    
    private function getPaymentDetails(PaymentMethod $method, array $data)
    {
        $details = [];
        
        switch ($method->code) {
            case 'cash':
                $details['cash_received'] = $data['cash_received'] ?? null;
                $details['change'] = isset($data['cash_received']) 
                    ? $data['cash_received'] - $data['amount'] 
                    : 0;
                break;
                
            case 'mpesa':
                $details['phone'] = $this->formatPhone($data['mpesa_phone'] ?? '');
                $details['transaction_id'] = $data['mpesa_transaction_id'] ?? null;
                break;
                
            case 'card':
                $details['card_last4'] = isset($data['card_number']) 
                    ? substr($data['card_number'], -4) 
                    : null;
                $details['card_type'] = $data['card_type'] ?? null;
                break;
                
            case 'bank':
                $details['bank_name'] = $data['bank_name'] ?? null;
                $details['reference'] = $data['bank_reference'] ?? null;
                break;
        }
        
        return $details;
    }
    
    private function generateReceiptData(Sale $sale, Payment $payment)
    {
        return [
            'id' => $sale->id,
            'invoice_no' => $sale->invoice_no,
            'date' => $sale->sale_date->format('Y-m-d H:i:s'),
            'customer' => $sale->customer ? [
                'name' => $sale->customer->name,
                'phone' => $sale->customer->phone,
            ] : null,
            'items' => $sale->items->map(function($item) {
                return [
                    'name' => $item->product->name,
                    'quantity' => $item->quantity,
                    'price' => $item->unit_price,
                    'total' => $item->total,
                ];
            }),
            'totals' => [
                'subtotal' => $sale->subtotal,
                'discount' => $sale->discount,
                'vat_rate' => $sale->vat_rate,
                'vat_amount' => $sale->vat_amount,
                'total' => $sale->total,
            ],
            'payment' => [
                'method' => $payment->method->name,
                'reference' => $payment->reference,
                'amount' => $payment->amount,
                'status' => $payment->status,
            ],
            'store' => [
                'name' => config('app.name'),
                'address' => config('pos.store_address'),
                'phone' => config('pos.store_phone'),
                'vat_number' => config('pos.vat_number'),
            ],
            'cashier' => auth()->user()->name,
        ];
    }
    
    private function formatPhone($phone)
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($phone) === 9 && preg_match('/^[17]\d{8}$/', $phone)) {
            return '0' . $phone;
        } elseif (strlen($phone) === 10 && str_starts_with($phone, '0')) {
            return $phone;
        } elseif (strlen($phone) === 12 && str_starts_with($phone, '254')) {
            return '0' . substr($phone, 3);
        }
        
        return $phone;
    }
}