<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Payment;
use App\Models\Product;
use App\Models\Customer;
use App\Models\HoldSale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class PaymentApiController extends Controller
{
    /**
     * Process payment and save transaction
     */
    public function processPayment(Request $request)
    {
        DB::beginTransaction();
        
        try {
            Log::info('Processing payment request:', $request->all());
            
            $validator = Validator::make($request->all(), [
                'cart' => 'required|array',
                'cart.*.id' => 'required|exists:products,id',
                'cart.*.quantity' => 'required|numeric|min:1',
                'cart.*.price' => 'required|numeric|min:0',
                'payment_method' => 'required|string|in:cash,mpesa,card,credit,multiple',
                'customer_id' => 'nullable|exists:customers,id',
                'amount_paid' => 'required|numeric|min:0',
                'cash_received' => 'nullable|numeric|min:0',
                'change' => 'nullable|numeric|min:0',
                'mpesa_phone' => 'nullable|string|max:20',
                'mpesa_transaction_id' => 'nullable|string|max:100',
                'card_transaction_id' => 'nullable|string|max:100',
                'bank_reference' => 'nullable|string|max:100',
                'discount_amount' => 'nullable|numeric|min:0',
                'notes' => 'nullable|string|max:500',
                'hold_sale_id' => 'nullable|exists:hold_sales,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();
            
            // Calculate totals
            $subtotal = 0;
            $vatTotal = 0;
            
            foreach ($validated['cart'] as $item) {
                $product = Product::find($item['id']);
                if (!$product) {
                    throw new \Exception("Product not found: {$item['id']}");
                }
                
                $itemTotal = $item['quantity'] * $item['price'];
                $subtotal += $itemTotal;
                
                if ($product->has_vat) {
                    $vatRate = $product->tax_rate ?? 16.00;
                    $vatTotal += $itemTotal * ($vatRate / 100);
                }
            }
            
            $discount = $validated['discount_amount'] ?? 0;
            $grandTotal = $subtotal + $vatTotal - $discount;
            
            // Validate payment amount
            if ($validated['amount_paid'] < $grandTotal) {
                throw new \Exception("Insufficient payment. Required: {$grandTotal}, Paid: {$validated['amount_paid']}");
            }
            
            // Generate invoice number
            $invoiceNo = $this->generateInvoiceNumber();
            
            // Create sale record
            $sale = Sale::create([
                'customer_id' => $validated['customer_id'] ?? null,
                'invoice_no' => $invoiceNo,
                'sale_date' => now(),
                'subtotal' => $subtotal,
                'vat_amount' => $vatTotal,
                'discount_amount' => $discount,
                'grand_total' => $grandTotal,
                'vat_included' => true,
                'vat_rate' => 16.00,
                'status' => 'completed',
                'payment_status' => 'paid',
                'payment_method' => $validated['payment_method'],
                'cash_received' => $validated['cash_received'] ?? null,
                'change' => $validated['change'] ?? 0,
                'notes' => $validated['notes'] ?? null,
                'created_by' => auth()->id(),
                'is_etims_invoice' => false,
            ]);
            
            // Create sale items and update stock
            foreach ($validated['cart'] as $item) {
                $product = Product::find($item['id']);
                
                // Calculate item tax
                $itemTotal = $item['quantity'] * $item['price'];
                $itemVat = $product->has_vat ? $itemTotal * (($product->tax_rate ?? 16.00) / 100) : 0;
                
                SaleItem::create([
                    'sale_id' => $sale->id,
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'sku' => $product->sku,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['price'],
                    'tax_percent' => $product->has_vat ? ($product->tax_rate ?? 16.00) : 0,
                    'tax_amount' => $itemVat,
                    'total' => $itemTotal,
                    'batch_id' => null,
                    'batch_number' => null,
                ]);
                
                // Update product stock
                if ($product->track_inventory) {
                    $product->decrement('stock', $item['quantity']);
                    
                    // Update stock status
                    if ($product->stock <= 0) {
                        $product->stock_status = 'out_of_stock';
                    } elseif ($product->stock <= $product->minimum_stock) {
                        $product->stock_status = 'low_stock';
                    }
                    $product->save();
                }
            }
            
            // Create payment record
            $payment = Payment::create([
                'sale_id' => $sale->id,
                'customer_id' => $validated['customer_id'] ?? null,
                'amount' => $validated['amount_paid'],
                'payment_method' => $validated['payment_method'],
                'reference' => $this->getPaymentReference($validated, $invoiceNo),
                'transaction_id' => $validated['mpesa_transaction_id'] ?? $validated['card_transaction_id'] ?? null,
                'phone' => $validated['mpesa_phone'] ?? null,
                'cash_received' => $validated['cash_received'] ?? null,
                'change' => $validated['change'] ?? 0,
                'status' => 'completed',
                'payment_date' => now(),
                'received_by' => auth()->id(),
                'notes' => $validated['notes'] ?? null,
            ]);
            
            // Update customer statistics
            if ($validated['customer_id']) {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    $customer->total_orders = $customer->total_orders + 1;
                    $customer->total_spent = $customer->total_spent + $grandTotal;
                    $customer->last_purchase_date = now();
                    $customer->save();
                }
            }
            
            // Delete hold sale if exists
            if (!empty($validated['hold_sale_id'])) {
                HoldSale::where('id', $validated['hold_sale_id'])->delete();
            }
            
            DB::commit();
            
            Log::info('Payment processed successfully:', [
                'sale_id' => $sale->id,
                'invoice_no' => $invoiceNo,
                'total' => $grandTotal,
                'payment_method' => $validated['payment_method']
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Payment processed successfully',
                'data' => [
                    'sale' => [
                        'id' => $sale->id,
                        'invoice_no' => $sale->invoice_no,
                        'total' => $sale->grand_total,
                        'date' => $sale->sale_date,
                        'cash_received' => $sale->cash_received,
                        'change' => $sale->change,
                    ],
                    'payment' => [
                        'id' => $payment->id,
                        'reference' => $payment->reference,
                        'method' => $payment->payment_method,
                    ]
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Payment processing error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'Payment failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Verify payment
     */
    public function verify(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'transaction_id' => 'required|string',
                'payment_method' => 'required|string|in:mpesa,card,bank_transfer',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // In a real application, you would verify with the payment gateway
            // For now, we'll simulate successful verification
            
            return response()->json([
                'success' => true,
                'message' => 'Payment verified successfully',
                'verified' => true,
                'transaction_id' => $request->transaction_id,
                'timestamp' => now()->toISOString(),
            ]);
            
        } catch (\Exception $e) {
            Log::error('Payment verification error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Verification failed'
            ], 500);
        }
    }
    
    /**
     * Generate invoice number
     */
    private function generateInvoiceNumber(): string
    {
        $prefix = 'INV';
        $year = date('Y');
        $month = date('m');
        $day = date('d');
        
        $lastInvoice = Sale::whereDate('created_at', today())
            ->orderBy('id', 'desc')
            ->first();
        
        $sequence = $lastInvoice ? (int) substr($lastInvoice->invoice_no, -4) + 1 : 1;
        
        return sprintf('%s-%s%s%s-%04d', $prefix, $year, $month, $day, $sequence);
    }
    
    /**
     * Get payment reference
     */
    private function getPaymentReference(array $data, string $invoiceNo): string
    {
        switch ($data['payment_method']) {
            case 'mpesa':
                return $data['mpesa_transaction_id'] ?? $invoiceNo . '-MPESA';
            case 'card':
                return $data['card_transaction_id'] ?? $invoiceNo . '-CARD';
            case 'bank_transfer':
                return $data['bank_reference'] ?? $invoiceNo . '-BANK';
            default:
                return $invoiceNo;
        }
    }
}