<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Api\ApiBaseController;
use Illuminate\Http\Request;
use App\Models\HoldSale;
use App\Models\HoldSaleItem;
use App\Models\Product;
use App\Models\Customer;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class HoldSaleApiController extends ApiBaseController
{
    /**
     * Get all hold sales
     */
    public function index(Request $request)
    {
        try {
            $holds = HoldSale::with(['customer', 'items.product', 'user'])
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($hold) {
                    return [
                        'id' => $hold->id,
                        'reference' => $hold->reference,
                        'name' => $hold->name,
                        'customer_id' => $hold->customer_id,
                        'customer_name' => $hold->customer->name ?? 'Walk-in',
                        'items' => $hold->items->map(function ($item) {
                            return [
                                'product_id' => $item->product_id,
                                'name' => $item->product->name ?? 'Unknown Product',
                                'sku' => $item->product->sku ?? 'N/A',
                                'barcode' => $item->product->barcode ?? 'N/A',
                                'quantity' => $item->quantity,
                                'price' => $item->price,
                                'total' => $item->total,
                                'tax_rate' => $item->tax_rate,
                                'is_vatable' => $item->is_vatable
                            ];
                        }),
                        'item_count' => $hold->items->count(),
                        'subtotal' => $hold->subtotal,
                        'vat' => $hold->vat,
                        'discount' => $hold->discount,
                        'total' => $hold->total,
                        'status' => $hold->status,
                        'notes' => $hold->notes,
                        'held_by' => $hold->user_id,
                        'held_by_name' => $hold->user->name ?? 'Unknown',
                        'created_at' => $hold->created_at,
                        'expires_at' => $hold->expires_at,
                        'is_expired' => Carbon::parse($hold->expires_at)->isPast()
                    ];
                });
            
            return $this->successResponse($holds, 'Hold sales retrieved successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to load hold sales');
        }
    }
    
    /**
     * Store a new hold sale
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validated = $request->validate([
                'name' => 'nullable|string|max:255',
                'customer_id' => 'nullable|exists:customers,id',
                'items' => 'required|array',
                'items.*.product_id' => 'required|exists:products,id',
                'items.*.name' => 'required|string',
                'items.*.quantity' => 'required|numeric|min:1',
                'items.*.price' => 'required|numeric|min:0',
                'items.*.tax_rate' => 'nullable|numeric|min:0',
                'items.*.is_vatable' => 'nullable|boolean',
                'subtotal' => 'required|numeric|min:0',
                'vat' => 'required|numeric|min:0',
                'discount' => 'nullable|numeric|min:0',
                'total' => 'required|numeric|min:0',
                'notes' => 'nullable|string'
            ]);
            
            // Generate reference
            $reference = 'HOLD-' . strtoupper(uniqid());
            
            // Create hold sale
            $holdSale = HoldSale::create([
                'reference' => $reference,
                'name' => $validated['name'] ?? 'Hold ' . Carbon::now()->format('H:i:s'),
                'customer_id' => $validated['customer_id'] ?? null,
                'user_id' => auth()->id(),
                'subtotal' => $validated['subtotal'],
                'vat' => $validated['vat'],
                'discount' => $validated['discount'] ?? 0,
                'total' => $validated['total'],
                'status' => 'active',
                'notes' => $validated['notes'] ?? null,
                'expires_at' => Carbon::now()->addHours(24) // Expire in 24 hours
            ]);
            
            // Create hold sale items
            foreach ($validated['items'] as $item) {
                HoldSaleItem::create([
                    'hold_sale_id' => $holdSale->id,
                    'product_id' => $item['product_id'],
                    'name' => $item['name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'total' => $item['price'] * $item['quantity'],
                    'tax_rate' => $item['tax_rate'] ?? 16,
                    'is_vatable' => $item['is_vatable'] ?? false
                ]);
            }
            
            DB::commit();
            
            return $this->successResponse([
                'id' => $holdSale->id,
                'reference' => $holdSale->reference,
                'name' => $holdSale->name,
                'customer_id' => $holdSale->customer_id,
                'total' => $holdSale->total,
                'items_count' => count($validated['items']),
                'expires_at' => $holdSale->expires_at
            ], 'Sale held successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to hold sale');
        }
    }
    
    /**
     * Load a hold sale into cart
     */
    public function load($id)
    {
        try {
            $holdSale = HoldSale::with(['items.product', 'customer'])->findOrFail($id);
            
            // Check if expired
            if (Carbon::parse($holdSale->expires_at)->isPast()) {
                return $this->errorResponse('Hold sale has expired', [], 400);
            }
            
            // Check if already converted
            if ($holdSale->status === 'converted') {
                return $this->errorResponse('Hold sale has already been converted', [], 400);
            }
            
            // Prepare items for cart
            $items = $holdSale->items->map(function ($item) {
                $product = $item->product;
                
                return [
                    'id' => $product->id,
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'barcode' => $product->barcode,
                    'price' => $item->price,
                    'cost_price' => $product->cost_price,
                    'quantity' => $item->quantity,
                    'tax_rate' => $item->tax_rate,
                    'is_vatable' => $item->is_vatable,
                    'stock' => $product->stock,
                    'category_id' => $product->category_id
                ];
            });
            
            // Prepare customer data if exists
            $customer = null;
            if ($holdSale->customer) {
                $customer = [
                    'id' => $holdSale->customer->id,
                    'name' => $holdSale->customer->name,
                    'phone' => $holdSale->customer->phone,
                    'email' => $holdSale->customer->email,
                    'customer_type' => $holdSale->customer->customer_type,
                    'credit_limit' => $holdSale->customer->credit_limit,
                    'available_credit' => $holdSale->customer->available_credit,
                    'total_spent' => $holdSale->customer->total_spent,
                    'total_orders' => $holdSale->customer->total_orders
                ];
            }
            
            return $this->successResponse([
                'id' => $holdSale->id,
                'reference' => $holdSale->reference,
                'name' => $holdSale->name,
                'items' => $items,
                'customer' => $customer,
                'subtotal' => $holdSale->subtotal,
                'vat' => $holdSale->vat,
                'discount' => $holdSale->discount,
                'total' => $holdSale->total,
                'notes' => $holdSale->notes
            ], 'Hold sale loaded successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to load hold sale');
        }
    }
    
    /**
     * Get hold sale statistics
     */
    public function stats()
    {
        try {
            $today = Carbon::today();
            
            $totalHolds = HoldSale::count();
            $activeHolds = HoldSale::where('status', 'active')->count();
            $expiredHolds = HoldSale::where('status', 'expired')->count();
            $convertedHolds = HoldSale::where('status', 'converted')->count();
            
            // Holds created today
            $todaysHolds = HoldSale::whereDate('created_at', $today)->count();
            
            // Holds expiring soon (within 1 hour)
            $expiringSoon = HoldSale::where('status', 'active')
                ->where('expires_at', '<=', Carbon::now()->addHour())
                ->where('expires_at', '>', Carbon::now())
                ->count();
            
            // Total value of active holds
            $totalValue = HoldSale::where('status', 'active')->sum('total');
            
            return $this->successResponse([
                'total_holds' => $totalHolds,
                'active_holds' => $activeHolds,
                'expired_holds' => $expiredHolds,
                'converted_holds' => $convertedHolds,
                'todays_holds' => $todaysHolds,
                'expiring_soon' => $expiringSoon,
                'total_value' => $totalValue
            ], 'Hold sale statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to load hold sale statistics');
        }
    }
    
    /**
     * Delete a hold sale
     */
    public function destroy($id)
    {
        try {
            $holdSale = HoldSale::findOrFail($id);
            
            // Delete items first
            $holdSale->items()->delete();
            
            // Delete hold sale
            $holdSale->delete();
            
            return $this->successResponse([], 'Hold sale deleted successfully');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to delete hold sale');
        }
    }
}