<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ActivityLogController extends Controller
{
    /**
     * Display a listing of the activity logs.
     */
    public function index(Request $request)
    {
        try {
            // For AJAX/API requests
            if ($request->ajax() || $request->is('api/*') || $request->expectsJson()) {
                return $this->getActivityLogsData($request);
            }

            // For web requests - get filter options
            $logTypes = ActivityLog::distinct('log_name')->orderBy('log_name')->pluck('log_name');
            $events = ActivityLog::distinct('event')->orderBy('event')->pluck('event');
            $users = User::whereIn('id', ActivityLog::distinct('causer_id')->pluck('causer_id'))
                ->orderBy('name')
                ->get();

            return view('activities.index', compact('logTypes', 'events', 'users'));

        } catch (\Exception $e) {
            \Log::error('Activity Log Index Error: ' . $e->getMessage());

            if ($request->ajax() || $request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to load activity logs',
                    'error' => config('app.debug') ? $e->getMessage() : null
                ], 500);
            }

            return redirect()->back()->with('error', 'Failed to load activity logs: ' . $e->getMessage());
        }
    }

    /**
     * Get activity logs data for AJAX/API requests.
     */
    private function getActivityLogsData(Request $request)
    {
        $perPage = $request->input('per_page', 25);
        $search = $request->input('search', '');
        $logName = $request->input('log_name', '');
        $event = $request->input('event', '');
        $userId = $request->input('user_id', '');
        $dateFrom = $request->input('date_from', '');
        $dateTo = $request->input('date_to', '');
        $orderBy = $request->input('order_by', 'created_at');
        $orderDir = $request->input('order_dir', 'desc');

        // Build query
        $query = ActivityLog::with(['causer', 'subject'])
            ->orderBy($orderBy, $orderDir);

        // Apply filters
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('log_name', 'like', "%{$search}%")
                  ->orWhere('event', 'like', "%{$search}%")
                  ->orWhere('properties', 'like', "%{$search}%")
                  ->orWhereHas('causer', function($q2) use ($search) {
                      $q2->where('name', 'like', "%{$search}%")
                         ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($logName) {
            $query->where('log_name', $logName);
        }

        if ($event) {
            $query->where('event', $event);
        }

        if ($userId) {
            $query->where('causer_id', $userId);
        }

        if ($dateFrom) {
            $query->whereDate('created_at', '>=', $dateFrom);
        }

        if ($dateTo) {
            $query->whereDate('created_at', '<=', $dateTo);
        }

        // Use paginate safely
        try {
            $activities = $query->paginate($perPage);
        } catch (\Exception $e) {
            \Log::error('Pagination error: ' . $e->getMessage());
            $activities = collect(); // Return empty collection
        }

        // Transform activities for response
        $transformedActivities = $activities->isEmpty() ? [] : $activities->map(function ($activity) {
            return [
                'id' => $activity->id,
                'log_name' => $activity->log_name,
                'description' => $activity->description,
                'event' => $activity->event,
                'created_at' => $activity->created_at ? $activity->created_at->format('Y-m-d H:i:s') : null,
                'created_at_formatted' => $activity->created_at ? $activity->created_at->format('M d, Y h:i A') : null,
                'causer' => $activity->causer ? [
                    'id' => $activity->causer->id,
                    'name' => $activity->causer->name,
                    'email' => $activity->causer->email,
                    'role' => $activity->causer->roles->first()->name ?? 'User',
                ] : null,
                'subject' => $activity->subject ? [
                    'type' => class_basename($activity->subject_type),
                    'id' => $activity->subject_id,
                    'name' => $activity->subject->name ?? $activity->subject->title ?? 'N/A',
                ] : null,
                'properties' => $activity->properties,
                'ip_address' => $activity->properties['ip_address'] ?? null,
                'user_agent' => $activity->properties['user_agent'] ?? null,
            ];
        });

        return response()->json([
            'success' => true,
            'activities' => $transformedActivities,
            'total' => $activities->total() ?? 0,
            'per_page' => $activities->perPage() ?? $perPage,
            'current_page' => $activities->currentPage() ?? 1,
            'last_page' => $activities->lastPage() ?? 1,
            'from' => $activities->firstItem() ?? 0,
            'to' => $activities->lastItem() ?? 0,
        ]);
    }

    /**
     * Display a single activity log.
     */
    public function show($id)
    {
        try {
            $activity = ActivityLog::with(['causer', 'subject'])->find($id);

            if (!$activity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Activity log not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'activity' => [
                    'id' => $activity->id,
                    'log_name' => $activity->log_name,
                    'description' => $activity->description,
                    'event' => $activity->event,
                    'event_name' => $activity->event_name,
                    'icon' => $activity->icon,
                    'badge_color' => $activity->badge_color,
                    'created_at' => $activity->created_at ? $activity->created_at->format('Y-m-d H:i:s') : null,
                    'created_at_human' => $activity->created_at ? $activity->created_at->diffForHumans() : null,
                    'causer' => $activity->causer ? [
                        'id' => $activity->causer->id,
                        'name' => $activity->causer->name,
                        'email' => $activity->causer->email,
                        'role' => $activity->causer->roles->first()->name ?? 'User',
                    ] : null,
                    'subject' => $activity->subject ? [
                        'type' => class_basename($activity->subject_type),
                        'id' => $activity->subject_id,
                        'name' => $activity->subject->name ?? $activity->subject->title ?? 'N/A',
                    ] : null,
                    'properties' => $activity->properties,
                    'formatted_properties' => $activity->formatted_properties,
                    'ip_address' => $activity->properties['ip_address'] ?? 'N/A',
                    'user_agent' => $activity->properties['user_agent'] ?? 'N/A',
                ],
            ]);

        } catch (\Exception $e) {
            \Log::error('Activity Log Show Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch activity log',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Delete a specific activity log.
     */
    public function destroy($id)
    {
        try {
            $activity = ActivityLog::find($id);

            if (!$activity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Activity log not found'
                ], 404);
            }

            $activity->delete();

            // Log the deletion
            activity()
                ->causedBy(Auth::user())
                ->performedOn($activity)
                ->withProperties(['ip_address' => request()->ip()])
                ->log('deleted activity log');

            return response()->json([
                'success' => true,
                'message' => 'Activity log deleted successfully.',
                'deleted_id' => $id,
            ]);

        } catch (\Exception $e) {
            \Log::error('Activity Log Delete Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete activity log: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete activity logs.
     */
    public function bulkDestroy(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'ids' => 'sometimes|array',
                'ids.*' => 'exists:activity_log,id',
                'filter' => 'sometimes|array',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $ids = $request->input('ids', []);
            $filter = $request->input('filter', []);

            // If no specific IDs provided but filter exists, get IDs from filter
            if (empty($ids) && !empty($filter)) {
                $query = ActivityLog::query();
                
                if (!empty($filter['log_name'])) {
                    $query->where('log_name', $filter['log_name']);
                }
                
                if (!empty($filter['event'])) {
                    $query->where('event', $filter['event']);
                }
                
                if (!empty($filter['date_from'])) {
                    $query->whereDate('created_at', '>=', $filter['date_from']);
                }
                
                if (!empty($filter['date_to'])) {
                    $query->whereDate('created_at', '<=', $filter['date_to']);
                }
                
                if (!empty($filter['older_than_days'])) {
                    $query->where('created_at', '<', now()->subDays($filter['older_than_days']));
                }
                
                $ids = $query->pluck('id')->toArray();
            }

            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No activity logs selected for deletion'
                ], 400);
            }

            $count = ActivityLog::whereIn('id', $ids)->delete();

            // Log the bulk deletion
            activity()
                ->causedBy(Auth::user())
                ->withProperties([
                    'count' => $count,
                    'filter' => $filter,
                    'ip_address' => request()->ip()
                ])
                ->log('bulk deleted activity logs');

            return response()->json([
                'success' => true,
                'message' => "Successfully deleted {$count} activity logs.",
                'deleted_count' => $count,
            ]);

        } catch (\Exception $e) {
            \Log::error('Bulk Delete Activity Logs Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete activity logs: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export activity logs in various formats.
     */
    public function export(Request $request)
    {
        try {
            $format = $request->input('format', 'csv');
            $exportType = $request->input('export_type', 'filtered');
            
            // Validate format
            $allowedFormats = ['csv', 'json', 'excel'];
            if (!in_array($format, $allowedFormats)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid export format. Allowed: csv, json, excel'
                ], 400);
            }
            
            // Build query based on filters
            $query = ActivityLog::with(['causer']);
            
            if ($exportType === 'filtered') {
                $search = $request->input('search', '');
                $logName = $request->input('log_name', '');
                $event = $request->input('event', '');
                $dateFrom = $request->input('date_from', '');
                $dateTo = $request->input('date_to', '');
                
                if ($search) {
                    $query->where('description', 'like', "%{$search}%");
                }
                
                if ($logName) {
                    $query->where('log_name', $logName);
                }
                
                if ($event) {
                    $query->where('event', $event);
                }
                
                if ($dateFrom) {
                    $query->whereDate('created_at', '>=', $dateFrom);
                }
                
                if ($dateTo) {
                    $query->whereDate('created_at', '<=', $dateTo);
                }
            } elseif ($exportType === 'selected') {
                $ids = explode(',', $request->input('ids', ''));
                if (!empty($ids)) {
                    $query->whereIn('id', $ids);
                }
            } elseif ($exportType === 'custom') {
                $startDate = $request->input('start_date', now()->subMonth()->format('Y-m-d'));
                $endDate = $request->input('end_date', now()->format('Y-m-d'));
                $query->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
            }
            
            $activities = $query->orderBy('created_at', 'desc')->get();
            
            if ($activities->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No activity logs found to export'
                ], 404);
            }
            
            // Log the export
            activity()
                ->causedBy(Auth::user())
                ->withProperties([
                    'format' => $format,
                    'count' => $activities->count(),
                    'export_type' => $exportType,
                    'ip_address' => request()->ip()
                ])
                ->log('exported activity logs');

            if ($format === 'csv') {
                return $this->exportToCSV($activities);
            } elseif ($format === 'excel') {
                return $this->exportToExcel($activities);
            } else {
                return $this->exportToJSON($activities);
            }

        } catch (\Exception $e) {
            \Log::error('Activity Log Export Error: ' . $e->getMessage());
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Export failed: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()->with('error', 'Export failed: ' . $e->getMessage());
        }
    }

    /**
     * Export to CSV format.
     */
    private function exportToCSV($activities)
    {
        $filename = 'activity_logs_' . date('Y-m-d_H-i-s') . '.csv';
        
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $file = fopen('php://output', 'w');
        
        // Add BOM for UTF-8
        fwrite($file, "\xEF\xBB\xBF");
        
        // Headers
        fputcsv($file, [
            'ID',
            'Date & Time',
            'Log Type',
            'Event',
            'Description',
            'User',
            'User Email',
            'Subject Type',
            'Subject ID',
            'IP Address',
            'User Agent',
        ]);
        
        // Data
        foreach ($activities as $activity) {
            fputcsv($file, [
                $activity->id,
                $activity->created_at ? $activity->created_at->format('Y-m-d H:i:s') : 'N/A',
                $activity->log_name,
                $activity->event,
                $activity->description,
                $activity->causer ? $activity->causer->name : 'System',
                $activity->causer ? $activity->causer->email : 'N/A',
                $activity->subject_type ?? 'N/A',
                $activity->subject_id ?? 'N/A',
                isset($activity->properties['ip_address']) ? $activity->properties['ip_address'] : 'N/A',
                isset($activity->properties['user_agent']) ? $activity->properties['user_agent'] : 'N/A',
            ]);
        }
        
        fclose($file);
        exit;
    }

    /**
     * Export to Excel format.
     */
    private function exportToExcel($activities)
    {
        $filename = 'activity_logs_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        return $this->exportToCSV($activities);
    }

    /**
     * Export to JSON format.
     */
    private function exportToJSON($activities)
    {
        $filename = 'activity_logs_' . date('Y-m-d_H-i-s') . '.json';
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $data = [
            'export_date' => now()->toISOString(),
            'total_records' => $activities->count(),
            'activities' => $activities->map(function($activity) {
                return [
                    'id' => $activity->id,
                    'log_name' => $activity->log_name,
                    'description' => $activity->description,
                    'event' => $activity->event,
                    'created_at' => $activity->created_at ? $activity->created_at->toISOString() : null,
                    'causer' => $activity->causer ? [
                        'name' => $activity->causer->name,
                        'email' => $activity->causer->email,
                    ] : null,
                    'subject' => $activity->subject ? [
                        'type' => $activity->subject_type,
                        'id' => $activity->subject_id,
                    ] : null,
                    'properties' => $activity->properties,
                ];
            })->toArray(),
        ];
        
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Clean old activity logs automatically.
     */
    public function cleanOldLogs(Request $request)
    {
        try {
            $days = $request->input('days', 90);
            
            if ($days < 7) {
                return response()->json([
                    'success' => false,
                    'message' => 'Minimum days for cleaning logs is 7'
                ], 400);
            }
            
            $count = ActivityLog::where('created_at', '<', now()->subDays($days))->delete();

            // Log the cleanup
            activity()
                ->causedBy(Auth::user())
                ->withProperties([
                    'days' => $days,
                    'count' => $count,
                    'ip_address' => request()->ip()
                ])
                ->log('cleaned old activity logs');

            return response()->json([
                'success' => true,
                'message' => "Successfully cleaned {$count} activity logs older than {$days} days.",
                'cleaned_count' => $count,
            ]);

        } catch (\Exception $e) {
            \Log::error('Clean Old Logs Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to clean old logs: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get activity log statistics.
     */
    public function statistics(Request $request)
    {
        try {
            // First check if the method exists in the model, otherwise calculate here
            $stats = method_exists(ActivityLog::class, 'statistics') 
                ? ActivityLog::statistics() 
                : $this->calculateStatistics();
            
            return response()->json([
                'success' => true,
                'statistics' => $stats,
            ]);

        } catch (\Exception $e) {
            \Log::error('Activity Log Statistics Error: ' . $e->getMessage());
            
            // Return safe default statistics
            return response()->json([
                'success' => true,
                'statistics' => [
                    'total' => 0,
                    'today' => 0,
                    'active_users' => 0,
                    'average_daily' => 0,
                    'top_users' => [],
                ],
                'warning' => 'Statistics calculated with safe defaults',
            ]);
        }
    }
    
    /**
     * Calculate statistics manually (fallback method)
     */
    private function calculateStatistics(): array
    {
        try {
            // Total activities
            $total = ActivityLog::count();
            
            // Today's activities
            $today = ActivityLog::whereDate('created_at', today())->count();
            
            // Unique active users (last 7 days)
            $activeUsers = ActivityLog::whereDate('created_at', '>=', now()->subDays(7))
                ->whereNotNull('causer_id')
                ->distinct('causer_id')
                ->count('causer_id');
            
            // Average daily activities (last 30 days)
            $averageDaily = ActivityLog::whereDate('created_at', '>=', now()->subDays(30))
                ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
                ->groupBy(DB::raw('DATE(created_at)'))
                ->get()
                ->avg('count') ?? 0;
            
            // Top users
            $topUsers = ActivityLog::whereNotNull('causer_id')
                ->select('causer_id', DB::raw('COUNT(*) as count'))
                ->groupBy('causer_id')
                ->with('causer')
                ->orderBy('count', 'desc')
                ->limit(5)
                ->get()
                ->map(function($item) {
                    return [
                        'user' => $item->causer ? $item->causer->name : 'Unknown',
                        'count' => $item->count,
                    ];
                });

            return [
                'total' => $total,
                'today' => $today,
                'active_users' => $activeUsers,
                'average_daily' => round($averageDaily, 1),
                'top_users' => $topUsers,
            ];
        } catch (\Exception $e) {
            \Log::error('Calculate Statistics Error: ' . $e->getMessage());
            return [
                'total' => 0,
                'today' => 0,
                'active_users' => 0,
                'average_daily' => 0,
                'top_users' => [],
            ];
        }
    }

    /**
     * Search activity logs.
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            $type = $request->input('type', 'all');
            $limit = $request->input('limit', 50);
            
            $query = ActivityLog::with(['causer'])
                ->orderBy('created_at', 'desc');
            
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('description', 'like', "%{$search}%")
                      ->orWhere('log_name', 'like', "%{$search}%")
                      ->orWhere('event', 'like', "%{$search}%")
                      ->orWhere('properties', 'like', "%{$search}%")
                      ->orWhereHas('causer', function($q2) use ($search) {
                          $q2->where('name', 'like', "%{$search}%")
                             ->orWhere('email', 'like', "%{$search}%");
                      });
                });
            }
            
            if ($type && $type !== 'all') {
                $query->where('log_name', $type);
            }
            
            $activities = $query->limit($limit)->get();
            
            return response()->json([
                'success' => true,
                'activities' => $activities->map(function($activity) {
                    return [
                        'id' => $activity->id,
                        'date' => $activity->created_at ? $activity->created_at->format('Y-m-d H:i:s') : null,
                        'user' => $activity->causer ? [
                            'name' => $activity->causer->name,
                            'email' => $activity->causer->email,
                        ] : null,
                        'activity' => $activity->description,
                        'type' => $activity->log_name,
                        'event' => $activity->event,
                        'details' => $activity->properties,
                    ];
                }),
                'count' => $activities->count(),
                'search_term' => $search,
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Activity Log Search Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Search failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get available log types for filtering.
     */
    public function getLogTypes()
    {
        try {
            $types = ActivityLog::distinct('log_name')->orderBy('log_name')->pluck('log_name');
            
            return response()->json([
                'success' => true,
                'log_types' => $types,
                'count' => $types->count(),
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Get Log Types Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get log types',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get activity log dashboard stats.
     */
    public function dashboardStats(Request $request)
    {
        try {
            $period = $request->input('period', 'today'); // today, week, month, year
            
            switch ($period) {
                case 'today':
                    $startDate = today();
                    $endDate = now();
                    break;
                case 'week':
                    $startDate = now()->startOfWeek();
                    $endDate = now()->endOfWeek();
                    break;
                case 'month':
                    $startDate = now()->startOfMonth();
                    $endDate = now()->endOfMonth();
                    break;
                case 'year':
                    $startDate = now()->startOfYear();
                    $endDate = now()->endOfYear();
                    break;
                default:
                    $startDate = today();
                    $endDate = now();
            }
            
            $stats = [
                'period' => $period,
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
                'total_activities' => ActivityLog::whereBetween('created_at', [$startDate, $endDate])->count(),
                'unique_users' => ActivityLog::whereBetween('created_at', [$startDate, $endDate])
                    ->whereNotNull('causer_id')
                    ->distinct('causer_id')
                    ->count('causer_id'),
                'top_events' => ActivityLog::whereBetween('created_at', [$startDate, $endDate])
                    ->select('event', DB::raw('COUNT(*) as count'))
                    ->groupBy('event')
                    ->orderBy('count', 'desc')
                    ->limit(10)
                    ->get()
                    ->map(function($item) {
                        return [
                            'event' => $item->event,
                            'count' => $item->count,
                        ];
                    }),
                'activity_by_hour' => $this->getActivityByHour($startDate, $endDate),
            ];
            
            return response()->json([
                'success' => true,
                'stats' => $stats,
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Dashboard Stats Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get dashboard stats: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get activity distribution by hour.
     */
    private function getActivityByHour($startDate, $endDate)
    {
        try {
            $activities = ActivityLog::whereBetween('created_at', [$startDate, $endDate])
                ->select(DB::raw('HOUR(created_at) as hour'), DB::raw('COUNT(*) as count'))
                ->groupBy(DB::raw('HOUR(created_at)'))
                ->orderBy('hour')
                ->get();
            
            $hours = [];
            for ($i = 0; $i < 24; $i++) {
                $hours[$i] = [
                    'hour' => $i,
                    'label' => sprintf('%02d:00', $i),
                    'count' => 0,
                ];
            }
            
            foreach ($activities as $activity) {
                $hour = (int) $activity->hour;
                if (isset($hours[$hour])) {
                    $hours[$hour]['count'] = $activity->count;
                }
            }
            
            return array_values($hours);
            
        } catch (\Exception $e) {
            \Log::error('Get Activity By Hour Error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get activity log summary.
     */
    public function summary()
    {
        try {
            // Get summary for the last 30 days
            $thirtyDaysAgo = now()->subDays(30);
            
            $summary = [
                'total_activities' => ActivityLog::count(),
                'activities_last_30_days' => ActivityLog::where('created_at', '>=', $thirtyDaysAgo)->count(),
                'unique_users_last_30_days' => ActivityLog::where('created_at', '>=', $thirtyDaysAgo)
                    ->whereNotNull('causer_id')
                    ->distinct('causer_id')
                    ->count('causer_id'),
                'most_active_log_type' => ActivityLog::select('log_name', DB::raw('COUNT(*) as count'))
                    ->groupBy('log_name')
                    ->orderBy('count', 'desc')
                    ->first(),
                'most_active_event' => ActivityLog::select('event', DB::raw('COUNT(*) as count'))
                    ->groupBy('event')
                    ->orderBy('count', 'desc')
                    ->first(),
                'recent_activities' => ActivityLog::with('causer')
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get()
                    ->map(function($activity) {
                        return [
                            'id' => $activity->id,
                            'description' => $activity->description,
                            'log_name' => $activity->log_name,
                            'event' => $activity->event,
                            'created_at' => $activity->created_at->format('Y-m-d H:i:s'),
                            'user' => $activity->causer ? $activity->causer->name : 'System',
                        ];
                    }),
            ];
            
            return response()->json([
                'success' => true,
                'summary' => $summary,
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Activity Log Summary Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get activity log summary'
            ], 500);
        }
    }
}