

<?php $__env->startSection('title', 'Point of Sale - Modern Kenyan Supermarket'); ?>

<?php $__env->startSection('style'); ?>
<style>
    .pos-container {
        display: grid;
        grid-template-columns: 3fr 1fr;
        gap: 20px;
        height: calc(100vh - 120px);
        overflow: hidden;
    }
    
    .pos-main {
        display: grid;
        grid-template-rows: auto auto 1fr;
        gap: 15px;
        overflow: hidden;
    }
    
    .pos-header {
        background: linear-gradient(135deg, #2a4365 0%, #1a365d 100%);
        color: white;
        padding: 15px 25px;
        border-radius: 12px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    }
    
    .store-info {
        flex: 1;
    }
    
    .store-name {
        font-size: 20px;
        font-weight: 700;
        margin-bottom: 5px;
        color: #fff;
    }
    
    .store-meta {
        font-size: 12px;
        opacity: 0.9;
    }
    
    .header-stats {
        display: flex;
        gap: 25px;
        font-size: 13px;
    }
    
    .stat-item {
        text-align: center;
        padding: 0 10px;
        position: relative;
    }
    
    .stat-item:not(:last-child)::after {
        content: '';
        position: absolute;
        right: -12px;
        top: 50%;
        transform: translateY(-50%);
        width: 1px;
        height: 30px;
        background: rgba(255, 255, 255, 0.2);
    }
    
    .stat-label {
        font-size: 11px;
        opacity: 0.8;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 3px;
    }
    
    .stat-value {
        font-weight: bold;
        font-size: 16px;
        color: #fff;
    }
    
    .quick-actions {
        display: flex;
        gap: 10px;
    }
    
    .action-btn {
        background: rgba(255, 255, 255, 0.15);
        border: 1px solid rgba(255, 255, 255, 0.2);
        color: white;
        padding: 8px 15px;
        border-radius: 8px;
        font-size: 13px;
        transition: all 0.3s;
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .action-btn:hover {
        background: rgba(255, 255, 255, 0.25);
        transform: translateY(-1px);
    }
    
    .fab-container {
        position: fixed;
        bottom: 30px;
        right: 30px;
        display: flex;
        flex-direction: column;
        gap: 15px;
        z-index: 1000;
    }
    
    .fab {
        width: 60px;
        height: 60px;
        border-radius: 16px;
        background: linear-gradient(135deg, #2a4365 0%, #1a365d 100%);
        color: white;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        box-shadow: 0 6px 20px rgba(42, 67, 101, 0.3);
        transition: all 0.3s;
        font-size: 22px;
        border: 2px solid white;
    }
    
    .fab:hover {
        transform: translateY(-3px) scale(1.05);
        box-shadow: 0 8px 25px rgba(42, 67, 101, 0.4);
    }
    
    .fab-label {
        position: absolute;
        right: 70px;
        background: #2a4365;
        color: white;
        padding: 6px 12px;
        border-radius: 6px;
        font-size: 12px;
        white-space: nowrap;
        opacity: 0;
        transform: translateX(10px);
        transition: all 0.3s;
        pointer-events: none;
        border: 1px solid rgba(255, 255, 255, 0.1);
    }
    
    .fab:hover .fab-label {
        opacity: 1;
        transform: translateX(0);
    }
    
    .keyboard-shortcuts {
        position: fixed;
        bottom: 30px;
        left: 30px;
        background: rgba(42, 67, 101, 0.95);
        color: white;
        padding: 15px;
        border-radius: 12px;
        font-size: 11px;
        backdrop-filter: blur(10px);
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
        z-index: 999;
    }
    
    .shortcut-item {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 8px;
    }
    
    .shortcut-item:last-child {
        margin-bottom: 0;
    }
    
    .shortcut-key {
        background: rgba(255, 255, 255, 0.2);
        padding: 2px 8px;
        border-radius: 4px;
        font-weight: bold;
        min-width: 30px;
        text-align: center;
        font-family: monospace;
    }
    
    .shortcut-desc {
        opacity: 0.9;
    }
    
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        display: none;
        align-items: center;
        justify-content: center;
        z-index: 9999;
        backdrop-filter: blur(4px);
    }
    
    .loading-spinner {
        width: 60px;
        height: 60px;
        border: 3px solid rgba(255, 255, 255, 0.3);
        border-radius: 50%;
        border-top-color: #2a4365;
        animation: spin 1s linear infinite;
    }
    
    @keyframes  spin {
        to { transform: rotate(360deg); }
    }
    
    /* Payment-specific styles */
    .payment-methods-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
        gap: 15px;
        margin-bottom: 20px;
    }
    
    .payment-method-card {
        border: 2px solid #e2e8f0;
        border-radius: 12px;
        padding: 15px;
        cursor: pointer;
        transition: all 0.3s;
        background: white;
        position: relative;
        overflow: hidden;
    }
    
    .payment-method-card:hover {
        border-color: #2a4365;
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(42, 67, 101, 0.15);
    }
    
    .payment-method-card.selected {
        border-color: #2a4365;
        background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
        box-shadow: 0 0 0 3px rgba(42, 67, 101, 0.1);
    }
    
    .method-icon {
        width: 50px;
        height: 50px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
        color: white;
        margin-bottom: 12px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    .method-info { flex: 1; }
    .method-name { 
        font-weight: 700; 
        font-size: 15px; 
        color: #1e293b; 
        margin-bottom: 4px;
    }
    .method-desc { 
        font-size: 13px; 
        color: #64748b; 
        margin-bottom: 6px;
    }
    .method-check { 
        position: absolute;
        top: 10px;
        right: 10px;
        color: #10b981; 
        font-size: 20px; 
        opacity: 0; 
        transition: opacity 0.3s;
    }
    .payment-method-card.selected .method-check { opacity: 1; }
    
    /* Progress Wizard */
    .wizard-progress {
        display: flex;
        justify-content: space-between;
        position: relative;
        margin: 20px 0 30px;
    }
    
    .wizard-progress:before {
        content: '';
        position: absolute;
        top: 20px;
        left: 10%;
        right: 10%;
        height: 3px;
        background: #e2e8f0;
        z-index: 1;
    }
    
    .progress-step { 
        position: relative; 
        z-index: 2; 
        text-align: center; 
        flex: 1; 
    }
    .step-circle {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        background: #e2e8f0;
        color: #64748b;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 8px;
        border: 4px solid white;
        font-weight: bold;
        font-size: 14px;
        transition: all 0.3s;
        position: relative;
        z-index: 2;
    }
    
    .step-circle.active { 
        background: #2a4365; 
        color: white; 
        box-shadow: 0 4px 10px rgba(42, 67, 101, 0.3);
    }
    .step-circle.completed { 
        background: #10b981; 
        color: white; 
        box-shadow: 0 4px 10px rgba(16, 185, 129, 0.3);
    }
    .step-label { 
        font-size: 13px; 
        color: #94a3b8; 
        font-weight: 600; 
        transition: all 0.3s; 
    }
    .step-label.active { 
        color: #2a4365; 
        font-weight: 700; 
    }
    
    @media (max-width: 1200px) {
        .pos-container {
            grid-template-columns: 2fr 1fr;
        }
    }
    
    @media (max-width: 768px) {
        .pos-container {
            grid-template-columns: 1fr;
            height: auto;
        }
        
        .pos-header {
            flex-direction: column;
            gap: 15px;
            text-align: center;
        }
        
        .header-stats {
            width: 100%;
            justify-content: space-around;
        }
        
        .fab-container {
            bottom: 20px;
            right: 20px;
        }
        
        .keyboard-shortcuts {
            display: none;
        }
        
        .payment-methods-grid { 
            grid-template-columns: repeat(2, 1fr); 
            gap: 10px;
        }
    }
    
    @media (max-width: 576px) {
        .payment-methods-grid { 
            grid-template-columns: 1fr; 
        }
    }
    
    /* Safe fallback styles */
    .safe-fallback {
        border: 2px dashed #dc3545;
        padding: 20px;
        border-radius: 8px;
        background: #fff8f8;
    }
    
    .safe-fallback .btn {
        margin: 5px;
    }
    
    .payment-form-template {
        display: none;
    }
    
    .payment-method-form {
        transition: all 0.3s ease;
    }
    
    .payment-method-form.hidden {
        display: none !important;
    }
    
    .payment-method-form.active {
        display: block !important;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper">
    <div class="page-content">
        <!-- Loading Overlay -->
        <div class="loading-overlay" id="loadingOverlay">
            <div class="loading-spinner"></div>
        </div>
        
        <!-- Safe Fallback Container (hidden by default) -->
        <div id="safeFallbackContainer" class="safe-fallback d-none">
            <h5><i class="bx bx-error-circle me-2"></i>POS System Error</h5>
            <p class="mb-3">The POS system encountered an error. Please try refreshing the page.</p>
            <button class="btn btn-sm btn-danger" onclick="location.reload()">
                <i class="bx bx-refresh me-1"></i> Reload Page
            </button>
            <button class="btn btn-sm btn-outline-secondary" onclick="document.getElementById('safeFallbackContainer').classList.add('d-none')">
                <i class="bx bx-x me-1"></i> Dismiss
            </button>
        </div>
        
        <!-- POS Header -->
        <div class="pos-header mb-3">
            <div class="store-info">
                <div class="store-name"><?php echo e(config('app.name', 'Kenyan Supermarket')); ?></div>
                <div class="store-meta">
                    <span><i class="bx bx-map me-1"></i><?php echo e(config('app.address', 'Nairobi CBD')); ?></span>
                    <span class="ms-3"><i class="bx bx-phone me-1"></i><?php echo e(config('app.phone', '0700 000 000')); ?></span>
                </div>
            </div>
            
            <div class="header-stats">
                <div class="stat-item">
                    <div class="stat-label">Today Sales</div>
                    <div class="stat-value" id="todaySalesCount">0</div>
                </div>
                <div class="stat-item">
                    <div class="stat-label">Revenue</div>
                    <div class="stat-value" id="todayRevenue">KES 0</div>
                </div>
                <div class="stat-item">
                    <div class="stat-label">Avg. Sale</div>
                    <div class="stat-value" id="avgSale">KES 0</div>
                </div>
                <div class="stat-item">
                    <div class="stat-label">Customers</div>
                    <div class="stat-value" id="todayCustomers">0</div>
                </div>
            </div>
            
            <div class="quick-actions">
                <button class="action-btn" onclick="safeNewSale()" title="Start New Sale (F3)">
                    <i class="bx bx-receipt"></i>
                    <span>New Sale</span>
                </button>
                <button class="action-btn" onclick="safeShowHeldSales()" title="View Held Sales">
                    <i class="bx bx-history"></i>
                    <span>Held Sales</span>
                </button>
                <button class="action-btn" onclick="safeShowLowStock()" title="Quick Stock Check">
                    <i class="bx bx-package"></i>
                    <span>Stock</span>
                </button>
                <button class="action-btn" onclick="safeShowPaymentModal()" title="Quick Payment (F8)">
                    <i class="bx bx-credit-card"></i>
                    <span>Payment</span>
                </button>
            </div>
        </div>
        
        <!-- Main POS Container -->
        <div class="pos-container">
            <!-- Left: Customer & Products -->
            <div class="pos-main">
                <!-- Customer Section -->
                <?php echo $__env->make('pos.components.pos-customer-section', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
                
                <!-- Quick Actions Bar -->
                <div class="quick-actions-bar card mb-2">
                    <div class="card-body py-2">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="d-flex gap-2">
                                <button class="btn btn-sm btn-outline-primary" onclick="safeOpenScanner()">
                                    <i class="bx bx-barcode me-1"></i> Scan Barcode
                                </button>
                                <button class="btn btn-sm btn-outline-secondary" onclick="safeShowDiscountModal()">
                                    <i class="bx bx-percent me-1"></i> Quick Discount
                                </button>
                                <button class="btn btn-sm btn-outline-info" onclick="safeApplySpecialOffer()">
                                    <i class="bx bx-gift me-1"></i> Special Offer
                                </button>
                                <button class="btn btn-sm btn-outline-success" onclick="safeShowPaymentModal()">
                                    <i class="bx bx-credit-card me-1"></i> Quick Pay
                                </button>
                            </div>
                            <div class="d-flex gap-2">
                                <div class="input-group input-group-sm" style="width: 250px;">
                                    <input type="text" class="form-control" id="globalSearch" 
                                           placeholder="Search products, customers, invoices...">
                                    <button class="btn btn-outline-secondary" type="button" onclick="safeGlobalSearch()">
                                        <i class="bx bx-search"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Product Grid -->
                <div class="card flex-grow-1">
                    <div class="card-body h-100">
                        <?php echo $__env->make('pos.components.pos-product-grid', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
                    </div>
                </div>
            </div>
            
            <!-- Right: Cart & Checkout -->
            <div class="pos-sidebar h-100">
                <?php echo $__env->make('pos.components.pos-cart', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
            </div>
        </div>
        
        <!-- Floating Action Buttons -->
        <div class="fab-container">
            <div class="fab" onclick="safeShowPaymentModal()" title="Quick Payment (F8)">
                <i class="bx bx-credit-card"></i>
                <span class="fab-label">Quick Payment</span>
            </div>
            <div class="fab" onclick="safeHoldCurrentSale()" title="Hold Sale">
                <i class="bx bx-pause"></i>
                <span class="fab-label">Hold Sale</span>
            </div>
            <div class="fab" onclick="printLastReceipt()" title="Reprint Receipt">
                <i class="bx bx-printer"></i>
                <span class="fab-label">Reprint</span>
            </div>
            <div class="fab" onclick="safeOpenCalculator()" title="Calculator">
                <i class="bx bx-calculator"></i>
                <span class="fab-label">Calculator</span>
            </div>
        </div>
        
        <!-- Keyboard Shortcuts -->
        <div class="keyboard-shortcuts">
            <div class="shortcut-item">
                <span class="shortcut-key">F1</span>
                <span class="shortcut-desc">Search Products</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">F2</span>
                <span class="shortcut-desc">Select Customer</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">F3</span>
                <span class="shortcut-desc">New Sale</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">F8</span>
                <span class="shortcut-desc">Payment</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">F9</span>
                <span class="shortcut-desc">M-Pesa Manager</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">ESC</span>
                <span class="shortcut-desc">Close Modal</span>
            </div>
            <div class="shortcut-item">
                <span class="shortcut-key">Ctrl+B</span>
                <span class="shortcut-desc">Barcode Scanner</span>
            </div>
        </div>
    </div>
</div>

<!-- Include All Modals -->
<?php echo $__env->make('pos.components.pos-payment-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-receipt-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-customer-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-discount-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-held-sales-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-barcode-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.pos-calculator-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.manual-mpesa-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>
<?php echo $__env->make('pos.components.mpesa-transactions-modal', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>

<!-- Toast Notifications Container -->
<div id="toastContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>

<?php $__env->stopSection(); ?>

<?php $__env->startSection('script'); ?>

<!-- Load Bootstrap first -->

<!-- Production-Grade POS Initialization System -->
<script>
(function() {
    'use strict';
    
    // =========================================================================
    // 1. SAFE UTILITIES & ERROR HANDLING
    // =========================================================================
    
    // Global error handler
    window.onerror = function(msg, url, lineNo, columnNo, error) {
        console.error('🚨 Global Error:', {msg, url, lineNo, columnNo, error});
        
        // Show user-friendly error
        safeShowFallback('An error occurred. Please refresh the page.');
        
        // Don't prevent default handling
        return false;
    };
    
    // Unhandled promise rejection handler
    window.addEventListener('unhandledrejection', function(event) {
        console.error('🚨 Unhandled Promise Rejection:', event.reason);
        safeShowFallback('System error. Some features may not work properly.');
    });
    
    // Safe fallback display
    window.safeShowFallback = function(message) {
        try {
            const fallback = document.getElementById('safeFallbackContainer');
            if (fallback) {
                fallback.querySelector('p').textContent = message;
                fallback.classList.remove('d-none');
            }
        } catch (e) {
            console.error('Cannot show fallback:', e);
        }
    };
    
    // Safe toast function (works even if POSUtils fails)
    window.safeToast = function(message, type = 'info') {
        try {
            if (window.POSUtils && typeof window.POSUtils.showToast === 'function') {
                window.POSUtils.showToast(message, type);
                return;
            }
            
            // Fallback toast
            console.log(`💬 ${type.toUpperCase()}: ${message}`);
            
            const toastContainer = document.getElementById('toastContainer');
            if (!toastContainer) return;
            
            const toastId = 'toast-' + Date.now();
            const typeClass = type === 'error' ? 'danger' : type;
            
            const toastHtml = `
                <div id="${toastId}" class="toast align-items-center border-0 bg-${typeClass} text-white" role="alert">
                    <div class="d-flex">
                        <div class="toast-body">
                            <i class="bx bx-${type === 'success' ? 'check-circle' : type === 'error' ? 'error-circle' : type === 'warning' ? 'error' : 'info-circle'} me-2"></i>
                            ${message}
                        </div>
                        <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="document.getElementById('${toastId}').remove()"></button>
                    </div>
                </div>
            `;
            
            toastContainer.insertAdjacentHTML('beforeend', toastHtml);
            
            // Auto-remove after 3 seconds
            setTimeout(() => {
                const toast = document.getElementById(toastId);
                if (toast) toast.remove();
            }, 3000);
            
        } catch (error) {
            console.error('Error showing toast:', error);
        }
    };
    
    // =========================================================================
    // 2. SAFE GLOBAL FUNCTIONS (for onclick handlers)
    // =========================================================================
    
    window.safeNewSale = function() {
        try {
            if (window.newSale) {
                window.newSale();
            } else if (window.POSCart && window.POSCart.clear) {
                if (window.posState && window.posState.cart && window.posState.cart.length > 0) {
                    if (!confirm('Start new sale? Current cart will be cleared.')) return;
                }
                window.posState?.clearCart();
                window.POSCart.updateDisplay();
                safeToast('New sale started', 'success');
            } else {
                safeToast('System not ready', 'warning');
            }
        } catch (error) {
            console.error('Error in safeNewSale:', error);
            safeToast('Failed to start new sale', 'error');
        }
    };
    
    window.safeShowPaymentModal = function() {
        try {
            if (window.POSPayments && window.POSPayments.showModal) {
                return window.POSPayments.showModal();
            } else if (window.showPaymentModal) {
                return window.showPaymentModal();
            } else {
                safeToast('Payment system not ready', 'warning');
                return false;
            }
        } catch (error) {
            console.error('Error showing payment modal:', error);
            safeToast('Payment system error', 'error');
            return false;
        }
    };
    
    window.safeOpenScanner = function() {
        try {
            if (window.POSScanner && window.POSScanner.open) {
                window.POSScanner.open();
            } else if (window.openBarcodeScanner) {
                window.openBarcodeScanner();
            } else {
                safeToast('Scanner not available', 'warning');
            }
        } catch (error) {
            console.error('Error opening scanner:', error);
            safeToast('Scanner error', 'error');
        }
    };
    
    window.safeShowHeldSales = function() {
        try {
            if (window.POSHeldSales && window.POSHeldSales.showModal) {
                window.POSHeldSales.showModal();
            } else if (window.showHeldSalesModal) {
                window.showHeldSalesModal();
            } else {
                safeToast('Held sales not available', 'warning');
            }
        } catch (error) {
            console.error('Error showing held sales:', error);
            safeToast('Cannot show held sales', 'error');
        }
    };
    
    window.safeShowLowStock = function() {
        try {
            if (window.POSProducts && window.POSProducts.showLowStock) {
                window.POSProducts.showLowStock();
            } else {
                safeToast('Stock check not available', 'warning');
            }
        } catch (error) {
            console.error('Error showing low stock:', error);
            safeToast('Stock check error', 'error');
        }
    };
    
    window.safeShowDiscountModal = function() {
        try {
            if (window.POSDiscounts && window.POSDiscounts.showModal) {
                window.POSDiscounts.showModal();
            } else {
                safeToast('Discount system not available', 'warning');
            }
        } catch (error) {
            console.error('Error showing discount modal:', error);
            safeToast('Discount error', 'error');
        }
    };
    
    window.safeApplySpecialOffer = function() {
        try {
            if (window.applySpecialOffer) {
                window.applySpecialOffer();
            } else {
                safeToast('Special offers not available', 'warning');
            }
        } catch (error) {
            console.error('Error applying special offer:', error);
            safeToast('Special offer error', 'error');
        }
    };
    
    window.safeGlobalSearch = function() {
        try {
            const query = document.getElementById('globalSearch')?.value.trim();
            if (!query) return;
            
            if (window.POSProducts && window.POSProducts.searchWithPreview) {
                window.POSProducts.searchWithPreview(query);
            } else {
                safeToast('Search not available', 'warning');
            }
        } catch (error) {
            console.error('Error in global search:', error);
            safeToast('Search error', 'error');
        }
    };
    
    window.safeHoldCurrentSale = function() {
        try {
            if (window.POSHeldSales && window.POSHeldSales.holdCurrent) {
                window.POSHeldSales.holdCurrent();
            } else if (window.holdCurrentSale) {
                window.holdCurrentSale();
            } else {
                safeToast('Hold sale not available', 'warning');
            }
        } catch (error) {
            console.error('Error holding sale:', error);
            safeToast('Cannot hold sale', 'error');
        }
    };
    
window.printLastReceipt = function() {
    try {
        console.log('🖨️ Printing last receipt...');
        
        // Get the last sale from localStorage or API
        const lastReceiptData = localStorage.getItem('pos_last_receipt_data');
        
        if (lastReceiptData) {
            try {
                const receipt = JSON.parse(lastReceiptData);
                const invoiceNo = receipt.invoice_no || receipt.receipt_no;
                
                if (invoiceNo) {
                    // Open receipt in new window
                    const receiptWindow = window.open(`/receipt/${invoiceNo}?autoprint=true`, '_blank');
                    
                    // Wait for window to load then print
                    setTimeout(() => {
                        if (receiptWindow && !receiptWindow.closed) {
                            receiptWindow.print();
                        }
                    }, 1000);
                }
            } catch (e) {
                console.error('Error parsing receipt data:', e);
                safeToast('Invalid receipt data', 'error');
            }
        } else {
            // Try to get from API
            fetch('/api/sales/latest')
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.sale) {
                        const receiptWindow = window.open(`/receipt/${data.sale.receipt_no}?autoprint=true`, '_blank');
                        
                        setTimeout(() => {
                            if (receiptWindow && !receiptWindow.closed) {
                                receiptWindow.print();
                            }
                        }, 1000);
                    } else {
                        safeToast('No recent receipt found', 'warning');
                    }
                })
                .catch(error => {
                    console.error('Error fetching last receipt:', error);
                    safeToast('Cannot find last receipt', 'error');
                });
        }
    } catch (error) {
        console.error('Error printing last receipt:', error);
        safeToast('Print failed', 'error');
    }
};
    
    window.safeOpenCalculator = function() {
        try {
            if (window.openCalculator) {
                window.openCalculator();
            } else {
                safeToast('Calculator not available', 'warning');
            }
        } catch (error) {
            console.error('Error opening calculator:', error);
            safeToast('Calculator error', 'error');
        }
    };
    
    // =========================================================================
    // 3. MODULE LOADER WITH PROPER DEPENDENCY MANAGEMENT
    // =========================================================================
    
    class POSModuleManager {
        constructor() {
            this.modules = new Map();
            this.moduleStatus = new Map();
            this.loadingPromises = new Map();
            this.dependencies = {
                'POSUtils': [],
                'POSErrorHandler': ['POSUtils'],
                'POSMonitor': ['POSUtils'],
                'VATService': ['POSUtils'],
                'posState': ['POSUtils', 'POSErrorHandler'],
                'POSCart': ['POSUtils', 'posState', 'VATService'],
                'POSProducts': ['POSUtils', 'posState'],
                'POSCustomers': ['POSUtils', 'posState'],
                'POSScanner': ['POSUtils'],
                'POSDiscounts': ['POSUtils', 'posState', 'POSCart'],
                'POSHeldSales': ['POSUtils', 'posState'],
                'POSStats': ['POSUtils'],
                'POSReceipts': ['POSUtils'],
                'PaymentManager': ['POSUtils', 'posState', 'VATService', 'POSErrorHandler'],
                'POSPayments': ['PaymentManager', 'POSUtils', 'posState']
            };
        }
        
        async loadModule(moduleName) {
            // If already loaded
            if (this.moduleStatus.get(moduleName) === 'loaded') {
                return window[moduleName];
            }
            
            // If already loading
            if (this.loadingPromises.has(moduleName)) {
                return this.loadingPromises.get(moduleName);
            }
            
            // Create loading promise
            const loadPromise = (async () => {
                try {
                    console.log(`📦 Loading module: ${moduleName}`);
                    
                    // Load dependencies first
                    const deps = this.dependencies[moduleName] || [];
                    for (const dep of deps) {
                        await this.loadModule(dep);
                    }
                    
                    // Get module from window (should be loaded by script tag)
                    const module = window[moduleName];
                    
                    if (!module) {
                        throw new Error(`Module ${moduleName} not found`);
                    }
                    
                    // Initialize module if it has init method
                    if (typeof module.init === 'function') {
                        await Promise.resolve(module.init());
                    } else if (typeof module === 'object' && module.init) {
                        await Promise.resolve(module.init());
                    } else if (typeof module === 'function' && module.prototype && module.prototype.init) {
                        const instance = new module();
                        await Promise.resolve(instance.init());
                        window[moduleName] = instance;
                    }
                    
                    this.moduleStatus.set(moduleName, 'loaded');
                    console.log(`✅ Module loaded: ${moduleName}`);
                    
                    return window[moduleName];
                    
                } catch (error) {
                    console.error(`❌ Failed to load module ${moduleName}:`, error);
                    this.moduleStatus.set(moduleName, 'failed');
                    
                    // If critical module fails, show fallback
                    if (['POSUtils', 'posState', 'POSCart', 'PaymentManager'].includes(moduleName)) {
                        safeShowFallback(`Critical module ${moduleName} failed to load. Please refresh.`);
                    }
                    
                    throw error;
                }
            })();
            
            this.loadingPromises.set(moduleName, loadPromise);
            return loadPromise;
        }
        
        async loadAll() {
            const modulesToLoad = [
                'POSUtils',
                'POSErrorHandler',
                'POSMonitor',
                'VATService',
                'posState',
                'POSCart',
                'POSProducts',
                'POSCustomers',
                'POSScanner',
                'POSDiscounts',
                'POSHeldSales',
                'POSStats',
                'POSReceipts',
                'PaymentManager',
                'POSPayments'
            ];
            
            const results = [];
            for (const moduleName of modulesToLoad) {
                try {
                    const module = await this.loadModule(moduleName);
                    results.push({ name: moduleName, success: true, module });
                } catch (error) {
                    console.warn(`⚠️ Module ${moduleName} failed:`, error.message);
                    results.push({ name: moduleName, success: false, error: error.message });
                    
                    // Don't fail completely for non-critical modules
                    if (!['POSUtils', 'posState', 'POSCart', 'PaymentManager'].includes(moduleName)) {
                        continue;
                    }
                }
            }
            
            return results;
        }
        
        isModuleLoaded(moduleName) {
            return this.moduleStatus.get(moduleName) === 'loaded';
        }
        
        getModuleStatus(moduleName) {
            return this.moduleStatus.get(moduleName) || 'not-loaded';
        }
    }
    
    // Create global module manager
    window.POSModuleManager = new POSModuleManager();
    
    // =========================================================================
    // 4. PRODUCTION POS INITIALIZATION
    // =========================================================================
    
    async function initializeProductionPOS() {
        console.log('🚀 PRODUCTION POS Initialization...');
        
        const startTime = performance.now();
        
        // Show loading overlay
        const loadingOverlay = document.getElementById('loadingOverlay');
        if (loadingOverlay) {
            loadingOverlay.style.display = 'flex';
        }
        
        try {
            // Step 1: Load core modules
            console.log('🔄 Step 1: Loading core modules...');
            const coreModules = [
                'POSUtils',
                'POSErrorHandler',
                'POSMonitor',
                'VATService',
                'posState',
                'POSCart',
                'PaymentManager'
            ];
            
            const coreResults = [];
            for (const moduleName of coreModules) {
                try {
                    const module = await window.POSModuleManager.loadModule(moduleName);
                    coreResults.push({ name: moduleName, success: true });
                } catch (error) {
                    console.error(`❌ Critical module ${moduleName} failed:`, error);
                    coreResults.push({ name: moduleName, success: false, error: error.message });
                    
                    // If critical module fails, abort
                    if (['POSUtils', 'posState', 'POSCart'].includes(moduleName)) {
                        throw new Error(`Critical module ${moduleName} failed: ${error.message}`);
                    }
                }
            }
            
            console.log(`✅ Core modules: ${coreResults.filter(r => r.success).length}/${coreModules.length} loaded`);
            
            // Step 2: Initialize essential functionality
            console.log('🔄 Step 2: Initializing essential functionality...');
            await initializeEssentialFeatures();
            
            // Step 3: Load non-critical modules in background
            console.log('🔄 Step 3: Loading non-critical modules in background...');
            setTimeout(() => {
                const nonCriticalModules = [
                    'POSProducts',
                    'POSCustomers',
                    'POSScanner',
                    'POSDiscounts',
                    'POSHeldSales',
                    'POSStats',
                    'POSReceipts',
                    'POSPayments'
                ];
                
                nonCriticalModules.forEach(moduleName => {
                    window.POSModuleManager.loadModule(moduleName)
                        .then(() => console.log(`✅ Background loaded: ${moduleName}`))
                        .catch(err => console.warn(`⚠️ Background load failed for ${moduleName}:`, err.message));
                });
            }, 500);
            
            // Step 4: Setup event listeners
            console.log('🔄 Step 4: Setting up event listeners...');
            setupProductionEventListeners();
            
            // Step 5: Load initial data
            console.log('🔄 Step 5: Loading initial data...');
            await loadProductionInitialData();
            
            // Record startup time
            const loadTime = performance.now() - startTime;
            
            // Update monitor
            if (window.POSMonitor) {
                window.POSMonitor.metrics.startupTime = loadTime;
                window.POSMonitor.metrics.startupSuccess = true;
            }
            
            // Hide loading overlay
            if (loadingOverlay) {
                loadingOverlay.style.display = 'none';
            }
            
            // Show success message
            safeToast(`POS Ready in ${Math.round(loadTime)}ms`, 'success');
            console.log(`🎉 PRODUCTION POS Ready in ${loadTime.toFixed(0)}ms`);
            
            // Track analytics
            if (window.POSAnalytics) {
                window.POSAnalytics.track('pos_production_initialized', {
                    loadTime: Math.round(loadTime),
                    coreModulesLoaded: coreResults.filter(r => r.success).length,
                    timestamp: new Date().toISOString()
                });
            }
            
        } catch (error) {
            console.error('❌ PRODUCTION POS Initialization failed:', error);
            
            // Hide loading overlay
            if (loadingOverlay) {
                loadingOverlay.style.display = 'none';
            }
            
            // Show fallback
            safeShowFallback(`POS Initialization failed: ${error.message}. Please refresh.`);
            
            // Track error
            if (window.POSMonitor) {
                window.POSMonitor.recordError(error, { phase: 'initialization' });
            }
            
            // Try to setup basic functionality anyway
            try {
                setupEssentialEventListeners();
            } catch (e) {
                console.error('Failed to setup fallback event listeners:', e);
            }
        }
    }
    
    async function initializeEssentialFeatures() {
        return new Promise(async (resolve) => {
            console.log('🔧 Initializing essential features...');
            
            try {
                // 1. Initialize cart
                if (window.POSCart && window.POSCart.init) {
                    await window.POSCart.init();
                }
                
                // 2. Load saved cart state
                if (window.posState && typeof window.posState.loadSavedState === 'function') {
                    window.posState.loadSavedState();
                } else if (window.posState) {
                    // Fallback: load from localStorage
                    try {
                        const savedCart = localStorage.getItem('pos_cart');
                        if (savedCart) {
                            window.posState.cart = JSON.parse(savedCart);
                        }
                    } catch (e) {
                        console.warn('Failed to load saved cart:', e);
                    }
                }
                
                // 3. Update cart display
                if (window.POSCart && window.POSCart.updateDisplay) {
                    window.POSCart.updateDisplay();
                }
                
                // 4. Initialize payment manager
                if (window.PaymentManager && window.PaymentManager.init) {
                    window.PaymentManager.init().catch(err => {
                        console.warn('Payment manager init warning:', err);
                    });
                }
                
                // 5. Setup basic UI
                updateBasicUI();
                
                console.log('✅ Essential features initialized');
                resolve();
                
            } catch (error) {
                console.error('❌ Essential features initialization failed:', error);
                resolve(); // Don't fail the whole initialization
            }
        });
    }
    
    function setupProductionEventListeners() {
        console.log('🎮 Setting up production event listeners...');
        
        // Keyboard shortcuts with production-grade error handling
        document.addEventListener('keydown', function(e) {
            // Don't trigger when typing in inputs
            if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA' || e.target.isContentEditable) {
                return;
            }
            
            try {
                switch(e.key) {
                    case 'F1':
                        e.preventDefault();
                        const searchInput = document.getElementById('productSearch') || document.getElementById('globalSearch');
                        if (searchInput) {
                            searchInput.focus();
                            searchInput.select();
                        }
                        break;
                        
                    case 'F2':
                        e.preventDefault();
                        if (window.POSCustomers && window.POSCustomers.showModal) {
                            window.POSCustomers.showModal();
                        } else {
                            safeToast('Customer selection not available', 'warning');
                        }
                        break;
                        
                    case 'F3':
                        e.preventDefault();
                        safeNewSale();
                        break;
                        
                    case 'F8':
                        e.preventDefault();
                        safeShowPaymentModal();
                        break;
                        
                    case 'F9':
                        e.preventDefault();
                        const mpesaModal = document.getElementById('manualMpesaModal');
                        if (mpesaModal) {
                            try {
                                new bootstrap.Modal(mpesaModal).show();
                            } catch (modalError) {
                                safeToast('Cannot open M-Pesa manager', 'error');
                            }
                        }
                        break;
                        
                    case 'Escape':
                        e.preventDefault();
                        // Close all modals safely
                        try {
                            document.querySelectorAll('.modal.show').forEach(modal => {
                                const bsModal = bootstrap.Modal.getInstance(modal);
                                if (bsModal) bsModal.hide();
                            });
                        } catch (modalError) {
                            console.error('Error closing modals:', modalError);
                        }
                        break;
                        
                    case 'b':
                        if (e.ctrlKey) {
                            e.preventDefault();
                            safeOpenScanner();
                        }
                        break;
                }
            } catch (error) {
                console.error('Keyboard shortcut error:', error);
                if (window.POSMonitor) {
                    window.POSMonitor.recordError(error, { type: 'keyboard_shortcut', key: e.key });
                }
            }
        });
        
        // Global search enter key
        const globalSearch = document.getElementById('globalSearch');
        if (globalSearch) {
            globalSearch.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    safeGlobalSearch();
                }
            });
        }
        
        // Handle beforeunload - warn about unsaved changes
        window.addEventListener('beforeunload', function(e) {
            if (window.posState && window.posState.cart && window.posState.cart.length > 0) {
                // Don't prompt in production if sale is complete
                const hasCompletedSale = localStorage.getItem('pos_last_completed_sale');
                if (!hasCompletedSale) {
                    e.preventDefault();
                    e.returnValue = 'You have unsaved items in your cart. Are you sure you want to leave?';
                    return e.returnValue;
                }
            }
        });
        
        // Safe click handlers for all checkout buttons
        document.addEventListener('click', function(e) {
            // Handle checkout button
            if (e.target.closest('#checkoutBtn')) {
                e.preventDefault();
                e.stopPropagation();
                safeShowPaymentModal();
            }
            
            // Handle payment method cards
            if (e.target.closest('.payment-method-card')) {
                const card = e.target.closest('.payment-method-card');
                if (card && card.dataset.method) {
                    e.preventDefault();
                    e.stopPropagation();
                    if (window.PaymentManager && window.PaymentManager.selectPaymentMethod) {
                        window.PaymentManager.selectPaymentMethod(card.dataset.method)
                            .catch(err => {
                                safeToast(`Payment method error: ${err.message}`, 'error');
                            });
                    }
                }
            }
        });
        
        // Network status monitoring
        window.addEventListener('online', function() {
            safeToast('Back online', 'success');
            // Try to sync any pending operations
            if (window.POSMonitor) {
                window.POSMonitor.healthCheck();
            }
        });
        
        window.addEventListener('offline', function() {
            safeToast('You are offline. Some features may be limited.', 'warning');
        });
        
        console.log('✅ Production event listeners setup complete');
    }
    
    function setupEssentialEventListeners() {
        // Basic event listeners for fallback mode
        const checkoutBtn = document.getElementById('checkoutBtn');
        if (checkoutBtn) {
            checkoutBtn.addEventListener('click', safeShowPaymentModal);
        }
    }
    
    function updateBasicUI() {
        // Update today's stats if available
        if (window.POSStats && window.POSStats.loadToday) {
            window.POSStats.loadToday().catch(() => {
                // Silently fail
            });
        }
        
        // Update cart count
        if (window.posState && window.posState.cart) {
            const cartCount = document.getElementById('cartCount');
            if (cartCount) {
                cartCount.textContent = window.posState.cart.length;
            }
        }
    }
    
    async function loadProductionInitialData() {
        console.log('📊 Loading production initial data...');
        
        const promises = [];
        
        // Load products (non-blocking)
        if (window.POSProducts && window.POSProducts.loadAll) {
            promises.push(
                window.POSProducts.loadAll()
                    .then(() => console.log('✅ Products loaded'))
                    .catch(err => console.warn('⚠️ Products load failed:', err.message))
            );
        }
        
        // Load customers (non-blocking)
        if (window.POSCustomers && window.POSCustomers.loadCustomers) {
            promises.push(
                window.POSCustomers.loadCustomers()
                    .then(() => console.log('✅ Customers loaded'))
                    .catch(err => console.warn('⚠️ Customers load failed:', err.message))
            );
        }
        
        // Load today's stats (non-blocking)
        if (window.POSStats && window.POSStats.loadToday) {
            promises.push(
                window.POSStats.loadToday()
                    .then(() => console.log('✅ Today stats loaded'))
                    .catch(err => console.warn('⚠️ Stats load failed:', err.message))
            );
        }
        
        // Wait for all with timeout
        await Promise.allSettled(promises.map(p => 
            Promise.race([
                p,
                new Promise(resolve => setTimeout(resolve, 10000)) // 10 second timeout
            ])
        ));
        
        console.log('✅ Initial data loading complete');
    }
    
    // =========================================================================
    // 5. MODULE FALLBACKS (in case scripts fail to load)
    // =========================================================================
    
    // Create minimal fallbacks if modules don't load
    setTimeout(() => {
        // POSUtils fallback
        if (!window.POSUtils) {
            console.warn('Creating POSUtils fallback');
            window.POSUtils = {
                showToast: safeToast,
                showLoading: function(show, elementId = null) {
                    const loadingOverlay = document.getElementById('loadingOverlay');
                    if (loadingOverlay) {
                        loadingOverlay.style.display = show ? 'flex' : 'none';
                    }
                },
                formatCurrency: function(amount) {
                    return `KES ${parseFloat(amount || 0).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
                },
                showError: function(message, error = null) {
                    console.error('POSUtils Error:', message, error);
                    safeToast(message, 'error');
                }
            };
        }
        
        // POSErrorHandler fallback
        if (!window.POSErrorHandler) {
            console.warn('Creating POSErrorHandler fallback');
            window.POSErrorHandler = {
                capture: function(error, context = {}) {
                    console.error('🚨 POS Error:', error, context);
                    safeToast('An error occurred', 'error');
                }
            };
        }
        
        // VATService fallback
        if (!window.VATService) {
            console.warn('Creating VATService fallback');
            window.VATService = {
                calculateItemVAT: function(item, customerStatus = 'vatable') {
                    const price = item.price || 0;
                    const quantity = item.quantity || 1;
                    const total = price * quantity;
                    
                    if (customerStatus === 'exempted') {
                        return {
                            totalExcludingVAT: total,
                            totalIncludingVAT: total,
                            taxAmount: 0,
                            taxRate: 0
                        };
                    }
                    
                    const taxRate = item.tax_rate || 16;
                    const taxAmount = (total * taxRate) / 100;
                    
                    return {
                        totalExcludingVAT: total,
                        totalIncludingVAT: total + taxAmount,
                        taxAmount: taxAmount,
                        taxRate: taxRate
                    };
                },
                
                calculateCartVAT: function(cartItems, customer = null, discount = 0) {
                    const customerStatus = customer?.vat_status || 'vatable';
                    let subtotal = 0;
                    let totalVAT = 0;
                    
                    cartItems.forEach(item => {
                        const vatData = this.calculateItemVAT(item, customerStatus);
                        subtotal += vatData.totalExcludingVAT;
                        totalVAT += vatData.taxAmount;
                    });
                    
                    const totalWithVAT = Math.max(0, (subtotal - discount) + totalVAT);
                    
                    return {
                        subtotal: subtotal,
                        discount: discount,
                        totalVAT: totalVAT,
                        totalWithVAT: totalWithVAT,
                        customerVatStatus: customerStatus
                    };
                }
            };
        }
        
        // POSMonitor fallback
        if (!window.POSMonitor) {
            console.warn('Creating POSMonitor fallback');
            window.POSMonitor = {
                metrics: {},
                recordError: function(error, context = {}) {
                    console.error('Monitor Error:', error, context);
                },
                healthCheck: function() {
                    return {
                        timestamp: Date.now(),
                        online: navigator.onLine,
                        cartItems: window.posState?.cart?.length || 0
                    };
                }
            };
        }
        
        // posState fallback
        if (!window.posState) {
            console.warn('Creating posState fallback');
            window.posState = {
                cart: [],
                currentCustomer: null,
                currentDiscount: 0,
                cartVATData: null,
                
                getCartSubtotal: function() {
                    return this.cart.reduce((total, item) => total + (item.price * item.quantity), 0);
                },
                
                getCartTotal: function() {
                    const subtotal = this.getCartSubtotal();
                    const vatData = window.VATService?.calculateCartVAT?.(this.cart, this.currentCustomer, this.currentDiscount) || {
                        totalWithVAT: subtotal - this.currentDiscount
                    };
                    return vatData.totalWithVAT;
                },
                
                clearCart: function() {
                    this.cart = [];
                    this.currentCustomer = null;
                    this.currentDiscount = 0;
                    this.cartVATData = null;
                    
                    // Save to localStorage
                    try {
                        localStorage.removeItem('pos_cart');
                        localStorage.removeItem('pos_current_customer');
                        localStorage.removeItem('pos_current_discount');
                    } catch (e) {}
                    
                    // Update UI if possible
                    if (window.POSCart && window.POSCart.updateDisplay) {
                        window.POSCart.updateDisplay();
                    }
                },
                
                saveCart: function() {
                    try {
                        localStorage.setItem('pos_cart', JSON.stringify(this.cart));
                        if (this.currentCustomer) {
                            localStorage.setItem('pos_current_customer', JSON.stringify(this.currentCustomer));
                        }
                        localStorage.setItem('pos_current_discount', this.currentDiscount.toString());
                    } catch (e) {
                        console.warn('Failed to save cart:', e);
                    }
                },
                
                loadSavedState: function() {
                    try {
                        const savedCart = localStorage.getItem('pos_cart');
                        if (savedCart) {
                            this.cart = JSON.parse(savedCart);
                        }
                        
                        const savedCustomer = localStorage.getItem('pos_current_customer');
                        if (savedCustomer) {
                            this.currentCustomer = JSON.parse(savedCustomer);
                        }
                        
                        const savedDiscount = localStorage.getItem('pos_current_discount');
                        if (savedDiscount) {
                            this.currentDiscount = parseFloat(savedDiscount);
                        }
                    } catch (e) {
                        console.warn('Failed to load saved state:', e);
                    }
                }
            };
            
            // Load saved state
            window.posState.loadSavedState();
        }
        
        // Show warning if critical modules missing
        if (!window.POSCart || !window.POSProducts) {
            console.warn('Critical POS modules missing. Some features disabled.');
        }
    }, 1000);
    
    // =========================================================================
    // 6. PERFORMANCE MONITORING
    // =========================================================================
    
    // Performance monitoring
    if (window.performance) {
        const posStart = performance.now();
        
        window.addEventListener('load', () => {
            const loadTime = performance.now() - posStart;
            console.log(`🎯 POS loaded in ${loadTime.toFixed(0)}ms`);
            
            // Track initial load time
            if (window.POSMonitor && window.POSMonitor.metrics) {
                window.POSMonitor.metrics.initialLoadTime = loadTime;
            }
        });
    }
    
    // Monitor memory usage (if available)
    if (window.performance && performance.memory) {
        setInterval(() => {
            const usedMB = performance.memory.usedJSHeapSize / (1024 * 1024);
            const totalMB = performance.memory.totalJSHeapSize / (1024 * 1024);
            
            if (usedMB > 150) { // 150MB threshold
                console.warn(`High memory usage: ${usedMB.toFixed(1)}MB / ${totalMB.toFixed(1)}MB`);
                
                // Suggest cleanup if memory is high
                if (usedMB > 200 && window.POSMonitor) {
                    window.POSMonitor.recordError(new Error('High memory usage'), {
                        usedMB,
                        totalMB
                    });
                }
            }
        }, 30000); // Check every 30 seconds
    }
    
    // =========================================================================
    // 7. START PRODUCTION INITIALIZATION
    // =========================================================================
    
    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            // Small delay to let other scripts load
            setTimeout(() => {
                initializeProductionPOS();
            }, 300);
        });
    } else {
        // DOM already loaded
        setTimeout(() => {
            initializeProductionPOS();
        }, 300);
    }
    
})();
</script>

<!-- Analytics (optional) -->
<script>
(function() {
    'use strict';
    
    // Simple analytics tracking
    window.POSAnalytics = {
        track: function(event, data = {}) {
            if (window.console && console.log) {
                console.log(`📊 Analytics: ${event}`, data);
            }
            
            // Send to server if endpoint exists
            if (window.fetch && typeof window.fetch === 'function') {
                fetch('/api/analytics/track', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                    },
                    body: JSON.stringify({
                        event: event,
                        data: data,
                        timestamp: new Date().toISOString(),
                        url: window.location.href
                    })
                }).catch(() => {
                    // Silently fail
                });
            }
        }
    };
    
    // Track page load
    document.addEventListener('DOMContentLoaded', () => {
        setTimeout(() => {
            window.POSAnalytics?.track('pos_page_loaded');
        }, 1000);
    });
    
})();

// Add this to update cart with customer VAT
window.updateCartWithCustomerVAT = function(customer) {
    if (window.posState) {
        window.posState.currentCustomer = customer;
        
        // Recalculate VAT with new customer status
        if (window.POSCart && window.POSCart.updateDisplay) {
            window.POSCart.updateDisplay();
        }
        
        // Update payment modal if open
        if (window.POSPayments && window.POSPayments.updatePaymentModalForCurrentCart) {
            window.POSPayments.updatePaymentModalForCurrentCart();
        }
        
        // Show notification
        const status = customer?.vat_status || 'vatable';
        let message = `Customer set to: ${customer?.name || 'Walk-in'}`;
        if (status === 'exempted') {
            message += ' (VAT Exempted)';
        } else if (status === 'zero_rated') {
            message += ' (Zero-Rated VAT)';
        }
        
        if (window.POSUtils && window.POSUtils.showToast) {
            window.POSUtils.showToast(message, 'info');
        }
    }
};

// Emergency backup functions if modules fail to load
setTimeout(() => {
    // Ensure all critical functions exist
    if (typeof window.printLastReceipt !== 'function') {
        window.printLastReceipt = function() {
            console.log('🖨️ Emergency print function called');
            const invoiceNo = localStorage.getItem('pos_last_invoice_no') || 'INV-' + Date.now();
            window.open(`/pos/receipt/print/${invoiceNo}`, '_blank');
        };
    }
    
    if (typeof window.clearCartSafely !== 'function') {
        window.clearCartSafely = function() {
            console.log('🧹 Emergency cart clear function called');
            localStorage.removeItem('pos_cart');
            localStorage.removeItem('pos_current_customer');
            localStorage.removeItem('pos_current_discount');
            
            // Try to update UI
            const cartCount = document.getElementById('cartCount');
            if (cartCount) cartCount.textContent = '0';
            
            const cartItemsList = document.getElementById('cartItemsList');
            if (cartItemsList) cartItemsList.innerHTML = '<div class="text-center py-4 text-muted"><i class="bx bx-cart"></i><p>Cart is empty</p></div>';
            
            safeToast('Cart cleared', 'success');
        };
    }
    
    // Ensure safeToast exists
    if (!window.safeToast) {
        window.safeToast = function(message, type = 'info') {
            console.log(`[${type.toUpperCase()}] ${message}`);
            alert(`${type.toUpperCase()}: ${message}`);
        };
    }
}, 2000);
</script>

<!-- Load POS Modules (in dependency order) -->
<script src="<?php echo e(asset('js/pos/core/error-handler.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/core/monitor.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/utils.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/services/vat-service.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/core/pos-state.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/cart.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/products.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/customers.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/scanner.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/discounts.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/held-sales.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/stats.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/receipts.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/payment/payment-manager.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/payment/payments.js')); ?>?v=1.0.0"></script>
<script src="<?php echo e(asset('js/pos/modules/payment/payment-vat-helper.js')); ?>?v=1.0.0"></script>

<?php $__env->stopSection(); ?>

<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\Autopos\resources\views/pos/index.blade.php ENDPATH**/ ?>