<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\PosController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\CustomerController;
use App\Http\Controllers\PurchaseController;
use App\Http\Controllers\SupplierController;
use App\Http\Controllers\SaleController;
use App\Http\Controllers\ReportController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\ProductBatchController;
use App\Http\Controllers\BrandController;
use App\Http\Controllers\PriceHistoryController;
use App\Http\Controllers\UnitController;
use App\Http\Controllers\TaxController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\NotificationController;
use App\Http\Controllers\ActivityLogController;
use App\Http\Controllers\AuditController;
use App\Http\Controllers\ExportController;
use App\Http\Controllers\ImportController;
use App\Http\Controllers\DatabaseController;
use App\Http\Controllers\PrinterController;
use App\Http\Controllers\MpesaTransactionController;
use App\Http\Controllers\MpesaController;
use App\Http\Controllers\HoldSaleController;
use App\Http\Controllers\StoreInformationWebController;
use App\Http\Controllers\ReceiptSettingsController;
use App\Http\Controllers\ReceiptPDFController;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\ETIMSController;
use App\Http\Controllers\VatReportController;

/*
|--------------------------------------------------------------------------
| Public Redirect & Health Check
|--------------------------------------------------------------------------
*/
Route::get('/', function () {
    return redirect()->route('login');
});

Route::get('/health', function () {
    return response()->json([
        'status' => 'healthy',
        'timestamp' => now(),
        'version' => config('app.version', '1.0.0'),
        'environment' => config('app.env'),
        'services' => [
            'database' => DB::connection()->getPdo() ? 'connected' : 'disconnected',
            'cache' => true,
            'queue' => true,
        ]
    ]);
});

/*
|--------------------------------------------------------------------------
| Webhook Routes (No Authentication)
|--------------------------------------------------------------------------
*/
Route::prefix('webhooks')->name('webhooks.')->group(function () {
    Route::prefix('mpesa')->name('mpesa.')->group(function () {
        Route::get('/transactions/search', [MpesaTransactionController::class, 'search'])->name('transactions.search');
        Route::post('/transactions/manual', [MpesaTransactionController::class, 'storeManual'])->name('transactions.manual');
        Route::post('/sync', [MpesaTransactionController::class, 'sync'])->name('sync');
        Route::get('/transactions/export', [MpesaTransactionController::class, 'export'])->name('transactions.export');

        Route::post('/callback', [MpesaTransactionController::class, 'storeCallback'])
            ->name('callback')
            ->withoutMiddleware(['web', 'csrf']);
        
        Route::post('/validation', [MpesaTransactionController::class, 'validateTransaction'])
            ->name('validation')
            ->withoutMiddleware(['web', 'csrf']);
    });
    
    Route::prefix('etims')->name('etims.')->group(function () {
        Route::post('/callback', [PosController::class, 'etimsCallback'])
            ->name('callback')
            ->withoutMiddleware(['web', 'csrf']);
    });
});

/*
|--------------------------------------------------------------------------
| Auth Routes
|--------------------------------------------------------------------------
*/
require __DIR__ . '/auth.php';

/*
|--------------------------------------------------------------------------
| Protected Routes (Authenticated + Verified)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth', 'verified'])->group(function () {

    /*
    |--------------------------------------------------------------------------
    | Dashboard Routes (All Authenticated Users)
    |--------------------------------------------------------------------------
    */
    Route::get('/dashboard', [DashboardController::class, 'index'])
        ->name('dashboard');

// Dashboard routes
Route::middleware(['auth'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/dashboard/data', [DashboardController::class, 'getDashboardData'])->name('dashboard.data');
    Route::get('/dashboard/activities', [DashboardController::class, 'getRecentActivities'])->name('dashboard.activities');
    Route::get('/dashboard/low-stock', [DashboardController::class, 'getLowStockProducts'])->name('dashboard.low-stock');
    Route::get('/dashboard/recent-sales', [DashboardController::class, 'getRecentSales'])->name('dashboard.recent-sales');
    Route::get('/dashboard/top-products', [DashboardController::class, 'getTopProducts'])->name('dashboard.top-products');
});

    /*
    |--------------------------------------------------------------------------
    | API Routes (Called by POS JavaScript)
    |--------------------------------------------------------------------------
    */
    Route::prefix('api')->name('api.')->group(function () {
        // Store settings API (used by POS)
        Route::get('/store-settings', [\App\Http\Controllers\Api\SettingsApiController::class, 'getStoreSettings'])->name('store-settings');
        Route::get('/printer-settings', [\App\Http\Controllers\Api\SettingsApiController::class, 'getPrinterSettings'])->name('printer-settings');
        Route::get('/printer/check', [\App\Http\Controllers\Api\SettingsApiController::class, 'checkPrinterConnection'])->name('printer.check');
        
        // Stats API
        Route::get('/stats/today', [\App\Http\Controllers\Api\StatsApiController::class, 'getTodayStats'])->name('stats.today');
        Route::get('/stats/dashboard', [\App\Http\Controllers\Api\DashboardApiController::class, 'getDashboardStats'])->name('stats.dashboard');
        
        // Hold sales API
        Route::get('/hold-sales', [\App\Http\Controllers\Api\HoldSaleApiController::class, 'index'])->name('hold-sales.index');
        Route::post('/hold-sales', [\App\Http\Controllers\Api\HoldSaleApiController::class, 'store'])->name('hold-sales.store');
        Route::get('/hold-sales/stats', [\App\Http\Controllers\Api\HoldSaleApiController::class, 'stats'])->name('hold-sales.stats');
        Route::get('/hold-sales/{id}/load', [\App\Http\Controllers\Api\HoldSaleApiController::class, 'load'])->name('hold-sales.load');
        Route::delete('/hold-sales/{id}', [\App\Http\Controllers\Api\HoldSaleApiController::class, 'destroy'])->name('hold-sales.destroy');
        
        // M-Pesa transactions API
        Route::get('/mpesa/transactions', [MpesaTransactionController::class, 'index'])->name('mpesa.transactions');
        Route::get('/mpesa/transactions/{id}', [MpesaTransactionController::class, 'show'])->name('mpesa.transactions.show');
        Route::post('/mpesa/check-payment', [MpesaTransactionController::class, 'checkPendingPayment'])->name('mpesa.check-payment');
        Route::post('/mpesa/transactions/{id}/match', [MpesaTransactionController::class, 'match'])->name('mpesa.transactions.match');
        Route::post('/mpesa/sync-transactions', [MpesaTransactionController::class, 'syncTransactions'])->name('mpesa.sync-transactions');
        
        // Payment API
        Route::post('/process-payment', [\App\Http\Controllers\Api\PaymentApiController::class, 'processPayment'])->name('payment.process');
        Route::post('/verify-payment', [\App\Http\Controllers\Api\PaymentApiController::class, 'verify'])->name('payment.verify');
        
        // Products API
        Route::get('/products', [ProductController::class, 'apiGetProducts'])->name('products.index');
        Route::get('/products/search', [ProductController::class, 'apiSearchProducts'])->name('products.search');
        Route::get('/products/barcode/{barcode}', [ProductController::class, 'findByBarcode'])->name('products.barcode');
        Route::get('/products/{id}', [ProductController::class, 'apiGetProduct'])->name('products.show');
        Route::post('/products/{id}/adjust-stock', [ProductController::class, 'apiAdjustStock'])->name('products.adjust-stock');
        
        // Categories API
        Route::get('/categories', [CategoryController::class, 'apiGetCategories'])->name('categories.index');
        Route::get('/categories/{id}/products', [CategoryController::class, 'apiGetCategoryProducts'])->name('categories.products');
        
        // Customers API
        Route::get('/customers', [CustomerController::class, 'apiGetCustomers'])->name('customers.index');
        Route::get('/customers/search', [CustomerController::class, 'apiSearchCustomers'])->name('customers.search');
        Route::get('/customers/{id}', [CustomerController::class, 'apiGetCustomer'])->name('customers.show');
        Route::post('/customers', [CustomerController::class, 'apiStore'])->name('customers.store');
        Route::post('/customers/quick-add', [CustomerController::class, 'apiQuickAdd'])->name('customers.quick-add');
        Route::get('/customers/{id}/credit-info', [CustomerController::class, 'apiGetCreditInfo'])->name('customers.credit-info');
        
        // Additional Customer API routes
        Route::get('/customers/stats', [CustomerController::class, 'apiGetCustomerStats'])->name('customers.stats');
        Route::get('/customers/{id}/purchase-history', [CustomerController::class, 'getPurchaseHistory'])->name('customers.purchase-history');
        Route::post('/customers/{id}/update-credit', [CustomerController::class, 'updateCreditBalance'])->name('customers.update-credit');
        Route::get('/customers/{id}/has-sales', [CustomerController::class, 'hasSales'])->name('customers.has-sales');
        Route::post('/validate-vat', [CustomerController::class, 'validateVatNumber'])->name('validate-vat');
        
        // Sales API
        Route::post('/sales/process', [SaleController::class, 'apiProcessSale'])->name('sales.process');
        Route::get('/sales/{id}/receipt', [SaleController::class, 'apiGetReceipt'])->name('sales.receipt');
        Route::post('/sales/{id}/refund', [SaleController::class, 'apiRefund'])->name('sales.refund');
        Route::get('/sales/recent', [SaleController::class, 'apiGetRecentSales'])->name('sales.recent');
        Route::get('/sales/search', [SaleController::class, 'apiSearchSales'])->name('sales.search');
        
        // POS Checkout API
        Route::post('/pos/checkout', [\App\Http\Controllers\Api\PosApiController::class, 'checkout'])->name('pos.checkout');
        
        // Store Information API Routes (for AJAX calls)
        Route::prefix('store')->name('store.')->group(function () {
            Route::get('/info', [StoreInformationWebController::class, 'index'])->name('info.index');
            Route::post('/info', [StoreInformationWebController::class, 'store'])->name('info.store');
            Route::get('/pos-info', [StoreInformationWebController::class, 'posInfo'])->name('pos-info');
            Route::get('/receipt-settings', [StoreInformationWebController::class, 'receiptSettings'])->name('receipt-settings');
            Route::post('/clear-cache', [StoreInformationWebController::class, 'clearCache'])->name('clear-cache');
            Route::get('/health', [StoreInformationWebController::class, 'health'])->name('health');
            Route::get('/config', [StoreInformationWebController::class, 'config'])->name('config');
        });
        
        // Test API
        Route::get('/test', function() {
            return response()->json([
                'success' => true,
                'message' => 'API is working',
                'timestamp' => now(),
                'endpoints' => [
                    '/api/store-settings',
                    '/api/stats/today',
                    '/api/products',
                    '/api/customers',
                    '/api/sales/process'
                ]
            ]);
        });
        
        // Mock endpoints for development
        Route::post('/mpesa/stk-push', function(Request $request) {
            return response()->json([
                'success' => true,
                'message' => 'STK Push simulated',
                'transaction_id' => 'MPESA_' . time(),
                'phone' => $request->input('phone'),
                'amount' => $request->input('amount')
            ]);
        });
    });

    /*
    |--------------------------------------------------------------------------
    | API Version 1 Routes (Modern POS & Mobile Apps)
    |--------------------------------------------------------------------------
    */
    Route::prefix('api/v1')->name('api.v1.')->group(function () {
        
        /*
        |--------------------------------------------------------------------------
        | Products API
        |--------------------------------------------------------------------------
        */
        Route::prefix('products')->name('products.')->group(function () {
            Route::get('/', [ProductController::class, 'apiGetProducts'])->name('index');
            Route::get('/search', [ProductController::class, 'apiSearchProducts'])->name('search');
            Route::get('/barcode/{barcode}', [ProductController::class, 'findByBarcode'])->name('barcode');
            Route::get('/{product}', [ProductController::class, 'apiGetProduct'])->name('show');
            Route::post('/{product}/adjust-stock', [ProductController::class, 'apiAdjustStock'])->name('adjust-stock');
            Route::get('/{product}/price-history', [ProductController::class, 'apiPriceHistory'])->name('price-history');
            Route::get('/top-selling', [ProductController::class, 'apiGetTopProducts'])->name('top-selling');
        });

        /*
        |--------------------------------------------------------------------------
        | Categories API
        |--------------------------------------------------------------------------
        */
        Route::prefix('categories')->name('categories.')->group(function () {
            Route::get('/', [CategoryController::class, 'apiGetCategories'])->name('index');
            Route::get('/tree', [CategoryController::class, 'apiGetCategoryTree'])->name('tree');
            Route::get('/{category}/products', [CategoryController::class, 'apiGetCategoryProducts'])->name('products');
        });

        /*
        |--------------------------------------------------------------------------
        | Customers API
        |--------------------------------------------------------------------------
        */
        Route::prefix('customers')->name('customers.')->group(function () {
            Route::get('/', [CustomerController::class, 'apiGetCustomers'])->name('index');
            Route::get('/search', [CustomerController::class, 'apiSearchCustomers'])->name('search');
            Route::get('/{id}', [CustomerController::class, 'apiGetCustomer'])->name('show');
            Route::post('/', [CustomerController::class, 'apiStore'])->name('store');
            Route::post('/quick-add', [CustomerController::class, 'apiQuickAdd'])->name('quick-add');
            Route::get('/{id}/credit-info', [CustomerController::class, 'apiGetCreditInfo'])->name('credit-info');
            Route::get('/stats', [CustomerController::class, 'apiGetStats'])->name('stats');
            Route::get('/{id}/purchase-history', [CustomerController::class, 'getPurchaseHistory'])->name('purchase-history');
            Route::post('/{id}/update-credit', [CustomerController::class, 'updateCreditBalance'])->name('update-credit');
            Route::get('/{id}/has-sales', [CustomerController::class, 'hasSales'])->name('has-sales');
            Route::post('/validate-vat', [CustomerController::class, 'validateVatNumber'])->name('validate-vat');
        });

        /*
        |--------------------------------------------------------------------------
        | Sales API
        |--------------------------------------------------------------------------
        */
        Route::prefix('sales')->name('sales.')->group(function () {
            Route::post('/process', [SaleController::class, 'apiProcessSale'])->name('process');
            Route::get('/{id}/receipt', [SaleController::class, 'apiGetReceipt'])->name('receipt');
            Route::post('/{id}/refund', [SaleController::class, 'apiRefund'])->name('refund');
            Route::post('/{id}/send-receipt', [SaleController::class, 'apiSendReceipt'])->name('send-receipt');
            Route::get('/{id}/details', [SaleController::class, 'apiGetDetails'])->name('details');
            Route::get('/recent', [SaleController::class, 'apiGetRecentSales'])->name('recent');
            Route::get('/search', [SaleController::class, 'apiSearchSales'])->name('search');
        });

        /*
        |--------------------------------------------------------------------------
        | Hold Sales API
        |--------------------------------------------------------------------------
        */
        Route::prefix('hold-sales')->name('hold-sales.')->group(function () {
            Route::get('/', [HoldSaleController::class, 'index'])->name('index');
            Route::get('/stats', [HoldSaleController::class, 'stats'])->name('stats');
            Route::post('/', [HoldSaleController::class, 'store'])->name('store');
            Route::get('/{id}', [HoldSaleController::class, 'show'])->name('show');
            Route::get('/{id}/load', [HoldSaleController::class, 'load'])->name('load');
            Route::delete('/{id}', [HoldSaleController::class, 'destroy'])->name('destroy');
            Route::post('/clear-expired', [HoldSaleController::class, 'clearExpired'])->name('clear-expired');
            Route::post('/{id}/extend', [HoldSaleController::class, 'extend'])->name('extend');
            Route::get('/search', [HoldSaleController::class, 'search'])->name('search');
        });

        /*
        |--------------------------------------------------------------------------
        | M-Pesa API
        |--------------------------------------------------------------------------
        */
        Route::prefix('mpesa')->name('mpesa.')->group(function () {
            Route::get('/transactions', [MpesaTransactionController::class, 'index'])->name('transactions.index');
            Route::get('/transactions/{id}', [MpesaTransactionController::class, 'show'])->name('transactions.show');
            Route::get('/transactions/phone/{phone}', [MpesaTransactionController::class, 'getByPhone'])->name('transactions.phone');
            Route::post('/transactions/{id}/match', [MpesaTransactionController::class, 'match'])->name('transactions.match');
            Route::post('/check-payment', [MpesaTransactionController::class, 'checkPendingPayment'])->name('check-payment');
            Route::post('/transactions/bulk-match', [MpesaTransactionController::class, 'bulkMatch'])->name('transactions.bulk-match');
            Route::get('/stats', [MpesaTransactionController::class, 'stats'])->name('stats');
            Route::get('/dashboard-stats', [MpesaTransactionController::class, 'dashboardStats'])->name('dashboard-stats');
            Route::post('/stk-push', [MpesaTransactionController::class, 'stkPush'])->name('stk-push');
        });

        /*
        |--------------------------------------------------------------------------
        | Store & System API
        |--------------------------------------------------------------------------
        */
        Route::prefix('store')->name('store.')->group(function () {
            Route::get('/settings', [SettingsController::class, 'apiGetSettings'])->name('settings');
            Route::get('/printer-settings', [PrinterController::class, 'apiGetSettings'])->name('printer-settings');
            Route::get('/printer/check', [PrinterController::class, 'apiCheckConnection'])->name('printer.check');
            Route::post('/print/thermal/{saleId}', [PrinterController::class, 'apiPrintThermal'])->name('print.thermal');
        });

        /*
        |--------------------------------------------------------------------------
        | eTIMS API
        |--------------------------------------------------------------------------
        */
        Route::prefix('etims')->name('etims.')->group(function () {
            Route::post('/send/{saleId}', [PosController::class, 'apiSendToEtims'])->name('send');
            Route::get('/status/{saleId}', [PosController::class, 'apiGetEtimsStatus'])->name('status');
        });

        /*
        |--------------------------------------------------------------------------
        | Statistics API
        |--------------------------------------------------------------------------
        */
        Route::prefix('stats')->name('stats.')->group(function () {
            Route::get('/today', [DashboardController::class, 'apiGetTodayStats'])->name('today');
            Route::get('/dashboard', [DashboardController::class, 'apiGetDashboardStats'])->name('dashboard');
        });

        /*
        |--------------------------------------------------------------------------
        | Operations API
        |--------------------------------------------------------------------------
        */
        Route::prefix('operations')->name('operations.')->group(function () {
            Route::post('/apply-discount', [PosController::class, 'apiApplyDiscount'])->name('apply-discount');
            Route::post('/email-receipt', [SaleController::class, 'apiEmailReceipt'])->name('email-receipt');
            Route::post('/download-receipt-pdf', [SaleController::class, 'apiDownloadReceiptPDF'])->name('download-receipt-pdf');
        });

        /*
        |--------------------------------------------------------------------------
        | Resources API
        |--------------------------------------------------------------------------
        */
        Route::prefix('resources')->name('resources.')->group(function () {
            Route::get('/suppliers', [SupplierController::class, 'apiGetSuppliers'])->name('suppliers');
            Route::get('/brands', [BrandController::class, 'apiGetBrands'])->name('brands');
            Route::get('/units', [UnitController::class, 'apiGetUnits'])->name('units');
            Route::get('/taxes', [TaxController::class, 'apiGetTaxes'])->name('taxes');
            Route::get('/printers/available', [PrinterController::class, 'getAvailablePrinters'])->name('printers.available');
            Route::get('/printers/{printer}/status', [PrinterController::class, 'checkHealth'])->name('printers.status');
        });

        /*
        |--------------------------------------------------------------------------
        | Payment API
        |--------------------------------------------------------------------------
        */
        Route::prefix('payments')->name('payments.')->group(function () {
            Route::post('/process', [\App\Http\Controllers\Api\PaymentApiController::class, 'processPayment'])->name('process');
            Route::post('/verify', [\App\Http\Controllers\Api\PaymentApiController::class, 'verify'])->name('verify');
            Route::post('/cash', [\App\Http\Controllers\Api\PaymentApiController::class, 'processCash'])->name('cash');
            Route::post('/mpesa', [\App\Http\Controllers\Api\PaymentApiController::class, 'processMpesa'])->name('mpesa');
            Route::post('/card', [\App\Http\Controllers\Api\PaymentApiController::class, 'processCard'])->name('card');
            Route::post('/split', [\App\Http\Controllers\Api\PaymentApiController::class, 'processSplit'])->name('split');
        });
    });

    /*
    |--------------------------------------------------------------------------
    | M-Pesa Management Routes (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('mpesa')->name('mpesa.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [MpesaController::class, 'index'])->name('index');
        Route::get('/configuration', [MpesaController::class, 'configuration'])->name('configuration');
        Route::post('/configuration', [MpesaController::class, 'updateConfiguration'])->name('configuration.update');
        Route::get('/test-connection', [MpesaController::class, 'testConnection'])->name('test-connection');
        
        Route::get('/transactions', [MpesaTransactionController::class, 'index'])->name('transactions.index');
        Route::get('/transactions/{transaction}', [MpesaTransactionController::class, 'show'])->name('transactions.show');
        Route::post('/transactions/{transaction}/match', [MpesaTransactionController::class, 'match'])->name('transactions.match');
        Route::post('/transactions/{transaction}/reverse', [MpesaTransactionController::class, 'reverse'])->name('transactions.reverse');
        Route::post('/transactions/bulk-match', [MpesaTransactionController::class, 'bulkMatch'])->name('transactions.bulk-match');
        
        Route::get('/reports/daily', [MpesaTransactionController::class, 'dailyReport'])->name('reports.daily');
        Route::get('/reports/unmatched', [MpesaTransactionController::class, 'unmatchedReport'])->name('reports.unmatched');
        Route::get('/reconciliation', [MpesaTransactionController::class, 'reconciliation'])->name('reconciliation');
        
        Route::get('/settings', [MpesaController::class, 'settings'])->name('settings');
        Route::post('/settings', [MpesaController::class, 'updateSettings'])->name('settings.update');
    });

    /*
    |--------------------------------------------------------------------------
    | POS Routes (Cashier, Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('pos')->name('pos.')->middleware('role:cashier|manager|admin')->group(function () {
        Route::get('/', [PosController::class, 'index'])->name('index');
        Route::get('/search-products', [PosController::class, 'searchProducts'])->name('search-products');
        Route::post('/scan-barcode', [PosController::class, 'scanBarcode'])->name('scan-barcode');
        Route::post('/process-sale', [PosController::class, 'processSale'])->name('process-sale');
        Route::post('/process-sale-modern', [PosController::class, 'processSaleModern'])->name('process.modern');
        Route::post('/initiate-mpesa', [PosController::class, 'initiateMpesaPayment'])->name('mpesa.initiate');
        Route::post('/check-mpesa-payment', [PosController::class, 'checkMpesaPayment'])->name('mpesa.check');
        Route::post('/hold-sale', [PosController::class, 'holdSale'])->name('hold-sale');
        Route::post('/hold-sale-modern', [PosController::class, 'holdSaleModern'])->name('hold.modern');
        Route::get('/held-sales', [PosController::class, 'getHeldSales'])->name('held-sales');
        Route::get('/held-sales-modern', [PosController::class, 'getHeldSalesModern'])->name('held.modern');
        Route::get('/load-held-sale/{id}', [PosController::class, 'loadHeldSale'])->name('load-held-sale');
        Route::delete('/delete-held-sale/{id}', [PosController::class, 'deleteHeldSale'])->name('delete-held-sale');
        Route::post('/print-receipt/{saleId?}', [PosController::class, 'printReceipt'])->name('print-receipt');
        Route::get('/receipt-modern/{sale}', [PosController::class, 'generateModernReceipt'])->name('receipt.modern');
        Route::get('/receipt-thermal/{sale}', [PosController::class, 'generateThermalReceipt'])->name('receipt.thermal');
        Route::get('/today-stats', [PosController::class, 'getTodayStats'])->name('today-stats');
        Route::get('/dashboard-stats', [PosController::class, 'getDashboardStats'])->name('dashboard-stats');
        Route::post('/quick-add-customer', [PosController::class, 'quickAddCustomer'])->name('quick-add-customer');
        Route::post('/quick-customer', [PosController::class, 'quickAddCustomer'])->name('quick.customer');
        Route::get('/search-customers', [PosController::class, 'searchCustomers'])->name('search-customers');
        Route::get('/customer-credit-info/{customerId}', [PosController::class, 'getCustomerCreditInfo'])->name('customer-credit-info');
        Route::post('/generate-report', [PosController::class, 'generateReport'])->name('generate-report');
        Route::post('/apply-discount', [PosController::class, 'applyDiscount'])->name('apply.discount');
    });

    /*
    |--------------------------------------------------------------------------
    | Products Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('products')->name('products.')->middleware('role:manager|admin')->group(function () {
        // Basic CRUD Operations
        Route::get('/', [ProductController::class, 'index'])->name('index');
        Route::get('/create', [ProductController::class, 'create'])->name('create');
        Route::post('/', [ProductController::class, 'store'])->name('store');
        Route::get('/{product}', [ProductController::class, 'show'])->name('show');
        Route::get('/{product}/edit', [ProductController::class, 'edit'])->name('edit');
        Route::put('/{product}', [ProductController::class, 'update'])->name('update');
        Route::delete('/{product}', [ProductController::class, 'destroy'])->name('destroy');
        
        // Price Management Routes
        Route::prefix('pricing')->name('pricing.')->group(function () {
            Route::get('/', [ProductController::class, 'pricingIndex'])->name('index');
            Route::post('/bulk-update', [ProductController::class, 'bulkPricingUpdate'])->name('bulk-update');
            Route::post('/update/{product}', [ProductController::class, 'updateSinglePrice'])->name('update');
            Route::post('/{product}/set-discount', [ProductController::class, 'setDiscount'])->name('set-discount');
            Route::post('/{product}/clear-discount', [ProductController::class, 'clearDiscount'])->name('clear-discount');
            Route::get('/tax-report', [ProductController::class, 'taxReport'])->name('tax-report');
        });
        
        // Individual Price Update
        Route::post('/{product}/update-price', [ProductController::class, 'updateSinglePrice'])->name('update-price');
        
        // Tax Report Route
        Route::get('/tax-report', [ProductController::class, 'taxReport'])->name('tax-report');
        
        // Stock Management Routes
        Route::get('/{product}/add-stock', [ProductController::class, 'addStock'])->name('add-stock.page');
        Route::post('/{product}/add-stock', [ProductController::class, 'addStock'])->name('add-stock');
        Route::post('/{product}/remove-stock', [ProductController::class, 'removeStock'])->name('remove-stock');
        
        // Stock Movements & History
        Route::get('/{product}/stock-movements', [ProductController::class, 'stockMovements'])->name('stock-movements');
        Route::get('/{product}/stock-movements/recent', [ProductController::class, 'getRecentStockMovements'])->name('stock-movements.recent');
        Route::get('/{product}/stock-movements/export', [ProductController::class, 'exportStockMovements'])->name('stock-movements.export');
        Route::get('/{product}/inventory-history', [ProductController::class, 'inventoryHistory'])->name('inventory-history');
        Route::get('/{product}/inventory-history/export', [ProductController::class, 'exportInventoryHistory'])->name('inventory-history.export');
        Route::get('/{product}/inventory-summary', [ProductController::class, 'getInventorySummary'])->name('inventory-summary');
        
        // Reports & Analytics
        Route::get('/{product}/price-history', [ProductController::class, 'priceHistory'])->name('price-history');
        Route::get('/{product}/batches', [ProductController::class, 'batches'])->name('batches');
        Route::get('/reports/expiry', [ProductController::class, 'expiryReport'])->name('expiry-report');
        Route::post('/{product}/set-expiry-discount', [ProductController::class, 'setExpiryDiscount'])->name('set-expiry-discount');
        Route::post('/bulk-set-expiry-discount', [ProductController::class, 'bulkSetExpiryDiscount'])->name('bulk-set-expiry-discount');
        
        // Bulk Operations
        Route::post('/bulk-update', [ProductController::class, 'bulkUpdate'])->name('bulk-update');
        Route::post('/bulk-delete', [ProductController::class, 'bulkDelete'])->name('bulk-delete');
        Route::post('/bulk-update-tax', [ProductController::class, 'bulkUpdateTax'])->name('bulk-update-tax');
        
        // Import/Export
        Route::get('/import', [ProductController::class, 'import'])->name('import');
        Route::post('/import', [ProductController::class, 'processImport'])->name('process-import');
        Route::get('/export', [ProductController::class, 'export'])->name('export');
        Route::get('/template/download', [ProductController::class, 'downloadTemplate'])->name('template.download');
        
        // Search & Statistics
        Route::get('/search', [ProductController::class, 'search'])->name('search');
        Route::get('/statistics', [ProductController::class, 'statistics'])->name('statistics');
    });

    /*
    |--------------------------------------------------------------------------
    | Product Batches Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('batches')->name('batches.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [ProductBatchController::class, 'index'])->name('index');
        Route::get('/create', [ProductBatchController::class, 'create'])->name('create');
        Route::post('/', [ProductBatchController::class, 'store'])->name('store');
        Route::get('/{batch}', [ProductBatchController::class, 'show'])->name('show');
        Route::get('/{batch}/edit', [ProductBatchController::class, 'edit'])->name('edit');
        Route::put('/{batch}', [ProductBatchController::class, 'update'])->name('update');
        Route::delete('/{batch}', [ProductBatchController::class, 'destroy'])->name('destroy');
        Route::post('/{batch}/adjust-stock', [ProductBatchController::class, 'adjustStock'])->name('adjust-stock');
        Route::get('/expiry-alerts', [ProductBatchController::class, 'expiryAlerts'])->name('expiry-alerts');
        Route::get('/export', [ProductBatchController::class, 'export'])->name('export');
    });

    /*
    |--------------------------------------------------------------------------
    | Categories Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('categories')->name('categories.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [CategoryController::class, 'index'])->name('index');
        Route::get('/create', [CategoryController::class, 'create'])->name('create');
        Route::post('/', [CategoryController::class, 'store'])->name('store');
        Route::get('/{category}', [CategoryController::class, 'show'])->name('show');
        Route::get('/{category}/edit', [CategoryController::class, 'edit'])->name('edit');
        Route::put('/{category}', [CategoryController::class, 'update'])->name('update');
        Route::delete('/{category}', [CategoryController::class, 'destroy'])->name('destroy');
        Route::post('/{category}/archive', [CategoryController::class, 'archive'])->name('archive');
        Route::post('/{category}/restore', [CategoryController::class, 'restore'])->name('restore');
        Route::delete('/{category}/force', [CategoryController::class, 'forceDelete'])->name('force-delete');
        Route::get('/export', [CategoryController::class, 'export'])->name('export');
        Route::get('/search', [CategoryController::class, 'search'])->name('search');
        Route::post('/bulk-delete', [CategoryController::class, 'bulkDelete'])->name('bulk-delete');
        Route::post('/bulk-update', [CategoryController::class, 'bulkUpdate'])->name('bulk-update');
        Route::get('/statistics', [CategoryController::class, 'statistics'])->name('statistics');
        Route::get('/dropdown', [CategoryController::class, 'dropdown'])->name('dropdown');
        Route::get('/{category}/products', [CategoryController::class, 'products'])->name('products');
        Route::get('/tree', [CategoryController::class, 'tree'])->name('tree');
        Route::post('/reorder', [CategoryController::class, 'reorder'])->name('reorder');
    });

    /*
    |--------------------------------------------------------------------------
    | Customers Management (All Authenticated Users)
    |--------------------------------------------------------------------------
    */
    Route::prefix('customers')->name('customers.')->group(function () {
        Route::get('/', [CustomerController::class, 'index'])->name('index');
        Route::get('/create', [CustomerController::class, 'create'])->name('create');
        Route::post('/', [CustomerController::class, 'store'])->name('store');
        Route::get('/{customer}', [CustomerController::class, 'show'])->name('show');
        Route::get('/{customer}/edit', [CustomerController::class, 'edit'])->name('edit');
        Route::put('/{customer}', [CustomerController::class, 'update'])->name('update');
        Route::delete('/{customer}', [CustomerController::class, 'destroy'])->name('destroy');
        Route::get('/export', [CustomerController::class, 'export'])->name('export');
        Route::get('/search', [CustomerController::class, 'search'])->name('search');
        Route::post('/quick-create', [CustomerController::class, 'quickCreate'])->name('quick-create');
        Route::get('/walk-in', [CustomerController::class, 'getWalkInCustomer'])->name('walk-in');
        Route::post('/bulk-delete', [CustomerController::class, 'bulkDelete'])->name('bulk-delete');
        Route::get('/statistics', [CustomerController::class, 'statistics'])->name('statistics');
        Route::post('/{customer}/update-balance', [CustomerController::class, 'updateBalance'])->name('update-balance');
        
        // VAT Report Routes
        Route::get('/vat-report', [CustomerController::class, 'vatReport'])->name('vat-report');
        Route::get('/export/vat', [CustomerController::class, 'exportVatReport'])->name('export.vat');
        
        // Customer Statement
        Route::get('/{customer}/statement', [CustomerController::class, 'generateStatement'])->name('statement');
        
        // Reset Statistics
        Route::post('/{customer}/reset-statistics', [CustomerController::class, 'resetStatistics'])->name('reset-statistics');
    });

    /*
    |--------------------------------------------------------------------------
    | Purchases Management (Purchaser, Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('purchases')->name('purchases.')->middleware('role:purchaser|manager|admin')->group(function () {
        Route::get('/', [PurchaseController::class, 'index'])->name('index');
        Route::get('/create', [PurchaseController::class, 'create'])->name('create');
        Route::post('/', [PurchaseController::class, 'store'])->name('store');
        Route::get('/{purchase}', [PurchaseController::class, 'show'])->name('show');
        Route::get('/{purchase}/edit', [PurchaseController::class, 'edit'])->name('edit');
        Route::put('/{purchase}', [PurchaseController::class, 'update'])->name('update');
        Route::delete('/{purchase}', [PurchaseController::class, 'destroy'])->name('destroy');
        Route::get('/{purchase}/quick-view', [PurchaseController::class, 'quickView'])->name('quick-view');
        Route::post('/{purchase}/complete', [PurchaseController::class, 'complete'])->name('complete');
        Route::post('/{purchase}/cancel', [PurchaseController::class, 'cancel'])->name('cancel');
        Route::get('/{purchase}/payment', [PurchaseController::class, 'payment'])->name('payment');
        Route::post('/{purchase}/payment', [PurchaseController::class, 'recordPayment'])->name('payment.store');
        Route::post('/{purchase}/approve', [PurchaseController::class, 'approve'])->name('approve');
        Route::post('/{purchase}/receive', [PurchaseController::class, 'receive'])->name('receive');
        Route::post('/{purchase}/deliver', [PurchaseController::class, 'deliver'])->name('deliver');
        Route::get('/{purchase}/check-inventory', [PurchaseController::class, 'checkInventoryAdd'])->name('check-inventory');
        Route::get('/{purchase}/details', [PurchaseController::class, 'getDetails'])->name('details');
        Route::get('/{purchase}/print-invoice', [PurchaseController::class, 'printInvoice'])->name('print-invoice');
        Route::post('/{purchase}/send-email', [PurchaseController::class, 'sendInvoiceEmail'])->name('send-email');
        Route::get('/export', [PurchaseController::class, 'export'])->name('export');
        Route::get('/export-template', [PurchaseController::class, 'exportTemplate'])->name('export-template');
        Route::post('/import', [PurchaseController::class, 'import'])->name('import');
        Route::get('/search', [PurchaseController::class, 'search'])->name('search');
        Route::get('/reports/summary', [PurchaseController::class, 'summary'])->name('reports.summary');
        Route::get('/reports/supplier', [PurchaseController::class, 'supplierReport'])->name('reports.supplier');
        Route::get('/reports/monthly', [PurchaseController::class, 'monthlyReport'])->name('reports.monthly');
        Route::get('/reports/credit', [PurchaseController::class, 'creditReport'])->name('reports.credit');
        Route::get('/{purchase}/credit-info', [PurchaseController::class, 'creditInfo'])->name('credit-info');
    });

    /*
    |--------------------------------------------------------------------------
    | Suppliers Management (Purchaser, Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('suppliers')->name('suppliers.')->middleware('role:purchaser|manager|admin')->group(function () {
        Route::get('/', [SupplierController::class, 'index'])->name('index');
        Route::get('/create', [SupplierController::class, 'create'])->name('create');
        Route::post('/', [SupplierController::class, 'store'])->name('store');
        Route::get('/{supplier}', [SupplierController::class, 'show'])->name('show');
        Route::get('/{supplier}/edit', [SupplierController::class, 'edit'])->name('edit');
        Route::put('/{supplier}', [SupplierController::class, 'update'])->name('update');
        Route::delete('/{supplier}', [SupplierController::class, 'destroy'])->name('destroy');
        Route::post('/bulk-delete', [SupplierController::class, 'bulkDelete'])->name('bulk-delete');
        Route::get('/export', [SupplierController::class, 'export'])->name('export');
        Route::get('/search', [SupplierController::class, 'search'])->name('search');
        Route::get('/statistics', [SupplierController::class, 'statistics'])->name('statistics');
        Route::get('/import', [SupplierController::class, 'import'])->name('import');
        Route::post('/import', [SupplierController::class, 'processImport'])->name('process-import');
        Route::post('/{supplier}/update-credit', [SupplierController::class, 'updateCredit'])->name('update-credit');
        Route::get('/{supplier}/credit-history', [SupplierController::class, 'creditHistory'])->name('credit-history');
        Route::get('/{supplier}/credit-info', [SupplierController::class, 'creditInfo'])->name('credit-info');
    });

    /*
    |--------------------------------------------------------------------------
    | Sales Management (Cashier, Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('sales')->name('sales.')->middleware('role:cashier|manager|admin')->group(function () {
        Route::get('/', [SaleController::class, 'index'])->name('index');
        Route::get('/create', [SaleController::class, 'create'])->name('create');
        Route::post('/', [SaleController::class, 'store'])->name('store');
        Route::get('/{sale}', [SaleController::class, 'show'])->name('show');
        Route::get('/{sale}/edit', [SaleController::class, 'edit'])->name('edit');
        Route::put('/{sale}', [SaleController::class, 'update'])->name('update');
        Route::delete('/{sale}', [SaleController::class, 'destroy'])->name('destroy');
        Route::post('/{sale}/refund', [SaleController::class, 'refund'])->name('refund');
        Route::get('/{sale}/receipt', [SaleController::class, 'receipt'])->name('receipt');
        Route::get('/{sale}/invoice', [SaleController::class, 'invoice'])->name('invoice');
        Route::post('/{sale}/send-receipt', [SaleController::class, 'sendReceipt'])->name('send-receipt');
    });

    /*
    |--------------------------------------------------------------------------
    | Printer Management Routes (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('printers')->name('printers.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [PrinterController::class, 'index'])->name('index');
        Route::get('/create', [PrinterController::class, 'create'])->name('create');
        Route::post('/', [PrinterController::class, 'store'])->name('store');
        Route::get('/{printer}', [PrinterController::class, 'show'])->name('show');
        Route::get('/{printer}/edit', [PrinterController::class, 'edit'])->name('edit');
        Route::put('/{printer}', [PrinterController::class, 'update'])->name('update');
        Route::delete('/{printer}', [PrinterController::class, 'destroy'])->name('destroy');
        
        Route::post('/{printer}/test', [PrinterController::class, 'testConnection'])->name('test');
        Route::post('/{printer}/test-print', [PrinterController::class, 'printTestPage'])->name('test-print');
        Route::get('/{printer}/health', [PrinterController::class, 'checkHealth'])->name('health');
        Route::post('/{printer}/maintenance', [PrinterController::class, 'performMaintenance'])->name('maintenance');
        Route::post('/{printer}/change-paper', [PrinterController::class, 'changePaper'])->name('change-paper');
        Route::post('/{printer}/clear-queue', [PrinterController::class, 'clearQueue'])->name('clear-queue');
        Route::post('/{printer}/reset-stats', [PrinterController::class, 'resetStatistics'])->name('reset-stats');
        Route::post('/{printer}/set-default', [PrinterController::class, 'setDefault'])->name('set-default');
        
        Route::get('/{printer}/statistics', [PrinterController::class, 'getStatistics'])->name('statistics');
        Route::get('/{printer}/queue', [PrinterController::class, 'getQueue'])->name('queue');
        Route::get('/{printer}/export', [PrinterController::class, 'exportConfiguration'])->name('export');
        Route::post('/import', [PrinterController::class, 'importConfiguration'])->name('import');
    });

    /*
    |--------------------------------------------------------------------------
    | System Settings (Admin Only)
    |--------------------------------------------------------------------------
    */
    Route::middleware(['auth', 'role:admin'])->prefix('settings')->name('settings.')->group(function () {
        // Main settings dashboard
        Route::get('/', [SettingsController::class, 'index'])->name('index');
        
        // General settings
        Route::get('/general', [SettingsController::class, 'general'])->name('general');
        Route::post('/general', [SettingsController::class, 'update'])->name('update');
        
        // Store information web routes
        Route::prefix('store-info')->name('store-information.')->group(function () {
            Route::get('/', [StoreInformationWebController::class, 'index'])->name('index');
            Route::post('/', [StoreInformationWebController::class, 'store'])->name('store');
            Route::post('/save', [StoreInformationWebController::class, 'store'])->name('save');
                
            // AJAX routes for store info
            Route::get('/health', [StoreInformationWebController::class, 'health'])->name('health');
            Route::get('/config', [StoreInformationWebController::class, 'config'])->name('config');
            Route::get('/receipt-settings', [StoreInformationWebController::class, 'receiptSettings'])->name('receipt-settings');
            Route::post('/clear-cache', [StoreInformationWebController::class, 'clearCache'])->name('clear-cache');
                Route::get('/test-receipt', [StoreInformationWebController::class, 'generateTestReceipt'])->name('test-receipt');
        });
        
        // Receipt settings routes
        Route::prefix('receipt')->name('receipt.')->group(function () {
            Route::get('/', [ReceiptSettingsController::class, 'index'])->name('index');
            Route::post('/', [ReceiptSettingsController::class, 'store'])->name('store');
            Route::get('/{id}/edit', [ReceiptSettingsController::class, 'edit'])->name('edit');
            Route::put('/{id}', [ReceiptSettingsController::class, 'update'])->name('update');
            Route::delete('/{id}', [ReceiptSettingsController::class, 'destroy'])->name('destroy');
            Route::post('/{id}/set-default', [ReceiptSettingsController::class, 'setDefault'])->name('set-default');
            Route::get('/test', [ReceiptSettingsController::class, 'testReceipt'])->name('test');
            Route::post('/clear-cache', [ReceiptSettingsController::class, 'clearCache'])->name('clear-cache');
            Route::get('/api-config', [ReceiptSettingsController::class, 'apiGetSettings'])->name('api-config');
        });
        
        // Backup settings
        Route::get('/backup', [SettingsController::class, 'backup'])->name('backup');
        Route::post('/backup/create', [SettingsController::class, 'createBackup'])->name('create-backup');
        Route::post('/backup/restore', [SettingsController::class, 'restoreBackup'])->name('restore-backup');
        
        // System logs view
        Route::get('/logs', function () {
            return view('settings.logs');
        })->name('logs');
        
        // System information view
        Route::get('/system', function () {
            $phpVersion = phpversion();
            $laravelVersion = app()->version();
            $mysqlVersion = \DB::select('SELECT VERSION() as version')[0]->version;
            $serverSoftware = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
            
            return view('settings.system', compact('phpVersion', 'laravelVersion', 'mysqlVersion', 'serverSoftware'));
        })->name('system');
    });

    /*
    |--------------------------------------------------------------------------
    | Reports & Analytics (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('reports')->name('reports.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [ReportController::class, 'index'])->name('index');
        Route::get('/sales', [ReportController::class, 'sales'])->name('sales');
        Route::get('/purchases', [ReportController::class, 'purchases'])->name('purchases');
        Route::get('/inventory', [ReportController::class, 'inventory'])->name('inventory');
        Route::get('/customers', [ReportController::class, 'customers'])->name('customers');
        Route::get('/suppliers', [ReportController::class, 'suppliers'])->name('suppliers');
        Route::get('/expiry', [ProductController::class, 'expiryReport'])->name('expiry');
        Route::get('/price-analytics', [PriceHistoryController::class, 'analytics'])->name('price-analytics');
        Route::get('/credit', [ReportController::class, 'credit'])->name('credit');
        Route::get('/payment', [ReportController::class, 'payment'])->name('payment');
        Route::get('/profit-loss', [ReportController::class, 'profitLoss'])->name('profit-loss');
        Route::get('/mpesa-transactions', [MpesaTransactionController::class, 'report'])->name('mpesa-transactions');
        Route::get('/vat-summary', [VatReportController::class, 'summary'])->name('vat-summary');
        Route::get('/vat-detailed', [VatReportController::class, 'detailed'])->name('vat-detailed');
    });

    /*
    |--------------------------------------------------------------------------
    | Price History Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('price-histories')->name('price-histories.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [PriceHistoryController::class, 'index'])->name('index');
        Route::get('/trends/{product}', [PriceHistoryController::class, 'trends'])->name('trends');
        Route::get('/compare', [PriceHistoryController::class, 'compare'])->name('compare');
        Route::get('/analytics', [PriceHistoryController::class, 'analytics'])->name('analytics');
        Route::get('/export', [PriceHistoryController::class, 'export'])->name('export');
    });

    /*
    |--------------------------------------------------------------------------
    | Units Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('units')->name('units.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [UnitController::class, 'index'])->name('index');
        Route::get('/create', [UnitController::class, 'create'])->name('create');
        Route::post('/', [UnitController::class, 'store'])->name('store');
        Route::get('/{unit}', [UnitController::class, 'show'])->name('show');
        Route::get('/{unit}/edit', [UnitController::class, 'edit'])->name('edit');
        Route::put('/{unit}', [UnitController::class, 'update'])->name('update');
        Route::delete('/{unit}', [UnitController::class, 'destroy'])->name('destroy');
        Route::get('/search', [UnitController::class, 'search'])->name('search');
        Route::post('/bulk-delete', [UnitController::class, 'bulkDelete'])->name('bulk-delete');
        Route::post('/bulk-update', [UnitController::class, 'bulkUpdate'])->name('bulk-update');
        Route::get('/export', [UnitController::class, 'export'])->name('export');
        Route::post('/convert', [UnitController::class, 'convert'])->name('convert');
    });

    /*
    |--------------------------------------------------------------------------
    | Taxes Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('taxes')->name('taxes.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [TaxController::class, 'index'])->name('index');
        Route::get('/create', [TaxController::class, 'create'])->name('create');
        Route::post('/', [TaxController::class, 'store'])->name('store');
        Route::get('/{tax}', [TaxController::class, 'show'])->name('show');
        Route::get('/{tax}/edit', [TaxController::class, 'edit'])->name('edit');
        Route::put('/{tax}', [TaxController::class, 'update'])->name('update');
        Route::delete('/{tax}', [TaxController::class, 'destroy'])->name('destroy');
        Route::get('/search', [TaxController::class, 'search'])->name('search');
        Route::post('/bulk-delete', [TaxController::class, 'bulkDelete'])->name('bulk-delete');
        Route::post('/bulk-update', [TaxController::class, 'bulkUpdate'])->name('bulk-update');
        Route::get('/export', [TaxController::class, 'export'])->name('export');
    });

    /*
    |--------------------------------------------------------------------------
    | Brands Management (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('brands')->name('brands.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [BrandController::class, 'index'])->name('index');
        Route::get('/create', [BrandController::class, 'create'])->name('create');
        Route::post('/', [BrandController::class, 'store'])->name('store');
        Route::get('/{brand}', [BrandController::class, 'show'])->name('show');
        Route::get('/{brand}/edit', [BrandController::class, 'edit'])->name('edit');
        Route::put('/{brand}', [BrandController::class, 'update'])->name('update');
        Route::delete('/{brand}', [BrandController::class, 'destroy'])->name('destroy');
        Route::get('/search', [BrandController::class, 'search'])->name('search');
        Route::post('/bulk-delete', [BrandController::class, 'bulkDelete'])->name('bulk-delete');
        Route::post('/bulk-update', [BrandController::class, 'bulkUpdate'])->name('bulk-update');
        Route::get('/export', [BrandController::class, 'export'])->name('export');
        Route::get('/statistics', [BrandController::class, 'statistics'])->name('statistics');
    });

    /*
    |--------------------------------------------------------------------------
    | User Profile (All Authenticated Users)
    |--------------------------------------------------------------------------
    */
 Route::middleware(['auth'])->group(function () {
    // Profile Routes
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [ProfileController::class, 'updatePassword'])->name('profile.password.update');
    Route::post('/profile/image', [ProfileController::class, 'updateImage'])->name('profile.image.update');
    Route::get('/profile/statistics', [ProfileController::class, 'statistics'])->name('profile.statistics');
});

    /*
    |--------------------------------------------------------------------------
    | Notifications (All Authenticated Users)
    |--------------------------------------------------------------------------
    */
    Route::prefix('notifications')->name('notifications.')->group(function () {
        Route::get('/', [NotificationController::class, 'index'])->name('index');
        Route::post('/mark-as-read', [NotificationController::class, 'markAsRead'])->name('mark-as-read');
        Route::post('/mark-all-read', [NotificationController::class, 'markAllRead'])->name('mark-all-read');
        Route::delete('/{notification}', [NotificationController::class, 'destroy'])->name('destroy');
        Route::delete('/', [NotificationController::class, 'clearAll'])->name('clear-all');
    });

    /*
    |--------------------------------------------------------------------------
    | Activity Logs (Manager, Admin)
    |--------------------------------------------------------------------------
    */
    Route::prefix('activities')->name('activities.')->middleware('role:manager|admin')->group(function () {
        Route::get('/', [ActivityLogController::class, 'index'])->name('index');
        Route::get('/export', [ActivityLogController::class, 'export'])->name('export');
        Route::get('/search', [ActivityLogController::class, 'search'])->name('search');
    });

    /*
    |--------------------------------------------------------------------------
    | Quick Search Routes (All Authenticated Users)
    |--------------------------------------------------------------------------
    */
    Route::prefix('search')->name('search.')->group(function () {
        Route::get('/products', [ProductController::class, 'search'])->name('products');
        Route::get('/customers', [CustomerController::class, 'search'])->name('customers');
        Route::get('/suppliers', [SupplierController::class, 'search'])->name('suppliers');
        Route::get('/brands', [BrandController::class, 'search'])->name('brands');
        Route::get('/categories', [CategoryController::class, 'search'])->name('categories');
        Route::get('/purchases', [PurchaseController::class, 'search'])->name('purchases');
        Route::get('/sales', [SaleController::class, 'search'])->name('sales');
    });

    /*
    |--------------------------------------------------------------------------
    | AJAX Routes for Dynamic Content
    |--------------------------------------------------------------------------
    */
    Route::prefix('ajax')->name('ajax.')->group(function () {
        Route::get('/products', [ProductController::class, 'ajaxSearch'])->name('products');
        Route::get('/categories', [CategoryController::class, 'ajaxDropdown'])->name('categories');
        Route::get('/customers', [CustomerController::class, 'ajaxSearch'])->name('customers');
        Route::get('/products/{product}', [ProductController::class, 'ajaxGetProduct'])->name('products.show');
        Route::get('/suppliers', [SupplierController::class, 'ajaxSearch'])->name('suppliers');
        Route::get('/purchases', [PurchaseController::class, 'ajaxSearch'])->name('purchases');
        Route::get('/categories/tree', [CategoryController::class, 'ajaxTree'])->name('categories.tree');
        Route::get('/brands', [BrandController::class, 'ajaxSearch'])->name('brands');
        Route::get('/units', [UnitController::class, 'ajaxSearch'])->name('units');
        Route::get('/taxes', [TaxController::class, 'ajaxSearch'])->name('taxes');
        Route::get('/mpesa/transactions', [MpesaTransactionController::class, 'ajaxSearch'])->name('mpesa.transactions');
    });

    /*
    |--------------------------------------------------------------------------
    | Print/Export Routes
    |--------------------------------------------------------------------------
    */
    Route::prefix('print')->name('print.')->group(function () {
        Route::get('/sale/{sale}', [SaleController::class, 'printReceipt'])->name('sale.receipt');
        Route::get('/sale/{sale}/invoice', [SaleController::class, 'printInvoice'])->name('sale.invoice');
        Route::get('/purchase/{purchase}', [PurchaseController::class, 'printInvoice'])->name('purchase.invoice');
        Route::get('/customer/{customer}/statement', [CustomerController::class, 'printStatement'])->name('customer.statement');
        Route::get('/supplier/{supplier}/statement', [SupplierController::class, 'printStatement'])->name('supplier.statement');
        Route::get('/pos-receipt/{sale}', [PosController::class, 'printReceipt'])->name('pos.receipt');
        Route::get('/mpesa-transaction/{transaction}', [MpesaTransactionController::class, 'printTransaction'])->name('mpesa.transaction');
        Route::post('/receipt-pdf', [ReceiptPDFController::class, 'generatePDF'])->name('receipt-pdf');
    });

    /*
    |--------------------------------------------------------------------------
    | Backup and Export Routes (Admin Only)
    |--------------------------------------------------------------------------
    */
    Route::middleware('role:admin')->prefix('tools')->name('tools.')->group(function () {
        Route::get('/export-data', [ExportController::class, 'index'])->name('export.index');
        Route::post('/export-data', [ExportController::class, 'export'])->name('export');
        
        Route::get('/import-data', [ImportController::class, 'index'])->name('import.index');
        Route::post('/import-data', [ImportController::class, 'import'])->name('import');
        
        Route::get('/database', [DatabaseController::class, 'index'])->name('database.index');
        Route::post('/database/optimize', [DatabaseController::class, 'optimize'])->name('database.optimize');
        Route::post('/database/clean', [DatabaseController::class, 'clean'])->name('database.clean');
    });

    /*
    |--------------------------------------------------------------------------
    | Audit Logs (Admin Only)
    |--------------------------------------------------------------------------
    */
    Route::middleware('role:admin')->prefix('audit')->name('audit.')->group(function () {
        Route::get('/', [AuditController::class, 'index'])->name('index');
        Route::get('/export', [AuditController::class, 'export'])->name('export');
        Route::get('/user/{user}', [AuditController::class, 'userLogs'])->name('user');
    });
    
    /*
    |--------------------------------------------------------------------------
    | Developer Routes (Local Environment Only)
    |--------------------------------------------------------------------------
    */
    if (app()->environment('local')) {
        Route::get('/dev/pos-test', function() {
            return view('pos.test');
        })->name('dev.pos-test');
        
        Route::get('/dev/clear-cache', function() {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            return 'Cache cleared successfully!';
        });
        
        Route::get('/dev/mpesa-test', [MpesaController::class, 'testPage'])->name('dev.mpesa-test');
        Route::post('/dev/mpesa-simulate', [MpesaTransactionController::class, 'simulateCallback'])->name('dev.mpesa-simulate');
    }
});

/*
|--------------------------------------------------------------------------
| Public M-Pesa Callback Route (No Authentication)
|--------------------------------------------------------------------------
*/
Route::prefix('api/mpesa')->group(function () {
    Route::post('/callback', [MpesaTransactionController::class, 'storeCallback'])
        ->withoutMiddleware(['web', 'csrf'])
        ->name('api.mpesa.callback.public');
});

/*
|--------------------------------------------------------------------------
| Emergency Routes
|--------------------------------------------------------------------------
*/
Route::get('/api/emergency/payment-methods', function() {
    return response()->json([
        'success' => true,
        'methods' => [
            ['code' => 'cash', 'name' => 'Cash', 'icon' => 'money', 'color' => '#28a745', 'is_active' => true],
            ['code' => 'mpesa', 'name' => 'M-Pesa', 'icon' => 'mobile-alt', 'color' => '#4facfe', 'is_active' => true],
            ['code' => 'card', 'name' => 'Card', 'icon' => 'credit-card', 'color' => '#007bff', 'is_active' => true],
        ],
        'emergency' => true,
        'timestamp' => now()->toISOString()
    ]);
})->name('api.emergency.payment-methods');



/*
|--------------------------------------------------------------------------
| POS Receipt Routes - Production Ready
|--------------------------------------------------------------------------
*/
Route::prefix('pos')->name('pos.')->group(function () {
    // ==================== PAYMENT PROCESSING ====================
    Route::post('/payments/process', [PaymentController::class, 'process'])
        ->name('payments.process');
    
    // ==================== RECEIPT ROUTES WITH MULTIPLE FALLBACKS ====================
    
    // Primary receipt route (supports invoice_no, receipt_no, or sale ID)
    Route::get('/receipt/print/{identifier}', [PaymentController::class, 'showReceipt'])
        ->where('identifier', '.*') // Accept any characters including dashes
        ->name('receipt.print');
    
    // Alternative receipt routes for compatibility
    Route::get('/receipt/{identifier}', [PaymentController::class, 'showReceipt'])
        ->where('identifier', '.*')
        ->name('receipt.show');
    
    Route::get('/invoice/{identifier}', [PaymentController::class, 'showReceipt'])
        ->where('identifier', '.*')
        ->name('invoice.show');
    
    Route::get('/print-receipt/{identifier}', [PaymentController::class, 'showReceipt'])
        ->where('identifier', '.*')
        ->name('print.receipt');
    
    // Receipt preview with data
    Route::get('/receipt/preview', function () {
        // Get receipt data from request or session
        $receiptData = request()->has('data') 
            ? json_decode(request('data'), true)
            : session()->get('receipt_preview_data', $this->getDefaultReceiptData());
        
        return view('pos.receipts.receipt', [
            'data' => $receiptData,
            'autoprint' => request('autoprint', false)
        ]);
    })->name('receipt.preview');
    
    // Receipt download as PDF
    Route::get('/receipt/{identifier}/download', [PaymentController::class, 'downloadReceipt'])
        ->name('receipt.download');
    
    // Quick receipt endpoint for testing
    Route::get('/receipt/test/{type?}', function ($type = 'simple') {
        $testData = [
            'simple' => [
                'store' => [
                    'name' => 'TEST STORE - ' . config('app.name', 'Kenyan Supermarket'),
                    'address' => config('app.address', 'Nairobi CBD'),
                    'phone' => config('app.phone', '0700 000 000'),
                    'pin' => 'P051234567N',
                    'vat_number' => 'VAT001234567',
                ],
                'invoice_no' => 'TEST-INV-' . date('Ymd') . '-001',
                'receipt_no' => 'TEST-RCP-' . date('Ymd') . '-001',
                'date' => date('Y-m-d'),
                'time' => date('H:i:s'),
                'cashier' => 'Test Cashier',
                'customer' => ['name' => 'Test Customer', 'phone' => '0712345678'],
                'items' => [
                    ['name' => 'Test Product 1', 'quantity' => 2, 'unit_price' => 50, 'total' => 100],
                    ['name' => 'Test Product 2', 'quantity' => 1, 'unit_price' => 100, 'total' => 100],
                ],
                'totals' => ['subtotal' => 200, 'vat_amount' => 32, 'grand_total' => 232],
                'payment' => ['method' => 'cash', 'amount_paid' => 250, 'change' => 18],
                'autoprint' => request('autoprint', false),
            ],
            'vat' => [
                'store' => [
                    'name' => 'VAT TEST STORE',
                    'address' => 'Nairobi CBD',
                    'phone' => '0700 000 000',
                    'pin' => 'P051234567N',
                    'vat_number' => 'VAT001234567',
                ],
                'invoice_no' => 'VAT-INV-' . date('Ymd') . '-001',
                'receipt_no' => 'VAT-RCP-' . date('Ymd') . '-001',
                'date' => date('Y-m-d'),
                'time' => date('H:i:s'),
                'cashier' => 'VAT Cashier',
                'customer' => [
                    'name' => 'VAT Customer',
                    'phone' => '0712345678',
                    'vat_status' => 'vatable',
                    'vat_number' => 'VAT987654321'
                ],
                'items' => [
                    [
                        'name' => 'VATable Product',
                        'quantity' => 2,
                        'unit_price' => 50,
                        'total' => 100,
                        'tax_rate' => 16,
                        'tax_amount' => 16,
                        'is_vatable' => true,
                    ],
                    [
                        'name' => 'Non-VAT Product',
                        'quantity' => 1,
                        'unit_price' => 100,
                        'total' => 100,
                        'tax_rate' => 0,
                        'tax_amount' => 0,
                        'is_vatable' => false,
                    ],
                ],
                'totals' => [
                    'subtotal' => 200,
                    'discount' => 10,
                    'taxable_amount' => 100,
                    'non_taxable_amount' => 100,
                    'exempted_amount' => 0,
                    'zero_rated_amount' => 0,
                    'vat_amount' => 16,
                    'grand_total' => 206,
                ],
                'vat_info' => [
                    'customer_status' => 'vatable',
                    'is_vat_applicable' => true,
                    'is_zero_rated' => false,
                    'is_exempted' => false,
                    'breakdown' => [['rate' => 16, 'amount' => 16, 'taxable_amount' => 100]],
                    'total_vat' => 16,
                ],
                'payment' => [
                    'method' => 'mpesa',
                    'amount_paid' => 206,
                    'mpesa_details' => ['phone' => '0712345678', 'transaction_id' => 'MPESA123456']
                ],
                'autoprint' => request('autoprint', false),
            ]
        ];
        
        return view('pos.receipts.receipt', [
            'data' => $testData[$type] ?? $testData['simple'],
            'autoprint' => request('autoprint', false)
        ]);
    })->name('receipt.test');
    
    // Legacy compatibility route (keep for existing references)
    Route::get('/receipt/legacy/{id}', function ($id) {
        try {
            if ($id === 'preview' && request()->has('data')) {
                $receiptData = json_decode(request('data'), true);
            } else {
                $sale = \App\Models\Sale::with(['items.product', 'customer', 'user', 'payments'])->findOrFail($id);
                $receiptData = [
                    'store' => [
                        'name' => config('app.name', 'Kenyan Supermarket'),
                        'address' => config('app.address', 'Nairobi CBD'),
                        'phone' => config('app.phone', '0700 000 000'),
                        'vat_number' => config('app.vat_number', 'VAT001234567'),
                        'pin' => config('app.pin', 'P051234567N'),
                    ],
                    'invoice_no' => $sale->invoice_no,
                    'receipt_no' => $sale->receipt_no,
                    'date' => $sale->sale_date->format('Y-m-d') ?? $sale->created_at->format('Y-m-d'),
                    'time' => $sale->sale_date ? $sale->sale_date->format('H:i:s') : $sale->created_at->format('H:i:s'),
                    'cashier' => $sale->user->name ?? 'Cashier',
                    'customer' => $sale->customer ? [
                        'name' => $sale->customer->name,
                        'phone' => $sale->customer->phone,
                        'email' => $sale->customer->email,
                        'pin' => $sale->customer->pin,
                        'vat_number' => $sale->customer->vat_number,
                        'vat_status' => $sale->customer->vat_status ?? 'vatable',
                    ] : null,
                    'items' => $sale->items->map(function ($item) {
                        return [
                            'name' => $item->product->name ?? 'Product',
                            'quantity' => $item->qty ?? $item->quantity ?? 1,
                            'unit_price' => $item->price,
                            'total' => $item->total,
                            'vat_included' => ($item->tax_rate ?? 0) > 0,
                            'tax_rate' => $item->tax_rate ?? 0,
                            'tax_amount' => $item->tax_amount ?? 0,
                        ];
                    }),
                    'totals' => [
                        'subtotal' => $sale->subtotal,
                        'discount' => $sale->discount,
                        'vat_amount' => $sale->tax_total ?? $sale->vat_amount ?? 0,
                        'vatable_amount' => $sale->taxable_amount ?? ($sale->subtotal - $sale->discount),
                        'non_taxable_amount' => $sale->non_taxable_amount ?? 0,
                        'grand_total' => $sale->grand_total,
                    ],
                    'payment' => [
                        'method' => $sale->payment_method,
                        'amount_paid' => $sale->grand_total,
                        'transaction_id' => $sale->transaction_id,
                        ...($sale->payment_method === 'cash' ? [
                            'cash_received' => $sale->cash_received,
                            'change' => $sale->customer_change,
                        ] : []),
                        ...($sale->payment_method === 'mpesa' ? [
                            'mpesa_details' => [
                                'phone' => $sale->customer->phone ?? '',
                                'transaction_id' => $sale->transaction_id,
                            ]
                        ] : []),
                    ],
                    'autoprint' => request('autoprint', false),
                ];
            }
            
            return view('pos.receipts.receipt', ['data' => $receiptData]);
            
        } catch (\Exception $e) {
            \Log::error('Legacy receipt error: ' . $e->getMessage());
            return response()->view('errors.404', [
                'message' => 'Receipt not found: ' . $id
            ], 404);
        }
    })->name('receipt.legacy');
});

// ==================== DIRECT RECEIPT ROUTES (NO POS PREFIX) ====================
// These routes are for direct access and external systems

Route::get('/receipt/{identifier}', [PaymentController::class, 'showReceipt'])
    ->where('identifier', '.*')
    ->name('receipt.direct');

Route::get('/invoice/{identifier}', [PaymentController::class, 'showReceipt'])
    ->where('identifier', '.*')
    ->name('invoice.direct');

Route::get('/print/{identifier}', [PaymentController::class, 'showReceipt'])
    ->where('identifier', '.*')
    ->name('print.direct');

// ==================== API RECEIPT ROUTES ====================
Route::prefix('api')->name('api.')->group(function () {
    Route::get('/receipt/{identifier}', function ($identifier) {
        try {
            $sale = \App\Models\Sale::with(['items.product', 'customer', 'user'])
                ->where('invoice_no', $identifier)
                ->orWhere('receipt_no', $identifier)
                ->orWhere('id', $identifier)
                ->firstOrFail();
            
            return response()->json([
                'success' => true,
                'receipt' => [
                    'invoice_no' => $sale->invoice_no,
                    'receipt_no' => $sale->receipt_no,
                    'date' => $sale->created_at->format('Y-m-d H:i:s'),
                    'grand_total' => $sale->grand_total,
                    'customer' => $sale->customer ? $sale->customer->name : 'Walk-in',
                    'items_count' => $sale->items->count(),
                    'print_url' => url("/pos/receipt/print/{$sale->invoice_no}?autoprint=true"),
                    'view_url' => url("/pos/receipt/{$sale->invoice_no}"),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Receipt not found',
                'identifier' => $identifier
            ], 404);
        }
    })->name('receipt.info');
});

// Helper function for default receipt data
if (!function_exists('getDefaultReceiptData')) {
    function getDefaultReceiptData() {
        return [
            'store' => [
                'name' => config('app.name', 'POS System'),
                'address' => config('app.address', ''),
                'phone' => config('app.phone', ''),
                'vat_number' => config('app.vat_number', ''),
            ],
            'invoice_no' => 'PREVIEW-' . date('Ymd-His'),
            'receipt_no' => 'PREVIEW-' . date('Ymd-His'),
            'date' => date('Y-m-d'),
            'time' => date('H:i:s'),
            'cashier' => 'Preview User',
            'items' => [
                ['name' => 'Preview Item 1', 'quantity' => 1, 'unit_price' => 100, 'total' => 100],
                ['name' => 'Preview Item 2', 'quantity' => 2, 'unit_price' => 50, 'total' => 100],
            ],
            'totals' => [
                'subtotal' => 200,
                'discount' => 0,
                'vat_amount' => 32,
                'grand_total' => 232,
            ],
            'payment' => [
                'method' => 'cash',
                'amount_paid' => 250,
                'change' => 18,
            ],
            'autoprint' => false,
            'notes' => ['This is a preview receipt for testing purposes.'],
        ];
    }
}

/*
|--------------------------------------------------------------------------
| Public Store Information API Routes
|--------------------------------------------------------------------------
*/
Route::prefix('api/store')->name('api.store.')->group(function () {
    Route::get('/info', [\App\Http\Controllers\StoreInformationController::class, 'index'])->name('info');
    Route::post('/info', [\App\Http\Controllers\StoreInformationController::class, 'store'])->middleware('auth')->name('update-info');
    Route::get('/pos-info', [\App\Http\Controllers\StoreInformationController::class, 'posInfo'])->name('pos-info');
    Route::get('/receipt-settings', [\App\Http\Controllers\StoreInformationController::class, 'receiptSettings'])->name('receipt-settings');
    Route::get('/health', [\App\Http\Controllers\StoreInformationController::class, 'health'])->name('health');
});


// User Management Routes
Route::prefix('users')->name('users.')->middleware(['auth'])->group(function () {
    Route::get('/', [UserController::class, 'index'])->name('index');
    Route::get('/create', [UserController::class, 'create'])->name('create');
    Route::post('/', [UserController::class, 'store'])->name('store');
    Route::get('/{user}', [UserController::class, 'show'])->name('show');
    Route::get('/{user}/edit', [UserController::class, 'edit'])->name('edit');
    Route::put('/{user}', [UserController::class, 'update'])->name('update');
    Route::delete('/{user}', [UserController::class, 'destroy'])->name('destroy');
    
    // Additional routes
    Route::put('/{user}/status', [UserController::class, 'updateStatus'])->name('status.update');
    Route::post('/{user}/profile-image', [UserController::class, 'updateProfileImage'])->name('profile-image.update');
    Route::post('/bulk-delete', [UserController::class, 'bulkDelete'])->name('bulk.delete');
    Route::get('/export', [UserController::class, 'export'])->name('export');
    Route::get('/import', [UserController::class, 'import'])->name('import');
    Route::post('/import', [UserController::class, 'processImport'])->name('import.process');
    Route::get('/statistics', [UserController::class, 'statistics'])->name('statistics');
    Route::delete('/{user}/force', [UserController::class, 'forceDelete'])->name('force.delete');
    Route::post('/{user}/reset-password', [UserController::class, 'resetPassword'])->name('reset-password');
});



/*
|--------------------------------------------------------------------------
| Fallback Route
|--------------------------------------------------------------------------
*/
Route::fallback(function () {
    if (request()->expectsJson()) {
        return response()->json([
            'error' => 'Route not found',
            'message' => 'The requested route does not exist'
        ], 404);
    }
    
    return response()->view('errors.404', [], 404);
});

// routes/web.php
Route::middleware(['web'])->group(function () {
    // Unified receipt system
    Route::prefix('receipt')->group(function () {
        Route::get('/{identifier}', [UnifiedReceiptController::class, 'show'])
            ->name('receipt.show');
        
        Route::get('/{identifier}/download', [UnifiedReceiptController::class, 'download'])
            ->name('receipt.download');
            
        Route::get('/{identifier}/verify', [UnifiedReceiptController::class, 'verify'])
            ->name('receipt.verify');
    });
});

// routes/api.php
Route::middleware(['api'])->group(function () {
    Route::get('/receipt/{identifier}', [UnifiedReceiptController::class, 'show']);
    Route::get('/receipt/{identifier}/verify', [UnifiedReceiptController::class, 'verify']);
});
