@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            border-left: 4px solid #667eea;
        }
        .form-section h5 {
            color: #495057;
            border-bottom: 2px solid #dee2e6;
            padding-bottom: 10px;
            margin-bottom: 20px;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .optional-field::after {
            content: " (optional)";
            color: #6c757d;
            font-weight: normal;
        }
        .switch-label {
            cursor: pointer;
        }
        .base-unit-info {
            background: #e7f3ff;
            border-radius: 8px;
            padding: 15px;
            margin-top: 10px;
            border: 1px solid #b3d7ff;
        }
        .conversion-help {
            font-size: 0.85em;
            color: #6c757d;
        }
        .info-text {
            color: #6c757d;
            font-size: 0.875rem;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Units</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('units.index') }}">Units</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Edit Unit</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('units.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to List
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Edit Unit: {{ $unit->name }}</h4>
                        
                        @if($errors->any())
                            <div class="alert alert-danger">
                                <ul class="mb-0">
                                    @foreach($errors->all() as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                </ul>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('units.update', $unit) }}" id="unitForm">
                            @csrf
                            @method('PUT')
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h5><i class="bx bx-info-circle"></i> Basic Information</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Unit Name</label>
                                        <input type="text" name="name" class="form-control" 
                                               value="{{ old('name', $unit->name) }}" required 
                                               placeholder="e.g., Kilogram, Liter, Piece">
                                        <div class="form-text">The full name of the unit</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Abbreviation</label>
                                        <input type="text" name="abbreviation" class="form-control" 
                                               value="{{ old('abbreviation', $unit->abbreviation) }}" required 
                                               placeholder="e.g., kg, L, pcs" maxlength="10">
                                        <div class="form-text">Short form (max 10 characters)</div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label optional-field">Description</label>
                                    <textarea name="description" class="form-control" rows="2" 
                                              placeholder="Optional description">{{ old('description', $unit->description) }}</textarea>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Status</label>
                                        <select name="status" class="form-select" required>
                                            <option value="active" {{ old('status', $unit->status) == 'active' ? 'selected' : '' }}>
                                                Active
                                            </option>
                                            <option value="inactive" {{ old('status', $unit->status) == 'inactive' ? 'selected' : '' }}>
                                                Inactive
                                            </option>
                                        </select>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <div class="form-check form-switch mt-4">
                                            <input class="form-check-input" type="checkbox" name="is_base_unit" 
                                                   id="isBaseUnit" value="1" 
                                                   {{ old('is_base_unit', $unit->is_base_unit) ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="isBaseUnit">
                                                Mark as Base Unit
                                            </label>
                                        </div>
                                        <div class="info-text">
                                            Base units don't require a parent unit. Uncheck to create a derived unit.
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Conversion Settings -->
                            <div class="form-section" id="conversionSection">
                                <h5><i class="bx bx-calculator"></i> Conversion Settings (Optional)</h5>
                                
                                <div class="info-text mb-3">
                                    <i class="bx bx-info-circle"></i> These fields are optional. Leave empty if this is a base unit.
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label optional-field">Base Unit</label>
                                        <select name="base_unit_id" id="baseUnitSelect" class="form-select">
                                            <option value="">-- Select Base Unit (Optional) --</option>
                                            @foreach($baseUnits as $baseUnit)
                                                <option value="{{ $baseUnit->id }}" 
                                                        {{ old('base_unit_id', $unit->base_unit_id) == $baseUnit->id ? 'selected' : '' }}>
                                                    {{ $baseUnit->name }} ({{ $baseUnit->abbreviation }})
                                                </option>
                                            @endforeach
                                        </select>
                                        <div class="form-text">Select if this unit is derived from another unit</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label optional-field">Conversion Rate</label>
                                        <div class="input-group">
                                            <span class="input-group-text">1</span>
                                            <span class="input-group-text" id="baseUnitAbbr">
                                                @if($unit->baseUnit)
                                                    {{ $unit->baseUnit->abbreviation }}
                                                @else
                                                    [Base]
                                                @endif
                                            </span>
                                            <span class="input-group-text">=</span>
                                            <input type="number" name="conversion_rate" id="conversionRate" 
                                                   class="form-control" step="0.0001" min="0.0001" 
                                                   value="{{ old('conversion_rate', $unit->conversion_rate) }}" placeholder="1.0000">
                                            <span class="input-group-text">[This Unit]</span>
                                        </div>
                                        <div class="conversion-help mt-1">
                                            <small>Example: 1 Meter = 100 Centimeters (rate = 100)</small>
                                        </div>
                                    </div>
                                </div>
                                
                                @if($unit->derivedUnits()->count() > 0)
                                    <div class="alert alert-warning mt-3">
                                        <i class="bx bx-error"></i> 
                                        <strong>Note:</strong> This unit is used as a base unit for 
                                        {{ $unit->derivedUnits()->count() }} other units. 
                                        Changing it to a derived unit may affect those conversions.
                                    </div>
                                @endif
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5">
                                                <i class="bx bx-save"></i> Update Unit
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5">
                                                <i class="bx bx-reset"></i> Reset Changes
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('units.index') }}" class="btn btn-light px-5">
                                                <i class="bx bx-x"></i> Cancel
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // Toggle conversion section based on base unit checkbox
        function toggleConversionSection() {
            const isBaseUnit = $('#isBaseUnit').is(':checked');
            
            if (isBaseUnit) {
                $('#conversionSection').hide();
                $('#baseUnitSelect').prop('required', false);
                $('#conversionRate').prop('required', false);
            } else {
                $('#conversionSection').show();
                $('#baseUnitSelect').prop('required', false);
                $('#conversionRate').prop('required', false);
            }
        }
        
        // Initial toggle
        toggleConversionSection();
        
        // Bind change event
        $('#isBaseUnit').change(toggleConversionSection);
        
        // Update base unit abbreviation display
        function updateBaseUnitAbbr() {
            const selectedOption = $('#baseUnitSelect option:selected');
            if (selectedOption.val()) {
                const text = selectedOption.text();
                const abbr = text.match(/\(([^)]+)\)/);
                $('#baseUnitAbbr').text(abbr ? abbr[1] : 'Base');
            } else {
                $('#baseUnitAbbr').text('[Base]');
            }
        }
        
        // Initial update
        updateBaseUnitAbbr();
        
        // Bind change event
        $('#baseUnitSelect').change(updateBaseUnitAbbr);
        
        // Form validation - Simplified for optional fields
        $('#unitForm').validate({
            rules: {
                name: {
                    required: true,
                    minlength: 2,
                    maxlength: 255
                },
                abbreviation: {
                    required: true,
                    minlength: 1,
                    maxlength: 10
                },
                conversion_rate: {
                    number: true,
                    min: 0.0001
                }
            },
            messages: {
                name: {
                    required: "Unit name is required",
                    minlength: "Unit name must be at least 2 characters"
                },
                abbreviation: {
                    required: "Abbreviation is required",
                    minlength: "Abbreviation must be at least 1 character"
                },
                conversion_rate: {
                    number: "Conversion rate must be a number",
                    min: "Conversion rate must be greater than 0"
                }
            },
            errorElement: 'span',
            errorPlacement: function (error, element) {
                error.addClass('invalid-feedback');
                element.closest('.mb-3').append(error);
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass('is-invalid');
            },
            submitHandler: function(form) {
                // Additional custom validation
                const isBaseUnit = $('#isBaseUnit').is(':checked');
                const baseUnitId = $('#baseUnitSelect').val();
                const conversionRate = $('#conversionRate').val();
                const unitId = {{ $unit->id }};
                
                if (!isBaseUnit && baseUnitId && !conversionRate) {
                    $('#conversionRate').addClass('is-invalid');
                    $('#conversionRate').after('<span class="invalid-feedback">Conversion rate is required when a base unit is selected.</span>');
                    return false;
                }
                
                if (isBaseUnit && baseUnitId) {
                    $('#baseUnitSelect').addClass('is-invalid');
                    $('#baseUnitSelect').after('<span class="invalid-feedback">Base unit cannot have a parent base unit.</span>');
                    return false;
                }
                
                if (baseUnitId == unitId) {
                    $('#baseUnitSelect').addClass('is-invalid');
                    $('#baseUnitSelect').after('<span class="invalid-feedback">A unit cannot be its own base unit.</span>');
                    return false;
                }
                
                return true;
            }
        });
        
        // Auto-toggle is_base_unit based on base_unit_id selection
        $('#baseUnitSelect').change(function() {
            if ($(this).val()) {
                // If base unit is selected, uncheck is_base_unit
                $('#isBaseUnit').prop('checked', false).trigger('change');
            }
        });
        
        // Clear base unit selection when marking as base unit
        $('#isBaseUnit').change(function() {
            if ($(this).is(':checked')) {
                $('#baseUnitSelect').val('').trigger('change');
            }
        });
    });
</script>
@endsection