@extends('layouts.app')

@section('title', 'Backup & Restore')


@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Settings</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('settings.index') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Backup & Restore</li>
                    </ol>
                </nav>
            </div>
        </div>
        <!--end breadcrumb-->
        
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <!-- Backup Statistics Cards -->
                        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-4 g-3 mb-4">
                            <div class="col">
                                <div class="card radius-10 border-start border-0 border-4 border-success">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <p class="mb-0 text-secondary">Total Backups</p>
                                                <h4 class="my-1" id="totalBackups">0</h4>
                                            </div>
                                            <div class="widgets-icons-2 rounded-circle bg-gradient-ohhappiness text-white ms-auto">
                                                <i class='bx bx-data'></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col">
                                <div class="card radius-10 border-start border-0 border-4 border-info">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <p class="mb-0 text-secondary">Last Backup</p>
                                                <h4 class="my-1" id="lastBackup">Never</h4>
                                            </div>
                                            <div class="widgets-icons-2 rounded-circle bg-gradient-scooter text-white ms-auto">
                                                <i class='bx bx-time-five'></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col">
                                <div class="card radius-10 border-start border-0 border-4 border-warning">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <p class="mb-0 text-secondary">Total Size</p>
                                                <h4 class="my-1" id="totalSize">0 MB</h4>
                                            </div>
                                            <div class="widgets-icons-2 rounded-circle bg-gradient-burning text-white ms-auto">
                                                <i class='bx bx-hdd'></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col">
                                <div class="card radius-10 border-start border-0 border-4 border-danger">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div>
                                                <p class="mb-0 text-secondary">Next Backup</p>
                                                <h4 class="my-1" id="nextBackup">Disabled</h4>
                                            </div>
                                            <div class="widgets-icons-2 rounded-circle bg-gradient-bloody text-white ms-auto">
                                                <i class='bx bx-calendar'></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Backup Actions -->
                        <div class="card mb-4">
                            <div class="card-body">
                                <div class="d-flex flex-wrap gap-2">
                                    <button type="button" class="btn btn-primary px-4" onclick="createBackup()">
                                        <i class="bx bx-plus-circle mr-1"></i>Create New Backup
                                    </button>
                                    <button type="button" class="btn btn-success px-4" onclick="createDatabaseBackup()">
                                        <i class="bx bx-data mr-1"></i>Database Only
                                    </button>
                                    <button type="button" class="btn btn-warning px-4" onclick="createFullBackup()">
                                        <i class="bx bx-server mr-1"></i>Full System
                                    </button>
                                    <div class="ms-auto">
                                        <button type="button" class="btn btn-danger px-4" onclick="deleteSelectedBackups()" id="deleteSelectedBtn" disabled>
                                            <i class="bx bx-trash mr-1"></i>Delete Selected
                                        </button>
                                        <button type="button" class="btn btn-info px-4" onclick="restoreBackupModal()">
                                            <i class="bx bx-reset mr-1"></i>Restore
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Backup List -->
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table id="backupTable" class="table table-striped table-bordered" style="width:100%">
                                        <thead>
                                            <tr>
                                                <th width="50">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                                    </div>
                                                </th>
                                                <th>Backup Name</th>
                                                <th width="100">Type</th>
                                                <th width="120">Size</th>
                                                <th width="150">Created At</th>
                                                <th width="100">Status</th>
                                                <th width="150">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody id="backupList">
                                            <!-- Backups will be loaded here -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Backup Settings -->
                        <div class="row mt-4">
                            <div class="col-lg-6">
                                <div class="card">
                                    <div class="card-header bg-transparent">
                                        <h5 class="mb-0"><i class="bx bx-cog mr-2"></i>Backup Settings</h5>
                                    </div>
                                    <div class="card-body">
                                        <form id="backupSettingsForm">
                                            <div class="row g-3">
                                                <div class="col-12">
                                                    <label class="form-label">Backup Frequency</label>
                                                    <select class="form-select" name="frequency">
                                                        <option value="disabled">Disabled</option>
                                                        <option value="hourly">Hourly</option>
                                                        <option value="daily" selected>Daily</option>
                                                        <option value="weekly">Weekly</option>
                                                        <option value="monthly">Monthly</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="col-12">
                                                    <label class="form-label">Backup Retention</label>
                                                    <select class="form-select" name="retention">
                                                        <option value="7">7 days</option>
                                                        <option value="30" selected>30 days</option>
                                                        <option value="90">90 days</option>
                                                        <option value="180">180 days</option>
                                                        <option value="365">1 year</option>
                                                        <option value="0">Keep forever</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="col-12">
                                                    <label class="form-label">Backup Type</label>
                                                    <div class="form-check mb-2">
                                                        <input class="form-check-input" type="checkbox" name="include_database" id="includeDatabase" checked>
                                                        <label class="form-check-label" for="includeDatabase">
                                                            Include Database
                                                        </label>
                                                    </div>
                                                    <div class="form-check mb-2">
                                                        <input class="form-check-input" type="checkbox" name="include_files" id="includeFiles">
                                                        <label class="form-check-label" for="includeFiles">
                                                            Include Uploaded Files
                                                        </label>
                                                    </div>
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="checkbox" name="compress" id="compress" checked>
                                                        <label class="form-check-label" for="compress">
                                                            Compress Backup (ZIP)
                                                        </label>
                                                    </div>
                                                </div>
                                                
                                                <div class="col-12">
                                                    <div class="d-grid">
                                                        <button type="button" class="btn btn-primary" onclick="saveBackupSettings()">
                                                            <i class="bx bx-save mr-1"></i>Save Settings
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-lg-6">
                                <div class="card">
                                    <div class="card-header bg-transparent">
                                        <h5 class="mb-0"><i class="bx bx-pie-chart-alt mr-2"></i>Statistics</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="mb-4">
                                            <h6 class="mb-3">Storage Usage</h6>
                                            <div class="progress mb-2" style="height: 20px;">
                                                <div class="progress-bar bg-success" role="progressbar" style="width: 65%;">65% Used</div>
                                            </div>
                                            <div class="d-flex justify-content-between small">
                                                <span>650 MB Used</span>
                                                <span>350 MB Free</span>
                                                <span>1 GB Total</span>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-4">
                                            <h6 class="mb-3">Last 30 Days</h6>
                                            <div class="d-flex align-items-center mb-2">
                                                <div class="flex-grow-1 me-3">
                                                    <small>Successful</small>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-success" style="width: 80%"></div>
                                                    </div>
                                                </div>
                                                <small class="text-success">24</small>
                                            </div>
                                            <div class="d-flex align-items-center mb-2">
                                                <div class="flex-grow-1 me-3">
                                                    <small>Partial</small>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-warning" style="width: 15%"></div>
                                                    </div>
                                                </div>
                                                <small class="text-warning">4</small>
                                            </div>
                                            <div class="d-flex align-items-center">
                                                <div class="flex-grow-1 me-3">
                                                    <small>Failed</small>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-danger" style="width: 5%"></div>
                                                    </div>
                                                </div>
                                                <small class="text-danger">2</small>
                                            </div>
                                        </div>
                                        
                                        <div class="alert alert-light border">
                                            <i class="bx bx-info-circle mr-2"></i>
                                            <strong>Backup Tips:</strong>
                                            <ul class="mb-0 mt-2 ps-3">
                                                <li>Regular backups prevent data loss</li>
                                                <li>Test restore periodically</li>
                                                <li>Store in multiple locations</li>
                                                <li>Encrypt sensitive data</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!--end page wrapper -->

<!-- Restore Backup Modal -->
<div class="modal fade" id="restoreModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-reset mr-2"></i>Restore Backup</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning border-0">
                    <i class="bx bx-error-circle mr-2"></i>
                    <strong>Warning:</strong> This will replace all current data!
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Select Backup</label>
                    <select class="form-select" id="restoreSelect">
                        <option value="">Choose backup...</option>
                    </select>
                </div>
                
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="confirmRestore">
                    <label class="form-check-label" for="confirmRestore">
                        I understand this cannot be undone
                    </label>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="performRestore()" id="restoreBtn" disabled>
                    <i class="bx bx-reset mr-1"></i>Restore Now
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Backup Progress Modal -->
<div class="modal fade" id="backupProgressModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bx bx-loader-circle bx-spin mr-2"></i>
                    <span id="progressTitle">Creating Backup</span>
                </h5>
            </div>
            <div class="modal-body">
                <div class="progress mb-3">
                    <div class="progress-bar progress-bar-striped progress-bar-animated" 
                         role="progressbar" style="width: 0%" id="backupProgressBar">0%</div>
                </div>
                <div id="backupStatus" class="text-center">Initializing...</div>
                <div id="backupDetails" class="small text-muted mt-2 text-center"></div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

<script>
// Global variables
let selectedBackups = [];
let backupInterval = null;

// Initialize on page load
$(document).ready(function() {
    loadBackups();
    loadBackupStats();
    
    // Watch for restore confirmation
    $('#confirmRestore').change(function() {
        $('#restoreBtn').prop('disabled', !$(this).is(':checked'));
    });
});

// Load backups list
function loadBackups() {
    // Simulate API call
    setTimeout(() => {
        const backups = [
            { id: 1, name: 'backup-2026-01-03.sql', type: 'Database', size: '24.5 MB', created_at: '2026-01-03 09:30:00', status: 'success' },
            { id: 2, name: 'backup-2026-01-02.zip', type: 'Full', size: '156.2 MB', created_at: '2026-01-02 23:00:00', status: 'success' },
            { id: 3, name: 'backup-2026-01-01.sql', type: 'Database', size: '24.3 MB', created_at: '2026-01-01 23:00:00', status: 'success' },
            { id: 4, name: 'backup-2025-12-31.zip', type: 'Full', size: '155.8 MB', created_at: '2025-12-31 23:00:00', status: 'success' },
            { id: 5, name: 'backup-2025-12-30.sql', type: 'Database', size: '24.1 MB', created_at: '2025-12-30 23:00:00', status: 'partial' },
        ];
        
        populateBackupList(backups);
        initializeDataTable();
    }, 1000);
}

// Populate backup list
function populateBackupList(backups) {
    const tbody = $('#backupList');
    tbody.empty();
    
    if (backups.length === 0) {
        tbody.html('<tr><td colspan="7" class="text-center py-4">No backups found</td></tr>');
        return;
    }
    
    backups.forEach(backup => {
        const isSelected = selectedBackups.includes(backup.id);
        const statusBadge = getStatusBadge(backup.status);
        const icon = backup.type === 'Database' ? 'bx-data' : 'bx-archive';
        
        const row = `
            <tr>
                <td>
                    <div class="form-check">
                        <input class="form-check-input backup-checkbox" type="checkbox" 
                               value="${backup.id}" ${isSelected ? 'checked' : ''}
                               onchange="toggleBackupSelection(${backup.id})">
                    </div>
                </td>
                <td>
                    <div class="d-flex align-items-center">
                        <div class="ms-2">
                            <h6 class="mb-0 font-14">${backup.name}</h6>
                        </div>
                    </div>
                </td>
                <td><span class="badge ${backup.type === 'Database' ? 'bg-primary' : 'bg-success'}">${backup.type}</span></td>
                <td>${backup.size}</td>
                <td>${backup.created_at}</td>
                <td>${statusBadge}</td>
                <td>
                    <div class="d-flex order-actions">
                        <a href="javascript:;" class="text-primary" onclick="downloadBackup(${backup.id})" title="Download">
                            <i class="bx bx-download"></i>
                        </a>
                        <a href="javascript:;" class="text-warning ms-3" onclick="viewBackup(${backup.id})" title="View">
                            <i class="bx bx-show"></i>
                        </a>
                        <a href="javascript:;" class="text-danger ms-3" onclick="deleteBackup(${backup.id})" title="Delete">
                            <i class="bx bx-trash"></i>
                        </a>
                    </div>
                </td>
            </tr>
        `;
        
        tbody.append(row);
    });
    
    updateDeleteButton();
}

// Get status badge
function getStatusBadge(status) {
    const statusMap = {
        'success': { class: 'bg-success', icon: 'bx-check-circle' },
        'partial': { class: 'bg-warning', icon: 'bx-error-circle' },
        'failed': { class: 'bg-danger', icon: 'bx-x-circle' }
    };
    
    const statusInfo = statusMap[status.toLowerCase()] || { class: 'bg-secondary', icon: 'bx-question-mark' };
    
    return `<span class="badge ${statusInfo.class}"><i class="bx ${statusInfo.icon} mr-1"></i>${status}</span>`;
}

// Initialize DataTable
function initializeDataTable() {
    $('#backupTable').DataTable({
        lengthMenu: [[10, 25, 50, -1], [10, 25, 50, "All"]],
        pageLength: 10,
        ordering: true,
        order: [[4, 'desc']], // Sort by created_at descending
        language: {
            search: "Search backups:",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ backups",
            paginate: {
                previous: "<i class='bx bx-chevron-left'>",
                next: "<i class='bx bx-chevron-right'>"
            }
        }
    });
}

// Load backup statistics
function loadBackupStats() {
    setTimeout(() => {
        $('#totalBackups').text('5');
        $('#lastBackup').text('Today 09:30');
        $('#totalSize').text('384.9 MB');
        $('#nextBackup').text('Tomorrow 23:00');
    }, 500);
}

// Create new backup
function createBackup() {
    showBackupProgress('Creating backup...');
    
    let progress = 0;
    backupInterval = setInterval(() => {
        progress += 10;
        updateBackupProgress(progress, `Processing... (${progress}%)`);
        
        if (progress >= 100) {
            clearInterval(backupInterval);
            setTimeout(() => {
                hideBackupProgress();
                showNotification('Backup created successfully!', 'success');
                loadBackups();
                loadBackupStats();
            }, 500);
        }
    }, 300);
}

// Create database backup
function createDatabaseBackup() {
    showBackupProgress('Creating database backup...');
    
    let progress = 0;
    backupInterval = setInterval(() => {
        progress += 20;
        updateBackupProgress(progress, `Exporting database... (${progress}%)`);
        
        if (progress >= 100) {
            clearInterval(backupInterval);
            setTimeout(() => {
                hideBackupProgress();
                showNotification('Database backup created!', 'success');
                loadBackups();
                loadBackupStats();
            }, 500);
        }
    }, 200);
}

// Create full backup
function createFullBackup() {
    showBackupProgress('Creating full system backup...');
    
    let progress = 0;
    backupInterval = setInterval(() => {
        progress += 5;
        updateBackupProgress(progress, `Backing up files... (${progress}%)`);
        
        if (progress >= 100) {
            clearInterval(backupInterval);
            setTimeout(() => {
                hideBackupProgress();
                showNotification('Full system backup created!', 'success');
                loadBackups();
                loadBackupStats();
            }, 500);
        }
    }, 200);
}

// Show backup progress modal
function showBackupProgress(title) {
    $('#progressTitle').text(title);
    $('#backupProgressBar').css('width', '0%').text('0%');
    $('#backupStatus').text('Initializing...');
    $('#backupDetails').text('');
    
    new bootstrap.Modal('#backupProgressModal').show();
}

// Update backup progress
function updateBackupProgress(percent, status, details = '') {
    $('#backupProgressBar').css('width', percent + '%').text(percent + '%');
    $('#backupStatus').text(status);
    if (details) {
        $('#backupDetails').text(details);
    }
}

// Hide backup progress modal
function hideBackupProgress() {
    const modal = bootstrap.Modal.getInstance('#backupProgressModal');
    if (modal) modal.hide();
}

// Restore backup modal
function restoreBackupModal() {
    const select = $('#restoreSelect');
    select.empty().append('<option value="">Choose backup...</option>');
    
    // Add options
    ['backup-2026-01-03.sql', 'backup-2026-01-02.zip', 'backup-2026-01-01.sql'].forEach(name => {
        select.append(`<option value="${name}">${name}</option>`);
    });
    
    $('#confirmRestore').prop('checked', false);
    $('#restoreBtn').prop('disabled', true);
    
    new bootstrap.Modal('#restoreModal').show();
}

// Perform restore
function performRestore() {
    const backupName = $('#restoreSelect').val();
    if (!backupName) {
        showNotification('Please select a backup!', 'error');
        return;
    }
    
    if (!confirm(`WARNING: This will replace ALL current data with "${backupName}"!\n\nAre you absolutely sure?`)) {
        return;
    }
    
    showBackupProgress('Restoring backup...');
    
    let progress = 0;
    backupInterval = setInterval(() => {
        progress += 25;
        updateBackupProgress(progress, `Restoring ${backupName}... (${progress}%)`);
        
        if (progress >= 100) {
            clearInterval(backupInterval);
            setTimeout(() => {
                hideBackupProgress();
                showNotification('Backup restored successfully! Reloading...', 'success');
                
                // Close restore modal
                bootstrap.Modal.getInstance('#restoreModal').hide();
                
                // Reload page
                setTimeout(() => window.location.reload(), 2000);
            }, 500);
        }
    }, 300);
}

// Toggle backup selection
function toggleBackupSelection(id) {
    const index = selectedBackups.indexOf(id);
    if (index === -1) {
        selectedBackups.push(id);
    } else {
        selectedBackups.splice(index, 1);
    }
    updateSelectAllCheckbox();
    updateDeleteButton();
}

// Toggle select all
function toggleSelectAll() {
    const selectAll = $('#selectAll').is(':checked');
    $('.backup-checkbox').prop('checked', selectAll);
    
    selectedBackups = [];
    if (selectAll) {
        $('.backup-checkbox').each(function() {
            selectedBackups.push(parseInt($(this).val()));
        });
    }
    
    updateDeleteButton();
}

// Update select all checkbox
function updateSelectAllCheckbox() {
    const checkboxes = $('.backup-checkbox');
    const selectAll = $('#selectAll');
    
    if (checkboxes.length === 0) {
        selectAll.prop('checked', false);
        selectAll.prop('indeterminate', false);
        return;
    }
    
    const checkedCount = $('.backup-checkbox:checked').length;
    
    if (checkedCount === 0) {
        selectAll.prop('checked', false);
        selectAll.prop('indeterminate', false);
    } else if (checkedCount === checkboxes.length) {
        selectAll.prop('checked', true);
        selectAll.prop('indeterminate', false);
    } else {
        selectAll.prop('checked', false);
        selectAll.prop('indeterminate', true);
    }
}

// Update delete button state
function updateDeleteButton() {
    $('#deleteSelectedBtn').prop('disabled', selectedBackups.length === 0);
}

// Delete selected backups
function deleteSelectedBackups() {
    if (selectedBackups.length === 0) {
        showNotification('Please select backups to delete!', 'warning');
        return;
    }
    
    if (!confirm(`Delete ${selectedBackups.length} selected backup(s)?\nThis action cannot be undone.`)) {
        return;
    }
    
    showLoading();
    
    setTimeout(() => {
        hideLoading();
        showNotification(`${selectedBackups.length} backup(s) deleted!`, 'success');
        
        selectedBackups = [];
        $('#selectAll').prop('checked', false);
        
        loadBackups();
        loadBackupStats();
    }, 1500);
}

// Download backup
function downloadBackup(id) {
    showNotification('Download started...', 'info');
    // Real implementation would trigger file download
}

// View backup
function viewBackup(id) {
    showNotification(`Viewing backup ${id}...`, 'info');
}

// Delete single backup
function deleteBackup(id) {
    if (!confirm('Delete this backup?')) return;
    
    showLoading('Deleting...');
    
    setTimeout(() => {
        hideLoading();
        showNotification('Backup deleted!', 'success');
        loadBackups();
        loadBackupStats();
    }, 1000);
}

// Save backup settings
function saveBackupSettings() {
    showLoading('Saving settings...');
    
    setTimeout(() => {
        hideLoading();
        showNotification('Settings saved!', 'success');
    }, 1000);
}

// Utility functions
function showLoading(message = 'Processing...') {
    Swal.fire({
        title: message,
        allowOutsideClick: false,
        showConfirmButton: false,
        willOpen: () => {
            Swal.showLoading();
        }
    });
}

function hideLoading() {
    Swal.close();
}

function showNotification(message, type = 'info') {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true
    });
    
    Toast.fire({
        icon: type,
        title: message
    });
}
</script>

<style>
.widgets-icons-2 {
    width: 56px;
    height: 56px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.bg-gradient-ohhappiness {
    background: linear-gradient(45deg, #00b09b, #96c93d) !important;
}

.bg-gradient-scooter {
    background: linear-gradient(45deg, #17ead9, #6078ea) !important;
}

.bg-gradient-burning {
    background: linear-gradient(45deg, #ff416c, #ff4b2b) !important;
}

.bg-gradient-bloody {
    background: linear-gradient(45deg, #f54ea2, #ff7676) !important;
}

.order-actions a {
    font-size: 1.2rem;
}
</style>
@endsection