@extends('layouts.app')

@section('style')
<style>
    /* Syndron Theme - Mobile Responsive Purchase Show */
    .purchase-status-badge {
        padding: 8px 20px;
        border-radius: 25px;
        font-size: 14px;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        display: inline-block;
        margin-bottom: 10px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    .status-pending {
        background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
        color: white;
    }
    
    .status-approved {
        background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
        color: white;
    }
    
    .status-received {
        background: linear-gradient(135deg, #8B5CF6 0%, #7C3AED 100%);
        color: white;
    }
    
    .status-completed {
        background: linear-gradient(135deg, #10B981 0%, #059669 100%);
        color: white;
    }
    
    .status-delivered {
        background: linear-gradient(135deg, #6366F1 0%, #4F46E5 100%);
        color: white;
    }
    
    .status-cancelled {
        background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
        color: white;
    }
    
    /* Info Cards */
    .info-card-syndron {
        background: white;
        border-radius: 15px;
        padding: 20px;
        margin-bottom: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        border-left: 5px solid #667eea;
        transition: all 0.3s ease;
    }
    
    @media (max-width: 768px) {
        .info-card-syndron {
            padding: 15px;
            margin-bottom: 10px;
        }
        
        .info-card-syndron h6 {
            font-size: 14px;
        }
        
        .info-card-syndron .fs-5 {
            font-size: 18px !important;
        }
    }
    
    .info-card-syndron:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.15);
    }
    
    .info-card-syndron.success {
        border-left-color: #10B981;
    }
    
    .info-card-syndron.warning {
        border-left-color: #F59E0B;
    }
    
    .info-card-syndron.info {
        border-left-color: #3B82F6;
    }
    
    .info-card-syndron.danger {
        border-left-color: #EF4444;
    }
    
    /* Action Buttons */
    .action-buttons-syndron {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-top: 20px;
    }
    
    @media (max-width: 768px) {
        .action-buttons-syndron {
            flex-direction: column;
        }
        
        .action-buttons-syndron .btn {
            width: 100%;
            justify-content: center;
            margin-bottom: 5px;
        }
    }
    
    .action-buttons-syndron .btn {
        border-radius: 10px;
        padding: 12px 20px;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
        transition: all 0.3s ease;
        border: none;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-size: 14px;
        min-width: 160px;
        justify-content: center;
    }
    
    .action-buttons-syndron .btn-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }
    
    .action-buttons-syndron .btn-success {
        background: linear-gradient(135deg, #10B981 0%, #059669 100%);
        box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
    }
    
    .action-buttons-syndron .btn-warning {
        background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
        box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
    }
    
    .action-buttons-syndron .btn-info {
        background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
        box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
    }
    
    .action-buttons-syndron .btn-danger {
        background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
        box-shadow: 0 4px 15px rgba(239, 68, 68, 0.3);
    }
    
    .action-buttons-syndron .btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(0,0,0,0.15);
    }
    
    /* Items Table */
    .items-table-syndron {
        border-collapse: separate;
        border-spacing: 0;
        border-radius: 10px;
        overflow: hidden;
        width: 100%;
        box-shadow: 0 4px 12px rgba(0,0,0,0.05);
    }
    
    .items-table-syndron thead th {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        padding: 15px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-size: 13px;
    }
    
    .items-table-syndron tbody tr {
        transition: all 0.3s ease;
    }
    
    .items-table-syndron tbody tr:hover {
        background-color: rgba(102, 126, 234, 0.05);
    }
    
    .items-table-syndron td {
        padding: 12px 15px;
        vertical-align: middle;
        border-bottom: 1px solid rgba(0,0,0,0.05);
    }
    
    @media (max-width: 768px) {
        .items-table-syndron {
            display: block;
        }
        
        .items-table-syndron thead {
            display: none;
        }
        
        .items-table-syndron tbody,
        .items-table-syndron tr,
        .items-table-syndron td {
            display: block;
            width: 100%;
        }
        
        .items-table-syndron tr {
            margin-bottom: 15px;
            border: 1px solid #dee2e6;
            border-radius: 10px;
            padding: 10px;
            background: white;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .items-table-syndron td {
            padding: 8px 0;
            border: none;
            position: relative;
            padding-left: 50%;
            text-align: right;
            border-bottom: 1px solid #f8f9fa;
        }
        
        .items-table-syndron td:before {
            content: attr(data-label);
            position: absolute;
            left: 0;
            width: 45%;
            padding-right: 10px;
            font-weight: 600;
            text-align: left;
            color: #667eea;
        }
        
        .items-table-syndron td:last-child {
            border-bottom: none;
        }
    }
    
    /* Progress Steps */
    .progress-steps-syndron {
        display: flex;
        justify-content: space-between;
        position: relative;
        margin: 30px 0;
        padding: 0 20px;
    }
    
    @media (max-width: 768px) {
        .progress-steps-syndron {
            padding: 0 10px;
            margin: 20px 0;
        }
    }
    
    .progress-steps-syndron:before {
        content: '';
        position: absolute;
        top: 15px;
        left: 0;
        right: 0;
        height: 2px;
        background: linear-gradient(to right, #667eea, #764ba2);
        z-index: 1;
    }
    
    .progress-step-syndron {
        position: relative;
        z-index: 2;
        text-align: center;
    }
    
    .step-circle-syndron {
        width: 30px;
        height: 30px;
        border-radius: 50%;
        background: white;
        border: 3px solid #e9ecef;
        color: #6c757d;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 10px;
        font-weight: bold;
        font-size: 12px;
    }
    
    .step-circle-syndron.active {
        background: #667eea;
        border-color: #667eea;
        color: white;
        box-shadow: 0 0 0 5px rgba(102, 126, 234, 0.2);
    }
    
    .step-circle-syndron.completed {
        background: #10B981;
        border-color: #10B981;
        color: white;
        box-shadow: 0 0 0 5px rgba(16, 185, 129, 0.2);
    }
    
    .step-label-syndron {
        font-size: 11px;
        color: #6c757d;
        font-weight: 500;
        white-space: nowrap;
    }
    
    .step-label-syndron.active {
        color: #667eea;
        font-weight: bold;
    }
    
    .step-label-syndron.completed {
        color: #10B981;
    }
    
    /* Timeline */
    .timeline-syndron {
        position: relative;
        padding: 20px 0;
    }
    
    .timeline-syndron:before {
        content: '';
        position: absolute;
        left: 20px;
        top: 0;
        bottom: 0;
        width: 2px;
        background: linear-gradient(to bottom, #667eea, #764ba2);
    }
    
    .timeline-item-syndron {
        position: relative;
        padding-left: 50px;
        margin-bottom: 25px;
    }
    
    .timeline-dot-syndron {
        position: absolute;
        left: 16px;
        top: 5px;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: #667eea;
        border: 3px solid white;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.2);
    }
    
    .timeline-dot-syndron.completed {
        background: #10B981;
        box-shadow: 0 0 0 3px rgba(16, 185, 129, 0.2);
    }
    
    .timeline-content-syndron {
        background: white;
        padding: 15px;
        border-radius: 10px;
        box-shadow: 0 3px 10px rgba(0,0,0,0.05);
    }
    
    /* Status Alert */
    .status-alert-syndron {
        border-left: 5px solid;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 20px;
        background: white;
        box-shadow: 0 3px 10px rgba(0,0,0,0.05);
    }
    
    .alert-pending {
        border-left-color: #F59E0B;
        background: linear-gradient(to right, #fff5e6, white);
    }
    
    .alert-approved {
        border-left-color: #3B82F6;
        background: linear-gradient(to right, #e6f0ff, white);
    }
    
    .alert-received {
        border-left-color: #8B5CF6;
        background: linear-gradient(to right, #f0e6ff, white);
    }
    
    /* Item Badges */
    .item-badge {
        font-size: 11px;
        padding: 4px 8px;
        border-radius: 12px;
        font-weight: 600;
    }
    
    .badge-inventory {
        background: linear-gradient(135deg, #10B981, #059669);
        color: white;
    }
    
    .badge-service {
        background: linear-gradient(135deg, #3B82F6, #1D4ED8);
        color: white;
    }
    
    /* Mobile Card View */
    @media (max-width: 768px) {
        .mobile-card-view {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .mobile-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            border-left: 4px solid #667eea;
        }
        
        .mobile-card .card-label {
            font-size: 12px;
            color: #6c757d;
            font-weight: 500;
            margin-bottom: 5px;
        }
        
        .mobile-card .card-value {
            font-size: 16px;
            font-weight: 600;
            color: #333;
        }
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">
                <i class="bx bx-cart me-1"></i> Purchase Order
            </div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('purchases.index') }}">Purchases</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ $purchase->invoice_no }}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('purchases.index') }}" class="btn btn-outline-secondary btn-sm">
                    <i class="bx bx-arrow-back"></i> Back to List
                </a>
            </div>
        </div>

        <!-- Status Alert -->
        <div class="status-alert-syndron alert-{{ $purchase->status }}">
            <div class="d-flex align-items-center">
                <div class="flex-grow-1">
                    <h5 class="mb-1">
                        <span class="purchase-status-badge status-{{ $purchase->status }}">
                            {{ ucfirst($purchase->status) }}
                        </span>
                    </h5>
                    <p class="mb-0 text-muted">
                        @switch($purchase->status)
                            @case('pending')
                                <i class="bx bx-time"></i> Purchase is pending approval
                                @break
                            @case('approved')
                                <i class="bx bx-check-circle"></i> Purchase approved, waiting to be received
                                @break
                            @case('received')
                                <i class="bx bx-package"></i> Purchase received, ready to add to inventory
                                @break
                            @case('completed')
                                <i class="bx bx-check-circle"></i> Purchase completed and added to inventory
                                @break
                            @case('delivered')
                                <i class="bx bx-truck"></i> Services delivered successfully
                                @break
                            @case('cancelled')
                                <i class="bx bx-x-circle"></i> Purchase cancelled
                                @break
                        @endswitch
                    </p>
                </div>
            </div>
        </div>

        <!-- Purchase Progress Steps -->
        <div class="progress-steps-syndron">
            @php
                $steps = [
                    'pending' => ['label' => 'Pending', 'icon' => 'bx-time'],
                    'approved' => ['label' => 'Approved', 'icon' => 'bx-check'],
                    'received' => ['label' => 'Received', 'icon' => 'bx-package'],
                    'completed' => ['label' => 'Inventory', 'icon' => 'bx-package-check'],
                    'delivered' => ['label' => 'Delivered', 'icon' => 'bx-truck']
                ];
                
                $currentStep = array_search($purchase->status, array_keys($steps));
                if ($currentStep === false) $currentStep = 0;
            @endphp
            
            @foreach($steps as $stepKey => $step)
                <div class="progress-step-syndron" style="width: {{ 100/count($steps) }}%">
                    <div class="step-circle-syndron 
                        @if($stepKey === $purchase->status) active 
                        @elseif(array_search($stepKey, array_keys($steps)) < $currentStep) completed 
                        @endif">
                        @if($stepKey === $purchase->status || array_search($stepKey, array_keys($steps)) < $currentStep)
                            <i class="bx {{ $step['icon'] }}"></i>
                        @else
                            {{ array_search($stepKey, array_keys($steps)) + 1 }}
                        @endif
                    </div>
                    <div class="step-label-syndron 
                        @if($stepKey === $purchase->status) active 
                        @elseif(array_search($stepKey, array_keys($steps)) < $currentStep) completed 
                        @endif">
                        {{ $step['label'] }}
                    </div>
                </div>
            @endforeach
        </div>

        <!-- Action Buttons -->
        <div class="action-buttons-syndron">
            @if($purchase->status === 'pending')
                <button class="btn btn-success" onclick="approvePurchase({{ $purchase->id }})">
                    <i class="bx bx-check"></i> Approve Purchase
                </button>
            @endif
            
            @if($purchase->status === 'approved')
                <button class="btn btn-info" onclick="receivePurchase({{ $purchase->id }})">
                    <i class="bx bx-package"></i> Mark as Received
                </button>
            @endif
            
            @if($purchase->status === 'received')
                <button class="btn btn-primary" onclick="addToInventory({{ $purchase->id }})">
                    <i class="bx bx-plus"></i> Add to Inventory
                </button>
            @endif
            
            @if($purchase->status === 'completed')
                @php
                    $hasServices = $purchase->items()
                        ->where(function($q) {
                            $q->where('item_type', 'service')
                              ->orWhereHas('product', function($q) {
                                  $q->where('is_service', true);
                              });
                        })
                        ->exists();
                @endphp
                
                @if($hasServices)
                    <button class="btn btn-warning" onclick="deliverServices({{ $purchase->id }})">
                        <i class="bx bx-truck"></i> Mark as Delivered
                    </button>
                @endif
            @endif
            
            @if(in_array($purchase->status, ['pending', 'approved', 'received']))
                <button class="btn btn-danger" onclick="showCancelModal({{ $purchase->id }})">
                    <i class="bx bx-x"></i> Cancel Purchase
                </button>
            @endif
            
            @if($purchase->status === 'pending')
                <a href="{{ route('purchases.edit', $purchase) }}" class="btn btn-warning">
                    <i class="bx bx-edit"></i> Edit Purchase
                </a>
            @endif
            
            @if($purchase->grand_total > $purchase->paid_amount)
                <a href="{{ route('purchases.payment', $purchase) }}" class="btn btn-success">
                    <i class="bx bx-money"></i> Record Payment
                </a>
            @endif
            
            <button class="btn btn-secondary" onclick="printPurchase({{ $purchase->id }})">
                <i class="bx bx-printer"></i> Print
            </button>
        </div>

        <!-- Purchase Details Cards (Mobile View) -->
        <div class="mobile-card-view d-block d-md-none">
            <div class="mobile-card">
                <div class="card-label">Invoice Number</div>
                <div class="card-value">{{ $purchase->invoice_no }}</div>
            </div>
            
            <div class="mobile-card">
                <div class="card-label">Supplier</div>
                <div class="card-value">{{ $purchase->supplier->name }}</div>
                <small class="text-muted">{{ $purchase->supplier->phone }}</small>
            </div>
            
            <div class="mobile-card">
                <div class="card-label">Purchase Date</div>
                <div class="card-value">{{ $purchase->purchase_date->format('M d, Y') }}</div>
            </div>
            
            <div class="mobile-card">
                <div class="card-label">Total Amount</div>
                <div class="card-value text-primary">KES {{ number_format($purchase->grand_total, 2) }}</div>
            </div>
            
            <div class="mobile-card">
                <div class="card-label">Payment Status</div>
                <div class="card-value">
                    <span class="badge bg-{{ $purchase->payment_status === 'paid' ? 'success' : ($purchase->payment_status === 'partial' ? 'warning' : 'secondary') }}">
                        {{ ucfirst($purchase->payment_status) }}
                    </span>
                </div>
                <small class="text-muted">
                    Paid: KES {{ number_format($purchase->paid_amount, 2) }}
                    @if($purchase->grand_total > $purchase->paid_amount)
                        <br>Due: KES {{ number_format($purchase->grand_total - $purchase->paid_amount, 2) }}
                    @endif
                </small>
            </div>
        </div>

        <!-- Purchase Details (Desktop View) -->
        <div class="row d-none d-md-flex">
            <div class="col-md-3">
                <div class="info-card-syndron">
                    <h6 class="text-muted mb-2">Invoice Number</h6>
                    <h4 class="mb-0">{{ $purchase->invoice_no }}</h4>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="info-card-syndron">
                    <h6 class="text-muted mb-2">Supplier</h6>
                    <h5 class="mb-1">{{ $purchase->supplier->name }}</h5>
                    <p class="mb-0 text-muted small">{{ $purchase->supplier->phone }}</p>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="info-card-syndron">
                    <h6 class="text-muted mb-2">Purchase Date</h6>
                    <h5 class="mb-1">{{ $purchase->purchase_date->format('M d, Y') }}</h5>
                    <p class="mb-0 text-muted small">{{ $purchase->purchase_date->format('h:i A') }}</p>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="info-card-syndron success">
                    <h6 class="text-muted mb-2">Total Amount</h6>
                    <h4 class="mb-0 text-primary">KES {{ number_format($purchase->grand_total, 2) }}</h4>
                </div>
            </div>
        </div>

        <!-- Purchase Items -->
        <div class="card border-0 shadow-sm mt-4">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="bx bx-package me-2"></i> Purchase Items</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="items-table-syndron">
                        <thead>
                            <tr>
                                <th>Type</th>
                                <th>Item</th>
                                <th class="text-end">Qty</th>
                                <th class="text-end">Unit Price</th>
                                <th class="text-end">Tax</th>
                                <th class="text-end">Total</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($purchase->items as $item)
                                <tr>
                                    <td data-label="Type">
                                        @if($item->item_type === 'service' || ($item->product && $item->product->is_service))
                                            <span class="item-badge badge-service">Service</span>
                                        @else
                                            <span class="item-badge badge-inventory">Inventory</span>
                                        @endif
                                    </td>
                                    <td data-label="Item">
                                        <strong>
                                            @if($item->item_type === 'service')
                                                {{ $item->description }}
                                            @else
                                                {{ $item->product->name ?? 'Product #' . $item->product_id }}
                                            @endif
                                        </strong>
                                        @if($item->item_type === 'product' && $item->product)
                                            <br>
                                            <small class="text-muted">SKU: {{ $item->product->sku }}</small>
                                        @endif
                                        @if($item->batch_number)
                                            <br>
                                            <small class="text-muted">Batch: {{ $item->batch_number }}</small>
                                        @endif
                                        @if($item->expiry_date)
                                            <br>
                                            <small class="text-muted">Expires: {{ $item->expiry_date->format('M d, Y') }}</small>
                                        @endif
                                    </td>
                                    <td data-label="Quantity" class="text-end">{{ $item->quantity }}</td>
                                    <td data-label="Unit Price" class="text-end">KES {{ number_format($item->unit_price, 2) }}</td>
                                    <td data-label="Tax" class="text-end">
                                        @if($item->tax_percent)
                                            {{ $item->tax_percent }}%<br>
                                            <small>KES {{ number_format($item->tax_amount, 2) }}</small>
                                        @else
                                            -
                                        @endif
                                    </td>
                                    <td data-label="Total" class="text-end">KES {{ number_format($item->total, 2) }}</td>
                                    <td data-label="Status">
                                        @if($item->added_to_stock)
                                            <span class="badge bg-success">In Stock</span>
                                        @elseif($item->item_type === 'service' || ($item->product && $item->product->is_service))
                                            <span class="badge bg-info">Service</span>
                                        @else
                                            <span class="badge bg-warning">Pending</span>
                                        @endif
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Timeline -->
        <div class="card border-0 shadow-sm mt-4">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="bx bx-history me-2"></i> Activity Timeline</h5>
            </div>
            <div class="card-body">
                <div class="timeline-syndron">
                    <div class="timeline-item-syndron">
                        <div class="timeline-dot-syndron completed"></div>
                        <div class="timeline-content-syndron">
                            <h6 class="mb-1">Purchase Created</h6>
                            <p class="mb-1 text-muted small">{{ $purchase->created_at->format('M d, Y h:i A') }}</p>
                            <p class="mb-0 small">Created by: {{ $purchase->creator->name ?? 'System' }}</p>
                        </div>
                    </div>
                    
                    @if($purchase->approved_at)
                        <div class="timeline-item-syndron">
                            <div class="timeline-dot-syndron completed"></div>
                            <div class="timeline-content-syndron">
                                <h6 class="mb-1">Purchase Approved</h6>
                                <p class="mb-1 text-muted small">{{ $purchase->approved_at->format('M d, Y h:i A') }}</p>
                                <p class="mb-0 small">Approved by: {{ $purchase->approver->name ?? 'System' }}</p>
                            </div>
                        </div>
                    @endif
                    
                    @if($purchase->received_at)
                        <div class="timeline-item-syndron">
                            <div class="timeline-dot-syndron completed"></div>
                            <div class="timeline-content-syndron">
                                <h6 class="mb-1">Purchase Received</h6>
                                <p class="mb-1 text-muted small">{{ $purchase->received_at->format('M d, Y h:i A') }}</p>
                                <p class="mb-0 small">Received by: {{ auth()->user()->name }}</p>
                            </div>
                        </div>
                    @endif
                    
                    @if($purchase->status === 'completed')
                        <div class="timeline-item-syndron">
                            <div class="timeline-dot-syndron completed"></div>
                            <div class="timeline-content-syndron">
                                <h6 class="mb-1">Added to Inventory</h6>
                                <p class="mb-1 text-muted small">{{ $purchase->updated_at->format('M d, Y h:i A') }}</p>
                                <p class="mb-0 small">Inventory updated by system</p>
                            </div>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Notes Section -->
        @if($purchase->notes)
            <div class="card border-0 shadow-sm mt-4">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="bx bx-note me-2"></i> Notes</h5>
                </div>
                <div class="card-body">
                    <p class="mb-0">{{ $purchase->notes }}</p>
                </div>
            </div>
        @endif
    </div>
</div>

<!-- Cancel Modal -->
<div class="modal fade" id="cancelModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-x me-2"></i> Cancel Purchase</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to cancel this purchase?</p>
                <div class="mb-3">
                    <label class="form-label">Cancellation Reason</label>
                    <textarea class="form-control" id="cancelReason" rows="3" placeholder="Enter reason for cancellation..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-danger" onclick="cancelPurchase()">Cancel Purchase</button>
            </div>
        </div>
    </div>
</div>

<!-- Add to Inventory Modal -->
<div class="modal fade" id="inventoryModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-package me-2"></i> Add to Inventory</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="inventoryCheckResult"></div>
                <div id="inventoryItemsList" class="mt-3" style="display: none;">
                    <h6>Items to Add:</h6>
                    <ul id="inventoryItems" class="list-group"></ul>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" id="addToInventoryBtn" style="display: none;" onclick="processAddToInventory()">Add to Inventory</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
$(document).ready(function() {
    // Initialize tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl)
    });
});

let currentPurchaseId = null;

// Add these functions to your existing JavaScript:

function approvePurchase(purchaseId) {
    Swal.fire({
        title: 'Approve Purchase Order',
        text: 'Are you sure you want to approve this purchase order?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, approve it!',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`/purchases/${purchaseId}/approve`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Approved!', data.message, 'success')
                        .then(() => location.reload());
                } else {
                    Swal.fire('Error!', data.message, 'error');
                }
            })
            .catch(error => {
                Swal.fire('Error!', 'Failed to approve purchase', 'error');
            });
        }
    });
}

function receivePurchase(purchaseId) {
    Swal.fire({
        title: 'Mark as Received',
        text: 'Have you received all the items in this purchase?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#17a2b8',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, mark as received',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`/purchases/${purchaseId}/receive`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Received!', data.message, 'success')
                        .then(() => location.reload());
                } else {
                    Swal.fire('Error!', data.message, 'error');
                }
            })
            .catch(error => {
                Swal.fire('Error!', 'Failed to mark as received', 'error');
            });
        }
    });
}

function addToInventory(purchaseId) {
    // First check if purchase can be added to inventory
    Swal.fire({
        title: 'Checking Inventory...',
        text: 'Please wait while we check purchase items',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    fetch(`/purchases/${purchaseId}/check-inventory`)
        .then(response => response.json())
        .then(data => {
            Swal.close();
            
            if (data.success) {
                if (data.can_add_to_inventory) {
                    Swal.fire({
                        title: 'Add to Inventory',
                        html: `This will add ${data.inventory_items_count} item(s) to inventory.<br>
                               ${data.service_items_count > 0 ? data.service_items_count + ' service item(s) will be completed.' : ''}<br><br>
                               Are you sure you want to continue?`,
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonColor: '#28a745',
                        cancelButtonColor: '#6c757d',
                        confirmButtonText: 'Yes, add to inventory',
                        cancelButtonText: 'Cancel'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            Swal.fire({
                                title: 'Adding to Inventory...',
                                text: 'Please wait while we update inventory',
                                allowOutsideClick: false,
                                didOpen: () => {
                                    Swal.showLoading();
                                }
                            });
                            
                            fetch(`/purchases/${purchaseId}/complete`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                }
                            })
                            .then(response => response.json())
                            .then(completeData => {
                                Swal.close();
                                if (completeData.success) {
                                    Swal.fire('Success!', completeData.message, 'success')
                                        .then(() => location.reload());
                                } else {
                                    Swal.fire('Error!', completeData.message, 'error');
                                }
                            })
                            .catch(error => {
                                Swal.close();
                                Swal.fire('Error!', 'Failed to add to inventory', 'error');
                            });
                        }
                    });
                } else {
                    Swal.fire('Cannot Add to Inventory', 
                        'Purchase must be approved and received before adding to inventory.', 
                        'warning');
                }
            } else {
                Swal.fire('Error!', data.message, 'error');
            }
        })
        .catch(error => {
            Swal.close();
            Swal.fire('Error!', 'Failed to check inventory status', 'error');
        });
}

function deliverServices(purchaseId) {
    Swal.fire({
        title: 'Mark Services as Delivered',
        text: 'Have all services been delivered to the customer?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#ffc107',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, mark as delivered',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`/purchases/${purchaseId}/deliver`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire('Delivered!', data.message, 'success')
                        .then(() => location.reload());
                } else {
                    Swal.fire('Error!', data.message, 'error');
                }
            })
            .catch(error => {
                Swal.fire('Error!', 'Failed to mark as delivered', 'error');
            });
        }
    });
}

function printPurchaseOrder(purchaseId) {
    window.open(`/purchases/${purchaseId}/print-invoice`, '_blank');
}

function receivePurchase(purchaseId) {
    Swal.fire({
        title: 'Mark as Received',
        text: 'Are you sure the purchase items have been received?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#3B82F6',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, received!',
        cancelButtonText: 'Cancel',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`/purchases/${purchaseId}/receive`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({})
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(response.statusText);
                }
                return response.json();
            })
            .catch(error => {
                Swal.showValidationMessage(`Request failed: ${error}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value.success) {
            Swal.fire({
                title: 'Received!',
                text: result.value.message,
                icon: 'success',
                confirmButtonColor: '#3B82F6',
            }).then(() => {
                location.reload();
            });
        } else if (result.value && !result.value.success) {
            Swal.fire({
                title: 'Error!',
                text: result.value.message,
                icon: 'error',
                confirmButtonColor: '#EF4444',
            });
        }
    });
}

function addToInventory(purchaseId) {
    currentPurchaseId = purchaseId;
    
    // First check if purchase can be added to inventory
    Swal.fire({
        title: 'Checking Inventory...',
        text: 'Please wait while we check purchase items',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    fetch(`/purchases/${purchaseId}/check-inventory`)
        .then(response => response.json())
        .then(data => {
            Swal.close();
            
            if (data.success) {
                if (data.can_add_to_inventory) {
                    // Show inventory modal with items
                    showInventoryModal(data);
                } else {
                    Swal.fire({
                        title: 'Cannot Add to Inventory',
                        text: data.message || 'Purchase must be received before adding to inventory.',
                        icon: 'warning',
                        confirmButtonColor: '#F59E0B',
                    });
                }
            } else {
                Swal.fire({
                    title: 'Error!',
                    text: data.message,
                    icon: 'error',
                    confirmButtonColor: '#EF4444',
                });
            }
        })
        .catch(error => {
            Swal.close();
            Swal.fire({
                title: 'Error!',
                text: 'Failed to check inventory status.',
                icon: 'error',
                confirmButtonColor: '#EF4444',
            });
        });
}

function showInventoryModal(data) {
    $('#inventoryCheckResult').html(`
        <div class="alert alert-info">
            <i class="bx bx-info-circle"></i>
            <strong>Ready to add to inventory</strong>
            <p class="mb-0 mt-1">${data.inventory_items_count} inventory item(s) will be added to stock.</p>
            ${data.service_items_count > 0 ? `<p class="mb-0">${data.service_items_count} service item(s) will be marked as delivered.</p>` : ''}
        </div>
    `);
    
    $('#inventoryItemsList').show();
    $('#addToInventoryBtn').show();
    
    const modal = new bootstrap.Modal(document.getElementById('inventoryModal'));
    modal.show();
}

function processAddToInventory() {
    Swal.fire({
        title: 'Adding to Inventory...',
        text: 'Please wait while we update inventory',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    fetch(`/purchases/${currentPurchaseId}/complete`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({})
    })
    .then(response => response.json())
    .then(data => {
        Swal.close();
        
        if (data.success) {
            Swal.fire({
                title: 'Success!',
                text: data.message,
                icon: 'success',
                confirmButtonColor: '#10B981',
            }).then(() => {
                $('#inventoryModal').modal('hide');
                location.reload();
            });
        } else {
            Swal.fire({
                title: 'Error!',
                text: data.message,
                icon: 'error',
                confirmButtonColor: '#EF4444',
            });
        }
    })
    .catch(error => {
        Swal.close();
        Swal.fire({
            title: 'Error!',
            text: 'Failed to add to inventory.',
            icon: 'error',
            confirmButtonColor: '#EF4444',
        });
    });
}

function deliverServices(purchaseId) {
    Swal.fire({
        title: 'Mark as Delivered',
        text: 'Are you sure all services have been delivered?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#F59E0B',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, delivered!',
        cancelButtonText: 'Cancel',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`/purchases/${purchaseId}/deliver`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({})
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(response.statusText);
                }
                return response.json();
            })
            .catch(error => {
                Swal.showValidationMessage(`Request failed: ${error}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value.success) {
            Swal.fire({
                title: 'Delivered!',
                text: result.value.message,
                icon: 'success',
                confirmButtonColor: '#F59E0B',
            }).then(() => {
                location.reload();
            });
        } else if (result.value && !result.value.success) {
            Swal.fire({
                title: 'Error!',
                text: result.value.message,
                icon: 'error',
                confirmButtonColor: '#EF4444',
            });
        }
    });
}

function showCancelModal(purchaseId) {
    currentPurchaseId = purchaseId;
    const modal = new bootstrap.Modal(document.getElementById('cancelModal'));
    modal.show();
}

function cancelPurchase() {
    const reason = $('#cancelReason').val();
    
    Swal.fire({
        title: 'Cancel Purchase',
        text: 'Are you sure you want to cancel this purchase?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, cancel it!',
        cancelButtonText: 'Cancel',
        showLoaderOnConfirm: true,
        preConfirm: () => {
            return fetch(`/purchases/${currentPurchaseId}/cancel`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ reason: reason })
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(response.statusText);
                }
                return response.json();
            })
            .catch(error => {
                Swal.showValidationMessage(`Request failed: ${error}`);
            });
        }
    }).then((result) => {
        if (result.isConfirmed && result.value.success) {
            Swal.fire({
                title: 'Cancelled!',
                text: result.value.message,
                icon: 'success',
                confirmButtonColor: '#EF4444',
            }).then(() => {
                $('#cancelModal').modal('hide');
                location.reload();
            });
        } else if (result.value && !result.value.success) {
            Swal.fire({
                title: 'Error!',
                text: result.value.message,
                icon: 'error',
                confirmButtonColor: '#EF4444',
            });
        }
    });
}

function printPurchase(purchaseId) {
    window.open(`/purchases/${purchaseId}/print-invoice`, '_blank');
}
</script>
@endsection