@extends('layouts.app')

@section('style')
<style>
    /* Syndron Theme Purchases - Enhanced Mobile Responsive */
    :root {
        --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --success-gradient: linear-gradient(135deg, #10B981 0%, #059669 100%);
        --warning-gradient: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
        --info-gradient: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
        --danger-gradient: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
        --secondary-gradient: linear-gradient(135deg, #4F46E5 0%, #7C3AED 100%);
    }
    
    .stat-card-syndron {
        background: var(--primary-gradient);
        border-radius: 15px;
        padding: 20px;
        margin-bottom: 15px;
        color: white;
        box-shadow: 0 10px 20px rgba(102, 126, 234, 0.25);
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
        border: none;
        min-height: 150px;
        display: flex;
        flex-direction: column;
        justify-content: center;
    }
    
    @media (max-width: 768px) {
        .stat-card-syndron {
            padding: 15px;
            margin-bottom: 10px;
            min-height: 120px;
        }
        
        .stat-value-syndron {
            font-size: 24px !important;
        }
        
        .stat-label-syndron {
            font-size: 12px !important;
        }
        
        .stat-icon-syndron {
            font-size: 35px !important;
            top: 15px !important;
            right: 15px !important;
        }
    }
    
    .stat-card-syndron:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 30px rgba(102, 126, 234, 0.4);
    }
    
    .stat-card-syndron.success { background: var(--success-gradient); box-shadow: 0 10px 20px rgba(16, 185, 129, 0.25); }
    .stat-card-syndron.warning { background: var(--warning-gradient); box-shadow: 0 10px 20px rgba(245, 158, 11, 0.25); }
    .stat-card-syndron.info { background: var(--info-gradient); box-shadow: 0 10px 20px rgba(59, 130, 246, 0.25); }
    .stat-card-syndron.secondary { background: var(--secondary-gradient); box-shadow: 0 10px 20px rgba(79, 70, 229, 0.25); }
    .stat-card-syndron.danger { background: var(--danger-gradient); box-shadow: 0 10px 20px rgba(239, 68, 68, 0.25); }
    
    .stat-icon-syndron {
        position: absolute;
        top: 20px;
        right: 20px;
        font-size: 50px;
        opacity: 0.2;
        color: white;
        transition: all 0.4s ease;
    }
    
    .stat-card-syndron:hover .stat-icon-syndron {
        transform: scale(1.1);
        opacity: 0.3;
    }
    
    .stat-value-syndron {
        font-size: 36px;
        font-weight: 800;
        margin-bottom: 5px;
        letter-spacing: -0.5px;
        text-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .stat-label-syndron {
        font-size: 14px;
        font-weight: 500;
        opacity: 0.9;
        letter-spacing: 0.5px;
    }
    
    /* Enhanced Filter Section */
    .filter-section-syndron {
        background: white;
        border-radius: 15px;
        padding: 20px;
        margin-bottom: 20px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #e5e7eb;
    }
    
    @media (max-width: 768px) {
        .filter-section-syndron {
            padding: 15px;
            margin-bottom: 15px;
        }
        
        .filter-section-syndron .row > div {
            margin-bottom: 15px;
        }
        
        .filter-section-syndron .input-group {
            flex-direction: column;
            gap: 10px;
        }
        
        .filter-section-syndron .input-group .input-group-text {
            border-radius: 8px !important;
            justify-content: center;
        }
    }
    
    /* Quick Actions */
    .quick-actions-syndron {
        display: flex;
        gap: 12px;
        margin-bottom: 20px;
        flex-wrap: wrap;
    }
    
    @media (max-width: 768px) {
        .quick-actions-syndron {
            flex-direction: column;
            gap: 8px;
        }
        
        .quick-actions-syndron .btn {
            width: 100%;
            justify-content: center;
        }
    }
    
    /* Purchase Cards - Card View */
    .purchase-card-view {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 20px;
        margin-bottom: 20px;
    }
    
    @media (max-width: 768px) {
        .purchase-card-view {
            grid-template-columns: 1fr;
            gap: 15px;
        }
    }
    
    .purchase-card {
        background: white;
        border-radius: 15px;
        padding: 20px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.08);
        transition: all 0.3s ease;
        border-left: 5px solid;
        position: relative;
        overflow: hidden;
    }
    
    @media (max-width: 768px) {
        .purchase-card {
            padding: 15px;
        }
    }
    
    .purchase-card:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.12);
    }
    
    .purchase-card.pending { border-left-color: #F59E0B; background: linear-gradient(to right, #fffaf0, white); }
    .purchase-card.approved { border-left-color: #3B82F6; background: linear-gradient(to right, #eff6ff, white); }
    .purchase-card.received { border-left-color: #8B5CF6; background: linear-gradient(to right, #f5f3ff, white); }
    .purchase-card.completed { border-left-color: #10B981; background: linear-gradient(to right, #f0fdf4, white); }
    .purchase-card.delivered { border-left-color: #6366F1; background: linear-gradient(to right, #eef2ff, white); }
    .purchase-card.cancelled { border-left-color: #EF4444; background: linear-gradient(to right, #fef2f2, white); }
    
    /* Card Status Badge */
    .card-status-badge {
        position: absolute;
        top: 15px;
        right: 15px;
        padding: 6px 15px;
        border-radius: 20px;
        font-size: 11px;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        box-shadow: 0 3px 10px rgba(0,0,0,0.1);
    }
    
    .badge-pending { background: var(--warning-gradient); color: white; }
    .badge-approved { background: var(--info-gradient); color: white; }
    .badge-received { background: var(--secondary-gradient); color: white; }
    .badge-completed { background: var(--success-gradient); color: white; }
    .badge-delivered { background: #6366F1; color: white; }
    .badge-cancelled { background: var(--danger-gradient); color: white; }
    
    /* Card Content */
    .card-header {
        margin-bottom: 15px;
        padding-bottom: 15px;
        border-bottom: 1px solid #f1f5f9;
    }
    
    .card-header h5 {
        font-weight: 700;
        color: #1e293b;
        margin: 0;
        font-size: 16px;
    }
    
    .card-header .invoice-number {
        color: #64748b;
        font-size: 13px;
        font-weight: 500;
    }
    
    .card-details {
        margin-bottom: 20px;
    }
    
    .detail-row {
        display: flex;
        justify-content: space-between;
        margin-bottom: 10px;
        font-size: 14px;
    }
    
    .detail-label {
        color: #64748b;
        font-weight: 500;
    }
    
    .detail-value {
        color: #1e293b;
        font-weight: 600;
        text-align: right;
    }
    
    .detail-value.text-success { color: #10B981; }
    .detail-value.text-warning { color: #F59E0B; }
    .detail-value.text-danger { color: #EF4444; }
    
    /* Action Buttons */
    .card-actions {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
        margin-top: 15px;
        padding-top: 15px;
        border-top: 1px solid #f1f5f9;
    }
    
    @media (max-width: 768px) {
        .card-actions {
            flex-direction: column;
        }
        
        .card-actions .btn {
            width: 100%;
            justify-content: center;
        }
    }
    
    .action-btn {
        flex: 1;
        padding: 8px 12px;
        border-radius: 8px;
        font-size: 13px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.3px;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 6px;
        transition: all 0.3s ease;
        border: none;
        min-width: 120px;
    }
    
    .action-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    }
    
    .btn-view { background: #e0e7ff; color: #4f46e5; }
    .btn-view:hover { background: #c7d2fe; }
    
    .btn-edit { background: #fef3c7; color: #d97706; }
    .btn-edit:hover { background: #fde68a; }
    
    .btn-approve { background: #dbeafe; color: #1d4ed8; }
    .btn-approve:hover { background: #bfdbfe; }
    
    .btn-receive { background: #ede9fe; color: #7c3aed; }
    .btn-receive:hover { background: #ddd6fe; }
    
    .btn-inventory { background: #d1fae5; color: #059669; }
    .btn-inventory:hover { background: #a7f3d0; }
    
    .btn-deliver { background: #e0e7ff; color: #4f46e5; }
    .btn-deliver:hover { background: #c7d2fe; }
    
    .btn-payment { background: #d1fae5; color: #059669; }
    .btn-payment:hover { background: #a7f3d0; }
    
    .btn-delete { background: #fee2e2; color: #dc2626; }
    .btn-delete:hover { background: #fecaca; }
    
    .btn-cancel { background: #fef3c7; color: #d97706; }
    .btn-cancel:hover { background: #fde68a; }
    
    /* View Toggle */
    .view-toggle {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
        background: white;
        padding: 10px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    .view-btn {
        padding: 8px 20px;
        border-radius: 8px;
        background: #f8fafc;
        border: 2px solid transparent;
        color: #64748b;
        font-weight: 600;
        font-size: 13px;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 8px;
    }
    
    .view-btn.active {
        background: var(--primary-gradient);
        color: white;
        box-shadow: 0 4px 10px rgba(102, 126, 234, 0.2);
    }
    
    .view-btn:hover:not(.active) {
        background: #e2e8f0;
        color: #475569;
    }
    
    /* Empty State */
    .empty-state-syndron {
        text-align: center;
        padding: 60px 20px;
        background: white;
        border-radius: 15px;
        margin: 30px 0;
        border: 2px dashed #e2e8f0;
        box-shadow: 0 4px 20px rgba(0,0,0,0.05);
    }
    
    @media (max-width: 768px) {
        .empty-state-syndron {
            padding: 40px 15px;
            margin: 20px 0;
        }
        
        .empty-state-syndron i {
            font-size: 60px !important;
        }
    }
    
    /* Payment Status Badge */
    .payment-badge {
        padding: 4px 10px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.3px;
    }
    
    .payment-paid { background: #10B981; color: white; }
    .payment-partial { background: #F59E0B; color: white; }
    .payment-pending { background: #64748b; color: white; }
    
    /* Progress Steps */
    .progress-steps-mini {
        display: flex;
        justify-content: space-between;
        position: relative;
        margin: 15px 0;
        padding: 0 10px;
    }
    
    .progress-steps-mini:before {
        content: '';
        position: absolute;
        top: 8px;
        left: 20px;
        right: 20px;
        height: 2px;
        background: #e2e8f0;
        z-index: 1;
    }
    
    .progress-step-mini {
        position: relative;
        z-index: 2;
        text-align: center;
        flex: 1;
    }
    
    .step-circle-mini {
        width: 18px;
        height: 18px;
        border-radius: 50%;
        background: white;
        border: 2px solid #e2e8f0;
        margin: 0 auto 5px;
        position: relative;
    }
    
    .step-circle-mini.active {
        background: #667eea;
        border-color: #667eea;
    }
    
    .step-circle-mini.completed {
        background: #10B981;
        border-color: #10B981;
    }
    
    .step-label-mini {
        font-size: 9px;
        color: #64748b;
        font-weight: 500;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }
    
    .step-label-mini.active {
        color: #667eea;
        font-weight: 600;
    }
    
    /* Bulk Actions */
    .bulk-actions {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
        padding: 15px;
        background: white;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    @media (max-width: 768px) {
        .bulk-actions {
            flex-direction: column;
        }
        
        .bulk-actions .btn {
            width: 100%;
            justify-content: center;
        }
    }
    
    /* Loading Animation */
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(255, 255, 255, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 9999;
    }
    
    .spinner {
        width: 50px;
        height: 50px;
        border: 3px solid #f3f3f3;
        border-top: 3px solid #667eea;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-4">
            <div class="breadcrumb-title pe-3">
                <i class="bx bx-cart me-1"></i> Purchases
            </div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Purchases</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('purchases.create') }}" class="btn btn-primary">
                        <i class="bx bx-plus"></i> New Purchase
                    </a>
                    <button type="button" class="btn btn-primary dropdown-toggle dropdown-toggle-split" data-bs-toggle="dropdown">
                        <span class="visually-hidden">Toggle Dropdown</span>
                    </button>
                    <div class="dropdown-menu dropdown-menu-end">
                        <a class="dropdown-item" href="{{ route('purchases.export') }}">
                            <i class="bx bx-download"></i> Export
                        </a>
                        <a class="dropdown-item" href="{{ route('reports.purchases') }}">
                            <i class="bx bx-line-chart"></i> Reports
                        </a>
                        <div class="dropdown-divider"></div>
                        <a class="dropdown-item" href="{{ route('purchases.index', ['status' => 'pending']) }}">
                            <i class="bx bx-time"></i> View Pending
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Statistics Cards -->
        <div class="row row-cols-1 row-cols-md-6 row-cols-xl-6 mb-4 g-3">
            <div class="col">
                <div class="stat-card-syndron info">
                    <i class="bx bx-cart-alt stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['total_purchases'] ?? 0 }}</div>
                        <div class="stat-label-syndron">Total Purchases</div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron success">
                    <i class="bx bx-check-circle stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['completed_purchases'] ?? 0 }}</div>
                        <div class="stat-label-syndron">Completed</div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron warning">
                    <i class="bx bx-time-five stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['pending_purchases'] ?? 0 }}</div>
                        <div class="stat-label-syndron">Pending</div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron secondary">
                    <i class="bx bx-money stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">KES {{ number_format($stats['total_amount'] ?? 0, 0) }}</div>
                        <div class="stat-label-syndron">Total Amount</div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron info">
                    <i class="bx bx-credit-card stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">KES {{ number_format($stats['total_paid'] ?? 0, 0) }}</div>
                        <div class="stat-label-syndron">Paid Amount</div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron danger">
                    <i class="bx bx-alarm stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">KES {{ number_format($stats['total_outstanding'] ?? 0, 0) }}</div>
                        <div class="stat-label-syndron">Outstanding</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="quick-actions-syndron">
            <a href="{{ route('purchases.index', ['status' => 'pending']) }}" class="btn btn-outline-primary">
                <i class="bx bx-time"></i> Pending
            </a>
            <a href="{{ route('purchases.index', ['status' => 'approved']) }}" class="btn btn-outline-info">
                <i class="bx bx-check"></i> Approved
            </a>
            <a href="{{ route('purchases.index', ['status' => 'received']) }}" class="btn btn-outline-secondary">
                <i class="bx bx-package"></i> Received
            </a>
            <a href="{{ route('purchases.index', ['status' => 'completed']) }}" class="btn btn-outline-success">
                <i class="bx bx-check-circle"></i> Completed
            </a>
            <a href="{{ route('purchases.index', ['status' => 'cancelled']) }}" class="btn btn-outline-danger">
                <i class="bx bx-x"></i> Cancelled
            </a>
            <a href="{{ route('purchases.create') }}" class="btn btn-primary">
                <i class="bx bx-plus"></i> New Purchase
            </a>
        </div>

        <!-- Filters Section -->
        <div class="filter-section-syndron">
            <form action="{{ route('purchases.index') }}" method="GET" id="filterForm">
                <div class="row align-items-center">
                    <div class="col-12 col-md-3 mb-3 mb-md-0">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select" onchange="applyFilters()">
                            <option value="">All Status</option>
                            <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                            <option value="approved" {{ request('status') == 'approved' ? 'selected' : '' }}>Approved</option>
                            <option value="received" {{ request('status') == 'received' ? 'selected' : '' }}>Received</option>
                            <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                            <option value="delivered" {{ request('status') == 'delivered' ? 'selected' : '' }}>Delivered</option>
                            <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                        </select>
                    </div>
                    <div class="col-12 col-md-3 mb-3 mb-md-0">
                        <label class="form-label">Supplier</label>
                        <select name="supplier_id" class="form-select" onchange="applyFilters()">
                            <option value="">All Suppliers</option>
                            @foreach($suppliers ?? [] as $supplier)
                                <option value="{{ $supplier->id }}" {{ request('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                    {{ $supplier->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-12 col-md-4 mb-3 mb-md-0">
                        <label class="form-label">Date Range</label>
                        <div class="input-group">
                            <input type="date" name="start_date" class="form-control" 
                                   value="{{ request('start_date') }}" onchange="applyFilters()">
                            <span class="input-group-text bg-white">to</span>
                            <input type="date" name="end_date" class="form-control" 
                                   value="{{ request('end_date') }}" onchange="applyFilters()">
                        </div>
                    </div>
                    <div class="col-12 col-md-2 d-flex align-items-end">
                        <div class="d-flex gap-2 w-100">
                            <button type="button" class="btn btn-outline-secondary w-50" onclick="resetFilters()">
                                <i class="bx bx-reset"></i> Reset
                            </button>
                            <button type="button" class="btn btn-primary w-50" onclick="applyFilters()">
                                <i class="bx bx-search"></i> Filter
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Search Box -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="input-group">
                    <input type="text" id="searchInput" class="form-control" 
                           placeholder="Search invoices, suppliers, reference numbers..."
                           value="{{ request('search') }}" 
                           onkeyup="debouncedSearch()">
                    <button class="btn btn-primary" type="button" onclick="applyFilters()">
                        <i class="bx bx-search"></i>
                    </button>
                </div>
            </div>
            <div class="col-md-6 text-end">
                <div class="view-toggle">
                    <button class="view-btn active" data-view="card">
                        <i class="bx bx-grid-alt"></i> Card View
                    </button>
                    <button class="view-btn" data-view="table">
                        <i class="bx bx-table"></i> Table View
                    </button>
                </div>
            </div>
        </div>

        <!-- Card View -->
        <div id="cardView" class="purchase-card-view">
            @forelse($purchases as $purchase)
                <div class="purchase-card {{ $purchase->status }}">
                    <!-- Status Badge -->
                    <span class="card-status-badge badge-{{ $purchase->status }}">
                        {{ ucfirst($purchase->status) }}
                    </span>
                    
                    <!-- Card Header -->
                    <div class="card-header">
                        <h5>{{ $purchase->supplier->name ?? 'No Supplier' }}</h5>
                        <div class="invoice-number">{{ $purchase->invoice_no }}</div>
                        @if($purchase->reference_number)
                            <small class="text-muted">{{ $purchase->reference_number }}</small>
                        @endif
                    </div>
                    
                    <!-- Card Details -->
                    <div class="card-details">
                        <div class="detail-row">
                            <span class="detail-label">Date</span>
                            <span class="detail-value">{{ $purchase->purchase_date->format('M d, Y') }}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Items</span>
                            <span class="detail-value">{{ $purchase->items_count ?? 0 }} items</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Total</span>
                            <span class="detail-value text-primary">KES {{ number_format($purchase->grand_total, 2) }}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Payment</span>
                            <span class="detail-value">
                                <span class="payment-badge payment-{{ $purchase->payment_status }}">
                                    {{ ucfirst($purchase->payment_status) }}
                                </span>
                            </span>
                        </div>
                        @if($purchase->is_credit_purchase)
                            <div class="detail-row">
                                <span class="detail-label">Credit</span>
                                <span class="detail-value text-warning">
                                    <i class="bx bx-credit-card"></i> Enabled
                                </span>
                            </div>
                        @endif
                        
                        <!-- Progress Steps -->
                        <div class="progress-steps-mini">
                            <div class="progress-step-mini">
                                <div class="step-circle-mini {{ in_array($purchase->status, ['approved', 'received', 'completed', 'delivered']) ? 'completed' : ($purchase->status == 'pending' ? 'active' : '') }}"></div>
                                <div class="step-label-mini {{ $purchase->status == 'pending' ? 'active' : '' }}">Pending</div>
                            </div>
                            <div class="progress-step-mini">
                                <div class="step-circle-mini {{ in_array($purchase->status, ['received', 'completed', 'delivered']) ? 'completed' : ($purchase->status == 'approved' ? 'active' : '') }}"></div>
                                <div class="step-label-mini {{ $purchase->status == 'approved' ? 'active' : '' }}">Approved</div>
                            </div>
                            <div class="progress-step-mini">
                                <div class="step-circle-mini {{ in_array($purchase->status, ['completed', 'delivered']) ? 'completed' : ($purchase->status == 'received' ? 'active' : '') }}"></div>
                                <div class="step-label-mini {{ $purchase->status == 'received' ? 'active' : '' }}">Received</div>
                            </div>
                            <div class="progress-step-mini">
                                <div class="step-circle-mini {{ $purchase->status == 'completed' ? 'completed' : ($purchase->status == 'delivered' ? 'active' : '') }}"></div>
                                <div class="step-label-mini {{ in_array($purchase->status, ['completed', 'delivered']) ? 'active' : '' }}">Inventory</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="card-actions">
                        <!-- View Button -->
                        <button class="action-btn btn-view" onclick="viewPurchase({{ $purchase->id }})">
                            <i class="bx bx-show"></i> View
                        </button>
                        
                        @if($purchase->status == 'pending')
                            <!-- Edit Button -->
                            <button class="action-btn btn-edit" onclick="editPurchase({{ $purchase->id }})">
                                <i class="bx bx-edit"></i> Edit
                            </button>
                            
                            <!-- Approve Button -->
                            <button class="action-btn btn-approve" onclick="approvePurchase({{ $purchase->id }})">
                                <i class="bx bx-check"></i> Approve
                            </button>
                            
                            <!-- Cancel Button -->
                            <button class="action-btn btn-cancel" onclick="cancelPurchase({{ $purchase->id }})">
                                <i class="bx bx-x"></i> Cancel
                            </button>
                            
                        @elseif($purchase->status == 'approved')
                            <!-- Receive Button -->
                            <button class="action-btn btn-receive" onclick="receivePurchase({{ $purchase->id }})">
                                <i class="bx bx-package"></i> Receive
                            </button>
                            
                        @elseif($purchase->status == 'received')
                            <!-- Add to Inventory Button -->
                            <button class="action-btn btn-inventory" onclick="addToInventory({{ $purchase->id }})">
                                <i class="bx bx-plus"></i> Add to Inventory
                            </button>
                            
                        @elseif($purchase->status == 'completed')
                            @php
                                $hasServices = $purchase->items()
                                    ->where(function($q) {
                                        $q->where('item_type', 'service')
                                          ->orWhereHas('product', function($q) {
                                              $q->where('is_service', true);
                                          });
                                    })
                                    ->exists();
                            @endphp
                            
                            @if($hasServices)
                                <!-- Deliver Services Button -->
                                <button class="action-btn btn-deliver" onclick="deliverServices({{ $purchase->id }})">
                                    <i class="bx bx-truck"></i> Deliver
                                </button>
                            @endif
                            
                        @endif
                        
                        @if($purchase->grand_total > $purchase->paid_amount)
                            <!-- Payment Button -->
                            <button class="action-btn btn-payment" onclick="recordPayment({{ $purchase->id }})">
                                <i class="bx bx-money"></i> Payment
                            </button>
                        @endif
                        
                        <!-- Delete Button (only for draft/cancelled with no stock) -->
                        @if(in_array($purchase->status, ['pending', 'draft', 'cancelled']) && $purchase->items()->where('added_to_stock', false)->count() == $purchase->items()->count())
                            <button class="action-btn btn-delete" onclick="deletePurchase({{ $purchase->id }})">
                                <i class="bx bx-trash"></i> Delete
                            </button>
                        @endif
                    </div>
                </div>
            @empty
                <div class="col-12">
                    <div class="empty-state-syndron">
                        <i class="bx bx-package"></i>
                        <h5 class="mt-3 mb-2">No purchases found</h5>
                        <p class="text-muted mb-4">Start by creating your first inventory purchase.</p>
                        <a href="{{ route('purchases.create') }}" class="btn btn-primary">
                            <i class="bx bx-plus"></i> Create Purchase
                        </a>
                    </div>
                </div>
            @endforelse
        </div>

        <!-- Table View (Hidden by default) -->
        <div id="tableView" class="card border-0 shadow-sm" style="display: none;">
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Invoice No.</th>
                                <th>Supplier</th>
                                <th>Date</th>
                                <th>Items</th>
                                <th>Total</th>
                                <th>Status</th>
                                <th>Payment</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($purchases as $purchase)
                                <tr>
                                    <td>
                                        <strong>{{ $purchase->invoice_no }}</strong>
                                        @if($purchase->reference_number)
                                            <br><small class="text-muted">{{ $purchase->reference_number }}</small>
                                        @endif
                                    </td>
                                    <td>{{ $purchase->supplier->name ?? 'No Supplier' }}</td>
                                    <td>{{ $purchase->purchase_date->format('M d, Y') }}</td>
                                    <td><span class="badge bg-light text-dark">{{ $purchase->items_count ?? 0 }} items</span></td>
                                    <td><strong>KES {{ number_format($purchase->grand_total, 2) }}</strong></td>
                                    <td>
                                        <span class="badge badge-{{ $purchase->status }}">
                                            {{ ucfirst($purchase->status) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge payment-{{ $purchase->payment_status }}">
                                            {{ ucfirst($purchase->payment_status) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="dropdown">
                                            <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                <i class="bx bx-dots-horizontal-rounded"></i>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li><a class="dropdown-item" href="{{ route('purchases.show', $purchase) }}"><i class="bx bx-show me-2"></i> View</a></li>
                                                
                                                @if($purchase->status == 'pending')
                                                    <li><a class="dropdown-item" href="{{ route('purchases.edit', $purchase) }}"><i class="bx bx-edit me-2"></i> Edit</a></li>
                                                    <li><button class="dropdown-item" onclick="approvePurchase({{ $purchase->id }})"><i class="bx bx-check me-2"></i> Approve</button></li>
                                                    <li><button class="dropdown-item" onclick="cancelPurchase({{ $purchase->id }})"><i class="bx bx-x me-2"></i> Cancel</button></li>
                                                @endif
                                                
                                                @if($purchase->status == 'approved')
                                                    <li><button class="dropdown-item" onclick="receivePurchase({{ $purchase->id }})"><i class="bx bx-package me-2"></i> Receive</button></li>
                                                @endif
                                                
                                                @if($purchase->status == 'received')
                                                    <li><button class="dropdown-item" onclick="addToInventory({{ $purchase->id }})"><i class="bx bx-plus me-2"></i> Add to Inventory</button></li>
                                                @endif
                                                
                                                @if($purchase->grand_total > $purchase->paid_amount)
                                                    <li><a class="dropdown-item" href="{{ route('purchases.payment', $purchase) }}"><i class="bx bx-money me-2"></i> Record Payment</a></li>
                                                @endif
                                                
                                                @if(in_array($purchase->status, ['pending', 'draft', 'cancelled']) && $purchase->items()->where('added_to_stock', false)->count() == $purchase->items()->count())
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li><button class="dropdown-item text-danger" onclick="deletePurchase({{ $purchase->id }})"><i class="bx bx-trash me-2"></i> Delete</button></li>
                                                @endif
                                            </ul>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8" class="text-center py-4">
                                        <div class="empty-state-syndron">
                                            <i class="bx bx-package"></i>
                                            <h5 class="mt-3 mb-2">No purchases found</h5>
                                            <p class="text-muted mb-4">Start by creating your first inventory purchase.</p>
                                            <a href="{{ route('purchases.create') }}" class="btn btn-primary">
                                                <i class="bx bx-plus"></i> Create Purchase
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Pagination -->
        @if($purchases->hasPages())
            <div class="mt-4">
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        {{ $purchases->withQueryString()->links() }}
                    </ul>
                </nav>
            </div>
        @endif

        <!-- Recent Suppliers -->
        @if(isset($recentSuppliers) && $recentSuppliers->count() > 0)
            <div class="card border-0 shadow-sm mt-4">
                <div class="card-header bg-white">
                    <h6 class="mb-0"><i class="bx bx-user me-2"></i> Recent Suppliers</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        @foreach($recentSuppliers as $supplier)
                            <div class="col-6 col-md-3 mb-3">
                                <div class="vendor-card">
                                    <div class="d-flex align-items-center">
                                        <div class="flex-shrink-0">
                                            <div class="bg-primary rounded-circle p-2 me-2">
                                                <i class="bx bx-user text-white"></i>
                                            </div>
                                        </div>
                                        <div class="flex-grow-1">
                                            <h6 class="mb-1">{{ $supplier->name }}</h6>
                                            <small class="text-muted">{{ $supplier->phone }}</small>
                                            <div class="small text-primary">
                                                {{ $supplier->purchases_count }} purchases
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        @endif
    </div>
</div>

<!-- Loading Overlay -->
<div id="loadingOverlay" class="loading-overlay" style="display: none;">
    <div class="spinner"></div>
</div>
@endsection

@section('script')
<script>
// Global variables
let loading = false;

// Show loading overlay
function showLoading() {
    loading = true;
    document.getElementById('loadingOverlay').style.display = 'flex';
}

// Hide loading overlay
function hideLoading() {
    loading = false;
    document.getElementById('loadingOverlay').style.display = 'none';
}

// SweetAlert wrapper
function showAlert(title, message, type = 'success', callback = null) {
    Swal.fire({
        title: title,
        text: message,
        icon: type,
        confirmButtonColor: type === 'error' ? '#EF4444' : '#667eea',
        confirmButtonText: 'OK',
        allowOutsideClick: false
    }).then((result) => {
        if (callback && typeof callback === 'function') {
            callback(result);
        }
    });
}

// Confirm dialog wrapper
function showConfirm(title, message, confirmText = 'Confirm', cancelText = 'Cancel', type = 'warning') {
    return Swal.fire({
        title: title,
        text: message,
        icon: type,
        showCancelButton: true,
        confirmButtonColor: '#667eea',
        cancelButtonColor: '#6c757d',
        confirmButtonText: confirmText,
        cancelButtonText: cancelText
    });
}

// Purchase Actions
function viewPurchase(purchaseId) {
    window.location.href = `/purchases/${purchaseId}`;
}

function editPurchase(purchaseId) {
    window.location.href = `/purchases/${purchaseId}/edit`;
}

function approvePurchase(purchaseId) {
    showConfirm('Approve Purchase', 'Are you sure you want to approve this purchase?', 'Approve', 'Cancel')
        .then((result) => {
            if (result.isConfirmed) {
                showLoading();
                
                fetch(`/purchases/${purchaseId}/approve`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        showAlert('Approved!', data.message, 'success', () => {
                            location.reload();
                        });
                    } else {
                        showAlert('Error!', data.message, 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showAlert('Error!', 'Failed to approve purchase.', 'error');
                });
            }
        });
}

function receivePurchase(purchaseId) {
    showConfirm('Mark as Received', 'Are you sure the purchase items have been received?', 'Mark Received', 'Cancel')
        .then((result) => {
            if (result.isConfirmed) {
                showLoading();
                
                fetch(`/purchases/${purchaseId}/receive`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        showAlert('Received!', data.message, 'success', () => {
                            location.reload();
                        });
                    } else {
                        showAlert('Error!', data.message, 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showAlert('Error!', 'Failed to mark as received.', 'error');
                });
            }
        });
}

function addToInventory(purchaseId) {
    showLoading();
    
    // First check if purchase can be added to inventory
    fetch(`/purchases/${purchaseId}/check-inventory`)
        .then(response => response.json())
        .then(data => {
            hideLoading();
            
            if (data.success && data.can_add_to_inventory) {
                showConfirm('Add to Inventory', 
                    `Add ${data.inventory_items_count} inventory item(s) to stock?${data.service_items_count > 0 ? `\n${data.service_items_count} service item(s) will be marked as delivered.` : ''}`, 
                    'Add to Inventory', 'Cancel')
                    .then((result) => {
                        if (result.isConfirmed) {
                            showLoading();
                            
                            fetch(`/purchases/${purchaseId}/complete`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                }
                            })
                            .then(response => response.json())
                            .then(data => {
                                hideLoading();
                                if (data.success) {
                                    showAlert('Success!', data.message, 'success', () => {
                                        location.reload();
                                    });
                                } else {
                                    showAlert('Error!', data.message, 'error');
                                }
                            })
                            .catch(error => {
                                hideLoading();
                                showAlert('Error!', 'Failed to add to inventory.', 'error');
                            });
                        }
                    });
            } else {
                showAlert('Cannot Add to Inventory', 
                    data.message || 'Purchase must be received before adding to inventory.', 
                    'warning');
            }
        })
        .catch(error => {
            hideLoading();
            showAlert('Error!', 'Failed to check inventory status.', 'error');
        });
}

function deliverServices(purchaseId) {
    showConfirm('Mark as Delivered', 'Are you sure all services have been delivered?', 'Mark Delivered', 'Cancel')
        .then((result) => {
            if (result.isConfirmed) {
                showLoading();
                
                fetch(`/purchases/${purchaseId}/deliver`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        showAlert('Delivered!', data.message, 'success', () => {
                            location.reload();
                        });
                    } else {
                        showAlert('Error!', data.message, 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showAlert('Error!', 'Failed to mark as delivered.', 'error');
                });
            }
        });
}

function cancelPurchase(purchaseId) {
    Swal.fire({
        title: 'Cancel Purchase',
        html: `<p>Are you sure you want to cancel this purchase?</p>
              <div class="mt-3">
                  <label class="form-label">Cancellation Reason</label>
                  <textarea id="cancelReason" class="form-control" rows="3" placeholder="Enter reason for cancellation..."></textarea>
              </div>`,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#F59E0B',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Cancel Purchase',
        cancelButtonText: 'Back',
        preConfirm: () => {
            const reason = document.getElementById('cancelReason').value;
            if (!reason) {
                Swal.showValidationMessage('Please provide a cancellation reason');
                return false;
            }
            return { reason: reason };
        }
    }).then((result) => {
        if (result.isConfirmed && result.value) {
            showLoading();
            
            fetch(`/purchases/${purchaseId}/cancel`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ reason: result.value.reason })
            })
            .then(response => response.json())
            .then(data => {
                hideLoading();
                if (data.success) {
                    showAlert('Cancelled!', data.message, 'success', () => {
                        location.reload();
                    });
                } else {
                    showAlert('Error!', data.message, 'error');
                }
            })
            .catch(error => {
                hideLoading();
                showAlert('Error!', 'Failed to cancel purchase.', 'error');
            });
        }
    });
}

function recordPayment(purchaseId) {
    window.location.href = `/purchases/${purchaseId}/payment`;
}

function deletePurchase(purchaseId) {
    showConfirm('Delete Purchase', 'Are you sure you want to delete this purchase? This action cannot be undone.', 'Delete', 'Cancel', 'error')
        .then((result) => {
            if (result.isConfirmed) {
                showLoading();
                
                fetch(`/purchases/${purchaseId}`, {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        showAlert('Deleted!', data.message, 'success', () => {
                            location.reload();
                        });
                    } else {
                        showAlert('Error!', data.message, 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showAlert('Error!', 'Failed to delete purchase.', 'error');
                });
            }
        });
}

// View Toggle
document.addEventListener('DOMContentLoaded', function() {
    const cardView = document.getElementById('cardView');
    const tableView = document.getElementById('tableView');
    const viewButtons = document.querySelectorAll('.view-btn');
    
    viewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const view = this.dataset.view;
            
            // Update active button
            viewButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Show selected view
            if (view === 'card') {
                cardView.style.display = 'grid';
                tableView.style.display = 'none';
            } else {
                cardView.style.display = 'none';
                tableView.style.display = 'block';
            }
        });
    });
});

// Filter Functions
$(document).ready(function() {
    // Debounced search
    let searchTimeout;
    function debouncedSearch() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            applyFilters();
        }, 500);
    }
    window.debouncedSearch = debouncedSearch;
});

function applyFilters() {
    if (loading) return;
    
    const form = document.getElementById('filterForm');
    const search = document.getElementById('searchInput').value;
    
    // Add search parameter if exists
    let searchInput = form.querySelector('input[name="search"]');
    if (search) {
        if (!searchInput) {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'search';
            input.value = search;
            form.appendChild(input);
        } else {
            searchInput.value = search;
        }
    } else if (searchInput) {
        searchInput.remove();
    }
    
    showLoading();
    setTimeout(() => {
        form.submit();
    }, 300);
}

function resetFilters() {
    if (loading) return;
    
    showLoading();
    window.location.href = '{{ route("purchases.index") }}';
}
</script>
@endsection