@extends('layouts.app')

@section('style')
<style>
    .wizard-step {
        display: none;
        animation: fadeIn 0.5s;
    }
    
    .wizard-step.active {
        display: block;
    }
    
    .item-row {
        margin-bottom: 15px;
        padding: 15px;
        background: #f8f9fa;
        border-radius: 6px;
        position: relative;
        border: 1px solid #dee2e6;
    }
    
    .remove-btn {
        position: absolute;
        top: 10px;
        right: 10px;
    }
    
    .item-type-badge {
        position: absolute;
        top: 10px;
        left: 15px;
    }
    
    .stock-info {
        font-size: 12px;
        color: #6c757d;
        margin-top: 5px;
    }
    
    .validation-error {
        color: #dc3545;
        font-size: 12px;
        margin-top: 5px;
        display: none;
    }
    
    .wizard-progress {
        display: flex;
        justify-content: space-between;
        margin-bottom: 30px;
        position: relative;
    }
    
    .wizard-progress:before {
        content: '';
        position: absolute;
        top: 20px;
        left: 0;
        right: 0;
        height: 2px;
        background: #e9ecef;
        z-index: 1;
    }
    
    .progress-step {
        position: relative;
        z-index: 2;
        text-align: center;
        flex: 1;
    }
    
    .step-circle {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        background: #e9ecef;
        color: #6c757d;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 10px;
        border: 3px solid #fff;
        font-weight: bold;
    }
    
    .step-circle.active {
        background: #0d6efd;
        color: #fff;
    }
    
    .step-circle.completed {
        background: #198754;
        color: #fff;
    }
    
    .step-label {
        font-size: 12px;
        color: #6c757d;
        font-weight: 500;
    }
    
    .step-label.active {
        color: #0d6efd;
        font-weight: bold;
    }
    
    .wizard-nav {
        margin-top: 30px;
        padding-top: 20px;
        border-top: 1px solid #dee2e6;
        display: flex;
        justify-content: space-between;
    }
    
    .batch-fields {
        background: #e9ecef;
        padding: 10px;
        border-radius: 5px;
        margin-top: 10px;
    }
    
    .item-summary {
        background: #fff;
        border: 1px solid #dee2e6;
        border-radius: 5px;
        padding: 15px;
        margin-bottom: 20px;
    }
    
    .summary-table {
        width: 100%;
        margin-bottom: 15px;
    }
    
    .summary-table th {
        background: #f8f9fa;
        padding: 10px;
        text-align: left;
        font-weight: 600;
        border-bottom: 1px solid #dee2e6;
    }
    
    .summary-table td {
        padding: 10px;
        border-bottom: 1px solid #dee2e6;
    }
    
    .summary-table tr:last-child td {
        border-bottom: none;
    }
    
    .credit-limit-alert {
        transition: all 0.3s;
    }
    
    @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
    }
</style>
@endsection

@section('content')
@php
    // Get default tax rate from database
    $defaultTax = \App\Models\Tax::where('is_default', true)->first();
    $defaultTaxRate = $defaultTax ? $defaultTax->rate : 16.00;
@endphp

<div class="page-wrapper">
    <div class="page-content">
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">
                <i class="bx bx-cart me-1"></i> Create Purchase Order
            </div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('purchases.index') }}">Purchases</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Create</li>
                    </ol>
                </nav>
            </div>
        </div>

        @if($errors->any())
        <div class="alert alert-danger">
            <h6 class="alert-heading">Please fix the following errors:</h6>
            <ul class="mb-0">
                @foreach($errors->all() as $error)
                <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
        @endif

        <div class="card border-0 shadow-sm">
            <div class="card-body">
                <!-- Wizard Progress -->
                <div class="wizard-progress">
                    <div class="progress-step">
                        <div class="step-circle active" id="step1-circle">1</div>
                        <div class="step-label active">Add Items</div>
                    </div>
                    <div class="progress-step">
                        <div class="step-circle" id="step2-circle">2</div>
                        <div class="step-label">Supplier & Payment</div>
                    </div>
                    <div class="progress-step">
                        <div class="step-circle" id="step3-circle">3</div>
                        <div class="step-label">Review & Complete</div>
                    </div>
                </div>

                <form action="{{ route('purchases.store') }}" method="POST" id="createPurchaseForm">
                    @csrf
                    
                    <!-- Step 1: Add Items -->
                    <div class="wizard-step active" id="step1">
                        <h4 class="mb-3"><i class="bx bx-package"></i> Purchase Items</h4>
                        
                        <div class="alert alert-info mb-3">
                            <i class="bx bx-info-circle"></i>
                            <small>Add products to your purchase order. These will be added to inventory when the purchase is completed.</small>
                        </div>
                        
                        <div id="purchaseItems">
                            <div class="item-row" data-index="0">
                                <span class="badge bg-secondary item-type-badge">Product</span>
                                
                                <button type="button" class="btn btn-sm btn-danger remove-btn" onclick="removeItem(this)" title="Remove item">
                                    <i class="bx bx-trash"></i>
                                </button>
                                
                                <div class="product-fields">
                                    <div class="row">
                                        <div class="col-md-5">
                                            <label class="form-label">Product *</label>
                                            <select name="items[0][product_id]" class="form-select product-select" 
                                                    onchange="onProductChange(this, 0)" required>
                                                <option value="">Select Product</option>
                                                @foreach($products as $product)
                                                    <option value="{{ $product->id }}" 
                                                            data-cost="{{ $product->cost_price }}"
                                                            data-stock="{{ $product->stock }}"
                                                            data-has-expiry="{{ $product->has_expiry }}"
                                                            data-track-batches="{{ $product->track_batches }}"
                                                            data-tax-rate="{{ $product->tax_rate ?? $defaultTaxRate }}">
                                                        {{ $product->name }} ({{ $product->sku }}) - Stock: {{ $product->stock }}
                                                        @if($product->tax_rate)
                                                            (Tax: {{ $product->tax_rate }}%)
                                                        @endif
                                                    </option>
                                                @endforeach
                                            </select>
                                            <div class="validation-error" id="product-error-0">Please select a product</div>
                                        </div>
                                        <div class="col-md-2">
                                            <label class="form-label">Quantity *</label>
                                            <input type="number" name="items[0][quantity]" 
                                                   class="form-control quantity-input" min="0.001" step="0.001" value="1" required
                                                   onchange="calculateItemTotal(0)">
                                            <div class="validation-error" id="quantity-error-0">Please enter a valid quantity</div>
                                        </div>
                                        <div class="col-md-3">
                                            <label class="form-label">Unit Price *</label>
                                            <div class="input-group">
                                                <span class="input-group-text">KES</span>
                                                <input type="number" name="items[0][unit_price]" 
                                                       class="form-control price-input" min="0" step="0.01" required
                                                       onchange="calculateItemTotal(0)">
                                            </div>
                                            <div class="validation-error" id="price-error-0">Please enter a valid price</div>
                                        </div>
                                        <div class="col-md-2">
                                            <label class="form-label">Total with Tax</label>
                                            <input type="text" class="form-control item-total" id="item-total-0" 
                                                   value="KES 0.00" readonly>
                                            <small class="text-muted" id="item-tax-info-0" style="font-size: 11px;"></small>
                                        </div>
                                    </div>
                                    
                                    <div class="row mt-2">
                                        <div class="col-md-12">
                                            <div class="batch-fields" style="display: none;">
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <label class="form-label">Batch Number</label>
                                                        <input type="text" name="items[0][batch_number]" 
                                                               class="form-control" placeholder="Optional">
                                                    </div>
                                                    <div class="col-md-6">
                                                        <label class="form-label">Expiry Date</label>
                                                        <input type="date" name="items[0][expiry_date]" 
                                                               class="form-control" 
                                                               min="{{ date('Y-m-d', strtotime('+1 day')) }}">
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row mt-2">
                                        <div class="col-md-12">
                                            <label class="form-label">Notes (Optional)</label>
                                            <textarea name="items[0][notes]" class="form-control" rows="2" placeholder="Additional notes about this item..."></textarea>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Hidden fields -->
                                <input type="hidden" name="items[0][type]" value="product">
                                <input type="hidden" name="items[0][description]" value="">
                                <input type="hidden" class="tax-rate-input" id="tax-rate-0" value="{{ $defaultTaxRate }}">
                            </div>
                        </div>
                        
                        <!-- Service Items Section (Optional) -->
                        <div class="mt-3 mb-3">
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="showServiceItems" onchange="toggleServiceItems()">
                                <label class="form-check-label" for="showServiceItems">
                                    <strong>Include Service Items</strong>
                                </label>
                            </div>
                            <small class="text-muted">Check this if you need to add service items that won't be added to inventory.</small>
                        </div>
                        
                        <div id="serviceItemsSection" style="display: none;">
                            <h5 class="mb-3"><i class="bx bx-cog"></i> Service Items</h5>
                            <div id="serviceItems"></div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" onclick="addServiceItem()">
                                <i class="bx bx-plus"></i> Add Service Item
                            </button>
                        </div>
                        
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <button type="button" class="btn btn-sm btn-primary" onclick="addItem()">
                                    <i class="bx bx-plus"></i> Add Another Product
                                </button>
                            </div>
                            <div class="col-md-6 text-end">
                                <div class="item-summary">
                                    <h6>Order Summary</h6>
                                    <div class="d-flex justify-content-between">
                                        <span>Subtotal:</span>
                                        <span id="step1-subtotal">KES 0.00</span>
                                    </div>
                                    <div class="d-flex justify-content-between">
                                        <span>Tax:</span>
                                        <span id="step1-tax">KES 0.00</span>
                                    </div>
                                    <div class="d-flex justify-content-between mt-2">
                                        <strong>Grand Total:</strong>
                                        <strong><span id="step1-grand-total">KES 0.00</span></strong>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="wizard-nav">
                            <div>
                                <a href="{{ route('purchases.index') }}" class="btn btn-secondary">
                                    <i class="bx bx-x"></i> Cancel
                                </a>
                            </div>
                            <div>
                                <button type="button" class="btn btn-primary" onclick="nextStep()">
                                    Continue <i class="bx bx-chevron-right"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Step 2: Supplier & Payment -->
                    <div class="wizard-step" id="step2">
                        <h4 class="mb-3"><i class="bx bx-info-circle"></i> Supplier & Payment Details</h4>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Supplier *</label>
                                    <select name="supplier_id" id="supplierSelect" class="form-select" required>
                                        <option value="">Select Supplier</option>
                                        @foreach($suppliers as $supplier)
                                            <option value="{{ $supplier->id }}"
                                                    data-credit-limit="{{ $supplier->credit_limit ?? 0 }}"
                                                    data-current-balance="{{ $supplier->current_balance ?? 0 }}">
                                                {{ $supplier->name }}
                                                @if($supplier->credit_limit > 0)
                                                    (Credit Limit: KES {{ number_format($supplier->credit_limit, 2) }})
                                                @endif
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="alert alert-warning credit-limit-alert mt-2" id="creditLimitAlert" style="display: none;">
                                        <i class="bx bx-alarm"></i>
                                        <small id="creditLimitMessage"></small>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Purchase Date *</label>
                                        <input type="date" name="purchase_date" class="form-control" 
                                               value="{{ date('Y-m-d') }}" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Invoice No</label>
                                        <input type="text" name="invoice_no" class="form-control" 
                                               value="{{ $invoiceNumber }}" readonly>
                                        <small class="text-muted">Auto-generated</small>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Reference Number</label>
                                        <input type="text" name="reference_number" class="form-control" 
                                               placeholder="Supplier reference number">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Expected Delivery</label>
                                        <input type="date" name="expected_delivery_date" class="form-control" 
                                               min="{{ date('Y-m-d') }}">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Payment Status *</label>
                                    <select name="payment_status" id="paymentStatus" class="form-select" required
                                            onchange="togglePaymentFields()">
                                        <option value="pending" selected>Pending</option>
                                        <option value="partial">Partial</option>
                                        <option value="paid">Paid</option>
                                    </select>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Payment Method</label>
                                        <select name="payment_method" class="form-select">
                                            <option value="">Select Method</option>
                                            <option value="cash">Cash</option>
                                            <option value="bank">Bank Transfer</option>
                                            <option value="mpesa">M-Pesa</option>
                                            <option value="cheque">Cheque</option>
                                            <option value="credit">Credit</option>
                                        </select>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label" id="paidAmountLabel">Amount Paid *</label>
                                        <div class="input-group">
                                            <span class="input-group-text">KES</span>
                                            <input type="number" name="paid_amount" id="paidAmount" class="form-control" 
                                                   value="0" min="0" step="0.01" required>
                                        </div>
                                        <small class="text-muted" id="paidAmountHelp">Enter amount paid now</small>
                                    </div>
                                </div>
                                
                                <div class="mb-3" id="creditFields" style="display: none;">
                                    <div class="form-check mb-2">
                                        <input type="checkbox" name="is_credit_purchase" id="isCreditPurchase" 
                                               class="form-check-input" onchange="toggleCreditFields()">
                                        <label class="form-check-label" for="isCreditPurchase">
                                            This is a credit purchase
                                        </label>
                                    </div>
                                    <div class="row mt-2" id="dueDateField" style="display: none;">
                                        <div class="col-md-12">
                                            <label class="form-label">Due Date</label>
                                            <input type="date" name="due_date" class="form-control" 
                                                   min="{{ date('Y-m-d') }}">
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Order Status *</label>
                                    <select name="status" class="form-select" required>
                                        <option value="draft">Draft</option>
                                        <option value="pending" selected>Pending</option>
                                        <option value="completed">Completed</option>
                                    </select>
                                </div>
                                
                                <div class="form-check mb-3">
                                    <input type="checkbox" name="add_to_stock" id="addToStock" 
                                           class="form-check-input" checked>
                                    <label class="form-check-label" for="addToStock">
                                        Add items to inventory stock
                                    </label>
                                    <small class="text-muted d-block">Only product items will be added to stock</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row mt-3">
                            <div class="col-md-12">
                                <label class="form-label">Notes</label>
                                <textarea name="notes" class="form-control" rows="3" placeholder="Any special notes about this purchase..."></textarea>
                            </div>
                        </div>
                        
                        <div class="wizard-nav">
                            <div>
                                <button type="button" class="btn btn-secondary" onclick="prevStep()">
                                    <i class="bx bx-chevron-left"></i> Back
                                </button>
                            </div>
                            <div>
                                <button type="button" class="btn btn-primary" onclick="nextStep()">
                                    Review Order <i class="bx bx-chevron-right"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Step 3: Review & Complete -->
                    <div class="wizard-step" id="step3">
                        <h4 class="mb-3"><i class="bx bx-check-circle"></i> Review & Complete Purchase</h4>
                        
                        <div class="alert alert-info mb-3">
                            <i class="bx bx-info-circle"></i>
                            <small>Please review all details before completing the purchase order.</small>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-8">
                                <!-- Items Summary -->
                                <div class="card mb-3">
                                    <div class="card-header">
                                        <h5 class="mb-0">Order Items</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="table-responsive">
                                            <table class="table summary-table">
                                                <thead>
                                                    <tr>
                                                        <th>Type</th>
                                                        <th>Description</th>
                                                        <th class="text-end">Qty</th>
                                                        <th class="text-end">Unit Price</th>
                                                        <th class="text-end">Tax</th>
                                                        <th class="text-end">Total</th>
                                                    </tr>
                                                </thead>
                                                <tbody id="reviewItemsTable">
                                                    <!-- Items will be populated by JavaScript -->
                                                </tbody>
                                                <tfoot>
                                                    <tr>
                                                        <td colspan="4" class="text-end"><strong>Subtotal:</strong></td>
                                                        <td class="text-end"></td>
                                                        <td class="text-end"><strong id="reviewSubtotal">KES 0.00</strong></td>
                                                    </tr>
                                                    <tr>
                                                        <td colspan="4" class="text-end">Tax:</td>
                                                        <td class="text-end" id="reviewTaxDetails"></td>
                                                        <td class="text-end" id="reviewTax">KES 0.00</td>
                                                    </tr>
                                                    <tr>
                                                        <td colspan="4" class="text-end"><strong>Grand Total:</strong></td>
                                                        <td class="text-end"></td>
                                                        <td class="text-end"><strong id="reviewGrandTotal">KES 0.00</strong></td>
                                                    </tr>
                                                </tfoot>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Supplier & Payment Summary -->
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card mb-3">
                                            <div class="card-header">
                                                <h5 class="mb-0">Supplier Details</h5>
                                            </div>
                                            <div class="card-body">
                                                <p class="mb-1"><strong>Supplier:</strong> <span id="reviewSupplier">Not selected</span></p>
                                                <p class="mb-1"><strong>Purchase Date:</strong> <span id="reviewPurchaseDate">{{ date('Y-m-d') }}</span></p>
                                                <p class="mb-1"><strong>Reference:</strong> <span id="reviewReference">N/A</span></p>
                                                <p class="mb-0"><strong>Delivery Date:</strong> <span id="reviewDeliveryDate">N/A</span></p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card mb-3">
                                            <div class="card-header">
                                                <h5 class="mb-0">Payment Details</h5>
                                            </div>
                                            <div class="card-body">
                                                <p class="mb-1"><strong>Status:</strong> <span id="reviewPaymentStatus">Pending</span></p>
                                                <p class="mb-1"><strong>Method:</strong> <span id="reviewPaymentMethod">N/A</span></p>
                                                <p class="mb-1"><strong>Amount Paid:</strong> <span id="reviewPaidAmount">KES 0.00</span></p>
                                                <p class="mb-0"><strong>Balance Due:</strong> <span id="reviewBalanceDue">KES 0.00</span></p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <!-- Order Summary -->
                                <div class="card mb-3">
                                    <div class="card-header">
                                        <h5 class="mb-0">Order Summary</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Order Status:</span>
                                            <strong id="reviewOrderStatus">Pending</strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Add to Stock:</span>
                                            <strong id="reviewAddToStock">Yes</strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-3">
                                            <span>Credit Purchase:</span>
                                            <strong id="reviewCreditPurchase">No</strong>
                                        </div>
                                        
                                        <div class="alert alert-warning" id="creditWarning" style="display: none;">
                                            <i class="bx bx-alarm"></i>
                                            <small>This is a credit purchase. Make sure supplier has sufficient credit limit.</small>
                                        </div>
                                        
                                        <div class="alert alert-info">
                                            <i class="bx bx-info-circle"></i>
                                            <small>Click "Complete Purchase" to create the purchase order.</small>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Final Actions -->
                                <div class="d-grid gap-2">
                                    <button type="button" class="btn btn-primary" onclick="completePurchase()">
                                        <i class="bx bx-check"></i> Complete Purchase
                                    </button>
                                    <button type="button" class="btn btn-outline-primary" onclick="saveAsDraft()">
                                        <i class="bx bx-save"></i> Save as Draft
                                    </button>
                                    <button type="button" class="btn btn-outline-secondary" onclick="prevStep()">
                                        <i class="bx bx-edit"></i> Edit Details
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
let itemCount = 1;
let serviceItemCount = 0;
let currentStep = 1;
let defaultTaxRate = {{ $defaultTaxRate }};
let itemTaxRates = {}; // Store tax rates for each item index

// Helper function for alerts
function showAlert(message, type = 'info') {
    if (typeof Swal !== 'undefined') {
        const config = {
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 3000,
            timerProgressBar: true,
            didOpen: (toast) => {
                toast.addEventListener('mouseenter', Swal.stopTimer);
                toast.addEventListener('mouseleave', Swal.resumeTimer);
            }
        };
        
        switch(type) {
            case 'success':
                Swal.fire({
                    ...config,
                    icon: 'success',
                    title: message
                });
                break;
            case 'error':
                Swal.fire({
                    ...config,
                    icon: 'error',
                    title: message
                });
                break;
            case 'warning':
                Swal.fire({
                    ...config,
                    icon: 'warning',
                    title: message
                });
                break;
            case 'info':
            default:
                Swal.fire({
                    ...config,
                    icon: 'info',
                    title: message
                });
                break;
        }
    } else {
        const alertClass = type === 'error' ? 'danger' : type;
        const alertDiv = $(`
            <div class="alert alert-${alertClass} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `);
        $('.page-content').prepend(alertDiv);
        setTimeout(() => alertDiv.alert('close'), 5000);
    }
}

// Wizard Navigation
function nextStep() {
    if (currentStep === 1 && !validateStep1()) {
        return false;
    }
    
    if (currentStep === 2 && !validateStep2()) {
        return false;
    }
    
    if (currentStep < 3) {
        document.getElementById(`step${currentStep}`).classList.remove('active');
        document.getElementById(`step${currentStep}-circle`).classList.remove('active');
        document.querySelector(`.progress-step:nth-child(${currentStep}) .step-label`).classList.remove('active');
        
        currentStep++;
        
        document.getElementById(`step${currentStep}`).classList.add('active');
        document.getElementById(`step${currentStep}-circle`).classList.add('active');
        document.querySelector(`.progress-step:nth-child(${currentStep}) .step-label`).classList.add('active');
        
        for (let i = 1; i < currentStep; i++) {
            document.getElementById(`step${i}-circle`).classList.add('completed');
        }
        
        if (currentStep === 3) {
            updateReviewSection();
        }
    }
}

function prevStep() {
    if (currentStep > 1) {
        document.getElementById(`step${currentStep}`).classList.remove('active');
        document.getElementById(`step${currentStep}-circle`).classList.remove('active');
        document.querySelector(`.progress-step:nth-child(${currentStep}) .step-label`).classList.remove('active');
        
        currentStep--;
        
        document.getElementById(`step${currentStep}`).classList.add('active');
        document.getElementById(`step${currentStep}-circle`).classList.add('active');
        document.querySelector(`.progress-step:nth-child(${currentStep}) .step-label`).classList.add('active');
        
        for (let i = 1; i <= 3; i++) {
            const circle = document.getElementById(`step${i}-circle`);
            if (i < currentStep) {
                circle.classList.add('completed');
                circle.classList.remove('active');
            } else if (i === currentStep) {
                circle.classList.add('active');
                circle.classList.remove('completed');
            } else {
                circle.classList.remove('active', 'completed');
            }
        }
    }
}

// Step 1: Product Items Functions
function addItem() {
    const html = `
        <div class="item-row" data-index="${itemCount}">
            <span class="badge bg-secondary item-type-badge">Product</span>
            
            <button type="button" class="btn btn-sm btn-danger remove-btn" onclick="removeItem(this)" title="Remove item">
                <i class="bx bx-trash"></i>
            </button>
            
            <div class="product-fields">
                <div class="row">
                    <div class="col-md-5">
                        <label class="form-label">Product *</label>
                        <select name="items[${itemCount}][product_id]" class="form-select product-select" 
                                onchange="onProductChange(this, ${itemCount})" required>
                            <option value="">Select Product</option>
                            @foreach($products as $product)
                                <option value="{{ $product->id }}" 
                                        data-cost="{{ $product->cost_price }}"
                                        data-stock="{{ $product->stock }}"
                                        data-has-expiry="{{ $product->has_expiry }}"
                                        data-track-batches="{{ $product->track_batches }}"
                                        data-tax-rate="{{ $product->tax_rate ?? $defaultTaxRate }}">
                                    {{ $product->name }} ({{ $product->sku }}) - Stock: {{ $product->stock }}
                                    @if($product->tax_rate)
                                        (Tax: {{ $product->tax_rate }}%)
                                    @endif
                                </option>
                            @endforeach
                        </select>
                        <div class="validation-error" id="product-error-${itemCount}">Please select a product</div>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Quantity *</label>
                        <input type="number" name="items[${itemCount}][quantity]" 
                               class="form-control quantity-input" min="0.001" step="0.001" value="1" required
                               onchange="calculateItemTotal(${itemCount})">
                        <div class="validation-error" id="quantity-error-${itemCount}">Please enter a valid quantity</div>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Unit Price *</label>
                        <div class="input-group">
                            <span class="input-group-text">KES</span>
                            <input type="number" name="items[${itemCount}][unit_price]" 
                                   class="form-control price-input" min="0" step="0.01" required
                                   onchange="calculateItemTotal(${itemCount})">
                        </div>
                        <div class="validation-error" id="price-error-${itemCount}">Please enter a valid price</div>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Total with Tax</label>
                        <input type="text" class="form-control item-total" id="item-total-${itemCount}" 
                               value="KES 0.00" readonly>
                        <small class="text-muted" id="item-tax-info-${itemCount}" style="font-size: 11px;"></small>
                    </div>
                </div>
                
                <div class="row mt-2">
                    <div class="col-md-12">
                        <div class="batch-fields" style="display: none;">
                            <div class="row">
                                <div class="col-md-6">
                                    <label class="form-label">Batch Number</label>
                                    <input type="text" name="items[${itemCount}][batch_number]" 
                                           class="form-control" placeholder="Optional">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Expiry Date</label>
                                    <input type="date" name="items[${itemCount}][expiry_date]" 
                                           class="form-control" 
                                           min="{{ date('Y-m-d', strtotime('+1 day')) }}">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row mt-2">
                    <div class="col-md-12">
                        <label class="form-label">Notes (Optional)</label>
                        <textarea name="items[${itemCount}][notes]" class="form-control" rows="2" placeholder="Additional notes..."></textarea>
                    </div>
                </div>
            </div>
            
            <!-- Hidden fields -->
            <input type="hidden" name="items[${itemCount}][type]" value="product">
            <input type="hidden" name="items[${itemCount}][description]" value="">
            <input type="hidden" class="tax-rate-input" id="tax-rate-${itemCount}" value="${defaultTaxRate}">
        </div>
    `;
    
    $('#purchaseItems').append(html);
    
    // Initialize Select2 if available
    if (typeof $.fn.select2 !== 'undefined') {
        $(`[name="items[${itemCount}][product_id]"]`).select2({
            theme: 'bootstrap4',
            placeholder: 'Select product'
        });
    }
    
    itemCount++;
    calculateTotals();
}

function addServiceItem() {
    const html = `
        <div class="item-row service-item" data-index="${serviceItemCount}" style="background: #e8f4fd; border-color: #b6d4fe;">
            <span class="badge bg-info item-type-badge">Service</span>
            
            <button type="button" class="btn btn-sm btn-danger remove-btn" onclick="removeServiceItem(this)" title="Remove service">
                <i class="bx bx-trash"></i>
            </button>
            
            <div class="row">
                <div class="col-md-4">
                    <label class="form-label">Service Description *</label>
                    <input type="text" name="items[${itemCount + serviceItemCount}][description]" 
                           class="form-control service-description" required placeholder="Enter service description">
                    <div class="validation-error" id="service-description-error-${serviceItemCount}">Please enter service description</div>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Quantity *</label>
                    <input type="number" name="items[${itemCount + serviceItemCount}][quantity]" 
                           class="form-control quantity-input" min="0.001" step="0.001" value="1" required
                           onchange="calculateItemTotal(${itemCount + serviceItemCount})">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Unit Price *</label>
                    <div class="input-group">
                        <span class="input-group-text">KES</span>
                        <input type="number" name="items[${itemCount + serviceItemCount}][unit_price]" 
                               class="form-control price-input" min="0" step="0.01" required
                               onchange="calculateItemTotal(${itemCount + serviceItemCount})">
                    </div>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Total with Tax</label>
                    <input type="text" class="form-control item-total" id="item-total-${itemCount + serviceItemCount}" 
                           value="KES 0.00" readonly>
                    <small class="text-muted" id="item-tax-info-${itemCount + serviceItemCount}" style="font-size: 11px;">
                        Default tax: {{ $defaultTaxRate }}%
                    </small>
                </div>
            </div>
            
            <div class="row mt-2">
                <div class="col-md-12">
                    <label class="form-label">Notes (Optional)</label>
                    <textarea name="items[${itemCount + serviceItemCount}][notes]" class="form-control" rows="2" placeholder="Service notes..."></textarea>
                </div>
            </div>
            
            <!-- Hidden fields -->
            <input type="hidden" name="items[${itemCount + serviceItemCount}][type]" value="service">
            <input type="hidden" class="tax-rate-input" id="tax-rate-${itemCount + serviceItemCount}" value="{{ $defaultTaxRate }}">
        </div>
    `;
    
    $('#serviceItems').append(html);
    serviceItemCount++;
    calculateTotals();
}

function removeItem(button) {
    const items = $('#purchaseItems .item-row').not('.service-item');
    
    if (items.length > 1) {
        const index = $(button).closest('.item-row').data('index');
        delete itemTaxRates[index];
        $(button).closest('.item-row').remove();
        calculateTotals();
    } else {
        showAlert('At least one product item is required', 'warning');
    }
}

function removeServiceItem(button) {
    const serviceItems = $('#serviceItems .service-item');
    
    if (serviceItems.length > 0) {
        const index = $(button).closest('.item-row').data('index');
        delete itemTaxRates[index];
        $(button).closest('.item-row').remove();
        serviceItemCount--;
        calculateTotals();
    }
}

function toggleServiceItems() {
    const showServiceItems = $('#showServiceItems').is(':checked');
    if (showServiceItems) {
        $('#serviceItemsSection').show();
    } else {
        $('#serviceItemsSection').hide();
        $('#serviceItems').empty();
        serviceItemCount = 0;
        calculateTotals();
    }
}

function onProductChange(select, index) {
    const selectedOption = $(select).find('option:selected');
    const hasExpiry = selectedOption.data('has-expiry') === '1' || selectedOption.data('has-expiry') === 1;
    const trackBatches = selectedOption.data('track-batches') === '1' || selectedOption.data('track-batches') === 1;
    const costPrice = selectedOption.data('cost') || 0;
    const taxRate = selectedOption.data('tax-rate') || defaultTaxRate;
    
    // Store the tax rate for this item
    itemTaxRates[index] = taxRate;
    $(`#tax-rate-${index}`).val(taxRate);
    
    // Update tax info display
    $(`#item-tax-info-${index}`).text(`Tax rate: ${taxRate}%`);
    
    // Set unit price to cost price if not set or zero
    const unitPriceInput = $(select).closest('.item-row').find('.price-input');
    if (!unitPriceInput.val() || unitPriceInput.val() === '0' || unitPriceInput.val() === '0.00') {
        unitPriceInput.val(costPrice);
    }
    
    // Show/hide batch fields
    const batchFields = $(select).closest('.item-row').find('.batch-fields');
    if (hasExpiry || trackBatches) {
        batchFields.show();
    } else {
        batchFields.hide();
    }
    
    calculateItemTotal(index);
}

function calculateItemTotal(index) {
    const row = $(`.item-row[data-index="${index}"]`);
    const quantity = parseFloat(row.find('.quantity-input').val()) || 0;
    const price = parseFloat(row.find('.price-input').val()) || 0;
    const taxRate = parseFloat(row.find('.tax-rate-input').val()) || defaultTaxRate;
    
    const subtotal = quantity * price;
    const taxAmount = subtotal * (taxRate / 100);
    const totalWithTax = subtotal + taxAmount;
    
    $(`#item-total-${index}`).val('KES ' + totalWithTax.toFixed(2));
    
    // Update tax info for product items
    if (row.find('.product-select').length > 0) {
        $(`#item-tax-info-${index}`).text(`Tax: ${taxRate}% (KES ${taxAmount.toFixed(2)})`);
    }
    
    calculateTotals();
}

function calculateTotals() {
    let subtotal = 0;
    let totalTax = 0;
    let taxBreakdown = {};
    
    $('.item-row').each(function() {
        const index = $(this).data('index');
        const quantity = parseFloat($(this).find('.quantity-input').val()) || 0;
        const price = parseFloat($(this).find('.price-input').val()) || 0;
        const taxRate = parseFloat($(this).find('.tax-rate-input').val()) || defaultTaxRate;
        
        const itemSubtotal = quantity * price;
        const itemTax = itemSubtotal * (taxRate / 100);
        
        subtotal += itemSubtotal;
        totalTax += itemTax;
        
        // Group tax by rate for breakdown
        if (!taxBreakdown[taxRate]) {
            taxBreakdown[taxRate] = 0;
        }
        taxBreakdown[taxRate] += itemTax;
    });
    
    const grandTotal = subtotal + totalTax;
    
    $('#step1-subtotal').text('KES ' + subtotal.toFixed(2));
    $('#step1-tax').text('KES ' + totalTax.toFixed(2));
    $('#step1-grand-total').text('KES ' + grandTotal.toFixed(2));
    
    // Update paid amount if payment status is "paid"
    if ($('#paymentStatus').val() === 'paid') {
        $('#paidAmount').val(grandTotal.toFixed(2)).trigger('change');
    }
    
    // Update credit limit check
    checkCreditLimit();
}

// Step 1 Validation
function validateStep1() {
    let isValid = true;
    
    $('.validation-error').hide();
    $('.form-select, .form-control').removeClass('is-invalid');
    
    // Validate product items
    $('.item-row').not('.service-item').each(function(index) {
        const productId = $(this).find('select[name*="product_id"]').val();
        const quantity = $(this).find('input[name*="quantity"]').val();
        const unitPrice = $(this).find('input[name*="unit_price"]').val();
        
        if (!productId) {
            $(`#product-error-${index}`).show();
            $(this).find('select[name*="product_id"]').addClass('is-invalid');
            isValid = false;
        }
        
        if (!quantity || parseFloat(quantity) <= 0) {
            $(`#quantity-error-${index}`).show();
            $(this).find('input[name*="quantity"]').addClass('is-invalid');
            isValid = false;
        }
        
        if (!unitPrice || parseFloat(unitPrice) < 0) {
            $(`#price-error-${index}`).show();
            $(this).find('input[name*="unit_price"]').addClass('is-invalid');
            isValid = false;
        }
    });
    
    // Validate service items
    $('.service-item').each(function(index) {
        const description = $(this).find('.service-description').val();
        const quantity = $(this).find('input[name*="quantity"]').val();
        const unitPrice = $(this).find('input[name*="unit_price"]').val();
        
        if (!description || description.trim() === '') {
            $(`#service-description-error-${index}`).show();
            $(this).find('.service-description').addClass('is-invalid');
            isValid = false;
        }
        
        if (!quantity || parseFloat(quantity) <= 0) {
            $(this).find('input[name*="quantity"]').addClass('is-invalid');
            isValid = false;
        }
        
        if (!unitPrice || parseFloat(unitPrice) < 0) {
            $(this).find('input[name*="unit_price"]').addClass('is-invalid');
            isValid = false;
        }
    });
    
    if (!isValid) {
        showAlert('Please fix the errors in the items section', 'error');
    }
    
    return isValid;
}

// Step 2: Payment Functions
function togglePaymentFields() {
    const paymentStatus = $('#paymentStatus').val();
    const paidAmountInput = $('#paidAmount');
    const paidAmountLabel = $('#paidAmountLabel');
    const paidAmountHelp = $('#paidAmountHelp');
    const creditFields = $('#creditFields');
    
    calculateTotals();
    const grandTotal = parseFloat($('#step1-grand-total').text().replace('KES ', '')) || 0;
    
    if (paymentStatus === 'paid') {
        paidAmountInput.val(grandTotal.toFixed(2)).prop('readonly', true).prop('disabled', false);
        paidAmountLabel.text('Amount Paid *');
        paidAmountHelp.text('Full amount will be marked as paid');
        creditFields.hide();
        $('#isCreditPurchase').prop('checked', false).trigger('change');
    } else if (paymentStatus === 'partial') {
        paidAmountInput.val('0').prop('readonly', false).prop('disabled', false);
        paidAmountLabel.text('Partial Payment *');
        paidAmountHelp.text('Enter partial amount paid now');
        creditFields.show();
    } else {
        paidAmountInput.val('0').prop('readonly', false).prop('disabled', false);
        paidAmountLabel.text('Amount Paid *');
        paidAmountHelp.text('Enter amount paid now (if any)');
        creditFields.show();
    }
    
    validatePaidAmount();
}

function validatePaidAmount() {
    const paidAmount = parseFloat($('#paidAmount').val()) || 0;
    const grandTotal = parseFloat($('#step1-grand-total').text().replace('KES ', '')) || 0;
    const paymentStatus = $('#paymentStatus').val();
    
    $('#paidAmount').removeClass('is-invalid');
    $('#paidAmountHelp').removeClass('text-danger');
    
    if (paymentStatus === 'paid') {
        if (paidAmount < grandTotal) {
            $('#paidAmount').addClass('is-invalid');
            $('#paidAmountHelp').addClass('text-danger').text('Paid amount must equal grand total for "Paid" status');
            return false;
        }
    }
    
    if (paidAmount > grandTotal) {
        $('#paidAmount').addClass('is-invalid');
        $('#paidAmountHelp').addClass('text-danger').text('Paid amount cannot exceed grand total');
        return false;
    }
    
    return true;
}

function checkCreditLimit() {
    const supplierSelect = $('#supplierSelect');
    const selectedOption = supplierSelect.find('option:selected');
    const creditLimit = parseFloat(selectedOption.data('credit-limit')) || 0;
    const currentBalance = parseFloat(selectedOption.data('current-balance')) || 0;
    
    if (creditLimit > 0) {
        const availableCredit = creditLimit - currentBalance;
        const grandTotal = parseFloat($('#step1-grand-total').text().replace('KES ', '')) || 0;
        
        if (grandTotal > availableCredit) {
            const exceedAmount = grandTotal - availableCredit;
            $('#creditLimitMessage').html(
                `⚠️ Warning: Purchase amount (KES ${grandTotal.toFixed(2)}) exceeds available credit by KES ${exceedAmount.toFixed(2)}.<br>
                Available credit: KES ${availableCredit.toFixed(2)}`
            );
            $('#creditLimitAlert').removeClass('alert-info').addClass('alert-warning').show();
        } else {
            $('#creditLimitMessage').html(
                `Available credit: KES ${availableCredit.toFixed(2)}<br>
                Purchase amount: KES ${grandTotal.toFixed(2)}`
            );
            $('#creditLimitAlert').removeClass('alert-warning').addClass('alert-info').show();
        }
    } else {
        $('#creditLimitAlert').hide();
    }
}

function toggleCreditFields() {
    const isCreditPurchase = $('#isCreditPurchase').is(':checked');
    if (isCreditPurchase) {
        $('#dueDateField').show();
    } else {
        $('#dueDateField').hide();
    }
}

// Step 2 Validation
function validateStep2() {
    let isValid = true;
    
    $('.is-invalid').removeClass('is-invalid');
    $('.text-danger').removeClass('text-danger');
    
    const supplierId = $('#supplierSelect').val();
    if (!supplierId) {
        $('#supplierSelect').addClass('is-invalid');
        showAlert('Please select a supplier', 'error');
        isValid = false;
    }
    
    const purchaseDate = $('input[name="purchase_date"]').val();
    if (!purchaseDate) {
        $('input[name="purchase_date"]').addClass('is-invalid');
        showAlert('Please select a purchase date', 'error');
        isValid = false;
    }
    
    const paidAmount = parseFloat($('#paidAmount').val()) || 0;
    const grandTotal = parseFloat($('#step1-grand-total').text().replace('KES ', '')) || 0;
    const paymentStatus = $('#paymentStatus').val();
    
    if (paymentStatus === 'paid') {
        if (paidAmount < grandTotal) {
            $('#paidAmount').addClass('is-invalid');
            $('#paidAmountHelp').addClass('text-danger').text('Paid amount must equal grand total when payment status is "Paid"');
            showAlert('Paid amount must equal grand total when payment status is "Paid"', 'error');
            isValid = false;
        }
    }
    
    if (paidAmount > grandTotal) {
        $('#paidAmount').addClass('is-invalid');
        $('#paidAmountHelp').addClass('text-danger').text('Paid amount cannot exceed grand total');
        showAlert('Paid amount cannot exceed grand total', 'error');
        isValid = false;
    }
    
    const isCreditPurchase = $('#isCreditPurchase').is(':checked');
    if (isCreditPurchase) {
        const dueDate = $('input[name="due_date"]').val();
        if (!dueDate) {
            $('input[name="due_date"]').addClass('is-invalid');
            showAlert('Please select a due date for credit purchase', 'error');
            isValid = false;
        }
    }
    
    return isValid;
}

// Step 3: Review Functions
function updateReviewSection() {
    const itemsTable = $('#reviewItemsTable');
    itemsTable.empty();
    
    let subtotal = 0;
    let totalTax = 0;
    let taxBreakdown = {};
    
    $('.item-row').each(function(index) {
        const row = $(this);
        let description = '';
        let type = '';
        let taxRate = 0;
        
        if (row.hasClass('service-item')) {
            description = row.find('.service-description').val();
            type = 'Service';
            taxRate = parseFloat(row.find('.tax-rate-input').val()) || defaultTaxRate;
        } else {
            const productSelect = row.find('select[name*="product_id"]');
            description = productSelect.find('option:selected').text().split(' - ')[0] || 'Unknown Product';
            type = 'Product';
            taxRate = parseFloat(row.find('.tax-rate-input').val()) || defaultTaxRate;
        }
        
        const quantity = parseFloat(row.find('.quantity-input').val()) || 0;
        const unitPrice = parseFloat(row.find('.price-input').val()) || 0;
        const itemSubtotal = quantity * unitPrice;
        const itemTax = itemSubtotal * (taxRate / 100);
        const totalWithTax = itemSubtotal + itemTax;
        
        subtotal += itemSubtotal;
        totalTax += itemTax;
        
        // Group tax by rate
        if (!taxBreakdown[taxRate]) {
            taxBreakdown[taxRate] = 0;
        }
        taxBreakdown[taxRate] += itemTax;
        
        itemsTable.append(`
            <tr>
                <td>${type}</td>
                <td>${description}</td>
                <td class="text-end">${quantity.toFixed(3)}</td>
                <td class="text-end">KES ${unitPrice.toFixed(2)}</td>
                <td class="text-end">${taxRate}%</td>
                <td class="text-end">KES ${totalWithTax.toFixed(2)}</td>
            </tr>
        `);
    });
    
    const grandTotal = subtotal + totalTax;
    
    $('#reviewSubtotal').text('KES ' + subtotal.toFixed(2));
    $('#reviewTax').text('KES ' + totalTax.toFixed(2));
    $('#reviewGrandTotal').text('KES ' + grandTotal.toFixed(2));
    
    // Update tax breakdown
    let taxDetails = '';
    Object.keys(taxBreakdown).forEach(rate => {
        if (taxBreakdown[rate] > 0) {
            taxDetails += `${rate}%: KES ${taxBreakdown[rate].toFixed(2)}<br>`;
        }
    });
    $('#reviewTaxDetails').html(taxDetails || '-');
    
    // Update supplier details
    const supplierSelect = $('#supplierSelect');
    const supplierName = supplierSelect.find('option:selected').text().split(' (')[0] || 'Not selected';
    $('#reviewSupplier').text(supplierName);
    
    $('#reviewPurchaseDate').text($('input[name="purchase_date"]').val());
    $('#reviewReference').text($('input[name="reference_number"]').val() || 'N/A');
    $('#reviewDeliveryDate').text($('input[name="expected_delivery_date"]').val() || 'N/A');
    
    // Update payment details
    $('#reviewPaymentStatus').text($('#paymentStatus').val().charAt(0).toUpperCase() + $('#paymentStatus').val().slice(1));
    $('#reviewPaymentMethod').text($('select[name="payment_method"]').val() || 'N/A');
    $('#reviewPaidAmount').text('KES ' + ($('#paidAmount').val() || '0.00'));
    $('#reviewBalanceDue').text('KES ' + (grandTotal - parseFloat($('#paidAmount').val() || 0)).toFixed(2));
    
    // Update order summary
    $('#reviewOrderStatus').text($('select[name="status"] option:selected').text());
    $('#reviewAddToStock').text($('#addToStock').is(':checked') ? 'Yes' : 'No');
    $('#reviewCreditPurchase').text($('#isCreditPurchase').is(':checked') ? 'Yes' : 'No');
    
    if ($('#isCreditPurchase').is(':checked')) {
        $('#creditWarning').show();
    } else {
        $('#creditWarning').hide();
    }
}

function completePurchase() {
    if ($('select[name="status"]').val() === 'draft') {
        $('select[name="status"]').val('pending');
    }
    
    if (typeof Swal !== 'undefined') {
        Swal.fire({
            title: 'Creating Purchase Order...',
            text: 'Please wait while we process your request',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
    } else {
        showAlert('Creating purchase order...', 'info');
    }
    
    setTimeout(() => {
        $('#createPurchaseForm').submit();
    }, 500);
}

function saveAsDraft() {
    $('select[name="status"]').val('draft');
    
    if (typeof Swal !== 'undefined') {
        Swal.fire({
            title: 'Saving as Draft...',
            text: 'Please wait while we save your purchase',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
    } else {
        showAlert('Saving as draft...', 'info');
    }
    
    setTimeout(() => {
        $('#createPurchaseForm').submit();
    }, 500);
}

// Initialize on page load
$(document).ready(function() {
    if (typeof $.fn.select2 !== 'undefined') {
        $('#supplierSelect').select2({
            theme: 'bootstrap4',
            placeholder: 'Select supplier'
        });
        
        $('.product-select').select2({
            theme: 'bootstrap4',
            placeholder: 'Select product'
        });
    }
    
    calculateTotals();
    togglePaymentFields();
    
    $('#supplierSelect').on('change', function() {
        checkCreditLimit();
    });
    
    $('#paymentStatus').on('change', togglePaymentFields);
    
    $('#paidAmount').on('change keyup', function() {
        const paidAmount = parseFloat($(this).val()) || 0;
        const grandTotal = parseFloat($('#step1-grand-total').text().replace('KES ', '')) || 0;
        
        validatePaidAmount();
        
        if (paidAmount >= grandTotal) {
            if ($('#paymentStatus').val() !== 'paid') {
                $('#paymentStatus').val('paid');
                togglePaymentFields();
            }
        } else if (paidAmount > 0) {
            if ($('#paymentStatus').val() !== 'partial') {
                $('#paymentStatus').val('partial');
                togglePaymentFields();
            }
        } else {
            if ($('#paymentStatus').val() !== 'pending') {
                $('#paymentStatus').val('pending');
                togglePaymentFields();
            }
        }
    });
    
    $('#isCreditPurchase').on('change', toggleCreditFields);
    
    const today = new Date().toISOString().split('T')[0];
    $('input[name="expected_delivery_date"]').val(today);
    
    const dueDate = new Date();
    dueDate.setDate(dueDate.getDate() + 30);
    $('input[name="due_date"]').val(dueDate.toISOString().split('T')[0]);
    
    toggleCreditFields();
    
    $(document).on('change keyup', '.quantity-input, .price-input, .service-description', function() {
        const row = $(this).closest('.item-row');
        const index = row.data('index');
        calculateItemTotal(index);
    });
    
    $(document).on('change', '.product-select', function() {
        const row = $(this).closest('.item-row');
        const index = row.data('index');
        onProductChange(this, index);
    });
});
</script>
@endsection