@extends('layouts.app')

@section('style')
    <style>
        /* Syndron Theme Statistics - Mobile Responsive */
        .stat-card-syndron {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            color: white;
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.25);
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
            border: none;
            min-height: 150px;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }
        
        @media (max-width: 768px) {
            .stat-card-syndron {
                padding: 15px;
                margin-bottom: 10px;
                min-height: 130px;
            }
            
            .stat-value-syndron {
                font-size: 28px !important;
            }
            
            .stat-label-syndron {
                font-size: 13px !important;
            }
            
            .stat-change-syndron {
                font-size: 11px !important;
                padding: 3px 8px !important;
            }
        }
        
        @media (max-width: 576px) {
            .stat-card-syndron {
                min-height: 120px;
                padding: 12px;
            }
            
            .stat-value-syndron {
                font-size: 24px !important;
            }
            
            .stat-icon-syndron {
                font-size: 40px !important;
                top: 15px !important;
                right: 15px !important;
            }
        }
        
        .stat-card-syndron:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(102, 126, 234, 0.4);
        }
        
        .stat-card-syndron::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0.05) 100%);
            border-radius: 15px;
        }
        
        .stat-card-syndron.secondary {
            background: linear-gradient(135deg, #4F46E5 0%, #7C3AED 100%);
            box-shadow: 0 10px 20px rgba(79, 70, 229, 0.25);
        }
        
        .stat-card-syndron.secondary:hover {
            box-shadow: 0 15px 30px rgba(79, 70, 229, 0.4);
        }
        
        .stat-card-syndron.success {
            background: linear-gradient(135deg, #10B981 0%, #059669 100%);
            box-shadow: 0 10px 20px rgba(16, 185, 129, 0.25);
        }
        
        .stat-card-syndron.success:hover {
            box-shadow: 0 15px 30px rgba(16, 185, 129, 0.4);
        }
        
        .stat-card-syndron.danger {
            background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
            box-shadow: 0 10px 20px rgba(239, 68, 68, 0.25);
        }
        
        .stat-card-syndron.danger:hover {
            box-shadow: 0 15px 30px rgba(239, 68, 68, 0.4);
        }
        
        .stat-card-syndron.warning {
            background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
            box-shadow: 0 10px 20px rgba(245, 158, 11, 0.25);
        }
        
        .stat-card-syndron.warning:hover {
            box-shadow: 0 15px 30px rgba(245, 158, 11, 0.4);
        }
        
        .stat-card-syndron.info {
            background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
            box-shadow: 0 10px 20px rgba(59, 130, 246, 0.25);
        }
        
        .stat-card-syndron.info:hover {
            box-shadow: 0 15px 30px rgba(59, 130, 246, 0.4);
        }
        
        .stat-icon-syndron {
            position: absolute;
            top: 20px;
            right: 20px;
            font-size: 50px;
            opacity: 0.2;
            color: white;
            transition: all 0.4s ease;
        }
        
        .stat-card-syndron:hover .stat-icon-syndron {
            transform: scale(1.1);
            opacity: 0.3;
        }
        
        .stat-content-syndron {
            position: relative;
            z-index: 2;
        }
        
        .stat-value-syndron {
            font-size: 36px;
            font-weight: 800;
            margin-bottom: 5px;
            letter-spacing: -0.5px;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .stat-label-syndron {
            font-size: 14px;
            font-weight: 500;
            opacity: 0.9;
            letter-spacing: 0.5px;
        }
        
        .stat-change-syndron {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            font-size: 12px;
            font-weight: 600;
            padding: 4px 10px;
            border-radius: 20px;
            background: rgba(255,255,255,0.2);
            margin-top: 8px;
            backdrop-filter: blur(10px);
        }
        
        /* Movement cards - Mobile Responsive */
        .movement-card-syndron {
            border: none;
            border-radius: 12px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            background: white;
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            position: relative;
            overflow: hidden;
        }
        
        @media (max-width: 768px) {
            .movement-card-syndron {
                padding: 15px !important;
                margin-bottom: 10px;
            }
            
            .movement-card-syndron .d-flex {
                flex-direction: column;
            }
            
            .movement-card-syndron .text-end {
                text-align: left !important;
                margin-top: 15px;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            
            .movement-type-badge-syndron {
                padding: 4px 12px !important;
                font-size: 11px !important;
            }
            
            .movement-details {
                font-size: 14px;
            }
        }
        
        .movement-card-syndron::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 5px;
            background: linear-gradient(to bottom, #667eea, #764ba2);
        }
        
        .movement-card-syndron.movement-in::before {
            background: linear-gradient(to bottom, #10B981, #059669);
        }
        
        .movement-card-syndron.movement-out::before {
            background: linear-gradient(to bottom, #EF4444, #DC2626);
        }
        
        .movement-card-syndron:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.15);
        }
        
        .movement-type-badge-syndron {
            padding: 6px 15px;
            border-radius: 25px;
            font-size: 12px;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border: none;
        }
        
        /* Filter section - Mobile Responsive */
        .filter-section-syndron {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            color: white;
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.25);
        }
        
        @media (max-width: 768px) {
            .filter-section-syndron {
                padding: 15px;
                margin-bottom: 15px;
            }
            
            .filter-section-syndron .row > div {
                margin-bottom: 15px;
            }
            
            .filter-section-syndron .d-flex {
                flex-direction: column;
                gap: 10px;
            }
            
            .filter-section-syndron .w-50 {
                width: 100% !important;
            }
            
            .input-group input {
                font-size: 14px;
                padding: 8px;
            }
        }
        
        /* Chart container - Mobile Responsive */
        .chart-container-syndron {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 8px 20px rgba(0,0,0,0.08);
            border: 1px solid rgba(102, 126, 234, 0.1);
            transition: all 0.3s ease;
        }
        
        @media (max-width: 768px) {
            .chart-container-syndron {
                padding: 15px;
                margin-bottom: 15px;
                border-radius: 12px;
            }
            
            .chart-wrapper {
                overflow-x: auto;
            }
            
            .chart-summary {
                flex-wrap: wrap;
                gap: 15px;
            }
            
            .summary-item {
                flex: 0 0 calc(50% - 15px);
                text-align: center;
            }
        }
        
        @media (max-width: 576px) {
            .chart-container-syndron {
                padding: 12px;
                margin-bottom: 12px;
            }
            
            .chart-title {
                flex-direction: column;
                align-items: flex-start !important;
                gap: 10px;
            }
            
            .chart-controls {
                flex-direction: column;
                gap: 10px;
            }
            
            .chart-controls .d-flex {
                flex-wrap: wrap;
                gap: 10px;
            }
            
            .summary-item {
                flex: 0 0 100%;
            }
        }
        
        /* Quick stats - Mobile Responsive */
        .quick-stats-syndron {
            background: linear-gradient(135deg, #f0f4ff 0%, #f9fafb 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border-left: 5px solid #667eea;
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.1);
        }
        
        @media (max-width: 768px) {
            .quick-stats-syndron {
                padding: 15px;
            }
            
            .quick-stats-syndron .display-5 {
                font-size: 24px !important;
            }
        }
        
        /* Timeline - Mobile Responsive */
        .timeline-syndron {
            position: relative;
            padding-left: 30px;
        }
        
        @media (max-width: 768px) {
            .timeline-syndron {
                padding-left: 25px;
            }
            
            .timeline-item-syndron::before {
                left: -22px !important;
                width: 12px !important;
                height: 12px !important;
            }
        }
        
        /* Export buttons - Mobile Responsive */
        .export-buttons-syndron .btn {
            border-radius: 10px;
            padding: 10px 15px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            border: none;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            font-size: 13px;
        }
        
        @media (max-width: 768px) {
            .export-buttons-syndron {
                flex-direction: column;
                gap: 10px;
                margin-top: 15px;
            }
            
            .export-buttons-syndron .btn {
                width: 100%;
                justify-content: center;
                padding: 12px;
            }
            
            .ms-auto {
                margin-left: 0 !important;
                width: 100%;
            }
        }
        
        /* Product header - Mobile Responsive */
        .product-header-syndron {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            color: white;
            box-shadow: 0 15px 30px rgba(102, 126, 234, 0.25);
            position: relative;
            overflow: hidden;
        }
        
        @media (max-width: 768px) {
            .product-header-syndron {
                padding: 15px;
                margin-bottom: 15px;
            }
            
            .product-header-syndron .d-flex {
                flex-direction: column;
                align-items: flex-start !important;
            }
            
            .product-header-syndron .row.g-3 {
                margin-top: 15px;
            }
            
            .product-header-syndron .col-sm-6 {
                margin-bottom: 10px;
            }
            
            .product-header-syndron .text-end {
                text-align: left !important;
                margin-top: 15px;
            }
            
            .display-6 {
                font-size: 28px !important;
            }
        }
        
        /* Pagination - Mobile Responsive */
        @media (max-width: 768px) {
            .pagination-container .d-flex {
                flex-direction: column;
                gap: 15px;
                align-items: center !important;
            }
            
            .pagination {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .page-item .page-link {
                padding: 6px 10px;
                font-size: 14px;
            }
        }
        
        /* Modal - Mobile Responsive */
        @media (max-width: 768px) {
            .modal-dialog {
                margin: 10px;
            }
            
            .modal-content {
                border-radius: 12px;
            }
            
            .modal-body {
                padding: 15px;
            }
        }
        
        /* Breadcrumb - Mobile Responsive */
        @media (max-width: 768px) {
            .page-breadcrumb {
                flex-direction: column;
                align-items: flex-start !important;
                gap: 10px;
            }
            
            .breadcrumb-title {
                font-size: 18px;
            }
            
            .breadcrumb {
                flex-wrap: wrap;
                margin-bottom: 0;
            }
            
            .breadcrumb-item {
                font-size: 13px;
            }
        }
        
        /* Grid system adjustments */
        @media (max-width: 768px) {
            .row-cols-md-4 > * {
                flex: 0 0 50%;
                max-width: 50%;
            }
        }
        
        @media (max-width: 576px) {
            .row-cols-md-4 > * {
                flex: 0 0 100%;
                max-width: 100%;
            }
        }
        
        /* Time toggle - Mobile Responsive */
        @media (max-width: 576px) {
            .time-toggle-syndron {
                flex-wrap: wrap;
                gap: 5px;
            }
            
            .time-toggle-btn-syndron {
                flex: 1;
                min-width: 60px;
                text-align: center;
                font-size: 12px;
                padding: 6px 8px;
            }
        }
        
        /* Today's transactions - Mobile Responsive */
        @media (max-width: 768px) {
            .transaction-grid {
                grid-template-columns: 1fr !important;
                gap: 10px;
            }
            
            .transaction-card {
                padding: 12px !important;
            }
        }
        
        /* Search box - Mobile Responsive */
        @media (max-width: 576px) {
            .search-box-syndron input {
                padding-left: 40px;
                font-size: 14px;
            }
            
            .search-box-syndron i {
                left: 12px;
            }
        }
        
        /* General responsive adjustments */
        @media (max-width: 768px) {
            .page-content {
                padding: 10px;
            }
            
            .card-body.p-4 {
                padding: 15px !important;
            }
            
            .h5, .h6 {
                font-size: 16px;
            }
            
            .h4 {
                font-size: 20px;
            }
            
            .small {
                font-size: 12px;
            }
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-4">
            <div class="breadcrumb-title pe-3">
                <i class="bx bx-transfer-alt me-1"></i> Stock Movements
            </div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">Products</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.show', $product) }}">{{ $product->name }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Stock Movements</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group export-buttons-syndron">
                    <a href="{{ route('products.stock-movements.export', $product) }}" class="btn btn-success">
                        <i class="bx bx-download"></i> Export CSV
                    </a>
                    <a href="{{ route('products.inventory-history', $product) }}" class="btn btn-info">
                        <i class="bx bx-history"></i> Full History
                    </a>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addStockModal">
                        <i class="bx bx-plus"></i> Add Stock
                    </button>
                    <a href="{{ route('products.show', $product) }}" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Product Header - Syndron Style -->
        <div class="product-header-syndron mb-4">
            <div class="product-header-content">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <div class="d-flex align-items-center">
                            <div class="flex-shrink-0">
                                <div class="bg-white rounded-circle p-3 me-3" style="opacity: 0.9;">
                                    <i class="bx bx-package text-primary fs-4"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1">
                                <h3 class="mb-2 text-white">{{ $product->name }}</h3>
                                <div class="row g-3">
                                    <div class="col-sm-6 col-md-4">
                                        <small class="text-white-50 d-block">SKU</small>
                                        <strong class="text-white">{{ $product->sku }}</strong>
                                    </div>
                                    <div class="col-sm-6 col-md-4">
                                        <small class="text-white-50 d-block">Current Stock</small>
                                        <strong class="h4 text-white">{{ $product->stock }}</strong>
                                    </div>
                                    <div class="col-sm-6 col-md-4">
                                        <small class="text-white-50 d-block">Last Movement</small>
                                        <strong class="text-white">
                                            @if($stockMovements->count() > 0)
                                                {{ $stockMovements->first()->created_at->diffForHumans() }}
                                            @else
                                                Never
                                            @endif
                                        </strong>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="text-end">
                            <div class="display-6 mb-2 text-white">KES {{ number_format($product->sale_price, 2) }}</div>
                            <small class="text-white-50">Current Sale Price</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics Cards - Syndron Style -->
        <div class="row row-cols-1 row-cols-md-4 row-cols-xl-4 mb-4 g-3">
            <div class="col">
                <div class="stat-card-syndron secondary" onclick="filterByType('')">
                    <i class="bx bx-transfer-alt stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['total_movements'] }}</div>
                        <div class="stat-label-syndron">Total Movements</div>
                        <div class="stat-change-syndron positive">
                            <i class="bx bx-trending-up"></i>
                            {{ round(($stats['total_movements'] / max($product->stock, 1)) * 100) }}% of stock activity
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron success" onclick="filterByType('in')">
                    <i class="bx bx-up-arrow-alt stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['total_in'] }}</div>
                        <div class="stat-label-syndron">Total Stock In</div>
                        <div class="stat-change-syndron positive">
                            <i class="bx bx-plus"></i>
                            {{ $stats['total_movements'] > 0 ? round(($stats['total_in'] / $stats['total_movements']) * 100) : 0 }}% of movements
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron danger" onclick="filterByType('out')">
                    <i class="bx bx-down-arrow-alt stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['total_out'] }}</div>
                        <div class="stat-label-syndron">Total Stock Out</div>
                        <div class="stat-change-syndron negative">
                            <i class="bx bx-minus"></i>
                            {{ $stats['total_movements'] > 0 ? round(($stats['total_out'] / $stats['total_movements']) * 100) : 0 }}% of movements
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="stat-card-syndron info" onclick="showCurrentStockDetails()">
                    <i class="bx bx-trending-up stat-icon-syndron"></i>
                    <div class="stat-content-syndron">
                        <div class="stat-value-syndron">{{ $stats['current_stock'] }}</div>
                        <div class="stat-label-syndron">Current Stock Level</div>
                        <div class="stat-change-syndron {{ $stats['net_movement'] >= 0 ? 'positive' : 'negative' }}">
                            <i class="bx bx-{{ $stats['net_movement'] >= 0 ? 'trending-up' : 'trending-down' }}"></i>
                            Net: {{ $stats['net_movement'] >= 0 ? '+' : '' }}{{ $stats['net_movement'] }}
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Stats - Syndron Style -->
        @if(isset($chartData['typeDistribution']))
        <div class="quick-stats-syndron">
            <div class="row">
                <div class="col-6 col-md-3 mb-3 mb-md-0">
                    <div class="text-center">
                        <div class="display-5 mb-2 text-success" style="font-weight: 800;">{{ $chartData['typeDistribution']['in'] ?? 0 }}</div>
                        <small class="text-muted">Stock In Quantity</small>
                    </div>
                </div>
                <div class="col-6 col-md-3 mb-3 mb-md-0">
                    <div class="text-center">
                        <div class="display-5 mb-2 text-danger" style="font-weight: 800;">{{ $chartData['typeDistribution']['out'] ?? 0 }}</div>
                        <small class="text-muted">Stock Out Quantity</small>
                    </div>
                </div>
                <div class="col-6 col-md-3 mb-3 mb-md-0">
                    <div class="text-center">
                        @php
                            $netChange = ($chartData['typeDistribution']['in'] ?? 0) - ($chartData['typeDistribution']['out'] ?? 0);
                        @endphp
                        <div class="display-5 mb-2 {{ $netChange >= 0 ? 'text-success' : 'text-danger' }}" style="font-weight: 800;">
                            {{ $netChange >= 0 ? '+' : '' }}{{ $netChange }}
                        </div>
                        <small class="text-muted">Net Movement</small>
                    </div>
                </div>
                <div class="col-6 col-md-3 mb-3 mb-md-0">
                    <div class="text-center">
                        <div class="display-5 mb-2 text-primary" style="font-weight: 800;">{{ $stockMovements->total() }}</div>
                        <small class="text-muted">Movement Records</small>
                    </div>
                </div>
            </div>
        </div>
        @endif

        <!-- Charts Section - Syndron Style -->
        <div class="row mb-4">
            <div class="col-lg-8">
                <div class="chart-container-syndron" id="stockChartContainer">
                    <div class="chart-title" onclick="toggleChart('stockChartContainer')">
                        <div>
                            <i class="bx bx-line-chart"></i> Stock Movement Timeline
                            <small class="text-muted ms-2">({{ $chartData['view_mode'] ?? 'Daily' }} View)</small>
                        </div>
                        <div class="chart-actions">
                            <div class="time-toggle-syndron">
                                <button class="time-toggle-btn-syndron {{ ($chartData['view_mode'] ?? 'daily') == 'daily' ? 'active' : '' }}" 
                                        onclick="changeViewMode('daily', event)">
                                    Daily
                                </button>
                                <button class="time-toggle-btn-syndron {{ ($chartData['view_mode'] ?? 'daily') == 'hourly' ? 'active' : '' }}" 
                                        onclick="changeViewMode('hourly', event)">
                                    Hourly
                                </button>
                            </div>
                            <i class="bx bx-chevron-up collapse-icon" id="stockChartIcon"></i>
                        </div>
                    </div>
                    <div class="chart-wrapper">
                        <div class="chart-controls">
                            <div class="d-flex gap-3">
                                <div class="legend-item">
                                    <div class="legend-color" style="background-color: #10b981;"></div>
                                    <span>Stock In</span>
                                </div>
                                <div class="legend-item">
                                    <div class="legend-color" style="background-color: #ef4444;"></div>
                                    <span>Stock Out</span>
                                </div>
                                <div class="legend-item">
                                    <div class="legend-color" style="background-color: #667eea;"></div>
                                    <span>Net Change</span>
                                </div>
                            </div>
                            <div class="dropdown">
                                <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                    <i class="bx bx-calendar"></i> {{ $chartData['days'] ?? 30 }} Days
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" onclick="updateChartRange(7)">Last 7 Days</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="updateChartRange(30)">Last 30 Days</a></li>
                                    <li><a class="dropdown-item" href="#" onclick="updateChartRange(90)">Last 90 Days</a></li>
                                </ul>
                            </div>
                        </div>
                        
                        @if(($chartData['view_mode'] ?? 'daily') == 'hourly')
                            <div class="hourly-chart">
                                @foreach($chartData['hourly_transactions'] ?? [] as $hour => $transactions)
                                    @if(!empty($transactions))
                                        <div class="hourly-bar {{ $transactions['type'] ?? '' }}">
                                            <div class="hourly-time">{{ $hour }}:00</div>
                                            <div class="hourly-details">
                                                <span class="hourly-quantity {{ $transactions['type'] ?? '' }}">
                                                    {{ $transactions['type'] == 'in' ? '+' : '-' }}{{ $transactions['quantity'] ?? 0 }}
                                                </span>
                                                <span class="hourly-reason">{{ $transactions['reason'] ?? 'Various transactions' }}</span>
                                                <span class="hourly-user">{{ $transactions['count'] ?? 0 }} transactions</span>
                                            </div>
                                        </div>
                                    @endif
                                @endforeach
                            </div>
                        @else
                            <canvas id="stockMovementChart" height="250"></canvas>
                            <div class="chart-summary">
                                <div class="summary-item">
                                    <div class="summary-value text-success">{{ $chartData['daily_summary']['total_in'] ?? 0 }}</div>
                                    <div class="summary-label">Total In</div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-value text-danger">{{ $chartData['daily_summary']['total_out'] ?? 0 }}</div>
                                    <div class="summary-label">Total Out</div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-value {{ ($chartData['daily_summary']['net_change'] ?? 0) >= 0 ? 'text-success' : 'text-danger' }}">
                                        {{ ($chartData['daily_summary']['net_change'] ?? 0) >= 0 ? '+' : '' }}{{ $chartData['daily_summary']['net_change'] ?? 0 }}
                                    </div>
                                    <div class="summary-label">Net Change</div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-value">{{ $chartData['daily_summary']['transaction_count'] ?? 0 }}</div>
                                    <div class="summary-label">Transactions</div>
                                </div>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="chart-container-syndron" id="typeChartContainer">
                    <div class="chart-title" onclick="toggleChart('typeChartContainer')">
                        <div>
                            <i class="bx bx-pie-chart-alt"></i> Movement Distribution
                        </div>
                        <i class="bx bx-chevron-up collapse-icon" id="typeChartIcon"></i>
                    </div>
                    <div class="chart-wrapper">
                        <canvas id="typeDistributionChart" height="250"></canvas>
                        <div class="text-center mt-3">
                            <div class="row">
                                <div class="col-6">
                                    <div class="text-success">
                                        <div class="display-6 mb-1" style="font-weight: 800;">{{ $chartData['typeDistribution']['in'] ?? 0 }}</div>
                                        <small>Stock In</small>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="text-danger">
                                        <div class="display-6 mb-1" style="font-weight: 800;">{{ $chartData['typeDistribution']['out'] ?? 0 }}</div>
                                        <small>Stock Out</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Today's Transactions - Syndron Style -->
        @if(isset($chartData['today_transactions']) && count($chartData['today_transactions']) > 0)
        <div class="chart-container-syndron mb-4" id="todayTransactionsContainer">
            <div class="chart-title" onclick="toggleChart('todayTransactionsContainer')">
                <div>
                    <i class="bx bx-calendar-check"></i> Today's Transactions
                    <small class="text-muted ms-2">({{ count($chartData['today_transactions']) }} movements)</small>
                </div>
                <i class="bx bx-chevron-up collapse-icon" id="todayTransactionsIcon"></i>
            </div>
            <div class="chart-wrapper">
                <div class="transaction-grid">
                    @foreach($chartData['today_transactions'] as $transaction)
                    <div class="transaction-card">
                        <div class="transaction-header">
                            <span class="transaction-time">{{ $transaction['time'] }}</span>
                            <span class="transaction-quantity {{ $transaction['type'] }}">
                                {{ $transaction['type'] == 'in' ? '+' : '-' }}{{ $transaction['quantity'] }}
                            </span>
                        </div>
                        <div class="transaction-reason">{{ $transaction['reason'] }}</div>
                        <div class="transaction-meta">
                            <span>{{ $transaction['user'] }}</span>
                            @if($transaction['batch'])
                            <span>Batch: {{ $transaction['batch'] }}</span>
                            @endif
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
        @endif

        <!-- Filters Section - Syndron Style -->
        <div class="filter-section-syndron">
            <div class="row align-items-center">
                <div class="col-12 col-md-3 mb-3 mb-md-0">
                    <label class="form-label">Movement Type</label>
                    <select name="movement_type" class="form-select" onchange="applyFilter()">
                        <option value="">All Types</option>
                        <option value="in" {{ request('movement_type') == 'in' ? 'selected' : '' }}>Stock In</option>
                        <option value="out" {{ request('movement_type') == 'out' ? 'selected' : '' }}>Stock Out</option>
                    </select>
                </div>
                <div class="col-12 col-md-3 mb-3 mb-md-0">
                    <label class="form-label">Date Range</label>
                    <div class="input-group">
                        <input type="date" name="start_date" class="form-control" 
                               value="{{ request('start_date') }}" onchange="applyFilter()">
                        <span class="input-group-text">to</span>
                        <input type="date" name="end_date" class="form-control" 
                               value="{{ request('end_date') }}" onchange="applyFilter()">
                    </div>
                </div>
                <div class="col-12 col-md-4 mb-3 mb-md-0">
                    <label class="form-label">Search</label>
                    <div class="search-box-syndron">
                        <i class="bx bx-search"></i>
                        <input type="text" name="search" class="form-control" placeholder="Search by reason, notes, batch..." 
                               value="{{ request('search') }}" onkeyup="applyFilter()">
                    </div>
                </div>
                <div class="col-12 col-md-2 d-flex align-items-end">
                    <div class="d-flex gap-2 w-100">
                        <button type="button" class="btn btn-light w-50" onclick="resetFilters()">
                            <i class="bx bx-reset"></i> Reset
                        </button>
                        <a href="{{ route('products.stock-movements.export', array_merge(['product' => $product->id], request()->all())) }}" 
                           class="btn btn-success w-50">
                            <i class="bx bx-download"></i> Export
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Stock Movements History - Syndron Style -->
        <div class="card border-0 shadow-sm">
            <div class="card-body p-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h5 class="mb-1">Stock Movement History</h5>
                        <p class="text-muted mb-0">Showing all stock movements for {{ $product->name }}</p>
                    </div>
                    <div class="text-end">
                        <div class="text-muted small">
                            {{ $stockMovements->total() }} records found
                        </div>
                    </div>
                </div>

                @if($stockMovements->isEmpty())
                    <div class="empty-state-syndron">
                        <i class="bx bx-history"></i>
                        <h5 class="mt-3 mb-2">No stock movements found</h5>
                        <p class="text-muted mb-4">This product hasn't had any stock movements yet.</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addStockModal">
                            <i class="bx bx-plus"></i> Add First Stock
                        </button>
                    </div>
                @else
                    <div class="timeline-syndron">
                        @foreach($stockMovements as $movement)
                            <div class="timeline-item-syndron">
                                <div class="movement-card-syndron p-4 movement-{{ $movement->movement_type == 'in' ? 'in' : 'out' }}">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div class="flex-grow-1">
                                            <div class="d-flex align-items-center mb-3 flex-wrap">
                                                <span class="movement-type-badge-syndron badge-{{ $movement->movement_type }}-syndron">
                                                    <i class="bx bx-{{ $movement->movement_type == 'in' ? 'up-arrow-alt' : 'down-arrow-alt' }} me-1"></i>
                                                    {{ $movement->movement_type == 'in' ? 'STOCK IN' : 'STOCK OUT' }}
                                                </span>
                                                <span class="ms-3 fw-bold h5 {{ $movement->movement_type == 'in' ? 'text-success' : 'text-danger' }}">
                                                    {{ $movement->movement_type == 'in' ? '+' : '-' }}{{ $movement->quantity }}
                                                </span>
                                                @if($movement->quantity_before)
                                                    <span class="ms-3 text-muted small">
                                                        Stock: {{ $movement->quantity_before }} → {{ $movement->quantity_before + $movement->quantity }}
                                                    </span>
                                                @endif
                                            </div>
                                            
                                            <h6 class="mb-2">
                                                <i class="bx bx-receipt me-2"></i>{{ $movement->reason }}
                                            </h6>
                                            
                                            @if($movement->batch || $movement->reference_number || $movement->notes)
                                                <div class="movement-details">
                                                    @if($movement->batch)
                                                        <div class="mb-2">
                                                            <i class="bx bx-package me-2"></i>
                                                            <strong>Batch:</strong> {{ $movement->batch->batch_number }}
                                                            @if($movement->batch->expiry_date)
                                                                <span class="ms-2 ms-md-3 d-block d-md-inline">
                                                                    <i class="bx bx-calendar me-1"></i>
                                                                    Expires: {{ $movement->batch->expiry_date->format('M d, Y') }}
                                                                </span>
                                                            @endif
                                                        </div>
                                                    @endif
                                                    
                                                    @if($movement->reference_number)
                                                        <div class="mb-2">
                                                            <i class="bx bx-hash me-2"></i>
                                                            <strong>Reference:</strong> {{ $movement->reference_number }}
                                                        </div>
                                                    @endif
                                                    
                                                    @if($movement->notes)
                                                        <div class="mb-0">
                                                            <i class="bx bx-note me-2"></i>
                                                            <strong>Notes:</strong> {{ $movement->notes }}
                                                        </div>
                                                    @endif
                                                </div>
                                            @endif
                                            
                                            <div class="d-flex align-items-center mt-3 flex-wrap">
                                                <div class="d-flex align-items-center me-3 mb-2 mb-md-0">
                                                    <i class="bx bx-user me-2 text-muted"></i>
                                                    <span class="small">{{ $movement->user->name ?? 'System' }}</span>
                                                </div>
                                                <div class="d-flex align-items-center mb-2 mb-md-0">
                                                    <i class="bx bx-time me-2 text-muted"></i>
                                                    <span class="small">{{ $movement->created_at->format('h:i A') }}</span>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="text-end ms-3">
                                            <div class="text-muted small mb-2">
                                                {{ $movement->created_at->format('M d, Y') }}
                                            </div>
                                            <div class="text-muted small mb-3">
                                                {{ $movement->created_at->diffForHumans() }}
                                            </div>
                                            <button class="btn btn-sm btn-outline-secondary" 
                                                    onclick="showMovementDetails({{ $movement->id }})"
                                                    title="View Details">
                                                <i class="bx bx-show"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>

                    <!-- Pagination -->
                    <div class="pagination-container">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="text-muted">
                                Showing {{ $stockMovements->firstItem() }} to {{ $stockMovements->lastItem() }} 
                                of {{ $stockMovements->total() }} entries
                            </div>
                            <nav>
                                {{ $stockMovements->withQueryString()->links() }}
                            </nav>
                        </div>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Add Stock Modal - Updated for Syndron -->
<div class="modal fade" id="addStockModal" tabindex="-1" aria-labelledby="addStockModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form action="{{ route('products.add-stock', $product) }}" method="POST" id="addStockForm">
                @csrf
                <div class="modal-header bg-primary text-white" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                    <h5 class="modal-title" id="addStockModalLabel">
                        <i class="bx bx-plus"></i> Add Stock to {{ $product->name }}
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-12 col-md-6 mb-3">
                            <label class="form-label">Quantity *</label>
                            <input type="number" name="quantity" class="form-control" min="1" required 
                                   placeholder="Enter quantity to add">
                        </div>
                        <div class="col-12 col-md-6 mb-3">
                            <label class="form-label">Reason *</label>
                            <select name="reason" class="form-select" required>
                                <option value="">Select Reason</option>
                                <option value="purchase">Purchase/Replenishment</option>
                                <option value="return">Customer Return</option>
                                <option value="adjustment">Stock Adjustment</option>
                                <option value="transfer">Transfer In</option>
                                <option value="production">Production</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                    </div>
                    
                    @if($product->track_batches && $product->has_expiry)
                    <div class="card mb-3 border-0 shadow-sm">
                        <div class="card-header bg-light">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="use_batch" 
                                       id="useBatch" value="1" checked>
                                <label class="form-check-label fw-bold" for="useBatch">
                                    Batch Information (Required for Expiry Tracking)
                                </label>
                            </div>
                        </div>
                        <div class="card-body" id="batchFields">
                            <div class="row">
                                <div class="col-12 col-md-6 mb-3">
                                    <label class="form-label">Batch Number *</label>
                                    <input type="text" name="batch_data[batch_number]" class="form-control" 
                                           value="{{ 'BATCH-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 6)) }}" 
                                           required>
                                </div>
                                <div class="col-12 col-md-6 mb-3">
                                    <label class="form-label">Expiry Date *</label>
                                    <input type="date" name="batch_data[expiry_date]" class="form-control" 
                                           min="{{ date('Y-m-d') }}" required>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-12 col-md-6 mb-3">
                                    <label class="form-label">Manufacture Date</label>
                                    <input type="date" name="batch_data[manufacture_date]" class="form-control" 
                                           max="{{ date('Y-m-d') }}">
                                </div>
                                <div class="col-12 col-md-6 mb-3">
                                    <label class="form-label">Cost Price</label>
                                    <input type="number" step="0.01" name="batch_data[cost_price]" 
                                           class="form-control" value="{{ $product->cost_price }}">
                                </div>
                            </div>
                        </div>
                    </div>
                    @endif
                    
                    <div class="mb-3">
                        <label class="form-label">Reference Number</label>
                        <input type="text" name="reference_number" class="form-control" 
                               placeholder="Optional reference number">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" class="form-control" rows="3" 
                                  placeholder="Additional notes about this stock addition"></textarea>
                    </div>
                    
                    @if(!$product->track_batches)
                    <div class="alert alert-info border-0 shadow-sm">
                        <i class="bx bx-info-circle"></i>
                        This product doesn't use batch tracking. Stock will be added to general inventory.
                    </div>
                    @endif
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none;">
                        <i class="bx bx-plus"></i> Add Stock
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Movement Details Modal -->
<div class="modal fade" id="movementDetailsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                <h5 class="modal-title">Movement Details</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="movementDetailsContent">
                Loading...
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
$(document).ready(function() {
    // Initialize Charts
    initializeCharts();
    
    // Set initial chart states from localStorage
    restoreChartStates();
    
    // Toggle batch fields
    $('#useBatch').change(function() {
        if ($(this).is(':checked')) {
            $('#batchFields').show();
            $('#batchFields input, #batchFields select').prop('required', true);
        } else {
            $('#batchFields').hide();
            $('#batchFields input, #batchFields select').prop('required', false);
        }
    });

    // Form validation
    $('#addStockForm').validate({
        rules: {
            quantity: {
                required: true,
                min: 1
            },
            reason: {
                required: true
            }
        },
        messages: {
            quantity: {
                required: "Please enter quantity",
                min: "Quantity must be at least 1"
            },
            reason: {
                required: "Please select a reason"
            }
        }
    });

    // Auto-refresh data every 30 seconds
    setInterval(refreshData, 30000);

    // Mobile detection and adjustments
    if ($(window).width() < 768) {
        $('.stat-card-syndron').css('cursor', 'pointer');
        $('.chart-title').css('cursor', 'pointer');
        
        // Make tables scrollable
        $('.table-responsive').addClass('mobile-scroll');
        
        // Adjust modal for mobile
        $(window).on('resize', function() {
            if ($(window).width() < 768) {
                $('.modal-dialog').css('margin', '10px');
            }
        }).trigger('resize');
    }

    // Touch events for mobile
    if ('ontouchstart' in window) {
        $('.stat-card-syndron').on('touchstart', function() {
            $(this).addClass('touched');
        }).on('touchend', function() {
            $(this).removeClass('touched');
        });
        
        // Prevent zoom on input focus for better mobile experience
        $('input, select, textarea').on('focus', function() {
            $('meta[name=viewport]').attr('content', 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=0');
        }).on('blur', function() {
            $('meta[name=viewport]').attr('content', 'width=device-width, initial-scale=1.0');
        });
    }
});

function initializeCharts() {
    @if(($chartData['view_mode'] ?? 'daily') == 'daily')
    // Stock Movement Chart (Daily View)
    const ctx = document.getElementById('stockMovementChart')?.getContext('2d');
    if (ctx) {
        const stockMovementChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: @json($chartData['daily']['labels'] ?? []),
                datasets: [
                    {
                        label: 'Stock In',
                        data: @json($chartData['daily']['in'] ?? []),
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        borderWidth: 3,
                        tension: 0.4,
                        fill: true,
                        pointRadius: 3,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Stock Out',
                        data: @json($chartData['daily']['out'] ?? []),
                        borderColor: '#ef4444',
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        borderWidth: 3,
                        tension: 0.4,
                        fill: true,
                        pointRadius: 3,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Net Change',
                        data: @json($chartData['daily']['net'] ?? []),
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        borderWidth: 2,
                        borderDash: [5, 5],
                        tension: 0.4,
                        fill: false,
                        pointRadius: 0
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    intersect: false,
                    mode: 'index'
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            drawBorder: false
                        },
                        title: {
                            display: true,
                            text: 'Quantity'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            maxRotation: 45,
                            minRotation: 45,
                            callback: function(value, index, values) {
                                // For mobile, show fewer labels
                                if ($(window).width() < 768 && values.length > 7) {
                                    return index % 2 === 0 ? this.getLabelForValue(value) : '';
                                }
                                return this.getLabelForValue(value);
                            }
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        backgroundColor: 'rgba(0, 0, 0, 0.7)',
                        titleFont: {
                            size: 14
                        },
                        bodyFont: {
                            size: 13
                        },
                        padding: 12,
                        boxPadding: 6,
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                let value = context.raw;
                                return `${label}: ${value}`;
                            }
                        }
                    }
                }
            }
        });
        window.stockMovementChart = stockMovementChart;
        
        // Adjust chart on window resize
        $(window).on('resize', function() {
            if (window.stockMovementChart) {
                window.stockMovementChart.resize();
            }
        });
    }
    @endif

    // Type Distribution Chart
    const typeCtx = document.getElementById('typeDistributionChart')?.getContext('2d');
    if (typeCtx) {
        const typeDistributionChart = new Chart(typeCtx, {
            type: 'doughnut',
            data: {
                labels: ['Stock In', 'Stock Out'],
                datasets: [{
                    data: [
                        {{ $chartData['typeDistribution']['in'] ?? 0 }},
                        {{ $chartData['typeDistribution']['out'] ?? 0 }}
                    ],
                    backgroundColor: [
                        '#10b981',
                        '#ef4444'
                    ],
                    borderWidth: 2,
                    borderColor: '#ffffff',
                    hoverOffset: 15
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '70%',
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.raw || 0;
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                return `${label}: ${value} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
        window.typeDistributionChart = typeDistributionChart;
    }
}

function toggleChart(containerId) {
    const container = document.getElementById(containerId);
    const icon = document.getElementById(containerId.replace('Container', 'Icon'));
    
    if (container) {
        container.classList.toggle('collapsed');
        if (icon) {
            icon.classList.toggle('collapsed');
        }
        
        // Save state to localStorage
        const isCollapsed = container.classList.contains('collapsed');
        localStorage.setItem(`chart_${containerId}_collapsed`, isCollapsed);
        
        // Update chart if it exists
        setTimeout(() => {
            if (window.stockMovementChart && containerId === 'stockChartContainer') {
                window.stockMovementChart.resize();
            }
            if (window.typeDistributionChart && containerId === 'typeChartContainer') {
                window.typeDistributionChart.resize();
            }
        }, 300);
    }
}

function restoreChartStates() {
    const containers = ['stockChartContainer', 'typeChartContainer', 'todayTransactionsContainer'];
    
    containers.forEach(containerId => {
        const isCollapsed = localStorage.getItem(`chart_${containerId}_collapsed`) === 'true';
        const container = document.getElementById(containerId);
        const icon = document.getElementById(containerId.replace('Container', 'Icon'));
        
        if (container && isCollapsed) {
            container.classList.add('collapsed');
            if (icon) {
                icon.classList.add('collapsed');
            }
        }
    });
}

function changeViewMode(mode, event) {
    if (event) event.stopPropagation();
    
    // Update active button
    document.querySelectorAll('.time-toggle-btn-syndron').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Update URL with view mode parameter
    const url = new URL(window.location.href);
    url.searchParams.set('view_mode', mode);
    window.location.href = url.toString();
}

function updateChartRange(days) {
    const endDate = new Date().toISOString().split('T')[0];
    const startDate = new Date(Date.now() - days * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
    
    const url = new URL(window.location.href);
    url.searchParams.set('start_date', startDate);
    url.searchParams.set('end_date', endDate);
    url.searchParams.set('days', days);
    
    window.location.href = url.toString();
}

function filterByType(type) {
    const select = document.querySelector('select[name="movement_type"]');
    if (select) {
        select.value = type;
        applyFilter();
    }
}

function applyFilter() {
    const form = document.createElement('form');
    form.method = 'GET';
    form.action = window.location.pathname;
    
    const movementType = document.querySelector('select[name="movement_type"]').value;
    const startDate = document.querySelector('input[name="start_date"]').value;
    const endDate = document.querySelector('input[name="end_date"]').value;
    const search = document.querySelector('input[name="search"]').value;
    
    if (movementType) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'movement_type';
        input.value = movementType;
        form.appendChild(input);
    }
    
    if (startDate) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'start_date';
        input.value = startDate;
        form.appendChild(input);
    }
    
    if (endDate) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'end_date';
        input.value = endDate;
        form.appendChild(input);
    }
    
    if (search) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'search';
        input.value = search;
        form.appendChild(input);
    }
    
    // Add view mode if exists
    const urlParams = new URLSearchParams(window.location.search);
    const viewMode = urlParams.get('view_mode');
    if (viewMode) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'view_mode';
        input.value = viewMode;
        form.appendChild(input);
    }
    
    document.body.appendChild(form);
    form.submit();
}

function resetFilters() {
    window.location.href = window.location.pathname;
}

function showMovementDetails(movementId) {
    // For mobile, use a simpler modal
    if ($(window).width() < 768) {
        window.location.href = `/stock-movements/${movementId}/details`;
        return;
    }
    
    $.ajax({
        url: `/api/stock-movements/${movementId}`,
        method: 'GET',
        success: function(response) {
            if (response.success) {
                const movement = response.data;
                let detailsHtml = `
                    <div class="row">
                        <div class="col-12 col-md-6">
                            <div class="mb-3">
                                <strong>Type:</strong><br>
                                <span class="badge badge-${movement.movement_type}">
                                    ${movement.movement_type.toUpperCase()}
                                </span>
                            </div>
                        </div>
                        <div class="col-12 col-md-6">
                            <div class="mb-3">
                                <strong>Quantity:</strong><br>
                                <span class="${movement.movement_type === 'in' ? 'text-success' : 'text-danger'}">
                                    ${movement.movement_type === 'in' ? '+' : '-'}${movement.quantity}
                                </span>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <strong>Reason:</strong><br>
                        ${movement.reason}
                    </div>
                `;
                
                if (movement.batch) {
                    detailsHtml += `
                        <div class="mb-3">
                            <strong>Batch Details:</strong><br>
                            <div class="ms-3">
                                <div>Number: ${movement.batch.batch_number}</div>
                                ${movement.batch.expiry_date ? `<div>Expiry: ${movement.batch.expiry_date}</div>` : ''}
                            </div>
                        </div>
                    `;
                }
                
                if (movement.notes) {
                    detailsHtml += `
                        <div class="mb-3">
                            <strong>Notes:</strong><br>
                            ${movement.notes}
                        </div>
                    `;
                }
                
                detailsHtml += `
                    <div class="row">
                        <div class="col-12 col-md-6">
                            <div class="mb-3">
                                <strong>User:</strong><br>
                                ${movement.user?.name || 'System'}
                            </div>
                        </div>
                        <div class="col-12 col-md-6">
                            <div class="mb-3">
                                <strong>Date & Time:</strong><br>
                                ${movement.created_at_formatted}
                            </div>
                        </div>
                    </div>
                `;
                
                $('#movementDetailsContent').html(detailsHtml);
                $('#movementDetailsModal').modal('show');
            }
        },
        error: function() {
            $('#movementDetailsContent').html('<div class="alert alert-danger">Failed to load movement details.</div>');
            $('#movementDetailsModal').modal('show');
        }
    });
}

function showCurrentStockDetails() {
    const detailsHtml = `
        <div class="alert alert-info border-0 shadow-sm">
            <h6><i class="bx bx-info-circle"></i> Current Stock Details</h6>
            <hr>
            <div class="row">
                <div class="col-6">
                    <strong>Product:</strong><br>
                    {{ $product->name }}
                </div>
                <div class="col-6">
                    <strong>SKU:</strong><br>
                    {{ $product->sku }}
                </div>
            </div>
            <div class="row mt-2">
                <div class="col-6">
                    <strong>Current Stock:</strong><br>
                    <span class="h4 {{ $product->stock > 0 ? 'text-success' : 'text-danger' }}">
                        {{ $product->stock }}
                    </span>
                </div>
                <div class="col-6">
                    <strong>Stock Value:</strong><br>
                    <span class="h4 text-success">
                        KES {{ number_format($product->stock * $product->cost_price, 2) }}
                    </span>
                </div>
            </div>
            @if($product->has_expiry)
            <hr>
            <div class="row">
                <div class="col-12">
                    <strong>Expiry Status:</strong><br>
                    <small class="text-muted">This product has expiry tracking enabled.</small>
                </div>
            </div>
            @endif
        </div>
    `;
    
    $('#movementDetailsContent').html(detailsHtml);
    $('#movementDetailsModal').modal('show');
}

function refreshData() {
    $.ajax({
        url: window.location.href,
        method: 'GET',
        data: {
            refresh: true,
            _token: '{{ csrf_token() }}'
        },
        success: function(response) {
            console.log('Data refreshed');
        }
    });
}
</script>
@endsection