@extends('layouts.app')

@section('style')
<style>
    .batch-status {
        padding: 4px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 500;
    }
    .status-active {
        background-color: #d1fae5;
        color: #065f46;
    }
    .status-expired {
        background-color: #fee2e2;
        color: #991b1b;
    }
    .status-expiring {
        background-color: #fef3c7;
        color: #92400e;
    }
    .status-empty {
        background-color: #e5e7eb;
        color: #6b7280;
    }
    .expiry-card {
        border-left: 4px solid;
        transition: all 0.3s;
    }
    .expiry-card.expired {
        border-left-color: #ef4444;
        background-color: #fef2f2;
    }
    .expiry-card.expiring {
        border-left-color: #f59e0b;
        background-color: #fffbeb;
    }
    .expiry-card.valid {
        border-left-color: #10b981;
        background-color: #f0fdf4;
    }
    .batch-card {
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 15px;
        transition: all 0.3s;
    }
    .batch-card:hover {
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        transform: translateY(-2px);
    }
    .batch-progress {
        height: 8px;
        border-radius: 4px;
        overflow: hidden;
    }
    .batch-progress-bar {
        height: 100%;
        transition: width 0.3s;
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Products</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">Products</a></li>
                        @if(isset($product))
                        <li class="breadcrumb-item"><a href="{{ route('products.show', $product) }}">{{ $product->name }}</a></li>
                        @endif
                        <li class="breadcrumb-item active" aria-current="page">Batches</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    @if(isset($product))
                    <a href="{{ route('products.show', $product) }}" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back to Product
                    </a>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addBatchModal">
                        <i class="bx bx-plus"></i> Add Batch
                    </button>
                    @else
                    <a href="{{ route('batches.create') }}" class="btn btn-primary">
                        <i class="bx bx-plus"></i> Add Batch
                    </a>
                    @endif
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        @if(isset($product))
        <!-- Product Info -->
        <div class="card mb-4">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-8">
                        <h5 class="mb-2">{{ $product->name }}</h5>
                        <div class="row g-3">
                            <div class="col-sm-6 col-md-3">
                                <small class="text-muted d-block">SKU</small>
                                <strong>{{ $product->sku }}</strong>
                            </div>
                            <div class="col-sm-6 col-md-3">
                                <small class="text-muted d-block">Current Stock</small>
                                <strong>{{ $product->stock }} units</strong>
                            </div>
                            <div class="col-sm-6 col-md-3">
                                <small class="text-muted d-block">Batch Tracked</small>
                                <strong>{{ $product->track_batches ? 'Yes' : 'No' }}</strong>
                            </div>
                            <div class="col-sm-6 col-md-3">
                                <small class="text-muted d-block">Expiry Tracking</small>
                                <strong>{{ $product->has_expiry ? 'Yes' : 'No' }}</strong>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="d-flex justify-content-end">
                            <div class="text-end">
                                <div class="h4 mb-1">${{ number_format($product->cost_price, 2) }}</div>
                                <small class="text-muted">Cost Price</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        @endif

        <!-- Batch Statistics -->
        <div class="row row-cols-1 row-cols-md-4 row-cols-xl-4 mb-4">
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-primary">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Total Batches</p>
                                <h4 class="my-1">{{ $batchStats['total_batches'] ?? 0 }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-primary text-primary ms-auto">
                                <i class="bx bx-package"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-success">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Active Batches</p>
                                <h4 class="my-1">{{ $batchStats['active_batches'] ?? 0 }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-success text-success ms-auto">
                                <i class="bx bx-check-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-warning">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Expiring Soon</p>
                                <h4 class="my-1">{{ $batchStats['expiring_soon'] ?? 0 }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-warning text-warning ms-auto">
                                <i class="bx bx-time"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-danger">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Expired</p>
                                <h4 class="my-1">{{ $batchStats['expired_batches'] ?? 0 }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-danger text-danger ms-auto">
                                <i class="bx bx-error-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters Section -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" action="{{ isset($product) ? route('products.batches', $product) : route('batches.index') }}" id="filterForm">
                    <div class="row g-3">
                        <!-- Search -->
                        <div class="col-md-3">
                            <label class="form-label">Search</label>
                            <input type="text" name="search" class="form-control" 
                                   placeholder="Batch number or product" 
                                   value="{{ request('search') }}">
                        </div>

                        <!-- Status Filter -->
                        <div class="col-md-2">
                            <label class="form-label">Status</label>
                            <select name="status" class="form-select">
                                <option value="">All Status</option>
                                <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                <option value="expiring" {{ request('status') == 'expiring' ? 'selected' : '' }}>Expiring Soon</option>
                                <option value="expired" {{ request('status') == 'expired' ? 'selected' : '' }}>Expired</option>
                                <option value="empty" {{ request('status') == 'empty' ? 'selected' : '' }}>Empty</option>
                            </select>
                        </div>

                        <!-- Supplier Filter -->
                        <div class="col-md-2">
                            <label class="form-label">Supplier</label>
                            <select name="supplier_id" class="form-select">
                                <option value="">All Suppliers</option>
                                @foreach($suppliers ?? [] as $supplier)
                                    <option value="{{ $supplier->id }}" 
                                            {{ request('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                        {{ $supplier->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Expiry Range -->
                        <div class="col-md-3">
                            <label class="form-label">Expiry Date</label>
                            <div class="input-group">
                                <input type="date" name="expiry_from" class="form-control" 
                                       value="{{ request('expiry_from') }}">
                                <span class="input-group-text">to</span>
                                <input type="date" name="expiry_to" class="form-control" 
                                       value="{{ request('expiry_to') }}">
                            </div>
                        </div>

                        <!-- Actions -->
                        <div class="col-md-2 d-flex align-items-end">
                            <div class="d-flex gap-2 w-100">
                                <button type="submit" class="btn btn-outline-primary">
                                    <i class="bx bx-filter"></i> Filter
                                </button>
                                <a href="{{ isset($product) ? route('products.batches', $product) : route('batches.index') }}" 
                                   class="btn btn-outline-secondary">
                                    <i class="bx bx-reset"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Batches Table -->
        <div class="card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h5 class="card-title mb-0">
                        @if(isset($product))
                            Product Batches
                        @else
                            All Batches
                        @endif
                    </h5>
                    <div class="d-flex gap-2">
                        <a href="{{ route('products.expiry-report') }}" class="btn btn-outline-warning">
                            <i class="bx bx-calendar-exclamation"></i> Expiry Report
                        </a>
                        <a href="{{ route('batches.export') }}" class="btn btn-outline-primary">
                            <i class="bx bx-export"></i> Export
                        </a>
                    </div>
                </div>

                @if(session('success'))
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bx bx-check-circle"></i> {{ session('success') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif

                @if(session('error'))
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bx bx-error-circle"></i> {{ session('error') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif

                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Batch Number</th>
                                <th>Product</th>
                                <th>Supplier</th>
                                <th>Manufacture Date</th>
                                <th>Expiry Date</th>
                                <th>Quantity</th>
                                <th>Cost Price</th>
                                <th>Status</th>
                                <th>Location</th>
                                <th class="text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($batches as $batch)
                                @php
                                    $expiryClass = '';
                                    $status = 'active';
                                    
                                    if ($batch->current_quantity <= 0) {
                                        $status = 'empty';
                                    } elseif ($batch->expiry_date && $batch->expiry_date < now()) {
                                        $status = 'expired';
                                        $expiryClass = 'expired';
                                    } elseif ($batch->expiry_date && $batch->expiry_date <= now()->addDays(30)) {
                                        $status = 'expiring';
                                        $expiryClass = 'expiring';
                                    } else {
                                        $expiryClass = 'valid';
                                    }
                                @endphp
                                <tr class="expiry-card {{ $expiryClass }}">
                                    <td>
                                        <strong>{{ $batch->batch_number }}</strong>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="flex-shrink-0 me-2">
                                                @if($batch->product->image)
                                                    <img src="{{ asset('storage/' . $batch->product->image) }}" 
                                                         alt="{{ $batch->product->name }}" 
                                                         width="30" height="30" class="rounded">
                                                @else
                                                    <div class="rounded bg-light d-flex align-items-center justify-content-center" 
                                                         style="width: 30px; height: 30px;">
                                                        <i class="bx bx-package text-muted"></i>
                                                    </div>
                                                @endif
                                            </div>
                                            <div class="flex-grow-1">
                                                <a href="{{ route('products.show', $batch->product) }}" class="text-dark">
                                                    {{ $batch->product->name }}
                                                </a>
                                                <div class="text-muted small">{{ $batch->product->sku }}</div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        {{ $batch->supplier->name ?? 'N/A' }}
                                    </td>
                                    <td>
                                        {{ $batch->manufacture_date ? $batch->manufacture_date->format('Y-m-d') : 'N/A' }}
                                    </td>
                                    <td>
                                        @if($batch->expiry_date)
                                            <div class="d-flex flex-column">
                                                <span>{{ $batch->expiry_date->format('Y-m-d') }}</span>
                                                @if($batch->expiry_date < now())
                                                    <small class="text-danger">Expired {{ $batch->expiry_date->diffForHumans() }}</small>
                                                @elseif($batch->expiry_date <= now()->addDays(30))
                                                    <small class="text-warning">Expires in {{ $batch->expiry_date->diffForHumans(['parts' => 2]) }}</small>
                                                @else
                                                    <small class="text-muted">Expires {{ $batch->expiry_date->diffForHumans() }}</small>
                                                @endif
                                            </div>
                                        @else
                                            <span class="text-muted">No expiry</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="batch-progress" style="width: 60px;">
                                                <div class="batch-progress-bar bg-{{ $status === 'empty' ? 'secondary' : ($status === 'expired' ? 'danger' : ($status === 'expiring' ? 'warning' : 'success')) }}" 
                                                     style="width: {{ $batch->initial_quantity > 0 ? ($batch->current_quantity / $batch->initial_quantity) * 100 : 0 }}%">
                                                </div>
                                            </div>
                                            <span class="ms-2">{{ $batch->current_quantity }}/{{ $batch->initial_quantity }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <strong>${{ number_format($batch->batch_cost_price, 2) }}</strong>
                                    </td>
                                    <td>
                                        <span class="batch-status status-{{ $status }}">
                                            {{ ucfirst($status) }}
                                        </span>
                                    </td>
                                    <td>
                                        {{ $batch->warehouse_location ?? 'Default' }}
                                    </td>
                                    <td class="text-end">
                                        <div class="d-flex justify-content-end gap-1">
                                            <!-- View Details -->
                                            <a href="{{ route('batches.show', $batch) }}" 
                                               class="btn btn-outline-info btn-sm"
                                               data-bs-toggle="tooltip" title="View Details">
                                                <i class="bx bx-show"></i>
                                            </a>
                                            
                                            <!-- Adjust Stock -->
                                            <button type="button" class="btn btn-outline-primary btn-sm"
                                                    data-bs-toggle="modal" data-bs-target="#adjustStockModal{{ $batch->id }}"
                                                    data-bs-toggle="tooltip" title="Adjust Stock">
                                                <i class="bx bx-transfer"></i>
                                            </button>
                                            
                                            <!-- Quick Actions -->
                                            <div class="dropdown">
                                                <button class="btn btn-outline-secondary btn-sm" 
                                                        type="button" data-bs-toggle="dropdown">
                                                    <i class="bx bx-dots-vertical-rounded"></i>
                                                </button>
                                                <ul class="dropdown-menu dropdown-menu-end">
                                                    <li>
                                                        <a class="dropdown-item" href="{{ route('batches.edit', $batch) }}">
                                                            <i class="bx bx-edit"></i> Edit Batch
                                                        </a>
                                                    </li>
                                                    @if($batch->current_quantity > 0 && $batch->expiry_date && $batch->expiry_date <= now()->addDays(30))
                                                    <li>
                                                        <button type="button" class="dropdown-item text-warning set-expiry-discount"
                                                                data-batch-id="{{ $batch->id }}"
                                                                data-product-id="{{ $batch->product_id }}">
                                                            <i class="bx bx-tag"></i> Set Expiry Discount
                                                        </button>
                                                    </li>
                                                    @endif
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li>
                                                        <button type="button" class="dropdown-item text-danger delete-batch"
                                                                data-batch-id="{{ $batch->id }}">
                                                            <i class="bx bx-trash"></i> Delete Batch
                                                        </button>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>

                                        <!-- Adjust Stock Modal -->
                                        <div class="modal fade" id="adjustStockModal{{ $batch->id }}" tabindex="-1" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">Adjust Stock - {{ $batch->batch_number }}</h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <form action="{{ route('batches.adjust-stock', $batch) }}" method="POST">
                                                        @csrf
                                                        <div class="modal-body">
                                                            <div class="mb-3">
                                                                <label class="form-label">Current Quantity</label>
                                                                <input type="text" class="form-control" 
                                                                       value="{{ $batch->current_quantity }}" readonly>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Adjustment Type</label>
                                                                <select name="adjustment_type" class="form-select" required>
                                                                    <option value="add">Add Stock</option>
                                                                    <option value="remove">Remove Stock</option>
                                                                    <option value="set">Set to Exact Value</option>
                                                                </select>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Quantity</label>
                                                                <input type="number" name="quantity" class="form-control" 
                                                                       min="0" step="1" required>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Reason</label>
                                                                <input type="text" name="reason" class="form-control" 
                                                                       placeholder="Physical count, damage, etc." required>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Notes (Optional)</label>
                                                                <textarea name="notes" class="form-control" rows="2"></textarea>
                                                            </div>
                                                        </div>
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                            <button type="submit" class="btn btn-primary">Save Adjustment</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="10" class="text-center py-5">
                                        <div class="py-5">
                                            <i class="bx bx-package display-4 text-muted mb-3"></i>
                                            <h5 class="text-muted">No batches found</h5>
                                            <p class="text-muted mb-4">
                                                @if(request()->hasAny(['search', 'status', 'supplier_id', 'expiry_from']))
                                                    Try changing your filters
                                                @else
                                                    No batches available
                                                @endif
                                            </p>
                                            @if(isset($product))
                                            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addBatchModal">
                                                <i class="bx bx-plus"></i> Create First Batch
                                            </button>
                                            @else
                                            <a href="{{ route('batches.create') }}" class="btn btn-primary">
                                                <i class="bx bx-plus"></i> Create Batch
                                            </a>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-4">
                    <div class="pagination-info">
                        Showing {{ $batches->firstItem() }} to {{ $batches->lastItem() }} of {{ $batches->total() }} batches
                    </div>
                    <nav>
                        {{ $batches->withQueryString()->links() }}
                    </nav>
                </div>
            </div>
        </div>
    </div>
</div>

@if(isset($product))
<!-- Add Batch Modal -->
<div class="modal fade" id="addBatchModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Batch</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('products.add-stock', $product) }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Batch Number *</label>
                            <input type="text" name="batch_data[batch_number]" class="form-control" 
                                   value="BATCH-{{ date('Ymd') }}-{{ strtoupper(substr(md5(uniqid()), 0, 6)) }}"
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Supplier</label>
                            <select name="batch_data[supplier_id]" class="form-select">
                                <option value="">Select Supplier</option>
                                @foreach($suppliers ?? [] as $supplier)
                                    <option value="{{ $supplier->id }}">{{ $supplier->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Manufacture Date</label>
                            <input type="date" name="batch_data[manufacture_date]" class="form-control"
                                   value="{{ date('Y-m-d') }}">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Expiry Date</label>
                            <input type="date" name="batch_data[expiry_date]" class="form-control" 
                                   min="{{ date('Y-m-d') }}">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Quantity *</label>
                            <input type="number" name="quantity" class="form-control" min="1" value="1" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Batch Cost Price *</label>
                            <input type="number" name="batch_data[cost_price]" class="form-control" 
                                   step="0.01" min="0" value="{{ $product->cost_price }}" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Batch Sale Price</label>
                            <input type="number" name="batch_data[sale_price]" class="form-control" 
                                   step="0.01" min="0" value="{{ $product->sale_price }}">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Warehouse Location</label>
                            <input type="text" name="batch_data[location]" class="form-control" 
                                   placeholder="Shelf A-1, Rack B-2, etc.">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Reason for Addition *</label>
                            <input type="text" name="reason" class="form-control" 
                                   placeholder="New stock purchase, production batch, etc." required>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Notes (Optional)</label>
                            <textarea name="notes" class="form-control" rows="2"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Batch</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endif
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // Tooltip initialization
        $('[data-bs-toggle="tooltip"]').tooltip();

        // Set expiry discount for batch
        $('.set-expiry-discount').click(function() {
            const batchId = $(this).data('batch-id');
            const productId = $(this).data('product-id');
            
            Swal.fire({
                title: 'Set Expiry Discount',
                input: 'number',
                inputLabel: 'Discount Percentage',
                inputPlaceholder: 'Enter discount %',
                inputAttributes: {
                    step: '0.01',
                    min: '0',
                    max: '100'
                },
                showCancelButton: true,
                confirmButtonText: 'Apply Discount',
                cancelButtonText: 'Cancel',
                inputValidator: (value) => {
                    if (!value || parseFloat(value) <= 0 || parseFloat(value) > 100) {
                        return 'Please enter a valid percentage (0-100)';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: `{{ route('products.set-expiry-discount', ':id') }}`.replace(':id', productId),
                        type: 'POST',
                        data: {
                            _token: '{{ csrf_token() }}',
                            discount_type: 'percent',
                            discount_value: result.value,
                            discount_reason: 'Batch expiry discount',
                            batch_id: batchId
                        },
                        success: function(response) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Discount Applied',
                                text: response.message
                            }).then(() => {
                                location.reload();
                            });
                        },
                        error: function(xhr) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: xhr.responseJSON?.message || 'Failed to apply discount'
                            });
                        }
                    });
                }
            });
        });

        // Delete batch
        $('.delete-batch').click(function() {
            const batchId = $(this).data('batch-id');
            
            Swal.fire({
                title: 'Delete Batch?',
                text: 'Are you sure you want to delete this batch? This action cannot be undone.',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Yes, delete it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: `{{ route('batches.index') }}/${batchId}`,
                        type: 'DELETE',
                        headers: {
                            'X-CSRF-TOKEN': '{{ csrf_token() }}'
                        },
                        success: function(response) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Deleted!',
                                text: response.message
                            }).then(() => {
                                location.reload();
                            });
                        },
                        error: function(xhr) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: xhr.responseJSON?.message || 'Failed to delete batch'
                            });
                        }
                    });
                }
            });
        });

        @if(isset($product))
        // Auto-generate batch number
        function generateBatchNumber() {
            const date = new Date().toISOString().slice(0,10).replace(/-/g, '');
            const random = Math.random().toString(36).substr(2, 6).toUpperCase();
            return `BATCH-${date}-${random}`;
        }

        // Update batch number field
        $('input[name="batch_data[batch_number]"]').on('click', function() {
            if ($(this).val().startsWith('BATCH-')) {
                $(this).val(generateBatchNumber());
            }
        });
        @endif
    });
</script>
@endsection