@extends('layouts.app')

@section('style')
    <style>
        .price-change-badge {
            padding: 4px 8px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
        }
        .price-increase {
            background-color: #d1fae5;
            color: #065f46;
        }
        .price-decrease {
            background-color: #fee2e2;
            color: #991b1b;
        }
        .price-no-change {
            background-color: #e5e7eb;
            color: #6b7280;
        }
        .pricing-card {
            transition: all 0.3s;
            border: 1px solid #e5e7eb;
        }
        .pricing-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        .filter-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            border-left: 4px solid #667eea;
        }
        .price-input {
            max-width: 120px;
        }
        .profit-margin-badge {
            padding: 3px 6px;
            border-radius: 10px;
            font-size: 10px;
            font-weight: 600;
        }
        .margin-high {
            background-color: #d1fae5;
            color: #065f46;
        }
        .margin-medium {
            background-color: #fef3c7;
            color: #92400e;
        }
        .margin-low {
            background-color: #fee2e2;
            color: #991b1b;
        }
        .bulk-action-panel {
            background: #f0f9ff;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #bae6fd;
        }
        .stock-status {
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 500;
        }
        .status-in-stock {
            background-color: #d1fae5;
            color: #065f46;
        }
        .status-low-stock {
            background-color: #fef3c7;
            color: #92400e;
        }
        .status-out-of-stock {
            background-color: #fee2e2;
            color: #991b1b;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Products</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">Products</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Pricing Management</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#bulkUpdateModal">
                        <i class="bx bx-cog"></i> Bulk Update
                    </button>
                    <button type="button" class="btn btn-primary dropdown-toggle dropdown-toggle-split" 
                            data-bs-toggle="dropdown" aria-expanded="false">
                        <span class="visually-hidden">Toggle Dropdown</span>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li>
                            <a class="dropdown-item" href="{{ route('price-histories.index') }}">
                                <i class="bx bx-history"></i> Price History
                            </a>
                        </li>
                        <li>
                            <a class="dropdown-item" href="{{ route('price-histories.analytics') }}">
                                <i class="bx bx-line-chart"></i> Price Analytics
                            </a>
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <li>
                            <a class="dropdown-item" href="{{ route('products.export') }}">
                                <i class="bx bx-export"></i> Export Prices
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Statistics Cards -->
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-4 row-cols-xl-4 mb-4">
            <div class="col">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Avg. Cost Price</p>
                                <h4 class="my-1">${{ number_format($products->avg('cost_price') ?? 0, 2) }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-primary text-primary ms-auto">
                                <i class="bx bx-dollar-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Avg. Sale Price</p>
                                <h4 class="my-1">${{ number_format($products->avg('sale_price') ?? 0, 2) }}</h4>
                            </div>
                            <div class="widgets-icons bg-light-success text-success ms-auto">
                                <i class="bx bx-money"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Avg. Profit Margin</p>
                                <h4 class="my-1">
                                    {{ number_format($products->avg(function($product) {
                                        return $product->sale_price > 0 ? 
                                            (($product->sale_price - $product->cost_price) / $product->sale_price) * 100 : 0;
                                    }) ?? 0, 1) }}%
                                </h4>
                            </div>
                            <div class="widgets-icons bg-light-info text-info ms-auto">
                                <i class="bx bx-trending-up"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">On Discount</p>
                                <h4 class="my-1">
                                    {{ $products->where(function($product) {
                                        return $product->discount_price > 0 || $product->discount_percent > 0;
                                    })->count() }}
                                </h4>
                            </div>
                            <div class="widgets-icons bg-light-warning text-warning ms-auto">
                                <i class="bx bx-tag"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Bulk Update Panel -->
        <div class="bulk-action-panel">
            <h5 class="mb-3"><i class="bx bx-cog"></i> Bulk Price Update</h5>
            <form id="bulkUpdateForm" method="POST" action="{{ route('products.pricing.bulk-update') }}">
                @csrf
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Price Type</label>
                        <select name="price_type" class="form-select" required>
                            <option value="sale_price">Sale Price</option>
                            <option value="cost_price">Cost Price</option>
                            <option value="wholesale_price">Wholesale Price</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Update Type</label>
                        <select name="update_type" class="form-select" required>
                            <option value="set">Set to exact value</option>
                            <option value="increase">Increase by amount</option>
                            <option value="decrease">Decrease by amount</option>
                            <option value="percentage">Change by percentage</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Value</label>
                        <input type="number" name="value" class="form-control" step="0.01" required 
                               placeholder="0.00">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Change Reason</label>
                        <input type="text" name="change_reason" class="form-control" required 
                               placeholder="Reason for change">
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="button" class="btn btn-primary w-100" onclick="applyBulkUpdate()">
                            <i class="bx bx-check"></i> Apply
                        </button>
                    </div>
                </div>
                <input type="hidden" name="product_ids" id="bulkProductIds" value="">
            </form>
        </div>

        <!-- Filters Section -->
        <div class="filter-section">
            <form method="GET" action="{{ route('products.pricing.index') }}" id="filterForm">
                <div class="row g-3">
                    <!-- Search Input -->
                    <div class="col-md-3">
                        <label class="form-label">Search</label>
                        <div class="input-group">
                            <input type="text" name="search" class="form-control" 
                                   placeholder="Product name or SKU" 
                                   value="{{ request('search') }}">
                            <button class="btn btn-outline-secondary" type="submit">
                                <i class="bx bx-search"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Category Filter -->
                    <div class="col-md-2">
                        <label class="form-label">Category</label>
                        <select name="category_id" class="form-select">
                            <option value="">All Categories</option>
                            @foreach($categories as $category)
                                <option value="{{ $category->id }}" 
                                        {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                    {{ $category->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Brand Filter -->
                    <div class="col-md-2">
                        <label class="form-label">Brand</label>
                        <select name="brand_id" class="form-select">
                            <option value="">All Brands</option>
                            @foreach($brands as $brand)
                                <option value="{{ $brand->id }}" 
                                        {{ request('brand_id') == $brand->id ? 'selected' : '' }}>
                                    {{ $brand->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Margin Filter -->
                    <div class="col-md-2">
                        <label class="form-label">Profit Margin</label>
                        <select name="margin_range" class="form-select">
                            <option value="">All Margins</option>
                            <option value="high" {{ request('margin_range') == 'high' ? 'selected' : '' }}>
                                High (&gt; 40%)
                            </option>
                            <option value="medium" {{ request('margin_range') == 'medium' ? 'selected' : '' }}>
                                Medium (20-40%)
                            </option>
                            <option value="low" {{ request('margin_range') == 'low' ? 'selected' : '' }}>
                                Low (&lt; 20%)
                            </option>
                            <option value="negative" {{ request('margin_range') == 'negative' ? 'selected' : '' }}>
                                Negative
                            </option>
                        </select>
                    </div>

                    <!-- Actions Column -->
                    <div class="col-md-1 d-flex align-items-end">
                        <div class="d-flex gap-2 w-100">
                            <button type="submit" class="btn btn-outline-primary">
                                <i class="bx bx-filter"></i>
                            </button>
                            <a href="{{ route('products.pricing.index') }}" class="btn btn-outline-secondary">
                                <i class="bx bx-reset"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Products Pricing Table -->
        <div class="card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h5 class="card-title mb-0">Product Pricing</h5>
                    <div class="d-flex gap-2">
                        <!-- Select All -->
                        <div class="form-check d-flex align-items-center">
                            <input class="form-check-input me-2" type="checkbox" id="selectAllProducts">
                            <label class="form-check-label" for="selectAllProducts">Select All</label>
                        </div>
                    </div>
                </div>

                @if(session('success'))
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bx bx-check-circle"></i> {{ session('success') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif

                @if(session('error'))
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bx bx-error-circle"></i> {{ session('error') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif

                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th style="width: 40px;">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="selectAllHeader">
                                    </div>
                                </th>
                                <th>Product</th>
                                <th>Stock</th>
                                <th>Cost Price</th>
                                <th>Sale Price</th>
                                <th>Discount Price</th>
                                <th>Profit Margin</th>
                                <th>Wholesale Price</th>
                                <th class="text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($products as $product)
                                <tr class="pricing-card" data-product-id="{{ $product->id }}">
                                    <td>
                                        <div class="form-check">
                                            <input class="form-check-input product-check" type="checkbox" 
                                                   value="{{ $product->id }}">
                                        </div>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="flex-shrink-0 me-2">
                                                @if($product->image)
                                                    <img src="{{ asset('storage/' . $product->image) }}" 
                                                         alt="{{ $product->name }}" 
                                                         width="40" height="40" class="rounded">
                                                @else
                                                    <div class="rounded bg-light d-flex align-items-center justify-content-center" 
                                                         style="width: 40px; height: 40px;">
                                                        <i class="bx bx-package text-muted"></i>
                                                    </div>
                                                @endif
                                            </div>
                                            <div class="flex-grow-1">
                                                <strong>{{ $product->name }}</strong>
                                                <div class="text-muted small">{{ $product->sku }}</div>
                                                <div class="small">
                                                    @if($product->category)
                                                        <span class="badge bg-light text-dark">{{ $product->category->name }}</span>
                                                    @endif
                                                    @if($product->brand)
                                                        <span class="badge bg-light text-dark">{{ $product->brand->name }}</span>
                                                    @endif
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="d-flex flex-column">
                                            <span class="fw-bold">{{ $product->stock }}</span>
                                            @if($product->stock <= 0)
                                                <span class="stock-status status-out-of-stock">Out of Stock</span>
                                            @elseif($product->stock <= $product->minimum_stock)
                                                <span class="stock-status status-low-stock">Low Stock</span>
                                            @else
                                                <span class="stock-status status-in-stock">In Stock</span>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="price-input-group">
                                            <div class="input-group input-group-sm">
                                                <span class="input-group-text">$</span>
                                                <input type="number" class="form-control price-input cost-price" 
                                                       value="{{ number_format($product->cost_price, 2) }}" 
                                                       step="0.01" min="0"
                                                       data-original="{{ $product->cost_price }}"
                                                       data-product-id="{{ $product->id }}"
                                                       data-price-type="cost_price">
                                            </div>
                                            <div class="mt-1">
                                                <button type="button" class="btn btn-sm btn-outline-secondary update-price"
                                                        data-product-id="{{ $product->id }}"
                                                        data-price-type="cost_price">
                                                    <i class="bx bx-save"></i> Update
                                                </button>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="price-input-group">
                                            <div class="input-group input-group-sm">
                                                <span class="input-group-text">$</span>
                                                <input type="number" class="form-control price-input sale-price" 
                                                       value="{{ number_format($product->sale_price, 2) }}" 
                                                       step="0.01" min="0"
                                                       data-original="{{ $product->sale_price }}"
                                                       data-product-id="{{ $product->id }}"
                                                       data-price-type="sale_price">
                                            </div>
                                            <div class="mt-1">
                                                <button type="button" class="btn btn-sm btn-outline-secondary update-price"
                                                        data-product-id="{{ $product->id }}"
                                                        data-price-type="sale_price">
                                                    <i class="bx bx-save"></i> Update
                                                </button>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        @if($product->discount_price || $product->discount_percent)
                                            <div class="d-flex flex-column">
                                                @if($product->discount_price)
                                                    <span class="text-success fw-bold">
                                                        ${{ number_format($product->discount_price, 2) }}
                                                    </span>
                                                    <small class="text-muted">
                                                        {{ number_format((($product->sale_price - $product->discount_price) / $product->sale_price) * 100, 1) }}% off
                                                    </small>
                                                @else
                                                    <span class="text-success fw-bold">
                                                        {{ $product->discount_percent }}% off
                                                    </span>
                                                    <small class="text-muted">
                                                        ${{ number_format($product->sale_price * (1 - ($product->discount_percent / 100)), 2) }}
                                                    </small>
                                                @endif
                                            </div>
                                        @else
                                            <span class="text-muted">No discount</span>
                                        @endif
                                    </td>
                                    <td>
                                        @php
                                            $profitMargin = $product->sale_price > 0 ? 
                                                (($product->sale_price - $product->cost_price) / $product->sale_price) * 100 : 0;
                                        @endphp
                                        <div class="d-flex align-items-center">
                                            <span class="me-2">{{ number_format($profitMargin, 1) }}%</span>
                                            @if($profitMargin > 40)
                                                <span class="profit-margin-badge margin-high">High</span>
                                            @elseif($profitMargin >= 20)
                                                <span class="profit-margin-badge margin-medium">Medium</span>
                                            @elseif($profitMargin > 0)
                                                <span class="profit-margin-badge margin-low">Low</span>
                                            @else
                                                <span class="profit-margin-badge" style="background-color: #9ca3af; color: white;">
                                                    Negative
                                                </span>
                                            @endif
                                        </div>
                                        <small class="text-muted">
                                            ${{ number_format($product->sale_price - $product->cost_price, 2) }}
                                        </small>
                                    </td>
                                    <td>
                                        @if($product->wholesale_price)
                                            <div class="input-group input-group-sm">
                                                <span class="input-group-text">$</span>
                                                <input type="number" class="form-control price-input wholesale-price" 
                                                       value="{{ number_format($product->wholesale_price, 2) }}" 
                                                       step="0.01" min="0"
                                                       data-original="{{ $product->wholesale_price }}"
                                                       data-product-id="{{ $product->id }}"
                                                       data-price-type="wholesale_price">
                                            </div>
                                            <div class="mt-1">
                                                <button type="button" class="btn btn-sm btn-outline-secondary update-price"
                                                        data-product-id="{{ $product->id }}"
                                                        data-price-type="wholesale_price">
                                                    <i class="bx bx-save"></i>
                                                </button>
                                            </div>
                                        @else
                                            <button type="button" class="btn btn-sm btn-outline-secondary set-wholesale"
                                                    data-product-id="{{ $product->id }}">
                                                <i class="bx bx-plus"></i> Add
                                            </button>
                                        @endif
                                    </td>
                                    <td class="text-end">
                                        <div class="d-flex justify-content-end gap-1">
                                            <!-- Price History -->
                                            <a href="{{ route('products.price-history', $product) }}" 
                                               class="btn btn-outline-info btn-sm"
                                               data-bs-toggle="tooltip" title="Price History">
                                                <i class="bx bx-history"></i>
                                            </a>
                                            
                                            <!-- Batches -->
                                            @if($product->track_batches)
                                            <a href="{{ route('products.batches', $product) }}" 
                                               class="btn btn-outline-primary btn-sm"
                                               data-bs-toggle="tooltip" title="View Batches">
                                                <i class="bx bx-package"></i>
                                            </a>
                                            @endif
                                            
                                            <!-- Set Discount -->
                                            <button type="button" class="btn btn-outline-warning btn-sm set-discount"
                                                    data-bs-toggle="modal" data-bs-target="#discountModal{{ $product->id }}"
                                                    data-product-id="{{ $product->id }}">
                                                <i class="bx bx-tag"></i>
                                            </button>
                                            
                                            <!-- Quick Actions Dropdown -->
                                            <div class="dropdown">
                                                <button class="btn btn-outline-secondary btn-sm" 
                                                        type="button" data-bs-toggle="dropdown">
                                                    <i class="bx bx-dots-vertical-rounded"></i>
                                                </button>
                                                <ul class="dropdown-menu dropdown-menu-end">
                                                    <li>
                                                        <a class="dropdown-item" href="{{ route('products.show', $product) }}">
                                                            <i class="bx bx-show"></i> View Product
                                                        </a>
                                                    </li>
                                                    <li>
                                                        <a class="dropdown-item" href="{{ route('products.edit', $product) }}">
                                                            <i class="bx bx-edit"></i> Edit Product
                                                        </a>
                                                    </li>
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li>
                                                        <a class="dropdown-item text-danger clear-discount" 
                                                           href="#" data-product-id="{{ $product->id }}">
                                                            <i class="bx bx-x-circle"></i> Clear Discount
                                                        </a>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>

                                        <!-- Discount Modal -->
                                        <div class="modal fade" id="discountModal{{ $product->id }}" tabindex="-1" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">Set Discount for {{ $product->name }}</h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <form class="discount-form" data-product-id="{{ $product->id }}">
                                                        @csrf
                                                        <div class="modal-body">
                                                            <div class="mb-3">
                                                                <label class="form-label">Current Price</label>
                                                                <input type="text" class="form-control" 
                                                                       value="${{ number_format($product->sale_price, 2) }}" readonly>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Discount Type</label>
                                                                <select name="discount_type" class="form-select" required>
                                                                    <option value="amount">Fixed Amount</option>
                                                                    <option value="percent">Percentage</option>
                                                                </select>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Discount Value</label>
                                                                <input type="number" name="discount_value" class="form-control" 
                                                                       step="0.01" min="0" required>
                                                            </div>
                                                            <div class="mb-3">
                                                                <label class="form-label">Reason (Optional)</label>
                                                                <input type="text" name="reason" class="form-control" 
                                                                       placeholder="Seasonal sale, clearance, etc.">
                                                            </div>
                                                        </div>
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                            <button type="submit" class="btn btn-primary">Apply Discount</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="9" class="text-center py-5">
                                        <div class="py-5">
                                            <i class="bx bx-dollar-circle display-4 text-muted mb-3"></i>
                                            <h5 class="text-muted">No products found</h5>
                                            <p class="text-muted mb-4">
                                                @if(request()->hasAny(['search', 'category_id', 'brand_id', 'margin_range']))
                                                    Try changing your filters
                                                @else
                                                    No products available for pricing management
                                                @endif
                                            </p>
                                            <a href="{{ route('products.index') }}" class="btn btn-primary">
                                                <i class="bx bx-package"></i> View Products
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-4">
                    <div class="pagination-info">
                        Showing {{ $products->firstItem() }} to {{ $products->lastItem() }} of {{ $products->total() }} entries
                    </div>
                    <nav>
                        {{ $products->withQueryString()->links() }}
                    </nav>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Update Modal -->
<div class="modal fade" id="bulkUpdateModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Bulk Price Update</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info">
                    <i class="bx bx-info-circle"></i>
                    Select products from the table first, then configure the bulk update here.
                </div>
                <p id="selectedCount" class="text-center fw-bold mb-4">0 products selected</p>
                <form id="bulkUpdateFormModal">
                    <div class="mb-3">
                        <label class="form-label">Price Type</label>
                        <select name="price_type" class="form-select" required>
                            <option value="sale_price">Sale Price</option>
                            <option value="cost_price">Cost Price</option>
                            <option value="wholesale_price">Wholesale Price</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Update Type</label>
                        <select name="update_type" class="form-select" required>
                            <option value="set">Set to exact value</option>
                            <option value="increase">Increase by amount</option>
                            <option value="decrease">Decrease by amount</option>
                            <option value="percentage">Change by percentage</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Value</label>
                        <input type="number" name="value" class="form-control" step="0.01" required 
                               placeholder="0.00">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Change Reason</label>
                        <input type="text" name="change_reason" class="form-control" required 
                               placeholder="Reason for change">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Notes (Optional)</label>
                        <textarea name="notes" class="form-control" rows="2" 
                                  placeholder="Additional notes"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="submitBulkUpdate()">Apply Update</button>
            </div>
        </div>
    </div>
</div>
@endsection
@section('script')
<script>
    // ============================================
    // PRICING MANAGEMENT JAVASCRIPT
    // ============================================
    
    // ROUTE DEFINITIONS WITH DEBUGGING
    const routes = {
        setDiscount: (productId) => {
            const url = '{{ route("products.pricing.set-discount", ["product" => "PRODUCT_ID"]) }}'.replace('PRODUCT_ID', productId);
            console.log('Generated setDiscount URL:', url);
            return url;
        },
        clearDiscount: (productId) => {
            const url = '{{ route("products.pricing.clear-discount", ["product" => "PRODUCT_ID"]) }}'.replace('PRODUCT_ID', productId);
            console.log('Generated clearDiscount URL:', url);
            return url;
        },
        updatePrice: (productId) => {
            const url = '{{ route("products.pricing.update", ["product" => "PRODUCT_ID"]) }}'.replace('PRODUCT_ID', productId);
            console.log('Generated updatePrice URL:', url);
            return url;
        },
        bulkUpdate: '{{ route("products.pricing.bulk-update") }}'
    };

    console.log('Pricing routes initialized:', {
        testSetDiscount: routes.setDiscount(123),
        testClearDiscount: routes.clearDiscount(123),
        testUpdatePrice: routes.updatePrice(123),
        bulkUpdate: routes.bulkUpdate
    });

    // Enhanced error handler for AJAX calls
    function handleAjaxError(xhr, defaultMessage = 'An error occurred') {
        console.error('AJAX Error Details:', {
            status: xhr.status,
            statusText: xhr.statusText,
            response: xhr.responseText,
            responseJSON: xhr.responseJSON,
            requestURL: xhr.responseURL
        });
        
        let errorMessage = defaultMessage;
        
        if (xhr.responseJSON) {
            if (xhr.responseJSON.message) {
                errorMessage = xhr.responseJSON.message;
            }
            if (xhr.responseJSON.errors) {
                const errors = Object.values(xhr.responseJSON.errors).flat();
                errorMessage = errors.join(', ');
            }
        } else if (xhr.status === 0) {
            errorMessage = 'Network error. Please check your connection.';
        } else if (xhr.status === 401) {
            errorMessage = 'Session expired. Please refresh the page.';
        } else if (xhr.status === 403) {
            errorMessage = 'You do not have permission to perform this action.';
        } else if (xhr.status === 404) {
            errorMessage = 'The requested resource was not found.';
        } else if (xhr.status === 422) {
            errorMessage = 'Validation error. Please check your input.';
        } else if (xhr.status === 500) {
            errorMessage = 'Server error. Please try again later.';
        }
        
        return errorMessage;
    }

    // DOCUMENT READY FUNCTION
    $(document).ready(function() {
        console.log('Pricing management page loaded');
        
        // Initialize tooltips
        $('[data-bs-toggle="tooltip"]').tooltip();
        
        // Initialize select all functionality
        initializeSelectAll();
        
        // Initialize event listeners
        initializeEventListeners();
        
        // Initialize keyboard shortcuts
        initializeKeyboardShortcuts();
    });

    // INITIALIZATION FUNCTIONS
    
    function initializeSelectAll() {
        $('#selectAllHeader').change(function() {
            const isChecked = $(this).prop('checked');
            $('.product-check').prop('checked', isChecked);
            updateSelectedCount();
        });

        $('#selectAllProducts').change(function() {
            const isChecked = $(this).prop('checked');
            $('.product-check').prop('checked', isChecked);
            updateSelectedCount();
        });

        // Update selected count when individual checkboxes change
        $('.product-check').change(function() {
            updateSelectedCount();
        });
    }

    function initializeEventListeners() {
        // Individual price update buttons
        $(document).on('click', '.update-price', function() {
            const productId = $(this).data('product-id');
            const priceType = $(this).data('price-type');
            const inputField = $(this).closest('td').find('.price-input[data-price-type="' + priceType + '"]');
            const newPrice = parseFloat(inputField.val());
            const originalPrice = parseFloat(inputField.data('original'));

            console.log('Price update requested:', { productId, priceType, newPrice, originalPrice });

            if (isNaN(newPrice) || newPrice < 0) {
                Swal.fire({
                    icon: 'error',
                    title: 'Invalid Price',
                    text: 'Please enter a valid price (0 or greater).'
                });
                return;
            }

            Swal.fire({
                title: 'Confirm Price Update',
                html: `Update <strong>${priceType.replace('_', ' ')}</strong><br>
                       From: <strong>$${originalPrice.toFixed(2)}</strong><br>
                       To: <strong>$${newPrice.toFixed(2)}</strong>`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Update',
                cancelButtonText: 'Cancel',
                confirmButtonColor: '#3085d6',
                cancelButtonColor: '#d33',
                showLoaderOnConfirm: true,
                preConfirm: () => {
                    return updatePrice(productId, priceType, newPrice, 'manual_update');
                },
                allowOutsideClick: () => !Swal.isLoading()
            }).then((result) => {
                if (result.isDismissed) {
                    // Reset to original value if cancelled
                    inputField.val(originalPrice.toFixed(2));
                }
            });
        });

        // Set wholesale price button
        $(document).on('click', '.set-wholesale', function() {
            const productId = $(this).data('product-id');
            const productRow = $(this).closest('tr');
            const currentSalePrice = parseFloat(productRow.find('.sale-price').val());
            
            Swal.fire({
                title: 'Set Wholesale Price',
                input: 'number',
                inputLabel: `Current sale price: $${currentSalePrice.toFixed(2)}`,
                inputPlaceholder: 'Enter wholesale price',
                inputValue: (currentSalePrice * 0.8).toFixed(2), // Suggest 20% discount
                inputAttributes: {
                    step: '0.01',
                    min: '0',
                    max: currentSalePrice.toString()
                },
                showCancelButton: true,
                confirmButtonText: 'Set Price',
                cancelButtonText: 'Cancel',
                inputValidator: (value) => {
                    const price = parseFloat(value);
                    if (!value || isNaN(price) || price < 0) {
                        return 'Please enter a valid price';
                    }
                    if (price > currentSalePrice) {
                        return 'Wholesale price cannot exceed sale price';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const wholesalePrice = parseFloat(result.value);
                    
                    Swal.fire({
                        title: 'Setting Wholesale Price',
                        text: 'Please wait...',
                        allowOutsideClick: false,
                        didOpen: () => {
                            Swal.showLoading();
                        }
                    });

                    updatePrice(productId, 'wholesale_price', wholesalePrice, 'wholesale_price_set')
                        .then(() => {
                            // Update UI on success
                            $(this).replaceWith(`
                                <div class="input-group input-group-sm">
                                    <span class="input-group-text">$</span>
                                    <input type="number" class="form-control price-input wholesale-price" 
                                           value="${wholesalePrice.toFixed(2)}" 
                                           step="0.01" min="0"
                                           data-original="${wholesalePrice}"
                                           data-product-id="${productId}"
                                           data-price-type="wholesale_price">
                                </div>
                                <div class="mt-1">
                                    <button type="button" class="btn btn-sm btn-outline-secondary update-price"
                                            data-product-id="${productId}"
                                            data-price-type="wholesale_price">
                                        <i class="bx bx-save"></i>
                                    </button>
                                </div>
                            `);
                            
                            Swal.fire({
                                icon: 'success',
                                title: 'Success',
                                text: 'Wholesale price set successfully!'
                            });
                        })
                        .catch((error) => {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: error
                            });
                        });
                }
            });
        });

        // Apply discount form submission
        $(document).on('submit', '.discount-form', function(e) {
            e.preventDefault();
            const productId = $(this).data('product-id');
            const formData = $(this).serializeArray();
            const data = {};
            
            formData.forEach(item => {
                data[item.name] = item.value;
            });
            
            console.log('Setting discount for product:', productId, data);
            
            Swal.fire({
                title: 'Applying Discount',
                text: 'Please wait...',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: routes.setDiscount(productId),
                type: 'POST',
                data: JSON.stringify(data),
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                success: function(response) {
                    Swal.fire({
                        icon: 'success',
                        title: 'Discount Applied',
                        text: response.message,
                        timer: 2000,
                        showConfirmButton: false
                    }).then(() => {
                        location.reload();
                    });
                },
                error: function(xhr) {
                    const errorMessage = handleAjaxError(xhr, 'Failed to apply discount');
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: errorMessage
                    });
                }
            });
        });

        // Clear discount link
        $(document).on('click', '.clear-discount', function(e) {
            e.preventDefault();
            const productId = $(this).data('product-id');
            const productName = $(this).closest('tr').find('td:nth-child(2) strong').text();
            
            Swal.fire({
                title: 'Clear Discount',
                html: `Are you sure you want to remove the discount from <strong>${productName}</strong>?`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Yes, clear it!',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    Swal.fire({
                        title: 'Clearing Discount',
                        text: 'Please wait...',
                        allowOutsideClick: false,
                        didOpen: () => {
                            Swal.showLoading();
                        }
                    });

                    $.ajax({
                        url: routes.clearDiscount(productId),
                        type: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        success: function(response) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Discount Cleared',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        },
                        error: function(xhr) {
                            const errorMessage = handleAjaxError(xhr, 'Failed to clear discount');
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: errorMessage
                            });
                        }
                    });
                }
            });
        });

        // Auto-submit filters on change
        $('select[name="category_id"], select[name="brand_id"], select[name="margin_range"]').change(function() {
            console.log('Filter changed, submitting form...');
            $('#filterForm').submit();
        });

        // Search input debounce
        let searchTimer;
        $('input[name="search"]').on('keyup', function() {
            clearTimeout(searchTimer);
            const searchValue = $(this).val();
            searchTimer = setTimeout(() => {
                if (searchValue.length >= 2 || searchValue.length === 0) {
                    console.log('Search triggered:', searchValue);
                    $('#filterForm').submit();
                }
            }, 500);
        });

        // Close modals on escape
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape') {
                $('.modal').modal('hide');
            }
        });
    }

    function initializeKeyboardShortcuts() {
        $(document).keydown(function(e) {
            // Ctrl/Cmd + B for bulk update modal
            if ((e.ctrlKey || e.metaKey) && e.key === 'b') {
                e.preventDefault();
                $('#bulkUpdateModal').modal('show');
            }
            
            // Ctrl/Cmd + F for search focus
            if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
                e.preventDefault();
                $('input[name="search"]').focus().select();
            }
            
            // Ctrl/Cmd + A to select all products
            if ((e.ctrlKey || e.metaKey) && e.key === 'a') {
                e.preventDefault();
                $('.product-check').prop('checked', true);
                updateSelectedCount();
            }
            
            // Escape to clear selection
            if (e.key === 'Escape' && !$('.modal').hasClass('show')) {
                $('.product-check').prop('checked', false);
                updateSelectedCount();
            }
        });
    }

    // CORE FUNCTIONS
    
    function updateSelectedCount() {
        const selectedCount = $('.product-check:checked').length;
        const totalCount = $('.product-check').length;
        
        $('#selectedCount').text(`${selectedCount} of ${totalCount} products selected`);
        
        // Update bulk form hidden input
        const selectedIds = [];
        $('.product-check:checked').each(function() {
            selectedIds.push($(this).val());
        });
        $('#bulkProductIds').val(selectedIds.join(','));
        
        // Update select all checkbox state
        const allChecked = selectedCount === totalCount && totalCount > 0;
        $('#selectAllHeader').prop('checked', allChecked);
        $('#selectAllProducts').prop('checked', allChecked);
        
        console.log('Selected products:', selectedIds);
    }

    function updatePrice(productId, priceType, newPrice, changeReason = 'manual_update') {
        return new Promise((resolve, reject) => {
            console.log('Updating price:', { productId, priceType, newPrice, changeReason });
            
            $.ajax({
                url: routes.updatePrice(productId),
                type: 'POST',
                data: JSON.stringify({
                    price_type: priceType,
                    new_price: newPrice,
                    change_reason: changeReason,
                    notes: 'Updated via pricing management interface'
                }),
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                success: function(response) {
                    console.log('Price update successful:', response);
                    
                    // Update original value in data attribute
                    $(`.price-input[data-product-id="${productId}"][data-price-type="${priceType}"]`)
                        .data('original', newPrice);
                    
                    // Update profit margin display if cost or sale price changed
                    if (priceType === 'cost_price' || priceType === 'sale_price') {
                        updateProfitMarginDisplay(productId);
                    }
                    
                    resolve(response);
                },
                error: function(xhr) {
                    const errorMessage = handleAjaxError(xhr, 'Failed to update price');
                    reject(errorMessage);
                }
            });
        });
    }

    function updateProfitMarginDisplay(productId) {
        const row = $(`[data-product-id="${productId}"]`);
        const costPrice = parseFloat(row.find('.cost-price').val());
        const salePrice = parseFloat(row.find('.sale-price').val());
        
        if (salePrice > 0) {
            const profitMargin = ((salePrice - costPrice) / salePrice) * 100;
            const profitAmount = salePrice - costPrice;
            
            // Update margin percentage
            const marginElement = row.find('.profit-margin-badge').parent().prev();
            marginElement.text(`${profitMargin.toFixed(1)}%`);
            
            // Update margin badge
            const marginBadge = row.find('.profit-margin-badge');
            marginBadge.removeClass('margin-high margin-medium margin-low');
            
            if (profitMargin > 40) {
                marginBadge.addClass('margin-high').text('High');
            } else if (profitMargin >= 20) {
                marginBadge.addClass('margin-medium').text('Medium');
            } else if (profitMargin > 0) {
                marginBadge.addClass('margin-low').text('Low');
            } else {
                marginBadge.css({
                    'background-color': '#9ca3af',
                    'color': 'white'
                }).text('Negative');
            }
            
            // Update profit amount
            row.find('.text-muted').text(`$${profitAmount.toFixed(2)}`);
        }
    }

    // BULK UPDATE FUNCTIONS
    
    function applyBulkUpdate() {
        const selectedIds = $('#bulkProductIds').val();
        if (!selectedIds) {
            Swal.fire({
                icon: 'warning',
                title: 'No Products Selected',
                text: 'Please select products first by checking the boxes in the table.'
            });
            return;
        }

        const productIdsArray = selectedIds.split(',').map(id => id.trim()).filter(id => id);
        
        if (productIdsArray.length === 0) {
            Swal.fire({
                icon: 'warning',
                title: 'No Valid Products',
                text: 'Please select at least one product.'
            });
            return;
        }

        // Get form data
        const formData = $('#bulkUpdateForm').serializeArray();
        const data = {};
        
        formData.forEach(item => {
            data[item.name] = item.value;
        });
        
        // Add product_ids as array
        data.product_ids = productIdsArray;
        
        console.log('Bulk update data:', data);
        
        Swal.fire({
            title: 'Confirm Bulk Update',
            html: `Apply price update to <strong>${productIdsArray.length}</strong> selected products?<br><br>
                   <strong>Type:</strong> ${data.price_type.replace('_', ' ')}<br>
                   <strong>Action:</strong> ${data.update_type}<br>
                   <strong>Value:</strong> ${data.value}`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Update All',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            showLoaderOnConfirm: true,
            preConfirm: () => {
                return new Promise((resolve) => {
                    $.ajax({
                        url: routes.bulkUpdate,
                        type: 'POST',
                        data: JSON.stringify(data),
                        contentType: 'application/json',
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        success: function(response) {
                            resolve(response);
                        },
                        error: function(xhr) {
                            const errorMessage = handleAjaxError(xhr, 'Failed to update prices');
                            Swal.showValidationMessage(errorMessage);
                        }
                    });
                });
            },
            allowOutsideClick: () => !Swal.isLoading()
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    icon: 'success',
                    title: 'Bulk Update Complete',
                    text: result.value.message,
                    timer: 3000,
                    showConfirmButton: false
                }).then(() => {
                    location.reload();
                });
            }
        });
    }

    function submitBulkUpdate() {
        const selectedIds = $('#bulkProductIds').val();
        if (!selectedIds) {
            Swal.fire({
                icon: 'warning',
                title: 'No Products Selected',
                text: 'Please select products first by checking the boxes in the table.'
            });
            return;
        }

        const productIdsArray = selectedIds.split(',').map(id => id.trim()).filter(id => id);
        
        if (productIdsArray.length === 0) {
            Swal.fire({
                icon: 'warning',
                title: 'No Valid Products',
                text: 'Please select at least one product.'
            });
            return;
        }

        // Get form data from modal
        const formData = $('#bulkUpdateFormModal').serializeArray();
        const data = {};
        
        formData.forEach(item => {
            data[item.name] = item.value;
        });
        
        // Add product_ids as array
        data.product_ids = productIdsArray;
        
        console.log('Modal bulk update data:', data);
        
        Swal.fire({
            title: 'Confirm Bulk Update',
            html: `Apply price update to <strong>${productIdsArray.length}</strong> selected products?`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Update All',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            showLoaderOnConfirm: true,
            preConfirm: () => {
                return new Promise((resolve) => {
                    $.ajax({
                        url: routes.bulkUpdate,
                        type: 'POST',
                        data: JSON.stringify(data),
                        contentType: 'application/json',
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        success: function(response) {
                            resolve(response);
                        },
                        error: function(xhr) {
                            const errorMessage = handleAjaxError(xhr, 'Failed to update prices');
                            Swal.showValidationMessage(errorMessage);
                        }
                    });
                });
            },
            allowOutsideClick: () => !Swal.isLoading()
        }).then((result) => {
            if (result.isConfirmed) {
                // Close modal
                $('#bulkUpdateModal').modal('hide');
                
                Swal.fire({
                    icon: 'success',
                    title: 'Bulk Update Complete',
                    text: result.value.message,
                    timer: 3000,
                    showConfirmButton: false
                }).then(() => {
                    location.reload();
                });
            }
        });
    }

    console.log('Pricing management JavaScript loaded successfully');
</script>
@endsection