@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            border-left: 4px solid #667eea;
        }
        .form-section h5 {
            color: #495057;
            border-bottom: 2px solid #dee2e6;
            padding-bottom: 10px;
            margin-bottom: 20px;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .switch-label {
            cursor: pointer;
        }
        .price-input-group {
            position: relative;
        }
        .price-input-group .input-group-text {
            min-width: 100px;
        }
        .expiry-fields {
            background: #fff3cd;
            border-radius: 8px;
            padding: 15px;
            border: 1px solid #ffeaa7;
        }
        .image-upload-container {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        .image-upload-container:hover {
            border-color: #667eea;
            background: #f8f9fa;
        }
        .image-preview {
            max-width: 200px;
            max-height: 200px;
            border-radius: 10px;
        }
        .current-image {
            border: 2px solid #dee2e6;
            border-radius: 8px;
            padding: 10px;
            background: white;
        }
        .image-actions {
            margin-top: 10px;
        }
        .form-check-input:checked {
            background-color: #667eea;
            border-color: #667eea;
        }
        .tax-badge {
            font-size: 0.75rem;
            padding: 0.25rem 0.5rem;
        }
        .btn-group-tax .btn {
            flex: 1;
        }
        .tax-status-card {
            border-left: 4px solid;
        }
        .tax-status-vat {
            border-left-color: #667eea;
        }
        .tax-status-zero {
            border-left-color: #38a169;
        }
        .tax-status-exempt {
            border-left-color: #718096;
        }
        .tax-preview-card {
            background: linear-gradient(135deg, #667eea0d 0%, #764ba20d 100%);
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Products</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">Products</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Edit Product: {{ $product->name }}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('products.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to List
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Edit Product: {{ $product->name }}</h4>
                        
                        @if($errors->any())
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <h6 class="alert-heading mb-2"><i class="bx bx-error-circle"></i> Validation Errors</h6>
                                <ul class="mb-0 ps-3">
                                    @foreach($errors->all() as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                </ul>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('success'))
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <i class="bx bx-check-circle"></i> {{ session('success') }}
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('error'))
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="bx bx-error-circle"></i> {{ session('error') }}
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('products.update', $product->id) }}" enctype="multipart/form-data" id="productForm" novalidate>
                            @csrf
                            @method('PUT')
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h5><i class="bx bx-info-circle"></i> Basic Information</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Product Name</label>
                                        <input type="text" name="name" class="form-control @error('name') is-invalid @enderror" 
                                               value="{{ old('name', $product->name) }}" required 
                                               placeholder="Enter product name">
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">SKU</label>
                                        <input type="text" name="sku" class="form-control @error('sku') is-invalid @enderror" 
                                               value="{{ old('sku', $product->sku) }}" required 
                                               placeholder="Enter SKU (must be unique)">
                                        @error('sku')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Stock Keeping Unit - must be unique</div>
                                        <input type="hidden" name="original_sku" value="{{ $product->sku }}">
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Barcode</label>
                                        <input type="text" name="barcode" class="form-control @error('barcode') is-invalid @enderror" 
                                               value="{{ old('barcode', $product->barcode) }}" 
                                               placeholder="Enter barcode (optional)">
                                        @error('barcode')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Reference Number</label>
                                        <input type="text" name="ref_number" class="form-control @error('ref_number') is-invalid @enderror" 
                                               value="{{ old('ref_number', $product->ref_number) }}" 
                                               placeholder="Your internal reference number">
                                        @error('ref_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Category</label>
                                        <select name="category_id" class="form-select @error('category_id') is-invalid @enderror" required>
                                            <option value="">Select Category</option>
                                            @foreach($categories as $category)
                                                <option value="{{ $category->id }}" 
                                                        {{ old('category_id', $product->category_id) == $category->id ? 'selected' : '' }}>
                                                    {{ $category->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('category_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Brand</label>
                                        <select name="brand_id" class="form-select @error('brand_id') is-invalid @enderror">
                                            <option value="">Select Brand</option>
                                            @foreach($brands as $brand)
                                                <option value="{{ $brand->id }}" 
                                                        {{ old('brand_id', $product->brand_id) == $brand->id ? 'selected' : '' }}>
                                                    {{ $brand->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('brand_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Supplier</label>
                                        <select name="supplier_id" class="form-select @error('supplier_id') is-invalid @enderror">
                                            <option value="">Select Supplier</option>
                                            @foreach($suppliers as $supplier)
                                                <option value="{{ $supplier->id }}" 
                                                        {{ old('supplier_id', $product->supplier_id) == $supplier->id ? 'selected' : '' }}>
                                                    {{ $supplier->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('supplier_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Unit</label>
                                        <select name="unit_id" class="form-select @error('unit_id') is-invalid @enderror">
                                            <option value="">Select Unit</option>
                                            @foreach($units as $unit)
                                                <option value="{{ $unit->id }}" 
                                                        {{ old('unit_id', $product->unit_id) == $unit->id ? 'selected' : '' }}>
                                                    {{ $unit->name }} ({{ $unit->abbreviation }})
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('unit_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Description</label>
                                    <textarea name="description" class="form-control @error('description') is-invalid @enderror" rows="3" 
                                              placeholder="Product description">{{ old('description', $product->description) }}</textarea>
                                    @error('description')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Short Description</label>
                                    <textarea name="short_description" class="form-control @error('short_description') is-invalid @enderror" rows="2" 
                                              placeholder="Brief description for listings">{{ old('short_description', $product->short_description) }}</textarea>
                                    @error('short_description')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Pricing -->
                            <div class="form-section">
                                <h5><i class="bx bx-dollar-circle"></i> Pricing</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Cost Price</label>
                                        <div class="input-group price-input-group">
                                            <span class="input-group-text">KES</span>
                                            <input type="number" name="cost_price" class="form-control @error('cost_price') is-invalid @enderror" 
                                                   step="0.01" min="0" value="{{ old('cost_price', $product->cost_price) }}" required 
                                                   placeholder="0.00" id="costPrice">
                                        </div>
                                        @error('cost_price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Sale Price</label>
                                        <div class="input-group price-input-group">
                                            <span class="input-group-text">KES</span>
                                            <input type="number" name="sale_price" class="form-control @error('sale_price') is-invalid @enderror" 
                                                   step="0.01" min="0" value="{{ old('sale_price', $product->sale_price) }}" required 
                                                   placeholder="0.00" id="salePrice">
                                        </div>
                                        @error('sale_price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Wholesale Price</label>
                                        <div class="input-group price-input-group">
                                            <span class="input-group-text">KES</span>
                                            <input type="number" name="wholesale_price" class="form-control @error('wholesale_price') is-invalid @enderror" 
                                                   step="0.01" min="0" value="{{ old('wholesale_price', $product->wholesale_price) }}" 
                                                   placeholder="Optional">
                                        </div>
                                        @error('wholesale_price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <div class="row">
                                            <div class="col-6">
                                                <label class="form-label">Discount Price</label>
                                                <div class="input-group price-input-group">
                                                    <span class="input-group-text">KES</span>
                                                    <input type="number" name="discount_price" class="form-control @error('discount_price') is-invalid @enderror" 
                                                           step="0.01" min="0" value="{{ old('discount_price', $product->discount_price) }}" 
                                                           placeholder="0.00" id="discountPrice">
                                                </div>
                                                @error('discount_price')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                            </div>
                                            <div class="col-6">
                                                <label class="form-label">Discount %</label>
                                                <div class="input-group price-input-group">
                                                    <input type="number" name="discount_percent" class="form-control @error('discount_percent') is-invalid @enderror" 
                                                           min="0" max="100" value="{{ old('discount_percent', $product->discount_percent) }}" 
                                                           placeholder="0" id="discountPercent">
                                                    <span class="input-group-text">%</span>
                                                </div>
                                                @error('discount_percent')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                            </div>
                                        </div>
                                        <div class="form-text">Enter either discount price or percentage (not both)</div>
                                    </div>
                                </div>
                                
                                <!-- Enhanced Tax Configuration -->
                                @php
                                    // Determine tax status based on product data
                                    $taxStatus = 'vat';
                                    $taxId = old('tax_id', $product->tax_id);
                                    $hasVat = old('has_vat', $product->has_vat);
                                    
                                    if ($taxId && $hasVat) {
                                        $taxRate = $taxes->where('id', $taxId)->first()->rate ?? 0;
                                        $taxStatus = $taxRate == 0 ? 'zero' : 'vat';
                                    } elseif (!$hasVat) {
                                        $taxStatus = 'exempt';
                                    }
                                    
                                    $zeroRatedTax = $taxes->where('rate', 0)->first();
                                @endphp
                                
                                <div class="card tax-status-card 
                                    @if($taxStatus == 'vat') tax-status-vat 
                                    @elseif($taxStatus == 'zero') tax-status-zero 
                                    @else tax-status-exempt @endif mb-3">
                                    <div class="card-header">
                                        <h6 class="mb-0"><i class="bx bx-receipt"></i> Tax Configuration</h6>
                                    </div>
                                    <div class="card-body">
                                        <!-- Tax Status Selection -->
                                        <div class="row mb-3">
                                            <div class="col-md-12">
                                                <label class="form-label required-field">Tax Status</label>
                                                <div class="btn-group btn-group-tax w-100" role="group">
                                                    <input type="radio" class="btn-check" name="tax_status" id="taxStatusVat" 
                                                           value="vat" {{ old('tax_status', $taxStatus) == 'vat' ? 'checked' : '' }}>
                                                    <label class="btn btn-outline-primary" for="taxStatusVat">
                                                        <i class="bx bx-check-shield"></i> VAT Taxable
                                                    </label>
                                                    
                                                    <input type="radio" class="btn-check" name="tax_status" id="taxStatusZero" 
                                                           value="zero" {{ old('tax_status', $taxStatus) == 'zero' ? 'checked' : '' }}>
                                                    <label class="btn btn-outline-success" for="taxStatusZero">
                                                        <i class="bx bx-shield-x"></i> Zero-Rated
                                                    </label>
                                                    
                                                    <input type="radio" class="btn-check" name="tax_status" id="taxStatusExempt" 
                                                           value="exempt" {{ old('tax_status', $taxStatus) == 'exempt' ? 'checked' : '' }}>
                                                    <label class="btn btn-outline-secondary" for="taxStatusExempt">
                                                        <i class="bx bx-shield"></i> VAT Exempt
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Tax Rate Selection (only for VAT Taxable) -->
                                        <div class="row mb-3" id="taxRateContainer" style="display: {{ old('tax_status', $taxStatus) == 'vat' ? 'block' : 'none' }};">
                                            <div class="col-md-6">
                                                <label class="form-label required-field">Select VAT Rate</label>
                                                <div class="input-group">
                                                    <select name="tax_id" class="form-select @error('tax_id') is-invalid @enderror" id="taxSelect">
                                                        <option value="">-- Select VAT Rate --</option>
                                                        @foreach($taxes as $tax)
                                                            @if($tax->rate > 0) {{-- Only show positive rates for VAT taxable --}}
                                                            <option value="{{ $tax->id }}" 
                                                                    data-rate="{{ $tax->rate }}"
                                                                    {{ old('tax_id', $product->tax_id) == $tax->id ? 'selected' : '' }}>
                                                                {{ $tax->name }} ({{ $tax->rate }}%)
                                                            </option>
                                                            @endif
                                                        @endforeach
                                                    </select>
                                                    <button type="button" class="btn btn-outline-primary" onclick="manageTaxes()">
                                                        <i class="bx bx-cog"></i>
                                                    </button>
                                                </div>
                                                @error('tax_id')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                                <div class="form-text">Select applicable VAT rate (16%, 8%, etc.)</div>
                                            </div>
                                            <div class="col-md-6">
                                                <div id="selectedTaxInfo" class="mt-4" style="display: {{ old('tax_id', $product->tax_id) ? 'block' : 'none' }};">
                                                    <div class="alert alert-info p-2">
                                                        <small>
                                                            <i class="bx bx-info-circle"></i>
                                                            Selected: <strong id="taxName"></strong> (<span id="taxRate"></span>%)
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Tax Explanation -->
                                        <div class="alert alert-info">
                                            <small>
                                                <i class="bx bx-info-circle"></i>
                                                <strong>Understanding Tax Options:</strong>
                                                <ul class="mb-0 mt-2">
                                                    <li><strong>VAT Taxable</strong>: Product is subject to VAT (select a rate above)</li>
                                                    <li><strong>Zero-Rated VAT</strong>: VAT applies at 0% rate (for exports/essentials)</li>
                                                    <li><strong>VAT Exempt</strong>: No VAT applies at all (exempt by law)</li>
                                                </ul>
                                            </small>
                                        </div>
                                        
                                        <!-- Hidden fields for backend processing -->
                                        <input type="hidden" name="has_vat" id="hasVatField" value="{{ old('has_vat', $product->has_vat ? '1' : '0') }}">
                                        <input type="hidden" name="tax_id_actual" id="taxIdField" value="{{ old('tax_id', $product->tax_id) }}">
                                        
                                        <!-- Zero-rated tax info (hidden field) -->
                                        <input type="hidden" id="zeroRatedTaxId" value="{{ $zeroRatedTax->id ?? '' }}">
                                    </div>
                                </div>
                                
                                <!-- Tax Preview -->
                                <div class="row mt-3">
                                    <div class="col-12">
                                        <div class="card tax-preview-card">
                                            <div class="card-body p-3">
                                                <h6 class="mb-2"><i class="bx bx-calculator"></i> Tax & Price Preview</h6>
                                                <div class="row">
                                                    <div class="col-md-3">
                                                        <div class="mb-2">
                                                            <small class="text-muted">Sale Price</small>
                                                            <div class="h5" id="salePricePreview">KES 0.00</div>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-3">
                                                        <div class="mb-2">
                                                            <small class="text-muted">Tax Rate</small>
                                                            <div class="h5" id="taxRatePreview">0%</div>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-3">
                                                        <div class="mb-2">
                                                            <small class="text-muted">Tax Amount</small>
                                                            <div class="h5 text-danger" id="taxAmountPreview">KES 0.00</div>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-3">
                                                        <div class="mb-2">
                                                            <small class="text-muted">Total Price</small>
                                                            <div class="h5 text-success" id="totalPricePreview">KES 0.00</div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="row mt-2">
                                                    <div class="col-12">
                                                        <div class="alert @if($taxStatus == 'vat') alert-warning @elseif($taxStatus == 'zero') alert-success @else alert-info @endif p-2" id="taxStatusPreview">
                                                            <small>
                                                                <i class="bx bx-info-circle"></i>
                                                                <strong>Current Status:</strong> 
                                                                <span id="taxStatusText">
                                                                    @if($taxStatus == 'vat')
                                                                        VAT Taxable 
                                                                        @if($taxId)
                                                                            ({{ $taxes->where('id', $taxId)->first()->rate ?? '0' }}%)
                                                                        @endif
                                                                    @elseif($taxStatus == 'zero')
                                                                        Zero-Rated VAT (0%)
                                                                    @else
                                                                        VAT Exempt (No VAT)
                                                                    @endif
                                                                </span>
                                                            </small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Inventory -->
                            <div class="form-section">
                                <h5><i class="bx bx-package"></i> Inventory Management</h5>
                                
                                <div class="row">
                                    <div class="col-md-4 mb-3">
                                        <label class="form-label required-field">Current Stock</label>
                                        <input type="number" name="stock" class="form-control @error('stock') is-invalid @enderror" 
                                               min="0" value="{{ old('stock', $product->stock) }}" required 
                                               placeholder="0">
                                        @error('stock')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-4 mb-3">
                                        <label class="form-label required-field">Reorder Point</label>
                                        <input type="number" name="reorder_point" class="form-control @error('reorder_point') is-invalid @enderror" 
                                               min="0" value="{{ old('reorder_point', $product->reorder_point) }}" required 
                                               placeholder="10">
                                        @error('reorder_point')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Alert when stock reaches this level</div>
                                    </div>
                                    
                                    <div class="col-md-4 mb-3">
                                        <label class="form-label required-field">Minimum Stock</label>
                                        <input type="number" name="minimum_stock" class="form-control @error('minimum_stock') is-invalid @enderror" 
                                               min="0" value="{{ old('minimum_stock', $product->minimum_stock) }}" required 
                                               placeholder="10">
                                        @error('minimum_stock')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Maximum Stock</label>
                                        <input type="number" name="maximum_stock" class="form-control @error('maximum_stock') is-invalid @enderror" 
                                               min="0" value="{{ old('maximum_stock', $product->maximum_stock) }}" 
                                               placeholder="Optional">
                                        @error('maximum_stock')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Maximum stock capacity (optional)</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <div class="alert alert-info h-100 d-flex align-items-center">
                                            <div>
                                                <i class="bx bx-info-circle"></i>
                                                <small><strong>Stock Status:</strong> {{ ucfirst($product->stock_status) }}</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Expiry & Batch Tracking -->
                                <div class="expiry-fields mt-3">
                                    <h6><i class="bx bx-calendar"></i> Expiry & Batch Tracking</h6>
                                    
                                    <div class="row">
                                        <div class="col-md-4 mb-3">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" name="has_expiry" 
                                                       id="hasExpiry" value="1" 
                                                       {{ old('has_expiry', $product->has_expiry) ? 'checked' : '' }}>
                                                <label class="form-check-label switch-label" for="hasExpiry">
                                                    Has Expiry Date
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4 mb-3">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" name="track_batches" 
                                                       id="trackBatches" value="1" 
                                                       {{ old('track_batches', $product->track_batches) ? 'checked' : '' }}>
                                                <label class="form-check-label switch-label" for="trackBatches">
                                                    Track by Batches
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4 mb-3">
                                            <label class="form-label">Expiry Warning (Days)</label>
                                            <input type="number" name="expiry_warning_days" class="form-control @error('expiry_warning_days') is-invalid @enderror" 
                                                   min="1" max="365" value="{{ old('expiry_warning_days', $product->expiry_warning_days ?? 30) }}" 
                                                   placeholder="30">
                                            @error('expiry_warning_days')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                            <div class="form-text">Alert before expiry</div>
                                        </div>
                                    </div>
                                    
                                    <div class="alert alert-info mt-2">
                                        <small>
                                            <i class="bx bx-info-circle"></i> 
                                            <strong>Expiry Tracking:</strong> For products with expiry dates (food, medicine, etc.)<br>
                                            <strong>Batch Tracking:</strong> Track stock by batches/lots. Required for expiry tracking.
                                        </small>
                                    </div>
                                </div>
                            </div>

                            <!-- Product Image -->
                            <div class="form-section">
                                <h5><i class="bx bx-image"></i> Product Image</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Current Image</label>
                                        <div class="current-image">
                                            @if($product->image)
                                                <img src="{{ asset('storage/' . $product->image) }}" 
                                                     alt="{{ $product->name }}" 
                                                     class="img-fluid image-preview mb-2"
                                                     style="max-height: 200px;">
                                                <div class="image-actions">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="checkbox" 
                                                               name="remove_image" id="removeImage" value="1">
                                                        <label class="form-check-label text-danger" for="removeImage">
                                                            <i class="bx bx-trash"></i> Remove current image
                                                        </label>
                                                    </div>
                                                </div>
                                            @else
                                                <div class="text-center py-4">
                                                    <i class="bx bx-image-alt display-4 text-muted"></i>
                                                    <p class="mt-2 text-muted">No image uploaded</p>
                                                </div>
                                            @endif
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label">Upload New Image</label>
                                        <div class="image-upload-container" onclick="document.getElementById('imageUpload').click();">
                                            <i class="bx bx-cloud-upload display-4 text-muted mb-3"></i>
                                            <p class="mb-2">Click to upload or drag and drop</p>
                                            <p class="text-muted small mb-0">PNG, JPG, GIF up to 2MB</p>
                                            <input type="file" name="image" id="imageUpload" class="d-none" 
                                                   accept="image/*">
                                        </div>
                                        <div class="mt-3" id="newImagePreview" style="display: none;">
                                            <img src="" alt="New Image Preview" 
                                                 class="img-fluid image-preview mb-2">
                                            <button type="button" class="btn btn-sm btn-outline-danger" 
                                                    onclick="removeNewImage()">
                                                <i class="bx bx-trash"></i> Remove
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Status & Options -->
                            <div class="form-section">
                                <h5><i class="bx bx-cog"></i> Status & Options</h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label required-field">Status</label>
                                        <select name="status" class="form-select @error('status') is-invalid @enderror" required>
                                            <option value="active" {{ old('status', $product->status) == 'active' ? 'selected' : '' }}>
                                                Active
                                            </option>
                                            <option value="inactive" {{ old('status', $product->status) == 'inactive' ? 'selected' : '' }}>
                                                Inactive
                                            </option>
                                            <option value="discontinued" {{ old('status', $product->status) == 'discontinued' ? 'selected' : '' }}>
                                                Discontinued
                                            </option>
                                        </select>
                                        @error('status')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <div class="form-check form-switch mt-4">
                                            <input class="form-check-input" type="checkbox" name="is_featured" 
                                                   id="isFeatured" value="1" 
                                                   {{ old('is_featured', $product->is_featured) ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="isFeatured">
                                                Featured Product
                                            </label>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-4 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" name="is_new" 
                                                   id="isNew" value="1" 
                                                   {{ old('is_new', $product->is_new) ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="isNew">
                                                New Product
                                            </label>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-4 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" name="is_bestseller" 
                                                   id="isBestseller" value="1" 
                                                   {{ old('is_bestseller', $product->is_bestseller) ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="isBestseller">
                                                Bestseller
                                            </label>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-4 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" name="track_inventory" 
                                                   id="trackInventory" value="1" 
                                                   {{ old('track_inventory', $product->track_inventory ?? true) ? 'checked' : '' }}>
                                            <label class="form-check-label switch-label" for="trackInventory">
                                                Track Inventory
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5" id="submitBtn">
                                                <i class="bx bx-save"></i> Update Product
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5">
                                                <i class="bx bx-reset"></i> Reset Changes
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('products.show', $product) }}" class="btn btn-outline-info px-5">
                                                <i class="bx bx-show"></i> View Product
                                            </a>
                                            <a href="{{ route('products.index') }}" class="btn btn-light px-5 ms-2">
                                                <i class="bx bx-x"></i> Cancel
                                            </a>
                                            <button type="button" class="btn btn-danger px-5 ms-2" 
                                                    data-bs-toggle="modal" data-bs-target="#deleteModal">
                                                <i class="bx bx-trash"></i> Delete
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-danger">
                    <i class="bx bx-error-circle"></i>
                    <strong>Warning!</strong> This action cannot be undone.
                </div>
                <p>Are you sure you want to delete the product "<strong>{{ $product->name }}</strong>" (SKU: {{ $product->sku }})?</p>
                <p class="text-danger">All product data, including inventory records, will be permanently deleted.</p>
                
                                @if($product->stock > 0)
                <div class="alert alert-warning mt-3">
                    <i class="bx bx-error-alt"></i>
                    <strong>Warning:</strong> This product has {{ $product->stock }} units in stock.
                    Deleting it will also delete all stock records.
                </div>
                @endif
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form action="{{ route('products.destroy', $product->id) }}" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete Product</button>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<!-- Include jQuery Validation Plugin -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery-validate/1.19.5/jquery.validate.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery-validate/1.19.5/additional-methods.min.js"></script>

<script>
$(document).ready(function() {
    // Toggle expiry fields
    $('#hasExpiry').change(function() {
        if ($(this).is(':checked')) {
            $('#trackBatches').prop('checked', true).prop('disabled', false);
        } else {
            $('#trackBatches').prop('disabled', false);
        }
    });
    
    // Prevent form submission if validation fails
    $('#productForm').on('submit', function(e) {
        // Get the submit button
        var submitBtn = $('#submitBtn');
        
        // Disable button and show loading state
        submitBtn.prop('disabled', true);
        submitBtn.html('<i class="bx bx-loader bx-spin"></i> Updating...');
        
        // Validate tax selection
        const taxStatus = $('input[name="tax_status"]:checked').val();
        const taxSelect = $('#taxSelect');
        
        if (taxStatus === 'vat' && !taxSelect.val()) {
            e.preventDefault();
            Swal.fire({
                icon: 'warning',
                title: 'Tax Rate Required',
                text: 'Please select a VAT rate for VAT taxable products.',
                confirmButtonColor: '#3085d6',
            });
            submitBtn.prop('disabled', false);
            submitBtn.html('<i class="bx bx-save"></i> Update Product');
            return false;
        }
        
        // Let the form submit normally
        return true;
    });
    
    // Check SKU uniqueness (skip if unchanged)
    $('input[name="sku"]').on('blur', function() {
        var sku = $(this).val();
        var originalSku = $('input[name="original_sku"]').val();
        
        if (sku.length > 0 && sku !== originalSku) {
            $.ajax({
                url: '{{ route("products.search") }}',
                type: 'GET',
                data: { 
                    q: sku,
                    check_sku: true,
                    exclude_id: {{ $product->id }}
                },
                success: function(response) {
                    if (response.exists) {
                        // Show warning but don't prevent submission
                        Swal.fire({
                            icon: 'warning',
                            title: 'SKU Already Exists',
                            text: 'This SKU is already in use. Please choose a different SKU to avoid conflicts.',
                            confirmButtonColor: '#3085d6',
                        });
                        $('input[name="sku"]').addClass('is-invalid');
                        $('#skuFeedback').remove();
                        $('input[name="sku"]').after('<div class="invalid-feedback" id="skuFeedback">This SKU already exists in the system.</div>');
                    } else {
                        $('input[name="sku"]').removeClass('is-invalid');
                        $('#skuFeedback').remove();
                    }
                }
            });
        }
    });
    
    // Validate discount fields
    $('input[name="discount_price"], input[name="discount_percent"]').on('input', function() {
        var discountPrice = $('input[name="discount_price"]').val();
        var discountPercent = $('input[name="discount_percent"]').val();
        var salePrice = parseFloat($('#salePrice').val()) || 0;
        
        if (discountPrice && discountPercent) {
            Swal.fire({
                icon: 'warning',
                title: 'Discount Conflict',
                text: 'Please enter either discount price OR discount percentage, not both.',
                confirmButtonColor: '#3085d6',
            });
            $(this).val('');
        }
        
        // Validate discount price doesn't exceed sale price
        if (discountPrice && salePrice > 0 && parseFloat(discountPrice) > salePrice) {
            Swal.fire({
                icon: 'warning',
                title: 'Invalid Discount',
                text: 'Discount price cannot be higher than sale price.',
                confirmButtonColor: '#3085d6',
            });
            $('input[name="discount_price"]').val('');
        }
    });
    
    // Auto-calculate markup
    $('input[name="cost_price"], input[name="sale_price"]').on('input', function() {
        var cost = parseFloat($('#costPrice').val()) || 0;
        var sale = parseFloat($('#salePrice').val()) || 0;
        
        if (cost > 0 && sale > 0) {
            var markup = ((sale - cost) / cost) * 100;
            var profit = sale - cost;
            
            // Update tax preview
            updateTaxPreview();
        }
    });
    
    // Form validation for required fields
    $('input[required], select[required]').on('blur', function() {
        if (!$(this).val()) {
            $(this).addClass('is-invalid');
            if (!$(this).next('.invalid-feedback').length) {
                $(this).after('<div class="invalid-feedback">This field is required.</div>');
            }
        } else {
            $(this).removeClass('is-invalid');
            $(this).next('.invalid-feedback').remove();
        }
    });

    // Tax Management Functions
    function manageTaxes() {
        window.location.href = '{{ route("taxes.index") }}';
    }
    
    // Tax status radio buttons handler
    $('input[name="tax_status"]').change(function() {
        const status = $(this).val();
        const taxContainer = $('#taxRateContainer');
        const taxSelect = $('#taxSelect');
        const hasVatField = $('#hasVatField');
        const taxIdField = $('#taxIdField');
        const zeroRatedTaxId = $('#zeroRatedTaxId').val();
        
        // Update UI based on selection
        if (status === 'vat') {
            taxContainer.show();
            hasVatField.val('1');
            // Keep selected tax ID if exists
            updateTaxPreview();
        } else if (status === 'zero') {
            taxContainer.hide();
            hasVatField.val('1');
            taxIdField.val(zeroRatedTaxId);
            updateTaxPreview(0); // 0% tax
        } else if (status === 'exempt') {
            taxContainer.hide();
            hasVatField.val('0');
            taxIdField.val('');
            updateTaxPreview(0, false);
        }
        
        // Update card border color
        const taxCard = $('.tax-status-card');
        taxCard.removeClass('tax-status-vat tax-status-zero tax-status-exempt');
        if (status === 'vat') {
            taxCard.addClass('tax-status-vat');
        } else if (status === 'zero') {
            taxCard.addClass('tax-status-zero');
        } else {
            taxCard.addClass('tax-status-exempt');
        }
    });
    
    // Tax rate selection handler
    $('#taxSelect').change(function() {
        const selectedOption = $(this).find('option:selected');
        const taxIdField = $('#taxIdField');
        taxIdField.val(selectedOption.val());
        
        // Show/hide selected tax info
        if (selectedOption.val()) {
            $('#selectedTaxInfo').show();
            $('#taxName').text(selectedOption.text().split('(')[0].trim());
            $('#taxRate').text(selectedOption.data('rate'));
        } else {
            $('#selectedTaxInfo').hide();
        }
        
        updateTaxPreview();
    });
    
    // Update tax preview
    function updateTaxPreview(customRate = null, hasVat = true) {
        const salePrice = parseFloat($('#salePrice').val()) || 0;
        const discountPrice = parseFloat($('#discountPrice').val()) || 0;
        const discountPercent = parseFloat($('#discountPercent').val()) || 0;
        
        // Calculate final price after discount
        let finalPrice = salePrice;
        if (discountPrice > 0) {
            finalPrice = discountPrice;
        } else if (discountPercent > 0) {
            finalPrice = salePrice * (1 - (discountPercent / 100));
        }
        
        let taxRate = 0;
        let statusText = '';
        
        const taxStatus = $('input[name="tax_status"]:checked').val();
        
        if (taxStatus === 'vat') {
            const selectedOption = $('#taxSelect').find('option:selected');
            taxRate = selectedOption.data('rate') || 0;
            statusText = `VAT Taxable (${taxRate}%)`;
        } else if (taxStatus === 'zero') {
            taxRate = 0;
            statusText = 'Zero-Rated VAT (0%)';
        } else if (taxStatus === 'exempt') {
            taxRate = 0;
            statusText = 'VAT Exempt (No VAT)';
        }
        
        // Use custom values if provided
        if (customRate !== null) taxRate = customRate;
        
        // Calculate tax amount
        let taxAmount = 0;
        if (hasVat && taxRate > 0) {
            taxAmount = finalPrice * (taxRate / 100);
        }
        
        const totalPrice = finalPrice + taxAmount;
        
        // Update displays
        $('#salePricePreview').text('KES ' + finalPrice.toFixed(2));
        $('#taxRatePreview').text(taxRate + '%');
        $('#taxAmountPreview').text('KES ' + taxAmount.toFixed(2));
        $('#totalPricePreview').text('KES ' + totalPrice.toFixed(2));
        $('#taxStatusText').text(statusText);
        
        // Update status color
        const statusPreview = $('#taxStatusPreview');
        statusPreview.removeClass('alert-warning alert-success alert-info');
        if (taxStatus === 'vat') {
            statusPreview.addClass('alert-warning');
        } else if (taxStatus === 'zero') {
            statusPreview.addClass('alert-success');
        } else {
            statusPreview.addClass('alert-info');
        }
    }
    
    // Update tax preview when price fields change
    $('#salePrice, #costPrice, #discountPrice, #discountPercent').on('input', function() {
        updateTaxPreview();
    });
    
    // Image upload preview
    $('#imageUpload').change(function(e) {
        var input = this;
        if (input.files && input.files[0]) {
            var reader = new FileReader();
            
            reader.onload = function(e) {
                $('#newImagePreview').find('img').attr('src', e.target.result);
                $('#newImagePreview').show();
            }
            
            reader.readAsDataURL(input.files[0]);
        }
    });
    
    function removeNewImage() {
        $('#imageUpload').val('');
        $('#newImagePreview').hide();
        $('#newImagePreview').find('img').attr('src', '');
    }
    
    // Initialize tax preview
    updateTaxPreview();
    
    // Enable form submission button when all required fields are filled
    function checkFormValidity() {
        var isValid = true;
        $('input[required], select[required]').each(function() {
            if (!$(this).val()) {
                isValid = false;
                return false; // Break loop
            }
        });
        
        // Check sale price vs cost price
        var cost = parseFloat($('#costPrice').val()) || 0;
        var sale = parseFloat($('#salePrice').val()) || 0;
        if (sale < cost) {
            Swal.fire({
                icon: 'warning',
                title: 'Price Validation',
                text: 'Sale price cannot be less than cost price.',
                confirmButtonColor: '#3085d6',
            });
            isValid = false;
        }
        
        $('#submitBtn').prop('disabled', !isValid);
        return isValid;
    }
    
    // Check form validity on input change
    $('input, select').on('input change', function() {
        checkFormValidity();
    });
    
    // Initialize form validation
    checkFormValidity();
});
</script>
@endsection